"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerNoteRoute = registerNoteRoute;

var _configSchema = require("@osd/config-schema");

var _notebooks = require("../../../common/constants/notebooks");

var _notebooks2 = _interopRequireDefault(require("../../adaptors/notebooks"));

var _metrics_helper = require("../../common/metrics/metrics_helper");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
function registerNoteRoute(router) {
  // Fetch all the notebooks available
  router.get({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/`,
    validate: {}
  }, async (context, request, response) => {
    (0, _metrics_helper.addRequestToMetric)('notebooks', 'get', 'count');
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    let notebooksData = [];

    try {
      notebooksData = await _notebooks2.default.viewNotes(opensearchNotebooksClient, _notebooks.wreckOptions);
      return response.ok({
        body: {
          data: notebooksData
        }
      });
    } catch (error) {
      (0, _metrics_helper.addRequestToMetric)('notebooks', 'get', error);
      console.log('Notebook:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // Get all paragraphs of notebooks

  router.get({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/{noteId}`,
    validate: {
      params: _configSchema.schema.object({
        noteId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    (0, _metrics_helper.addRequestToMetric)('notebooks', 'get', 'count');
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const notebookinfo = await _notebooks2.default.fetchNote(opensearchNotebooksClient, request.params.noteId, _notebooks.wreckOptions);
      return response.ok({
        body: notebookinfo
      });
    } catch (error) {
      (0, _metrics_helper.addRequestToMetric)('notebooks', 'get', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // Add a Notebook

  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note`,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    (0, _metrics_helper.addRequestToMetric)('notebooks', 'create', 'count');
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const addResponse = await _notebooks2.default.addNote(opensearchNotebooksClient, request.body, _notebooks.wreckOptions);
      return response.ok({
        body: addResponse.message.objectId
      });
    } catch (error) {
      (0, _metrics_helper.addRequestToMetric)('notebooks', 'create', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // Rename a notebook

  router.put({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/rename`,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        noteId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    (0, _metrics_helper.addRequestToMetric)('notebooks', 'update', 'count');
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const renameResponse = await _notebooks2.default.renameNote(opensearchNotebooksClient, request.body, _notebooks.wreckOptions);
      return response.ok({
        body: renameResponse
      });
    } catch (error) {
      (0, _metrics_helper.addRequestToMetric)('notebooks', 'update', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // Clone a notebook

  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/clone`,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        noteId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    (0, _metrics_helper.addRequestToMetric)('notebooks', 'create', 'count');
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const cloneResponse = await _notebooks2.default.cloneNote(opensearchNotebooksClient, request.body, _notebooks.wreckOptions);
      return response.ok({
        body: cloneResponse
      });
    } catch (error) {
      (0, _metrics_helper.addRequestToMetric)('notebooks', 'create', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // Delete notebooks

  router.delete({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/{noteList}`,
    validate: {
      params: _configSchema.schema.object({
        noteList: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    (0, _metrics_helper.addRequestToMetric)('notebooks', 'delete', 'count');
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const delResponse = await _notebooks2.default.deleteNote(opensearchNotebooksClient, request.params.noteList, _notebooks.wreckOptions);
      return response.ok({
        body: delResponse
      });
    } catch (error) {
      (0, _metrics_helper.addRequestToMetric)('notebooks', 'delete', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // Add sample notebooks

  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/addSampleNotebooks`,
    validate: {
      body: _configSchema.schema.object({
        visIds: _configSchema.schema.arrayOf(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    (0, _metrics_helper.addRequestToMetric)('notebooks', 'add_samples', 'count');
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const addSampleNotesResponse = await _notebooks2.default.addSampleNotes(opensearchNotebooksClient, request.body.visIds, _notebooks.wreckOptions);
      return response.ok({
        body: addSampleNotesResponse
      });
    } catch (error) {
      (0, _metrics_helper.addRequestToMetric)('notebooks', 'add_samples', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIm5vdGVSb3V0ZXIudHMiXSwibmFtZXMiOlsicmVnaXN0ZXJOb3RlUm91dGUiLCJyb3V0ZXIiLCJnZXQiLCJwYXRoIiwiTk9URUJPT0tTX0FQSV9QUkVGSVgiLCJ2YWxpZGF0ZSIsImNvbnRleHQiLCJyZXF1ZXN0IiwicmVzcG9uc2UiLCJvcGVuc2VhcmNoTm90ZWJvb2tzQ2xpZW50Iiwib2JzZXJ2YWJpbGl0eV9wbHVnaW4iLCJvYnNlcnZhYmlsaXR5Q2xpZW50IiwiYXNTY29wZWQiLCJub3RlYm9va3NEYXRhIiwiQkFDS0VORCIsInZpZXdOb3RlcyIsIndyZWNrT3B0aW9ucyIsIm9rIiwiYm9keSIsImRhdGEiLCJlcnJvciIsImNvbnNvbGUiLCJsb2ciLCJjdXN0b20iLCJzdGF0dXNDb2RlIiwibWVzc2FnZSIsInBhcmFtcyIsInNjaGVtYSIsIm9iamVjdCIsIm5vdGVJZCIsInN0cmluZyIsIm5vdGVib29raW5mbyIsImZldGNoTm90ZSIsInBvc3QiLCJuYW1lIiwiYWRkUmVzcG9uc2UiLCJhZGROb3RlIiwib2JqZWN0SWQiLCJwdXQiLCJyZW5hbWVSZXNwb25zZSIsInJlbmFtZU5vdGUiLCJjbG9uZVJlc3BvbnNlIiwiY2xvbmVOb3RlIiwiZGVsZXRlIiwibm90ZUxpc3QiLCJkZWxSZXNwb25zZSIsImRlbGV0ZU5vdGUiLCJ2aXNJZHMiLCJhcnJheU9mIiwiYWRkU2FtcGxlTm90ZXNSZXNwb25zZSIsImFkZFNhbXBsZU5vdGVzIl0sIm1hcHBpbmdzIjoiOzs7Ozs7O0FBS0E7O0FBT0E7O0FBQ0E7O0FBQ0E7Ozs7QUFkQTtBQUNBO0FBQ0E7QUFDQTtBQWFPLFNBQVNBLGlCQUFULENBQTJCQyxNQUEzQixFQUE0QztBQUNqRDtBQUNBQSxFQUFBQSxNQUFNLENBQUNDLEdBQVAsQ0FDRTtBQUNFQyxJQUFBQSxJQUFJLEVBQUcsR0FBRUMsK0JBQXFCLEdBRGhDO0FBRUVDLElBQUFBLFFBQVEsRUFBRTtBQUZaLEdBREYsRUFLRSxPQUNFQyxPQURGLEVBRUVDLE9BRkYsRUFHRUMsUUFIRixLQUlrRTtBQUNoRSw0Q0FBbUIsV0FBbkIsRUFBZ0MsS0FBaEMsRUFBdUMsT0FBdkM7QUFDQSxVQUFNQyx5QkFBcUQsR0FBR0gsT0FBTyxDQUFDSSxvQkFBUixDQUE2QkMsbUJBQTdCLENBQWlEQyxRQUFqRCxDQUM1REwsT0FENEQsQ0FBOUQ7QUFHQSxRQUFJTSxhQUFhLEdBQUcsRUFBcEI7O0FBQ0EsUUFBSTtBQUNGQSxNQUFBQSxhQUFhLEdBQUcsTUFBTUMsb0JBQVFDLFNBQVIsQ0FBa0JOLHlCQUFsQixFQUE2Q08sdUJBQTdDLENBQXRCO0FBQ0EsYUFBT1IsUUFBUSxDQUFDUyxFQUFULENBQVk7QUFDakJDLFFBQUFBLElBQUksRUFBRTtBQUNKQyxVQUFBQSxJQUFJLEVBQUVOO0FBREY7QUFEVyxPQUFaLENBQVA7QUFLRCxLQVBELENBT0UsT0FBT08sS0FBUCxFQUFjO0FBQ2QsOENBQW1CLFdBQW5CLEVBQWdDLEtBQWhDLEVBQXVDQSxLQUF2QztBQUNBQyxNQUFBQSxPQUFPLENBQUNDLEdBQVIsQ0FBWSxXQUFaLEVBQXlCRixLQUF6QjtBQUNBLGFBQU9aLFFBQVEsQ0FBQ2UsTUFBVCxDQUFnQjtBQUNyQkMsUUFBQUEsVUFBVSxFQUFFSixLQUFLLENBQUNJLFVBQU4sSUFBb0IsR0FEWDtBQUVyQk4sUUFBQUEsSUFBSSxFQUFFRSxLQUFLLENBQUNLO0FBRlMsT0FBaEIsQ0FBUDtBQUlEO0FBQ0YsR0E5QkgsRUFGaUQsQ0FtQ2pEOztBQUNBeEIsRUFBQUEsTUFBTSxDQUFDQyxHQUFQLENBQ0U7QUFDRUMsSUFBQUEsSUFBSSxFQUFHLEdBQUVDLCtCQUFxQixnQkFEaEM7QUFFRUMsSUFBQUEsUUFBUSxFQUFFO0FBQ1JxQixNQUFBQSxNQUFNLEVBQUVDLHFCQUFPQyxNQUFQLENBQWM7QUFDcEJDLFFBQUFBLE1BQU0sRUFBRUYscUJBQU9HLE1BQVA7QUFEWSxPQUFkO0FBREE7QUFGWixHQURGLEVBU0UsT0FDRXhCLE9BREYsRUFFRUMsT0FGRixFQUdFQyxRQUhGLEtBSWtFO0FBQ2hFLDRDQUFtQixXQUFuQixFQUFnQyxLQUFoQyxFQUF1QyxPQUF2QztBQUNBLFVBQU1DLHlCQUFxRCxHQUFHSCxPQUFPLENBQUNJLG9CQUFSLENBQTZCQyxtQkFBN0IsQ0FBaURDLFFBQWpELENBQzVETCxPQUQ0RCxDQUE5RDs7QUFHQSxRQUFJO0FBQ0YsWUFBTXdCLFlBQVksR0FBRyxNQUFNakIsb0JBQVFrQixTQUFSLENBQ3pCdkIseUJBRHlCLEVBRXpCRixPQUFPLENBQUNtQixNQUFSLENBQWVHLE1BRlUsRUFHekJiLHVCQUh5QixDQUEzQjtBQUtBLGFBQU9SLFFBQVEsQ0FBQ1MsRUFBVCxDQUFZO0FBQ2pCQyxRQUFBQSxJQUFJLEVBQUVhO0FBRFcsT0FBWixDQUFQO0FBR0QsS0FURCxDQVNFLE9BQU9YLEtBQVAsRUFBYztBQUNkLDhDQUFtQixXQUFuQixFQUFnQyxLQUFoQyxFQUF1Q0EsS0FBdkM7QUFDQSxhQUFPWixRQUFRLENBQUNlLE1BQVQsQ0FBZ0I7QUFDckJDLFFBQUFBLFVBQVUsRUFBRUosS0FBSyxDQUFDSSxVQUFOLElBQW9CLEdBRFg7QUFFckJOLFFBQUFBLElBQUksRUFBRUUsS0FBSyxDQUFDSztBQUZTLE9BQWhCLENBQVA7QUFJRDtBQUNGLEdBbENILEVBcENpRCxDQXlFakQ7O0FBQ0F4QixFQUFBQSxNQUFNLENBQUNnQyxJQUFQLENBQ0U7QUFDRTlCLElBQUFBLElBQUksRUFBRyxHQUFFQywrQkFBcUIsT0FEaEM7QUFFRUMsSUFBQUEsUUFBUSxFQUFFO0FBQ1JhLE1BQUFBLElBQUksRUFBRVMscUJBQU9DLE1BQVAsQ0FBYztBQUNsQk0sUUFBQUEsSUFBSSxFQUFFUCxxQkFBT0csTUFBUDtBQURZLE9BQWQ7QUFERTtBQUZaLEdBREYsRUFTRSxPQUNFeEIsT0FERixFQUVFQyxPQUZGLEVBR0VDLFFBSEYsS0FJa0U7QUFDaEUsNENBQW1CLFdBQW5CLEVBQWdDLFFBQWhDLEVBQTBDLE9BQTFDO0FBQ0EsVUFBTUMseUJBQXFELEdBQUdILE9BQU8sQ0FBQ0ksb0JBQVIsQ0FBNkJDLG1CQUE3QixDQUFpREMsUUFBakQsQ0FDNURMLE9BRDRELENBQTlEOztBQUdBLFFBQUk7QUFDRixZQUFNNEIsV0FBVyxHQUFHLE1BQU1yQixvQkFBUXNCLE9BQVIsQ0FDeEIzQix5QkFEd0IsRUFFeEJGLE9BQU8sQ0FBQ1csSUFGZ0IsRUFHeEJGLHVCQUh3QixDQUExQjtBQUtBLGFBQU9SLFFBQVEsQ0FBQ1MsRUFBVCxDQUFZO0FBQ2pCQyxRQUFBQSxJQUFJLEVBQUVpQixXQUFXLENBQUNWLE9BQVosQ0FBb0JZO0FBRFQsT0FBWixDQUFQO0FBR0QsS0FURCxDQVNFLE9BQU9qQixLQUFQLEVBQWM7QUFDZCw4Q0FBbUIsV0FBbkIsRUFBZ0MsUUFBaEMsRUFBMENBLEtBQTFDO0FBQ0EsYUFBT1osUUFBUSxDQUFDZSxNQUFULENBQWdCO0FBQ3JCQyxRQUFBQSxVQUFVLEVBQUVKLEtBQUssQ0FBQ0ksVUFBTixJQUFvQixHQURYO0FBRXJCTixRQUFBQSxJQUFJLEVBQUVFLEtBQUssQ0FBQ0s7QUFGUyxPQUFoQixDQUFQO0FBSUQ7QUFDRixHQWxDSCxFQTFFaUQsQ0ErR2pEOztBQUNBeEIsRUFBQUEsTUFBTSxDQUFDcUMsR0FBUCxDQUNFO0FBQ0VuQyxJQUFBQSxJQUFJLEVBQUcsR0FBRUMsK0JBQXFCLGNBRGhDO0FBRUVDLElBQUFBLFFBQVEsRUFBRTtBQUNSYSxNQUFBQSxJQUFJLEVBQUVTLHFCQUFPQyxNQUFQLENBQWM7QUFDbEJNLFFBQUFBLElBQUksRUFBRVAscUJBQU9HLE1BQVAsRUFEWTtBQUVsQkQsUUFBQUEsTUFBTSxFQUFFRixxQkFBT0csTUFBUDtBQUZVLE9BQWQ7QUFERTtBQUZaLEdBREYsRUFVRSxPQUNFeEIsT0FERixFQUVFQyxPQUZGLEVBR0VDLFFBSEYsS0FJa0U7QUFDaEUsNENBQW1CLFdBQW5CLEVBQWdDLFFBQWhDLEVBQTBDLE9BQTFDO0FBQ0EsVUFBTUMseUJBQXFELEdBQUdILE9BQU8sQ0FBQ0ksb0JBQVIsQ0FBNkJDLG1CQUE3QixDQUFpREMsUUFBakQsQ0FDNURMLE9BRDRELENBQTlEOztBQUdBLFFBQUk7QUFDRixZQUFNZ0MsY0FBYyxHQUFHLE1BQU16QixvQkFBUTBCLFVBQVIsQ0FDM0IvQix5QkFEMkIsRUFFM0JGLE9BQU8sQ0FBQ1csSUFGbUIsRUFHM0JGLHVCQUgyQixDQUE3QjtBQUtBLGFBQU9SLFFBQVEsQ0FBQ1MsRUFBVCxDQUFZO0FBQ2pCQyxRQUFBQSxJQUFJLEVBQUVxQjtBQURXLE9BQVosQ0FBUDtBQUdELEtBVEQsQ0FTRSxPQUFPbkIsS0FBUCxFQUFjO0FBQ2QsOENBQW1CLFdBQW5CLEVBQWdDLFFBQWhDLEVBQTBDQSxLQUExQztBQUNBLGFBQU9aLFFBQVEsQ0FBQ2UsTUFBVCxDQUFnQjtBQUNyQkMsUUFBQUEsVUFBVSxFQUFFSixLQUFLLENBQUNJLFVBQU4sSUFBb0IsR0FEWDtBQUVyQk4sUUFBQUEsSUFBSSxFQUFFRSxLQUFLLENBQUNLO0FBRlMsT0FBaEIsQ0FBUDtBQUlEO0FBQ0YsR0FuQ0gsRUFoSGlELENBc0pqRDs7QUFDQXhCLEVBQUFBLE1BQU0sQ0FBQ2dDLElBQVAsQ0FDRTtBQUNFOUIsSUFBQUEsSUFBSSxFQUFHLEdBQUVDLCtCQUFxQixhQURoQztBQUVFQyxJQUFBQSxRQUFRLEVBQUU7QUFDUmEsTUFBQUEsSUFBSSxFQUFFUyxxQkFBT0MsTUFBUCxDQUFjO0FBQ2xCTSxRQUFBQSxJQUFJLEVBQUVQLHFCQUFPRyxNQUFQLEVBRFk7QUFFbEJELFFBQUFBLE1BQU0sRUFBRUYscUJBQU9HLE1BQVA7QUFGVSxPQUFkO0FBREU7QUFGWixHQURGLEVBVUUsT0FDRXhCLE9BREYsRUFFRUMsT0FGRixFQUdFQyxRQUhGLEtBSWtFO0FBQ2hFLDRDQUFtQixXQUFuQixFQUFnQyxRQUFoQyxFQUEwQyxPQUExQztBQUNBLFVBQU1DLHlCQUFxRCxHQUFHSCxPQUFPLENBQUNJLG9CQUFSLENBQTZCQyxtQkFBN0IsQ0FBaURDLFFBQWpELENBQzVETCxPQUQ0RCxDQUE5RDs7QUFHQSxRQUFJO0FBQ0YsWUFBTWtDLGFBQWEsR0FBRyxNQUFNM0Isb0JBQVE0QixTQUFSLENBQzFCakMseUJBRDBCLEVBRTFCRixPQUFPLENBQUNXLElBRmtCLEVBRzFCRix1QkFIMEIsQ0FBNUI7QUFLQSxhQUFPUixRQUFRLENBQUNTLEVBQVQsQ0FBWTtBQUNqQkMsUUFBQUEsSUFBSSxFQUFFdUI7QUFEVyxPQUFaLENBQVA7QUFHRCxLQVRELENBU0UsT0FBT3JCLEtBQVAsRUFBYztBQUNkLDhDQUFtQixXQUFuQixFQUFnQyxRQUFoQyxFQUEwQ0EsS0FBMUM7QUFDQSxhQUFPWixRQUFRLENBQUNlLE1BQVQsQ0FBZ0I7QUFDckJDLFFBQUFBLFVBQVUsRUFBRUosS0FBSyxDQUFDSSxVQUFOLElBQW9CLEdBRFg7QUFFckJOLFFBQUFBLElBQUksRUFBRUUsS0FBSyxDQUFDSztBQUZTLE9BQWhCLENBQVA7QUFJRDtBQUNGLEdBbkNILEVBdkppRCxDQTZMakQ7O0FBQ0F4QixFQUFBQSxNQUFNLENBQUMwQyxNQUFQLENBQ0U7QUFDRXhDLElBQUFBLElBQUksRUFBRyxHQUFFQywrQkFBcUIsa0JBRGhDO0FBRUVDLElBQUFBLFFBQVEsRUFBRTtBQUNScUIsTUFBQUEsTUFBTSxFQUFFQyxxQkFBT0MsTUFBUCxDQUFjO0FBQ3BCZ0IsUUFBQUEsUUFBUSxFQUFFakIscUJBQU9HLE1BQVA7QUFEVSxPQUFkO0FBREE7QUFGWixHQURGLEVBU0UsT0FDRXhCLE9BREYsRUFFRUMsT0FGRixFQUdFQyxRQUhGLEtBSWtFO0FBQ2hFLDRDQUFtQixXQUFuQixFQUFnQyxRQUFoQyxFQUEwQyxPQUExQztBQUNBLFVBQU1DLHlCQUFxRCxHQUFHSCxPQUFPLENBQUNJLG9CQUFSLENBQTZCQyxtQkFBN0IsQ0FBaURDLFFBQWpELENBQzVETCxPQUQ0RCxDQUE5RDs7QUFHQSxRQUFJO0FBQ0YsWUFBTXNDLFdBQVcsR0FBRyxNQUFNL0Isb0JBQVFnQyxVQUFSLENBQ3hCckMseUJBRHdCLEVBRXhCRixPQUFPLENBQUNtQixNQUFSLENBQWVrQixRQUZTLEVBR3hCNUIsdUJBSHdCLENBQTFCO0FBS0EsYUFBT1IsUUFBUSxDQUFDUyxFQUFULENBQVk7QUFDakJDLFFBQUFBLElBQUksRUFBRTJCO0FBRFcsT0FBWixDQUFQO0FBR0QsS0FURCxDQVNFLE9BQU96QixLQUFQLEVBQWM7QUFDZCw4Q0FBbUIsV0FBbkIsRUFBZ0MsUUFBaEMsRUFBMENBLEtBQTFDO0FBQ0EsYUFBT1osUUFBUSxDQUFDZSxNQUFULENBQWdCO0FBQ3JCQyxRQUFBQSxVQUFVLEVBQUVKLEtBQUssQ0FBQ0ksVUFBTixJQUFvQixHQURYO0FBRXJCTixRQUFBQSxJQUFJLEVBQUVFLEtBQUssQ0FBQ0s7QUFGUyxPQUFoQixDQUFQO0FBSUQ7QUFDRixHQWxDSCxFQTlMaUQsQ0FtT2pEOztBQUNBeEIsRUFBQUEsTUFBTSxDQUFDZ0MsSUFBUCxDQUNFO0FBQ0U5QixJQUFBQSxJQUFJLEVBQUcsR0FBRUMsK0JBQXFCLDBCQURoQztBQUVFQyxJQUFBQSxRQUFRLEVBQUU7QUFDUmEsTUFBQUEsSUFBSSxFQUFFUyxxQkFBT0MsTUFBUCxDQUFjO0FBQ2xCbUIsUUFBQUEsTUFBTSxFQUFFcEIscUJBQU9xQixPQUFQLENBQWVyQixxQkFBT0csTUFBUCxFQUFmO0FBRFUsT0FBZDtBQURFO0FBRlosR0FERixFQVNFLE9BQ0V4QixPQURGLEVBRUVDLE9BRkYsRUFHRUMsUUFIRixLQUlrRTtBQUNoRSw0Q0FBbUIsV0FBbkIsRUFBZ0MsYUFBaEMsRUFBK0MsT0FBL0M7QUFDQSxVQUFNQyx5QkFBcUQsR0FBR0gsT0FBTyxDQUFDSSxvQkFBUixDQUE2QkMsbUJBQTdCLENBQWlEQyxRQUFqRCxDQUM1REwsT0FENEQsQ0FBOUQ7O0FBR0EsUUFBSTtBQUNGLFlBQU0wQyxzQkFBc0IsR0FBRyxNQUFNbkMsb0JBQVFvQyxjQUFSLENBQ25DekMseUJBRG1DLEVBRW5DRixPQUFPLENBQUNXLElBQVIsQ0FBYTZCLE1BRnNCLEVBR25DL0IsdUJBSG1DLENBQXJDO0FBS0EsYUFBT1IsUUFBUSxDQUFDUyxFQUFULENBQVk7QUFDakJDLFFBQUFBLElBQUksRUFBRStCO0FBRFcsT0FBWixDQUFQO0FBR0QsS0FURCxDQVNFLE9BQU83QixLQUFQLEVBQWM7QUFDZCw4Q0FBbUIsV0FBbkIsRUFBZ0MsYUFBaEMsRUFBK0NBLEtBQS9DO0FBQ0EsYUFBT1osUUFBUSxDQUFDZSxNQUFULENBQWdCO0FBQ3JCQyxRQUFBQSxVQUFVLEVBQUVKLEtBQUssQ0FBQ0ksVUFBTixJQUFvQixHQURYO0FBRXJCTixRQUFBQSxJQUFJLEVBQUVFLEtBQUssQ0FBQ0s7QUFGUyxPQUFoQixDQUFQO0FBSUQ7QUFDRixHQWxDSDtBQW9DRCIsInNvdXJjZXNDb250ZW50IjpbIi8qXG4gKiBDb3B5cmlnaHQgT3BlblNlYXJjaCBDb250cmlidXRvcnNcbiAqIFNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBBcGFjaGUtMi4wXG4gKi9cblxuaW1wb3J0IHsgc2NoZW1hIH0gZnJvbSAnQG9zZC9jb25maWctc2NoZW1hJztcbmltcG9ydCB7XG4gIElMZWdhY3lTY29wZWRDbHVzdGVyQ2xpZW50LFxuICBJT3BlblNlYXJjaERhc2hib2FyZHNSZXNwb25zZSxcbiAgSVJvdXRlcixcbiAgUmVzcG9uc2VFcnJvcixcbn0gZnJvbSAnLi4vLi4vLi4vLi4vLi4vc3JjL2NvcmUvc2VydmVyJztcbmltcG9ydCB7IE5PVEVCT09LU19BUElfUFJFRklYLCB3cmVja09wdGlvbnMgfSBmcm9tICcuLi8uLi8uLi9jb21tb24vY29uc3RhbnRzL25vdGVib29rcyc7XG5pbXBvcnQgQkFDS0VORCBmcm9tICcuLi8uLi9hZGFwdG9ycy9ub3RlYm9va3MnO1xuaW1wb3J0IHsgYWRkUmVxdWVzdFRvTWV0cmljIH0gZnJvbSAnLi4vLi4vY29tbW9uL21ldHJpY3MvbWV0cmljc19oZWxwZXInO1xuXG5leHBvcnQgZnVuY3Rpb24gcmVnaXN0ZXJOb3RlUm91dGUocm91dGVyOiBJUm91dGVyKSB7XG4gIC8vIEZldGNoIGFsbCB0aGUgbm90ZWJvb2tzIGF2YWlsYWJsZVxuICByb3V0ZXIuZ2V0KFxuICAgIHtcbiAgICAgIHBhdGg6IGAke05PVEVCT09LU19BUElfUFJFRklYfS9gLFxuICAgICAgdmFsaWRhdGU6IHt9LFxuICAgIH0sXG4gICAgYXN5bmMgKFxuICAgICAgY29udGV4dCxcbiAgICAgIHJlcXVlc3QsXG4gICAgICByZXNwb25zZVxuICAgICk6IFByb21pc2U8SU9wZW5TZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2U8YW55IHwgUmVzcG9uc2VFcnJvcj4+ID0+IHtcbiAgICAgIGFkZFJlcXVlc3RUb01ldHJpYygnbm90ZWJvb2tzJywgJ2dldCcsICdjb3VudCcpO1xuICAgICAgY29uc3Qgb3BlbnNlYXJjaE5vdGVib29rc0NsaWVudDogSUxlZ2FjeVNjb3BlZENsdXN0ZXJDbGllbnQgPSBjb250ZXh0Lm9ic2VydmFiaWxpdHlfcGx1Z2luLm9ic2VydmFiaWxpdHlDbGllbnQuYXNTY29wZWQoXG4gICAgICAgIHJlcXVlc3RcbiAgICAgICk7XG4gICAgICBsZXQgbm90ZWJvb2tzRGF0YSA9IFtdO1xuICAgICAgdHJ5IHtcbiAgICAgICAgbm90ZWJvb2tzRGF0YSA9IGF3YWl0IEJBQ0tFTkQudmlld05vdGVzKG9wZW5zZWFyY2hOb3RlYm9va3NDbGllbnQsIHdyZWNrT3B0aW9ucyk7XG4gICAgICAgIHJldHVybiByZXNwb25zZS5vayh7XG4gICAgICAgICAgYm9keToge1xuICAgICAgICAgICAgZGF0YTogbm90ZWJvb2tzRGF0YSxcbiAgICAgICAgICB9LFxuICAgICAgICB9KTtcbiAgICAgIH0gY2F0Y2ggKGVycm9yKSB7XG4gICAgICAgIGFkZFJlcXVlc3RUb01ldHJpYygnbm90ZWJvb2tzJywgJ2dldCcsIGVycm9yKTtcbiAgICAgICAgY29uc29sZS5sb2coJ05vdGVib29rOicsIGVycm9yKTtcbiAgICAgICAgcmV0dXJuIHJlc3BvbnNlLmN1c3RvbSh7XG4gICAgICAgICAgc3RhdHVzQ29kZTogZXJyb3Iuc3RhdHVzQ29kZSB8fCA1MDAsXG4gICAgICAgICAgYm9keTogZXJyb3IubWVzc2FnZSxcbiAgICAgICAgfSk7XG4gICAgICB9XG4gICAgfVxuICApO1xuXG4gIC8vIEdldCBhbGwgcGFyYWdyYXBocyBvZiBub3RlYm9va3NcbiAgcm91dGVyLmdldChcbiAgICB7XG4gICAgICBwYXRoOiBgJHtOT1RFQk9PS1NfQVBJX1BSRUZJWH0vbm90ZS97bm90ZUlkfWAsXG4gICAgICB2YWxpZGF0ZToge1xuICAgICAgICBwYXJhbXM6IHNjaGVtYS5vYmplY3Qoe1xuICAgICAgICAgIG5vdGVJZDogc2NoZW1hLnN0cmluZygpLFxuICAgICAgICB9KSxcbiAgICAgIH0sXG4gICAgfSxcbiAgICBhc3luYyAoXG4gICAgICBjb250ZXh0LFxuICAgICAgcmVxdWVzdCxcbiAgICAgIHJlc3BvbnNlXG4gICAgKTogUHJvbWlzZTxJT3BlblNlYXJjaERhc2hib2FyZHNSZXNwb25zZTxhbnkgfCBSZXNwb25zZUVycm9yPj4gPT4ge1xuICAgICAgYWRkUmVxdWVzdFRvTWV0cmljKCdub3RlYm9va3MnLCAnZ2V0JywgJ2NvdW50Jyk7XG4gICAgICBjb25zdCBvcGVuc2VhcmNoTm90ZWJvb2tzQ2xpZW50OiBJTGVnYWN5U2NvcGVkQ2x1c3RlckNsaWVudCA9IGNvbnRleHQub2JzZXJ2YWJpbGl0eV9wbHVnaW4ub2JzZXJ2YWJpbGl0eUNsaWVudC5hc1Njb3BlZChcbiAgICAgICAgcmVxdWVzdFxuICAgICAgKTtcbiAgICAgIHRyeSB7XG4gICAgICAgIGNvbnN0IG5vdGVib29raW5mbyA9IGF3YWl0IEJBQ0tFTkQuZmV0Y2hOb3RlKFxuICAgICAgICAgIG9wZW5zZWFyY2hOb3RlYm9va3NDbGllbnQsXG4gICAgICAgICAgcmVxdWVzdC5wYXJhbXMubm90ZUlkLFxuICAgICAgICAgIHdyZWNrT3B0aW9uc1xuICAgICAgICApO1xuICAgICAgICByZXR1cm4gcmVzcG9uc2Uub2soe1xuICAgICAgICAgIGJvZHk6IG5vdGVib29raW5mbyxcbiAgICAgICAgfSk7XG4gICAgICB9IGNhdGNoIChlcnJvcikge1xuICAgICAgICBhZGRSZXF1ZXN0VG9NZXRyaWMoJ25vdGVib29rcycsICdnZXQnLCBlcnJvcik7XG4gICAgICAgIHJldHVybiByZXNwb25zZS5jdXN0b20oe1xuICAgICAgICAgIHN0YXR1c0NvZGU6IGVycm9yLnN0YXR1c0NvZGUgfHwgNTAwLFxuICAgICAgICAgIGJvZHk6IGVycm9yLm1lc3NhZ2UsXG4gICAgICAgIH0pO1xuICAgICAgfVxuICAgIH1cbiAgKTtcblxuICAvLyBBZGQgYSBOb3RlYm9va1xuICByb3V0ZXIucG9zdChcbiAgICB7XG4gICAgICBwYXRoOiBgJHtOT1RFQk9PS1NfQVBJX1BSRUZJWH0vbm90ZWAsXG4gICAgICB2YWxpZGF0ZToge1xuICAgICAgICBib2R5OiBzY2hlbWEub2JqZWN0KHtcbiAgICAgICAgICBuYW1lOiBzY2hlbWEuc3RyaW5nKCksXG4gICAgICAgIH0pLFxuICAgICAgfSxcbiAgICB9LFxuICAgIGFzeW5jIChcbiAgICAgIGNvbnRleHQsXG4gICAgICByZXF1ZXN0LFxuICAgICAgcmVzcG9uc2VcbiAgICApOiBQcm9taXNlPElPcGVuU2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlPGFueSB8IFJlc3BvbnNlRXJyb3I+PiA9PiB7XG4gICAgICBhZGRSZXF1ZXN0VG9NZXRyaWMoJ25vdGVib29rcycsICdjcmVhdGUnLCAnY291bnQnKTtcbiAgICAgIGNvbnN0IG9wZW5zZWFyY2hOb3RlYm9va3NDbGllbnQ6IElMZWdhY3lTY29wZWRDbHVzdGVyQ2xpZW50ID0gY29udGV4dC5vYnNlcnZhYmlsaXR5X3BsdWdpbi5vYnNlcnZhYmlsaXR5Q2xpZW50LmFzU2NvcGVkKFxuICAgICAgICByZXF1ZXN0XG4gICAgICApO1xuICAgICAgdHJ5IHtcbiAgICAgICAgY29uc3QgYWRkUmVzcG9uc2UgPSBhd2FpdCBCQUNLRU5ELmFkZE5vdGUoXG4gICAgICAgICAgb3BlbnNlYXJjaE5vdGVib29rc0NsaWVudCxcbiAgICAgICAgICByZXF1ZXN0LmJvZHksXG4gICAgICAgICAgd3JlY2tPcHRpb25zXG4gICAgICAgICk7XG4gICAgICAgIHJldHVybiByZXNwb25zZS5vayh7XG4gICAgICAgICAgYm9keTogYWRkUmVzcG9uc2UubWVzc2FnZS5vYmplY3RJZCxcbiAgICAgICAgfSk7XG4gICAgICB9IGNhdGNoIChlcnJvcikge1xuICAgICAgICBhZGRSZXF1ZXN0VG9NZXRyaWMoJ25vdGVib29rcycsICdjcmVhdGUnLCBlcnJvcik7XG4gICAgICAgIHJldHVybiByZXNwb25zZS5jdXN0b20oe1xuICAgICAgICAgIHN0YXR1c0NvZGU6IGVycm9yLnN0YXR1c0NvZGUgfHwgNTAwLFxuICAgICAgICAgIGJvZHk6IGVycm9yLm1lc3NhZ2UsXG4gICAgICAgIH0pO1xuICAgICAgfVxuICAgIH1cbiAgKTtcblxuICAvLyBSZW5hbWUgYSBub3RlYm9va1xuICByb3V0ZXIucHV0KFxuICAgIHtcbiAgICAgIHBhdGg6IGAke05PVEVCT09LU19BUElfUFJFRklYfS9ub3RlL3JlbmFtZWAsXG4gICAgICB2YWxpZGF0ZToge1xuICAgICAgICBib2R5OiBzY2hlbWEub2JqZWN0KHtcbiAgICAgICAgICBuYW1lOiBzY2hlbWEuc3RyaW5nKCksXG4gICAgICAgICAgbm90ZUlkOiBzY2hlbWEuc3RyaW5nKCksXG4gICAgICAgIH0pLFxuICAgICAgfSxcbiAgICB9LFxuICAgIGFzeW5jIChcbiAgICAgIGNvbnRleHQsXG4gICAgICByZXF1ZXN0LFxuICAgICAgcmVzcG9uc2VcbiAgICApOiBQcm9taXNlPElPcGVuU2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlPGFueSB8IFJlc3BvbnNlRXJyb3I+PiA9PiB7XG4gICAgICBhZGRSZXF1ZXN0VG9NZXRyaWMoJ25vdGVib29rcycsICd1cGRhdGUnLCAnY291bnQnKTtcbiAgICAgIGNvbnN0IG9wZW5zZWFyY2hOb3RlYm9va3NDbGllbnQ6IElMZWdhY3lTY29wZWRDbHVzdGVyQ2xpZW50ID0gY29udGV4dC5vYnNlcnZhYmlsaXR5X3BsdWdpbi5vYnNlcnZhYmlsaXR5Q2xpZW50LmFzU2NvcGVkKFxuICAgICAgICByZXF1ZXN0XG4gICAgICApO1xuICAgICAgdHJ5IHtcbiAgICAgICAgY29uc3QgcmVuYW1lUmVzcG9uc2UgPSBhd2FpdCBCQUNLRU5ELnJlbmFtZU5vdGUoXG4gICAgICAgICAgb3BlbnNlYXJjaE5vdGVib29rc0NsaWVudCxcbiAgICAgICAgICByZXF1ZXN0LmJvZHksXG4gICAgICAgICAgd3JlY2tPcHRpb25zXG4gICAgICAgICk7XG4gICAgICAgIHJldHVybiByZXNwb25zZS5vayh7XG4gICAgICAgICAgYm9keTogcmVuYW1lUmVzcG9uc2UsXG4gICAgICAgIH0pO1xuICAgICAgfSBjYXRjaCAoZXJyb3IpIHtcbiAgICAgICAgYWRkUmVxdWVzdFRvTWV0cmljKCdub3RlYm9va3MnLCAndXBkYXRlJywgZXJyb3IpO1xuICAgICAgICByZXR1cm4gcmVzcG9uc2UuY3VzdG9tKHtcbiAgICAgICAgICBzdGF0dXNDb2RlOiBlcnJvci5zdGF0dXNDb2RlIHx8IDUwMCxcbiAgICAgICAgICBib2R5OiBlcnJvci5tZXNzYWdlLFxuICAgICAgICB9KTtcbiAgICAgIH1cbiAgICB9XG4gICk7XG5cbiAgLy8gQ2xvbmUgYSBub3RlYm9va1xuICByb3V0ZXIucG9zdChcbiAgICB7XG4gICAgICBwYXRoOiBgJHtOT1RFQk9PS1NfQVBJX1BSRUZJWH0vbm90ZS9jbG9uZWAsXG4gICAgICB2YWxpZGF0ZToge1xuICAgICAgICBib2R5OiBzY2hlbWEub2JqZWN0KHtcbiAgICAgICAgICBuYW1lOiBzY2hlbWEuc3RyaW5nKCksXG4gICAgICAgICAgbm90ZUlkOiBzY2hlbWEuc3RyaW5nKCksXG4gICAgICAgIH0pLFxuICAgICAgfSxcbiAgICB9LFxuICAgIGFzeW5jIChcbiAgICAgIGNvbnRleHQsXG4gICAgICByZXF1ZXN0LFxuICAgICAgcmVzcG9uc2VcbiAgICApOiBQcm9taXNlPElPcGVuU2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlPGFueSB8IFJlc3BvbnNlRXJyb3I+PiA9PiB7XG4gICAgICBhZGRSZXF1ZXN0VG9NZXRyaWMoJ25vdGVib29rcycsICdjcmVhdGUnLCAnY291bnQnKTtcbiAgICAgIGNvbnN0IG9wZW5zZWFyY2hOb3RlYm9va3NDbGllbnQ6IElMZWdhY3lTY29wZWRDbHVzdGVyQ2xpZW50ID0gY29udGV4dC5vYnNlcnZhYmlsaXR5X3BsdWdpbi5vYnNlcnZhYmlsaXR5Q2xpZW50LmFzU2NvcGVkKFxuICAgICAgICByZXF1ZXN0XG4gICAgICApO1xuICAgICAgdHJ5IHtcbiAgICAgICAgY29uc3QgY2xvbmVSZXNwb25zZSA9IGF3YWl0IEJBQ0tFTkQuY2xvbmVOb3RlKFxuICAgICAgICAgIG9wZW5zZWFyY2hOb3RlYm9va3NDbGllbnQsXG4gICAgICAgICAgcmVxdWVzdC5ib2R5LFxuICAgICAgICAgIHdyZWNrT3B0aW9uc1xuICAgICAgICApO1xuICAgICAgICByZXR1cm4gcmVzcG9uc2Uub2soe1xuICAgICAgICAgIGJvZHk6IGNsb25lUmVzcG9uc2UsXG4gICAgICAgIH0pO1xuICAgICAgfSBjYXRjaCAoZXJyb3IpIHtcbiAgICAgICAgYWRkUmVxdWVzdFRvTWV0cmljKCdub3RlYm9va3MnLCAnY3JlYXRlJywgZXJyb3IpO1xuICAgICAgICByZXR1cm4gcmVzcG9uc2UuY3VzdG9tKHtcbiAgICAgICAgICBzdGF0dXNDb2RlOiBlcnJvci5zdGF0dXNDb2RlIHx8IDUwMCxcbiAgICAgICAgICBib2R5OiBlcnJvci5tZXNzYWdlLFxuICAgICAgICB9KTtcbiAgICAgIH1cbiAgICB9XG4gICk7XG5cbiAgLy8gRGVsZXRlIG5vdGVib29rc1xuICByb3V0ZXIuZGVsZXRlKFxuICAgIHtcbiAgICAgIHBhdGg6IGAke05PVEVCT09LU19BUElfUFJFRklYfS9ub3RlL3tub3RlTGlzdH1gLFxuICAgICAgdmFsaWRhdGU6IHtcbiAgICAgICAgcGFyYW1zOiBzY2hlbWEub2JqZWN0KHtcbiAgICAgICAgICBub3RlTGlzdDogc2NoZW1hLnN0cmluZygpLFxuICAgICAgICB9KSxcbiAgICAgIH0sXG4gICAgfSxcbiAgICBhc3luYyAoXG4gICAgICBjb250ZXh0LFxuICAgICAgcmVxdWVzdCxcbiAgICAgIHJlc3BvbnNlXG4gICAgKTogUHJvbWlzZTxJT3BlblNlYXJjaERhc2hib2FyZHNSZXNwb25zZTxhbnkgfCBSZXNwb25zZUVycm9yPj4gPT4ge1xuICAgICAgYWRkUmVxdWVzdFRvTWV0cmljKCdub3RlYm9va3MnLCAnZGVsZXRlJywgJ2NvdW50Jyk7XG4gICAgICBjb25zdCBvcGVuc2VhcmNoTm90ZWJvb2tzQ2xpZW50OiBJTGVnYWN5U2NvcGVkQ2x1c3RlckNsaWVudCA9IGNvbnRleHQub2JzZXJ2YWJpbGl0eV9wbHVnaW4ub2JzZXJ2YWJpbGl0eUNsaWVudC5hc1Njb3BlZChcbiAgICAgICAgcmVxdWVzdFxuICAgICAgKTtcbiAgICAgIHRyeSB7XG4gICAgICAgIGNvbnN0IGRlbFJlc3BvbnNlID0gYXdhaXQgQkFDS0VORC5kZWxldGVOb3RlKFxuICAgICAgICAgIG9wZW5zZWFyY2hOb3RlYm9va3NDbGllbnQsXG4gICAgICAgICAgcmVxdWVzdC5wYXJhbXMubm90ZUxpc3QsXG4gICAgICAgICAgd3JlY2tPcHRpb25zXG4gICAgICAgICk7XG4gICAgICAgIHJldHVybiByZXNwb25zZS5vayh7XG4gICAgICAgICAgYm9keTogZGVsUmVzcG9uc2UsXG4gICAgICAgIH0pO1xuICAgICAgfSBjYXRjaCAoZXJyb3IpIHtcbiAgICAgICAgYWRkUmVxdWVzdFRvTWV0cmljKCdub3RlYm9va3MnLCAnZGVsZXRlJywgZXJyb3IpO1xuICAgICAgICByZXR1cm4gcmVzcG9uc2UuY3VzdG9tKHtcbiAgICAgICAgICBzdGF0dXNDb2RlOiBlcnJvci5zdGF0dXNDb2RlIHx8IDUwMCxcbiAgICAgICAgICBib2R5OiBlcnJvci5tZXNzYWdlLFxuICAgICAgICB9KTtcbiAgICAgIH1cbiAgICB9XG4gICk7XG5cbiAgLy8gQWRkIHNhbXBsZSBub3RlYm9va3NcbiAgcm91dGVyLnBvc3QoXG4gICAge1xuICAgICAgcGF0aDogYCR7Tk9URUJPT0tTX0FQSV9QUkVGSVh9L25vdGUvYWRkU2FtcGxlTm90ZWJvb2tzYCxcbiAgICAgIHZhbGlkYXRlOiB7XG4gICAgICAgIGJvZHk6IHNjaGVtYS5vYmplY3Qoe1xuICAgICAgICAgIHZpc0lkczogc2NoZW1hLmFycmF5T2Yoc2NoZW1hLnN0cmluZygpKSxcbiAgICAgICAgfSksXG4gICAgICB9LFxuICAgIH0sXG4gICAgYXN5bmMgKFxuICAgICAgY29udGV4dCxcbiAgICAgIHJlcXVlc3QsXG4gICAgICByZXNwb25zZVxuICAgICk6IFByb21pc2U8SU9wZW5TZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2U8YW55IHwgUmVzcG9uc2VFcnJvcj4+ID0+IHtcbiAgICAgIGFkZFJlcXVlc3RUb01ldHJpYygnbm90ZWJvb2tzJywgJ2FkZF9zYW1wbGVzJywgJ2NvdW50Jyk7XG4gICAgICBjb25zdCBvcGVuc2VhcmNoTm90ZWJvb2tzQ2xpZW50OiBJTGVnYWN5U2NvcGVkQ2x1c3RlckNsaWVudCA9IGNvbnRleHQub2JzZXJ2YWJpbGl0eV9wbHVnaW4ub2JzZXJ2YWJpbGl0eUNsaWVudC5hc1Njb3BlZChcbiAgICAgICAgcmVxdWVzdFxuICAgICAgKTtcbiAgICAgIHRyeSB7XG4gICAgICAgIGNvbnN0IGFkZFNhbXBsZU5vdGVzUmVzcG9uc2UgPSBhd2FpdCBCQUNLRU5ELmFkZFNhbXBsZU5vdGVzKFxuICAgICAgICAgIG9wZW5zZWFyY2hOb3RlYm9va3NDbGllbnQsXG4gICAgICAgICAgcmVxdWVzdC5ib2R5LnZpc0lkcyxcbiAgICAgICAgICB3cmVja09wdGlvbnNcbiAgICAgICAgKTtcbiAgICAgICAgcmV0dXJuIHJlc3BvbnNlLm9rKHtcbiAgICAgICAgICBib2R5OiBhZGRTYW1wbGVOb3Rlc1Jlc3BvbnNlLFxuICAgICAgICB9KTtcbiAgICAgIH0gY2F0Y2ggKGVycm9yKSB7XG4gICAgICAgIGFkZFJlcXVlc3RUb01ldHJpYygnbm90ZWJvb2tzJywgJ2FkZF9zYW1wbGVzJywgZXJyb3IpO1xuICAgICAgICByZXR1cm4gcmVzcG9uc2UuY3VzdG9tKHtcbiAgICAgICAgICBzdGF0dXNDb2RlOiBlcnJvci5zdGF0dXNDb2RlIHx8IDUwMCxcbiAgICAgICAgICBib2R5OiBlcnJvci5tZXNzYWdlLFxuICAgICAgICB9KTtcbiAgICAgIH1cbiAgICB9XG4gICk7XG59XG4iXX0=