"use strict";
/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
Object.defineProperty(exports, "__esModule", { value: true });
exports.createLanguageService = exports.githubSlugifier = exports.LogLevel = exports.RenameNotSupportedAtLocationError = exports.IncludeWorkspaceHeaderCompletions = exports.DiagnosticLevel = exports.DiagnosticCode = exports.PreferredMdPathExtensionStyle = void 0;
const config_1 = require("./config");
const extractLinkDef_1 = require("./languageFeatures/codeActions/extractLinkDef");
const removeLinkDefinition_1 = require("./languageFeatures/codeActions/removeLinkDefinition");
const definitions_1 = require("./languageFeatures/definitions");
const diagnostics_1 = require("./languageFeatures/diagnostics");
const documentHighlights_1 = require("./languageFeatures/documentHighlights");
const documentLinks_1 = require("./languageFeatures/documentLinks");
const documentSymbols_1 = require("./languageFeatures/documentSymbols");
const fileRename_1 = require("./languageFeatures/fileRename");
const folding_1 = require("./languageFeatures/folding");
const organizeLinkDefs_1 = require("./languageFeatures/organizeLinkDefs");
const pathCompletions_1 = require("./languageFeatures/pathCompletions");
const references_1 = require("./languageFeatures/references");
const rename_1 = require("./languageFeatures/rename");
const smartSelect_1 = require("./languageFeatures/smartSelect");
const workspaceSymbols_1 = require("./languageFeatures/workspaceSymbols");
const tableOfContents_1 = require("./tableOfContents");
const workspace_1 = require("./workspace");
var config_2 = require("./config");
Object.defineProperty(exports, "PreferredMdPathExtensionStyle", { enumerable: true, get: function () { return config_2.PreferredMdPathExtensionStyle; } });
var diagnostics_2 = require("./languageFeatures/diagnostics");
Object.defineProperty(exports, "DiagnosticCode", { enumerable: true, get: function () { return diagnostics_2.DiagnosticCode; } });
Object.defineProperty(exports, "DiagnosticLevel", { enumerable: true, get: function () { return diagnostics_2.DiagnosticLevel; } });
var pathCompletions_2 = require("./languageFeatures/pathCompletions");
Object.defineProperty(exports, "IncludeWorkspaceHeaderCompletions", { enumerable: true, get: function () { return pathCompletions_2.IncludeWorkspaceHeaderCompletions; } });
var rename_2 = require("./languageFeatures/rename");
Object.defineProperty(exports, "RenameNotSupportedAtLocationError", { enumerable: true, get: function () { return rename_2.RenameNotSupportedAtLocationError; } });
var logging_1 = require("./logging");
Object.defineProperty(exports, "LogLevel", { enumerable: true, get: function () { return logging_1.LogLevel; } });
var slugify_1 = require("./slugify");
Object.defineProperty(exports, "githubSlugifier", { enumerable: true, get: function () { return slugify_1.githubSlugifier; } });
/**
 * Create a new instance of the {@link IMdLanguageService language service}.
 */
function createLanguageService(init) {
    const config = (0, config_1.getLsConfiguration)(init);
    const logger = init.logger;
    const tocProvider = new tableOfContents_1.MdTableOfContentsProvider(init.parser, init.workspace, logger);
    const smartSelectProvider = new smartSelect_1.MdSelectionRangeProvider(init.parser, tocProvider, logger);
    const foldingProvider = new folding_1.MdFoldingProvider(init.parser, tocProvider, logger);
    const linkProvider = new documentLinks_1.MdLinkProvider(config, init.parser, init.workspace, tocProvider, logger);
    const pathCompletionProvider = new pathCompletions_1.MdPathCompletionProvider(config, init.workspace, init.parser, linkProvider, tocProvider);
    const linkCache = (0, documentLinks_1.createWorkspaceLinkCache)(init.parser, init.workspace);
    const referencesProvider = new references_1.MdReferencesProvider(config, init.parser, init.workspace, tocProvider, linkCache, logger);
    const definitionsProvider = new definitions_1.MdDefinitionProvider(config, init.workspace, tocProvider, linkCache);
    const renameProvider = new rename_1.MdRenameProvider(config, init.workspace, referencesProvider, init.parser.slugifier, logger);
    const fileRenameProvider = new fileRename_1.MdFileRenameProvider(config, init.workspace, linkCache, referencesProvider);
    const diagnosticsComputer = new diagnostics_1.DiagnosticComputer(config, init.workspace, linkProvider, tocProvider, logger);
    const docSymbolProvider = new documentSymbols_1.MdDocumentSymbolProvider(tocProvider, linkProvider, logger);
    const workspaceSymbolProvider = new workspaceSymbols_1.MdWorkspaceSymbolProvider(init.workspace, docSymbolProvider);
    const organizeLinkDefinitions = new organizeLinkDefs_1.MdOrganizeLinkDefinitionProvider(linkProvider);
    const documentHighlightProvider = new documentHighlights_1.MdDocumentHighlightProvider(config, tocProvider, linkProvider);
    const extractCodeActionProvider = new extractLinkDef_1.MdExtractLinkDefinitionCodeActionProvider(linkProvider);
    const removeLinkDefinitionActionProvider = new removeLinkDefinition_1.MdRemoveLinkDefinitionCodeActionProvider();
    return Object.freeze({
        dispose: () => {
            linkCache.dispose();
            tocProvider.dispose();
            workspaceSymbolProvider.dispose();
            linkProvider.dispose();
            referencesProvider.dispose();
        },
        getDocumentLinks: linkProvider.provideDocumentLinks.bind(linkProvider),
        resolveDocumentLink: linkProvider.resolveDocumentLink.bind(linkProvider),
        resolveLinkTarget: linkProvider.resolveLinkTarget.bind(linkProvider),
        getDocumentSymbols: docSymbolProvider.provideDocumentSymbols.bind(docSymbolProvider),
        getFoldingRanges: foldingProvider.provideFoldingRanges.bind(foldingProvider),
        getSelectionRanges: smartSelectProvider.provideSelectionRanges.bind(smartSelectProvider),
        getWorkspaceSymbols: workspaceSymbolProvider.provideWorkspaceSymbols.bind(workspaceSymbolProvider),
        getCompletionItems: pathCompletionProvider.provideCompletionItems.bind(pathCompletionProvider),
        getReferences: referencesProvider.provideReferences.bind(referencesProvider),
        getFileReferences: async (resource, token) => {
            return (await referencesProvider.getReferencesToFileInWorkspace(resource, token)).map(x => x.location);
        },
        getDefinition: definitionsProvider.provideDefinition.bind(definitionsProvider),
        organizeLinkDefinitions: organizeLinkDefinitions.getOrganizeLinkDefinitionEdits.bind(organizeLinkDefinitions),
        prepareRename: renameProvider.prepareRename.bind(renameProvider),
        getRenameEdit: renameProvider.provideRenameEdits.bind(renameProvider),
        getRenameFilesInWorkspaceEdit: fileRenameProvider.getRenameFilesInWorkspaceEdit.bind(fileRenameProvider),
        getCodeActions: async (doc, range, context, token) => {
            return (await Promise.all([
                extractCodeActionProvider.getActions(doc, range, context, token),
                Array.from(removeLinkDefinitionActionProvider.getActions(doc, range, context)),
            ])).flat();
        },
        getDocumentHighlights: (document, position, token) => {
            return documentHighlightProvider.getDocumentHighlights(document, position, token);
        },
        computeDiagnostics: async (doc, options, token) => {
            return (await diagnosticsComputer.compute(doc, options, token))?.diagnostics;
        },
        createPullDiagnosticsManager: () => {
            if (!(0, workspace_1.isWorkspaceWithFileWatching)(init.workspace)) {
                throw new Error(`Workspace does not support file watching. Diagnostics manager not supported`);
            }
            return new diagnostics_1.DiagnosticsManager(config, init.workspace, linkProvider, tocProvider, logger);
        }
    });
}
exports.createLanguageService = createLanguageService;
//# sourceMappingURL=index.js.map