/*! *****************************************************************************
Copyright (c) Microsoft Corporation. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License"); you may not use
this file except in compliance with the License. You may obtain a copy of the
License at http://www.apache.org/licenses/LICENSE-2.0

THIS CODE IS PROVIDED ON AN *AS IS* BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
KIND, EITHER EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION ANY IMPLIED
WARRANTIES OR CONDITIONS OF TITLE, FITNESS FOR A PARTICULAR PURPOSE,
MERCHANTABLITY OR NON-INFRINGEMENT.

See the Apache Version 2.0 License for specific language governing permissions
and limitations under the License.
***************************************************************************** */


"use strict";

// src/compiler/corePublic.ts
var versionMajorMinor = "5.7";
var version = "5.7.3";

// src/compiler/core.ts
var emptyArray = [];
var emptyMap = /* @__PURE__ */ new Map();
function length(array) {
  return array !== void 0 ? array.length : 0;
}
function forEach(array, callback) {
  if (array !== void 0) {
    for (let i = 0; i < array.length; i++) {
      const result = callback(array[i], i);
      if (result) {
        return result;
      }
    }
  }
  return void 0;
}
function firstDefined(array, callback) {
  if (array === void 0) {
    return void 0;
  }
  for (let i = 0; i < array.length; i++) {
    const result = callback(array[i], i);
    if (result !== void 0) {
      return result;
    }
  }
  return void 0;
}
function firstDefinedIterator(iter, callback) {
  for (const value of iter) {
    const result = callback(value);
    if (result !== void 0) {
      return result;
    }
  }
  return void 0;
}
function reduceLeftIterator(iterator, f, initial) {
  let result = initial;
  if (iterator) {
    let pos = 0;
    for (const value of iterator) {
      result = f(result, value, pos);
      pos++;
    }
  }
  return result;
}
function zipWith(arrayA, arrayB, callback) {
  const result = [];
  Debug.assertEqual(arrayA.length, arrayB.length);
  for (let i = 0; i < arrayA.length; i++) {
    result.push(callback(arrayA[i], arrayB[i], i));
  }
  return result;
}
function every(array, callback) {
  if (array !== void 0) {
    for (let i = 0; i < array.length; i++) {
      if (!callback(array[i], i)) {
        return false;
      }
    }
  }
  return true;
}
function find(array, predicate, startIndex) {
  if (array === void 0) return void 0;
  for (let i = startIndex ?? 0; i < array.length; i++) {
    const value = array[i];
    if (predicate(value, i)) {
      return value;
    }
  }
  return void 0;
}
function findLast(array, predicate, startIndex) {
  if (array === void 0) return void 0;
  for (let i = startIndex ?? array.length - 1; i >= 0; i--) {
    const value = array[i];
    if (predicate(value, i)) {
      return value;
    }
  }
  return void 0;
}
function findIndex(array, predicate, startIndex) {
  if (array === void 0) return -1;
  for (let i = startIndex ?? 0; i < array.length; i++) {
    if (predicate(array[i], i)) {
      return i;
    }
  }
  return -1;
}
function findLastIndex(array, predicate, startIndex) {
  if (array === void 0) return -1;
  for (let i = startIndex ?? array.length - 1; i >= 0; i--) {
    if (predicate(array[i], i)) {
      return i;
    }
  }
  return -1;
}
function contains(array, value, equalityComparer = equateValues) {
  if (array !== void 0) {
    for (let i = 0; i < array.length; i++) {
      if (equalityComparer(array[i], value)) {
        return true;
      }
    }
  }
  return false;
}
function indexOfAnyCharCode(text, charCodes, start) {
  for (let i = start ?? 0; i < text.length; i++) {
    if (contains(charCodes, text.charCodeAt(i))) {
      return i;
    }
  }
  return -1;
}
function countWhere(array, predicate) {
  let count = 0;
  if (array !== void 0) {
    for (let i = 0; i < array.length; i++) {
      const v = array[i];
      if (predicate(v, i)) {
        count++;
      }
    }
  }
  return count;
}
function filter(array, f) {
  if (array !== void 0) {
    const len = array.length;
    let i = 0;
    while (i < len && f(array[i])) i++;
    if (i < len) {
      const result = array.slice(0, i);
      i++;
      while (i < len) {
        const item = array[i];
        if (f(item)) {
          result.push(item);
        }
        i++;
      }
      return result;
    }
  }
  return array;
}
function filterMutate(array, f) {
  let outIndex = 0;
  for (let i = 0; i < array.length; i++) {
    if (f(array[i], i, array)) {
      array[outIndex] = array[i];
      outIndex++;
    }
  }
  array.length = outIndex;
}
function clear(array) {
  array.length = 0;
}
function map(array, f) {
  let result;
  if (array !== void 0) {
    result = [];
    for (let i = 0; i < array.length; i++) {
      result.push(f(array[i], i));
    }
  }
  return result;
}
function* mapIterator(iter, mapFn) {
  for (const x of iter) {
    yield mapFn(x);
  }
}
function sameMap(array, f) {
  if (array !== void 0) {
    for (let i = 0; i < array.length; i++) {
      const item = array[i];
      const mapped = f(item, i);
      if (item !== mapped) {
        const result = array.slice(0, i);
        result.push(mapped);
        for (i++; i < array.length; i++) {
          result.push(f(array[i], i));
        }
        return result;
      }
    }
  }
  return array;
}
function flatten(array) {
  const result = [];
  for (let i = 0; i < array.length; i++) {
    const v = array[i];
    if (v) {
      if (isArray(v)) {
        addRange(result, v);
      } else {
        result.push(v);
      }
    }
  }
  return result;
}
function flatMap(array, mapfn) {
  let result;
  if (array !== void 0) {
    for (let i = 0; i < array.length; i++) {
      const v = mapfn(array[i], i);
      if (v) {
        if (isArray(v)) {
          result = addRange(result, v);
        } else {
          result = append(result, v);
        }
      }
    }
  }
  return result ?? emptyArray;
}
function flatMapToMutable(array, mapfn) {
  const result = [];
  if (array !== void 0) {
    for (let i = 0; i < array.length; i++) {
      const v = mapfn(array[i], i);
      if (v) {
        if (isArray(v)) {
          addRange(result, v);
        } else {
          result.push(v);
        }
      }
    }
  }
  return result;
}
function sameFlatMap(array, mapfn) {
  let result;
  if (array !== void 0) {
    for (let i = 0; i < array.length; i++) {
      const item = array[i];
      const mapped = mapfn(item, i);
      if (result || item !== mapped || isArray(mapped)) {
        if (!result) {
          result = array.slice(0, i);
        }
        if (isArray(mapped)) {
          addRange(result, mapped);
        } else {
          result.push(mapped);
        }
      }
    }
  }
  return result ?? array;
}
function mapDefined(array, mapFn) {
  const result = [];
  if (array !== void 0) {
    for (let i = 0; i < array.length; i++) {
      const mapped = mapFn(array[i], i);
      if (mapped !== void 0) {
        result.push(mapped);
      }
    }
  }
  return result;
}
function* mapDefinedIterator(iter, mapFn) {
  for (const x of iter) {
    const value = mapFn(x);
    if (value !== void 0) {
      yield value;
    }
  }
}
function getOrUpdate(map2, key, callback) {
  if (map2.has(key)) {
    return map2.get(key);
  }
  const value = callback();
  map2.set(key, value);
  return value;
}
function tryAddToSet(set, value) {
  if (!set.has(value)) {
    set.add(value);
    return true;
  }
  return false;
}
function spanMap(array, keyfn, mapfn) {
  let result;
  if (array !== void 0) {
    result = [];
    const len = array.length;
    let previousKey;
    let key;
    let start = 0;
    let pos = 0;
    while (start < len) {
      while (pos < len) {
        const value = array[pos];
        key = keyfn(value, pos);
        if (pos === 0) {
          previousKey = key;
        } else if (key !== previousKey) {
          break;
        }
        pos++;
      }
      if (start < pos) {
        const v = mapfn(array.slice(start, pos), previousKey, start, pos);
        if (v) {
          result.push(v);
        }
        start = pos;
      }
      previousKey = key;
      pos++;
    }
  }
  return result;
}
function some(array, predicate) {
  if (array !== void 0) {
    if (predicate !== void 0) {
      for (let i = 0; i < array.length; i++) {
        if (predicate(array[i])) {
          return true;
        }
      }
    } else {
      return array.length > 0;
    }
  }
  return false;
}
function getRangesWhere(arr, pred, cb) {
  let start;
  for (let i = 0; i < arr.length; i++) {
    if (pred(arr[i])) {
      start = start === void 0 ? i : start;
    } else {
      if (start !== void 0) {
        cb(start, i);
        start = void 0;
      }
    }
  }
  if (start !== void 0) cb(start, arr.length);
}
function concatenate(array1, array2) {
  if (array2 === void 0 || array2.length === 0) return array1;
  if (array1 === void 0 || array1.length === 0) return array2;
  return [...array1, ...array2];
}
function selectIndex(_, i) {
  return i;
}
function indicesOf(array) {
  return array.map(selectIndex);
}
function deduplicateRelational(array, equalityComparer, comparer) {
  const indices = indicesOf(array);
  stableSortIndices(array, indices, comparer);
  let last2 = array[indices[0]];
  const deduplicated = [indices[0]];
  for (let i = 1; i < indices.length; i++) {
    const index = indices[i];
    const item = array[index];
    if (!equalityComparer(last2, item)) {
      deduplicated.push(index);
      last2 = item;
    }
  }
  deduplicated.sort();
  return deduplicated.map((i) => array[i]);
}
function deduplicateEquality(array, equalityComparer) {
  const result = [];
  for (let i = 0; i < array.length; i++) {
    pushIfUnique(result, array[i], equalityComparer);
  }
  return result;
}
function deduplicate(array, equalityComparer, comparer) {
  return array.length === 0 ? [] : array.length === 1 ? array.slice() : comparer ? deduplicateRelational(array, equalityComparer, comparer) : deduplicateEquality(array, equalityComparer);
}
function deduplicateSorted(array, comparer) {
  if (array.length === 0) return emptyArray;
  let last2 = array[0];
  const deduplicated = [last2];
  for (let i = 1; i < array.length; i++) {
    const next = array[i];
    switch (comparer(next, last2)) {
      // equality comparison
      case true:
      // relational comparison
      // falls through
      case 0 /* EqualTo */:
        continue;
      case -1 /* LessThan */:
        return Debug.fail("Array is unsorted.");
    }
    deduplicated.push(last2 = next);
  }
  return deduplicated;
}
function insertSorted(array, insert, compare, equalityComparer, allowDuplicates) {
  if (array.length === 0) {
    array.push(insert);
    return true;
  }
  const insertIndex = binarySearch(array, insert, identity, compare);
  if (insertIndex < 0) {
    if (equalityComparer && !allowDuplicates) {
      const idx = ~insertIndex;
      if (idx > 0 && equalityComparer(insert, array[idx - 1])) {
        return false;
      }
      if (idx < array.length && equalityComparer(insert, array[idx])) {
        array.splice(idx, 1, insert);
        return true;
      }
    }
    array.splice(~insertIndex, 0, insert);
    return true;
  }
  if (allowDuplicates) {
    array.splice(insertIndex, 0, insert);
    return true;
  }
  return false;
}
function sortAndDeduplicate(array, comparer, equalityComparer) {
  return deduplicateSorted(toSorted(array, comparer), equalityComparer ?? comparer ?? compareStringsCaseSensitive);
}
function arrayIsEqualTo(array1, array2, equalityComparer = equateValues) {
  if (array1 === void 0 || array2 === void 0) {
    return array1 === array2;
  }
  if (array1.length !== array2.length) {
    return false;
  }
  for (let i = 0; i < array1.length; i++) {
    if (!equalityComparer(array1[i], array2[i], i)) {
      return false;
    }
  }
  return true;
}
function compact(array) {
  let result;
  if (array !== void 0) {
    for (let i = 0; i < array.length; i++) {
      const v = array[i];
      if (result ?? !v) {
        result ?? (result = array.slice(0, i));
        if (v) {
          result.push(v);
        }
      }
    }
  }
  return result ?? array;
}
function relativeComplement(arrayA, arrayB, comparer) {
  if (!arrayB || !arrayA || arrayB.length === 0 || arrayA.length === 0) return arrayB;
  const result = [];
  loopB:
    for (let offsetA = 0, offsetB = 0; offsetB < arrayB.length; offsetB++) {
      if (offsetB > 0) {
        Debug.assertGreaterThanOrEqual(comparer(arrayB[offsetB], arrayB[offsetB - 1]), 0 /* EqualTo */);
      }
      loopA:
        for (const startA = offsetA; offsetA < arrayA.length; offsetA++) {
          if (offsetA > startA) {
            Debug.assertGreaterThanOrEqual(comparer(arrayA[offsetA], arrayA[offsetA - 1]), 0 /* EqualTo */);
          }
          switch (comparer(arrayB[offsetB], arrayA[offsetA])) {
            case -1 /* LessThan */:
              result.push(arrayB[offsetB]);
              continue loopB;
            case 0 /* EqualTo */:
              continue loopB;
            case 1 /* GreaterThan */:
              continue loopA;
          }
        }
    }
  return result;
}
function append(to, value) {
  if (value === void 0) return to;
  if (to === void 0) return [value];
  to.push(value);
  return to;
}
function toOffset(array, offset) {
  return offset < 0 ? array.length + offset : offset;
}
function addRange(to, from, start, end) {
  if (from === void 0 || from.length === 0) return to;
  if (to === void 0) return from.slice(start, end);
  start = start === void 0 ? 0 : toOffset(from, start);
  end = end === void 0 ? from.length : toOffset(from, end);
  for (let i = start; i < end && i < from.length; i++) {
    if (from[i] !== void 0) {
      to.push(from[i]);
    }
  }
  return to;
}
function pushIfUnique(array, toAdd, equalityComparer) {
  if (contains(array, toAdd, equalityComparer)) {
    return false;
  } else {
    array.push(toAdd);
    return true;
  }
}
function appendIfUnique(array, toAdd, equalityComparer) {
  if (array !== void 0) {
    pushIfUnique(array, toAdd, equalityComparer);
    return array;
  } else {
    return [toAdd];
  }
}
function stableSortIndices(array, indices, comparer) {
  indices.sort((x, y) => comparer(array[x], array[y]) || compareValues(x, y));
}
function toSorted(array, comparer) {
  return array.length === 0 ? emptyArray : array.slice().sort(comparer);
}
function rangeEquals(array1, array2, pos, end) {
  while (pos < end) {
    if (array1[pos] !== array2[pos]) {
      return false;
    }
    pos++;
  }
  return true;
}
var elementAt = !!Array.prototype.at ? (array, offset) => array == null ? void 0 : array.at(offset) : (array, offset) => {
  if (array !== void 0) {
    offset = toOffset(array, offset);
    if (offset < array.length) {
      return array[offset];
    }
  }
  return void 0;
};
function firstOrUndefined(array) {
  return array === void 0 || array.length === 0 ? void 0 : array[0];
}
function firstOrUndefinedIterator(iter) {
  if (iter !== void 0) {
    for (const value of iter) {
      return value;
    }
  }
  return void 0;
}
function first(array) {
  Debug.assert(array.length !== 0);
  return array[0];
}
function firstIterator(iter) {
  for (const value of iter) {
    return value;
  }
  Debug.fail("iterator is empty");
}
function lastOrUndefined(array) {
  return array === void 0 || array.length === 0 ? void 0 : array[array.length - 1];
}
function last(array) {
  Debug.assert(array.length !== 0);
  return array[array.length - 1];
}
function singleOrUndefined(array) {
  return array !== void 0 && array.length === 1 ? array[0] : void 0;
}
function singleOrMany(array) {
  return array !== void 0 && array.length === 1 ? array[0] : array;
}
function replaceElement(array, index, value) {
  const result = array.slice(0);
  result[index] = value;
  return result;
}
function binarySearch(array, value, keySelector, keyComparer, offset) {
  return binarySearchKey(array, keySelector(value), keySelector, keyComparer, offset);
}
function binarySearchKey(array, key, keySelector, keyComparer, offset) {
  if (!some(array)) {
    return -1;
  }
  let low = offset ?? 0;
  let high = array.length - 1;
  while (low <= high) {
    const middle = low + (high - low >> 1);
    const midKey = keySelector(array[middle], middle);
    switch (keyComparer(midKey, key)) {
      case -1 /* LessThan */:
        low = middle + 1;
        break;
      case 0 /* EqualTo */:
        return middle;
      case 1 /* GreaterThan */:
        high = middle - 1;
        break;
    }
  }
  return ~low;
}
function reduceLeft(array, f, initial, start, count) {
  if (array && array.length > 0) {
    const size = array.length;
    if (size > 0) {
      let pos = start === void 0 || start < 0 ? 0 : start;
      const end = count === void 0 || pos + count > size - 1 ? size - 1 : pos + count;
      let result;
      if (arguments.length <= 2) {
        result = array[pos];
        pos++;
      } else {
        result = initial;
      }
      while (pos <= end) {
        result = f(result, array[pos], pos);
        pos++;
      }
      return result;
    }
  }
  return initial;
}
var hasOwnProperty = Object.prototype.hasOwnProperty;
function hasProperty(map2, key) {
  return hasOwnProperty.call(map2, key);
}
function getOwnKeys(map2) {
  const keys = [];
  for (const key in map2) {
    if (hasOwnProperty.call(map2, key)) {
      keys.push(key);
    }
  }
  return keys;
}
function getOwnValues(collection) {
  const values = [];
  for (const key in collection) {
    if (hasOwnProperty.call(collection, key)) {
      values.push(collection[key]);
    }
  }
  return values;
}
function arrayOf(count, f) {
  const result = new Array(count);
  for (let i = 0; i < count; i++) {
    result[i] = f(i);
  }
  return result;
}
function arrayFrom(iterator, map2) {
  const result = [];
  for (const value of iterator) {
    result.push(map2 ? map2(value) : value);
  }
  return result;
}
function assign(t, ...args) {
  for (const arg of args) {
    if (arg === void 0) continue;
    for (const p in arg) {
      if (hasProperty(arg, p)) {
        t[p] = arg[p];
      }
    }
  }
  return t;
}
function equalOwnProperties(left, right, equalityComparer = equateValues) {
  if (left === right) return true;
  if (!left || !right) return false;
  for (const key in left) {
    if (hasOwnProperty.call(left, key)) {
      if (!hasOwnProperty.call(right, key)) return false;
      if (!equalityComparer(left[key], right[key])) return false;
    }
  }
  for (const key in right) {
    if (hasOwnProperty.call(right, key)) {
      if (!hasOwnProperty.call(left, key)) return false;
    }
  }
  return true;
}
function arrayToMap(array, makeKey, makeValue = identity) {
  const result = /* @__PURE__ */ new Map();
  for (let i = 0; i < array.length; i++) {
    const value = array[i];
    const key = makeKey(value);
    if (key !== void 0) result.set(key, makeValue(value));
  }
  return result;
}
function arrayToMultiMap(values, makeKey, makeValue = identity) {
  const result = createMultiMap();
  for (let i = 0; i < values.length; i++) {
    const value = values[i];
    result.add(makeKey(value), makeValue(value));
  }
  return result;
}
function group(values, getGroupId, resultSelector = identity) {
  return arrayFrom(arrayToMultiMap(values, getGroupId).values(), resultSelector);
}
function groupBy(values, keySelector) {
  const result = {};
  if (values !== void 0) {
    for (let i = 0; i < values.length; i++) {
      const value = values[i];
      const key = `${keySelector(value)}`;
      const array = result[key] ?? (result[key] = []);
      array.push(value);
    }
  }
  return result;
}
function extend(first2, second) {
  const result = {};
  for (const id in second) {
    if (hasOwnProperty.call(second, id)) {
      result[id] = second[id];
    }
  }
  for (const id in first2) {
    if (hasOwnProperty.call(first2, id)) {
      result[id] = first2[id];
    }
  }
  return result;
}
function copyProperties(first2, second) {
  for (const id in second) {
    if (hasOwnProperty.call(second, id)) {
      first2[id] = second[id];
    }
  }
}
function maybeBind(obj, fn) {
  return fn == null ? void 0 : fn.bind(obj);
}
function createMultiMap() {
  const map2 = /* @__PURE__ */ new Map();
  map2.add = multiMapAdd;
  map2.remove = multiMapRemove;
  return map2;
}
function multiMapAdd(key, value) {
  let values = this.get(key);
  if (values !== void 0) {
    values.push(value);
  } else {
    this.set(key, values = [value]);
  }
  return values;
}
function multiMapRemove(key, value) {
  const values = this.get(key);
  if (values !== void 0) {
    unorderedRemoveItem(values, value);
    if (!values.length) {
      this.delete(key);
    }
  }
}
function createQueue(items) {
  const elements = (items == null ? void 0 : items.slice()) ?? [];
  let headIndex = 0;
  function isEmpty() {
    return headIndex === elements.length;
  }
  function enqueue(...items2) {
    elements.push(...items2);
  }
  function dequeue() {
    if (isEmpty()) {
      throw new Error("Queue is empty");
    }
    const result = elements[headIndex];
    elements[headIndex] = void 0;
    headIndex++;
    if (headIndex > 100 && headIndex > elements.length >> 1) {
      const newLength = elements.length - headIndex;
      elements.copyWithin(
        /*target*/
        0,
        /*start*/
        headIndex
      );
      elements.length = newLength;
      headIndex = 0;
    }
    return result;
  }
  return {
    enqueue,
    dequeue,
    isEmpty
  };
}
function isArray(value) {
  return Array.isArray(value);
}
function toArray(value) {
  return isArray(value) ? value : [value];
}
function isString(text) {
  return typeof text === "string";
}
function isNumber(x) {
  return typeof x === "number";
}
function tryCast(value, test) {
  return value !== void 0 && test(value) ? value : void 0;
}
function cast(value, test) {
  if (value !== void 0 && test(value)) return value;
  return Debug.fail(`Invalid cast. The supplied value ${value} did not pass the test '${Debug.getFunctionName(test)}'.`);
}
function noop(_) {
}
function returnFalse() {
  return false;
}
function returnTrue() {
  return true;
}
function returnUndefined() {
  return void 0;
}
function identity(x) {
  return x;
}
function toLowerCase(x) {
  return x.toLowerCase();
}
var fileNameLowerCaseRegExp = /[^\u0130\u0131\u00DFa-z0-9\\/:\-_. ]+/g;
function toFileNameLowerCase(x) {
  return fileNameLowerCaseRegExp.test(x) ? x.replace(fileNameLowerCaseRegExp, toLowerCase) : x;
}
function notImplemented() {
  throw new Error("Not implemented");
}
function memoize(callback) {
  let value;
  return () => {
    if (callback) {
      value = callback();
      callback = void 0;
    }
    return value;
  };
}
function memoizeOne(callback) {
  const map2 = /* @__PURE__ */ new Map();
  return (arg) => {
    const key = `${typeof arg}:${arg}`;
    let value = map2.get(key);
    if (value === void 0 && !map2.has(key)) {
      value = callback(arg);
      map2.set(key, value);
    }
    return value;
  };
}
function equateValues(a, b) {
  return a === b;
}
function equateStringsCaseInsensitive(a, b) {
  return a === b || a !== void 0 && b !== void 0 && a.toUpperCase() === b.toUpperCase();
}
function equateStringsCaseSensitive(a, b) {
  return equateValues(a, b);
}
function compareComparableValues(a, b) {
  return a === b ? 0 /* EqualTo */ : a === void 0 ? -1 /* LessThan */ : b === void 0 ? 1 /* GreaterThan */ : a < b ? -1 /* LessThan */ : 1 /* GreaterThan */;
}
function compareValues(a, b) {
  return compareComparableValues(a, b);
}
function maxBy(arr, init, mapper) {
  for (let i = 0; i < arr.length; i++) {
    init = Math.max(init, mapper(arr[i]));
  }
  return init;
}
function min(items, compare) {
  return reduceLeft(items, (x, y) => compare(x, y) === -1 /* LessThan */ ? x : y);
}
function compareStringsCaseInsensitive(a, b) {
  if (a === b) return 0 /* EqualTo */;
  if (a === void 0) return -1 /* LessThan */;
  if (b === void 0) return 1 /* GreaterThan */;
  a = a.toUpperCase();
  b = b.toUpperCase();
  return a < b ? -1 /* LessThan */ : a > b ? 1 /* GreaterThan */ : 0 /* EqualTo */;
}
function compareStringsCaseSensitive(a, b) {
  return compareComparableValues(a, b);
}
function getStringComparer(ignoreCase) {
  return ignoreCase ? compareStringsCaseInsensitive : compareStringsCaseSensitive;
}
var uiComparerCaseSensitive;
var uiLocale;
function setUILocale(value) {
  if (uiLocale !== value) {
    uiLocale = value;
    uiComparerCaseSensitive = void 0;
  }
}
function compareBooleans(a, b) {
  return compareValues(a ? 1 : 0, b ? 1 : 0);
}
function getSpellingSuggestion(name, candidates, getName) {
  const maximumLengthDifference = Math.max(2, Math.floor(name.length * 0.34));
  let bestDistance = Math.floor(name.length * 0.4) + 1;
  let bestCandidate;
  for (const candidate of candidates) {
    const candidateName = getName(candidate);
    if (candidateName !== void 0 && Math.abs(candidateName.length - name.length) <= maximumLengthDifference) {
      if (candidateName === name) {
        continue;
      }
      if (candidateName.length < 3 && candidateName.toLowerCase() !== name.toLowerCase()) {
        continue;
      }
      const distance = levenshteinWithMax(name, candidateName, bestDistance - 0.1);
      if (distance === void 0) {
        continue;
      }
      Debug.assert(distance < bestDistance);
      bestDistance = distance;
      bestCandidate = candidate;
    }
  }
  return bestCandidate;
}
function levenshteinWithMax(s1, s2, max) {
  let previous = new Array(s2.length + 1);
  let current = new Array(s2.length + 1);
  const big = max + 0.01;
  for (let i = 0; i <= s2.length; i++) {
    previous[i] = i;
  }
  for (let i = 1; i <= s1.length; i++) {
    const c1 = s1.charCodeAt(i - 1);
    const minJ = Math.ceil(i > max ? i - max : 1);
    const maxJ = Math.floor(s2.length > max + i ? max + i : s2.length);
    current[0] = i;
    let colMin = i;
    for (let j = 1; j < minJ; j++) {
      current[j] = big;
    }
    for (let j = minJ; j <= maxJ; j++) {
      const substitutionDistance = s1[i - 1].toLowerCase() === s2[j - 1].toLowerCase() ? previous[j - 1] + 0.1 : previous[j - 1] + 2;
      const dist = c1 === s2.charCodeAt(j - 1) ? previous[j - 1] : Math.min(
        /*delete*/
        previous[j] + 1,
        /*insert*/
        current[j - 1] + 1,
        /*substitute*/
        substitutionDistance
      );
      current[j] = dist;
      colMin = Math.min(colMin, dist);
    }
    for (let j = maxJ + 1; j <= s2.length; j++) {
      current[j] = big;
    }
    if (colMin > max) {
      return void 0;
    }
    const temp = previous;
    previous = current;
    current = temp;
  }
  const res = previous[s2.length];
  return res > max ? void 0 : res;
}
function endsWith(str, suffix, ignoreCase) {
  const expectedPos = str.length - suffix.length;
  return expectedPos >= 0 && (ignoreCase ? equateStringsCaseInsensitive(str.slice(expectedPos), suffix) : str.indexOf(suffix, expectedPos) === expectedPos);
}
function removeSuffix(str, suffix) {
  return endsWith(str, suffix) ? str.slice(0, str.length - suffix.length) : str;
}
function orderedRemoveItem(array, item) {
  for (let i = 0; i < array.length; i++) {
    if (array[i] === item) {
      orderedRemoveItemAt(array, i);
      return true;
    }
  }
  return false;
}
function orderedRemoveItemAt(array, index) {
  for (let i = index; i < array.length - 1; i++) {
    array[i] = array[i + 1];
  }
  array.pop();
}
function unorderedRemoveItemAt(array, index) {
  array[index] = array[array.length - 1];
  array.pop();
}
function unorderedRemoveItem(array, item) {
  return unorderedRemoveFirstItemWhere(array, (element) => element === item);
}
function unorderedRemoveFirstItemWhere(array, predicate) {
  for (let i = 0; i < array.length; i++) {
    if (predicate(array[i])) {
      unorderedRemoveItemAt(array, i);
      return true;
    }
  }
  return false;
}
function createGetCanonicalFileName(useCaseSensitiveFileNames2) {
  return useCaseSensitiveFileNames2 ? identity : toFileNameLowerCase;
}
function patternText({ prefix, suffix }) {
  return `${prefix}*${suffix}`;
}
function matchedText(pattern, candidate) {
  Debug.assert(isPatternMatch(pattern, candidate));
  return candidate.substring(pattern.prefix.length, candidate.length - pattern.suffix.length);
}
function findBestPatternMatch(values, getPattern, candidate) {
  let matchedValue;
  let longestMatchPrefixLength = -1;
  for (let i = 0; i < values.length; i++) {
    const v = values[i];
    const pattern = getPattern(v);
    if (pattern.prefix.length > longestMatchPrefixLength && isPatternMatch(pattern, candidate)) {
      longestMatchPrefixLength = pattern.prefix.length;
      matchedValue = v;
    }
  }
  return matchedValue;
}
function startsWith(str, prefix, ignoreCase) {
  return ignoreCase ? equateStringsCaseInsensitive(str.slice(0, prefix.length), prefix) : str.lastIndexOf(prefix, 0) === 0;
}
function removePrefix(str, prefix) {
  return startsWith(str, prefix) ? str.substr(prefix.length) : str;
}
function isPatternMatch({ prefix, suffix }, candidate) {
  return candidate.length >= prefix.length + suffix.length && startsWith(candidate, prefix) && endsWith(candidate, suffix);
}
function and(f, g) {
  return (arg) => f(arg) && g(arg);
}
function or(...fs) {
  return (...args) => {
    let lastResult;
    for (const f of fs) {
      lastResult = f(...args);
      if (lastResult) {
        return lastResult;
      }
    }
    return lastResult;
  };
}
function not(fn) {
  return (...args) => !fn(...args);
}
function assertType(_) {
}
function singleElementArray(t) {
  return t === void 0 ? void 0 : [t];
}
function enumerateInsertsAndDeletes(newItems, oldItems, comparer, inserted, deleted, unchanged) {
  unchanged ?? (unchanged = noop);
  let newIndex = 0;
  let oldIndex = 0;
  const newLen = newItems.length;
  const oldLen = oldItems.length;
  let hasChanges = false;
  while (newIndex < newLen && oldIndex < oldLen) {
    const newItem = newItems[newIndex];
    const oldItem = oldItems[oldIndex];
    const compareResult = comparer(newItem, oldItem);
    if (compareResult === -1 /* LessThan */) {
      inserted(newItem);
      newIndex++;
      hasChanges = true;
    } else if (compareResult === 1 /* GreaterThan */) {
      deleted(oldItem);
      oldIndex++;
      hasChanges = true;
    } else {
      unchanged(oldItem, newItem);
      newIndex++;
      oldIndex++;
    }
  }
  while (newIndex < newLen) {
    inserted(newItems[newIndex++]);
    hasChanges = true;
  }
  while (oldIndex < oldLen) {
    deleted(oldItems[oldIndex++]);
    hasChanges = true;
  }
  return hasChanges;
}
function cartesianProduct(arrays) {
  const result = [];
  cartesianProductWorker(
    arrays,
    result,
    /*outer*/
    void 0,
    0
  );
  return result;
}
function cartesianProductWorker(arrays, result, outer, index) {
  for (const element of arrays[index]) {
    let inner;
    if (outer) {
      inner = outer.slice();
      inner.push(element);
    } else {
      inner = [element];
    }
    if (index === arrays.length - 1) {
      result.push(inner);
    } else {
      cartesianProductWorker(arrays, result, inner, index + 1);
    }
  }
}
function takeWhile(array, predicate) {
  if (array !== void 0) {
    const len = array.length;
    let index = 0;
    while (index < len && predicate(array[index])) {
      index++;
    }
    return array.slice(0, index);
  }
}
function skipWhile(array, predicate) {
  if (array !== void 0) {
    const len = array.length;
    let index = 0;
    while (index < len && predicate(array[index])) {
      index++;
    }
    return array.slice(index);
  }
}
function isNodeLikeSystem() {
  return typeof process !== "undefined" && !!process.nextTick && !process.browser && typeof require !== "undefined";
}

// src/compiler/debug.ts
var Debug;
((Debug2) => {
  let currentAssertionLevel = 0 /* None */;
  Debug2.currentLogLevel = 2 /* Warning */;
  Debug2.isDebugging = false;
  function shouldLog(level) {
    return Debug2.currentLogLevel <= level;
  }
  Debug2.shouldLog = shouldLog;
  function logMessage(level, s) {
    if (Debug2.loggingHost && shouldLog(level)) {
      Debug2.loggingHost.log(level, s);
    }
  }
  function log(s) {
    logMessage(3 /* Info */, s);
  }
  Debug2.log = log;
  ((_log) => {
    function error(s) {
      logMessage(1 /* Error */, s);
    }
    _log.error = error;
    function warn(s) {
      logMessage(2 /* Warning */, s);
    }
    _log.warn = warn;
    function log2(s) {
      logMessage(3 /* Info */, s);
    }
    _log.log = log2;
    function trace2(s) {
      logMessage(4 /* Verbose */, s);
    }
    _log.trace = trace2;
  })(log = Debug2.log || (Debug2.log = {}));
  const assertionCache = {};
  function getAssertionLevel() {
    return currentAssertionLevel;
  }
  Debug2.getAssertionLevel = getAssertionLevel;
  function setAssertionLevel(level) {
    const prevAssertionLevel = currentAssertionLevel;
    currentAssertionLevel = level;
    if (level > prevAssertionLevel) {
      for (const key of getOwnKeys(assertionCache)) {
        const cachedFunc = assertionCache[key];
        if (cachedFunc !== void 0 && Debug2[key] !== cachedFunc.assertion && level >= cachedFunc.level) {
          Debug2[key] = cachedFunc;
          assertionCache[key] = void 0;
        }
      }
    }
  }
  Debug2.setAssertionLevel = setAssertionLevel;
  function shouldAssert(level) {
    return currentAssertionLevel >= level;
  }
  Debug2.shouldAssert = shouldAssert;
  function shouldAssertFunction(level, name) {
    if (!shouldAssert(level)) {
      assertionCache[name] = { level, assertion: Debug2[name] };
      Debug2[name] = noop;
      return false;
    }
    return true;
  }
  function fail(message, stackCrawlMark) {
    debugger;
    const e = new Error(message ? `Debug Failure. ${message}` : "Debug Failure.");
    if (Error.captureStackTrace) {
      Error.captureStackTrace(e, stackCrawlMark || fail);
    }
    throw e;
  }
  Debug2.fail = fail;
  function failBadSyntaxKind(node, message, stackCrawlMark) {
    return fail(
      `${message || "Unexpected node."}\r
Node ${formatSyntaxKind(node.kind)} was unexpected.`,
      stackCrawlMark || failBadSyntaxKind
    );
  }
  Debug2.failBadSyntaxKind = failBadSyntaxKind;
  function assert(expression, message, verboseDebugInfo, stackCrawlMark) {
    if (!expression) {
      message = message ? `False expression: ${message}` : "False expression.";
      if (verboseDebugInfo) {
        message += "\r\nVerbose Debug Information: " + (typeof verboseDebugInfo === "string" ? verboseDebugInfo : verboseDebugInfo());
      }
      fail(message, stackCrawlMark || assert);
    }
  }
  Debug2.assert = assert;
  function assertEqual(a, b, msg, msg2, stackCrawlMark) {
    if (a !== b) {
      const message = msg ? msg2 ? `${msg} ${msg2}` : msg : "";
      fail(`Expected ${a} === ${b}. ${message}`, stackCrawlMark || assertEqual);
    }
  }
  Debug2.assertEqual = assertEqual;
  function assertLessThan(a, b, msg, stackCrawlMark) {
    if (a >= b) {
      fail(`Expected ${a} < ${b}. ${msg || ""}`, stackCrawlMark || assertLessThan);
    }
  }
  Debug2.assertLessThan = assertLessThan;
  function assertLessThanOrEqual(a, b, stackCrawlMark) {
    if (a > b) {
      fail(`Expected ${a} <= ${b}`, stackCrawlMark || assertLessThanOrEqual);
    }
  }
  Debug2.assertLessThanOrEqual = assertLessThanOrEqual;
  function assertGreaterThanOrEqual(a, b, stackCrawlMark) {
    if (a < b) {
      fail(`Expected ${a} >= ${b}`, stackCrawlMark || assertGreaterThanOrEqual);
    }
  }
  Debug2.assertGreaterThanOrEqual = assertGreaterThanOrEqual;
  function assertIsDefined(value, message, stackCrawlMark) {
    if (value === void 0 || value === null) {
      fail(message, stackCrawlMark || assertIsDefined);
    }
  }
  Debug2.assertIsDefined = assertIsDefined;
  function checkDefined(value, message, stackCrawlMark) {
    assertIsDefined(value, message, stackCrawlMark || checkDefined);
    return value;
  }
  Debug2.checkDefined = checkDefined;
  function assertEachIsDefined(value, message, stackCrawlMark) {
    for (const v of value) {
      assertIsDefined(v, message, stackCrawlMark || assertEachIsDefined);
    }
  }
  Debug2.assertEachIsDefined = assertEachIsDefined;
  function checkEachDefined(value, message, stackCrawlMark) {
    assertEachIsDefined(value, message, stackCrawlMark || checkEachDefined);
    return value;
  }
  Debug2.checkEachDefined = checkEachDefined;
  function assertNever(member, message = "Illegal value:", stackCrawlMark) {
    const detail = typeof member === "object" && hasProperty(member, "kind") && hasProperty(member, "pos") ? "SyntaxKind: " + formatSyntaxKind(member.kind) : JSON.stringify(member);
    return fail(`${message} ${detail}`, stackCrawlMark || assertNever);
  }
  Debug2.assertNever = assertNever;
  function assertEachNode(nodes, test, message, stackCrawlMark) {
    if (shouldAssertFunction(1 /* Normal */, "assertEachNode")) {
      assert(
        test === void 0 || every(nodes, test),
        message || "Unexpected node.",
        () => `Node array did not pass test '${getFunctionName(test)}'.`,
        stackCrawlMark || assertEachNode
      );
    }
  }
  Debug2.assertEachNode = assertEachNode;
  function assertNode(node, test, message, stackCrawlMark) {
    if (shouldAssertFunction(1 /* Normal */, "assertNode")) {
      assert(
        node !== void 0 && (test === void 0 || test(node)),
        message || "Unexpected node.",
        () => `Node ${formatSyntaxKind(node == null ? void 0 : node.kind)} did not pass test '${getFunctionName(test)}'.`,
        stackCrawlMark || assertNode
      );
    }
  }
  Debug2.assertNode = assertNode;
  function assertNotNode(node, test, message, stackCrawlMark) {
    if (shouldAssertFunction(1 /* Normal */, "assertNotNode")) {
      assert(
        node === void 0 || test === void 0 || !test(node),
        message || "Unexpected node.",
        () => `Node ${formatSyntaxKind(node.kind)} should not have passed test '${getFunctionName(test)}'.`,
        stackCrawlMark || assertNotNode
      );
    }
  }
  Debug2.assertNotNode = assertNotNode;
  function assertOptionalNode(node, test, message, stackCrawlMark) {
    if (shouldAssertFunction(1 /* Normal */, "assertOptionalNode")) {
      assert(
        test === void 0 || node === void 0 || test(node),
        message || "Unexpected node.",
        () => `Node ${formatSyntaxKind(node == null ? void 0 : node.kind)} did not pass test '${getFunctionName(test)}'.`,
        stackCrawlMark || assertOptionalNode
      );
    }
  }
  Debug2.assertOptionalNode = assertOptionalNode;
  function assertOptionalToken(node, kind, message, stackCrawlMark) {
    if (shouldAssertFunction(1 /* Normal */, "assertOptionalToken")) {
      assert(
        kind === void 0 || node === void 0 || node.kind === kind,
        message || "Unexpected node.",
        () => `Node ${formatSyntaxKind(node == null ? void 0 : node.kind)} was not a '${formatSyntaxKind(kind)}' token.`,
        stackCrawlMark || assertOptionalToken
      );
    }
  }
  Debug2.assertOptionalToken = assertOptionalToken;
  function assertMissingNode(node, message, stackCrawlMark) {
    if (shouldAssertFunction(1 /* Normal */, "assertMissingNode")) {
      assert(
        node === void 0,
        message || "Unexpected node.",
        () => `Node ${formatSyntaxKind(node.kind)} was unexpected'.`,
        stackCrawlMark || assertMissingNode
      );
    }
  }
  Debug2.assertMissingNode = assertMissingNode;
  function type(_value) {
  }
  Debug2.type = type;
  function getFunctionName(func) {
    if (typeof func !== "function") {
      return "";
    } else if (hasProperty(func, "name")) {
      return func.name;
    } else {
      const text = Function.prototype.toString.call(func);
      const match = /^function\s+([\w$]+)\s*\(/.exec(text);
      return match ? match[1] : "";
    }
  }
  Debug2.getFunctionName = getFunctionName;
  function formatSymbol(symbol) {
    return `{ name: ${unescapeLeadingUnderscores(symbol.escapedName)}; flags: ${formatSymbolFlags(symbol.flags)}; declarations: ${map(symbol.declarations, (node) => formatSyntaxKind(node.kind))} }`;
  }
  Debug2.formatSymbol = formatSymbol;
  function formatEnum(value = 0, enumObject, isFlags) {
    const members = getEnumMembers(enumObject);
    if (value === 0) {
      return members.length > 0 && members[0][0] === 0 ? members[0][1] : "0";
    }
    if (isFlags) {
      const result = [];
      let remainingFlags = value;
      for (const [enumValue, enumName] of members) {
        if (enumValue > value) {
          break;
        }
        if (enumValue !== 0 && enumValue & value) {
          result.push(enumName);
          remainingFlags &= ~enumValue;
        }
      }
      if (remainingFlags === 0) {
        return result.join("|");
      }
    } else {
      for (const [enumValue, enumName] of members) {
        if (enumValue === value) {
          return enumName;
        }
      }
    }
    return value.toString();
  }
  Debug2.formatEnum = formatEnum;
  const enumMemberCache = /* @__PURE__ */ new Map();
  function getEnumMembers(enumObject) {
    const existing = enumMemberCache.get(enumObject);
    if (existing) {
      return existing;
    }
    const result = [];
    for (const name in enumObject) {
      const value = enumObject[name];
      if (typeof value === "number") {
        result.push([value, name]);
      }
    }
    const sorted = toSorted(result, (x, y) => compareValues(x[0], y[0]));
    enumMemberCache.set(enumObject, sorted);
    return sorted;
  }
  function formatSyntaxKind(kind) {
    return formatEnum(
      kind,
      SyntaxKind,
      /*isFlags*/
      false
    );
  }
  Debug2.formatSyntaxKind = formatSyntaxKind;
  function formatSnippetKind(kind) {
    return formatEnum(
      kind,
      SnippetKind,
      /*isFlags*/
      false
    );
  }
  Debug2.formatSnippetKind = formatSnippetKind;
  function formatScriptKind(kind) {
    return formatEnum(
      kind,
      ScriptKind,
      /*isFlags*/
      false
    );
  }
  Debug2.formatScriptKind = formatScriptKind;
  function formatNodeFlags(flags) {
    return formatEnum(
      flags,
      NodeFlags,
      /*isFlags*/
      true
    );
  }
  Debug2.formatNodeFlags = formatNodeFlags;
  function formatNodeCheckFlags(flags) {
    return formatEnum(
      flags,
      NodeCheckFlags,
      /*isFlags*/
      true
    );
  }
  Debug2.formatNodeCheckFlags = formatNodeCheckFlags;
  function formatModifierFlags(flags) {
    return formatEnum(
      flags,
      ModifierFlags,
      /*isFlags*/
      true
    );
  }
  Debug2.formatModifierFlags = formatModifierFlags;
  function formatTransformFlags(flags) {
    return formatEnum(
      flags,
      TransformFlags,
      /*isFlags*/
      true
    );
  }
  Debug2.formatTransformFlags = formatTransformFlags;
  function formatEmitFlags(flags) {
    return formatEnum(
      flags,
      EmitFlags,
      /*isFlags*/
      true
    );
  }
  Debug2.formatEmitFlags = formatEmitFlags;
  function formatSymbolFlags(flags) {
    return formatEnum(
      flags,
      SymbolFlags,
      /*isFlags*/
      true
    );
  }
  Debug2.formatSymbolFlags = formatSymbolFlags;
  function formatTypeFlags(flags) {
    return formatEnum(
      flags,
      TypeFlags,
      /*isFlags*/
      true
    );
  }
  Debug2.formatTypeFlags = formatTypeFlags;
  function formatSignatureFlags(flags) {
    return formatEnum(
      flags,
      SignatureFlags,
      /*isFlags*/
      true
    );
  }
  Debug2.formatSignatureFlags = formatSignatureFlags;
  function formatObjectFlags(flags) {
    return formatEnum(
      flags,
      ObjectFlags,
      /*isFlags*/
      true
    );
  }
  Debug2.formatObjectFlags = formatObjectFlags;
  function formatFlowFlags(flags) {
    return formatEnum(
      flags,
      FlowFlags,
      /*isFlags*/
      true
    );
  }
  Debug2.formatFlowFlags = formatFlowFlags;
  function formatRelationComparisonResult(result) {
    return formatEnum(
      result,
      RelationComparisonResult,
      /*isFlags*/
      true
    );
  }
  Debug2.formatRelationComparisonResult = formatRelationComparisonResult;
  function formatCheckMode(mode) {
    return formatEnum(
      mode,
      CheckMode,
      /*isFlags*/
      true
    );
  }
  Debug2.formatCheckMode = formatCheckMode;
  function formatSignatureCheckMode(mode) {
    return formatEnum(
      mode,
      SignatureCheckMode,
      /*isFlags*/
      true
    );
  }
  Debug2.formatSignatureCheckMode = formatSignatureCheckMode;
  function formatTypeFacts(facts) {
    return formatEnum(
      facts,
      TypeFacts,
      /*isFlags*/
      true
    );
  }
  Debug2.formatTypeFacts = formatTypeFacts;
  let isDebugInfoEnabled = false;
  let flowNodeProto;
  function attachFlowNodeDebugInfoWorker(flowNode) {
    if (!("__debugFlowFlags" in flowNode)) {
      Object.defineProperties(flowNode, {
        // for use with vscode-js-debug's new customDescriptionGenerator in launch.json
        __tsDebuggerDisplay: {
          value() {
            const flowHeader = this.flags & 2 /* Start */ ? "FlowStart" : this.flags & 4 /* BranchLabel */ ? "FlowBranchLabel" : this.flags & 8 /* LoopLabel */ ? "FlowLoopLabel" : this.flags & 16 /* Assignment */ ? "FlowAssignment" : this.flags & 32 /* TrueCondition */ ? "FlowTrueCondition" : this.flags & 64 /* FalseCondition */ ? "FlowFalseCondition" : this.flags & 128 /* SwitchClause */ ? "FlowSwitchClause" : this.flags & 256 /* ArrayMutation */ ? "FlowArrayMutation" : this.flags & 512 /* Call */ ? "FlowCall" : this.flags & 1024 /* ReduceLabel */ ? "FlowReduceLabel" : this.flags & 1 /* Unreachable */ ? "FlowUnreachable" : "UnknownFlow";
            const remainingFlags = this.flags & ~(2048 /* Referenced */ - 1);
            return `${flowHeader}${remainingFlags ? ` (${formatFlowFlags(remainingFlags)})` : ""}`;
          }
        },
        __debugFlowFlags: {
          get() {
            return formatEnum(
              this.flags,
              FlowFlags,
              /*isFlags*/
              true
            );
          }
        },
        __debugToString: {
          value() {
            return formatControlFlowGraph(this);
          }
        }
      });
    }
  }
  function attachFlowNodeDebugInfo(flowNode) {
    if (isDebugInfoEnabled) {
      if (typeof Object.setPrototypeOf === "function") {
        if (!flowNodeProto) {
          flowNodeProto = Object.create(Object.prototype);
          attachFlowNodeDebugInfoWorker(flowNodeProto);
        }
        Object.setPrototypeOf(flowNode, flowNodeProto);
      } else {
        attachFlowNodeDebugInfoWorker(flowNode);
      }
    }
    return flowNode;
  }
  Debug2.attachFlowNodeDebugInfo = attachFlowNodeDebugInfo;
  let nodeArrayProto;
  function attachNodeArrayDebugInfoWorker(array) {
    if (!("__tsDebuggerDisplay" in array)) {
      Object.defineProperties(array, {
        __tsDebuggerDisplay: {
          value(defaultValue) {
            defaultValue = String(defaultValue).replace(/(?:,[\s\w]+:[^,]+)+\]$/, "]");
            return `NodeArray ${defaultValue}`;
          }
        }
      });
    }
  }
  function attachNodeArrayDebugInfo(array) {
    if (isDebugInfoEnabled) {
      if (typeof Object.setPrototypeOf === "function") {
        if (!nodeArrayProto) {
          nodeArrayProto = Object.create(Array.prototype);
          attachNodeArrayDebugInfoWorker(nodeArrayProto);
        }
        Object.setPrototypeOf(array, nodeArrayProto);
      } else {
        attachNodeArrayDebugInfoWorker(array);
      }
    }
  }
  Debug2.attachNodeArrayDebugInfo = attachNodeArrayDebugInfo;
  function enableDebugInfo() {
    if (isDebugInfoEnabled) return;
    const weakTypeTextMap = /* @__PURE__ */ new WeakMap();
    const weakNodeTextMap = /* @__PURE__ */ new WeakMap();
    Object.defineProperties(objectAllocator.getSymbolConstructor().prototype, {
      // for use with vscode-js-debug's new customDescriptionGenerator in launch.json
      __tsDebuggerDisplay: {
        value() {
          const symbolHeader = this.flags & 33554432 /* Transient */ ? "TransientSymbol" : "Symbol";
          const remainingSymbolFlags = this.flags & ~33554432 /* Transient */;
          return `${symbolHeader} '${symbolName(this)}'${remainingSymbolFlags ? ` (${formatSymbolFlags(remainingSymbolFlags)})` : ""}`;
        }
      },
      __debugFlags: {
        get() {
          return formatSymbolFlags(this.flags);
        }
      }
    });
    Object.defineProperties(objectAllocator.getTypeConstructor().prototype, {
      // for use with vscode-js-debug's new customDescriptionGenerator in launch.json
      __tsDebuggerDisplay: {
        value() {
          const typeHeader = this.flags & 67359327 /* Intrinsic */ ? `IntrinsicType ${this.intrinsicName}${this.debugIntrinsicName ? ` (${this.debugIntrinsicName})` : ""}` : this.flags & 98304 /* Nullable */ ? "NullableType" : this.flags & 384 /* StringOrNumberLiteral */ ? `LiteralType ${JSON.stringify(this.value)}` : this.flags & 2048 /* BigIntLiteral */ ? `LiteralType ${this.value.negative ? "-" : ""}${this.value.base10Value}n` : this.flags & 8192 /* UniqueESSymbol */ ? "UniqueESSymbolType" : this.flags & 32 /* Enum */ ? "EnumType" : this.flags & 1048576 /* Union */ ? "UnionType" : this.flags & 2097152 /* Intersection */ ? "IntersectionType" : this.flags & 4194304 /* Index */ ? "IndexType" : this.flags & 8388608 /* IndexedAccess */ ? "IndexedAccessType" : this.flags & 16777216 /* Conditional */ ? "ConditionalType" : this.flags & 33554432 /* Substitution */ ? "SubstitutionType" : this.flags & 262144 /* TypeParameter */ ? "TypeParameter" : this.flags & 524288 /* Object */ ? this.objectFlags & 3 /* ClassOrInterface */ ? "InterfaceType" : this.objectFlags & 4 /* Reference */ ? "TypeReference" : this.objectFlags & 8 /* Tuple */ ? "TupleType" : this.objectFlags & 16 /* Anonymous */ ? "AnonymousType" : this.objectFlags & 32 /* Mapped */ ? "MappedType" : this.objectFlags & 1024 /* ReverseMapped */ ? "ReverseMappedType" : this.objectFlags & 256 /* EvolvingArray */ ? "EvolvingArrayType" : "ObjectType" : "Type";
          const remainingObjectFlags = this.flags & 524288 /* Object */ ? this.objectFlags & ~1343 /* ObjectTypeKindMask */ : 0;
          return `${typeHeader}${this.symbol ? ` '${symbolName(this.symbol)}'` : ""}${remainingObjectFlags ? ` (${formatObjectFlags(remainingObjectFlags)})` : ""}`;
        }
      },
      __debugFlags: {
        get() {
          return formatTypeFlags(this.flags);
        }
      },
      __debugObjectFlags: {
        get() {
          return this.flags & 524288 /* Object */ ? formatObjectFlags(this.objectFlags) : "";
        }
      },
      __debugTypeToString: {
        value() {
          let text = weakTypeTextMap.get(this);
          if (text === void 0) {
            text = this.checker.typeToString(this);
            weakTypeTextMap.set(this, text);
          }
          return text;
        }
      }
    });
    Object.defineProperties(objectAllocator.getSignatureConstructor().prototype, {
      __debugFlags: {
        get() {
          return formatSignatureFlags(this.flags);
        }
      },
      __debugSignatureToString: {
        value() {
          var _a;
          return (_a = this.checker) == null ? void 0 : _a.signatureToString(this);
        }
      }
    });
    const nodeConstructors = [
      objectAllocator.getNodeConstructor(),
      objectAllocator.getIdentifierConstructor(),
      objectAllocator.getTokenConstructor(),
      objectAllocator.getSourceFileConstructor()
    ];
    for (const ctor of nodeConstructors) {
      if (!hasProperty(ctor.prototype, "__debugKind")) {
        Object.defineProperties(ctor.prototype, {
          // for use with vscode-js-debug's new customDescriptionGenerator in launch.json
          __tsDebuggerDisplay: {
            value() {
              const nodeHeader = isGeneratedIdentifier(this) ? "GeneratedIdentifier" : isIdentifier(this) ? `Identifier '${idText(this)}'` : isPrivateIdentifier(this) ? `PrivateIdentifier '${idText(this)}'` : isStringLiteral(this) ? `StringLiteral ${JSON.stringify(this.text.length < 10 ? this.text : this.text.slice(10) + "...")}` : isNumericLiteral(this) ? `NumericLiteral ${this.text}` : isBigIntLiteral(this) ? `BigIntLiteral ${this.text}n` : isTypeParameterDeclaration(this) ? "TypeParameterDeclaration" : isParameter(this) ? "ParameterDeclaration" : isConstructorDeclaration(this) ? "ConstructorDeclaration" : isGetAccessorDeclaration(this) ? "GetAccessorDeclaration" : isSetAccessorDeclaration(this) ? "SetAccessorDeclaration" : isCallSignatureDeclaration(this) ? "CallSignatureDeclaration" : isConstructSignatureDeclaration(this) ? "ConstructSignatureDeclaration" : isIndexSignatureDeclaration(this) ? "IndexSignatureDeclaration" : isTypePredicateNode(this) ? "TypePredicateNode" : isTypeReferenceNode(this) ? "TypeReferenceNode" : isFunctionTypeNode(this) ? "FunctionTypeNode" : isConstructorTypeNode(this) ? "ConstructorTypeNode" : isTypeQueryNode(this) ? "TypeQueryNode" : isTypeLiteralNode(this) ? "TypeLiteralNode" : isArrayTypeNode(this) ? "ArrayTypeNode" : isTupleTypeNode(this) ? "TupleTypeNode" : isOptionalTypeNode(this) ? "OptionalTypeNode" : isRestTypeNode(this) ? "RestTypeNode" : isUnionTypeNode(this) ? "UnionTypeNode" : isIntersectionTypeNode(this) ? "IntersectionTypeNode" : isConditionalTypeNode(this) ? "ConditionalTypeNode" : isInferTypeNode(this) ? "InferTypeNode" : isParenthesizedTypeNode(this) ? "ParenthesizedTypeNode" : isThisTypeNode(this) ? "ThisTypeNode" : isTypeOperatorNode(this) ? "TypeOperatorNode" : isIndexedAccessTypeNode(this) ? "IndexedAccessTypeNode" : isMappedTypeNode(this) ? "MappedTypeNode" : isLiteralTypeNode(this) ? "LiteralTypeNode" : isNamedTupleMember(this) ? "NamedTupleMember" : isImportTypeNode(this) ? "ImportTypeNode" : formatSyntaxKind(this.kind);
              return `${nodeHeader}${this.flags ? ` (${formatNodeFlags(this.flags)})` : ""}`;
            }
          },
          __debugKind: {
            get() {
              return formatSyntaxKind(this.kind);
            }
          },
          __debugNodeFlags: {
            get() {
              return formatNodeFlags(this.flags);
            }
          },
          __debugModifierFlags: {
            get() {
              return formatModifierFlags(getEffectiveModifierFlagsNoCache(this));
            }
          },
          __debugTransformFlags: {
            get() {
              return formatTransformFlags(this.transformFlags);
            }
          },
          __debugIsParseTreeNode: {
            get() {
              return isParseTreeNode(this);
            }
          },
          __debugEmitFlags: {
            get() {
              return formatEmitFlags(getEmitFlags(this));
            }
          },
          __debugGetText: {
            value(includeTrivia) {
              if (nodeIsSynthesized(this)) return "";
              let text = weakNodeTextMap.get(this);
              if (text === void 0) {
                const parseNode = getParseTreeNode(this);
                const sourceFile = parseNode && getSourceFileOfNode(parseNode);
                text = sourceFile ? getSourceTextOfNodeFromSourceFile(sourceFile, parseNode, includeTrivia) : "";
                weakNodeTextMap.set(this, text);
              }
              return text;
            }
          }
        });
      }
    }
    isDebugInfoEnabled = true;
  }
  Debug2.enableDebugInfo = enableDebugInfo;
  function formatVariance(varianceFlags) {
    const variance = varianceFlags & 7 /* VarianceMask */;
    let result = variance === 0 /* Invariant */ ? "in out" : variance === 3 /* Bivariant */ ? "[bivariant]" : variance === 2 /* Contravariant */ ? "in" : variance === 1 /* Covariant */ ? "out" : variance === 4 /* Independent */ ? "[independent]" : "";
    if (varianceFlags & 8 /* Unmeasurable */) {
      result += " (unmeasurable)";
    } else if (varianceFlags & 16 /* Unreliable */) {
      result += " (unreliable)";
    }
    return result;
  }
  Debug2.formatVariance = formatVariance;
  class DebugTypeMapper {
    __debugToString() {
      var _a;
      type(this);
      switch (this.kind) {
        case 3 /* Function */:
          return ((_a = this.debugInfo) == null ? void 0 : _a.call(this)) || "(function mapper)";
        case 0 /* Simple */:
          return `${this.source.__debugTypeToString()} -> ${this.target.__debugTypeToString()}`;
        case 1 /* Array */:
          return zipWith(
            this.sources,
            this.targets || map(this.sources, () => "any"),
            (s, t) => `${s.__debugTypeToString()} -> ${typeof t === "string" ? t : t.__debugTypeToString()}`
          ).join(", ");
        case 2 /* Deferred */:
          return zipWith(
            this.sources,
            this.targets,
            (s, t) => `${s.__debugTypeToString()} -> ${t().__debugTypeToString()}`
          ).join(", ");
        case 5 /* Merged */:
        case 4 /* Composite */:
          return `m1: ${this.mapper1.__debugToString().split("\n").join("\n    ")}
m2: ${this.mapper2.__debugToString().split("\n").join("\n    ")}`;
        default:
          return assertNever(this);
      }
    }
  }
  Debug2.DebugTypeMapper = DebugTypeMapper;
  function attachDebugPrototypeIfDebug(mapper) {
    if (Debug2.isDebugging) {
      return Object.setPrototypeOf(mapper, DebugTypeMapper.prototype);
    }
    return mapper;
  }
  Debug2.attachDebugPrototypeIfDebug = attachDebugPrototypeIfDebug;
  function printControlFlowGraph(flowNode) {
    return console.log(formatControlFlowGraph(flowNode));
  }
  Debug2.printControlFlowGraph = printControlFlowGraph;
  function formatControlFlowGraph(flowNode) {
    let nextDebugFlowId = -1;
    function getDebugFlowNodeId(f) {
      if (!f.id) {
        f.id = nextDebugFlowId;
        nextDebugFlowId--;
      }
      return f.id;
    }
    let BoxCharacter;
    ((BoxCharacter2) => {
      BoxCharacter2["lr"] = "\u2500";
      BoxCharacter2["ud"] = "\u2502";
      BoxCharacter2["dr"] = "\u256D";
      BoxCharacter2["dl"] = "\u256E";
      BoxCharacter2["ul"] = "\u256F";
      BoxCharacter2["ur"] = "\u2570";
      BoxCharacter2["udr"] = "\u251C";
      BoxCharacter2["udl"] = "\u2524";
      BoxCharacter2["dlr"] = "\u252C";
      BoxCharacter2["ulr"] = "\u2534";
      BoxCharacter2["udlr"] = "\u256B";
    })(BoxCharacter || (BoxCharacter = {}));
    let Connection;
    ((Connection2) => {
      Connection2[Connection2["None"] = 0] = "None";
      Connection2[Connection2["Up"] = 1] = "Up";
      Connection2[Connection2["Down"] = 2] = "Down";
      Connection2[Connection2["Left"] = 4] = "Left";
      Connection2[Connection2["Right"] = 8] = "Right";
      Connection2[Connection2["UpDown"] = 3] = "UpDown";
      Connection2[Connection2["LeftRight"] = 12] = "LeftRight";
      Connection2[Connection2["UpLeft"] = 5] = "UpLeft";
      Connection2[Connection2["UpRight"] = 9] = "UpRight";
      Connection2[Connection2["DownLeft"] = 6] = "DownLeft";
      Connection2[Connection2["DownRight"] = 10] = "DownRight";
      Connection2[Connection2["UpDownLeft"] = 7] = "UpDownLeft";
      Connection2[Connection2["UpDownRight"] = 11] = "UpDownRight";
      Connection2[Connection2["UpLeftRight"] = 13] = "UpLeftRight";
      Connection2[Connection2["DownLeftRight"] = 14] = "DownLeftRight";
      Connection2[Connection2["UpDownLeftRight"] = 15] = "UpDownLeftRight";
      Connection2[Connection2["NoChildren"] = 16] = "NoChildren";
    })(Connection || (Connection = {}));
    const hasAntecedentFlags = 16 /* Assignment */ | 96 /* Condition */ | 128 /* SwitchClause */ | 256 /* ArrayMutation */ | 512 /* Call */ | 1024 /* ReduceLabel */;
    const hasNodeFlags = 2 /* Start */ | 16 /* Assignment */ | 512 /* Call */ | 96 /* Condition */ | 256 /* ArrayMutation */;
    const links = /* @__PURE__ */ Object.create(
      /*o*/
      null
    );
    const nodes = [];
    const edges = [];
    const root = buildGraphNode(flowNode, /* @__PURE__ */ new Set());
    for (const node of nodes) {
      node.text = renderFlowNode(node.flowNode, node.circular);
      computeLevel(node);
    }
    const height = computeHeight(root);
    const columnWidths = computeColumnWidths(height);
    computeLanes(root, 0);
    return renderGraph();
    function isFlowSwitchClause(f) {
      return !!(f.flags & 128 /* SwitchClause */);
    }
    function hasAntecedents(f) {
      return !!(f.flags & 12 /* Label */) && !!f.antecedent;
    }
    function hasAntecedent(f) {
      return !!(f.flags & hasAntecedentFlags);
    }
    function hasNode(f) {
      return !!(f.flags & hasNodeFlags);
    }
    function getChildren(node) {
      const children = [];
      for (const edge of node.edges) {
        if (edge.source === node) {
          children.push(edge.target);
        }
      }
      return children;
    }
    function getParents(node) {
      const parents = [];
      for (const edge of node.edges) {
        if (edge.target === node) {
          parents.push(edge.source);
        }
      }
      return parents;
    }
    function buildGraphNode(flowNode2, seen) {
      const id = getDebugFlowNodeId(flowNode2);
      let graphNode = links[id];
      if (graphNode && seen.has(flowNode2)) {
        graphNode.circular = true;
        graphNode = {
          id: -1,
          flowNode: flowNode2,
          edges: [],
          text: "",
          lane: -1,
          endLane: -1,
          level: -1,
          circular: "circularity"
        };
        nodes.push(graphNode);
        return graphNode;
      }
      seen.add(flowNode2);
      if (!graphNode) {
        links[id] = graphNode = { id, flowNode: flowNode2, edges: [], text: "", lane: -1, endLane: -1, level: -1, circular: false };
        nodes.push(graphNode);
        if (hasAntecedents(flowNode2)) {
          for (const antecedent of flowNode2.antecedent) {
            buildGraphEdge(graphNode, antecedent, seen);
          }
        } else if (hasAntecedent(flowNode2)) {
          buildGraphEdge(graphNode, flowNode2.antecedent, seen);
        }
      }
      seen.delete(flowNode2);
      return graphNode;
    }
    function buildGraphEdge(source, antecedent, seen) {
      const target = buildGraphNode(antecedent, seen);
      const edge = { source, target };
      edges.push(edge);
      source.edges.push(edge);
      target.edges.push(edge);
    }
    function computeLevel(node) {
      if (node.level !== -1) {
        return node.level;
      }
      let level = 0;
      for (const parent of getParents(node)) {
        level = Math.max(level, computeLevel(parent) + 1);
      }
      return node.level = level;
    }
    function computeHeight(node) {
      let height2 = 0;
      for (const child of getChildren(node)) {
        height2 = Math.max(height2, computeHeight(child));
      }
      return height2 + 1;
    }
    function computeColumnWidths(height2) {
      const columns = fill(Array(height2), 0);
      for (const node of nodes) {
        columns[node.level] = Math.max(columns[node.level], node.text.length);
      }
      return columns;
    }
    function computeLanes(node, lane) {
      if (node.lane === -1) {
        node.lane = lane;
        node.endLane = lane;
        const children = getChildren(node);
        for (let i = 0; i < children.length; i++) {
          if (i > 0) lane++;
          const child = children[i];
          computeLanes(child, lane);
          if (child.endLane > node.endLane) {
            lane = child.endLane;
          }
        }
        node.endLane = lane;
      }
    }
    function getHeader2(flags) {
      if (flags & 2 /* Start */) return "Start";
      if (flags & 4 /* BranchLabel */) return "Branch";
      if (flags & 8 /* LoopLabel */) return "Loop";
      if (flags & 16 /* Assignment */) return "Assignment";
      if (flags & 32 /* TrueCondition */) return "True";
      if (flags & 64 /* FalseCondition */) return "False";
      if (flags & 128 /* SwitchClause */) return "SwitchClause";
      if (flags & 256 /* ArrayMutation */) return "ArrayMutation";
      if (flags & 512 /* Call */) return "Call";
      if (flags & 1024 /* ReduceLabel */) return "ReduceLabel";
      if (flags & 1 /* Unreachable */) return "Unreachable";
      throw new Error();
    }
    function getNodeText(node) {
      const sourceFile = getSourceFileOfNode(node);
      return getSourceTextOfNodeFromSourceFile(
        sourceFile,
        node,
        /*includeTrivia*/
        false
      );
    }
    function renderFlowNode(flowNode2, circular) {
      let text = getHeader2(flowNode2.flags);
      if (circular) {
        text = `${text}#${getDebugFlowNodeId(flowNode2)}`;
      }
      if (isFlowSwitchClause(flowNode2)) {
        const clauses = [];
        const { switchStatement, clauseStart, clauseEnd } = flowNode2.node;
        for (let i = clauseStart; i < clauseEnd; i++) {
          const clause = switchStatement.caseBlock.clauses[i];
          if (isDefaultClause(clause)) {
            clauses.push("default");
          } else {
            clauses.push(getNodeText(clause.expression));
          }
        }
        text += ` (${clauses.join(", ")})`;
      } else if (hasNode(flowNode2)) {
        if (flowNode2.node) {
          text += ` (${getNodeText(flowNode2.node)})`;
        }
      }
      return circular === "circularity" ? `Circular(${text})` : text;
    }
    function renderGraph() {
      const columnCount = columnWidths.length;
      const laneCount = maxBy(nodes, 0, (n) => n.lane) + 1;
      const lanes = fill(Array(laneCount), "");
      const grid = columnWidths.map(() => Array(laneCount));
      const connectors = columnWidths.map(() => fill(Array(laneCount), 0));
      for (const node of nodes) {
        grid[node.level][node.lane] = node;
        const children = getChildren(node);
        for (let i = 0; i < children.length; i++) {
          const child = children[i];
          let connector = 8 /* Right */;
          if (child.lane === node.lane) connector |= 4 /* Left */;
          if (i > 0) connector |= 1 /* Up */;
          if (i < children.length - 1) connector |= 2 /* Down */;
          connectors[node.level][child.lane] |= connector;
        }
        if (children.length === 0) {
          connectors[node.level][node.lane] |= 16 /* NoChildren */;
        }
        const parents = getParents(node);
        for (let i = 0; i < parents.length; i++) {
          const parent = parents[i];
          let connector = 4 /* Left */;
          if (i > 0) connector |= 1 /* Up */;
          if (i < parents.length - 1) connector |= 2 /* Down */;
          connectors[node.level - 1][parent.lane] |= connector;
        }
      }
      for (let column = 0; column < columnCount; column++) {
        for (let lane = 0; lane < laneCount; lane++) {
          const left = column > 0 ? connectors[column - 1][lane] : 0;
          const above = lane > 0 ? connectors[column][lane - 1] : 0;
          let connector = connectors[column][lane];
          if (!connector) {
            if (left & 8 /* Right */) connector |= 12 /* LeftRight */;
            if (above & 2 /* Down */) connector |= 3 /* UpDown */;
            connectors[column][lane] = connector;
          }
        }
      }
      for (let column = 0; column < columnCount; column++) {
        for (let lane = 0; lane < lanes.length; lane++) {
          const connector = connectors[column][lane];
          const fill2 = connector & 4 /* Left */ ? "\u2500" /* lr */ : " ";
          const node = grid[column][lane];
          if (!node) {
            if (column < columnCount - 1) {
              writeLane(lane, repeat(fill2, columnWidths[column] + 1));
            }
          } else {
            writeLane(lane, node.text);
            if (column < columnCount - 1) {
              writeLane(lane, " ");
              writeLane(lane, repeat(fill2, columnWidths[column] - node.text.length));
            }
          }
          writeLane(lane, getBoxCharacter(connector));
          writeLane(lane, connector & 8 /* Right */ && column < columnCount - 1 && !grid[column + 1][lane] ? "\u2500" /* lr */ : " ");
        }
      }
      return `
${lanes.join("\n")}
`;
      function writeLane(lane, text) {
        lanes[lane] += text;
      }
    }
    function getBoxCharacter(connector) {
      switch (connector) {
        case 3 /* UpDown */:
          return "\u2502" /* ud */;
        case 12 /* LeftRight */:
          return "\u2500" /* lr */;
        case 5 /* UpLeft */:
          return "\u256F" /* ul */;
        case 9 /* UpRight */:
          return "\u2570" /* ur */;
        case 6 /* DownLeft */:
          return "\u256E" /* dl */;
        case 10 /* DownRight */:
          return "\u256D" /* dr */;
        case 7 /* UpDownLeft */:
          return "\u2524" /* udl */;
        case 11 /* UpDownRight */:
          return "\u251C" /* udr */;
        case 13 /* UpLeftRight */:
          return "\u2534" /* ulr */;
        case 14 /* DownLeftRight */:
          return "\u252C" /* dlr */;
        case 15 /* UpDownLeftRight */:
          return "\u256B" /* udlr */;
      }
      return " ";
    }
    function fill(array, value) {
      if (array.fill) {
        array.fill(value);
      } else {
        for (let i = 0; i < array.length; i++) {
          array[i] = value;
        }
      }
      return array;
    }
    function repeat(ch, length2) {
      if (ch.repeat) {
        return length2 > 0 ? ch.repeat(length2) : "";
      }
      let s = "";
      while (s.length < length2) {
        s += ch;
      }
      return s;
    }
  }
  Debug2.formatControlFlowGraph = formatControlFlowGraph;
})(Debug || (Debug = {}));

// src/compiler/semver.ts
var versionRegExp = /^(0|[1-9]\d*)(?:\.(0|[1-9]\d*)(?:\.(0|[1-9]\d*)(?:-([a-z0-9-.]+))?(?:\+([a-z0-9-.]+))?)?)?$/i;
var prereleaseRegExp = /^(?:0|[1-9]\d*|[a-z-][a-z0-9-]*)(?:\.(?:0|[1-9]\d*|[a-z-][a-z0-9-]*))*$/i;
var prereleasePartRegExp = /^(?:0|[1-9]\d*|[a-z-][a-z0-9-]*)$/i;
var buildRegExp = /^[a-z0-9-]+(?:\.[a-z0-9-]+)*$/i;
var buildPartRegExp = /^[a-z0-9-]+$/i;
var numericIdentifierRegExp = /^(?:0|[1-9]\d*)$/;
var _Version = class _Version {
  constructor(major, minor = 0, patch = 0, prerelease = "", build2 = "") {
    if (typeof major === "string") {
      const result = Debug.checkDefined(tryParseComponents(major), "Invalid version");
      ({ major, minor, patch, prerelease, build: build2 } = result);
    }
    Debug.assert(major >= 0, "Invalid argument: major");
    Debug.assert(minor >= 0, "Invalid argument: minor");
    Debug.assert(patch >= 0, "Invalid argument: patch");
    const prereleaseArray = prerelease ? isArray(prerelease) ? prerelease : prerelease.split(".") : emptyArray;
    const buildArray = build2 ? isArray(build2) ? build2 : build2.split(".") : emptyArray;
    Debug.assert(every(prereleaseArray, (s) => prereleasePartRegExp.test(s)), "Invalid argument: prerelease");
    Debug.assert(every(buildArray, (s) => buildPartRegExp.test(s)), "Invalid argument: build");
    this.major = major;
    this.minor = minor;
    this.patch = patch;
    this.prerelease = prereleaseArray;
    this.build = buildArray;
  }
  static tryParse(text) {
    const result = tryParseComponents(text);
    if (!result) return void 0;
    const { major, minor, patch, prerelease, build: build2 } = result;
    return new _Version(major, minor, patch, prerelease, build2);
  }
  compareTo(other) {
    if (this === other) return 0 /* EqualTo */;
    if (other === void 0) return 1 /* GreaterThan */;
    return compareValues(this.major, other.major) || compareValues(this.minor, other.minor) || compareValues(this.patch, other.patch) || comparePrereleaseIdentifiers(this.prerelease, other.prerelease);
  }
  increment(field) {
    switch (field) {
      case "major":
        return new _Version(this.major + 1, 0, 0);
      case "minor":
        return new _Version(this.major, this.minor + 1, 0);
      case "patch":
        return new _Version(this.major, this.minor, this.patch + 1);
      default:
        return Debug.assertNever(field);
    }
  }
  with(fields) {
    const {
      major = this.major,
      minor = this.minor,
      patch = this.patch,
      prerelease = this.prerelease,
      build: build2 = this.build
    } = fields;
    return new _Version(major, minor, patch, prerelease, build2);
  }
  toString() {
    let result = `${this.major}.${this.minor}.${this.patch}`;
    if (some(this.prerelease)) result += `-${this.prerelease.join(".")}`;
    if (some(this.build)) result += `+${this.build.join(".")}`;
    return result;
  }
};
_Version.zero = new _Version(0, 0, 0, ["0"]);
var Version = _Version;
function tryParseComponents(text) {
  const match = versionRegExp.exec(text);
  if (!match) return void 0;
  const [, major, minor = "0", patch = "0", prerelease = "", build2 = ""] = match;
  if (prerelease && !prereleaseRegExp.test(prerelease)) return void 0;
  if (build2 && !buildRegExp.test(build2)) return void 0;
  return {
    major: parseInt(major, 10),
    minor: parseInt(minor, 10),
    patch: parseInt(patch, 10),
    prerelease,
    build: build2
  };
}
function comparePrereleaseIdentifiers(left, right) {
  if (left === right) return 0 /* EqualTo */;
  if (left.length === 0) return right.length === 0 ? 0 /* EqualTo */ : 1 /* GreaterThan */;
  if (right.length === 0) return -1 /* LessThan */;
  const length2 = Math.min(left.length, right.length);
  for (let i = 0; i < length2; i++) {
    const leftIdentifier = left[i];
    const rightIdentifier = right[i];
    if (leftIdentifier === rightIdentifier) continue;
    const leftIsNumeric = numericIdentifierRegExp.test(leftIdentifier);
    const rightIsNumeric = numericIdentifierRegExp.test(rightIdentifier);
    if (leftIsNumeric || rightIsNumeric) {
      if (leftIsNumeric !== rightIsNumeric) return leftIsNumeric ? -1 /* LessThan */ : 1 /* GreaterThan */;
      const result = compareValues(+leftIdentifier, +rightIdentifier);
      if (result) return result;
    } else {
      const result = compareStringsCaseSensitive(leftIdentifier, rightIdentifier);
      if (result) return result;
    }
  }
  return compareValues(left.length, right.length);
}
var VersionRange = class _VersionRange {
  constructor(spec) {
    this._alternatives = spec ? Debug.checkDefined(parseRange(spec), "Invalid range spec.") : emptyArray;
  }
  static tryParse(text) {
    const sets = parseRange(text);
    if (sets) {
      const range = new _VersionRange("");
      range._alternatives = sets;
      return range;
    }
    return void 0;
  }
  /**
   * Tests whether a version matches the range. This is equivalent to `satisfies(version, range, { includePrerelease: true })`.
   * in `node-semver`.
   */
  test(version2) {
    if (typeof version2 === "string") version2 = new Version(version2);
    return testDisjunction(version2, this._alternatives);
  }
  toString() {
    return formatDisjunction(this._alternatives);
  }
};
var logicalOrRegExp = /\|\|/;
var whitespaceRegExp = /\s+/;
var partialRegExp = /^([x*0]|[1-9]\d*)(?:\.([x*0]|[1-9]\d*)(?:\.([x*0]|[1-9]\d*)(?:-([a-z0-9-.]+))?(?:\+([a-z0-9-.]+))?)?)?$/i;
var hyphenRegExp = /^\s*([a-z0-9-+.*]+)\s+-\s+([a-z0-9-+.*]+)\s*$/i;
var rangeRegExp = /^([~^<>=]|<=|>=)?\s*([a-z0-9-+.*]+)$/i;
function parseRange(text) {
  const alternatives = [];
  for (let range of text.trim().split(logicalOrRegExp)) {
    if (!range) continue;
    const comparators = [];
    range = range.trim();
    const match = hyphenRegExp.exec(range);
    if (match) {
      if (!parseHyphen(match[1], match[2], comparators)) return void 0;
    } else {
      for (const simple of range.split(whitespaceRegExp)) {
        const match2 = rangeRegExp.exec(simple.trim());
        if (!match2 || !parseComparator(match2[1], match2[2], comparators)) return void 0;
      }
    }
    alternatives.push(comparators);
  }
  return alternatives;
}
function parsePartial(text) {
  const match = partialRegExp.exec(text);
  if (!match) return void 0;
  const [, major, minor = "*", patch = "*", prerelease, build2] = match;
  const version2 = new Version(
    isWildcard(major) ? 0 : parseInt(major, 10),
    isWildcard(major) || isWildcard(minor) ? 0 : parseInt(minor, 10),
    isWildcard(major) || isWildcard(minor) || isWildcard(patch) ? 0 : parseInt(patch, 10),
    prerelease,
    build2
  );
  return { version: version2, major, minor, patch };
}
function parseHyphen(left, right, comparators) {
  const leftResult = parsePartial(left);
  if (!leftResult) return false;
  const rightResult = parsePartial(right);
  if (!rightResult) return false;
  if (!isWildcard(leftResult.major)) {
    comparators.push(createComparator(">=", leftResult.version));
  }
  if (!isWildcard(rightResult.major)) {
    comparators.push(
      isWildcard(rightResult.minor) ? createComparator("<", rightResult.version.increment("major")) : isWildcard(rightResult.patch) ? createComparator("<", rightResult.version.increment("minor")) : createComparator("<=", rightResult.version)
    );
  }
  return true;
}
function parseComparator(operator, text, comparators) {
  const result = parsePartial(text);
  if (!result) return false;
  const { version: version2, major, minor, patch } = result;
  if (!isWildcard(major)) {
    switch (operator) {
      case "~":
        comparators.push(createComparator(">=", version2));
        comparators.push(createComparator(
          "<",
          version2.increment(
            isWildcard(minor) ? "major" : "minor"
          )
        ));
        break;
      case "^":
        comparators.push(createComparator(">=", version2));
        comparators.push(createComparator(
          "<",
          version2.increment(
            version2.major > 0 || isWildcard(minor) ? "major" : version2.minor > 0 || isWildcard(patch) ? "minor" : "patch"
          )
        ));
        break;
      case "<":
      case ">=":
        comparators.push(
          isWildcard(minor) || isWildcard(patch) ? createComparator(operator, version2.with({ prerelease: "0" })) : createComparator(operator, version2)
        );
        break;
      case "<=":
      case ">":
        comparators.push(
          isWildcard(minor) ? createComparator(operator === "<=" ? "<" : ">=", version2.increment("major").with({ prerelease: "0" })) : isWildcard(patch) ? createComparator(operator === "<=" ? "<" : ">=", version2.increment("minor").with({ prerelease: "0" })) : createComparator(operator, version2)
        );
        break;
      case "=":
      case void 0:
        if (isWildcard(minor) || isWildcard(patch)) {
          comparators.push(createComparator(">=", version2.with({ prerelease: "0" })));
          comparators.push(createComparator("<", version2.increment(isWildcard(minor) ? "major" : "minor").with({ prerelease: "0" })));
        } else {
          comparators.push(createComparator("=", version2));
        }
        break;
      default:
        return false;
    }
  } else if (operator === "<" || operator === ">") {
    comparators.push(createComparator("<", Version.zero));
  }
  return true;
}
function isWildcard(part) {
  return part === "*" || part === "x" || part === "X";
}
function createComparator(operator, operand) {
  return { operator, operand };
}
function testDisjunction(version2, alternatives) {
  if (alternatives.length === 0) return true;
  for (const alternative of alternatives) {
    if (testAlternative(version2, alternative)) return true;
  }
  return false;
}
function testAlternative(version2, comparators) {
  for (const comparator of comparators) {
    if (!testComparator(version2, comparator.operator, comparator.operand)) return false;
  }
  return true;
}
function testComparator(version2, operator, operand) {
  const cmp = version2.compareTo(operand);
  switch (operator) {
    case "<":
      return cmp < 0;
    case "<=":
      return cmp <= 0;
    case ">":
      return cmp > 0;
    case ">=":
      return cmp >= 0;
    case "=":
      return cmp === 0;
    default:
      return Debug.assertNever(operator);
  }
}
function formatDisjunction(alternatives) {
  return map(alternatives, formatAlternative).join(" || ") || "*";
}
function formatAlternative(comparators) {
  return map(comparators, formatComparator).join(" ");
}
function formatComparator(comparator) {
  return `${comparator.operator}${comparator.operand}`;
}

// src/compiler/performanceCore.ts
function tryGetPerformance() {
  if (isNodeLikeSystem()) {
    try {
      const { performance: performance2 } = require("perf_hooks");
      if (performance2) {
        return {
          shouldWriteNativeEvents: false,
          performance: performance2
        };
      }
    } catch {
    }
  }
  if (typeof performance === "object") {
    return {
      shouldWriteNativeEvents: true,
      performance
    };
  }
  return void 0;
}
function tryGetPerformanceHooks() {
  const p = tryGetPerformance();
  if (!p) return void 0;
  const { shouldWriteNativeEvents, performance: performance2 } = p;
  const hooks = {
    shouldWriteNativeEvents,
    performance: void 0,
    performanceTime: void 0
  };
  if (typeof performance2.timeOrigin === "number" && typeof performance2.now === "function") {
    hooks.performanceTime = performance2;
  }
  if (hooks.performanceTime && typeof performance2.mark === "function" && typeof performance2.measure === "function" && typeof performance2.clearMarks === "function" && typeof performance2.clearMeasures === "function") {
    hooks.performance = performance2;
  }
  return hooks;
}
var nativePerformanceHooks = tryGetPerformanceHooks();
var nativePerformanceTime = nativePerformanceHooks == null ? void 0 : nativePerformanceHooks.performanceTime;
function tryGetNativePerformanceHooks() {
  return nativePerformanceHooks;
}
var timestamp = nativePerformanceTime ? () => nativePerformanceTime.now() : Date.now;

// src/compiler/performance.ts
var perfHooks;
var performanceImpl;
function createTimerIf(condition, measureName, startMarkName, endMarkName) {
  return condition ? createTimer(measureName, startMarkName, endMarkName) : nullTimer;
}
function createTimer(measureName, startMarkName, endMarkName) {
  let enterCount = 0;
  return {
    enter,
    exit
  };
  function enter() {
    if (++enterCount === 1) {
      mark(startMarkName);
    }
  }
  function exit() {
    if (--enterCount === 0) {
      mark(endMarkName);
      measure(measureName, startMarkName, endMarkName);
    } else if (enterCount < 0) {
      Debug.fail("enter/exit count does not match.");
    }
  }
}
var nullTimer = { enter: noop, exit: noop };
var enabled = false;
var timeorigin = timestamp();
var marks = /* @__PURE__ */ new Map();
var counts = /* @__PURE__ */ new Map();
var durations = /* @__PURE__ */ new Map();
function mark(markName) {
  if (enabled) {
    const count = counts.get(markName) ?? 0;
    counts.set(markName, count + 1);
    marks.set(markName, timestamp());
    performanceImpl == null ? void 0 : performanceImpl.mark(markName);
    if (typeof onProfilerEvent === "function") {
      onProfilerEvent(markName);
    }
  }
}
function measure(measureName, startMarkName, endMarkName) {
  if (enabled) {
    const end = (endMarkName !== void 0 ? marks.get(endMarkName) : void 0) ?? timestamp();
    const start = (startMarkName !== void 0 ? marks.get(startMarkName) : void 0) ?? timeorigin;
    const previousDuration = durations.get(measureName) || 0;
    durations.set(measureName, previousDuration + (end - start));
    performanceImpl == null ? void 0 : performanceImpl.measure(measureName, startMarkName, endMarkName);
  }
}
function getCount(markName) {
  return counts.get(markName) || 0;
}
function getDuration(measureName) {
  return durations.get(measureName) || 0;
}
function forEachMeasure(cb) {
  durations.forEach((duration, measureName) => cb(measureName, duration));
}
function forEachMark(cb) {
  marks.forEach((_time, markName) => cb(markName));
}
function clearMeasures(name) {
  if (name !== void 0) durations.delete(name);
  else durations.clear();
  performanceImpl == null ? void 0 : performanceImpl.clearMeasures(name);
}
function clearMarks(name) {
  if (name !== void 0) {
    counts.delete(name);
    marks.delete(name);
  } else {
    counts.clear();
    marks.clear();
  }
  performanceImpl == null ? void 0 : performanceImpl.clearMarks(name);
}
function isEnabled() {
  return enabled;
}
function enable(system = sys) {
  var _a;
  if (!enabled) {
    enabled = true;
    perfHooks || (perfHooks = tryGetNativePerformanceHooks());
    if (perfHooks == null ? void 0 : perfHooks.performance) {
      timeorigin = perfHooks.performance.timeOrigin;
      if (perfHooks.shouldWriteNativeEvents || ((_a = system == null ? void 0 : system.cpuProfilingEnabled) == null ? void 0 : _a.call(system)) || (system == null ? void 0 : system.debugMode)) {
        performanceImpl = perfHooks.performance;
      }
    }
  }
  return true;
}
function disable() {
  if (enabled) {
    marks.clear();
    counts.clear();
    durations.clear();
    performanceImpl = void 0;
    enabled = false;
  }
}

// src/compiler/tracing.ts
var tracing;
var tracingEnabled;
((tracingEnabled2) => {
  let fs;
  let traceCount = 0;
  let traceFd = 0;
  let mode;
  const typeCatalog = [];
  let legendPath;
  const legend = [];
  function startTracing2(tracingMode, traceDir, configFilePath) {
    Debug.assert(!tracing, "Tracing already started");
    if (fs === void 0) {
      try {
        fs = require("fs");
      } catch (e) {
        throw new Error(`tracing requires having fs
(original error: ${e.message || e})`);
      }
    }
    mode = tracingMode;
    typeCatalog.length = 0;
    if (legendPath === void 0) {
      legendPath = combinePaths(traceDir, "legend.json");
    }
    if (!fs.existsSync(traceDir)) {
      fs.mkdirSync(traceDir, { recursive: true });
    }
    const countPart = mode === "build" ? `.${process.pid}-${++traceCount}` : mode === "server" ? `.${process.pid}` : ``;
    const tracePath = combinePaths(traceDir, `trace${countPart}.json`);
    const typesPath = combinePaths(traceDir, `types${countPart}.json`);
    legend.push({
      configFilePath,
      tracePath,
      typesPath
    });
    traceFd = fs.openSync(tracePath, "w");
    tracing = tracingEnabled2;
    const meta = { cat: "__metadata", ph: "M", ts: 1e3 * timestamp(), pid: 1, tid: 1 };
    fs.writeSync(
      traceFd,
      "[\n" + [{ name: "process_name", args: { name: "tsc" }, ...meta }, { name: "thread_name", args: { name: "Main" }, ...meta }, { name: "TracingStartedInBrowser", ...meta, cat: "disabled-by-default-devtools.timeline" }].map((v) => JSON.stringify(v)).join(",\n")
    );
  }
  tracingEnabled2.startTracing = startTracing2;
  function stopTracing() {
    Debug.assert(tracing, "Tracing is not in progress");
    Debug.assert(!!typeCatalog.length === (mode !== "server"));
    fs.writeSync(traceFd, `
]
`);
    fs.closeSync(traceFd);
    tracing = void 0;
    if (typeCatalog.length) {
      dumpTypes(typeCatalog);
    } else {
      legend[legend.length - 1].typesPath = void 0;
    }
  }
  tracingEnabled2.stopTracing = stopTracing;
  function recordType(type) {
    if (mode !== "server") {
      typeCatalog.push(type);
    }
  }
  tracingEnabled2.recordType = recordType;
  let Phase;
  ((Phase2) => {
    Phase2["Parse"] = "parse";
    Phase2["Program"] = "program";
    Phase2["Bind"] = "bind";
    Phase2["Check"] = "check";
    Phase2["CheckTypes"] = "checkTypes";
    Phase2["Emit"] = "emit";
    Phase2["Session"] = "session";
  })(Phase = tracingEnabled2.Phase || (tracingEnabled2.Phase = {}));
  function instant(phase, name, args) {
    writeEvent("I", phase, name, args, `"s":"g"`);
  }
  tracingEnabled2.instant = instant;
  const eventStack = [];
  function push(phase, name, args, separateBeginAndEnd = false) {
    if (separateBeginAndEnd) {
      writeEvent("B", phase, name, args);
    }
    eventStack.push({ phase, name, args, time: 1e3 * timestamp(), separateBeginAndEnd });
  }
  tracingEnabled2.push = push;
  function pop(results) {
    Debug.assert(eventStack.length > 0);
    writeStackEvent(eventStack.length - 1, 1e3 * timestamp(), results);
    eventStack.length--;
  }
  tracingEnabled2.pop = pop;
  function popAll() {
    const endTime = 1e3 * timestamp();
    for (let i = eventStack.length - 1; i >= 0; i--) {
      writeStackEvent(i, endTime);
    }
    eventStack.length = 0;
  }
  tracingEnabled2.popAll = popAll;
  const sampleInterval = 1e3 * 10;
  function writeStackEvent(index, endTime, results) {
    const { phase, name, args, time, separateBeginAndEnd } = eventStack[index];
    if (separateBeginAndEnd) {
      Debug.assert(!results, "`results` are not supported for events with `separateBeginAndEnd`");
      writeEvent(
        "E",
        phase,
        name,
        args,
        /*extras*/
        void 0,
        endTime
      );
    } else if (sampleInterval - time % sampleInterval <= endTime - time) {
      writeEvent("X", phase, name, { ...args, results }, `"dur":${endTime - time}`, time);
    }
  }
  function writeEvent(eventType, phase, name, args, extras, time = 1e3 * timestamp()) {
    if (mode === "server" && phase === "checkTypes" /* CheckTypes */) return;
    mark("beginTracing");
    fs.writeSync(traceFd, `,
{"pid":1,"tid":1,"ph":"${eventType}","cat":"${phase}","ts":${time},"name":"${name}"`);
    if (extras) fs.writeSync(traceFd, `,${extras}`);
    if (args) fs.writeSync(traceFd, `,"args":${JSON.stringify(args)}`);
    fs.writeSync(traceFd, `}`);
    mark("endTracing");
    measure("Tracing", "beginTracing", "endTracing");
  }
  function getLocation(node) {
    const file = getSourceFileOfNode(node);
    return !file ? void 0 : {
      path: file.path,
      start: indexFromOne(getLineAndCharacterOfPosition(file, node.pos)),
      end: indexFromOne(getLineAndCharacterOfPosition(file, node.end))
    };
    function indexFromOne(lc) {
      return {
        line: lc.line + 1,
        character: lc.character + 1
      };
    }
  }
  function dumpTypes(types) {
    var _a, _b, _c, _d, _e, _f, _g, _h, _i, _j, _k, _l, _m, _n, _o, _p, _q, _r, _s;
    mark("beginDumpTypes");
    const typesPath = legend[legend.length - 1].typesPath;
    const typesFd = fs.openSync(typesPath, "w");
    const recursionIdentityMap = /* @__PURE__ */ new Map();
    fs.writeSync(typesFd, "[");
    const numTypes = types.length;
    for (let i = 0; i < numTypes; i++) {
      const type = types[i];
      const objectFlags = type.objectFlags;
      const symbol = type.aliasSymbol ?? type.symbol;
      let display;
      if (objectFlags & 16 /* Anonymous */ | type.flags & 2944 /* Literal */) {
        try {
          display = (_a = type.checker) == null ? void 0 : _a.typeToString(type);
        } catch {
          display = void 0;
        }
      }
      let indexedAccessProperties = {};
      if (type.flags & 8388608 /* IndexedAccess */) {
        const indexedAccessType = type;
        indexedAccessProperties = {
          indexedAccessObjectType: (_b = indexedAccessType.objectType) == null ? void 0 : _b.id,
          indexedAccessIndexType: (_c = indexedAccessType.indexType) == null ? void 0 : _c.id
        };
      }
      let referenceProperties = {};
      if (objectFlags & 4 /* Reference */) {
        const referenceType = type;
        referenceProperties = {
          instantiatedType: (_d = referenceType.target) == null ? void 0 : _d.id,
          typeArguments: (_e = referenceType.resolvedTypeArguments) == null ? void 0 : _e.map((t) => t.id),
          referenceLocation: getLocation(referenceType.node)
        };
      }
      let conditionalProperties = {};
      if (type.flags & 16777216 /* Conditional */) {
        const conditionalType = type;
        conditionalProperties = {
          conditionalCheckType: (_f = conditionalType.checkType) == null ? void 0 : _f.id,
          conditionalExtendsType: (_g = conditionalType.extendsType) == null ? void 0 : _g.id,
          conditionalTrueType: ((_h = conditionalType.resolvedTrueType) == null ? void 0 : _h.id) ?? -1,
          conditionalFalseType: ((_i = conditionalType.resolvedFalseType) == null ? void 0 : _i.id) ?? -1
        };
      }
      let substitutionProperties = {};
      if (type.flags & 33554432 /* Substitution */) {
        const substitutionType = type;
        substitutionProperties = {
          substitutionBaseType: (_j = substitutionType.baseType) == null ? void 0 : _j.id,
          constraintType: (_k = substitutionType.constraint) == null ? void 0 : _k.id
        };
      }
      let reverseMappedProperties = {};
      if (objectFlags & 1024 /* ReverseMapped */) {
        const reverseMappedType = type;
        reverseMappedProperties = {
          reverseMappedSourceType: (_l = reverseMappedType.source) == null ? void 0 : _l.id,
          reverseMappedMappedType: (_m = reverseMappedType.mappedType) == null ? void 0 : _m.id,
          reverseMappedConstraintType: (_n = reverseMappedType.constraintType) == null ? void 0 : _n.id
        };
      }
      let evolvingArrayProperties = {};
      if (objectFlags & 256 /* EvolvingArray */) {
        const evolvingArrayType = type;
        evolvingArrayProperties = {
          evolvingArrayElementType: evolvingArrayType.elementType.id,
          evolvingArrayFinalType: (_o = evolvingArrayType.finalArrayType) == null ? void 0 : _o.id
        };
      }
      let recursionToken;
      const recursionIdentity = type.checker.getRecursionIdentity(type);
      if (recursionIdentity) {
        recursionToken = recursionIdentityMap.get(recursionIdentity);
        if (!recursionToken) {
          recursionToken = recursionIdentityMap.size;
          recursionIdentityMap.set(recursionIdentity, recursionToken);
        }
      }
      const descriptor = {
        id: type.id,
        intrinsicName: type.intrinsicName,
        symbolName: (symbol == null ? void 0 : symbol.escapedName) && unescapeLeadingUnderscores(symbol.escapedName),
        recursionId: recursionToken,
        isTuple: objectFlags & 8 /* Tuple */ ? true : void 0,
        unionTypes: type.flags & 1048576 /* Union */ ? (_p = type.types) == null ? void 0 : _p.map((t) => t.id) : void 0,
        intersectionTypes: type.flags & 2097152 /* Intersection */ ? type.types.map((t) => t.id) : void 0,
        aliasTypeArguments: (_q = type.aliasTypeArguments) == null ? void 0 : _q.map((t) => t.id),
        keyofType: type.flags & 4194304 /* Index */ ? (_r = type.type) == null ? void 0 : _r.id : void 0,
        ...indexedAccessProperties,
        ...referenceProperties,
        ...conditionalProperties,
        ...substitutionProperties,
        ...reverseMappedProperties,
        ...evolvingArrayProperties,
        destructuringPattern: getLocation(type.pattern),
        firstDeclaration: getLocation((_s = symbol == null ? void 0 : symbol.declarations) == null ? void 0 : _s[0]),
        flags: Debug.formatTypeFlags(type.flags).split("|"),
        display
      };
      fs.writeSync(typesFd, JSON.stringify(descriptor));
      if (i < numTypes - 1) {
        fs.writeSync(typesFd, ",\n");
      }
    }
    fs.writeSync(typesFd, "]\n");
    fs.closeSync(typesFd);
    mark("endDumpTypes");
    measure("Dump types", "beginDumpTypes", "endDumpTypes");
  }
  function dumpLegend() {
    if (!legendPath) {
      return;
    }
    fs.writeFileSync(legendPath, JSON.stringify(legend));
  }
  tracingEnabled2.dumpLegend = dumpLegend;
})(tracingEnabled || (tracingEnabled = {}));
var startTracing = tracingEnabled.startTracing;
var dumpTracingLegend = tracingEnabled.dumpLegend;

// src/compiler/types.ts
var SyntaxKind = /* @__PURE__ */ ((SyntaxKind4) => {
  SyntaxKind4[SyntaxKind4["Unknown"] = 0] = "Unknown";
  SyntaxKind4[SyntaxKind4["EndOfFileToken"] = 1] = "EndOfFileToken";
  SyntaxKind4[SyntaxKind4["SingleLineCommentTrivia"] = 2] = "SingleLineCommentTrivia";
  SyntaxKind4[SyntaxKind4["MultiLineCommentTrivia"] = 3] = "MultiLineCommentTrivia";
  SyntaxKind4[SyntaxKind4["NewLineTrivia"] = 4] = "NewLineTrivia";
  SyntaxKind4[SyntaxKind4["WhitespaceTrivia"] = 5] = "WhitespaceTrivia";
  SyntaxKind4[SyntaxKind4["ShebangTrivia"] = 6] = "ShebangTrivia";
  SyntaxKind4[SyntaxKind4["ConflictMarkerTrivia"] = 7] = "ConflictMarkerTrivia";
  SyntaxKind4[SyntaxKind4["NonTextFileMarkerTrivia"] = 8] = "NonTextFileMarkerTrivia";
  SyntaxKind4[SyntaxKind4["NumericLiteral"] = 9] = "NumericLiteral";
  SyntaxKind4[SyntaxKind4["BigIntLiteral"] = 10] = "BigIntLiteral";
  SyntaxKind4[SyntaxKind4["StringLiteral"] = 11] = "StringLiteral";
  SyntaxKind4[SyntaxKind4["JsxText"] = 12] = "JsxText";
  SyntaxKind4[SyntaxKind4["JsxTextAllWhiteSpaces"] = 13] = "JsxTextAllWhiteSpaces";
  SyntaxKind4[SyntaxKind4["RegularExpressionLiteral"] = 14] = "RegularExpressionLiteral";
  SyntaxKind4[SyntaxKind4["NoSubstitutionTemplateLiteral"] = 15] = "NoSubstitutionTemplateLiteral";
  SyntaxKind4[SyntaxKind4["TemplateHead"] = 16] = "TemplateHead";
  SyntaxKind4[SyntaxKind4["TemplateMiddle"] = 17] = "TemplateMiddle";
  SyntaxKind4[SyntaxKind4["TemplateTail"] = 18] = "TemplateTail";
  SyntaxKind4[SyntaxKind4["OpenBraceToken"] = 19] = "OpenBraceToken";
  SyntaxKind4[SyntaxKind4["CloseBraceToken"] = 20] = "CloseBraceToken";
  SyntaxKind4[SyntaxKind4["OpenParenToken"] = 21] = "OpenParenToken";
  SyntaxKind4[SyntaxKind4["CloseParenToken"] = 22] = "CloseParenToken";
  SyntaxKind4[SyntaxKind4["OpenBracketToken"] = 23] = "OpenBracketToken";
  SyntaxKind4[SyntaxKind4["CloseBracketToken"] = 24] = "CloseBracketToken";
  SyntaxKind4[SyntaxKind4["DotToken"] = 25] = "DotToken";
  SyntaxKind4[SyntaxKind4["DotDotDotToken"] = 26] = "DotDotDotToken";
  SyntaxKind4[SyntaxKind4["SemicolonToken"] = 27] = "SemicolonToken";
  SyntaxKind4[SyntaxKind4["CommaToken"] = 28] = "CommaToken";
  SyntaxKind4[SyntaxKind4["QuestionDotToken"] = 29] = "QuestionDotToken";
  SyntaxKind4[SyntaxKind4["LessThanToken"] = 30] = "LessThanToken";
  SyntaxKind4[SyntaxKind4["LessThanSlashToken"] = 31] = "LessThanSlashToken";
  SyntaxKind4[SyntaxKind4["GreaterThanToken"] = 32] = "GreaterThanToken";
  SyntaxKind4[SyntaxKind4["LessThanEqualsToken"] = 33] = "LessThanEqualsToken";
  SyntaxKind4[SyntaxKind4["GreaterThanEqualsToken"] = 34] = "GreaterThanEqualsToken";
  SyntaxKind4[SyntaxKind4["EqualsEqualsToken"] = 35] = "EqualsEqualsToken";
  SyntaxKind4[SyntaxKind4["ExclamationEqualsToken"] = 36] = "ExclamationEqualsToken";
  SyntaxKind4[SyntaxKind4["EqualsEqualsEqualsToken"] = 37] = "EqualsEqualsEqualsToken";
  SyntaxKind4[SyntaxKind4["ExclamationEqualsEqualsToken"] = 38] = "ExclamationEqualsEqualsToken";
  SyntaxKind4[SyntaxKind4["EqualsGreaterThanToken"] = 39] = "EqualsGreaterThanToken";
  SyntaxKind4[SyntaxKind4["PlusToken"] = 40] = "PlusToken";
  SyntaxKind4[SyntaxKind4["MinusToken"] = 41] = "MinusToken";
  SyntaxKind4[SyntaxKind4["AsteriskToken"] = 42] = "AsteriskToken";
  SyntaxKind4[SyntaxKind4["AsteriskAsteriskToken"] = 43] = "AsteriskAsteriskToken";
  SyntaxKind4[SyntaxKind4["SlashToken"] = 44] = "SlashToken";
  SyntaxKind4[SyntaxKind4["PercentToken"] = 45] = "PercentToken";
  SyntaxKind4[SyntaxKind4["PlusPlusToken"] = 46] = "PlusPlusToken";
  SyntaxKind4[SyntaxKind4["MinusMinusToken"] = 47] = "MinusMinusToken";
  SyntaxKind4[SyntaxKind4["LessThanLessThanToken"] = 48] = "LessThanLessThanToken";
  SyntaxKind4[SyntaxKind4["GreaterThanGreaterThanToken"] = 49] = "GreaterThanGreaterThanToken";
  SyntaxKind4[SyntaxKind4["GreaterThanGreaterThanGreaterThanToken"] = 50] = "GreaterThanGreaterThanGreaterThanToken";
  SyntaxKind4[SyntaxKind4["AmpersandToken"] = 51] = "AmpersandToken";
  SyntaxKind4[SyntaxKind4["BarToken"] = 52] = "BarToken";
  SyntaxKind4[SyntaxKind4["CaretToken"] = 53] = "CaretToken";
  SyntaxKind4[SyntaxKind4["ExclamationToken"] = 54] = "ExclamationToken";
  SyntaxKind4[SyntaxKind4["TildeToken"] = 55] = "TildeToken";
  SyntaxKind4[SyntaxKind4["AmpersandAmpersandToken"] = 56] = "AmpersandAmpersandToken";
  SyntaxKind4[SyntaxKind4["BarBarToken"] = 57] = "BarBarToken";
  SyntaxKind4[SyntaxKind4["QuestionToken"] = 58] = "QuestionToken";
  SyntaxKind4[SyntaxKind4["ColonToken"] = 59] = "ColonToken";
  SyntaxKind4[SyntaxKind4["AtToken"] = 60] = "AtToken";
  SyntaxKind4[SyntaxKind4["QuestionQuestionToken"] = 61] = "QuestionQuestionToken";
  SyntaxKind4[SyntaxKind4["BacktickToken"] = 62] = "BacktickToken";
  SyntaxKind4[SyntaxKind4["HashToken"] = 63] = "HashToken";
  SyntaxKind4[SyntaxKind4["EqualsToken"] = 64] = "EqualsToken";
  SyntaxKind4[SyntaxKind4["PlusEqualsToken"] = 65] = "PlusEqualsToken";
  SyntaxKind4[SyntaxKind4["MinusEqualsToken"] = 66] = "MinusEqualsToken";
  SyntaxKind4[SyntaxKind4["AsteriskEqualsToken"] = 67] = "AsteriskEqualsToken";
  SyntaxKind4[SyntaxKind4["AsteriskAsteriskEqualsToken"] = 68] = "AsteriskAsteriskEqualsToken";
  SyntaxKind4[SyntaxKind4["SlashEqualsToken"] = 69] = "SlashEqualsToken";
  SyntaxKind4[SyntaxKind4["PercentEqualsToken"] = 70] = "PercentEqualsToken";
  SyntaxKind4[SyntaxKind4["LessThanLessThanEqualsToken"] = 71] = "LessThanLessThanEqualsToken";
  SyntaxKind4[SyntaxKind4["GreaterThanGreaterThanEqualsToken"] = 72] = "GreaterThanGreaterThanEqualsToken";
  SyntaxKind4[SyntaxKind4["GreaterThanGreaterThanGreaterThanEqualsToken"] = 73] = "GreaterThanGreaterThanGreaterThanEqualsToken";
  SyntaxKind4[SyntaxKind4["AmpersandEqualsToken"] = 74] = "AmpersandEqualsToken";
  SyntaxKind4[SyntaxKind4["BarEqualsToken"] = 75] = "BarEqualsToken";
  SyntaxKind4[SyntaxKind4["BarBarEqualsToken"] = 76] = "BarBarEqualsToken";
  SyntaxKind4[SyntaxKind4["AmpersandAmpersandEqualsToken"] = 77] = "AmpersandAmpersandEqualsToken";
  SyntaxKind4[SyntaxKind4["QuestionQuestionEqualsToken"] = 78] = "QuestionQuestionEqualsToken";
  SyntaxKind4[SyntaxKind4["CaretEqualsToken"] = 79] = "CaretEqualsToken";
  SyntaxKind4[SyntaxKind4["Identifier"] = 80] = "Identifier";
  SyntaxKind4[SyntaxKind4["PrivateIdentifier"] = 81] = "PrivateIdentifier";
  SyntaxKind4[SyntaxKind4["JSDocCommentTextToken"] = 82] = "JSDocCommentTextToken";
  SyntaxKind4[SyntaxKind4["BreakKeyword"] = 83] = "BreakKeyword";
  SyntaxKind4[SyntaxKind4["CaseKeyword"] = 84] = "CaseKeyword";
  SyntaxKind4[SyntaxKind4["CatchKeyword"] = 85] = "CatchKeyword";
  SyntaxKind4[SyntaxKind4["ClassKeyword"] = 86] = "ClassKeyword";
  SyntaxKind4[SyntaxKind4["ConstKeyword"] = 87] = "ConstKeyword";
  SyntaxKind4[SyntaxKind4["ContinueKeyword"] = 88] = "ContinueKeyword";
  SyntaxKind4[SyntaxKind4["DebuggerKeyword"] = 89] = "DebuggerKeyword";
  SyntaxKind4[SyntaxKind4["DefaultKeyword"] = 90] = "DefaultKeyword";
  SyntaxKind4[SyntaxKind4["DeleteKeyword"] = 91] = "DeleteKeyword";
  SyntaxKind4[SyntaxKind4["DoKeyword"] = 92] = "DoKeyword";
  SyntaxKind4[SyntaxKind4["ElseKeyword"] = 93] = "ElseKeyword";
  SyntaxKind4[SyntaxKind4["EnumKeyword"] = 94] = "EnumKeyword";
  SyntaxKind4[SyntaxKind4["ExportKeyword"] = 95] = "ExportKeyword";
  SyntaxKind4[SyntaxKind4["ExtendsKeyword"] = 96] = "ExtendsKeyword";
  SyntaxKind4[SyntaxKind4["FalseKeyword"] = 97] = "FalseKeyword";
  SyntaxKind4[SyntaxKind4["FinallyKeyword"] = 98] = "FinallyKeyword";
  SyntaxKind4[SyntaxKind4["ForKeyword"] = 99] = "ForKeyword";
  SyntaxKind4[SyntaxKind4["FunctionKeyword"] = 100] = "FunctionKeyword";
  SyntaxKind4[SyntaxKind4["IfKeyword"] = 101] = "IfKeyword";
  SyntaxKind4[SyntaxKind4["ImportKeyword"] = 102] = "ImportKeyword";
  SyntaxKind4[SyntaxKind4["InKeyword"] = 103] = "InKeyword";
  SyntaxKind4[SyntaxKind4["InstanceOfKeyword"] = 104] = "InstanceOfKeyword";
  SyntaxKind4[SyntaxKind4["NewKeyword"] = 105] = "NewKeyword";
  SyntaxKind4[SyntaxKind4["NullKeyword"] = 106] = "NullKeyword";
  SyntaxKind4[SyntaxKind4["ReturnKeyword"] = 107] = "ReturnKeyword";
  SyntaxKind4[SyntaxKind4["SuperKeyword"] = 108] = "SuperKeyword";
  SyntaxKind4[SyntaxKind4["SwitchKeyword"] = 109] = "SwitchKeyword";
  SyntaxKind4[SyntaxKind4["ThisKeyword"] = 110] = "ThisKeyword";
  SyntaxKind4[SyntaxKind4["ThrowKeyword"] = 111] = "ThrowKeyword";
  SyntaxKind4[SyntaxKind4["TrueKeyword"] = 112] = "TrueKeyword";
  SyntaxKind4[SyntaxKind4["TryKeyword"] = 113] = "TryKeyword";
  SyntaxKind4[SyntaxKind4["TypeOfKeyword"] = 114] = "TypeOfKeyword";
  SyntaxKind4[SyntaxKind4["VarKeyword"] = 115] = "VarKeyword";
  SyntaxKind4[SyntaxKind4["VoidKeyword"] = 116] = "VoidKeyword";
  SyntaxKind4[SyntaxKind4["WhileKeyword"] = 117] = "WhileKeyword";
  SyntaxKind4[SyntaxKind4["WithKeyword"] = 118] = "WithKeyword";
  SyntaxKind4[SyntaxKind4["ImplementsKeyword"] = 119] = "ImplementsKeyword";
  SyntaxKind4[SyntaxKind4["InterfaceKeyword"] = 120] = "InterfaceKeyword";
  SyntaxKind4[SyntaxKind4["LetKeyword"] = 121] = "LetKeyword";
  SyntaxKind4[SyntaxKind4["PackageKeyword"] = 122] = "PackageKeyword";
  SyntaxKind4[SyntaxKind4["PrivateKeyword"] = 123] = "PrivateKeyword";
  SyntaxKind4[SyntaxKind4["ProtectedKeyword"] = 124] = "ProtectedKeyword";
  SyntaxKind4[SyntaxKind4["PublicKeyword"] = 125] = "PublicKeyword";
  SyntaxKind4[SyntaxKind4["StaticKeyword"] = 126] = "StaticKeyword";
  SyntaxKind4[SyntaxKind4["YieldKeyword"] = 127] = "YieldKeyword";
  SyntaxKind4[SyntaxKind4["AbstractKeyword"] = 128] = "AbstractKeyword";
  SyntaxKind4[SyntaxKind4["AccessorKeyword"] = 129] = "AccessorKeyword";
  SyntaxKind4[SyntaxKind4["AsKeyword"] = 130] = "AsKeyword";
  SyntaxKind4[SyntaxKind4["AssertsKeyword"] = 131] = "AssertsKeyword";
  SyntaxKind4[SyntaxKind4["AssertKeyword"] = 132] = "AssertKeyword";
  SyntaxKind4[SyntaxKind4["AnyKeyword"] = 133] = "AnyKeyword";
  SyntaxKind4[SyntaxKind4["AsyncKeyword"] = 134] = "AsyncKeyword";
  SyntaxKind4[SyntaxKind4["AwaitKeyword"] = 135] = "AwaitKeyword";
  SyntaxKind4[SyntaxKind4["BooleanKeyword"] = 136] = "BooleanKeyword";
  SyntaxKind4[SyntaxKind4["ConstructorKeyword"] = 137] = "ConstructorKeyword";
  SyntaxKind4[SyntaxKind4["DeclareKeyword"] = 138] = "DeclareKeyword";
  SyntaxKind4[SyntaxKind4["GetKeyword"] = 139] = "GetKeyword";
  SyntaxKind4[SyntaxKind4["InferKeyword"] = 140] = "InferKeyword";
  SyntaxKind4[SyntaxKind4["IntrinsicKeyword"] = 141] = "IntrinsicKeyword";
  SyntaxKind4[SyntaxKind4["IsKeyword"] = 142] = "IsKeyword";
  SyntaxKind4[SyntaxKind4["KeyOfKeyword"] = 143] = "KeyOfKeyword";
  SyntaxKind4[SyntaxKind4["ModuleKeyword"] = 144] = "ModuleKeyword";
  SyntaxKind4[SyntaxKind4["NamespaceKeyword"] = 145] = "NamespaceKeyword";
  SyntaxKind4[SyntaxKind4["NeverKeyword"] = 146] = "NeverKeyword";
  SyntaxKind4[SyntaxKind4["OutKeyword"] = 147] = "OutKeyword";
  SyntaxKind4[SyntaxKind4["ReadonlyKeyword"] = 148] = "ReadonlyKeyword";
  SyntaxKind4[SyntaxKind4["RequireKeyword"] = 149] = "RequireKeyword";
  SyntaxKind4[SyntaxKind4["NumberKeyword"] = 150] = "NumberKeyword";
  SyntaxKind4[SyntaxKind4["ObjectKeyword"] = 151] = "ObjectKeyword";
  SyntaxKind4[SyntaxKind4["SatisfiesKeyword"] = 152] = "SatisfiesKeyword";
  SyntaxKind4[SyntaxKind4["SetKeyword"] = 153] = "SetKeyword";
  SyntaxKind4[SyntaxKind4["StringKeyword"] = 154] = "StringKeyword";
  SyntaxKind4[SyntaxKind4["SymbolKeyword"] = 155] = "SymbolKeyword";
  SyntaxKind4[SyntaxKind4["TypeKeyword"] = 156] = "TypeKeyword";
  SyntaxKind4[SyntaxKind4["UndefinedKeyword"] = 157] = "UndefinedKeyword";
  SyntaxKind4[SyntaxKind4["UniqueKeyword"] = 158] = "UniqueKeyword";
  SyntaxKind4[SyntaxKind4["UnknownKeyword"] = 159] = "UnknownKeyword";
  SyntaxKind4[SyntaxKind4["UsingKeyword"] = 160] = "UsingKeyword";
  SyntaxKind4[SyntaxKind4["FromKeyword"] = 161] = "FromKeyword";
  SyntaxKind4[SyntaxKind4["GlobalKeyword"] = 162] = "GlobalKeyword";
  SyntaxKind4[SyntaxKind4["BigIntKeyword"] = 163] = "BigIntKeyword";
  SyntaxKind4[SyntaxKind4["OverrideKeyword"] = 164] = "OverrideKeyword";
  SyntaxKind4[SyntaxKind4["OfKeyword"] = 165] = "OfKeyword";
  SyntaxKind4[SyntaxKind4["QualifiedName"] = 166] = "QualifiedName";
  SyntaxKind4[SyntaxKind4["ComputedPropertyName"] = 167] = "ComputedPropertyName";
  SyntaxKind4[SyntaxKind4["TypeParameter"] = 168] = "TypeParameter";
  SyntaxKind4[SyntaxKind4["Parameter"] = 169] = "Parameter";
  SyntaxKind4[SyntaxKind4["Decorator"] = 170] = "Decorator";
  SyntaxKind4[SyntaxKind4["PropertySignature"] = 171] = "PropertySignature";
  SyntaxKind4[SyntaxKind4["PropertyDeclaration"] = 172] = "PropertyDeclaration";
  SyntaxKind4[SyntaxKind4["MethodSignature"] = 173] = "MethodSignature";
  SyntaxKind4[SyntaxKind4["MethodDeclaration"] = 174] = "MethodDeclaration";
  SyntaxKind4[SyntaxKind4["ClassStaticBlockDeclaration"] = 175] = "ClassStaticBlockDeclaration";
  SyntaxKind4[SyntaxKind4["Constructor"] = 176] = "Constructor";
  SyntaxKind4[SyntaxKind4["GetAccessor"] = 177] = "GetAccessor";
  SyntaxKind4[SyntaxKind4["SetAccessor"] = 178] = "SetAccessor";
  SyntaxKind4[SyntaxKind4["CallSignature"] = 179] = "CallSignature";
  SyntaxKind4[SyntaxKind4["ConstructSignature"] = 180] = "ConstructSignature";
  SyntaxKind4[SyntaxKind4["IndexSignature"] = 181] = "IndexSignature";
  SyntaxKind4[SyntaxKind4["TypePredicate"] = 182] = "TypePredicate";
  SyntaxKind4[SyntaxKind4["TypeReference"] = 183] = "TypeReference";
  SyntaxKind4[SyntaxKind4["FunctionType"] = 184] = "FunctionType";
  SyntaxKind4[SyntaxKind4["ConstructorType"] = 185] = "ConstructorType";
  SyntaxKind4[SyntaxKind4["TypeQuery"] = 186] = "TypeQuery";
  SyntaxKind4[SyntaxKind4["TypeLiteral"] = 187] = "TypeLiteral";
  SyntaxKind4[SyntaxKind4["ArrayType"] = 188] = "ArrayType";
  SyntaxKind4[SyntaxKind4["TupleType"] = 189] = "TupleType";
  SyntaxKind4[SyntaxKind4["OptionalType"] = 190] = "OptionalType";
  SyntaxKind4[SyntaxKind4["RestType"] = 191] = "RestType";
  SyntaxKind4[SyntaxKind4["UnionType"] = 192] = "UnionType";
  SyntaxKind4[SyntaxKind4["IntersectionType"] = 193] = "IntersectionType";
  SyntaxKind4[SyntaxKind4["ConditionalType"] = 194] = "ConditionalType";
  SyntaxKind4[SyntaxKind4["InferType"] = 195] = "InferType";
  SyntaxKind4[SyntaxKind4["ParenthesizedType"] = 196] = "ParenthesizedType";
  SyntaxKind4[SyntaxKind4["ThisType"] = 197] = "ThisType";
  SyntaxKind4[SyntaxKind4["TypeOperator"] = 198] = "TypeOperator";
  SyntaxKind4[SyntaxKind4["IndexedAccessType"] = 199] = "IndexedAccessType";
  SyntaxKind4[SyntaxKind4["MappedType"] = 200] = "MappedType";
  SyntaxKind4[SyntaxKind4["LiteralType"] = 201] = "LiteralType";
  SyntaxKind4[SyntaxKind4["NamedTupleMember"] = 202] = "NamedTupleMember";
  SyntaxKind4[SyntaxKind4["TemplateLiteralType"] = 203] = "TemplateLiteralType";
  SyntaxKind4[SyntaxKind4["TemplateLiteralTypeSpan"] = 204] = "TemplateLiteralTypeSpan";
  SyntaxKind4[SyntaxKind4["ImportType"] = 205] = "ImportType";
  SyntaxKind4[SyntaxKind4["ObjectBindingPattern"] = 206] = "ObjectBindingPattern";
  SyntaxKind4[SyntaxKind4["ArrayBindingPattern"] = 207] = "ArrayBindingPattern";
  SyntaxKind4[SyntaxKind4["BindingElement"] = 208] = "BindingElement";
  SyntaxKind4[SyntaxKind4["ArrayLiteralExpression"] = 209] = "ArrayLiteralExpression";
  SyntaxKind4[SyntaxKind4["ObjectLiteralExpression"] = 210] = "ObjectLiteralExpression";
  SyntaxKind4[SyntaxKind4["PropertyAccessExpression"] = 211] = "PropertyAccessExpression";
  SyntaxKind4[SyntaxKind4["ElementAccessExpression"] = 212] = "ElementAccessExpression";
  SyntaxKind4[SyntaxKind4["CallExpression"] = 213] = "CallExpression";
  SyntaxKind4[SyntaxKind4["NewExpression"] = 214] = "NewExpression";
  SyntaxKind4[SyntaxKind4["TaggedTemplateExpression"] = 215] = "TaggedTemplateExpression";
  SyntaxKind4[SyntaxKind4["TypeAssertionExpression"] = 216] = "TypeAssertionExpression";
  SyntaxKind4[SyntaxKind4["ParenthesizedExpression"] = 217] = "ParenthesizedExpression";
  SyntaxKind4[SyntaxKind4["FunctionExpression"] = 218] = "FunctionExpression";
  SyntaxKind4[SyntaxKind4["ArrowFunction"] = 219] = "ArrowFunction";
  SyntaxKind4[SyntaxKind4["DeleteExpression"] = 220] = "DeleteExpression";
  SyntaxKind4[SyntaxKind4["TypeOfExpression"] = 221] = "TypeOfExpression";
  SyntaxKind4[SyntaxKind4["VoidExpression"] = 222] = "VoidExpression";
  SyntaxKind4[SyntaxKind4["AwaitExpression"] = 223] = "AwaitExpression";
  SyntaxKind4[SyntaxKind4["PrefixUnaryExpression"] = 224] = "PrefixUnaryExpression";
  SyntaxKind4[SyntaxKind4["PostfixUnaryExpression"] = 225] = "PostfixUnaryExpression";
  SyntaxKind4[SyntaxKind4["BinaryExpression"] = 226] = "BinaryExpression";
  SyntaxKind4[SyntaxKind4["ConditionalExpression"] = 227] = "ConditionalExpression";
  SyntaxKind4[SyntaxKind4["TemplateExpression"] = 228] = "TemplateExpression";
  SyntaxKind4[SyntaxKind4["YieldExpression"] = 229] = "YieldExpression";
  SyntaxKind4[SyntaxKind4["SpreadElement"] = 230] = "SpreadElement";
  SyntaxKind4[SyntaxKind4["ClassExpression"] = 231] = "ClassExpression";
  SyntaxKind4[SyntaxKind4["OmittedExpression"] = 232] = "OmittedExpression";
  SyntaxKind4[SyntaxKind4["ExpressionWithTypeArguments"] = 233] = "ExpressionWithTypeArguments";
  SyntaxKind4[SyntaxKind4["AsExpression"] = 234] = "AsExpression";
  SyntaxKind4[SyntaxKind4["NonNullExpression"] = 235] = "NonNullExpression";
  SyntaxKind4[SyntaxKind4["MetaProperty"] = 236] = "MetaProperty";
  SyntaxKind4[SyntaxKind4["SyntheticExpression"] = 237] = "SyntheticExpression";
  SyntaxKind4[SyntaxKind4["SatisfiesExpression"] = 238] = "SatisfiesExpression";
  SyntaxKind4[SyntaxKind4["TemplateSpan"] = 239] = "TemplateSpan";
  SyntaxKind4[SyntaxKind4["SemicolonClassElement"] = 240] = "SemicolonClassElement";
  SyntaxKind4[SyntaxKind4["Block"] = 241] = "Block";
  SyntaxKind4[SyntaxKind4["EmptyStatement"] = 242] = "EmptyStatement";
  SyntaxKind4[SyntaxKind4["VariableStatement"] = 243] = "VariableStatement";
  SyntaxKind4[SyntaxKind4["ExpressionStatement"] = 244] = "ExpressionStatement";
  SyntaxKind4[SyntaxKind4["IfStatement"] = 245] = "IfStatement";
  SyntaxKind4[SyntaxKind4["DoStatement"] = 246] = "DoStatement";
  SyntaxKind4[SyntaxKind4["WhileStatement"] = 247] = "WhileStatement";
  SyntaxKind4[SyntaxKind4["ForStatement"] = 248] = "ForStatement";
  SyntaxKind4[SyntaxKind4["ForInStatement"] = 249] = "ForInStatement";
  SyntaxKind4[SyntaxKind4["ForOfStatement"] = 250] = "ForOfStatement";
  SyntaxKind4[SyntaxKind4["ContinueStatement"] = 251] = "ContinueStatement";
  SyntaxKind4[SyntaxKind4["BreakStatement"] = 252] = "BreakStatement";
  SyntaxKind4[SyntaxKind4["ReturnStatement"] = 253] = "ReturnStatement";
  SyntaxKind4[SyntaxKind4["WithStatement"] = 254] = "WithStatement";
  SyntaxKind4[SyntaxKind4["SwitchStatement"] = 255] = "SwitchStatement";
  SyntaxKind4[SyntaxKind4["LabeledStatement"] = 256] = "LabeledStatement";
  SyntaxKind4[SyntaxKind4["ThrowStatement"] = 257] = "ThrowStatement";
  SyntaxKind4[SyntaxKind4["TryStatement"] = 258] = "TryStatement";
  SyntaxKind4[SyntaxKind4["DebuggerStatement"] = 259] = "DebuggerStatement";
  SyntaxKind4[SyntaxKind4["VariableDeclaration"] = 260] = "VariableDeclaration";
  SyntaxKind4[SyntaxKind4["VariableDeclarationList"] = 261] = "VariableDeclarationList";
  SyntaxKind4[SyntaxKind4["FunctionDeclaration"] = 262] = "FunctionDeclaration";
  SyntaxKind4[SyntaxKind4["ClassDeclaration"] = 263] = "ClassDeclaration";
  SyntaxKind4[SyntaxKind4["InterfaceDeclaration"] = 264] = "InterfaceDeclaration";
  SyntaxKind4[SyntaxKind4["TypeAliasDeclaration"] = 265] = "TypeAliasDeclaration";
  SyntaxKind4[SyntaxKind4["EnumDeclaration"] = 266] = "EnumDeclaration";
  SyntaxKind4[SyntaxKind4["ModuleDeclaration"] = 267] = "ModuleDeclaration";
  SyntaxKind4[SyntaxKind4["ModuleBlock"] = 268] = "ModuleBlock";
  SyntaxKind4[SyntaxKind4["CaseBlock"] = 269] = "CaseBlock";
  SyntaxKind4[SyntaxKind4["NamespaceExportDeclaration"] = 270] = "NamespaceExportDeclaration";
  SyntaxKind4[SyntaxKind4["ImportEqualsDeclaration"] = 271] = "ImportEqualsDeclaration";
  SyntaxKind4[SyntaxKind4["ImportDeclaration"] = 272] = "ImportDeclaration";
  SyntaxKind4[SyntaxKind4["ImportClause"] = 273] = "ImportClause";
  SyntaxKind4[SyntaxKind4["NamespaceImport"] = 274] = "NamespaceImport";
  SyntaxKind4[SyntaxKind4["NamedImports"] = 275] = "NamedImports";
  SyntaxKind4[SyntaxKind4["ImportSpecifier"] = 276] = "ImportSpecifier";
  SyntaxKind4[SyntaxKind4["ExportAssignment"] = 277] = "ExportAssignment";
  SyntaxKind4[SyntaxKind4["ExportDeclaration"] = 278] = "ExportDeclaration";
  SyntaxKind4[SyntaxKind4["NamedExports"] = 279] = "NamedExports";
  SyntaxKind4[SyntaxKind4["NamespaceExport"] = 280] = "NamespaceExport";
  SyntaxKind4[SyntaxKind4["ExportSpecifier"] = 281] = "ExportSpecifier";
  SyntaxKind4[SyntaxKind4["MissingDeclaration"] = 282] = "MissingDeclaration";
  SyntaxKind4[SyntaxKind4["ExternalModuleReference"] = 283] = "ExternalModuleReference";
  SyntaxKind4[SyntaxKind4["JsxElement"] = 284] = "JsxElement";
  SyntaxKind4[SyntaxKind4["JsxSelfClosingElement"] = 285] = "JsxSelfClosingElement";
  SyntaxKind4[SyntaxKind4["JsxOpeningElement"] = 286] = "JsxOpeningElement";
  SyntaxKind4[SyntaxKind4["JsxClosingElement"] = 287] = "JsxClosingElement";
  SyntaxKind4[SyntaxKind4["JsxFragment"] = 288] = "JsxFragment";
  SyntaxKind4[SyntaxKind4["JsxOpeningFragment"] = 289] = "JsxOpeningFragment";
  SyntaxKind4[SyntaxKind4["JsxClosingFragment"] = 290] = "JsxClosingFragment";
  SyntaxKind4[SyntaxKind4["JsxAttribute"] = 291] = "JsxAttribute";
  SyntaxKind4[SyntaxKind4["JsxAttributes"] = 292] = "JsxAttributes";
  SyntaxKind4[SyntaxKind4["JsxSpreadAttribute"] = 293] = "JsxSpreadAttribute";
  SyntaxKind4[SyntaxKind4["JsxExpression"] = 294] = "JsxExpression";
  SyntaxKind4[SyntaxKind4["JsxNamespacedName"] = 295] = "JsxNamespacedName";
  SyntaxKind4[SyntaxKind4["CaseClause"] = 296] = "CaseClause";
  SyntaxKind4[SyntaxKind4["DefaultClause"] = 297] = "DefaultClause";
  SyntaxKind4[SyntaxKind4["HeritageClause"] = 298] = "HeritageClause";
  SyntaxKind4[SyntaxKind4["CatchClause"] = 299] = "CatchClause";
  SyntaxKind4[SyntaxKind4["ImportAttributes"] = 300] = "ImportAttributes";
  SyntaxKind4[SyntaxKind4["ImportAttribute"] = 301] = "ImportAttribute";
  SyntaxKind4[SyntaxKind4["AssertClause"] = 300 /* ImportAttributes */] = "AssertClause";
  SyntaxKind4[SyntaxKind4["AssertEntry"] = 301 /* ImportAttribute */] = "AssertEntry";
  SyntaxKind4[SyntaxKind4["ImportTypeAssertionContainer"] = 302] = "ImportTypeAssertionContainer";
  SyntaxKind4[SyntaxKind4["PropertyAssignment"] = 303] = "PropertyAssignment";
  SyntaxKind4[SyntaxKind4["ShorthandPropertyAssignment"] = 304] = "ShorthandPropertyAssignment";
  SyntaxKind4[SyntaxKind4["SpreadAssignment"] = 305] = "SpreadAssignment";
  SyntaxKind4[SyntaxKind4["EnumMember"] = 306] = "EnumMember";
  SyntaxKind4[SyntaxKind4["SourceFile"] = 307] = "SourceFile";
  SyntaxKind4[SyntaxKind4["Bundle"] = 308] = "Bundle";
  SyntaxKind4[SyntaxKind4["JSDocTypeExpression"] = 309] = "JSDocTypeExpression";
  SyntaxKind4[SyntaxKind4["JSDocNameReference"] = 310] = "JSDocNameReference";
  SyntaxKind4[SyntaxKind4["JSDocMemberName"] = 311] = "JSDocMemberName";
  SyntaxKind4[SyntaxKind4["JSDocAllType"] = 312] = "JSDocAllType";
  SyntaxKind4[SyntaxKind4["JSDocUnknownType"] = 313] = "JSDocUnknownType";
  SyntaxKind4[SyntaxKind4["JSDocNullableType"] = 314] = "JSDocNullableType";
  SyntaxKind4[SyntaxKind4["JSDocNonNullableType"] = 315] = "JSDocNonNullableType";
  SyntaxKind4[SyntaxKind4["JSDocOptionalType"] = 316] = "JSDocOptionalType";
  SyntaxKind4[SyntaxKind4["JSDocFunctionType"] = 317] = "JSDocFunctionType";
  SyntaxKind4[SyntaxKind4["JSDocVariadicType"] = 318] = "JSDocVariadicType";
  SyntaxKind4[SyntaxKind4["JSDocNamepathType"] = 319] = "JSDocNamepathType";
  SyntaxKind4[SyntaxKind4["JSDoc"] = 320] = "JSDoc";
  SyntaxKind4[SyntaxKind4["JSDocComment"] = 320 /* JSDoc */] = "JSDocComment";
  SyntaxKind4[SyntaxKind4["JSDocText"] = 321] = "JSDocText";
  SyntaxKind4[SyntaxKind4["JSDocTypeLiteral"] = 322] = "JSDocTypeLiteral";
  SyntaxKind4[SyntaxKind4["JSDocSignature"] = 323] = "JSDocSignature";
  SyntaxKind4[SyntaxKind4["JSDocLink"] = 324] = "JSDocLink";
  SyntaxKind4[SyntaxKind4["JSDocLinkCode"] = 325] = "JSDocLinkCode";
  SyntaxKind4[SyntaxKind4["JSDocLinkPlain"] = 326] = "JSDocLinkPlain";
  SyntaxKind4[SyntaxKind4["JSDocTag"] = 327] = "JSDocTag";
  SyntaxKind4[SyntaxKind4["JSDocAugmentsTag"] = 328] = "JSDocAugmentsTag";
  SyntaxKind4[SyntaxKind4["JSDocImplementsTag"] = 329] = "JSDocImplementsTag";
  SyntaxKind4[SyntaxKind4["JSDocAuthorTag"] = 330] = "JSDocAuthorTag";
  SyntaxKind4[SyntaxKind4["JSDocDeprecatedTag"] = 331] = "JSDocDeprecatedTag";
  SyntaxKind4[SyntaxKind4["JSDocClassTag"] = 332] = "JSDocClassTag";
  SyntaxKind4[SyntaxKind4["JSDocPublicTag"] = 333] = "JSDocPublicTag";
  SyntaxKind4[SyntaxKind4["JSDocPrivateTag"] = 334] = "JSDocPrivateTag";
  SyntaxKind4[SyntaxKind4["JSDocProtectedTag"] = 335] = "JSDocProtectedTag";
  SyntaxKind4[SyntaxKind4["JSDocReadonlyTag"] = 336] = "JSDocReadonlyTag";
  SyntaxKind4[SyntaxKind4["JSDocOverrideTag"] = 337] = "JSDocOverrideTag";
  SyntaxKind4[SyntaxKind4["JSDocCallbackTag"] = 338] = "JSDocCallbackTag";
  SyntaxKind4[SyntaxKind4["JSDocOverloadTag"] = 339] = "JSDocOverloadTag";
  SyntaxKind4[SyntaxKind4["JSDocEnumTag"] = 340] = "JSDocEnumTag";
  SyntaxKind4[SyntaxKind4["JSDocParameterTag"] = 341] = "JSDocParameterTag";
  SyntaxKind4[SyntaxKind4["JSDocReturnTag"] = 342] = "JSDocReturnTag";
  SyntaxKind4[SyntaxKind4["JSDocThisTag"] = 343] = "JSDocThisTag";
  SyntaxKind4[SyntaxKind4["JSDocTypeTag"] = 344] = "JSDocTypeTag";
  SyntaxKind4[SyntaxKind4["JSDocTemplateTag"] = 345] = "JSDocTemplateTag";
  SyntaxKind4[SyntaxKind4["JSDocTypedefTag"] = 346] = "JSDocTypedefTag";
  SyntaxKind4[SyntaxKind4["JSDocSeeTag"] = 347] = "JSDocSeeTag";
  SyntaxKind4[SyntaxKind4["JSDocPropertyTag"] = 348] = "JSDocPropertyTag";
  SyntaxKind4[SyntaxKind4["JSDocThrowsTag"] = 349] = "JSDocThrowsTag";
  SyntaxKind4[SyntaxKind4["JSDocSatisfiesTag"] = 350] = "JSDocSatisfiesTag";
  SyntaxKind4[SyntaxKind4["JSDocImportTag"] = 351] = "JSDocImportTag";
  SyntaxKind4[SyntaxKind4["SyntaxList"] = 352] = "SyntaxList";
  SyntaxKind4[SyntaxKind4["NotEmittedStatement"] = 353] = "NotEmittedStatement";
  SyntaxKind4[SyntaxKind4["NotEmittedTypeElement"] = 354] = "NotEmittedTypeElement";
  SyntaxKind4[SyntaxKind4["PartiallyEmittedExpression"] = 355] = "PartiallyEmittedExpression";
  SyntaxKind4[SyntaxKind4["CommaListExpression"] = 356] = "CommaListExpression";
  SyntaxKind4[SyntaxKind4["SyntheticReferenceExpression"] = 357] = "SyntheticReferenceExpression";
  SyntaxKind4[SyntaxKind4["Count"] = 358] = "Count";
  SyntaxKind4[SyntaxKind4["FirstAssignment"] = 64 /* EqualsToken */] = "FirstAssignment";
  SyntaxKind4[SyntaxKind4["LastAssignment"] = 79 /* CaretEqualsToken */] = "LastAssignment";
  SyntaxKind4[SyntaxKind4["FirstCompoundAssignment"] = 65 /* PlusEqualsToken */] = "FirstCompoundAssignment";
  SyntaxKind4[SyntaxKind4["LastCompoundAssignment"] = 79 /* CaretEqualsToken */] = "LastCompoundAssignment";
  SyntaxKind4[SyntaxKind4["FirstReservedWord"] = 83 /* BreakKeyword */] = "FirstReservedWord";
  SyntaxKind4[SyntaxKind4["LastReservedWord"] = 118 /* WithKeyword */] = "LastReservedWord";
  SyntaxKind4[SyntaxKind4["FirstKeyword"] = 83 /* BreakKeyword */] = "FirstKeyword";
  SyntaxKind4[SyntaxKind4["LastKeyword"] = 165 /* OfKeyword */] = "LastKeyword";
  SyntaxKind4[SyntaxKind4["FirstFutureReservedWord"] = 119 /* ImplementsKeyword */] = "FirstFutureReservedWord";
  SyntaxKind4[SyntaxKind4["LastFutureReservedWord"] = 127 /* YieldKeyword */] = "LastFutureReservedWord";
  SyntaxKind4[SyntaxKind4["FirstTypeNode"] = 182 /* TypePredicate */] = "FirstTypeNode";
  SyntaxKind4[SyntaxKind4["LastTypeNode"] = 205 /* ImportType */] = "LastTypeNode";
  SyntaxKind4[SyntaxKind4["FirstPunctuation"] = 19 /* OpenBraceToken */] = "FirstPunctuation";
  SyntaxKind4[SyntaxKind4["LastPunctuation"] = 79 /* CaretEqualsToken */] = "LastPunctuation";
  SyntaxKind4[SyntaxKind4["FirstToken"] = 0 /* Unknown */] = "FirstToken";
  SyntaxKind4[SyntaxKind4["LastToken"] = 165 /* LastKeyword */] = "LastToken";
  SyntaxKind4[SyntaxKind4["FirstTriviaToken"] = 2 /* SingleLineCommentTrivia */] = "FirstTriviaToken";
  SyntaxKind4[SyntaxKind4["LastTriviaToken"] = 7 /* ConflictMarkerTrivia */] = "LastTriviaToken";
  SyntaxKind4[SyntaxKind4["FirstLiteralToken"] = 9 /* NumericLiteral */] = "FirstLiteralToken";
  SyntaxKind4[SyntaxKind4["LastLiteralToken"] = 15 /* NoSubstitutionTemplateLiteral */] = "LastLiteralToken";
  SyntaxKind4[SyntaxKind4["FirstTemplateToken"] = 15 /* NoSubstitutionTemplateLiteral */] = "FirstTemplateToken";
  SyntaxKind4[SyntaxKind4["LastTemplateToken"] = 18 /* TemplateTail */] = "LastTemplateToken";
  SyntaxKind4[SyntaxKind4["FirstBinaryOperator"] = 30 /* LessThanToken */] = "FirstBinaryOperator";
  SyntaxKind4[SyntaxKind4["LastBinaryOperator"] = 79 /* CaretEqualsToken */] = "LastBinaryOperator";
  SyntaxKind4[SyntaxKind4["FirstStatement"] = 243 /* VariableStatement */] = "FirstStatement";
  SyntaxKind4[SyntaxKind4["LastStatement"] = 259 /* DebuggerStatement */] = "LastStatement";
  SyntaxKind4[SyntaxKind4["FirstNode"] = 166 /* QualifiedName */] = "FirstNode";
  SyntaxKind4[SyntaxKind4["FirstJSDocNode"] = 309 /* JSDocTypeExpression */] = "FirstJSDocNode";
  SyntaxKind4[SyntaxKind4["LastJSDocNode"] = 351 /* JSDocImportTag */] = "LastJSDocNode";
  SyntaxKind4[SyntaxKind4["FirstJSDocTagNode"] = 327 /* JSDocTag */] = "FirstJSDocTagNode";
  SyntaxKind4[SyntaxKind4["LastJSDocTagNode"] = 351 /* JSDocImportTag */] = "LastJSDocTagNode";
  SyntaxKind4[SyntaxKind4["FirstContextualKeyword"] = 128 /* AbstractKeyword */] = "FirstContextualKeyword";
  SyntaxKind4[SyntaxKind4["LastContextualKeyword"] = 165 /* OfKeyword */] = "LastContextualKeyword";
  return SyntaxKind4;
})(SyntaxKind || {});
var NodeFlags = /* @__PURE__ */ ((NodeFlags3) => {
  NodeFlags3[NodeFlags3["None"] = 0] = "None";
  NodeFlags3[NodeFlags3["Let"] = 1] = "Let";
  NodeFlags3[NodeFlags3["Const"] = 2] = "Const";
  NodeFlags3[NodeFlags3["Using"] = 4] = "Using";
  NodeFlags3[NodeFlags3["AwaitUsing"] = 6] = "AwaitUsing";
  NodeFlags3[NodeFlags3["NestedNamespace"] = 8] = "NestedNamespace";
  NodeFlags3[NodeFlags3["Synthesized"] = 16] = "Synthesized";
  NodeFlags3[NodeFlags3["Namespace"] = 32] = "Namespace";
  NodeFlags3[NodeFlags3["OptionalChain"] = 64] = "OptionalChain";
  NodeFlags3[NodeFlags3["ExportContext"] = 128] = "ExportContext";
  NodeFlags3[NodeFlags3["ContainsThis"] = 256] = "ContainsThis";
  NodeFlags3[NodeFlags3["HasImplicitReturn"] = 512] = "HasImplicitReturn";
  NodeFlags3[NodeFlags3["HasExplicitReturn"] = 1024] = "HasExplicitReturn";
  NodeFlags3[NodeFlags3["GlobalAugmentation"] = 2048] = "GlobalAugmentation";
  NodeFlags3[NodeFlags3["HasAsyncFunctions"] = 4096] = "HasAsyncFunctions";
  NodeFlags3[NodeFlags3["DisallowInContext"] = 8192] = "DisallowInContext";
  NodeFlags3[NodeFlags3["YieldContext"] = 16384] = "YieldContext";
  NodeFlags3[NodeFlags3["DecoratorContext"] = 32768] = "DecoratorContext";
  NodeFlags3[NodeFlags3["AwaitContext"] = 65536] = "AwaitContext";
  NodeFlags3[NodeFlags3["DisallowConditionalTypesContext"] = 131072] = "DisallowConditionalTypesContext";
  NodeFlags3[NodeFlags3["ThisNodeHasError"] = 262144] = "ThisNodeHasError";
  NodeFlags3[NodeFlags3["JavaScriptFile"] = 524288] = "JavaScriptFile";
  NodeFlags3[NodeFlags3["ThisNodeOrAnySubNodesHasError"] = 1048576] = "ThisNodeOrAnySubNodesHasError";
  NodeFlags3[NodeFlags3["HasAggregatedChildData"] = 2097152] = "HasAggregatedChildData";
  NodeFlags3[NodeFlags3["PossiblyContainsDynamicImport"] = 4194304] = "PossiblyContainsDynamicImport";
  NodeFlags3[NodeFlags3["PossiblyContainsImportMeta"] = 8388608] = "PossiblyContainsImportMeta";
  NodeFlags3[NodeFlags3["JSDoc"] = 16777216] = "JSDoc";
  NodeFlags3[NodeFlags3["Ambient"] = 33554432] = "Ambient";
  NodeFlags3[NodeFlags3["InWithStatement"] = 67108864] = "InWithStatement";
  NodeFlags3[NodeFlags3["JsonFile"] = 134217728] = "JsonFile";
  NodeFlags3[NodeFlags3["TypeCached"] = 268435456] = "TypeCached";
  NodeFlags3[NodeFlags3["Deprecated"] = 536870912] = "Deprecated";
  NodeFlags3[NodeFlags3["BlockScoped"] = 7] = "BlockScoped";
  NodeFlags3[NodeFlags3["Constant"] = 6] = "Constant";
  NodeFlags3[NodeFlags3["ReachabilityCheckFlags"] = 1536] = "ReachabilityCheckFlags";
  NodeFlags3[NodeFlags3["ReachabilityAndEmitFlags"] = 5632] = "ReachabilityAndEmitFlags";
  NodeFlags3[NodeFlags3["ContextFlags"] = 101441536] = "ContextFlags";
  NodeFlags3[NodeFlags3["TypeExcludesFlags"] = 81920] = "TypeExcludesFlags";
  NodeFlags3[NodeFlags3["PermanentlySetIncrementalFlags"] = 12582912] = "PermanentlySetIncrementalFlags";
  NodeFlags3[NodeFlags3["IdentifierHasExtendedUnicodeEscape"] = 256 /* ContainsThis */] = "IdentifierHasExtendedUnicodeEscape";
  NodeFlags3[NodeFlags3["IdentifierIsInJSDocNamespace"] = 4096 /* HasAsyncFunctions */] = "IdentifierIsInJSDocNamespace";
  return NodeFlags3;
})(NodeFlags || {});
var ModifierFlags = /* @__PURE__ */ ((ModifierFlags3) => {
  ModifierFlags3[ModifierFlags3["None"] = 0] = "None";
  ModifierFlags3[ModifierFlags3["Public"] = 1] = "Public";
  ModifierFlags3[ModifierFlags3["Private"] = 2] = "Private";
  ModifierFlags3[ModifierFlags3["Protected"] = 4] = "Protected";
  ModifierFlags3[ModifierFlags3["Readonly"] = 8] = "Readonly";
  ModifierFlags3[ModifierFlags3["Override"] = 16] = "Override";
  ModifierFlags3[ModifierFlags3["Export"] = 32] = "Export";
  ModifierFlags3[ModifierFlags3["Abstract"] = 64] = "Abstract";
  ModifierFlags3[ModifierFlags3["Ambient"] = 128] = "Ambient";
  ModifierFlags3[ModifierFlags3["Static"] = 256] = "Static";
  ModifierFlags3[ModifierFlags3["Accessor"] = 512] = "Accessor";
  ModifierFlags3[ModifierFlags3["Async"] = 1024] = "Async";
  ModifierFlags3[ModifierFlags3["Default"] = 2048] = "Default";
  ModifierFlags3[ModifierFlags3["Const"] = 4096] = "Const";
  ModifierFlags3[ModifierFlags3["In"] = 8192] = "In";
  ModifierFlags3[ModifierFlags3["Out"] = 16384] = "Out";
  ModifierFlags3[ModifierFlags3["Decorator"] = 32768] = "Decorator";
  ModifierFlags3[ModifierFlags3["Deprecated"] = 65536] = "Deprecated";
  ModifierFlags3[ModifierFlags3["JSDocPublic"] = 8388608] = "JSDocPublic";
  ModifierFlags3[ModifierFlags3["JSDocPrivate"] = 16777216] = "JSDocPrivate";
  ModifierFlags3[ModifierFlags3["JSDocProtected"] = 33554432] = "JSDocProtected";
  ModifierFlags3[ModifierFlags3["JSDocReadonly"] = 67108864] = "JSDocReadonly";
  ModifierFlags3[ModifierFlags3["JSDocOverride"] = 134217728] = "JSDocOverride";
  ModifierFlags3[ModifierFlags3["SyntacticOrJSDocModifiers"] = 31] = "SyntacticOrJSDocModifiers";
  ModifierFlags3[ModifierFlags3["SyntacticOnlyModifiers"] = 65504] = "SyntacticOnlyModifiers";
  ModifierFlags3[ModifierFlags3["SyntacticModifiers"] = 65535] = "SyntacticModifiers";
  ModifierFlags3[ModifierFlags3["JSDocCacheOnlyModifiers"] = 260046848] = "JSDocCacheOnlyModifiers";
  ModifierFlags3[ModifierFlags3["JSDocOnlyModifiers"] = 65536 /* Deprecated */] = "JSDocOnlyModifiers";
  ModifierFlags3[ModifierFlags3["NonCacheOnlyModifiers"] = 131071] = "NonCacheOnlyModifiers";
  ModifierFlags3[ModifierFlags3["HasComputedJSDocModifiers"] = 268435456] = "HasComputedJSDocModifiers";
  ModifierFlags3[ModifierFlags3["HasComputedFlags"] = 536870912] = "HasComputedFlags";
  ModifierFlags3[ModifierFlags3["AccessibilityModifier"] = 7] = "AccessibilityModifier";
  ModifierFlags3[ModifierFlags3["ParameterPropertyModifier"] = 31] = "ParameterPropertyModifier";
  ModifierFlags3[ModifierFlags3["NonPublicAccessibilityModifier"] = 6] = "NonPublicAccessibilityModifier";
  ModifierFlags3[ModifierFlags3["TypeScriptModifier"] = 28895] = "TypeScriptModifier";
  ModifierFlags3[ModifierFlags3["ExportDefault"] = 2080] = "ExportDefault";
  ModifierFlags3[ModifierFlags3["All"] = 131071] = "All";
  ModifierFlags3[ModifierFlags3["Modifier"] = 98303] = "Modifier";
  return ModifierFlags3;
})(ModifierFlags || {});
var RelationComparisonResult = /* @__PURE__ */ ((RelationComparisonResult3) => {
  RelationComparisonResult3[RelationComparisonResult3["None"] = 0] = "None";
  RelationComparisonResult3[RelationComparisonResult3["Succeeded"] = 1] = "Succeeded";
  RelationComparisonResult3[RelationComparisonResult3["Failed"] = 2] = "Failed";
  RelationComparisonResult3[RelationComparisonResult3["ReportsUnmeasurable"] = 8] = "ReportsUnmeasurable";
  RelationComparisonResult3[RelationComparisonResult3["ReportsUnreliable"] = 16] = "ReportsUnreliable";
  RelationComparisonResult3[RelationComparisonResult3["ReportsMask"] = 24] = "ReportsMask";
  RelationComparisonResult3[RelationComparisonResult3["ComplexityOverflow"] = 32] = "ComplexityOverflow";
  RelationComparisonResult3[RelationComparisonResult3["StackDepthOverflow"] = 64] = "StackDepthOverflow";
  RelationComparisonResult3[RelationComparisonResult3["Overflow"] = 96] = "Overflow";
  return RelationComparisonResult3;
})(RelationComparisonResult || {});
var GeneratedIdentifierFlags = /* @__PURE__ */ ((GeneratedIdentifierFlags2) => {
  GeneratedIdentifierFlags2[GeneratedIdentifierFlags2["None"] = 0] = "None";
  GeneratedIdentifierFlags2[GeneratedIdentifierFlags2["Auto"] = 1] = "Auto";
  GeneratedIdentifierFlags2[GeneratedIdentifierFlags2["Loop"] = 2] = "Loop";
  GeneratedIdentifierFlags2[GeneratedIdentifierFlags2["Unique"] = 3] = "Unique";
  GeneratedIdentifierFlags2[GeneratedIdentifierFlags2["Node"] = 4] = "Node";
  GeneratedIdentifierFlags2[GeneratedIdentifierFlags2["KindMask"] = 7] = "KindMask";
  GeneratedIdentifierFlags2[GeneratedIdentifierFlags2["ReservedInNestedScopes"] = 8] = "ReservedInNestedScopes";
  GeneratedIdentifierFlags2[GeneratedIdentifierFlags2["Optimistic"] = 16] = "Optimistic";
  GeneratedIdentifierFlags2[GeneratedIdentifierFlags2["FileLevel"] = 32] = "FileLevel";
  GeneratedIdentifierFlags2[GeneratedIdentifierFlags2["AllowNameSubstitution"] = 64] = "AllowNameSubstitution";
  return GeneratedIdentifierFlags2;
})(GeneratedIdentifierFlags || {});
var FlowFlags = /* @__PURE__ */ ((FlowFlags2) => {
  FlowFlags2[FlowFlags2["Unreachable"] = 1] = "Unreachable";
  FlowFlags2[FlowFlags2["Start"] = 2] = "Start";
  FlowFlags2[FlowFlags2["BranchLabel"] = 4] = "BranchLabel";
  FlowFlags2[FlowFlags2["LoopLabel"] = 8] = "LoopLabel";
  FlowFlags2[FlowFlags2["Assignment"] = 16] = "Assignment";
  FlowFlags2[FlowFlags2["TrueCondition"] = 32] = "TrueCondition";
  FlowFlags2[FlowFlags2["FalseCondition"] = 64] = "FalseCondition";
  FlowFlags2[FlowFlags2["SwitchClause"] = 128] = "SwitchClause";
  FlowFlags2[FlowFlags2["ArrayMutation"] = 256] = "ArrayMutation";
  FlowFlags2[FlowFlags2["Call"] = 512] = "Call";
  FlowFlags2[FlowFlags2["ReduceLabel"] = 1024] = "ReduceLabel";
  FlowFlags2[FlowFlags2["Referenced"] = 2048] = "Referenced";
  FlowFlags2[FlowFlags2["Shared"] = 4096] = "Shared";
  FlowFlags2[FlowFlags2["Label"] = 12] = "Label";
  FlowFlags2[FlowFlags2["Condition"] = 96] = "Condition";
  return FlowFlags2;
})(FlowFlags || {});
var OperationCanceledException = class {
};
var FileIncludeKind = /* @__PURE__ */ ((FileIncludeKind2) => {
  FileIncludeKind2[FileIncludeKind2["RootFile"] = 0] = "RootFile";
  FileIncludeKind2[FileIncludeKind2["SourceFromProjectReference"] = 1] = "SourceFromProjectReference";
  FileIncludeKind2[FileIncludeKind2["OutputFromProjectReference"] = 2] = "OutputFromProjectReference";
  FileIncludeKind2[FileIncludeKind2["Import"] = 3] = "Import";
  FileIncludeKind2[FileIncludeKind2["ReferenceFile"] = 4] = "ReferenceFile";
  FileIncludeKind2[FileIncludeKind2["TypeReferenceDirective"] = 5] = "TypeReferenceDirective";
  FileIncludeKind2[FileIncludeKind2["LibFile"] = 6] = "LibFile";
  FileIncludeKind2[FileIncludeKind2["LibReferenceDirective"] = 7] = "LibReferenceDirective";
  FileIncludeKind2[FileIncludeKind2["AutomaticTypeDirectiveFile"] = 8] = "AutomaticTypeDirectiveFile";
  return FileIncludeKind2;
})(FileIncludeKind || {});
var SymbolFlags = /* @__PURE__ */ ((SymbolFlags2) => {
  SymbolFlags2[SymbolFlags2["None"] = 0] = "None";
  SymbolFlags2[SymbolFlags2["FunctionScopedVariable"] = 1] = "FunctionScopedVariable";
  SymbolFlags2[SymbolFlags2["BlockScopedVariable"] = 2] = "BlockScopedVariable";
  SymbolFlags2[SymbolFlags2["Property"] = 4] = "Property";
  SymbolFlags2[SymbolFlags2["EnumMember"] = 8] = "EnumMember";
  SymbolFlags2[SymbolFlags2["Function"] = 16] = "Function";
  SymbolFlags2[SymbolFlags2["Class"] = 32] = "Class";
  SymbolFlags2[SymbolFlags2["Interface"] = 64] = "Interface";
  SymbolFlags2[SymbolFlags2["ConstEnum"] = 128] = "ConstEnum";
  SymbolFlags2[SymbolFlags2["RegularEnum"] = 256] = "RegularEnum";
  SymbolFlags2[SymbolFlags2["ValueModule"] = 512] = "ValueModule";
  SymbolFlags2[SymbolFlags2["NamespaceModule"] = 1024] = "NamespaceModule";
  SymbolFlags2[SymbolFlags2["TypeLiteral"] = 2048] = "TypeLiteral";
  SymbolFlags2[SymbolFlags2["ObjectLiteral"] = 4096] = "ObjectLiteral";
  SymbolFlags2[SymbolFlags2["Method"] = 8192] = "Method";
  SymbolFlags2[SymbolFlags2["Constructor"] = 16384] = "Constructor";
  SymbolFlags2[SymbolFlags2["GetAccessor"] = 32768] = "GetAccessor";
  SymbolFlags2[SymbolFlags2["SetAccessor"] = 65536] = "SetAccessor";
  SymbolFlags2[SymbolFlags2["Signature"] = 131072] = "Signature";
  SymbolFlags2[SymbolFlags2["TypeParameter"] = 262144] = "TypeParameter";
  SymbolFlags2[SymbolFlags2["TypeAlias"] = 524288] = "TypeAlias";
  SymbolFlags2[SymbolFlags2["ExportValue"] = 1048576] = "ExportValue";
  SymbolFlags2[SymbolFlags2["Alias"] = 2097152] = "Alias";
  SymbolFlags2[SymbolFlags2["Prototype"] = 4194304] = "Prototype";
  SymbolFlags2[SymbolFlags2["ExportStar"] = 8388608] = "ExportStar";
  SymbolFlags2[SymbolFlags2["Optional"] = 16777216] = "Optional";
  SymbolFlags2[SymbolFlags2["Transient"] = 33554432] = "Transient";
  SymbolFlags2[SymbolFlags2["Assignment"] = 67108864] = "Assignment";
  SymbolFlags2[SymbolFlags2["ModuleExports"] = 134217728] = "ModuleExports";
  SymbolFlags2[SymbolFlags2["All"] = -1] = "All";
  SymbolFlags2[SymbolFlags2["Enum"] = 384] = "Enum";
  SymbolFlags2[SymbolFlags2["Variable"] = 3] = "Variable";
  SymbolFlags2[SymbolFlags2["Value"] = 111551] = "Value";
  SymbolFlags2[SymbolFlags2["Type"] = 788968] = "Type";
  SymbolFlags2[SymbolFlags2["Namespace"] = 1920] = "Namespace";
  SymbolFlags2[SymbolFlags2["Module"] = 1536] = "Module";
  SymbolFlags2[SymbolFlags2["Accessor"] = 98304] = "Accessor";
  SymbolFlags2[SymbolFlags2["FunctionScopedVariableExcludes"] = 111550] = "FunctionScopedVariableExcludes";
  SymbolFlags2[SymbolFlags2["BlockScopedVariableExcludes"] = 111551 /* Value */] = "BlockScopedVariableExcludes";
  SymbolFlags2[SymbolFlags2["ParameterExcludes"] = 111551 /* Value */] = "ParameterExcludes";
  SymbolFlags2[SymbolFlags2["PropertyExcludes"] = 0 /* None */] = "PropertyExcludes";
  SymbolFlags2[SymbolFlags2["EnumMemberExcludes"] = 900095] = "EnumMemberExcludes";
  SymbolFlags2[SymbolFlags2["FunctionExcludes"] = 110991] = "FunctionExcludes";
  SymbolFlags2[SymbolFlags2["ClassExcludes"] = 899503] = "ClassExcludes";
  SymbolFlags2[SymbolFlags2["InterfaceExcludes"] = 788872] = "InterfaceExcludes";
  SymbolFlags2[SymbolFlags2["RegularEnumExcludes"] = 899327] = "RegularEnumExcludes";
  SymbolFlags2[SymbolFlags2["ConstEnumExcludes"] = 899967] = "ConstEnumExcludes";
  SymbolFlags2[SymbolFlags2["ValueModuleExcludes"] = 110735] = "ValueModuleExcludes";
  SymbolFlags2[SymbolFlags2["NamespaceModuleExcludes"] = 0] = "NamespaceModuleExcludes";
  SymbolFlags2[SymbolFlags2["MethodExcludes"] = 103359] = "MethodExcludes";
  SymbolFlags2[SymbolFlags2["GetAccessorExcludes"] = 46015] = "GetAccessorExcludes";
  SymbolFlags2[SymbolFlags2["SetAccessorExcludes"] = 78783] = "SetAccessorExcludes";
  SymbolFlags2[SymbolFlags2["AccessorExcludes"] = 13247] = "AccessorExcludes";
  SymbolFlags2[SymbolFlags2["TypeParameterExcludes"] = 526824] = "TypeParameterExcludes";
  SymbolFlags2[SymbolFlags2["TypeAliasExcludes"] = 788968 /* Type */] = "TypeAliasExcludes";
  SymbolFlags2[SymbolFlags2["AliasExcludes"] = 2097152 /* Alias */] = "AliasExcludes";
  SymbolFlags2[SymbolFlags2["ModuleMember"] = 2623475] = "ModuleMember";
  SymbolFlags2[SymbolFlags2["ExportHasLocal"] = 944] = "ExportHasLocal";
  SymbolFlags2[SymbolFlags2["BlockScoped"] = 418] = "BlockScoped";
  SymbolFlags2[SymbolFlags2["PropertyOrAccessor"] = 98308] = "PropertyOrAccessor";
  SymbolFlags2[SymbolFlags2["ClassMember"] = 106500] = "ClassMember";
  SymbolFlags2[SymbolFlags2["ExportSupportsDefaultModifier"] = 112] = "ExportSupportsDefaultModifier";
  SymbolFlags2[SymbolFlags2["ExportDoesNotSupportDefaultModifier"] = -113] = "ExportDoesNotSupportDefaultModifier";
  SymbolFlags2[SymbolFlags2["Classifiable"] = 2885600] = "Classifiable";
  SymbolFlags2[SymbolFlags2["LateBindingContainer"] = 6256] = "LateBindingContainer";
  return SymbolFlags2;
})(SymbolFlags || {});
var NodeCheckFlags = /* @__PURE__ */ ((NodeCheckFlags3) => {
  NodeCheckFlags3[NodeCheckFlags3["None"] = 0] = "None";
  NodeCheckFlags3[NodeCheckFlags3["TypeChecked"] = 1] = "TypeChecked";
  NodeCheckFlags3[NodeCheckFlags3["LexicalThis"] = 2] = "LexicalThis";
  NodeCheckFlags3[NodeCheckFlags3["CaptureThis"] = 4] = "CaptureThis";
  NodeCheckFlags3[NodeCheckFlags3["CaptureNewTarget"] = 8] = "CaptureNewTarget";
  NodeCheckFlags3[NodeCheckFlags3["SuperInstance"] = 16] = "SuperInstance";
  NodeCheckFlags3[NodeCheckFlags3["SuperStatic"] = 32] = "SuperStatic";
  NodeCheckFlags3[NodeCheckFlags3["ContextChecked"] = 64] = "ContextChecked";
  NodeCheckFlags3[NodeCheckFlags3["MethodWithSuperPropertyAccessInAsync"] = 128] = "MethodWithSuperPropertyAccessInAsync";
  NodeCheckFlags3[NodeCheckFlags3["MethodWithSuperPropertyAssignmentInAsync"] = 256] = "MethodWithSuperPropertyAssignmentInAsync";
  NodeCheckFlags3[NodeCheckFlags3["CaptureArguments"] = 512] = "CaptureArguments";
  NodeCheckFlags3[NodeCheckFlags3["EnumValuesComputed"] = 1024] = "EnumValuesComputed";
  NodeCheckFlags3[NodeCheckFlags3["LexicalModuleMergesWithClass"] = 2048] = "LexicalModuleMergesWithClass";
  NodeCheckFlags3[NodeCheckFlags3["LoopWithCapturedBlockScopedBinding"] = 4096] = "LoopWithCapturedBlockScopedBinding";
  NodeCheckFlags3[NodeCheckFlags3["ContainsCapturedBlockScopeBinding"] = 8192] = "ContainsCapturedBlockScopeBinding";
  NodeCheckFlags3[NodeCheckFlags3["CapturedBlockScopedBinding"] = 16384] = "CapturedBlockScopedBinding";
  NodeCheckFlags3[NodeCheckFlags3["BlockScopedBindingInLoop"] = 32768] = "BlockScopedBindingInLoop";
  NodeCheckFlags3[NodeCheckFlags3["NeedsLoopOutParameter"] = 65536] = "NeedsLoopOutParameter";
  NodeCheckFlags3[NodeCheckFlags3["AssignmentsMarked"] = 131072] = "AssignmentsMarked";
  NodeCheckFlags3[NodeCheckFlags3["ContainsConstructorReference"] = 262144] = "ContainsConstructorReference";
  NodeCheckFlags3[NodeCheckFlags3["ConstructorReference"] = 536870912] = "ConstructorReference";
  NodeCheckFlags3[NodeCheckFlags3["ContainsClassWithPrivateIdentifiers"] = 1048576] = "ContainsClassWithPrivateIdentifiers";
  NodeCheckFlags3[NodeCheckFlags3["ContainsSuperPropertyInStaticInitializer"] = 2097152] = "ContainsSuperPropertyInStaticInitializer";
  NodeCheckFlags3[NodeCheckFlags3["InCheckIdentifier"] = 4194304] = "InCheckIdentifier";
  NodeCheckFlags3[NodeCheckFlags3["PartiallyTypeChecked"] = 8388608] = "PartiallyTypeChecked";
  NodeCheckFlags3[NodeCheckFlags3["LazyFlags"] = 539358128] = "LazyFlags";
  return NodeCheckFlags3;
})(NodeCheckFlags || {});
var TypeFlags = /* @__PURE__ */ ((TypeFlags2) => {
  TypeFlags2[TypeFlags2["Any"] = 1] = "Any";
  TypeFlags2[TypeFlags2["Unknown"] = 2] = "Unknown";
  TypeFlags2[TypeFlags2["String"] = 4] = "String";
  TypeFlags2[TypeFlags2["Number"] = 8] = "Number";
  TypeFlags2[TypeFlags2["Boolean"] = 16] = "Boolean";
  TypeFlags2[TypeFlags2["Enum"] = 32] = "Enum";
  TypeFlags2[TypeFlags2["BigInt"] = 64] = "BigInt";
  TypeFlags2[TypeFlags2["StringLiteral"] = 128] = "StringLiteral";
  TypeFlags2[TypeFlags2["NumberLiteral"] = 256] = "NumberLiteral";
  TypeFlags2[TypeFlags2["BooleanLiteral"] = 512] = "BooleanLiteral";
  TypeFlags2[TypeFlags2["EnumLiteral"] = 1024] = "EnumLiteral";
  TypeFlags2[TypeFlags2["BigIntLiteral"] = 2048] = "BigIntLiteral";
  TypeFlags2[TypeFlags2["ESSymbol"] = 4096] = "ESSymbol";
  TypeFlags2[TypeFlags2["UniqueESSymbol"] = 8192] = "UniqueESSymbol";
  TypeFlags2[TypeFlags2["Void"] = 16384] = "Void";
  TypeFlags2[TypeFlags2["Undefined"] = 32768] = "Undefined";
  TypeFlags2[TypeFlags2["Null"] = 65536] = "Null";
  TypeFlags2[TypeFlags2["Never"] = 131072] = "Never";
  TypeFlags2[TypeFlags2["TypeParameter"] = 262144] = "TypeParameter";
  TypeFlags2[TypeFlags2["Object"] = 524288] = "Object";
  TypeFlags2[TypeFlags2["Union"] = 1048576] = "Union";
  TypeFlags2[TypeFlags2["Intersection"] = 2097152] = "Intersection";
  TypeFlags2[TypeFlags2["Index"] = 4194304] = "Index";
  TypeFlags2[TypeFlags2["IndexedAccess"] = 8388608] = "IndexedAccess";
  TypeFlags2[TypeFlags2["Conditional"] = 16777216] = "Conditional";
  TypeFlags2[TypeFlags2["Substitution"] = 33554432] = "Substitution";
  TypeFlags2[TypeFlags2["NonPrimitive"] = 67108864] = "NonPrimitive";
  TypeFlags2[TypeFlags2["TemplateLiteral"] = 134217728] = "TemplateLiteral";
  TypeFlags2[TypeFlags2["StringMapping"] = 268435456] = "StringMapping";
  TypeFlags2[TypeFlags2["Reserved1"] = 536870912] = "Reserved1";
  TypeFlags2[TypeFlags2["Reserved2"] = 1073741824] = "Reserved2";
  TypeFlags2[TypeFlags2["AnyOrUnknown"] = 3] = "AnyOrUnknown";
  TypeFlags2[TypeFlags2["Nullable"] = 98304] = "Nullable";
  TypeFlags2[TypeFlags2["Literal"] = 2944] = "Literal";
  TypeFlags2[TypeFlags2["Unit"] = 109472] = "Unit";
  TypeFlags2[TypeFlags2["Freshable"] = 2976] = "Freshable";
  TypeFlags2[TypeFlags2["StringOrNumberLiteral"] = 384] = "StringOrNumberLiteral";
  TypeFlags2[TypeFlags2["StringOrNumberLiteralOrUnique"] = 8576] = "StringOrNumberLiteralOrUnique";
  TypeFlags2[TypeFlags2["DefinitelyFalsy"] = 117632] = "DefinitelyFalsy";
  TypeFlags2[TypeFlags2["PossiblyFalsy"] = 117724] = "PossiblyFalsy";
  TypeFlags2[TypeFlags2["Intrinsic"] = 67359327] = "Intrinsic";
  TypeFlags2[TypeFlags2["StringLike"] = 402653316] = "StringLike";
  TypeFlags2[TypeFlags2["NumberLike"] = 296] = "NumberLike";
  TypeFlags2[TypeFlags2["BigIntLike"] = 2112] = "BigIntLike";
  TypeFlags2[TypeFlags2["BooleanLike"] = 528] = "BooleanLike";
  TypeFlags2[TypeFlags2["EnumLike"] = 1056] = "EnumLike";
  TypeFlags2[TypeFlags2["ESSymbolLike"] = 12288] = "ESSymbolLike";
  TypeFlags2[TypeFlags2["VoidLike"] = 49152] = "VoidLike";
  TypeFlags2[TypeFlags2["Primitive"] = 402784252] = "Primitive";
  TypeFlags2[TypeFlags2["DefinitelyNonNullable"] = 470302716] = "DefinitelyNonNullable";
  TypeFlags2[TypeFlags2["DisjointDomains"] = 469892092] = "DisjointDomains";
  TypeFlags2[TypeFlags2["UnionOrIntersection"] = 3145728] = "UnionOrIntersection";
  TypeFlags2[TypeFlags2["StructuredType"] = 3670016] = "StructuredType";
  TypeFlags2[TypeFlags2["TypeVariable"] = 8650752] = "TypeVariable";
  TypeFlags2[TypeFlags2["InstantiableNonPrimitive"] = 58982400] = "InstantiableNonPrimitive";
  TypeFlags2[TypeFlags2["InstantiablePrimitive"] = 406847488] = "InstantiablePrimitive";
  TypeFlags2[TypeFlags2["Instantiable"] = 465829888] = "Instantiable";
  TypeFlags2[TypeFlags2["StructuredOrInstantiable"] = 469499904] = "StructuredOrInstantiable";
  TypeFlags2[TypeFlags2["ObjectFlagsType"] = 3899393] = "ObjectFlagsType";
  TypeFlags2[TypeFlags2["Simplifiable"] = 25165824] = "Simplifiable";
  TypeFlags2[TypeFlags2["Singleton"] = 67358815] = "Singleton";
  TypeFlags2[TypeFlags2["Narrowable"] = 536624127] = "Narrowable";
  TypeFlags2[TypeFlags2["IncludesMask"] = 473694207] = "IncludesMask";
  TypeFlags2[TypeFlags2["IncludesMissingType"] = 262144 /* TypeParameter */] = "IncludesMissingType";
  TypeFlags2[TypeFlags2["IncludesNonWideningType"] = 4194304 /* Index */] = "IncludesNonWideningType";
  TypeFlags2[TypeFlags2["IncludesWildcard"] = 8388608 /* IndexedAccess */] = "IncludesWildcard";
  TypeFlags2[TypeFlags2["IncludesEmptyObject"] = 16777216 /* Conditional */] = "IncludesEmptyObject";
  TypeFlags2[TypeFlags2["IncludesInstantiable"] = 33554432 /* Substitution */] = "IncludesInstantiable";
  TypeFlags2[TypeFlags2["IncludesConstrainedTypeVariable"] = 536870912 /* Reserved1 */] = "IncludesConstrainedTypeVariable";
  TypeFlags2[TypeFlags2["IncludesError"] = 1073741824 /* Reserved2 */] = "IncludesError";
  TypeFlags2[TypeFlags2["NotPrimitiveUnion"] = 36323331] = "NotPrimitiveUnion";
  return TypeFlags2;
})(TypeFlags || {});
var ObjectFlags = /* @__PURE__ */ ((ObjectFlags3) => {
  ObjectFlags3[ObjectFlags3["None"] = 0] = "None";
  ObjectFlags3[ObjectFlags3["Class"] = 1] = "Class";
  ObjectFlags3[ObjectFlags3["Interface"] = 2] = "Interface";
  ObjectFlags3[ObjectFlags3["Reference"] = 4] = "Reference";
  ObjectFlags3[ObjectFlags3["Tuple"] = 8] = "Tuple";
  ObjectFlags3[ObjectFlags3["Anonymous"] = 16] = "Anonymous";
  ObjectFlags3[ObjectFlags3["Mapped"] = 32] = "Mapped";
  ObjectFlags3[ObjectFlags3["Instantiated"] = 64] = "Instantiated";
  ObjectFlags3[ObjectFlags3["ObjectLiteral"] = 128] = "ObjectLiteral";
  ObjectFlags3[ObjectFlags3["EvolvingArray"] = 256] = "EvolvingArray";
  ObjectFlags3[ObjectFlags3["ObjectLiteralPatternWithComputedProperties"] = 512] = "ObjectLiteralPatternWithComputedProperties";
  ObjectFlags3[ObjectFlags3["ReverseMapped"] = 1024] = "ReverseMapped";
  ObjectFlags3[ObjectFlags3["JsxAttributes"] = 2048] = "JsxAttributes";
  ObjectFlags3[ObjectFlags3["JSLiteral"] = 4096] = "JSLiteral";
  ObjectFlags3[ObjectFlags3["FreshLiteral"] = 8192] = "FreshLiteral";
  ObjectFlags3[ObjectFlags3["ArrayLiteral"] = 16384] = "ArrayLiteral";
  ObjectFlags3[ObjectFlags3["PrimitiveUnion"] = 32768] = "PrimitiveUnion";
  ObjectFlags3[ObjectFlags3["ContainsWideningType"] = 65536] = "ContainsWideningType";
  ObjectFlags3[ObjectFlags3["ContainsObjectOrArrayLiteral"] = 131072] = "ContainsObjectOrArrayLiteral";
  ObjectFlags3[ObjectFlags3["NonInferrableType"] = 262144] = "NonInferrableType";
  ObjectFlags3[ObjectFlags3["CouldContainTypeVariablesComputed"] = 524288] = "CouldContainTypeVariablesComputed";
  ObjectFlags3[ObjectFlags3["CouldContainTypeVariables"] = 1048576] = "CouldContainTypeVariables";
  ObjectFlags3[ObjectFlags3["ClassOrInterface"] = 3] = "ClassOrInterface";
  ObjectFlags3[ObjectFlags3["RequiresWidening"] = 196608] = "RequiresWidening";
  ObjectFlags3[ObjectFlags3["PropagatingFlags"] = 458752] = "PropagatingFlags";
  ObjectFlags3[ObjectFlags3["InstantiatedMapped"] = 96] = "InstantiatedMapped";
  ObjectFlags3[ObjectFlags3["ObjectTypeKindMask"] = 1343] = "ObjectTypeKindMask";
  ObjectFlags3[ObjectFlags3["ContainsSpread"] = 2097152] = "ContainsSpread";
  ObjectFlags3[ObjectFlags3["ObjectRestType"] = 4194304] = "ObjectRestType";
  ObjectFlags3[ObjectFlags3["InstantiationExpressionType"] = 8388608] = "InstantiationExpressionType";
  ObjectFlags3[ObjectFlags3["SingleSignatureType"] = 134217728] = "SingleSignatureType";
  ObjectFlags3[ObjectFlags3["IsClassInstanceClone"] = 16777216] = "IsClassInstanceClone";
  ObjectFlags3[ObjectFlags3["IdenticalBaseTypeCalculated"] = 33554432] = "IdenticalBaseTypeCalculated";
  ObjectFlags3[ObjectFlags3["IdenticalBaseTypeExists"] = 67108864] = "IdenticalBaseTypeExists";
  ObjectFlags3[ObjectFlags3["IsGenericTypeComputed"] = 2097152] = "IsGenericTypeComputed";
  ObjectFlags3[ObjectFlags3["IsGenericObjectType"] = 4194304] = "IsGenericObjectType";
  ObjectFlags3[ObjectFlags3["IsGenericIndexType"] = 8388608] = "IsGenericIndexType";
  ObjectFlags3[ObjectFlags3["IsGenericType"] = 12582912] = "IsGenericType";
  ObjectFlags3[ObjectFlags3["ContainsIntersections"] = 16777216] = "ContainsIntersections";
  ObjectFlags3[ObjectFlags3["IsUnknownLikeUnionComputed"] = 33554432] = "IsUnknownLikeUnionComputed";
  ObjectFlags3[ObjectFlags3["IsUnknownLikeUnion"] = 67108864] = "IsUnknownLikeUnion";
  ObjectFlags3[ObjectFlags3["IsNeverIntersectionComputed"] = 16777216] = "IsNeverIntersectionComputed";
  ObjectFlags3[ObjectFlags3["IsNeverIntersection"] = 33554432] = "IsNeverIntersection";
  ObjectFlags3[ObjectFlags3["IsConstrainedTypeVariable"] = 67108864] = "IsConstrainedTypeVariable";
  return ObjectFlags3;
})(ObjectFlags || {});
var SignatureFlags = /* @__PURE__ */ ((SignatureFlags4) => {
  SignatureFlags4[SignatureFlags4["None"] = 0] = "None";
  SignatureFlags4[SignatureFlags4["HasRestParameter"] = 1] = "HasRestParameter";
  SignatureFlags4[SignatureFlags4["HasLiteralTypes"] = 2] = "HasLiteralTypes";
  SignatureFlags4[SignatureFlags4["Abstract"] = 4] = "Abstract";
  SignatureFlags4[SignatureFlags4["IsInnerCallChain"] = 8] = "IsInnerCallChain";
  SignatureFlags4[SignatureFlags4["IsOuterCallChain"] = 16] = "IsOuterCallChain";
  SignatureFlags4[SignatureFlags4["IsUntypedSignatureInJSFile"] = 32] = "IsUntypedSignatureInJSFile";
  SignatureFlags4[SignatureFlags4["IsNonInferrable"] = 64] = "IsNonInferrable";
  SignatureFlags4[SignatureFlags4["IsSignatureCandidateForOverloadFailure"] = 128] = "IsSignatureCandidateForOverloadFailure";
  SignatureFlags4[SignatureFlags4["PropagatingFlags"] = 167] = "PropagatingFlags";
  SignatureFlags4[SignatureFlags4["CallChainFlags"] = 24] = "CallChainFlags";
  return SignatureFlags4;
})(SignatureFlags || {});
var DiagnosticCategory = /* @__PURE__ */ ((DiagnosticCategory2) => {
  DiagnosticCategory2[DiagnosticCategory2["Warning"] = 0] = "Warning";
  DiagnosticCategory2[DiagnosticCategory2["Error"] = 1] = "Error";
  DiagnosticCategory2[DiagnosticCategory2["Suggestion"] = 2] = "Suggestion";
  DiagnosticCategory2[DiagnosticCategory2["Message"] = 3] = "Message";
  return DiagnosticCategory2;
})(DiagnosticCategory || {});
function diagnosticCategoryName(d, lowerCase = true) {
  const name = DiagnosticCategory[d.category];
  return lowerCase ? name.toLowerCase() : name;
}
var ModuleResolutionKind = /* @__PURE__ */ ((ModuleResolutionKind2) => {
  ModuleResolutionKind2[ModuleResolutionKind2["Classic"] = 1] = "Classic";
  ModuleResolutionKind2[ModuleResolutionKind2["NodeJs"] = 2] = "NodeJs";
  ModuleResolutionKind2[ModuleResolutionKind2["Node10"] = 2] = "Node10";
  ModuleResolutionKind2[ModuleResolutionKind2["Node16"] = 3] = "Node16";
  ModuleResolutionKind2[ModuleResolutionKind2["NodeNext"] = 99] = "NodeNext";
  ModuleResolutionKind2[ModuleResolutionKind2["Bundler"] = 100] = "Bundler";
  return ModuleResolutionKind2;
})(ModuleResolutionKind || {});
var ModuleKind = /* @__PURE__ */ ((ModuleKind2) => {
  ModuleKind2[ModuleKind2["None"] = 0] = "None";
  ModuleKind2[ModuleKind2["CommonJS"] = 1] = "CommonJS";
  ModuleKind2[ModuleKind2["AMD"] = 2] = "AMD";
  ModuleKind2[ModuleKind2["UMD"] = 3] = "UMD";
  ModuleKind2[ModuleKind2["System"] = 4] = "System";
  ModuleKind2[ModuleKind2["ES2015"] = 5] = "ES2015";
  ModuleKind2[ModuleKind2["ES2020"] = 6] = "ES2020";
  ModuleKind2[ModuleKind2["ES2022"] = 7] = "ES2022";
  ModuleKind2[ModuleKind2["ESNext"] = 99] = "ESNext";
  ModuleKind2[ModuleKind2["Node16"] = 100] = "Node16";
  ModuleKind2[ModuleKind2["NodeNext"] = 199] = "NodeNext";
  ModuleKind2[ModuleKind2["Preserve"] = 200] = "Preserve";
  return ModuleKind2;
})(ModuleKind || {});
var ScriptKind = /* @__PURE__ */ ((ScriptKind3) => {
  ScriptKind3[ScriptKind3["Unknown"] = 0] = "Unknown";
  ScriptKind3[ScriptKind3["JS"] = 1] = "JS";
  ScriptKind3[ScriptKind3["JSX"] = 2] = "JSX";
  ScriptKind3[ScriptKind3["TS"] = 3] = "TS";
  ScriptKind3[ScriptKind3["TSX"] = 4] = "TSX";
  ScriptKind3[ScriptKind3["External"] = 5] = "External";
  ScriptKind3[ScriptKind3["JSON"] = 6] = "JSON";
  ScriptKind3[ScriptKind3["Deferred"] = 7] = "Deferred";
  return ScriptKind3;
})(ScriptKind || {});
var TransformFlags = /* @__PURE__ */ ((TransformFlags3) => {
  TransformFlags3[TransformFlags3["None"] = 0] = "None";
  TransformFlags3[TransformFlags3["ContainsTypeScript"] = 1] = "ContainsTypeScript";
  TransformFlags3[TransformFlags3["ContainsJsx"] = 2] = "ContainsJsx";
  TransformFlags3[TransformFlags3["ContainsESNext"] = 4] = "ContainsESNext";
  TransformFlags3[TransformFlags3["ContainsES2022"] = 8] = "ContainsES2022";
  TransformFlags3[TransformFlags3["ContainsES2021"] = 16] = "ContainsES2021";
  TransformFlags3[TransformFlags3["ContainsES2020"] = 32] = "ContainsES2020";
  TransformFlags3[TransformFlags3["ContainsES2019"] = 64] = "ContainsES2019";
  TransformFlags3[TransformFlags3["ContainsES2018"] = 128] = "ContainsES2018";
  TransformFlags3[TransformFlags3["ContainsES2017"] = 256] = "ContainsES2017";
  TransformFlags3[TransformFlags3["ContainsES2016"] = 512] = "ContainsES2016";
  TransformFlags3[TransformFlags3["ContainsES2015"] = 1024] = "ContainsES2015";
  TransformFlags3[TransformFlags3["ContainsGenerator"] = 2048] = "ContainsGenerator";
  TransformFlags3[TransformFlags3["ContainsDestructuringAssignment"] = 4096] = "ContainsDestructuringAssignment";
  TransformFlags3[TransformFlags3["ContainsTypeScriptClassSyntax"] = 8192] = "ContainsTypeScriptClassSyntax";
  TransformFlags3[TransformFlags3["ContainsLexicalThis"] = 16384] = "ContainsLexicalThis";
  TransformFlags3[TransformFlags3["ContainsRestOrSpread"] = 32768] = "ContainsRestOrSpread";
  TransformFlags3[TransformFlags3["ContainsObjectRestOrSpread"] = 65536] = "ContainsObjectRestOrSpread";
  TransformFlags3[TransformFlags3["ContainsComputedPropertyName"] = 131072] = "ContainsComputedPropertyName";
  TransformFlags3[TransformFlags3["ContainsBlockScopedBinding"] = 262144] = "ContainsBlockScopedBinding";
  TransformFlags3[TransformFlags3["ContainsBindingPattern"] = 524288] = "ContainsBindingPattern";
  TransformFlags3[TransformFlags3["ContainsYield"] = 1048576] = "ContainsYield";
  TransformFlags3[TransformFlags3["ContainsAwait"] = 2097152] = "ContainsAwait";
  TransformFlags3[TransformFlags3["ContainsHoistedDeclarationOrCompletion"] = 4194304] = "ContainsHoistedDeclarationOrCompletion";
  TransformFlags3[TransformFlags3["ContainsDynamicImport"] = 8388608] = "ContainsDynamicImport";
  TransformFlags3[TransformFlags3["ContainsClassFields"] = 16777216] = "ContainsClassFields";
  TransformFlags3[TransformFlags3["ContainsDecorators"] = 33554432] = "ContainsDecorators";
  TransformFlags3[TransformFlags3["ContainsPossibleTopLevelAwait"] = 67108864] = "ContainsPossibleTopLevelAwait";
  TransformFlags3[TransformFlags3["ContainsLexicalSuper"] = 134217728] = "ContainsLexicalSuper";
  TransformFlags3[TransformFlags3["ContainsUpdateExpressionForIdentifier"] = 268435456] = "ContainsUpdateExpressionForIdentifier";
  TransformFlags3[TransformFlags3["ContainsPrivateIdentifierInExpression"] = 536870912] = "ContainsPrivateIdentifierInExpression";
  TransformFlags3[TransformFlags3["HasComputedFlags"] = -2147483648] = "HasComputedFlags";
  TransformFlags3[TransformFlags3["AssertTypeScript"] = 1 /* ContainsTypeScript */] = "AssertTypeScript";
  TransformFlags3[TransformFlags3["AssertJsx"] = 2 /* ContainsJsx */] = "AssertJsx";
  TransformFlags3[TransformFlags3["AssertESNext"] = 4 /* ContainsESNext */] = "AssertESNext";
  TransformFlags3[TransformFlags3["AssertES2022"] = 8 /* ContainsES2022 */] = "AssertES2022";
  TransformFlags3[TransformFlags3["AssertES2021"] = 16 /* ContainsES2021 */] = "AssertES2021";
  TransformFlags3[TransformFlags3["AssertES2020"] = 32 /* ContainsES2020 */] = "AssertES2020";
  TransformFlags3[TransformFlags3["AssertES2019"] = 64 /* ContainsES2019 */] = "AssertES2019";
  TransformFlags3[TransformFlags3["AssertES2018"] = 128 /* ContainsES2018 */] = "AssertES2018";
  TransformFlags3[TransformFlags3["AssertES2017"] = 256 /* ContainsES2017 */] = "AssertES2017";
  TransformFlags3[TransformFlags3["AssertES2016"] = 512 /* ContainsES2016 */] = "AssertES2016";
  TransformFlags3[TransformFlags3["AssertES2015"] = 1024 /* ContainsES2015 */] = "AssertES2015";
  TransformFlags3[TransformFlags3["AssertGenerator"] = 2048 /* ContainsGenerator */] = "AssertGenerator";
  TransformFlags3[TransformFlags3["AssertDestructuringAssignment"] = 4096 /* ContainsDestructuringAssignment */] = "AssertDestructuringAssignment";
  TransformFlags3[TransformFlags3["OuterExpressionExcludes"] = -2147483648 /* HasComputedFlags */] = "OuterExpressionExcludes";
  TransformFlags3[TransformFlags3["PropertyAccessExcludes"] = -2147483648 /* OuterExpressionExcludes */] = "PropertyAccessExcludes";
  TransformFlags3[TransformFlags3["NodeExcludes"] = -2147483648 /* PropertyAccessExcludes */] = "NodeExcludes";
  TransformFlags3[TransformFlags3["ArrowFunctionExcludes"] = -2072174592] = "ArrowFunctionExcludes";
  TransformFlags3[TransformFlags3["FunctionExcludes"] = -1937940480] = "FunctionExcludes";
  TransformFlags3[TransformFlags3["ConstructorExcludes"] = -1937948672] = "ConstructorExcludes";
  TransformFlags3[TransformFlags3["MethodOrAccessorExcludes"] = -2005057536] = "MethodOrAccessorExcludes";
  TransformFlags3[TransformFlags3["PropertyExcludes"] = -2013249536] = "PropertyExcludes";
  TransformFlags3[TransformFlags3["ClassExcludes"] = -2147344384] = "ClassExcludes";
  TransformFlags3[TransformFlags3["ModuleExcludes"] = -1941676032] = "ModuleExcludes";
  TransformFlags3[TransformFlags3["TypeExcludes"] = -2] = "TypeExcludes";
  TransformFlags3[TransformFlags3["ObjectLiteralExcludes"] = -2147278848] = "ObjectLiteralExcludes";
  TransformFlags3[TransformFlags3["ArrayLiteralOrCallOrNewExcludes"] = -2147450880] = "ArrayLiteralOrCallOrNewExcludes";
  TransformFlags3[TransformFlags3["VariableDeclarationListExcludes"] = -2146893824] = "VariableDeclarationListExcludes";
  TransformFlags3[TransformFlags3["ParameterExcludes"] = -2147483648 /* NodeExcludes */] = "ParameterExcludes";
  TransformFlags3[TransformFlags3["CatchClauseExcludes"] = -2147418112] = "CatchClauseExcludes";
  TransformFlags3[TransformFlags3["BindingPatternExcludes"] = -2147450880] = "BindingPatternExcludes";
  TransformFlags3[TransformFlags3["ContainsLexicalThisOrSuper"] = 134234112] = "ContainsLexicalThisOrSuper";
  TransformFlags3[TransformFlags3["PropertyNamePropagatingFlags"] = 134234112] = "PropertyNamePropagatingFlags";
  return TransformFlags3;
})(TransformFlags || {});
var SnippetKind = /* @__PURE__ */ ((SnippetKind3) => {
  SnippetKind3[SnippetKind3["TabStop"] = 0] = "TabStop";
  SnippetKind3[SnippetKind3["Placeholder"] = 1] = "Placeholder";
  SnippetKind3[SnippetKind3["Choice"] = 2] = "Choice";
  SnippetKind3[SnippetKind3["Variable"] = 3] = "Variable";
  return SnippetKind3;
})(SnippetKind || {});
var EmitFlags = /* @__PURE__ */ ((EmitFlags3) => {
  EmitFlags3[EmitFlags3["None"] = 0] = "None";
  EmitFlags3[EmitFlags3["SingleLine"] = 1] = "SingleLine";
  EmitFlags3[EmitFlags3["MultiLine"] = 2] = "MultiLine";
  EmitFlags3[EmitFlags3["AdviseOnEmitNode"] = 4] = "AdviseOnEmitNode";
  EmitFlags3[EmitFlags3["NoSubstitution"] = 8] = "NoSubstitution";
  EmitFlags3[EmitFlags3["CapturesThis"] = 16] = "CapturesThis";
  EmitFlags3[EmitFlags3["NoLeadingSourceMap"] = 32] = "NoLeadingSourceMap";
  EmitFlags3[EmitFlags3["NoTrailingSourceMap"] = 64] = "NoTrailingSourceMap";
  EmitFlags3[EmitFlags3["NoSourceMap"] = 96] = "NoSourceMap";
  EmitFlags3[EmitFlags3["NoNestedSourceMaps"] = 128] = "NoNestedSourceMaps";
  EmitFlags3[EmitFlags3["NoTokenLeadingSourceMaps"] = 256] = "NoTokenLeadingSourceMaps";
  EmitFlags3[EmitFlags3["NoTokenTrailingSourceMaps"] = 512] = "NoTokenTrailingSourceMaps";
  EmitFlags3[EmitFlags3["NoTokenSourceMaps"] = 768] = "NoTokenSourceMaps";
  EmitFlags3[EmitFlags3["NoLeadingComments"] = 1024] = "NoLeadingComments";
  EmitFlags3[EmitFlags3["NoTrailingComments"] = 2048] = "NoTrailingComments";
  EmitFlags3[EmitFlags3["NoComments"] = 3072] = "NoComments";
  EmitFlags3[EmitFlags3["NoNestedComments"] = 4096] = "NoNestedComments";
  EmitFlags3[EmitFlags3["HelperName"] = 8192] = "HelperName";
  EmitFlags3[EmitFlags3["ExportName"] = 16384] = "ExportName";
  EmitFlags3[EmitFlags3["LocalName"] = 32768] = "LocalName";
  EmitFlags3[EmitFlags3["InternalName"] = 65536] = "InternalName";
  EmitFlags3[EmitFlags3["Indented"] = 131072] = "Indented";
  EmitFlags3[EmitFlags3["NoIndentation"] = 262144] = "NoIndentation";
  EmitFlags3[EmitFlags3["AsyncFunctionBody"] = 524288] = "AsyncFunctionBody";
  EmitFlags3[EmitFlags3["ReuseTempVariableScope"] = 1048576] = "ReuseTempVariableScope";
  EmitFlags3[EmitFlags3["CustomPrologue"] = 2097152] = "CustomPrologue";
  EmitFlags3[EmitFlags3["NoHoisting"] = 4194304] = "NoHoisting";
  EmitFlags3[EmitFlags3["Iterator"] = 8388608] = "Iterator";
  EmitFlags3[EmitFlags3["NoAsciiEscaping"] = 16777216] = "NoAsciiEscaping";
  return EmitFlags3;
})(EmitFlags || {});
var LanguageFeatureMinimumTarget = {
  Classes: 2 /* ES2015 */,
  ForOf: 2 /* ES2015 */,
  Generators: 2 /* ES2015 */,
  Iteration: 2 /* ES2015 */,
  SpreadElements: 2 /* ES2015 */,
  RestElements: 2 /* ES2015 */,
  TaggedTemplates: 2 /* ES2015 */,
  DestructuringAssignment: 2 /* ES2015 */,
  BindingPatterns: 2 /* ES2015 */,
  ArrowFunctions: 2 /* ES2015 */,
  BlockScopedVariables: 2 /* ES2015 */,
  ObjectAssign: 2 /* ES2015 */,
  RegularExpressionFlagsUnicode: 2 /* ES2015 */,
  RegularExpressionFlagsSticky: 2 /* ES2015 */,
  Exponentiation: 3 /* ES2016 */,
  AsyncFunctions: 4 /* ES2017 */,
  ForAwaitOf: 5 /* ES2018 */,
  AsyncGenerators: 5 /* ES2018 */,
  AsyncIteration: 5 /* ES2018 */,
  ObjectSpreadRest: 5 /* ES2018 */,
  RegularExpressionFlagsDotAll: 5 /* ES2018 */,
  BindinglessCatch: 6 /* ES2019 */,
  BigInt: 7 /* ES2020 */,
  NullishCoalesce: 7 /* ES2020 */,
  OptionalChaining: 7 /* ES2020 */,
  LogicalAssignment: 8 /* ES2021 */,
  TopLevelAwait: 9 /* ES2022 */,
  ClassFields: 9 /* ES2022 */,
  PrivateNamesAndClassStaticBlocks: 9 /* ES2022 */,
  RegularExpressionFlagsHasIndices: 9 /* ES2022 */,
  ShebangComments: 10 /* ES2023 */,
  RegularExpressionFlagsUnicodeSets: 11 /* ES2024 */,
  UsingAndAwaitUsing: 99 /* ESNext */,
  ClassAndClassElementDecorators: 99 /* ESNext */
};
var commentPragmas = {
  "reference": {
    args: [
      { name: "types", optional: true, captureSpan: true },
      { name: "lib", optional: true, captureSpan: true },
      { name: "path", optional: true, captureSpan: true },
      { name: "no-default-lib", optional: true },
      { name: "resolution-mode", optional: true },
      { name: "preserve", optional: true }
    ],
    kind: 1 /* TripleSlashXML */
  },
  "amd-dependency": {
    args: [{ name: "path" }, { name: "name", optional: true }],
    kind: 1 /* TripleSlashXML */
  },
  "amd-module": {
    args: [{ name: "name" }],
    kind: 1 /* TripleSlashXML */
  },
  "ts-check": {
    kind: 2 /* SingleLine */
  },
  "ts-nocheck": {
    kind: 2 /* SingleLine */
  },
  "jsx": {
    args: [{ name: "factory" }],
    kind: 4 /* MultiLine */
  },
  "jsxfrag": {
    args: [{ name: "factory" }],
    kind: 4 /* MultiLine */
  },
  "jsximportsource": {
    args: [{ name: "factory" }],
    kind: 4 /* MultiLine */
  },
  "jsxruntime": {
    args: [{ name: "factory" }],
    kind: 4 /* MultiLine */
  }
};

// src/compiler/sys.ts
function generateDjb2Hash(data) {
  let acc = 5381;
  for (let i = 0; i < data.length; i++) {
    acc = (acc << 5) + acc + data.charCodeAt(i);
  }
  return acc.toString();
}
var PollingInterval = /* @__PURE__ */ ((PollingInterval3) => {
  PollingInterval3[PollingInterval3["High"] = 2e3] = "High";
  PollingInterval3[PollingInterval3["Medium"] = 500] = "Medium";
  PollingInterval3[PollingInterval3["Low"] = 250] = "Low";
  return PollingInterval3;
})(PollingInterval || {});
var missingFileModifiedTime = /* @__PURE__ */ new Date(0);
function getModifiedTime(host, fileName) {
  return host.getModifiedTime(fileName) || missingFileModifiedTime;
}
function createPollingIntervalBasedLevels(levels) {
  return {
    [250 /* Low */]: levels.Low,
    [500 /* Medium */]: levels.Medium,
    [2e3 /* High */]: levels.High
  };
}
var defaultChunkLevels = { Low: 32, Medium: 64, High: 256 };
var pollingChunkSize = createPollingIntervalBasedLevels(defaultChunkLevels);
var unchangedPollThresholds = createPollingIntervalBasedLevels(defaultChunkLevels);
function setCustomPollingValues(system) {
  if (!system.getEnvironmentVariable) {
    return;
  }
  const pollingIntervalChanged = setCustomLevels("TSC_WATCH_POLLINGINTERVAL", PollingInterval);
  pollingChunkSize = getCustomPollingBasedLevels("TSC_WATCH_POLLINGCHUNKSIZE", defaultChunkLevels) || pollingChunkSize;
  unchangedPollThresholds = getCustomPollingBasedLevels("TSC_WATCH_UNCHANGEDPOLLTHRESHOLDS", defaultChunkLevels) || unchangedPollThresholds;
  function getLevel(envVar, level) {
    return system.getEnvironmentVariable(`${envVar}_${level.toUpperCase()}`);
  }
  function getCustomLevels(baseVariable) {
    let customLevels;
    setCustomLevel("Low");
    setCustomLevel("Medium");
    setCustomLevel("High");
    return customLevels;
    function setCustomLevel(level) {
      const customLevel = getLevel(baseVariable, level);
      if (customLevel) {
        (customLevels || (customLevels = {}))[level] = Number(customLevel);
      }
    }
  }
  function setCustomLevels(baseVariable, levels) {
    const customLevels = getCustomLevels(baseVariable);
    if (customLevels) {
      setLevel("Low");
      setLevel("Medium");
      setLevel("High");
      return true;
    }
    return false;
    function setLevel(level) {
      levels[level] = customLevels[level] || levels[level];
    }
  }
  function getCustomPollingBasedLevels(baseVariable, defaultLevels) {
    const customLevels = getCustomLevels(baseVariable);
    return (pollingIntervalChanged || customLevels) && createPollingIntervalBasedLevels(customLevels ? { ...defaultLevels, ...customLevels } : defaultLevels);
  }
}
function pollWatchedFileQueue(host, queue, pollIndex, chunkSize, callbackOnWatchFileStat) {
  let definedValueCopyToIndex = pollIndex;
  for (let canVisit = queue.length; chunkSize && canVisit; nextPollIndex(), canVisit--) {
    const watchedFile = queue[pollIndex];
    if (!watchedFile) {
      continue;
    } else if (watchedFile.isClosed) {
      queue[pollIndex] = void 0;
      continue;
    }
    chunkSize--;
    const fileChanged = onWatchedFileStat(watchedFile, getModifiedTime(host, watchedFile.fileName));
    if (watchedFile.isClosed) {
      queue[pollIndex] = void 0;
      continue;
    }
    callbackOnWatchFileStat == null ? void 0 : callbackOnWatchFileStat(watchedFile, pollIndex, fileChanged);
    if (queue[pollIndex]) {
      if (definedValueCopyToIndex < pollIndex) {
        queue[definedValueCopyToIndex] = watchedFile;
        queue[pollIndex] = void 0;
      }
      definedValueCopyToIndex++;
    }
  }
  return pollIndex;
  function nextPollIndex() {
    pollIndex++;
    if (pollIndex === queue.length) {
      if (definedValueCopyToIndex < pollIndex) {
        queue.length = definedValueCopyToIndex;
      }
      pollIndex = 0;
      definedValueCopyToIndex = 0;
    }
  }
}
function createDynamicPriorityPollingWatchFile(host) {
  const watchedFiles = [];
  const changedFilesInLastPoll = [];
  const lowPollingIntervalQueue = createPollingIntervalQueue(250 /* Low */);
  const mediumPollingIntervalQueue = createPollingIntervalQueue(500 /* Medium */);
  const highPollingIntervalQueue = createPollingIntervalQueue(2e3 /* High */);
  return watchFile2;
  function watchFile2(fileName, callback, defaultPollingInterval) {
    const file = {
      fileName,
      callback,
      unchangedPolls: 0,
      mtime: getModifiedTime(host, fileName)
    };
    watchedFiles.push(file);
    addToPollingIntervalQueue(file, defaultPollingInterval);
    return {
      close: () => {
        file.isClosed = true;
        unorderedRemoveItem(watchedFiles, file);
      }
    };
  }
  function createPollingIntervalQueue(pollingInterval) {
    const queue = [];
    queue.pollingInterval = pollingInterval;
    queue.pollIndex = 0;
    queue.pollScheduled = false;
    return queue;
  }
  function pollPollingIntervalQueue(_timeoutType, queue) {
    queue.pollIndex = pollQueue(queue, queue.pollingInterval, queue.pollIndex, pollingChunkSize[queue.pollingInterval]);
    if (queue.length) {
      scheduleNextPoll(queue.pollingInterval);
    } else {
      Debug.assert(queue.pollIndex === 0);
      queue.pollScheduled = false;
    }
  }
  function pollLowPollingIntervalQueue(_timeoutType, queue) {
    pollQueue(
      changedFilesInLastPoll,
      250 /* Low */,
      /*pollIndex*/
      0,
      changedFilesInLastPoll.length
    );
    pollPollingIntervalQueue(_timeoutType, queue);
    if (!queue.pollScheduled && changedFilesInLastPoll.length) {
      scheduleNextPoll(250 /* Low */);
    }
  }
  function pollQueue(queue, pollingInterval, pollIndex, chunkSize) {
    return pollWatchedFileQueue(
      host,
      queue,
      pollIndex,
      chunkSize,
      onWatchFileStat
    );
    function onWatchFileStat(watchedFile, pollIndex2, fileChanged) {
      if (fileChanged) {
        watchedFile.unchangedPolls = 0;
        if (queue !== changedFilesInLastPoll) {
          queue[pollIndex2] = void 0;
          addChangedFileToLowPollingIntervalQueue(watchedFile);
        }
      } else if (watchedFile.unchangedPolls !== unchangedPollThresholds[pollingInterval]) {
        watchedFile.unchangedPolls++;
      } else if (queue === changedFilesInLastPoll) {
        watchedFile.unchangedPolls = 1;
        queue[pollIndex2] = void 0;
        addToPollingIntervalQueue(watchedFile, 250 /* Low */);
      } else if (pollingInterval !== 2e3 /* High */) {
        watchedFile.unchangedPolls++;
        queue[pollIndex2] = void 0;
        addToPollingIntervalQueue(watchedFile, pollingInterval === 250 /* Low */ ? 500 /* Medium */ : 2e3 /* High */);
      }
    }
  }
  function pollingIntervalQueue(pollingInterval) {
    switch (pollingInterval) {
      case 250 /* Low */:
        return lowPollingIntervalQueue;
      case 500 /* Medium */:
        return mediumPollingIntervalQueue;
      case 2e3 /* High */:
        return highPollingIntervalQueue;
    }
  }
  function addToPollingIntervalQueue(file, pollingInterval) {
    pollingIntervalQueue(pollingInterval).push(file);
    scheduleNextPollIfNotAlreadyScheduled(pollingInterval);
  }
  function addChangedFileToLowPollingIntervalQueue(file) {
    changedFilesInLastPoll.push(file);
    scheduleNextPollIfNotAlreadyScheduled(250 /* Low */);
  }
  function scheduleNextPollIfNotAlreadyScheduled(pollingInterval) {
    if (!pollingIntervalQueue(pollingInterval).pollScheduled) {
      scheduleNextPoll(pollingInterval);
    }
  }
  function scheduleNextPoll(pollingInterval) {
    pollingIntervalQueue(pollingInterval).pollScheduled = host.setTimeout(pollingInterval === 250 /* Low */ ? pollLowPollingIntervalQueue : pollPollingIntervalQueue, pollingInterval, pollingInterval === 250 /* Low */ ? "pollLowPollingIntervalQueue" : "pollPollingIntervalQueue", pollingIntervalQueue(pollingInterval));
  }
}
function createUseFsEventsOnParentDirectoryWatchFile(fsWatch, useCaseSensitiveFileNames2, getModifiedTime3, fsWatchWithTimestamp) {
  const fileWatcherCallbacks = createMultiMap();
  const fileTimestamps = fsWatchWithTimestamp ? /* @__PURE__ */ new Map() : void 0;
  const dirWatchers = /* @__PURE__ */ new Map();
  const toCanonicalName = createGetCanonicalFileName(useCaseSensitiveFileNames2);
  return nonPollingWatchFile;
  function nonPollingWatchFile(fileName, callback, _pollingInterval, fallbackOptions) {
    const filePath = toCanonicalName(fileName);
    if (fileWatcherCallbacks.add(filePath, callback).length === 1 && fileTimestamps) {
      fileTimestamps.set(filePath, getModifiedTime3(fileName) || missingFileModifiedTime);
    }
    const dirPath = getDirectoryPath(filePath) || ".";
    const watcher = dirWatchers.get(dirPath) || createDirectoryWatcher(getDirectoryPath(fileName) || ".", dirPath, fallbackOptions);
    watcher.referenceCount++;
    return {
      close: () => {
        if (watcher.referenceCount === 1) {
          watcher.close();
          dirWatchers.delete(dirPath);
        } else {
          watcher.referenceCount--;
        }
        fileWatcherCallbacks.remove(filePath, callback);
      }
    };
  }
  function createDirectoryWatcher(dirName, dirPath, fallbackOptions) {
    const watcher = fsWatch(
      dirName,
      1 /* Directory */,
      (eventName, relativeFileName) => {
        if (!isString(relativeFileName)) return;
        const fileName = getNormalizedAbsolutePath(relativeFileName, dirName);
        const filePath = toCanonicalName(fileName);
        const callbacks = fileName && fileWatcherCallbacks.get(filePath);
        if (callbacks) {
          let currentModifiedTime;
          let eventKind = 1 /* Changed */;
          if (fileTimestamps) {
            const existingTime = fileTimestamps.get(filePath);
            if (eventName === "change") {
              currentModifiedTime = getModifiedTime3(fileName) || missingFileModifiedTime;
              if (currentModifiedTime.getTime() === existingTime.getTime()) return;
            }
            currentModifiedTime || (currentModifiedTime = getModifiedTime3(fileName) || missingFileModifiedTime);
            fileTimestamps.set(filePath, currentModifiedTime);
            if (existingTime === missingFileModifiedTime) eventKind = 0 /* Created */;
            else if (currentModifiedTime === missingFileModifiedTime) eventKind = 2 /* Deleted */;
          }
          for (const fileCallback of callbacks) {
            fileCallback(fileName, eventKind, currentModifiedTime);
          }
        }
      },
      /*recursive*/
      false,
      500 /* Medium */,
      fallbackOptions
    );
    watcher.referenceCount = 0;
    dirWatchers.set(dirPath, watcher);
    return watcher;
  }
}
function createFixedChunkSizePollingWatchFile(host) {
  const watchedFiles = [];
  let pollIndex = 0;
  let pollScheduled;
  return watchFile2;
  function watchFile2(fileName, callback) {
    const file = {
      fileName,
      callback,
      mtime: getModifiedTime(host, fileName)
    };
    watchedFiles.push(file);
    scheduleNextPoll();
    return {
      close: () => {
        file.isClosed = true;
        unorderedRemoveItem(watchedFiles, file);
      }
    };
  }
  function pollQueue() {
    pollScheduled = void 0;
    pollIndex = pollWatchedFileQueue(host, watchedFiles, pollIndex, pollingChunkSize[250 /* Low */]);
    scheduleNextPoll();
  }
  function scheduleNextPoll() {
    if (!watchedFiles.length || pollScheduled) return;
    pollScheduled = host.setTimeout(pollQueue, 2e3 /* High */, "pollQueue");
  }
}
function createSingleWatcherPerName(cache, useCaseSensitiveFileNames2, name, callback, createWatcher) {
  const toCanonicalFileName = createGetCanonicalFileName(useCaseSensitiveFileNames2);
  const path = toCanonicalFileName(name);
  const existing = cache.get(path);
  if (existing) {
    existing.callbacks.push(callback);
  } else {
    cache.set(path, {
      watcher: createWatcher(
        // Cant infer types correctly so lets satisfy checker
        (param1, param2, param3) => {
          var _a;
          return (_a = cache.get(path)) == null ? void 0 : _a.callbacks.slice().forEach((cb) => cb(param1, param2, param3));
        }
      ),
      callbacks: [callback]
    });
  }
  return {
    close: () => {
      const watcher = cache.get(path);
      if (!watcher) return;
      if (!orderedRemoveItem(watcher.callbacks, callback) || watcher.callbacks.length) return;
      cache.delete(path);
      closeFileWatcherOf(watcher);
    }
  };
}
function onWatchedFileStat(watchedFile, modifiedTime) {
  const oldTime = watchedFile.mtime.getTime();
  const newTime = modifiedTime.getTime();
  if (oldTime !== newTime) {
    watchedFile.mtime = modifiedTime;
    watchedFile.callback(watchedFile.fileName, getFileWatcherEventKind(oldTime, newTime), modifiedTime);
    return true;
  }
  return false;
}
function getFileWatcherEventKind(oldTime, newTime) {
  return oldTime === 0 ? 0 /* Created */ : newTime === 0 ? 2 /* Deleted */ : 1 /* Changed */;
}
var ignoredPaths = ["/node_modules/.", "/.git", "/.#"];
var curSysLog = noop;
function sysLog(s) {
  return curSysLog(s);
}
function setSysLog(logger) {
  curSysLog = logger;
}
function createDirectoryWatcherSupportingRecursive({
  watchDirectory,
  useCaseSensitiveFileNames: useCaseSensitiveFileNames2,
  getCurrentDirectory,
  getAccessibleSortedChildDirectories,
  fileSystemEntryExists,
  realpath,
  setTimeout: setTimeout2,
  clearTimeout: clearTimeout2
}) {
  const cache = /* @__PURE__ */ new Map();
  const callbackCache = createMultiMap();
  const cacheToUpdateChildWatches = /* @__PURE__ */ new Map();
  let timerToUpdateChildWatches;
  const filePathComparer = getStringComparer(!useCaseSensitiveFileNames2);
  const toCanonicalFilePath = createGetCanonicalFileName(useCaseSensitiveFileNames2);
  return (dirName, callback, recursive, options) => recursive ? createDirectoryWatcher(dirName, options, callback) : watchDirectory(dirName, callback, recursive, options);
  function createDirectoryWatcher(dirName, options, callback, link) {
    const dirPath = toCanonicalFilePath(dirName);
    let directoryWatcher = cache.get(dirPath);
    if (directoryWatcher) {
      directoryWatcher.refCount++;
    } else {
      directoryWatcher = {
        watcher: watchDirectory(
          dirName,
          (fileName) => {
            var _a;
            if (isIgnoredPath(fileName, options)) return;
            if (options == null ? void 0 : options.synchronousWatchDirectory) {
              if (!((_a = cache.get(dirPath)) == null ? void 0 : _a.targetWatcher)) invokeCallbacks(dirName, dirPath, fileName);
              updateChildWatches(dirName, dirPath, options);
            } else {
              nonSyncUpdateChildWatches(dirName, dirPath, fileName, options);
            }
          },
          /*recursive*/
          false,
          options
        ),
        refCount: 1,
        childWatches: emptyArray,
        targetWatcher: void 0,
        links: void 0
      };
      cache.set(dirPath, directoryWatcher);
      updateChildWatches(dirName, dirPath, options);
    }
    if (link) (directoryWatcher.links ?? (directoryWatcher.links = /* @__PURE__ */ new Set())).add(link);
    const callbackToAdd = callback && { dirName, callback };
    if (callbackToAdd) {
      callbackCache.add(dirPath, callbackToAdd);
    }
    return {
      dirName,
      close: () => {
        var _a;
        const directoryWatcher2 = Debug.checkDefined(cache.get(dirPath));
        if (callbackToAdd) callbackCache.remove(dirPath, callbackToAdd);
        if (link) (_a = directoryWatcher2.links) == null ? void 0 : _a.delete(link);
        directoryWatcher2.refCount--;
        if (directoryWatcher2.refCount) return;
        cache.delete(dirPath);
        directoryWatcher2.links = void 0;
        closeFileWatcherOf(directoryWatcher2);
        closeTargetWatcher(directoryWatcher2);
        directoryWatcher2.childWatches.forEach(closeFileWatcher);
      }
    };
  }
  function invokeCallbacks(dirName, dirPath, fileNameOrInvokeMap, fileNames) {
    var _a, _b;
    let fileName;
    let invokeMap;
    if (isString(fileNameOrInvokeMap)) {
      fileName = fileNameOrInvokeMap;
    } else {
      invokeMap = fileNameOrInvokeMap;
    }
    callbackCache.forEach((callbacks, rootDirName) => {
      if (invokeMap && invokeMap.get(rootDirName) === true) return;
      if (rootDirName === dirPath || startsWith(dirPath, rootDirName) && dirPath[rootDirName.length] === directorySeparator) {
        if (invokeMap) {
          if (fileNames) {
            const existing = invokeMap.get(rootDirName);
            if (existing) {
              existing.push(...fileNames);
            } else {
              invokeMap.set(rootDirName, fileNames.slice());
            }
          } else {
            invokeMap.set(rootDirName, true);
          }
        } else {
          callbacks.forEach(({ callback }) => callback(fileName));
        }
      }
    });
    (_b = (_a = cache.get(dirPath)) == null ? void 0 : _a.links) == null ? void 0 : _b.forEach((link) => {
      const toPathInLink = (fileName2) => combinePaths(link, getRelativePathFromDirectory(dirName, fileName2, toCanonicalFilePath));
      if (invokeMap) {
        invokeCallbacks(link, toCanonicalFilePath(link), invokeMap, fileNames == null ? void 0 : fileNames.map(toPathInLink));
      } else {
        invokeCallbacks(link, toCanonicalFilePath(link), toPathInLink(fileName));
      }
    });
  }
  function nonSyncUpdateChildWatches(dirName, dirPath, fileName, options) {
    const parentWatcher = cache.get(dirPath);
    if (parentWatcher && fileSystemEntryExists(dirName, 1 /* Directory */)) {
      scheduleUpdateChildWatches(dirName, dirPath, fileName, options);
      return;
    }
    invokeCallbacks(dirName, dirPath, fileName);
    closeTargetWatcher(parentWatcher);
    removeChildWatches(parentWatcher);
  }
  function scheduleUpdateChildWatches(dirName, dirPath, fileName, options) {
    const existing = cacheToUpdateChildWatches.get(dirPath);
    if (existing) {
      existing.fileNames.push(fileName);
    } else {
      cacheToUpdateChildWatches.set(dirPath, { dirName, options, fileNames: [fileName] });
    }
    if (timerToUpdateChildWatches) {
      clearTimeout2(timerToUpdateChildWatches);
      timerToUpdateChildWatches = void 0;
    }
    timerToUpdateChildWatches = setTimeout2(onTimerToUpdateChildWatches, 1e3, "timerToUpdateChildWatches");
  }
  function onTimerToUpdateChildWatches() {
    var _a;
    timerToUpdateChildWatches = void 0;
    sysLog(`sysLog:: onTimerToUpdateChildWatches:: ${cacheToUpdateChildWatches.size}`);
    const start = timestamp();
    const invokeMap = /* @__PURE__ */ new Map();
    while (!timerToUpdateChildWatches && cacheToUpdateChildWatches.size) {
      const result = cacheToUpdateChildWatches.entries().next();
      Debug.assert(!result.done);
      const { value: [dirPath, { dirName, options, fileNames }] } = result;
      cacheToUpdateChildWatches.delete(dirPath);
      const hasChanges = updateChildWatches(dirName, dirPath, options);
      if (!((_a = cache.get(dirPath)) == null ? void 0 : _a.targetWatcher)) invokeCallbacks(dirName, dirPath, invokeMap, hasChanges ? void 0 : fileNames);
    }
    sysLog(`sysLog:: invokingWatchers:: Elapsed:: ${timestamp() - start}ms:: ${cacheToUpdateChildWatches.size}`);
    callbackCache.forEach((callbacks, rootDirName) => {
      const existing = invokeMap.get(rootDirName);
      if (existing) {
        callbacks.forEach(({ callback, dirName }) => {
          if (isArray(existing)) {
            existing.forEach(callback);
          } else {
            callback(dirName);
          }
        });
      }
    });
    const elapsed = timestamp() - start;
    sysLog(`sysLog:: Elapsed:: ${elapsed}ms:: onTimerToUpdateChildWatches:: ${cacheToUpdateChildWatches.size} ${timerToUpdateChildWatches}`);
  }
  function removeChildWatches(parentWatcher) {
    if (!parentWatcher) return;
    const existingChildWatches = parentWatcher.childWatches;
    parentWatcher.childWatches = emptyArray;
    for (const childWatcher of existingChildWatches) {
      childWatcher.close();
      removeChildWatches(cache.get(toCanonicalFilePath(childWatcher.dirName)));
    }
  }
  function closeTargetWatcher(watcher) {
    if (watcher == null ? void 0 : watcher.targetWatcher) {
      watcher.targetWatcher.close();
      watcher.targetWatcher = void 0;
    }
  }
  function updateChildWatches(parentDir, parentDirPath, options) {
    const parentWatcher = cache.get(parentDirPath);
    if (!parentWatcher) return false;
    const target = normalizePath(realpath(parentDir));
    let hasChanges;
    let newChildWatches;
    if (filePathComparer(target, parentDir) === 0 /* EqualTo */) {
      hasChanges = enumerateInsertsAndDeletes(
        fileSystemEntryExists(parentDir, 1 /* Directory */) ? mapDefined(getAccessibleSortedChildDirectories(parentDir), (child) => {
          const childFullName = getNormalizedAbsolutePath(child, parentDir);
          return !isIgnoredPath(childFullName, options) && filePathComparer(childFullName, normalizePath(realpath(childFullName))) === 0 /* EqualTo */ ? childFullName : void 0;
        }) : emptyArray,
        parentWatcher.childWatches,
        (child, childWatcher) => filePathComparer(child, childWatcher.dirName),
        createAndAddChildDirectoryWatcher,
        closeFileWatcher,
        addChildDirectoryWatcher
      );
    } else if (parentWatcher.targetWatcher && filePathComparer(target, parentWatcher.targetWatcher.dirName) === 0 /* EqualTo */) {
      hasChanges = false;
      Debug.assert(parentWatcher.childWatches === emptyArray);
    } else {
      closeTargetWatcher(parentWatcher);
      parentWatcher.targetWatcher = createDirectoryWatcher(
        target,
        options,
        /*callback*/
        void 0,
        parentDir
      );
      parentWatcher.childWatches.forEach(closeFileWatcher);
      hasChanges = true;
    }
    parentWatcher.childWatches = newChildWatches || emptyArray;
    return hasChanges;
    function createAndAddChildDirectoryWatcher(childName) {
      const result = createDirectoryWatcher(childName, options);
      addChildDirectoryWatcher(result);
    }
    function addChildDirectoryWatcher(childWatcher) {
      (newChildWatches || (newChildWatches = [])).push(childWatcher);
    }
  }
  function isIgnoredPath(path, options) {
    return some(ignoredPaths, (searchPath) => isInPath(path, searchPath)) || isIgnoredByWatchOptions(path, options, useCaseSensitiveFileNames2, getCurrentDirectory);
  }
  function isInPath(path, searchPath) {
    if (path.includes(searchPath)) return true;
    if (useCaseSensitiveFileNames2) return false;
    return toCanonicalFilePath(path).includes(searchPath);
  }
}
function createFileWatcherCallback(callback) {
  return (_fileName, eventKind, modifiedTime) => callback(eventKind === 1 /* Changed */ ? "change" : "rename", "", modifiedTime);
}
function createFsWatchCallbackForFileWatcherCallback(fileName, callback, getModifiedTime3) {
  return (eventName, _relativeFileName, modifiedTime) => {
    if (eventName === "rename") {
      modifiedTime || (modifiedTime = getModifiedTime3(fileName) || missingFileModifiedTime);
      callback(fileName, modifiedTime !== missingFileModifiedTime ? 0 /* Created */ : 2 /* Deleted */, modifiedTime);
    } else {
      callback(fileName, 1 /* Changed */, modifiedTime);
    }
  };
}
function isIgnoredByWatchOptions(pathToCheck, options, useCaseSensitiveFileNames2, getCurrentDirectory) {
  return ((options == null ? void 0 : options.excludeDirectories) || (options == null ? void 0 : options.excludeFiles)) && (matchesExclude(pathToCheck, options == null ? void 0 : options.excludeFiles, useCaseSensitiveFileNames2, getCurrentDirectory()) || matchesExclude(pathToCheck, options == null ? void 0 : options.excludeDirectories, useCaseSensitiveFileNames2, getCurrentDirectory()));
}
function createFsWatchCallbackForDirectoryWatcherCallback(directoryName, callback, options, useCaseSensitiveFileNames2, getCurrentDirectory) {
  return (eventName, relativeFileName) => {
    if (eventName === "rename") {
      const fileName = !relativeFileName ? directoryName : normalizePath(combinePaths(directoryName, relativeFileName));
      if (!relativeFileName || !isIgnoredByWatchOptions(fileName, options, useCaseSensitiveFileNames2, getCurrentDirectory)) {
        callback(fileName);
      }
    }
  };
}
function createSystemWatchFunctions({
  pollingWatchFileWorker,
  getModifiedTime: getModifiedTime3,
  setTimeout: setTimeout2,
  clearTimeout: clearTimeout2,
  fsWatchWorker,
  fileSystemEntryExists,
  useCaseSensitiveFileNames: useCaseSensitiveFileNames2,
  getCurrentDirectory,
  fsSupportsRecursiveFsWatch,
  getAccessibleSortedChildDirectories,
  realpath,
  tscWatchFile,
  useNonPollingWatchers,
  tscWatchDirectory,
  inodeWatching,
  fsWatchWithTimestamp,
  sysLog: sysLog2
}) {
  const pollingWatches = /* @__PURE__ */ new Map();
  const fsWatches = /* @__PURE__ */ new Map();
  const fsWatchesRecursive = /* @__PURE__ */ new Map();
  let dynamicPollingWatchFile;
  let fixedChunkSizePollingWatchFile;
  let nonPollingWatchFile;
  let hostRecursiveDirectoryWatcher;
  let hitSystemWatcherLimit = false;
  return {
    watchFile: watchFile2,
    watchDirectory
  };
  function watchFile2(fileName, callback, pollingInterval, options) {
    options = updateOptionsForWatchFile(options, useNonPollingWatchers);
    const watchFileKind = Debug.checkDefined(options.watchFile);
    switch (watchFileKind) {
      case 0 /* FixedPollingInterval */:
        return pollingWatchFile(
          fileName,
          callback,
          250 /* Low */,
          /*options*/
          void 0
        );
      case 1 /* PriorityPollingInterval */:
        return pollingWatchFile(
          fileName,
          callback,
          pollingInterval,
          /*options*/
          void 0
        );
      case 2 /* DynamicPriorityPolling */:
        return ensureDynamicPollingWatchFile()(
          fileName,
          callback,
          pollingInterval,
          /*options*/
          void 0
        );
      case 3 /* FixedChunkSizePolling */:
        return ensureFixedChunkSizePollingWatchFile()(
          fileName,
          callback,
          /* pollingInterval */
          void 0,
          /*options*/
          void 0
        );
      case 4 /* UseFsEvents */:
        return fsWatch(
          fileName,
          0 /* File */,
          createFsWatchCallbackForFileWatcherCallback(fileName, callback, getModifiedTime3),
          /*recursive*/
          false,
          pollingInterval,
          getFallbackOptions(options)
        );
      case 5 /* UseFsEventsOnParentDirectory */:
        if (!nonPollingWatchFile) {
          nonPollingWatchFile = createUseFsEventsOnParentDirectoryWatchFile(fsWatch, useCaseSensitiveFileNames2, getModifiedTime3, fsWatchWithTimestamp);
        }
        return nonPollingWatchFile(fileName, callback, pollingInterval, getFallbackOptions(options));
      default:
        Debug.assertNever(watchFileKind);
    }
  }
  function ensureDynamicPollingWatchFile() {
    return dynamicPollingWatchFile || (dynamicPollingWatchFile = createDynamicPriorityPollingWatchFile({ getModifiedTime: getModifiedTime3, setTimeout: setTimeout2 }));
  }
  function ensureFixedChunkSizePollingWatchFile() {
    return fixedChunkSizePollingWatchFile || (fixedChunkSizePollingWatchFile = createFixedChunkSizePollingWatchFile({ getModifiedTime: getModifiedTime3, setTimeout: setTimeout2 }));
  }
  function updateOptionsForWatchFile(options, useNonPollingWatchers2) {
    if (options && options.watchFile !== void 0) return options;
    switch (tscWatchFile) {
      case "PriorityPollingInterval":
        return { watchFile: 1 /* PriorityPollingInterval */ };
      case "DynamicPriorityPolling":
        return { watchFile: 2 /* DynamicPriorityPolling */ };
      case "UseFsEvents":
        return generateWatchFileOptions(4 /* UseFsEvents */, 1 /* PriorityInterval */, options);
      case "UseFsEventsWithFallbackDynamicPolling":
        return generateWatchFileOptions(4 /* UseFsEvents */, 2 /* DynamicPriority */, options);
      case "UseFsEventsOnParentDirectory":
        useNonPollingWatchers2 = true;
      // fall through
      default:
        return useNonPollingWatchers2 ? (
          // Use notifications from FS to watch with falling back to fs.watchFile
          generateWatchFileOptions(5 /* UseFsEventsOnParentDirectory */, 1 /* PriorityInterval */, options)
        ) : (
          // Default to using fs events
          { watchFile: 4 /* UseFsEvents */ }
        );
    }
  }
  function generateWatchFileOptions(watchFile3, fallbackPolling, options) {
    const defaultFallbackPolling = options == null ? void 0 : options.fallbackPolling;
    return {
      watchFile: watchFile3,
      fallbackPolling: defaultFallbackPolling === void 0 ? fallbackPolling : defaultFallbackPolling
    };
  }
  function watchDirectory(directoryName, callback, recursive, options) {
    if (fsSupportsRecursiveFsWatch) {
      return fsWatch(
        directoryName,
        1 /* Directory */,
        createFsWatchCallbackForDirectoryWatcherCallback(directoryName, callback, options, useCaseSensitiveFileNames2, getCurrentDirectory),
        recursive,
        500 /* Medium */,
        getFallbackOptions(options)
      );
    }
    if (!hostRecursiveDirectoryWatcher) {
      hostRecursiveDirectoryWatcher = createDirectoryWatcherSupportingRecursive({
        useCaseSensitiveFileNames: useCaseSensitiveFileNames2,
        getCurrentDirectory,
        fileSystemEntryExists,
        getAccessibleSortedChildDirectories,
        watchDirectory: nonRecursiveWatchDirectory,
        realpath,
        setTimeout: setTimeout2,
        clearTimeout: clearTimeout2
      });
    }
    return hostRecursiveDirectoryWatcher(directoryName, callback, recursive, options);
  }
  function nonRecursiveWatchDirectory(directoryName, callback, recursive, options) {
    Debug.assert(!recursive);
    const watchDirectoryOptions = updateOptionsForWatchDirectory(options);
    const watchDirectoryKind = Debug.checkDefined(watchDirectoryOptions.watchDirectory);
    switch (watchDirectoryKind) {
      case 1 /* FixedPollingInterval */:
        return pollingWatchFile(
          directoryName,
          () => callback(directoryName),
          500 /* Medium */,
          /*options*/
          void 0
        );
      case 2 /* DynamicPriorityPolling */:
        return ensureDynamicPollingWatchFile()(
          directoryName,
          () => callback(directoryName),
          500 /* Medium */,
          /*options*/
          void 0
        );
      case 3 /* FixedChunkSizePolling */:
        return ensureFixedChunkSizePollingWatchFile()(
          directoryName,
          () => callback(directoryName),
          /* pollingInterval */
          void 0,
          /*options*/
          void 0
        );
      case 0 /* UseFsEvents */:
        return fsWatch(
          directoryName,
          1 /* Directory */,
          createFsWatchCallbackForDirectoryWatcherCallback(directoryName, callback, options, useCaseSensitiveFileNames2, getCurrentDirectory),
          recursive,
          500 /* Medium */,
          getFallbackOptions(watchDirectoryOptions)
        );
      default:
        Debug.assertNever(watchDirectoryKind);
    }
  }
  function updateOptionsForWatchDirectory(options) {
    if (options && options.watchDirectory !== void 0) return options;
    switch (tscWatchDirectory) {
      case "RecursiveDirectoryUsingFsWatchFile":
        return { watchDirectory: 1 /* FixedPollingInterval */ };
      case "RecursiveDirectoryUsingDynamicPriorityPolling":
        return { watchDirectory: 2 /* DynamicPriorityPolling */ };
      default:
        const defaultFallbackPolling = options == null ? void 0 : options.fallbackPolling;
        return {
          watchDirectory: 0 /* UseFsEvents */,
          fallbackPolling: defaultFallbackPolling !== void 0 ? defaultFallbackPolling : void 0
        };
    }
  }
  function pollingWatchFile(fileName, callback, pollingInterval, options) {
    return createSingleWatcherPerName(
      pollingWatches,
      useCaseSensitiveFileNames2,
      fileName,
      callback,
      (cb) => pollingWatchFileWorker(fileName, cb, pollingInterval, options)
    );
  }
  function fsWatch(fileOrDirectory, entryKind, callback, recursive, fallbackPollingInterval, fallbackOptions) {
    return createSingleWatcherPerName(
      recursive ? fsWatchesRecursive : fsWatches,
      useCaseSensitiveFileNames2,
      fileOrDirectory,
      callback,
      (cb) => fsWatchHandlingExistenceOnHost(fileOrDirectory, entryKind, cb, recursive, fallbackPollingInterval, fallbackOptions)
    );
  }
  function fsWatchHandlingExistenceOnHost(fileOrDirectory, entryKind, callback, recursive, fallbackPollingInterval, fallbackOptions) {
    let lastDirectoryPartWithDirectorySeparator;
    let lastDirectoryPart;
    if (inodeWatching) {
      lastDirectoryPartWithDirectorySeparator = fileOrDirectory.substring(fileOrDirectory.lastIndexOf(directorySeparator));
      lastDirectoryPart = lastDirectoryPartWithDirectorySeparator.slice(directorySeparator.length);
    }
    let watcher = !fileSystemEntryExists(fileOrDirectory, entryKind) ? watchMissingFileSystemEntry() : watchPresentFileSystemEntry();
    return {
      close: () => {
        if (watcher) {
          watcher.close();
          watcher = void 0;
        }
      }
    };
    function updateWatcher(createWatcher) {
      if (watcher) {
        sysLog2(`sysLog:: ${fileOrDirectory}:: Changing watcher to ${createWatcher === watchPresentFileSystemEntry ? "Present" : "Missing"}FileSystemEntryWatcher`);
        watcher.close();
        watcher = createWatcher();
      }
    }
    function watchPresentFileSystemEntry() {
      if (hitSystemWatcherLimit) {
        sysLog2(`sysLog:: ${fileOrDirectory}:: Defaulting to watchFile`);
        return watchPresentFileSystemEntryWithFsWatchFile();
      }
      try {
        const presentWatcher = (entryKind === 1 /* Directory */ || !fsWatchWithTimestamp ? fsWatchWorker : fsWatchWorkerHandlingTimestamp)(
          fileOrDirectory,
          recursive,
          inodeWatching ? callbackChangingToMissingFileSystemEntry : callback
        );
        presentWatcher.on("error", () => {
          callback("rename", "");
          updateWatcher(watchMissingFileSystemEntry);
        });
        return presentWatcher;
      } catch (e) {
        hitSystemWatcherLimit || (hitSystemWatcherLimit = e.code === "ENOSPC");
        sysLog2(`sysLog:: ${fileOrDirectory}:: Changing to watchFile`);
        return watchPresentFileSystemEntryWithFsWatchFile();
      }
    }
    function callbackChangingToMissingFileSystemEntry(event, relativeName) {
      let originalRelativeName;
      if (relativeName && endsWith(relativeName, "~")) {
        originalRelativeName = relativeName;
        relativeName = relativeName.slice(0, relativeName.length - 1);
      }
      if (event === "rename" && (!relativeName || relativeName === lastDirectoryPart || endsWith(relativeName, lastDirectoryPartWithDirectorySeparator))) {
        const modifiedTime = getModifiedTime3(fileOrDirectory) || missingFileModifiedTime;
        if (originalRelativeName) callback(event, originalRelativeName, modifiedTime);
        callback(event, relativeName, modifiedTime);
        if (inodeWatching) {
          updateWatcher(modifiedTime === missingFileModifiedTime ? watchMissingFileSystemEntry : watchPresentFileSystemEntry);
        } else if (modifiedTime === missingFileModifiedTime) {
          updateWatcher(watchMissingFileSystemEntry);
        }
      } else {
        if (originalRelativeName) callback(event, originalRelativeName);
        callback(event, relativeName);
      }
    }
    function watchPresentFileSystemEntryWithFsWatchFile() {
      return watchFile2(
        fileOrDirectory,
        createFileWatcherCallback(callback),
        fallbackPollingInterval,
        fallbackOptions
      );
    }
    function watchMissingFileSystemEntry() {
      return watchFile2(
        fileOrDirectory,
        (_fileName, eventKind, modifiedTime) => {
          if (eventKind === 0 /* Created */) {
            modifiedTime || (modifiedTime = getModifiedTime3(fileOrDirectory) || missingFileModifiedTime);
            if (modifiedTime !== missingFileModifiedTime) {
              callback("rename", "", modifiedTime);
              updateWatcher(watchPresentFileSystemEntry);
            }
          }
        },
        fallbackPollingInterval,
        fallbackOptions
      );
    }
  }
  function fsWatchWorkerHandlingTimestamp(fileOrDirectory, recursive, callback) {
    let modifiedTime = getModifiedTime3(fileOrDirectory) || missingFileModifiedTime;
    return fsWatchWorker(fileOrDirectory, recursive, (eventName, relativeFileName, currentModifiedTime) => {
      if (eventName === "change") {
        currentModifiedTime || (currentModifiedTime = getModifiedTime3(fileOrDirectory) || missingFileModifiedTime);
        if (currentModifiedTime.getTime() === modifiedTime.getTime()) return;
      }
      modifiedTime = currentModifiedTime || getModifiedTime3(fileOrDirectory) || missingFileModifiedTime;
      callback(eventName, relativeFileName, modifiedTime);
    });
  }
}
function patchWriteFileEnsuringDirectory(sys2) {
  const originalWriteFile = sys2.writeFile;
  sys2.writeFile = (path, data, writeBom) => writeFileEnsuringDirectories(
    path,
    data,
    !!writeBom,
    (path2, data2, writeByteOrderMark) => originalWriteFile.call(sys2, path2, data2, writeByteOrderMark),
    (path2) => sys2.createDirectory(path2),
    (path2) => sys2.directoryExists(path2)
  );
}
var sys = (() => {
  const byteOrderMarkIndicator = "\uFEFF";
  function getNodeSystem() {
    const nativePattern = /^native |^\([^)]+\)$|^(?:internal[\\/]|[\w\s]+(?:\.js)?$)/;
    const _fs = require("fs");
    const _path = require("path");
    const _os = require("os");
    let _crypto;
    try {
      _crypto = require("crypto");
    } catch {
      _crypto = void 0;
    }
    let activeSession;
    let profilePath = "./profile.cpuprofile";
    const isMacOs = process.platform === "darwin";
    const isLinuxOrMacOs = process.platform === "linux" || isMacOs;
    const statSyncOptions = { throwIfNoEntry: false };
    const platform = _os.platform();
    const useCaseSensitiveFileNames2 = isFileSystemCaseSensitive();
    const fsRealpath = !!_fs.realpathSync.native ? process.platform === "win32" ? fsRealPathHandlingLongPath : _fs.realpathSync.native : _fs.realpathSync;
    const executingFilePath = __filename.endsWith("sys.js") ? _path.join(_path.dirname(__dirname), "__fake__.js") : __filename;
    const fsSupportsRecursiveFsWatch = process.platform === "win32" || isMacOs;
    const getCurrentDirectory = memoize(() => process.cwd());
    const { watchFile: watchFile2, watchDirectory } = createSystemWatchFunctions({
      pollingWatchFileWorker: fsWatchFileWorker,
      getModifiedTime: getModifiedTime3,
      setTimeout,
      clearTimeout,
      fsWatchWorker,
      useCaseSensitiveFileNames: useCaseSensitiveFileNames2,
      getCurrentDirectory,
      fileSystemEntryExists,
      // Node 4.0 `fs.watch` function supports the "recursive" option on both OSX and Windows
      // (ref: https://github.com/nodejs/node/pull/2649 and https://github.com/Microsoft/TypeScript/issues/4643)
      fsSupportsRecursiveFsWatch,
      getAccessibleSortedChildDirectories: (path) => getAccessibleFileSystemEntries(path).directories,
      realpath,
      tscWatchFile: process.env.TSC_WATCHFILE,
      useNonPollingWatchers: !!process.env.TSC_NONPOLLING_WATCHER,
      tscWatchDirectory: process.env.TSC_WATCHDIRECTORY,
      inodeWatching: isLinuxOrMacOs,
      fsWatchWithTimestamp: isMacOs,
      sysLog
    });
    const nodeSystem = {
      args: process.argv.slice(2),
      newLine: _os.EOL,
      useCaseSensitiveFileNames: useCaseSensitiveFileNames2,
      write(s) {
        process.stdout.write(s);
      },
      getWidthOfTerminal() {
        return process.stdout.columns;
      },
      writeOutputIsTTY() {
        return process.stdout.isTTY;
      },
      readFile,
      writeFile: writeFile2,
      watchFile: watchFile2,
      watchDirectory,
      preferNonRecursiveWatch: !fsSupportsRecursiveFsWatch,
      resolvePath: (path) => _path.resolve(path),
      fileExists,
      directoryExists,
      getAccessibleFileSystemEntries,
      createDirectory(directoryName) {
        if (!nodeSystem.directoryExists(directoryName)) {
          try {
            _fs.mkdirSync(directoryName);
          } catch (e) {
            if (e.code !== "EEXIST") {
              throw e;
            }
          }
        }
      },
      getExecutingFilePath() {
        return executingFilePath;
      },
      getCurrentDirectory,
      getDirectories,
      getEnvironmentVariable(name) {
        return process.env[name] || "";
      },
      readDirectory,
      getModifiedTime: getModifiedTime3,
      setModifiedTime,
      deleteFile,
      createHash: _crypto ? createSHA256Hash : generateDjb2Hash,
      createSHA256Hash: _crypto ? createSHA256Hash : void 0,
      getMemoryUsage() {
        if (global.gc) {
          global.gc();
        }
        return process.memoryUsage().heapUsed;
      },
      getFileSize(path) {
        const stat = statSync(path);
        if (stat == null ? void 0 : stat.isFile()) {
          return stat.size;
        }
        return 0;
      },
      exit(exitCode) {
        disableCPUProfiler(() => process.exit(exitCode));
      },
      enableCPUProfiler,
      disableCPUProfiler,
      cpuProfilingEnabled: () => !!activeSession || contains(process.execArgv, "--cpu-prof") || contains(process.execArgv, "--prof"),
      realpath,
      debugMode: !!process.env.NODE_INSPECTOR_IPC || !!process.env.VSCODE_INSPECTOR_OPTIONS || some(process.execArgv, (arg) => /^--(?:inspect|debug)(?:-brk)?(?:=\d+)?$/i.test(arg)) || !!process.recordreplay,
      tryEnableSourceMapsForHost() {
        try {
          require("source-map-support").install();
        } catch {
        }
      },
      setTimeout,
      clearTimeout,
      clearScreen: () => {
        process.stdout.write("\x1B[2J\x1B[3J\x1B[H");
      },
      setBlocking: () => {
        var _a;
        const handle = (_a = process.stdout) == null ? void 0 : _a._handle;
        if (handle && handle.setBlocking) {
          handle.setBlocking(true);
        }
      },
      base64decode: (input) => Buffer.from(input, "base64").toString("utf8"),
      base64encode: (input) => Buffer.from(input).toString("base64"),
      require: (baseDir, moduleName) => {
        try {
          const modulePath = resolveJSModule(moduleName, baseDir, nodeSystem);
          return { module: require(modulePath), modulePath, error: void 0 };
        } catch (error) {
          return { module: void 0, modulePath: void 0, error };
        }
      }
    };
    return nodeSystem;
    function statSync(path) {
      try {
        return _fs.statSync(path, statSyncOptions);
      } catch {
        return void 0;
      }
    }
    function enableCPUProfiler(path, cb) {
      if (activeSession) {
        cb();
        return false;
      }
      const inspector = require("inspector");
      if (!inspector || !inspector.Session) {
        cb();
        return false;
      }
      const session = new inspector.Session();
      session.connect();
      session.post("Profiler.enable", () => {
        session.post("Profiler.start", () => {
          activeSession = session;
          profilePath = path;
          cb();
        });
      });
      return true;
    }
    function cleanupPaths(profile) {
      let externalFileCounter = 0;
      const remappedPaths = /* @__PURE__ */ new Map();
      const normalizedDir = normalizeSlashes(_path.dirname(executingFilePath));
      const fileUrlRoot = `file://${getRootLength(normalizedDir) === 1 ? "" : "/"}${normalizedDir}`;
      for (const node of profile.nodes) {
        if (node.callFrame.url) {
          const url = normalizeSlashes(node.callFrame.url);
          if (containsPath(fileUrlRoot, url, useCaseSensitiveFileNames2)) {
            node.callFrame.url = getRelativePathToDirectoryOrUrl(
              fileUrlRoot,
              url,
              fileUrlRoot,
              createGetCanonicalFileName(useCaseSensitiveFileNames2),
              /*isAbsolutePathAnUrl*/
              true
            );
          } else if (!nativePattern.test(url)) {
            node.callFrame.url = (remappedPaths.has(url) ? remappedPaths : remappedPaths.set(url, `external${externalFileCounter}.js`)).get(url);
            externalFileCounter++;
          }
        }
      }
      return profile;
    }
    function disableCPUProfiler(cb) {
      if (activeSession && activeSession !== "stopping") {
        const s = activeSession;
        activeSession.post("Profiler.stop", (err, { profile }) => {
          var _a;
          if (!err) {
            if ((_a = statSync(profilePath)) == null ? void 0 : _a.isDirectory()) {
              profilePath = _path.join(profilePath, `${(/* @__PURE__ */ new Date()).toISOString().replace(/:/g, "-")}+P${process.pid}.cpuprofile`);
            }
            try {
              _fs.mkdirSync(_path.dirname(profilePath), { recursive: true });
            } catch {
            }
            _fs.writeFileSync(profilePath, JSON.stringify(cleanupPaths(profile)));
          }
          activeSession = void 0;
          s.disconnect();
          cb();
        });
        activeSession = "stopping";
        return true;
      } else {
        cb();
        return false;
      }
    }
    function isFileSystemCaseSensitive() {
      if (typeof process.versions.pnp !== `undefined`) {
        return true;
      }
      if (platform === "win32" || platform === "win64") {
        return false;
      }
      return !fileExists(swapCase(__filename));
    }
    function swapCase(s) {
      return s.replace(/\w/g, (ch) => {
        const up = ch.toUpperCase();
        return ch === up ? ch.toLowerCase() : up;
      });
    }
    function fsWatchFileWorker(fileName, callback, pollingInterval) {
      _fs.watchFile(fileName, { persistent: true, interval: pollingInterval }, fileChanged);
      let eventKind;
      return {
        close: () => _fs.unwatchFile(fileName, fileChanged)
      };
      function fileChanged(curr, prev) {
        const isPreviouslyDeleted = +prev.mtime === 0 || eventKind === 2 /* Deleted */;
        if (+curr.mtime === 0) {
          if (isPreviouslyDeleted) {
            return;
          }
          eventKind = 2 /* Deleted */;
        } else if (isPreviouslyDeleted) {
          eventKind = 0 /* Created */;
        } else if (+curr.mtime === +prev.mtime) {
          return;
        } else {
          eventKind = 1 /* Changed */;
        }
        callback(fileName, eventKind, curr.mtime);
      }
    }
    function fsWatchWorker(fileOrDirectory, recursive, callback) {
      return _fs.watch(
        fileOrDirectory,
        fsSupportsRecursiveFsWatch ? { persistent: true, recursive: !!recursive } : { persistent: true },
        callback
      );
    }
    function readFile(fileName, _encoding) {
      let buffer;
      try {
        buffer = _fs.readFileSync(fileName);
      } catch {
        return void 0;
      }
      let len = buffer.length;
      if (len >= 2 && buffer[0] === 254 && buffer[1] === 255) {
        len &= ~1;
        for (let i = 0; i < len; i += 2) {
          const temp = buffer[i];
          buffer[i] = buffer[i + 1];
          buffer[i + 1] = temp;
        }
        return buffer.toString("utf16le", 2);
      }
      if (len >= 2 && buffer[0] === 255 && buffer[1] === 254) {
        return buffer.toString("utf16le", 2);
      }
      if (len >= 3 && buffer[0] === 239 && buffer[1] === 187 && buffer[2] === 191) {
        return buffer.toString("utf8", 3);
      }
      return buffer.toString("utf8");
    }
    function writeFile2(fileName, data, writeByteOrderMark) {
      if (writeByteOrderMark) {
        data = byteOrderMarkIndicator + data;
      }
      let fd;
      try {
        fd = _fs.openSync(fileName, "w");
        _fs.writeSync(
          fd,
          data,
          /*position*/
          void 0,
          "utf8"
        );
      } finally {
        if (fd !== void 0) {
          _fs.closeSync(fd);
        }
      }
    }
    function getAccessibleFileSystemEntries(path) {
      try {
        const entries = _fs.readdirSync(path || ".", { withFileTypes: true });
        const files = [];
        const directories = [];
        for (const dirent of entries) {
          const entry = typeof dirent === "string" ? dirent : dirent.name;
          if (entry === "." || entry === "..") {
            continue;
          }
          let stat;
          if (typeof dirent === "string" || dirent.isSymbolicLink()) {
            const name = combinePaths(path, entry);
            stat = statSync(name);
            if (!stat) {
              continue;
            }
          } else {
            stat = dirent;
          }
          if (stat.isFile()) {
            files.push(entry);
          } else if (stat.isDirectory()) {
            directories.push(entry);
          }
        }
        files.sort();
        directories.sort();
        return { files, directories };
      } catch {
        return emptyFileSystemEntries;
      }
    }
    function readDirectory(path, extensions, excludes, includes, depth) {
      return matchFiles(path, extensions, excludes, includes, useCaseSensitiveFileNames2, process.cwd(), depth, getAccessibleFileSystemEntries, realpath);
    }
    function fileSystemEntryExists(path, entryKind) {
      const stat = statSync(path);
      if (!stat) {
        return false;
      }
      switch (entryKind) {
        case 0 /* File */:
          return stat.isFile();
        case 1 /* Directory */:
          return stat.isDirectory();
        default:
          return false;
      }
    }
    function fileExists(path) {
      return fileSystemEntryExists(path, 0 /* File */);
    }
    function directoryExists(path) {
      return fileSystemEntryExists(path, 1 /* Directory */);
    }
    function getDirectories(path) {
      return getAccessibleFileSystemEntries(path).directories.slice();
    }
    function fsRealPathHandlingLongPath(path) {
      return path.length < 260 ? _fs.realpathSync.native(path) : _fs.realpathSync(path);
    }
    function realpath(path) {
      try {
        return fsRealpath(path);
      } catch {
        return path;
      }
    }
    function getModifiedTime3(path) {
      var _a;
      return (_a = statSync(path)) == null ? void 0 : _a.mtime;
    }
    function setModifiedTime(path, time) {
      try {
        _fs.utimesSync(path, time, time);
      } catch {
        return;
      }
    }
    function deleteFile(path) {
      try {
        return _fs.unlinkSync(path);
      } catch {
        return;
      }
    }
    function createSHA256Hash(data) {
      const hash = _crypto.createHash("sha256");
      hash.update(data);
      return hash.digest("hex");
    }
  }
  let sys2;
  if (isNodeLikeSystem()) {
    sys2 = getNodeSystem();
  }
  if (sys2) {
    patchWriteFileEnsuringDirectory(sys2);
  }
  return sys2;
})();
if (sys && sys.getEnvironmentVariable) {
  setCustomPollingValues(sys);
  Debug.setAssertionLevel(
    /^development$/i.test(sys.getEnvironmentVariable("NODE_ENV")) ? 1 /* Normal */ : 0 /* None */
  );
}
if (sys && sys.debugMode) {
  Debug.isDebugging = true;
}

// src/compiler/path.ts
var directorySeparator = "/";
var altDirectorySeparator = "\\";
var urlSchemeSeparator = "://";
var backslashRegExp = /\\/g;
function isAnyDirectorySeparator(charCode) {
  return charCode === 47 /* slash */ || charCode === 92 /* backslash */;
}
function isRootedDiskPath(path) {
  return getEncodedRootLength(path) > 0;
}
function isDiskPathRoot(path) {
  const rootLength = getEncodedRootLength(path);
  return rootLength > 0 && rootLength === path.length;
}
function pathIsAbsolute(path) {
  return getEncodedRootLength(path) !== 0;
}
function pathIsRelative(path) {
  return /^\.\.?(?:$|[\\/])/.test(path);
}
function pathIsBareSpecifier(path) {
  return !pathIsAbsolute(path) && !pathIsRelative(path);
}
function hasExtension(fileName) {
  return getBaseFileName(fileName).includes(".");
}
function fileExtensionIs(path, extension) {
  return path.length > extension.length && endsWith(path, extension);
}
function fileExtensionIsOneOf(path, extensions) {
  for (const extension of extensions) {
    if (fileExtensionIs(path, extension)) {
      return true;
    }
  }
  return false;
}
function hasTrailingDirectorySeparator(path) {
  return path.length > 0 && isAnyDirectorySeparator(path.charCodeAt(path.length - 1));
}
function isVolumeCharacter(charCode) {
  return charCode >= 97 /* a */ && charCode <= 122 /* z */ || charCode >= 65 /* A */ && charCode <= 90 /* Z */;
}
function getFileUrlVolumeSeparatorEnd(url, start) {
  const ch0 = url.charCodeAt(start);
  if (ch0 === 58 /* colon */) return start + 1;
  if (ch0 === 37 /* percent */ && url.charCodeAt(start + 1) === 51 /* _3 */) {
    const ch2 = url.charCodeAt(start + 2);
    if (ch2 === 97 /* a */ || ch2 === 65 /* A */) return start + 3;
  }
  return -1;
}
function getEncodedRootLength(path) {
  if (!path) return 0;
  const ch0 = path.charCodeAt(0);
  if (ch0 === 47 /* slash */ || ch0 === 92 /* backslash */) {
    if (path.charCodeAt(1) !== ch0) return 1;
    const p1 = path.indexOf(ch0 === 47 /* slash */ ? directorySeparator : altDirectorySeparator, 2);
    if (p1 < 0) return path.length;
    return p1 + 1;
  }
  if (isVolumeCharacter(ch0) && path.charCodeAt(1) === 58 /* colon */) {
    const ch2 = path.charCodeAt(2);
    if (ch2 === 47 /* slash */ || ch2 === 92 /* backslash */) return 3;
    if (path.length === 2) return 2;
  }
  const schemeEnd = path.indexOf(urlSchemeSeparator);
  if (schemeEnd !== -1) {
    const authorityStart = schemeEnd + urlSchemeSeparator.length;
    const authorityEnd = path.indexOf(directorySeparator, authorityStart);
    if (authorityEnd !== -1) {
      const scheme = path.slice(0, schemeEnd);
      const authority = path.slice(authorityStart, authorityEnd);
      if (scheme === "file" && (authority === "" || authority === "localhost") && isVolumeCharacter(path.charCodeAt(authorityEnd + 1))) {
        const volumeSeparatorEnd = getFileUrlVolumeSeparatorEnd(path, authorityEnd + 2);
        if (volumeSeparatorEnd !== -1) {
          if (path.charCodeAt(volumeSeparatorEnd) === 47 /* slash */) {
            return ~(volumeSeparatorEnd + 1);
          }
          if (volumeSeparatorEnd === path.length) {
            return ~volumeSeparatorEnd;
          }
        }
      }
      return ~(authorityEnd + 1);
    }
    return ~path.length;
  }
  return 0;
}
function getRootLength(path) {
  const rootLength = getEncodedRootLength(path);
  return rootLength < 0 ? ~rootLength : rootLength;
}
function getDirectoryPath(path) {
  path = normalizeSlashes(path);
  const rootLength = getRootLength(path);
  if (rootLength === path.length) return path;
  path = removeTrailingDirectorySeparator(path);
  return path.slice(0, Math.max(rootLength, path.lastIndexOf(directorySeparator)));
}
function getBaseFileName(path, extensions, ignoreCase) {
  path = normalizeSlashes(path);
  const rootLength = getRootLength(path);
  if (rootLength === path.length) return "";
  path = removeTrailingDirectorySeparator(path);
  const name = path.slice(Math.max(getRootLength(path), path.lastIndexOf(directorySeparator) + 1));
  const extension = extensions !== void 0 && ignoreCase !== void 0 ? getAnyExtensionFromPath(name, extensions, ignoreCase) : void 0;
  return extension ? name.slice(0, name.length - extension.length) : name;
}
function tryGetExtensionFromPath(path, extension, stringEqualityComparer) {
  if (!startsWith(extension, ".")) extension = "." + extension;
  if (path.length >= extension.length && path.charCodeAt(path.length - extension.length) === 46 /* dot */) {
    const pathExtension = path.slice(path.length - extension.length);
    if (stringEqualityComparer(pathExtension, extension)) {
      return pathExtension;
    }
  }
}
function getAnyExtensionFromPathWorker(path, extensions, stringEqualityComparer) {
  if (typeof extensions === "string") {
    return tryGetExtensionFromPath(path, extensions, stringEqualityComparer) || "";
  }
  for (const extension of extensions) {
    const result = tryGetExtensionFromPath(path, extension, stringEqualityComparer);
    if (result) return result;
  }
  return "";
}
function getAnyExtensionFromPath(path, extensions, ignoreCase) {
  if (extensions) {
    return getAnyExtensionFromPathWorker(removeTrailingDirectorySeparator(path), extensions, ignoreCase ? equateStringsCaseInsensitive : equateStringsCaseSensitive);
  }
  const baseFileName = getBaseFileName(path);
  const extensionIndex = baseFileName.lastIndexOf(".");
  if (extensionIndex >= 0) {
    return baseFileName.substring(extensionIndex);
  }
  return "";
}
function pathComponents(path, rootLength) {
  const root = path.substring(0, rootLength);
  const rest = path.substring(rootLength).split(directorySeparator);
  if (rest.length && !lastOrUndefined(rest)) rest.pop();
  return [root, ...rest];
}
function getPathComponents(path, currentDirectory = "") {
  path = combinePaths(currentDirectory, path);
  return pathComponents(path, getRootLength(path));
}
function getPathFromPathComponents(pathComponents2, length2) {
  if (pathComponents2.length === 0) return "";
  const root = pathComponents2[0] && ensureTrailingDirectorySeparator(pathComponents2[0]);
  return root + pathComponents2.slice(1, length2).join(directorySeparator);
}
function normalizeSlashes(path) {
  return path.includes("\\") ? path.replace(backslashRegExp, directorySeparator) : path;
}
function reducePathComponents(components) {
  if (!some(components)) return [];
  const reduced = [components[0]];
  for (let i = 1; i < components.length; i++) {
    const component = components[i];
    if (!component) continue;
    if (component === ".") continue;
    if (component === "..") {
      if (reduced.length > 1) {
        if (reduced[reduced.length - 1] !== "..") {
          reduced.pop();
          continue;
        }
      } else if (reduced[0]) continue;
    }
    reduced.push(component);
  }
  return reduced;
}
function combinePaths(path, ...paths) {
  if (path) path = normalizeSlashes(path);
  for (let relativePath of paths) {
    if (!relativePath) continue;
    relativePath = normalizeSlashes(relativePath);
    if (!path || getRootLength(relativePath) !== 0) {
      path = relativePath;
    } else {
      path = ensureTrailingDirectorySeparator(path) + relativePath;
    }
  }
  return path;
}
function resolvePath(path, ...paths) {
  return normalizePath(some(paths) ? combinePaths(path, ...paths) : normalizeSlashes(path));
}
function getNormalizedPathComponents(path, currentDirectory) {
  return reducePathComponents(getPathComponents(path, currentDirectory));
}
function getNormalizedAbsolutePath(fileName, currentDirectory) {
  return getPathFromPathComponents(getNormalizedPathComponents(fileName, currentDirectory));
}
function normalizePath(path) {
  path = normalizeSlashes(path);
  if (!relativePathSegmentRegExp.test(path)) {
    return path;
  }
  const simplified = path.replace(/\/\.\//g, "/").replace(/^\.\//, "");
  if (simplified !== path) {
    path = simplified;
    if (!relativePathSegmentRegExp.test(path)) {
      return path;
    }
  }
  const normalized = getPathFromPathComponents(reducePathComponents(getPathComponents(path)));
  return normalized && hasTrailingDirectorySeparator(path) ? ensureTrailingDirectorySeparator(normalized) : normalized;
}
function getPathWithoutRoot(pathComponents2) {
  if (pathComponents2.length === 0) return "";
  return pathComponents2.slice(1).join(directorySeparator);
}
function getNormalizedAbsolutePathWithoutRoot(fileName, currentDirectory) {
  return getPathWithoutRoot(getNormalizedPathComponents(fileName, currentDirectory));
}
function toPath(fileName, basePath, getCanonicalFileName) {
  const nonCanonicalizedPath = isRootedDiskPath(fileName) ? normalizePath(fileName) : getNormalizedAbsolutePath(fileName, basePath);
  return getCanonicalFileName(nonCanonicalizedPath);
}
function removeTrailingDirectorySeparator(path) {
  if (hasTrailingDirectorySeparator(path)) {
    return path.substr(0, path.length - 1);
  }
  return path;
}
function ensureTrailingDirectorySeparator(path) {
  if (!hasTrailingDirectorySeparator(path)) {
    return path + directorySeparator;
  }
  return path;
}
function ensurePathIsNonModuleName(path) {
  return !pathIsAbsolute(path) && !pathIsRelative(path) ? "./" + path : path;
}
function changeAnyExtension(path, ext, extensions, ignoreCase) {
  const pathext = extensions !== void 0 && ignoreCase !== void 0 ? getAnyExtensionFromPath(path, extensions, ignoreCase) : getAnyExtensionFromPath(path);
  return pathext ? path.slice(0, path.length - pathext.length) + (startsWith(ext, ".") ? ext : "." + ext) : path;
}
function changeFullExtension(path, newExtension) {
  const declarationExtension = getDeclarationFileExtension(path);
  if (declarationExtension) {
    return path.slice(0, path.length - declarationExtension.length) + (startsWith(newExtension, ".") ? newExtension : "." + newExtension);
  }
  return changeAnyExtension(path, newExtension);
}
var relativePathSegmentRegExp = /\/\/|(?:^|\/)\.\.?(?:$|\/)/;
function comparePathsWorker(a, b, componentComparer) {
  if (a === b) return 0 /* EqualTo */;
  if (a === void 0) return -1 /* LessThan */;
  if (b === void 0) return 1 /* GreaterThan */;
  const aRoot = a.substring(0, getRootLength(a));
  const bRoot = b.substring(0, getRootLength(b));
  const result = compareStringsCaseInsensitive(aRoot, bRoot);
  if (result !== 0 /* EqualTo */) {
    return result;
  }
  const aRest = a.substring(aRoot.length);
  const bRest = b.substring(bRoot.length);
  if (!relativePathSegmentRegExp.test(aRest) && !relativePathSegmentRegExp.test(bRest)) {
    return componentComparer(aRest, bRest);
  }
  const aComponents = reducePathComponents(getPathComponents(a));
  const bComponents = reducePathComponents(getPathComponents(b));
  const sharedLength = Math.min(aComponents.length, bComponents.length);
  for (let i = 1; i < sharedLength; i++) {
    const result2 = componentComparer(aComponents[i], bComponents[i]);
    if (result2 !== 0 /* EqualTo */) {
      return result2;
    }
  }
  return compareValues(aComponents.length, bComponents.length);
}
function comparePaths(a, b, currentDirectory, ignoreCase) {
  if (typeof currentDirectory === "string") {
    a = combinePaths(currentDirectory, a);
    b = combinePaths(currentDirectory, b);
  } else if (typeof currentDirectory === "boolean") {
    ignoreCase = currentDirectory;
  }
  return comparePathsWorker(a, b, getStringComparer(ignoreCase));
}
function containsPath(parent, child, currentDirectory, ignoreCase) {
  if (typeof currentDirectory === "string") {
    parent = combinePaths(currentDirectory, parent);
    child = combinePaths(currentDirectory, child);
  } else if (typeof currentDirectory === "boolean") {
    ignoreCase = currentDirectory;
  }
  if (parent === void 0 || child === void 0) return false;
  if (parent === child) return true;
  const parentComponents = reducePathComponents(getPathComponents(parent));
  const childComponents = reducePathComponents(getPathComponents(child));
  if (childComponents.length < parentComponents.length) {
    return false;
  }
  const componentEqualityComparer = ignoreCase ? equateStringsCaseInsensitive : equateStringsCaseSensitive;
  for (let i = 0; i < parentComponents.length; i++) {
    const equalityComparer = i === 0 ? equateStringsCaseInsensitive : componentEqualityComparer;
    if (!equalityComparer(parentComponents[i], childComponents[i])) {
      return false;
    }
  }
  return true;
}
function startsWithDirectory(fileName, directoryName, getCanonicalFileName) {
  const canonicalFileName = getCanonicalFileName(fileName);
  const canonicalDirectoryName = getCanonicalFileName(directoryName);
  return startsWith(canonicalFileName, canonicalDirectoryName + "/") || startsWith(canonicalFileName, canonicalDirectoryName + "\\");
}
function getPathComponentsRelativeTo(from, to, stringEqualityComparer, getCanonicalFileName) {
  const fromComponents = reducePathComponents(getPathComponents(from));
  const toComponents = reducePathComponents(getPathComponents(to));
  let start;
  for (start = 0; start < fromComponents.length && start < toComponents.length; start++) {
    const fromComponent = getCanonicalFileName(fromComponents[start]);
    const toComponent = getCanonicalFileName(toComponents[start]);
    const comparer = start === 0 ? equateStringsCaseInsensitive : stringEqualityComparer;
    if (!comparer(fromComponent, toComponent)) break;
  }
  if (start === 0) {
    return toComponents;
  }
  const components = toComponents.slice(start);
  const relative = [];
  for (; start < fromComponents.length; start++) {
    relative.push("..");
  }
  return ["", ...relative, ...components];
}
function getRelativePathFromDirectory(fromDirectory, to, getCanonicalFileNameOrIgnoreCase) {
  Debug.assert(getRootLength(fromDirectory) > 0 === getRootLength(to) > 0, "Paths must either both be absolute or both be relative");
  const getCanonicalFileName = typeof getCanonicalFileNameOrIgnoreCase === "function" ? getCanonicalFileNameOrIgnoreCase : identity;
  const ignoreCase = typeof getCanonicalFileNameOrIgnoreCase === "boolean" ? getCanonicalFileNameOrIgnoreCase : false;
  const pathComponents2 = getPathComponentsRelativeTo(fromDirectory, to, ignoreCase ? equateStringsCaseInsensitive : equateStringsCaseSensitive, getCanonicalFileName);
  return getPathFromPathComponents(pathComponents2);
}
function convertToRelativePath(absoluteOrRelativePath, basePath, getCanonicalFileName) {
  return !isRootedDiskPath(absoluteOrRelativePath) ? absoluteOrRelativePath : getRelativePathToDirectoryOrUrl(
    basePath,
    absoluteOrRelativePath,
    basePath,
    getCanonicalFileName,
    /*isAbsolutePathAnUrl*/
    false
  );
}
function getRelativePathFromFile(from, to, getCanonicalFileName) {
  return ensurePathIsNonModuleName(getRelativePathFromDirectory(getDirectoryPath(from), to, getCanonicalFileName));
}
function getRelativePathToDirectoryOrUrl(directoryPathOrUrl, relativeOrAbsolutePath, currentDirectory, getCanonicalFileName, isAbsolutePathAnUrl) {
  const pathComponents2 = getPathComponentsRelativeTo(
    resolvePath(currentDirectory, directoryPathOrUrl),
    resolvePath(currentDirectory, relativeOrAbsolutePath),
    equateStringsCaseSensitive,
    getCanonicalFileName
  );
  const firstComponent = pathComponents2[0];
  if (isAbsolutePathAnUrl && isRootedDiskPath(firstComponent)) {
    const prefix = firstComponent.charAt(0) === directorySeparator ? "file://" : "file:///";
    pathComponents2[0] = prefix + firstComponent;
  }
  return getPathFromPathComponents(pathComponents2);
}
function forEachAncestorDirectory(directory, callback) {
  while (true) {
    const result = callback(directory);
    if (result !== void 0) {
      return result;
    }
    const parentPath = getDirectoryPath(directory);
    if (parentPath === directory) {
      return void 0;
    }
    directory = parentPath;
  }
}
function isNodeModulesDirectory(dirPath) {
  return endsWith(dirPath, "/node_modules");
}

// src/compiler/diagnosticInformationMap.generated.ts
function diag(code, category, key, message, reportsUnnecessary, elidedInCompatabilityPyramid, reportsDeprecated) {
  return { code, category, key, message, reportsUnnecessary, elidedInCompatabilityPyramid, reportsDeprecated };
}
var Diagnostics = {
  Unterminated_string_literal: diag(1002, 1 /* Error */, "Unterminated_string_literal_1002", "Unterminated string literal."),
  Identifier_expected: diag(1003, 1 /* Error */, "Identifier_expected_1003", "Identifier expected."),
  _0_expected: diag(1005, 1 /* Error */, "_0_expected_1005", "'{0}' expected."),
  A_file_cannot_have_a_reference_to_itself: diag(1006, 1 /* Error */, "A_file_cannot_have_a_reference_to_itself_1006", "A file cannot have a reference to itself."),
  The_parser_expected_to_find_a_1_to_match_the_0_token_here: diag(1007, 1 /* Error */, "The_parser_expected_to_find_a_1_to_match_the_0_token_here_1007", "The parser expected to find a '{1}' to match the '{0}' token here."),
  Trailing_comma_not_allowed: diag(1009, 1 /* Error */, "Trailing_comma_not_allowed_1009", "Trailing comma not allowed."),
  Asterisk_Slash_expected: diag(1010, 1 /* Error */, "Asterisk_Slash_expected_1010", "'*/' expected."),
  An_element_access_expression_should_take_an_argument: diag(1011, 1 /* Error */, "An_element_access_expression_should_take_an_argument_1011", "An element access expression should take an argument."),
  Unexpected_token: diag(1012, 1 /* Error */, "Unexpected_token_1012", "Unexpected token."),
  A_rest_parameter_or_binding_pattern_may_not_have_a_trailing_comma: diag(1013, 1 /* Error */, "A_rest_parameter_or_binding_pattern_may_not_have_a_trailing_comma_1013", "A rest parameter or binding pattern may not have a trailing comma."),
  A_rest_parameter_must_be_last_in_a_parameter_list: diag(1014, 1 /* Error */, "A_rest_parameter_must_be_last_in_a_parameter_list_1014", "A rest parameter must be last in a parameter list."),
  Parameter_cannot_have_question_mark_and_initializer: diag(1015, 1 /* Error */, "Parameter_cannot_have_question_mark_and_initializer_1015", "Parameter cannot have question mark and initializer."),
  A_required_parameter_cannot_follow_an_optional_parameter: diag(1016, 1 /* Error */, "A_required_parameter_cannot_follow_an_optional_parameter_1016", "A required parameter cannot follow an optional parameter."),
  An_index_signature_cannot_have_a_rest_parameter: diag(1017, 1 /* Error */, "An_index_signature_cannot_have_a_rest_parameter_1017", "An index signature cannot have a rest parameter."),
  An_index_signature_parameter_cannot_have_an_accessibility_modifier: diag(1018, 1 /* Error */, "An_index_signature_parameter_cannot_have_an_accessibility_modifier_1018", "An index signature parameter cannot have an accessibility modifier."),
  An_index_signature_parameter_cannot_have_a_question_mark: diag(1019, 1 /* Error */, "An_index_signature_parameter_cannot_have_a_question_mark_1019", "An index signature parameter cannot have a question mark."),
  An_index_signature_parameter_cannot_have_an_initializer: diag(1020, 1 /* Error */, "An_index_signature_parameter_cannot_have_an_initializer_1020", "An index signature parameter cannot have an initializer."),
  An_index_signature_must_have_a_type_annotation: diag(1021, 1 /* Error */, "An_index_signature_must_have_a_type_annotation_1021", "An index signature must have a type annotation."),
  An_index_signature_parameter_must_have_a_type_annotation: diag(1022, 1 /* Error */, "An_index_signature_parameter_must_have_a_type_annotation_1022", "An index signature parameter must have a type annotation."),
  readonly_modifier_can_only_appear_on_a_property_declaration_or_index_signature: diag(1024, 1 /* Error */, "readonly_modifier_can_only_appear_on_a_property_declaration_or_index_signature_1024", "'readonly' modifier can only appear on a property declaration or index signature."),
  An_index_signature_cannot_have_a_trailing_comma: diag(1025, 1 /* Error */, "An_index_signature_cannot_have_a_trailing_comma_1025", "An index signature cannot have a trailing comma."),
  Accessibility_modifier_already_seen: diag(1028, 1 /* Error */, "Accessibility_modifier_already_seen_1028", "Accessibility modifier already seen."),
  _0_modifier_must_precede_1_modifier: diag(1029, 1 /* Error */, "_0_modifier_must_precede_1_modifier_1029", "'{0}' modifier must precede '{1}' modifier."),
  _0_modifier_already_seen: diag(1030, 1 /* Error */, "_0_modifier_already_seen_1030", "'{0}' modifier already seen."),
  _0_modifier_cannot_appear_on_class_elements_of_this_kind: diag(1031, 1 /* Error */, "_0_modifier_cannot_appear_on_class_elements_of_this_kind_1031", "'{0}' modifier cannot appear on class elements of this kind."),
  super_must_be_followed_by_an_argument_list_or_member_access: diag(1034, 1 /* Error */, "super_must_be_followed_by_an_argument_list_or_member_access_1034", "'super' must be followed by an argument list or member access."),
  Only_ambient_modules_can_use_quoted_names: diag(1035, 1 /* Error */, "Only_ambient_modules_can_use_quoted_names_1035", "Only ambient modules can use quoted names."),
  Statements_are_not_allowed_in_ambient_contexts: diag(1036, 1 /* Error */, "Statements_are_not_allowed_in_ambient_contexts_1036", "Statements are not allowed in ambient contexts."),
  A_declare_modifier_cannot_be_used_in_an_already_ambient_context: diag(1038, 1 /* Error */, "A_declare_modifier_cannot_be_used_in_an_already_ambient_context_1038", "A 'declare' modifier cannot be used in an already ambient context."),
  Initializers_are_not_allowed_in_ambient_contexts: diag(1039, 1 /* Error */, "Initializers_are_not_allowed_in_ambient_contexts_1039", "Initializers are not allowed in ambient contexts."),
  _0_modifier_cannot_be_used_in_an_ambient_context: diag(1040, 1 /* Error */, "_0_modifier_cannot_be_used_in_an_ambient_context_1040", "'{0}' modifier cannot be used in an ambient context."),
  _0_modifier_cannot_be_used_here: diag(1042, 1 /* Error */, "_0_modifier_cannot_be_used_here_1042", "'{0}' modifier cannot be used here."),
  _0_modifier_cannot_appear_on_a_module_or_namespace_element: diag(1044, 1 /* Error */, "_0_modifier_cannot_appear_on_a_module_or_namespace_element_1044", "'{0}' modifier cannot appear on a module or namespace element."),
  Top_level_declarations_in_d_ts_files_must_start_with_either_a_declare_or_export_modifier: diag(1046, 1 /* Error */, "Top_level_declarations_in_d_ts_files_must_start_with_either_a_declare_or_export_modifier_1046", "Top-level declarations in .d.ts files must start with either a 'declare' or 'export' modifier."),
  A_rest_parameter_cannot_be_optional: diag(1047, 1 /* Error */, "A_rest_parameter_cannot_be_optional_1047", "A rest parameter cannot be optional."),
  A_rest_parameter_cannot_have_an_initializer: diag(1048, 1 /* Error */, "A_rest_parameter_cannot_have_an_initializer_1048", "A rest parameter cannot have an initializer."),
  A_set_accessor_must_have_exactly_one_parameter: diag(1049, 1 /* Error */, "A_set_accessor_must_have_exactly_one_parameter_1049", "A 'set' accessor must have exactly one parameter."),
  A_set_accessor_cannot_have_an_optional_parameter: diag(1051, 1 /* Error */, "A_set_accessor_cannot_have_an_optional_parameter_1051", "A 'set' accessor cannot have an optional parameter."),
  A_set_accessor_parameter_cannot_have_an_initializer: diag(1052, 1 /* Error */, "A_set_accessor_parameter_cannot_have_an_initializer_1052", "A 'set' accessor parameter cannot have an initializer."),
  A_set_accessor_cannot_have_rest_parameter: diag(1053, 1 /* Error */, "A_set_accessor_cannot_have_rest_parameter_1053", "A 'set' accessor cannot have rest parameter."),
  A_get_accessor_cannot_have_parameters: diag(1054, 1 /* Error */, "A_get_accessor_cannot_have_parameters_1054", "A 'get' accessor cannot have parameters."),
  Type_0_is_not_a_valid_async_function_return_type_in_ES5_because_it_does_not_refer_to_a_Promise_compatible_constructor_value: diag(1055, 1 /* Error */, "Type_0_is_not_a_valid_async_function_return_type_in_ES5_because_it_does_not_refer_to_a_Promise_compa_1055", "Type '{0}' is not a valid async function return type in ES5 because it does not refer to a Promise-compatible constructor value."),
  Accessors_are_only_available_when_targeting_ECMAScript_5_and_higher: diag(1056, 1 /* Error */, "Accessors_are_only_available_when_targeting_ECMAScript_5_and_higher_1056", "Accessors are only available when targeting ECMAScript 5 and higher."),
  The_return_type_of_an_async_function_must_either_be_a_valid_promise_or_must_not_contain_a_callable_then_member: diag(1058, 1 /* Error */, "The_return_type_of_an_async_function_must_either_be_a_valid_promise_or_must_not_contain_a_callable_t_1058", "The return type of an async function must either be a valid promise or must not contain a callable 'then' member."),
  A_promise_must_have_a_then_method: diag(1059, 1 /* Error */, "A_promise_must_have_a_then_method_1059", "A promise must have a 'then' method."),
  The_first_parameter_of_the_then_method_of_a_promise_must_be_a_callback: diag(1060, 1 /* Error */, "The_first_parameter_of_the_then_method_of_a_promise_must_be_a_callback_1060", "The first parameter of the 'then' method of a promise must be a callback."),
  Enum_member_must_have_initializer: diag(1061, 1 /* Error */, "Enum_member_must_have_initializer_1061", "Enum member must have initializer."),
  Type_is_referenced_directly_or_indirectly_in_the_fulfillment_callback_of_its_own_then_method: diag(1062, 1 /* Error */, "Type_is_referenced_directly_or_indirectly_in_the_fulfillment_callback_of_its_own_then_method_1062", "Type is referenced directly or indirectly in the fulfillment callback of its own 'then' method."),
  An_export_assignment_cannot_be_used_in_a_namespace: diag(1063, 1 /* Error */, "An_export_assignment_cannot_be_used_in_a_namespace_1063", "An export assignment cannot be used in a namespace."),
  The_return_type_of_an_async_function_or_method_must_be_the_global_Promise_T_type_Did_you_mean_to_write_Promise_0: diag(1064, 1 /* Error */, "The_return_type_of_an_async_function_or_method_must_be_the_global_Promise_T_type_Did_you_mean_to_wri_1064", "The return type of an async function or method must be the global Promise<T> type. Did you mean to write 'Promise<{0}>'?"),
  The_return_type_of_an_async_function_or_method_must_be_the_global_Promise_T_type: diag(1065, 1 /* Error */, "The_return_type_of_an_async_function_or_method_must_be_the_global_Promise_T_type_1065", "The return type of an async function or method must be the global Promise<T> type."),
  In_ambient_enum_declarations_member_initializer_must_be_constant_expression: diag(1066, 1 /* Error */, "In_ambient_enum_declarations_member_initializer_must_be_constant_expression_1066", "In ambient enum declarations member initializer must be constant expression."),
  Unexpected_token_A_constructor_method_accessor_or_property_was_expected: diag(1068, 1 /* Error */, "Unexpected_token_A_constructor_method_accessor_or_property_was_expected_1068", "Unexpected token. A constructor, method, accessor, or property was expected."),
  Unexpected_token_A_type_parameter_name_was_expected_without_curly_braces: diag(1069, 1 /* Error */, "Unexpected_token_A_type_parameter_name_was_expected_without_curly_braces_1069", "Unexpected token. A type parameter name was expected without curly braces."),
  _0_modifier_cannot_appear_on_a_type_member: diag(1070, 1 /* Error */, "_0_modifier_cannot_appear_on_a_type_member_1070", "'{0}' modifier cannot appear on a type member."),
  _0_modifier_cannot_appear_on_an_index_signature: diag(1071, 1 /* Error */, "_0_modifier_cannot_appear_on_an_index_signature_1071", "'{0}' modifier cannot appear on an index signature."),
  A_0_modifier_cannot_be_used_with_an_import_declaration: diag(1079, 1 /* Error */, "A_0_modifier_cannot_be_used_with_an_import_declaration_1079", "A '{0}' modifier cannot be used with an import declaration."),
  Invalid_reference_directive_syntax: diag(1084, 1 /* Error */, "Invalid_reference_directive_syntax_1084", "Invalid 'reference' directive syntax."),
  _0_modifier_cannot_appear_on_a_constructor_declaration: diag(1089, 1 /* Error */, "_0_modifier_cannot_appear_on_a_constructor_declaration_1089", "'{0}' modifier cannot appear on a constructor declaration."),
  _0_modifier_cannot_appear_on_a_parameter: diag(1090, 1 /* Error */, "_0_modifier_cannot_appear_on_a_parameter_1090", "'{0}' modifier cannot appear on a parameter."),
  Only_a_single_variable_declaration_is_allowed_in_a_for_in_statement: diag(1091, 1 /* Error */, "Only_a_single_variable_declaration_is_allowed_in_a_for_in_statement_1091", "Only a single variable declaration is allowed in a 'for...in' statement."),
  Type_parameters_cannot_appear_on_a_constructor_declaration: diag(1092, 1 /* Error */, "Type_parameters_cannot_appear_on_a_constructor_declaration_1092", "Type parameters cannot appear on a constructor declaration."),
  Type_annotation_cannot_appear_on_a_constructor_declaration: diag(1093, 1 /* Error */, "Type_annotation_cannot_appear_on_a_constructor_declaration_1093", "Type annotation cannot appear on a constructor declaration."),
  An_accessor_cannot_have_type_parameters: diag(1094, 1 /* Error */, "An_accessor_cannot_have_type_parameters_1094", "An accessor cannot have type parameters."),
  A_set_accessor_cannot_have_a_return_type_annotation: diag(1095, 1 /* Error */, "A_set_accessor_cannot_have_a_return_type_annotation_1095", "A 'set' accessor cannot have a return type annotation."),
  An_index_signature_must_have_exactly_one_parameter: diag(1096, 1 /* Error */, "An_index_signature_must_have_exactly_one_parameter_1096", "An index signature must have exactly one parameter."),
  _0_list_cannot_be_empty: diag(1097, 1 /* Error */, "_0_list_cannot_be_empty_1097", "'{0}' list cannot be empty."),
  Type_parameter_list_cannot_be_empty: diag(1098, 1 /* Error */, "Type_parameter_list_cannot_be_empty_1098", "Type parameter list cannot be empty."),
  Type_argument_list_cannot_be_empty: diag(1099, 1 /* Error */, "Type_argument_list_cannot_be_empty_1099", "Type argument list cannot be empty."),
  Invalid_use_of_0_in_strict_mode: diag(1100, 1 /* Error */, "Invalid_use_of_0_in_strict_mode_1100", "Invalid use of '{0}' in strict mode."),
  with_statements_are_not_allowed_in_strict_mode: diag(1101, 1 /* Error */, "with_statements_are_not_allowed_in_strict_mode_1101", "'with' statements are not allowed in strict mode."),
  delete_cannot_be_called_on_an_identifier_in_strict_mode: diag(1102, 1 /* Error */, "delete_cannot_be_called_on_an_identifier_in_strict_mode_1102", "'delete' cannot be called on an identifier in strict mode."),
  for_await_loops_are_only_allowed_within_async_functions_and_at_the_top_levels_of_modules: diag(1103, 1 /* Error */, "for_await_loops_are_only_allowed_within_async_functions_and_at_the_top_levels_of_modules_1103", "'for await' loops are only allowed within async functions and at the top levels of modules."),
  A_continue_statement_can_only_be_used_within_an_enclosing_iteration_statement: diag(1104, 1 /* Error */, "A_continue_statement_can_only_be_used_within_an_enclosing_iteration_statement_1104", "A 'continue' statement can only be used within an enclosing iteration statement."),
  A_break_statement_can_only_be_used_within_an_enclosing_iteration_or_switch_statement: diag(1105, 1 /* Error */, "A_break_statement_can_only_be_used_within_an_enclosing_iteration_or_switch_statement_1105", "A 'break' statement can only be used within an enclosing iteration or switch statement."),
  The_left_hand_side_of_a_for_of_statement_may_not_be_async: diag(1106, 1 /* Error */, "The_left_hand_side_of_a_for_of_statement_may_not_be_async_1106", "The left-hand side of a 'for...of' statement may not be 'async'."),
  Jump_target_cannot_cross_function_boundary: diag(1107, 1 /* Error */, "Jump_target_cannot_cross_function_boundary_1107", "Jump target cannot cross function boundary."),
  A_return_statement_can_only_be_used_within_a_function_body: diag(1108, 1 /* Error */, "A_return_statement_can_only_be_used_within_a_function_body_1108", "A 'return' statement can only be used within a function body."),
  Expression_expected: diag(1109, 1 /* Error */, "Expression_expected_1109", "Expression expected."),
  Type_expected: diag(1110, 1 /* Error */, "Type_expected_1110", "Type expected."),
  Private_field_0_must_be_declared_in_an_enclosing_class: diag(1111, 1 /* Error */, "Private_field_0_must_be_declared_in_an_enclosing_class_1111", "Private field '{0}' must be declared in an enclosing class."),
  A_default_clause_cannot_appear_more_than_once_in_a_switch_statement: diag(1113, 1 /* Error */, "A_default_clause_cannot_appear_more_than_once_in_a_switch_statement_1113", "A 'default' clause cannot appear more than once in a 'switch' statement."),
  Duplicate_label_0: diag(1114, 1 /* Error */, "Duplicate_label_0_1114", "Duplicate label '{0}'."),
  A_continue_statement_can_only_jump_to_a_label_of_an_enclosing_iteration_statement: diag(1115, 1 /* Error */, "A_continue_statement_can_only_jump_to_a_label_of_an_enclosing_iteration_statement_1115", "A 'continue' statement can only jump to a label of an enclosing iteration statement."),
  A_break_statement_can_only_jump_to_a_label_of_an_enclosing_statement: diag(1116, 1 /* Error */, "A_break_statement_can_only_jump_to_a_label_of_an_enclosing_statement_1116", "A 'break' statement can only jump to a label of an enclosing statement."),
  An_object_literal_cannot_have_multiple_properties_with_the_same_name: diag(1117, 1 /* Error */, "An_object_literal_cannot_have_multiple_properties_with_the_same_name_1117", "An object literal cannot have multiple properties with the same name."),
  An_object_literal_cannot_have_multiple_get_Slashset_accessors_with_the_same_name: diag(1118, 1 /* Error */, "An_object_literal_cannot_have_multiple_get_Slashset_accessors_with_the_same_name_1118", "An object literal cannot have multiple get/set accessors with the same name."),
  An_object_literal_cannot_have_property_and_accessor_with_the_same_name: diag(1119, 1 /* Error */, "An_object_literal_cannot_have_property_and_accessor_with_the_same_name_1119", "An object literal cannot have property and accessor with the same name."),
  An_export_assignment_cannot_have_modifiers: diag(1120, 1 /* Error */, "An_export_assignment_cannot_have_modifiers_1120", "An export assignment cannot have modifiers."),
  Octal_literals_are_not_allowed_Use_the_syntax_0: diag(1121, 1 /* Error */, "Octal_literals_are_not_allowed_Use_the_syntax_0_1121", "Octal literals are not allowed. Use the syntax '{0}'."),
  Variable_declaration_list_cannot_be_empty: diag(1123, 1 /* Error */, "Variable_declaration_list_cannot_be_empty_1123", "Variable declaration list cannot be empty."),
  Digit_expected: diag(1124, 1 /* Error */, "Digit_expected_1124", "Digit expected."),
  Hexadecimal_digit_expected: diag(1125, 1 /* Error */, "Hexadecimal_digit_expected_1125", "Hexadecimal digit expected."),
  Unexpected_end_of_text: diag(1126, 1 /* Error */, "Unexpected_end_of_text_1126", "Unexpected end of text."),
  Invalid_character: diag(1127, 1 /* Error */, "Invalid_character_1127", "Invalid character."),
  Declaration_or_statement_expected: diag(1128, 1 /* Error */, "Declaration_or_statement_expected_1128", "Declaration or statement expected."),
  Statement_expected: diag(1129, 1 /* Error */, "Statement_expected_1129", "Statement expected."),
  case_or_default_expected: diag(1130, 1 /* Error */, "case_or_default_expected_1130", "'case' or 'default' expected."),
  Property_or_signature_expected: diag(1131, 1 /* Error */, "Property_or_signature_expected_1131", "Property or signature expected."),
  Enum_member_expected: diag(1132, 1 /* Error */, "Enum_member_expected_1132", "Enum member expected."),
  Variable_declaration_expected: diag(1134, 1 /* Error */, "Variable_declaration_expected_1134", "Variable declaration expected."),
  Argument_expression_expected: diag(1135, 1 /* Error */, "Argument_expression_expected_1135", "Argument expression expected."),
  Property_assignment_expected: diag(1136, 1 /* Error */, "Property_assignment_expected_1136", "Property assignment expected."),
  Expression_or_comma_expected: diag(1137, 1 /* Error */, "Expression_or_comma_expected_1137", "Expression or comma expected."),
  Parameter_declaration_expected: diag(1138, 1 /* Error */, "Parameter_declaration_expected_1138", "Parameter declaration expected."),
  Type_parameter_declaration_expected: diag(1139, 1 /* Error */, "Type_parameter_declaration_expected_1139", "Type parameter declaration expected."),
  Type_argument_expected: diag(1140, 1 /* Error */, "Type_argument_expected_1140", "Type argument expected."),
  String_literal_expected: diag(1141, 1 /* Error */, "String_literal_expected_1141", "String literal expected."),
  Line_break_not_permitted_here: diag(1142, 1 /* Error */, "Line_break_not_permitted_here_1142", "Line break not permitted here."),
  or_expected: diag(1144, 1 /* Error */, "or_expected_1144", "'{' or ';' expected."),
  or_JSX_element_expected: diag(1145, 1 /* Error */, "or_JSX_element_expected_1145", "'{' or JSX element expected."),
  Declaration_expected: diag(1146, 1 /* Error */, "Declaration_expected_1146", "Declaration expected."),
  Import_declarations_in_a_namespace_cannot_reference_a_module: diag(1147, 1 /* Error */, "Import_declarations_in_a_namespace_cannot_reference_a_module_1147", "Import declarations in a namespace cannot reference a module."),
  Cannot_use_imports_exports_or_module_augmentations_when_module_is_none: diag(1148, 1 /* Error */, "Cannot_use_imports_exports_or_module_augmentations_when_module_is_none_1148", "Cannot use imports, exports, or module augmentations when '--module' is 'none'."),
  File_name_0_differs_from_already_included_file_name_1_only_in_casing: diag(1149, 1 /* Error */, "File_name_0_differs_from_already_included_file_name_1_only_in_casing_1149", "File name '{0}' differs from already included file name '{1}' only in casing."),
  _0_declarations_must_be_initialized: diag(1155, 1 /* Error */, "_0_declarations_must_be_initialized_1155", "'{0}' declarations must be initialized."),
  _0_declarations_can_only_be_declared_inside_a_block: diag(1156, 1 /* Error */, "_0_declarations_can_only_be_declared_inside_a_block_1156", "'{0}' declarations can only be declared inside a block."),
  Unterminated_template_literal: diag(1160, 1 /* Error */, "Unterminated_template_literal_1160", "Unterminated template literal."),
  Unterminated_regular_expression_literal: diag(1161, 1 /* Error */, "Unterminated_regular_expression_literal_1161", "Unterminated regular expression literal."),
  An_object_member_cannot_be_declared_optional: diag(1162, 1 /* Error */, "An_object_member_cannot_be_declared_optional_1162", "An object member cannot be declared optional."),
  A_yield_expression_is_only_allowed_in_a_generator_body: diag(1163, 1 /* Error */, "A_yield_expression_is_only_allowed_in_a_generator_body_1163", "A 'yield' expression is only allowed in a generator body."),
  Computed_property_names_are_not_allowed_in_enums: diag(1164, 1 /* Error */, "Computed_property_names_are_not_allowed_in_enums_1164", "Computed property names are not allowed in enums."),
  A_computed_property_name_in_an_ambient_context_must_refer_to_an_expression_whose_type_is_a_literal_type_or_a_unique_symbol_type: diag(1165, 1 /* Error */, "A_computed_property_name_in_an_ambient_context_must_refer_to_an_expression_whose_type_is_a_literal_t_1165", "A computed property name in an ambient context must refer to an expression whose type is a literal type or a 'unique symbol' type."),
  A_computed_property_name_in_a_class_property_declaration_must_have_a_simple_literal_type_or_a_unique_symbol_type: diag(1166, 1 /* Error */, "A_computed_property_name_in_a_class_property_declaration_must_have_a_simple_literal_type_or_a_unique_1166", "A computed property name in a class property declaration must have a simple literal type or a 'unique symbol' type."),
  A_computed_property_name_in_a_method_overload_must_refer_to_an_expression_whose_type_is_a_literal_type_or_a_unique_symbol_type: diag(1168, 1 /* Error */, "A_computed_property_name_in_a_method_overload_must_refer_to_an_expression_whose_type_is_a_literal_ty_1168", "A computed property name in a method overload must refer to an expression whose type is a literal type or a 'unique symbol' type."),
  A_computed_property_name_in_an_interface_must_refer_to_an_expression_whose_type_is_a_literal_type_or_a_unique_symbol_type: diag(1169, 1 /* Error */, "A_computed_property_name_in_an_interface_must_refer_to_an_expression_whose_type_is_a_literal_type_or_1169", "A computed property name in an interface must refer to an expression whose type is a literal type or a 'unique symbol' type."),
  A_computed_property_name_in_a_type_literal_must_refer_to_an_expression_whose_type_is_a_literal_type_or_a_unique_symbol_type: diag(1170, 1 /* Error */, "A_computed_property_name_in_a_type_literal_must_refer_to_an_expression_whose_type_is_a_literal_type__1170", "A computed property name in a type literal must refer to an expression whose type is a literal type or a 'unique symbol' type."),
  A_comma_expression_is_not_allowed_in_a_computed_property_name: diag(1171, 1 /* Error */, "A_comma_expression_is_not_allowed_in_a_computed_property_name_1171", "A comma expression is not allowed in a computed property name."),
  extends_clause_already_seen: diag(1172, 1 /* Error */, "extends_clause_already_seen_1172", "'extends' clause already seen."),
  extends_clause_must_precede_implements_clause: diag(1173, 1 /* Error */, "extends_clause_must_precede_implements_clause_1173", "'extends' clause must precede 'implements' clause."),
  Classes_can_only_extend_a_single_class: diag(1174, 1 /* Error */, "Classes_can_only_extend_a_single_class_1174", "Classes can only extend a single class."),
  implements_clause_already_seen: diag(1175, 1 /* Error */, "implements_clause_already_seen_1175", "'implements' clause already seen."),
  Interface_declaration_cannot_have_implements_clause: diag(1176, 1 /* Error */, "Interface_declaration_cannot_have_implements_clause_1176", "Interface declaration cannot have 'implements' clause."),
  Binary_digit_expected: diag(1177, 1 /* Error */, "Binary_digit_expected_1177", "Binary digit expected."),
  Octal_digit_expected: diag(1178, 1 /* Error */, "Octal_digit_expected_1178", "Octal digit expected."),
  Unexpected_token_expected: diag(1179, 1 /* Error */, "Unexpected_token_expected_1179", "Unexpected token. '{' expected."),
  Property_destructuring_pattern_expected: diag(1180, 1 /* Error */, "Property_destructuring_pattern_expected_1180", "Property destructuring pattern expected."),
  Array_element_destructuring_pattern_expected: diag(1181, 1 /* Error */, "Array_element_destructuring_pattern_expected_1181", "Array element destructuring pattern expected."),
  A_destructuring_declaration_must_have_an_initializer: diag(1182, 1 /* Error */, "A_destructuring_declaration_must_have_an_initializer_1182", "A destructuring declaration must have an initializer."),
  An_implementation_cannot_be_declared_in_ambient_contexts: diag(1183, 1 /* Error */, "An_implementation_cannot_be_declared_in_ambient_contexts_1183", "An implementation cannot be declared in ambient contexts."),
  Modifiers_cannot_appear_here: diag(1184, 1 /* Error */, "Modifiers_cannot_appear_here_1184", "Modifiers cannot appear here."),
  Merge_conflict_marker_encountered: diag(1185, 1 /* Error */, "Merge_conflict_marker_encountered_1185", "Merge conflict marker encountered."),
  A_rest_element_cannot_have_an_initializer: diag(1186, 1 /* Error */, "A_rest_element_cannot_have_an_initializer_1186", "A rest element cannot have an initializer."),
  A_parameter_property_may_not_be_declared_using_a_binding_pattern: diag(1187, 1 /* Error */, "A_parameter_property_may_not_be_declared_using_a_binding_pattern_1187", "A parameter property may not be declared using a binding pattern."),
  Only_a_single_variable_declaration_is_allowed_in_a_for_of_statement: diag(1188, 1 /* Error */, "Only_a_single_variable_declaration_is_allowed_in_a_for_of_statement_1188", "Only a single variable declaration is allowed in a 'for...of' statement."),
  The_variable_declaration_of_a_for_in_statement_cannot_have_an_initializer: diag(1189, 1 /* Error */, "The_variable_declaration_of_a_for_in_statement_cannot_have_an_initializer_1189", "The variable declaration of a 'for...in' statement cannot have an initializer."),
  The_variable_declaration_of_a_for_of_statement_cannot_have_an_initializer: diag(1190, 1 /* Error */, "The_variable_declaration_of_a_for_of_statement_cannot_have_an_initializer_1190", "The variable declaration of a 'for...of' statement cannot have an initializer."),
  An_import_declaration_cannot_have_modifiers: diag(1191, 1 /* Error */, "An_import_declaration_cannot_have_modifiers_1191", "An import declaration cannot have modifiers."),
  Module_0_has_no_default_export: diag(1192, 1 /* Error */, "Module_0_has_no_default_export_1192", "Module '{0}' has no default export."),
  An_export_declaration_cannot_have_modifiers: diag(1193, 1 /* Error */, "An_export_declaration_cannot_have_modifiers_1193", "An export declaration cannot have modifiers."),
  Export_declarations_are_not_permitted_in_a_namespace: diag(1194, 1 /* Error */, "Export_declarations_are_not_permitted_in_a_namespace_1194", "Export declarations are not permitted in a namespace."),
  export_Asterisk_does_not_re_export_a_default: diag(1195, 1 /* Error */, "export_Asterisk_does_not_re_export_a_default_1195", "'export *' does not re-export a default."),
  Catch_clause_variable_type_annotation_must_be_any_or_unknown_if_specified: diag(1196, 1 /* Error */, "Catch_clause_variable_type_annotation_must_be_any_or_unknown_if_specified_1196", "Catch clause variable type annotation must be 'any' or 'unknown' if specified."),
  Catch_clause_variable_cannot_have_an_initializer: diag(1197, 1 /* Error */, "Catch_clause_variable_cannot_have_an_initializer_1197", "Catch clause variable cannot have an initializer."),
  An_extended_Unicode_escape_value_must_be_between_0x0_and_0x10FFFF_inclusive: diag(1198, 1 /* Error */, "An_extended_Unicode_escape_value_must_be_between_0x0_and_0x10FFFF_inclusive_1198", "An extended Unicode escape value must be between 0x0 and 0x10FFFF inclusive."),
  Unterminated_Unicode_escape_sequence: diag(1199, 1 /* Error */, "Unterminated_Unicode_escape_sequence_1199", "Unterminated Unicode escape sequence."),
  Line_terminator_not_permitted_before_arrow: diag(1200, 1 /* Error */, "Line_terminator_not_permitted_before_arrow_1200", "Line terminator not permitted before arrow."),
  Import_assignment_cannot_be_used_when_targeting_ECMAScript_modules_Consider_using_import_Asterisk_as_ns_from_mod_import_a_from_mod_import_d_from_mod_or_another_module_format_instead: diag(1202, 1 /* Error */, "Import_assignment_cannot_be_used_when_targeting_ECMAScript_modules_Consider_using_import_Asterisk_as_1202", `Import assignment cannot be used when targeting ECMAScript modules. Consider using 'import * as ns from "mod"', 'import {a} from "mod"', 'import d from "mod"', or another module format instead.`),
  Export_assignment_cannot_be_used_when_targeting_ECMAScript_modules_Consider_using_export_default_or_another_module_format_instead: diag(1203, 1 /* Error */, "Export_assignment_cannot_be_used_when_targeting_ECMAScript_modules_Consider_using_export_default_or__1203", "Export assignment cannot be used when targeting ECMAScript modules. Consider using 'export default' or another module format instead."),
  Re_exporting_a_type_when_0_is_enabled_requires_using_export_type: diag(1205, 1 /* Error */, "Re_exporting_a_type_when_0_is_enabled_requires_using_export_type_1205", "Re-exporting a type when '{0}' is enabled requires using 'export type'."),
  Decorators_are_not_valid_here: diag(1206, 1 /* Error */, "Decorators_are_not_valid_here_1206", "Decorators are not valid here."),
  Decorators_cannot_be_applied_to_multiple_get_Slashset_accessors_of_the_same_name: diag(1207, 1 /* Error */, "Decorators_cannot_be_applied_to_multiple_get_Slashset_accessors_of_the_same_name_1207", "Decorators cannot be applied to multiple get/set accessors of the same name."),
  Invalid_optional_chain_from_new_expression_Did_you_mean_to_call_0: diag(1209, 1 /* Error */, "Invalid_optional_chain_from_new_expression_Did_you_mean_to_call_0_1209", "Invalid optional chain from new expression. Did you mean to call '{0}()'?"),
  Code_contained_in_a_class_is_evaluated_in_JavaScript_s_strict_mode_which_does_not_allow_this_use_of_0_For_more_information_see_https_Colon_Slash_Slashdeveloper_mozilla_org_Slashen_US_Slashdocs_SlashWeb_SlashJavaScript_SlashReference_SlashStrict_mode: diag(1210, 1 /* Error */, "Code_contained_in_a_class_is_evaluated_in_JavaScript_s_strict_mode_which_does_not_allow_this_use_of__1210", "Code contained in a class is evaluated in JavaScript's strict mode which does not allow this use of '{0}'. For more information, see https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Strict_mode."),
  A_class_declaration_without_the_default_modifier_must_have_a_name: diag(1211, 1 /* Error */, "A_class_declaration_without_the_default_modifier_must_have_a_name_1211", "A class declaration without the 'default' modifier must have a name."),
  Identifier_expected_0_is_a_reserved_word_in_strict_mode: diag(1212, 1 /* Error */, "Identifier_expected_0_is_a_reserved_word_in_strict_mode_1212", "Identifier expected. '{0}' is a reserved word in strict mode."),
  Identifier_expected_0_is_a_reserved_word_in_strict_mode_Class_definitions_are_automatically_in_strict_mode: diag(1213, 1 /* Error */, "Identifier_expected_0_is_a_reserved_word_in_strict_mode_Class_definitions_are_automatically_in_stric_1213", "Identifier expected. '{0}' is a reserved word in strict mode. Class definitions are automatically in strict mode."),
  Identifier_expected_0_is_a_reserved_word_in_strict_mode_Modules_are_automatically_in_strict_mode: diag(1214, 1 /* Error */, "Identifier_expected_0_is_a_reserved_word_in_strict_mode_Modules_are_automatically_in_strict_mode_1214", "Identifier expected. '{0}' is a reserved word in strict mode. Modules are automatically in strict mode."),
  Invalid_use_of_0_Modules_are_automatically_in_strict_mode: diag(1215, 1 /* Error */, "Invalid_use_of_0_Modules_are_automatically_in_strict_mode_1215", "Invalid use of '{0}'. Modules are automatically in strict mode."),
  Identifier_expected_esModule_is_reserved_as_an_exported_marker_when_transforming_ECMAScript_modules: diag(1216, 1 /* Error */, "Identifier_expected_esModule_is_reserved_as_an_exported_marker_when_transforming_ECMAScript_modules_1216", "Identifier expected. '__esModule' is reserved as an exported marker when transforming ECMAScript modules."),
  Export_assignment_is_not_supported_when_module_flag_is_system: diag(1218, 1 /* Error */, "Export_assignment_is_not_supported_when_module_flag_is_system_1218", "Export assignment is not supported when '--module' flag is 'system'."),
  Generators_are_not_allowed_in_an_ambient_context: diag(1221, 1 /* Error */, "Generators_are_not_allowed_in_an_ambient_context_1221", "Generators are not allowed in an ambient context."),
  An_overload_signature_cannot_be_declared_as_a_generator: diag(1222, 1 /* Error */, "An_overload_signature_cannot_be_declared_as_a_generator_1222", "An overload signature cannot be declared as a generator."),
  _0_tag_already_specified: diag(1223, 1 /* Error */, "_0_tag_already_specified_1223", "'{0}' tag already specified."),
  Signature_0_must_be_a_type_predicate: diag(1224, 1 /* Error */, "Signature_0_must_be_a_type_predicate_1224", "Signature '{0}' must be a type predicate."),
  Cannot_find_parameter_0: diag(1225, 1 /* Error */, "Cannot_find_parameter_0_1225", "Cannot find parameter '{0}'."),
  Type_predicate_0_is_not_assignable_to_1: diag(1226, 1 /* Error */, "Type_predicate_0_is_not_assignable_to_1_1226", "Type predicate '{0}' is not assignable to '{1}'."),
  Parameter_0_is_not_in_the_same_position_as_parameter_1: diag(1227, 1 /* Error */, "Parameter_0_is_not_in_the_same_position_as_parameter_1_1227", "Parameter '{0}' is not in the same position as parameter '{1}'."),
  A_type_predicate_is_only_allowed_in_return_type_position_for_functions_and_methods: diag(1228, 1 /* Error */, "A_type_predicate_is_only_allowed_in_return_type_position_for_functions_and_methods_1228", "A type predicate is only allowed in return type position for functions and methods."),
  A_type_predicate_cannot_reference_a_rest_parameter: diag(1229, 1 /* Error */, "A_type_predicate_cannot_reference_a_rest_parameter_1229", "A type predicate cannot reference a rest parameter."),
  A_type_predicate_cannot_reference_element_0_in_a_binding_pattern: diag(1230, 1 /* Error */, "A_type_predicate_cannot_reference_element_0_in_a_binding_pattern_1230", "A type predicate cannot reference element '{0}' in a binding pattern."),
  An_export_assignment_must_be_at_the_top_level_of_a_file_or_module_declaration: diag(1231, 1 /* Error */, "An_export_assignment_must_be_at_the_top_level_of_a_file_or_module_declaration_1231", "An export assignment must be at the top level of a file or module declaration."),
  An_import_declaration_can_only_be_used_at_the_top_level_of_a_namespace_or_module: diag(1232, 1 /* Error */, "An_import_declaration_can_only_be_used_at_the_top_level_of_a_namespace_or_module_1232", "An import declaration can only be used at the top level of a namespace or module."),
  An_export_declaration_can_only_be_used_at_the_top_level_of_a_namespace_or_module: diag(1233, 1 /* Error */, "An_export_declaration_can_only_be_used_at_the_top_level_of_a_namespace_or_module_1233", "An export declaration can only be used at the top level of a namespace or module."),
  An_ambient_module_declaration_is_only_allowed_at_the_top_level_in_a_file: diag(1234, 1 /* Error */, "An_ambient_module_declaration_is_only_allowed_at_the_top_level_in_a_file_1234", "An ambient module declaration is only allowed at the top level in a file."),
  A_namespace_declaration_is_only_allowed_at_the_top_level_of_a_namespace_or_module: diag(1235, 1 /* Error */, "A_namespace_declaration_is_only_allowed_at_the_top_level_of_a_namespace_or_module_1235", "A namespace declaration is only allowed at the top level of a namespace or module."),
  The_return_type_of_a_property_decorator_function_must_be_either_void_or_any: diag(1236, 1 /* Error */, "The_return_type_of_a_property_decorator_function_must_be_either_void_or_any_1236", "The return type of a property decorator function must be either 'void' or 'any'."),
  The_return_type_of_a_parameter_decorator_function_must_be_either_void_or_any: diag(1237, 1 /* Error */, "The_return_type_of_a_parameter_decorator_function_must_be_either_void_or_any_1237", "The return type of a parameter decorator function must be either 'void' or 'any'."),
  Unable_to_resolve_signature_of_class_decorator_when_called_as_an_expression: diag(1238, 1 /* Error */, "Unable_to_resolve_signature_of_class_decorator_when_called_as_an_expression_1238", "Unable to resolve signature of class decorator when called as an expression."),
  Unable_to_resolve_signature_of_parameter_decorator_when_called_as_an_expression: diag(1239, 1 /* Error */, "Unable_to_resolve_signature_of_parameter_decorator_when_called_as_an_expression_1239", "Unable to resolve signature of parameter decorator when called as an expression."),
  Unable_to_resolve_signature_of_property_decorator_when_called_as_an_expression: diag(1240, 1 /* Error */, "Unable_to_resolve_signature_of_property_decorator_when_called_as_an_expression_1240", "Unable to resolve signature of property decorator when called as an expression."),
  Unable_to_resolve_signature_of_method_decorator_when_called_as_an_expression: diag(1241, 1 /* Error */, "Unable_to_resolve_signature_of_method_decorator_when_called_as_an_expression_1241", "Unable to resolve signature of method decorator when called as an expression."),
  abstract_modifier_can_only_appear_on_a_class_method_or_property_declaration: diag(1242, 1 /* Error */, "abstract_modifier_can_only_appear_on_a_class_method_or_property_declaration_1242", "'abstract' modifier can only appear on a class, method, or property declaration."),
  _0_modifier_cannot_be_used_with_1_modifier: diag(1243, 1 /* Error */, "_0_modifier_cannot_be_used_with_1_modifier_1243", "'{0}' modifier cannot be used with '{1}' modifier."),
  Abstract_methods_can_only_appear_within_an_abstract_class: diag(1244, 1 /* Error */, "Abstract_methods_can_only_appear_within_an_abstract_class_1244", "Abstract methods can only appear within an abstract class."),
  Method_0_cannot_have_an_implementation_because_it_is_marked_abstract: diag(1245, 1 /* Error */, "Method_0_cannot_have_an_implementation_because_it_is_marked_abstract_1245", "Method '{0}' cannot have an implementation because it is marked abstract."),
  An_interface_property_cannot_have_an_initializer: diag(1246, 1 /* Error */, "An_interface_property_cannot_have_an_initializer_1246", "An interface property cannot have an initializer."),
  A_type_literal_property_cannot_have_an_initializer: diag(1247, 1 /* Error */, "A_type_literal_property_cannot_have_an_initializer_1247", "A type literal property cannot have an initializer."),
  A_class_member_cannot_have_the_0_keyword: diag(1248, 1 /* Error */, "A_class_member_cannot_have_the_0_keyword_1248", "A class member cannot have the '{0}' keyword."),
  A_decorator_can_only_decorate_a_method_implementation_not_an_overload: diag(1249, 1 /* Error */, "A_decorator_can_only_decorate_a_method_implementation_not_an_overload_1249", "A decorator can only decorate a method implementation, not an overload."),
  Function_declarations_are_not_allowed_inside_blocks_in_strict_mode_when_targeting_ES5: diag(1250, 1 /* Error */, "Function_declarations_are_not_allowed_inside_blocks_in_strict_mode_when_targeting_ES5_1250", "Function declarations are not allowed inside blocks in strict mode when targeting 'ES5'."),
  Function_declarations_are_not_allowed_inside_blocks_in_strict_mode_when_targeting_ES5_Class_definitions_are_automatically_in_strict_mode: diag(1251, 1 /* Error */, "Function_declarations_are_not_allowed_inside_blocks_in_strict_mode_when_targeting_ES5_Class_definiti_1251", "Function declarations are not allowed inside blocks in strict mode when targeting 'ES5'. Class definitions are automatically in strict mode."),
  Function_declarations_are_not_allowed_inside_blocks_in_strict_mode_when_targeting_ES5_Modules_are_automatically_in_strict_mode: diag(1252, 1 /* Error */, "Function_declarations_are_not_allowed_inside_blocks_in_strict_mode_when_targeting_ES5_Modules_are_au_1252", "Function declarations are not allowed inside blocks in strict mode when targeting 'ES5'. Modules are automatically in strict mode."),
  Abstract_properties_can_only_appear_within_an_abstract_class: diag(1253, 1 /* Error */, "Abstract_properties_can_only_appear_within_an_abstract_class_1253", "Abstract properties can only appear within an abstract class."),
  A_const_initializer_in_an_ambient_context_must_be_a_string_or_numeric_literal_or_literal_enum_reference: diag(1254, 1 /* Error */, "A_const_initializer_in_an_ambient_context_must_be_a_string_or_numeric_literal_or_literal_enum_refere_1254", "A 'const' initializer in an ambient context must be a string or numeric literal or literal enum reference."),
  A_definite_assignment_assertion_is_not_permitted_in_this_context: diag(1255, 1 /* Error */, "A_definite_assignment_assertion_is_not_permitted_in_this_context_1255", "A definite assignment assertion '!' is not permitted in this context."),
  A_required_element_cannot_follow_an_optional_element: diag(1257, 1 /* Error */, "A_required_element_cannot_follow_an_optional_element_1257", "A required element cannot follow an optional element."),
  A_default_export_must_be_at_the_top_level_of_a_file_or_module_declaration: diag(1258, 1 /* Error */, "A_default_export_must_be_at_the_top_level_of_a_file_or_module_declaration_1258", "A default export must be at the top level of a file or module declaration."),
  Module_0_can_only_be_default_imported_using_the_1_flag: diag(1259, 1 /* Error */, "Module_0_can_only_be_default_imported_using_the_1_flag_1259", "Module '{0}' can only be default-imported using the '{1}' flag"),
  Keywords_cannot_contain_escape_characters: diag(1260, 1 /* Error */, "Keywords_cannot_contain_escape_characters_1260", "Keywords cannot contain escape characters."),
  Already_included_file_name_0_differs_from_file_name_1_only_in_casing: diag(1261, 1 /* Error */, "Already_included_file_name_0_differs_from_file_name_1_only_in_casing_1261", "Already included file name '{0}' differs from file name '{1}' only in casing."),
  Identifier_expected_0_is_a_reserved_word_at_the_top_level_of_a_module: diag(1262, 1 /* Error */, "Identifier_expected_0_is_a_reserved_word_at_the_top_level_of_a_module_1262", "Identifier expected. '{0}' is a reserved word at the top-level of a module."),
  Declarations_with_initializers_cannot_also_have_definite_assignment_assertions: diag(1263, 1 /* Error */, "Declarations_with_initializers_cannot_also_have_definite_assignment_assertions_1263", "Declarations with initializers cannot also have definite assignment assertions."),
  Declarations_with_definite_assignment_assertions_must_also_have_type_annotations: diag(1264, 1 /* Error */, "Declarations_with_definite_assignment_assertions_must_also_have_type_annotations_1264", "Declarations with definite assignment assertions must also have type annotations."),
  A_rest_element_cannot_follow_another_rest_element: diag(1265, 1 /* Error */, "A_rest_element_cannot_follow_another_rest_element_1265", "A rest element cannot follow another rest element."),
  An_optional_element_cannot_follow_a_rest_element: diag(1266, 1 /* Error */, "An_optional_element_cannot_follow_a_rest_element_1266", "An optional element cannot follow a rest element."),
  Property_0_cannot_have_an_initializer_because_it_is_marked_abstract: diag(1267, 1 /* Error */, "Property_0_cannot_have_an_initializer_because_it_is_marked_abstract_1267", "Property '{0}' cannot have an initializer because it is marked abstract."),
  An_index_signature_parameter_type_must_be_string_number_symbol_or_a_template_literal_type: diag(1268, 1 /* Error */, "An_index_signature_parameter_type_must_be_string_number_symbol_or_a_template_literal_type_1268", "An index signature parameter type must be 'string', 'number', 'symbol', or a template literal type."),
  Cannot_use_export_import_on_a_type_or_type_only_namespace_when_0_is_enabled: diag(1269, 1 /* Error */, "Cannot_use_export_import_on_a_type_or_type_only_namespace_when_0_is_enabled_1269", "Cannot use 'export import' on a type or type-only namespace when '{0}' is enabled."),
  Decorator_function_return_type_0_is_not_assignable_to_type_1: diag(1270, 1 /* Error */, "Decorator_function_return_type_0_is_not_assignable_to_type_1_1270", "Decorator function return type '{0}' is not assignable to type '{1}'."),
  Decorator_function_return_type_is_0_but_is_expected_to_be_void_or_any: diag(1271, 1 /* Error */, "Decorator_function_return_type_is_0_but_is_expected_to_be_void_or_any_1271", "Decorator function return type is '{0}' but is expected to be 'void' or 'any'."),
  A_type_referenced_in_a_decorated_signature_must_be_imported_with_import_type_or_a_namespace_import_when_isolatedModules_and_emitDecoratorMetadata_are_enabled: diag(1272, 1 /* Error */, "A_type_referenced_in_a_decorated_signature_must_be_imported_with_import_type_or_a_namespace_import_w_1272", "A type referenced in a decorated signature must be imported with 'import type' or a namespace import when 'isolatedModules' and 'emitDecoratorMetadata' are enabled."),
  _0_modifier_cannot_appear_on_a_type_parameter: diag(1273, 1 /* Error */, "_0_modifier_cannot_appear_on_a_type_parameter_1273", "'{0}' modifier cannot appear on a type parameter"),
  _0_modifier_can_only_appear_on_a_type_parameter_of_a_class_interface_or_type_alias: diag(1274, 1 /* Error */, "_0_modifier_can_only_appear_on_a_type_parameter_of_a_class_interface_or_type_alias_1274", "'{0}' modifier can only appear on a type parameter of a class, interface or type alias"),
  accessor_modifier_can_only_appear_on_a_property_declaration: diag(1275, 1 /* Error */, "accessor_modifier_can_only_appear_on_a_property_declaration_1275", "'accessor' modifier can only appear on a property declaration."),
  An_accessor_property_cannot_be_declared_optional: diag(1276, 1 /* Error */, "An_accessor_property_cannot_be_declared_optional_1276", "An 'accessor' property cannot be declared optional."),
  _0_modifier_can_only_appear_on_a_type_parameter_of_a_function_method_or_class: diag(1277, 1 /* Error */, "_0_modifier_can_only_appear_on_a_type_parameter_of_a_function_method_or_class_1277", "'{0}' modifier can only appear on a type parameter of a function, method or class"),
  The_runtime_will_invoke_the_decorator_with_1_arguments_but_the_decorator_expects_0: diag(1278, 1 /* Error */, "The_runtime_will_invoke_the_decorator_with_1_arguments_but_the_decorator_expects_0_1278", "The runtime will invoke the decorator with {1} arguments, but the decorator expects {0}."),
  The_runtime_will_invoke_the_decorator_with_1_arguments_but_the_decorator_expects_at_least_0: diag(1279, 1 /* Error */, "The_runtime_will_invoke_the_decorator_with_1_arguments_but_the_decorator_expects_at_least_0_1279", "The runtime will invoke the decorator with {1} arguments, but the decorator expects at least {0}."),
  Namespaces_are_not_allowed_in_global_script_files_when_0_is_enabled_If_this_file_is_not_intended_to_be_a_global_script_set_moduleDetection_to_force_or_add_an_empty_export_statement: diag(1280, 1 /* Error */, "Namespaces_are_not_allowed_in_global_script_files_when_0_is_enabled_If_this_file_is_not_intended_to__1280", "Namespaces are not allowed in global script files when '{0}' is enabled. If this file is not intended to be a global script, set 'moduleDetection' to 'force' or add an empty 'export {}' statement."),
  Cannot_access_0_from_another_file_without_qualification_when_1_is_enabled_Use_2_instead: diag(1281, 1 /* Error */, "Cannot_access_0_from_another_file_without_qualification_when_1_is_enabled_Use_2_instead_1281", "Cannot access '{0}' from another file without qualification when '{1}' is enabled. Use '{2}' instead."),
  An_export_declaration_must_reference_a_value_when_verbatimModuleSyntax_is_enabled_but_0_only_refers_to_a_type: diag(1282, 1 /* Error */, "An_export_declaration_must_reference_a_value_when_verbatimModuleSyntax_is_enabled_but_0_only_refers__1282", "An 'export =' declaration must reference a value when 'verbatimModuleSyntax' is enabled, but '{0}' only refers to a type."),
  An_export_declaration_must_reference_a_real_value_when_verbatimModuleSyntax_is_enabled_but_0_resolves_to_a_type_only_declaration: diag(1283, 1 /* Error */, "An_export_declaration_must_reference_a_real_value_when_verbatimModuleSyntax_is_enabled_but_0_resolve_1283", "An 'export =' declaration must reference a real value when 'verbatimModuleSyntax' is enabled, but '{0}' resolves to a type-only declaration."),
  An_export_default_must_reference_a_value_when_verbatimModuleSyntax_is_enabled_but_0_only_refers_to_a_type: diag(1284, 1 /* Error */, "An_export_default_must_reference_a_value_when_verbatimModuleSyntax_is_enabled_but_0_only_refers_to_a_1284", "An 'export default' must reference a value when 'verbatimModuleSyntax' is enabled, but '{0}' only refers to a type."),
  An_export_default_must_reference_a_real_value_when_verbatimModuleSyntax_is_enabled_but_0_resolves_to_a_type_only_declaration: diag(1285, 1 /* Error */, "An_export_default_must_reference_a_real_value_when_verbatimModuleSyntax_is_enabled_but_0_resolves_to_1285", "An 'export default' must reference a real value when 'verbatimModuleSyntax' is enabled, but '{0}' resolves to a type-only declaration."),
  ESM_syntax_is_not_allowed_in_a_CommonJS_module_when_verbatimModuleSyntax_is_enabled: diag(1286, 1 /* Error */, "ESM_syntax_is_not_allowed_in_a_CommonJS_module_when_verbatimModuleSyntax_is_enabled_1286", "ESM syntax is not allowed in a CommonJS module when 'verbatimModuleSyntax' is enabled."),
  A_top_level_export_modifier_cannot_be_used_on_value_declarations_in_a_CommonJS_module_when_verbatimModuleSyntax_is_enabled: diag(1287, 1 /* Error */, "A_top_level_export_modifier_cannot_be_used_on_value_declarations_in_a_CommonJS_module_when_verbatimM_1287", "A top-level 'export' modifier cannot be used on value declarations in a CommonJS module when 'verbatimModuleSyntax' is enabled."),
  An_import_alias_cannot_resolve_to_a_type_or_type_only_declaration_when_verbatimModuleSyntax_is_enabled: diag(1288, 1 /* Error */, "An_import_alias_cannot_resolve_to_a_type_or_type_only_declaration_when_verbatimModuleSyntax_is_enabl_1288", "An import alias cannot resolve to a type or type-only declaration when 'verbatimModuleSyntax' is enabled."),
  _0_resolves_to_a_type_only_declaration_and_must_be_marked_type_only_in_this_file_before_re_exporting_when_1_is_enabled_Consider_using_import_type_where_0_is_imported: diag(1289, 1 /* Error */, "_0_resolves_to_a_type_only_declaration_and_must_be_marked_type_only_in_this_file_before_re_exporting_1289", "'{0}' resolves to a type-only declaration and must be marked type-only in this file before re-exporting when '{1}' is enabled. Consider using 'import type' where '{0}' is imported."),
  _0_resolves_to_a_type_only_declaration_and_must_be_marked_type_only_in_this_file_before_re_exporting_when_1_is_enabled_Consider_using_export_type_0_as_default: diag(1290, 1 /* Error */, "_0_resolves_to_a_type_only_declaration_and_must_be_marked_type_only_in_this_file_before_re_exporting_1290", "'{0}' resolves to a type-only declaration and must be marked type-only in this file before re-exporting when '{1}' is enabled. Consider using 'export type { {0} as default }'."),
  _0_resolves_to_a_type_and_must_be_marked_type_only_in_this_file_before_re_exporting_when_1_is_enabled_Consider_using_import_type_where_0_is_imported: diag(1291, 1 /* Error */, "_0_resolves_to_a_type_and_must_be_marked_type_only_in_this_file_before_re_exporting_when_1_is_enable_1291", "'{0}' resolves to a type and must be marked type-only in this file before re-exporting when '{1}' is enabled. Consider using 'import type' where '{0}' is imported."),
  _0_resolves_to_a_type_and_must_be_marked_type_only_in_this_file_before_re_exporting_when_1_is_enabled_Consider_using_export_type_0_as_default: diag(1292, 1 /* Error */, "_0_resolves_to_a_type_and_must_be_marked_type_only_in_this_file_before_re_exporting_when_1_is_enable_1292", "'{0}' resolves to a type and must be marked type-only in this file before re-exporting when '{1}' is enabled. Consider using 'export type { {0} as default }'."),
  ESM_syntax_is_not_allowed_in_a_CommonJS_module_when_module_is_set_to_preserve: diag(1293, 1 /* Error */, "ESM_syntax_is_not_allowed_in_a_CommonJS_module_when_module_is_set_to_preserve_1293", "ESM syntax is not allowed in a CommonJS module when 'module' is set to 'preserve'."),
  with_statements_are_not_allowed_in_an_async_function_block: diag(1300, 1 /* Error */, "with_statements_are_not_allowed_in_an_async_function_block_1300", "'with' statements are not allowed in an async function block."),
  await_expressions_are_only_allowed_within_async_functions_and_at_the_top_levels_of_modules: diag(1308, 1 /* Error */, "await_expressions_are_only_allowed_within_async_functions_and_at_the_top_levels_of_modules_1308", "'await' expressions are only allowed within async functions and at the top levels of modules."),
  The_current_file_is_a_CommonJS_module_and_cannot_use_await_at_the_top_level: diag(1309, 1 /* Error */, "The_current_file_is_a_CommonJS_module_and_cannot_use_await_at_the_top_level_1309", "The current file is a CommonJS module and cannot use 'await' at the top level."),
  Did_you_mean_to_use_a_Colon_An_can_only_follow_a_property_name_when_the_containing_object_literal_is_part_of_a_destructuring_pattern: diag(1312, 1 /* Error */, "Did_you_mean_to_use_a_Colon_An_can_only_follow_a_property_name_when_the_containing_object_literal_is_1312", "Did you mean to use a ':'? An '=' can only follow a property name when the containing object literal is part of a destructuring pattern."),
  The_body_of_an_if_statement_cannot_be_the_empty_statement: diag(1313, 1 /* Error */, "The_body_of_an_if_statement_cannot_be_the_empty_statement_1313", "The body of an 'if' statement cannot be the empty statement."),
  Global_module_exports_may_only_appear_in_module_files: diag(1314, 1 /* Error */, "Global_module_exports_may_only_appear_in_module_files_1314", "Global module exports may only appear in module files."),
  Global_module_exports_may_only_appear_in_declaration_files: diag(1315, 1 /* Error */, "Global_module_exports_may_only_appear_in_declaration_files_1315", "Global module exports may only appear in declaration files."),
  Global_module_exports_may_only_appear_at_top_level: diag(1316, 1 /* Error */, "Global_module_exports_may_only_appear_at_top_level_1316", "Global module exports may only appear at top level."),
  A_parameter_property_cannot_be_declared_using_a_rest_parameter: diag(1317, 1 /* Error */, "A_parameter_property_cannot_be_declared_using_a_rest_parameter_1317", "A parameter property cannot be declared using a rest parameter."),
  An_abstract_accessor_cannot_have_an_implementation: diag(1318, 1 /* Error */, "An_abstract_accessor_cannot_have_an_implementation_1318", "An abstract accessor cannot have an implementation."),
  A_default_export_can_only_be_used_in_an_ECMAScript_style_module: diag(1319, 1 /* Error */, "A_default_export_can_only_be_used_in_an_ECMAScript_style_module_1319", "A default export can only be used in an ECMAScript-style module."),
  Type_of_await_operand_must_either_be_a_valid_promise_or_must_not_contain_a_callable_then_member: diag(1320, 1 /* Error */, "Type_of_await_operand_must_either_be_a_valid_promise_or_must_not_contain_a_callable_then_member_1320", "Type of 'await' operand must either be a valid promise or must not contain a callable 'then' member."),
  Type_of_yield_operand_in_an_async_generator_must_either_be_a_valid_promise_or_must_not_contain_a_callable_then_member: diag(1321, 1 /* Error */, "Type_of_yield_operand_in_an_async_generator_must_either_be_a_valid_promise_or_must_not_contain_a_cal_1321", "Type of 'yield' operand in an async generator must either be a valid promise or must not contain a callable 'then' member."),
  Type_of_iterated_elements_of_a_yield_Asterisk_operand_must_either_be_a_valid_promise_or_must_not_contain_a_callable_then_member: diag(1322, 1 /* Error */, "Type_of_iterated_elements_of_a_yield_Asterisk_operand_must_either_be_a_valid_promise_or_must_not_con_1322", "Type of iterated elements of a 'yield*' operand must either be a valid promise or must not contain a callable 'then' member."),
  Dynamic_imports_are_only_supported_when_the_module_flag_is_set_to_es2020_es2022_esnext_commonjs_amd_system_umd_node16_or_nodenext: diag(1323, 1 /* Error */, "Dynamic_imports_are_only_supported_when_the_module_flag_is_set_to_es2020_es2022_esnext_commonjs_amd__1323", "Dynamic imports are only supported when the '--module' flag is set to 'es2020', 'es2022', 'esnext', 'commonjs', 'amd', 'system', 'umd', 'node16', or 'nodenext'."),
  Dynamic_imports_only_support_a_second_argument_when_the_module_option_is_set_to_esnext_node16_nodenext_or_preserve: diag(1324, 1 /* Error */, "Dynamic_imports_only_support_a_second_argument_when_the_module_option_is_set_to_esnext_node16_nodene_1324", "Dynamic imports only support a second argument when the '--module' option is set to 'esnext', 'node16', 'nodenext', or 'preserve'."),
  Argument_of_dynamic_import_cannot_be_spread_element: diag(1325, 1 /* Error */, "Argument_of_dynamic_import_cannot_be_spread_element_1325", "Argument of dynamic import cannot be spread element."),
  This_use_of_import_is_invalid_import_calls_can_be_written_but_they_must_have_parentheses_and_cannot_have_type_arguments: diag(1326, 1 /* Error */, "This_use_of_import_is_invalid_import_calls_can_be_written_but_they_must_have_parentheses_and_cannot__1326", "This use of 'import' is invalid. 'import()' calls can be written, but they must have parentheses and cannot have type arguments."),
  String_literal_with_double_quotes_expected: diag(1327, 1 /* Error */, "String_literal_with_double_quotes_expected_1327", "String literal with double quotes expected."),
  Property_value_can_only_be_string_literal_numeric_literal_true_false_null_object_literal_or_array_literal: diag(1328, 1 /* Error */, "Property_value_can_only_be_string_literal_numeric_literal_true_false_null_object_literal_or_array_li_1328", "Property value can only be string literal, numeric literal, 'true', 'false', 'null', object literal or array literal."),
  _0_accepts_too_few_arguments_to_be_used_as_a_decorator_here_Did_you_mean_to_call_it_first_and_write_0: diag(1329, 1 /* Error */, "_0_accepts_too_few_arguments_to_be_used_as_a_decorator_here_Did_you_mean_to_call_it_first_and_write__1329", "'{0}' accepts too few arguments to be used as a decorator here. Did you mean to call it first and write '@{0}()'?"),
  A_property_of_an_interface_or_type_literal_whose_type_is_a_unique_symbol_type_must_be_readonly: diag(1330, 1 /* Error */, "A_property_of_an_interface_or_type_literal_whose_type_is_a_unique_symbol_type_must_be_readonly_1330", "A property of an interface or type literal whose type is a 'unique symbol' type must be 'readonly'."),
  A_property_of_a_class_whose_type_is_a_unique_symbol_type_must_be_both_static_and_readonly: diag(1331, 1 /* Error */, "A_property_of_a_class_whose_type_is_a_unique_symbol_type_must_be_both_static_and_readonly_1331", "A property of a class whose type is a 'unique symbol' type must be both 'static' and 'readonly'."),
  A_variable_whose_type_is_a_unique_symbol_type_must_be_const: diag(1332, 1 /* Error */, "A_variable_whose_type_is_a_unique_symbol_type_must_be_const_1332", "A variable whose type is a 'unique symbol' type must be 'const'."),
  unique_symbol_types_may_not_be_used_on_a_variable_declaration_with_a_binding_name: diag(1333, 1 /* Error */, "unique_symbol_types_may_not_be_used_on_a_variable_declaration_with_a_binding_name_1333", "'unique symbol' types may not be used on a variable declaration with a binding name."),
  unique_symbol_types_are_only_allowed_on_variables_in_a_variable_statement: diag(1334, 1 /* Error */, "unique_symbol_types_are_only_allowed_on_variables_in_a_variable_statement_1334", "'unique symbol' types are only allowed on variables in a variable statement."),
  unique_symbol_types_are_not_allowed_here: diag(1335, 1 /* Error */, "unique_symbol_types_are_not_allowed_here_1335", "'unique symbol' types are not allowed here."),
  An_index_signature_parameter_type_cannot_be_a_literal_type_or_generic_type_Consider_using_a_mapped_object_type_instead: diag(1337, 1 /* Error */, "An_index_signature_parameter_type_cannot_be_a_literal_type_or_generic_type_Consider_using_a_mapped_o_1337", "An index signature parameter type cannot be a literal type or generic type. Consider using a mapped object type instead."),
  infer_declarations_are_only_permitted_in_the_extends_clause_of_a_conditional_type: diag(1338, 1 /* Error */, "infer_declarations_are_only_permitted_in_the_extends_clause_of_a_conditional_type_1338", "'infer' declarations are only permitted in the 'extends' clause of a conditional type."),
  Module_0_does_not_refer_to_a_value_but_is_used_as_a_value_here: diag(1339, 1 /* Error */, "Module_0_does_not_refer_to_a_value_but_is_used_as_a_value_here_1339", "Module '{0}' does not refer to a value, but is used as a value here."),
  Module_0_does_not_refer_to_a_type_but_is_used_as_a_type_here_Did_you_mean_typeof_import_0: diag(1340, 1 /* Error */, "Module_0_does_not_refer_to_a_type_but_is_used_as_a_type_here_Did_you_mean_typeof_import_0_1340", "Module '{0}' does not refer to a type, but is used as a type here. Did you mean 'typeof import('{0}')'?"),
  Class_constructor_may_not_be_an_accessor: diag(1341, 1 /* Error */, "Class_constructor_may_not_be_an_accessor_1341", "Class constructor may not be an accessor."),
  The_import_meta_meta_property_is_only_allowed_when_the_module_option_is_es2020_es2022_esnext_system_node16_or_nodenext: diag(1343, 1 /* Error */, "The_import_meta_meta_property_is_only_allowed_when_the_module_option_is_es2020_es2022_esnext_system__1343", "The 'import.meta' meta-property is only allowed when the '--module' option is 'es2020', 'es2022', 'esnext', 'system', 'node16', or 'nodenext'."),
  A_label_is_not_allowed_here: diag(1344, 1 /* Error */, "A_label_is_not_allowed_here_1344", "'A label is not allowed here."),
  An_expression_of_type_void_cannot_be_tested_for_truthiness: diag(1345, 1 /* Error */, "An_expression_of_type_void_cannot_be_tested_for_truthiness_1345", "An expression of type 'void' cannot be tested for truthiness."),
  This_parameter_is_not_allowed_with_use_strict_directive: diag(1346, 1 /* Error */, "This_parameter_is_not_allowed_with_use_strict_directive_1346", "This parameter is not allowed with 'use strict' directive."),
  use_strict_directive_cannot_be_used_with_non_simple_parameter_list: diag(1347, 1 /* Error */, "use_strict_directive_cannot_be_used_with_non_simple_parameter_list_1347", "'use strict' directive cannot be used with non-simple parameter list."),
  Non_simple_parameter_declared_here: diag(1348, 1 /* Error */, "Non_simple_parameter_declared_here_1348", "Non-simple parameter declared here."),
  use_strict_directive_used_here: diag(1349, 1 /* Error */, "use_strict_directive_used_here_1349", "'use strict' directive used here."),
  Print_the_final_configuration_instead_of_building: diag(1350, 3 /* Message */, "Print_the_final_configuration_instead_of_building_1350", "Print the final configuration instead of building."),
  An_identifier_or_keyword_cannot_immediately_follow_a_numeric_literal: diag(1351, 1 /* Error */, "An_identifier_or_keyword_cannot_immediately_follow_a_numeric_literal_1351", "An identifier or keyword cannot immediately follow a numeric literal."),
  A_bigint_literal_cannot_use_exponential_notation: diag(1352, 1 /* Error */, "A_bigint_literal_cannot_use_exponential_notation_1352", "A bigint literal cannot use exponential notation."),
  A_bigint_literal_must_be_an_integer: diag(1353, 1 /* Error */, "A_bigint_literal_must_be_an_integer_1353", "A bigint literal must be an integer."),
  readonly_type_modifier_is_only_permitted_on_array_and_tuple_literal_types: diag(1354, 1 /* Error */, "readonly_type_modifier_is_only_permitted_on_array_and_tuple_literal_types_1354", "'readonly' type modifier is only permitted on array and tuple literal types."),
  A_const_assertions_can_only_be_applied_to_references_to_enum_members_or_string_number_boolean_array_or_object_literals: diag(1355, 1 /* Error */, "A_const_assertions_can_only_be_applied_to_references_to_enum_members_or_string_number_boolean_array__1355", "A 'const' assertions can only be applied to references to enum members, or string, number, boolean, array, or object literals."),
  Did_you_mean_to_mark_this_function_as_async: diag(1356, 1 /* Error */, "Did_you_mean_to_mark_this_function_as_async_1356", "Did you mean to mark this function as 'async'?"),
  An_enum_member_name_must_be_followed_by_a_or: diag(1357, 1 /* Error */, "An_enum_member_name_must_be_followed_by_a_or_1357", "An enum member name must be followed by a ',', '=', or '}'."),
  Tagged_template_expressions_are_not_permitted_in_an_optional_chain: diag(1358, 1 /* Error */, "Tagged_template_expressions_are_not_permitted_in_an_optional_chain_1358", "Tagged template expressions are not permitted in an optional chain."),
  Identifier_expected_0_is_a_reserved_word_that_cannot_be_used_here: diag(1359, 1 /* Error */, "Identifier_expected_0_is_a_reserved_word_that_cannot_be_used_here_1359", "Identifier expected. '{0}' is a reserved word that cannot be used here."),
  Type_0_does_not_satisfy_the_expected_type_1: diag(1360, 1 /* Error */, "Type_0_does_not_satisfy_the_expected_type_1_1360", "Type '{0}' does not satisfy the expected type '{1}'."),
  _0_cannot_be_used_as_a_value_because_it_was_imported_using_import_type: diag(1361, 1 /* Error */, "_0_cannot_be_used_as_a_value_because_it_was_imported_using_import_type_1361", "'{0}' cannot be used as a value because it was imported using 'import type'."),
  _0_cannot_be_used_as_a_value_because_it_was_exported_using_export_type: diag(1362, 1 /* Error */, "_0_cannot_be_used_as_a_value_because_it_was_exported_using_export_type_1362", "'{0}' cannot be used as a value because it was exported using 'export type'."),
  A_type_only_import_can_specify_a_default_import_or_named_bindings_but_not_both: diag(1363, 1 /* Error */, "A_type_only_import_can_specify_a_default_import_or_named_bindings_but_not_both_1363", "A type-only import can specify a default import or named bindings, but not both."),
  Convert_to_type_only_export: diag(1364, 3 /* Message */, "Convert_to_type_only_export_1364", "Convert to type-only export"),
  Convert_all_re_exported_types_to_type_only_exports: diag(1365, 3 /* Message */, "Convert_all_re_exported_types_to_type_only_exports_1365", "Convert all re-exported types to type-only exports"),
  Split_into_two_separate_import_declarations: diag(1366, 3 /* Message */, "Split_into_two_separate_import_declarations_1366", "Split into two separate import declarations"),
  Split_all_invalid_type_only_imports: diag(1367, 3 /* Message */, "Split_all_invalid_type_only_imports_1367", "Split all invalid type-only imports"),
  Class_constructor_may_not_be_a_generator: diag(1368, 1 /* Error */, "Class_constructor_may_not_be_a_generator_1368", "Class constructor may not be a generator."),
  Did_you_mean_0: diag(1369, 3 /* Message */, "Did_you_mean_0_1369", "Did you mean '{0}'?"),
  await_expressions_are_only_allowed_at_the_top_level_of_a_file_when_that_file_is_a_module_but_this_file_has_no_imports_or_exports_Consider_adding_an_empty_export_to_make_this_file_a_module: diag(1375, 1 /* Error */, "await_expressions_are_only_allowed_at_the_top_level_of_a_file_when_that_file_is_a_module_but_this_fi_1375", "'await' expressions are only allowed at the top level of a file when that file is a module, but this file has no imports or exports. Consider adding an empty 'export {}' to make this file a module."),
  _0_was_imported_here: diag(1376, 3 /* Message */, "_0_was_imported_here_1376", "'{0}' was imported here."),
  _0_was_exported_here: diag(1377, 3 /* Message */, "_0_was_exported_here_1377", "'{0}' was exported here."),
  Top_level_await_expressions_are_only_allowed_when_the_module_option_is_set_to_es2022_esnext_system_node16_nodenext_or_preserve_and_the_target_option_is_set_to_es2017_or_higher: diag(1378, 1 /* Error */, "Top_level_await_expressions_are_only_allowed_when_the_module_option_is_set_to_es2022_esnext_system_n_1378", "Top-level 'await' expressions are only allowed when the 'module' option is set to 'es2022', 'esnext', 'system', 'node16', 'nodenext', or 'preserve', and the 'target' option is set to 'es2017' or higher."),
  An_import_alias_cannot_reference_a_declaration_that_was_exported_using_export_type: diag(1379, 1 /* Error */, "An_import_alias_cannot_reference_a_declaration_that_was_exported_using_export_type_1379", "An import alias cannot reference a declaration that was exported using 'export type'."),
  An_import_alias_cannot_reference_a_declaration_that_was_imported_using_import_type: diag(1380, 1 /* Error */, "An_import_alias_cannot_reference_a_declaration_that_was_imported_using_import_type_1380", "An import alias cannot reference a declaration that was imported using 'import type'."),
  Unexpected_token_Did_you_mean_or_rbrace: diag(1381, 1 /* Error */, "Unexpected_token_Did_you_mean_or_rbrace_1381", "Unexpected token. Did you mean `{'}'}` or `&rbrace;`?"),
  Unexpected_token_Did_you_mean_or_gt: diag(1382, 1 /* Error */, "Unexpected_token_Did_you_mean_or_gt_1382", "Unexpected token. Did you mean `{'>'}` or `&gt;`?"),
  Function_type_notation_must_be_parenthesized_when_used_in_a_union_type: diag(1385, 1 /* Error */, "Function_type_notation_must_be_parenthesized_when_used_in_a_union_type_1385", "Function type notation must be parenthesized when used in a union type."),
  Constructor_type_notation_must_be_parenthesized_when_used_in_a_union_type: diag(1386, 1 /* Error */, "Constructor_type_notation_must_be_parenthesized_when_used_in_a_union_type_1386", "Constructor type notation must be parenthesized when used in a union type."),
  Function_type_notation_must_be_parenthesized_when_used_in_an_intersection_type: diag(1387, 1 /* Error */, "Function_type_notation_must_be_parenthesized_when_used_in_an_intersection_type_1387", "Function type notation must be parenthesized when used in an intersection type."),
  Constructor_type_notation_must_be_parenthesized_when_used_in_an_intersection_type: diag(1388, 1 /* Error */, "Constructor_type_notation_must_be_parenthesized_when_used_in_an_intersection_type_1388", "Constructor type notation must be parenthesized when used in an intersection type."),
  _0_is_not_allowed_as_a_variable_declaration_name: diag(1389, 1 /* Error */, "_0_is_not_allowed_as_a_variable_declaration_name_1389", "'{0}' is not allowed as a variable declaration name."),
  _0_is_not_allowed_as_a_parameter_name: diag(1390, 1 /* Error */, "_0_is_not_allowed_as_a_parameter_name_1390", "'{0}' is not allowed as a parameter name."),
  An_import_alias_cannot_use_import_type: diag(1392, 1 /* Error */, "An_import_alias_cannot_use_import_type_1392", "An import alias cannot use 'import type'"),
  Imported_via_0_from_file_1: diag(1393, 3 /* Message */, "Imported_via_0_from_file_1_1393", "Imported via {0} from file '{1}'"),
  Imported_via_0_from_file_1_with_packageId_2: diag(1394, 3 /* Message */, "Imported_via_0_from_file_1_with_packageId_2_1394", "Imported via {0} from file '{1}' with packageId '{2}'"),
  Imported_via_0_from_file_1_to_import_importHelpers_as_specified_in_compilerOptions: diag(1395, 3 /* Message */, "Imported_via_0_from_file_1_to_import_importHelpers_as_specified_in_compilerOptions_1395", "Imported via {0} from file '{1}' to import 'importHelpers' as specified in compilerOptions"),
  Imported_via_0_from_file_1_with_packageId_2_to_import_importHelpers_as_specified_in_compilerOptions: diag(1396, 3 /* Message */, "Imported_via_0_from_file_1_with_packageId_2_to_import_importHelpers_as_specified_in_compilerOptions_1396", "Imported via {0} from file '{1}' with packageId '{2}' to import 'importHelpers' as specified in compilerOptions"),
  Imported_via_0_from_file_1_to_import_jsx_and_jsxs_factory_functions: diag(1397, 3 /* Message */, "Imported_via_0_from_file_1_to_import_jsx_and_jsxs_factory_functions_1397", "Imported via {0} from file '{1}' to import 'jsx' and 'jsxs' factory functions"),
  Imported_via_0_from_file_1_with_packageId_2_to_import_jsx_and_jsxs_factory_functions: diag(1398, 3 /* Message */, "Imported_via_0_from_file_1_with_packageId_2_to_import_jsx_and_jsxs_factory_functions_1398", "Imported via {0} from file '{1}' with packageId '{2}' to import 'jsx' and 'jsxs' factory functions"),
  File_is_included_via_import_here: diag(1399, 3 /* Message */, "File_is_included_via_import_here_1399", "File is included via import here."),
  Referenced_via_0_from_file_1: diag(1400, 3 /* Message */, "Referenced_via_0_from_file_1_1400", "Referenced via '{0}' from file '{1}'"),
  File_is_included_via_reference_here: diag(1401, 3 /* Message */, "File_is_included_via_reference_here_1401", "File is included via reference here."),
  Type_library_referenced_via_0_from_file_1: diag(1402, 3 /* Message */, "Type_library_referenced_via_0_from_file_1_1402", "Type library referenced via '{0}' from file '{1}'"),
  Type_library_referenced_via_0_from_file_1_with_packageId_2: diag(1403, 3 /* Message */, "Type_library_referenced_via_0_from_file_1_with_packageId_2_1403", "Type library referenced via '{0}' from file '{1}' with packageId '{2}'"),
  File_is_included_via_type_library_reference_here: diag(1404, 3 /* Message */, "File_is_included_via_type_library_reference_here_1404", "File is included via type library reference here."),
  Library_referenced_via_0_from_file_1: diag(1405, 3 /* Message */, "Library_referenced_via_0_from_file_1_1405", "Library referenced via '{0}' from file '{1}'"),
  File_is_included_via_library_reference_here: diag(1406, 3 /* Message */, "File_is_included_via_library_reference_here_1406", "File is included via library reference here."),
  Matched_by_include_pattern_0_in_1: diag(1407, 3 /* Message */, "Matched_by_include_pattern_0_in_1_1407", "Matched by include pattern '{0}' in '{1}'"),
  File_is_matched_by_include_pattern_specified_here: diag(1408, 3 /* Message */, "File_is_matched_by_include_pattern_specified_here_1408", "File is matched by include pattern specified here."),
  Part_of_files_list_in_tsconfig_json: diag(1409, 3 /* Message */, "Part_of_files_list_in_tsconfig_json_1409", "Part of 'files' list in tsconfig.json"),
  File_is_matched_by_files_list_specified_here: diag(1410, 3 /* Message */, "File_is_matched_by_files_list_specified_here_1410", "File is matched by 'files' list specified here."),
  Output_from_referenced_project_0_included_because_1_specified: diag(1411, 3 /* Message */, "Output_from_referenced_project_0_included_because_1_specified_1411", "Output from referenced project '{0}' included because '{1}' specified"),
  Output_from_referenced_project_0_included_because_module_is_specified_as_none: diag(1412, 3 /* Message */, "Output_from_referenced_project_0_included_because_module_is_specified_as_none_1412", "Output from referenced project '{0}' included because '--module' is specified as 'none'"),
  File_is_output_from_referenced_project_specified_here: diag(1413, 3 /* Message */, "File_is_output_from_referenced_project_specified_here_1413", "File is output from referenced project specified here."),
  Source_from_referenced_project_0_included_because_1_specified: diag(1414, 3 /* Message */, "Source_from_referenced_project_0_included_because_1_specified_1414", "Source from referenced project '{0}' included because '{1}' specified"),
  Source_from_referenced_project_0_included_because_module_is_specified_as_none: diag(1415, 3 /* Message */, "Source_from_referenced_project_0_included_because_module_is_specified_as_none_1415", "Source from referenced project '{0}' included because '--module' is specified as 'none'"),
  File_is_source_from_referenced_project_specified_here: diag(1416, 3 /* Message */, "File_is_source_from_referenced_project_specified_here_1416", "File is source from referenced project specified here."),
  Entry_point_of_type_library_0_specified_in_compilerOptions: diag(1417, 3 /* Message */, "Entry_point_of_type_library_0_specified_in_compilerOptions_1417", "Entry point of type library '{0}' specified in compilerOptions"),
  Entry_point_of_type_library_0_specified_in_compilerOptions_with_packageId_1: diag(1418, 3 /* Message */, "Entry_point_of_type_library_0_specified_in_compilerOptions_with_packageId_1_1418", "Entry point of type library '{0}' specified in compilerOptions with packageId '{1}'"),
  File_is_entry_point_of_type_library_specified_here: diag(1419, 3 /* Message */, "File_is_entry_point_of_type_library_specified_here_1419", "File is entry point of type library specified here."),
  Entry_point_for_implicit_type_library_0: diag(1420, 3 /* Message */, "Entry_point_for_implicit_type_library_0_1420", "Entry point for implicit type library '{0}'"),
  Entry_point_for_implicit_type_library_0_with_packageId_1: diag(1421, 3 /* Message */, "Entry_point_for_implicit_type_library_0_with_packageId_1_1421", "Entry point for implicit type library '{0}' with packageId '{1}'"),
  Library_0_specified_in_compilerOptions: diag(1422, 3 /* Message */, "Library_0_specified_in_compilerOptions_1422", "Library '{0}' specified in compilerOptions"),
  File_is_library_specified_here: diag(1423, 3 /* Message */, "File_is_library_specified_here_1423", "File is library specified here."),
  Default_library: diag(1424, 3 /* Message */, "Default_library_1424", "Default library"),
  Default_library_for_target_0: diag(1425, 3 /* Message */, "Default_library_for_target_0_1425", "Default library for target '{0}'"),
  File_is_default_library_for_target_specified_here: diag(1426, 3 /* Message */, "File_is_default_library_for_target_specified_here_1426", "File is default library for target specified here."),
  Root_file_specified_for_compilation: diag(1427, 3 /* Message */, "Root_file_specified_for_compilation_1427", "Root file specified for compilation"),
  File_is_output_of_project_reference_source_0: diag(1428, 3 /* Message */, "File_is_output_of_project_reference_source_0_1428", "File is output of project reference source '{0}'"),
  File_redirects_to_file_0: diag(1429, 3 /* Message */, "File_redirects_to_file_0_1429", "File redirects to file '{0}'"),
  The_file_is_in_the_program_because_Colon: diag(1430, 3 /* Message */, "The_file_is_in_the_program_because_Colon_1430", "The file is in the program because:"),
  for_await_loops_are_only_allowed_at_the_top_level_of_a_file_when_that_file_is_a_module_but_this_file_has_no_imports_or_exports_Consider_adding_an_empty_export_to_make_this_file_a_module: diag(1431, 1 /* Error */, "for_await_loops_are_only_allowed_at_the_top_level_of_a_file_when_that_file_is_a_module_but_this_file_1431", "'for await' loops are only allowed at the top level of a file when that file is a module, but this file has no imports or exports. Consider adding an empty 'export {}' to make this file a module."),
  Top_level_for_await_loops_are_only_allowed_when_the_module_option_is_set_to_es2022_esnext_system_node16_nodenext_or_preserve_and_the_target_option_is_set_to_es2017_or_higher: diag(1432, 1 /* Error */, "Top_level_for_await_loops_are_only_allowed_when_the_module_option_is_set_to_es2022_esnext_system_nod_1432", "Top-level 'for await' loops are only allowed when the 'module' option is set to 'es2022', 'esnext', 'system', 'node16', 'nodenext', or 'preserve', and the 'target' option is set to 'es2017' or higher."),
  Neither_decorators_nor_modifiers_may_be_applied_to_this_parameters: diag(1433, 1 /* Error */, "Neither_decorators_nor_modifiers_may_be_applied_to_this_parameters_1433", "Neither decorators nor modifiers may be applied to 'this' parameters."),
  Unexpected_keyword_or_identifier: diag(1434, 1 /* Error */, "Unexpected_keyword_or_identifier_1434", "Unexpected keyword or identifier."),
  Unknown_keyword_or_identifier_Did_you_mean_0: diag(1435, 1 /* Error */, "Unknown_keyword_or_identifier_Did_you_mean_0_1435", "Unknown keyword or identifier. Did you mean '{0}'?"),
  Decorators_must_precede_the_name_and_all_keywords_of_property_declarations: diag(1436, 1 /* Error */, "Decorators_must_precede_the_name_and_all_keywords_of_property_declarations_1436", "Decorators must precede the name and all keywords of property declarations."),
  Namespace_must_be_given_a_name: diag(1437, 1 /* Error */, "Namespace_must_be_given_a_name_1437", "Namespace must be given a name."),
  Interface_must_be_given_a_name: diag(1438, 1 /* Error */, "Interface_must_be_given_a_name_1438", "Interface must be given a name."),
  Type_alias_must_be_given_a_name: diag(1439, 1 /* Error */, "Type_alias_must_be_given_a_name_1439", "Type alias must be given a name."),
  Variable_declaration_not_allowed_at_this_location: diag(1440, 1 /* Error */, "Variable_declaration_not_allowed_at_this_location_1440", "Variable declaration not allowed at this location."),
  Cannot_start_a_function_call_in_a_type_annotation: diag(1441, 1 /* Error */, "Cannot_start_a_function_call_in_a_type_annotation_1441", "Cannot start a function call in a type annotation."),
  Expected_for_property_initializer: diag(1442, 1 /* Error */, "Expected_for_property_initializer_1442", "Expected '=' for property initializer."),
  Module_declaration_names_may_only_use_or_quoted_strings: diag(1443, 1 /* Error */, "Module_declaration_names_may_only_use_or_quoted_strings_1443", `Module declaration names may only use ' or " quoted strings.`),
  _0_resolves_to_a_type_only_declaration_and_must_be_re_exported_using_a_type_only_re_export_when_1_is_enabled: diag(1448, 1 /* Error */, "_0_resolves_to_a_type_only_declaration_and_must_be_re_exported_using_a_type_only_re_export_when_1_is_1448", "'{0}' resolves to a type-only declaration and must be re-exported using a type-only re-export when '{1}' is enabled."),
  Preserve_unused_imported_values_in_the_JavaScript_output_that_would_otherwise_be_removed: diag(1449, 3 /* Message */, "Preserve_unused_imported_values_in_the_JavaScript_output_that_would_otherwise_be_removed_1449", "Preserve unused imported values in the JavaScript output that would otherwise be removed."),
  Dynamic_imports_can_only_accept_a_module_specifier_and_an_optional_set_of_attributes_as_arguments: diag(1450, 3 /* Message */, "Dynamic_imports_can_only_accept_a_module_specifier_and_an_optional_set_of_attributes_as_arguments_1450", "Dynamic imports can only accept a module specifier and an optional set of attributes as arguments"),
  Private_identifiers_are_only_allowed_in_class_bodies_and_may_only_be_used_as_part_of_a_class_member_declaration_property_access_or_on_the_left_hand_side_of_an_in_expression: diag(1451, 1 /* Error */, "Private_identifiers_are_only_allowed_in_class_bodies_and_may_only_be_used_as_part_of_a_class_member__1451", "Private identifiers are only allowed in class bodies and may only be used as part of a class member declaration, property access, or on the left-hand-side of an 'in' expression"),
  resolution_mode_should_be_either_require_or_import: diag(1453, 1 /* Error */, "resolution_mode_should_be_either_require_or_import_1453", "`resolution-mode` should be either `require` or `import`."),
  resolution_mode_can_only_be_set_for_type_only_imports: diag(1454, 1 /* Error */, "resolution_mode_can_only_be_set_for_type_only_imports_1454", "`resolution-mode` can only be set for type-only imports."),
  resolution_mode_is_the_only_valid_key_for_type_import_assertions: diag(1455, 1 /* Error */, "resolution_mode_is_the_only_valid_key_for_type_import_assertions_1455", "`resolution-mode` is the only valid key for type import assertions."),
  Type_import_assertions_should_have_exactly_one_key_resolution_mode_with_value_import_or_require: diag(1456, 1 /* Error */, "Type_import_assertions_should_have_exactly_one_key_resolution_mode_with_value_import_or_require_1456", "Type import assertions should have exactly one key - `resolution-mode` - with value `import` or `require`."),
  Matched_by_default_include_pattern_Asterisk_Asterisk_Slash_Asterisk: diag(1457, 3 /* Message */, "Matched_by_default_include_pattern_Asterisk_Asterisk_Slash_Asterisk_1457", "Matched by default include pattern '**/*'"),
  File_is_ECMAScript_module_because_0_has_field_type_with_value_module: diag(1458, 3 /* Message */, "File_is_ECMAScript_module_because_0_has_field_type_with_value_module_1458", `File is ECMAScript module because '{0}' has field "type" with value "module"`),
  File_is_CommonJS_module_because_0_has_field_type_whose_value_is_not_module: diag(1459, 3 /* Message */, "File_is_CommonJS_module_because_0_has_field_type_whose_value_is_not_module_1459", `File is CommonJS module because '{0}' has field "type" whose value is not "module"`),
  File_is_CommonJS_module_because_0_does_not_have_field_type: diag(1460, 3 /* Message */, "File_is_CommonJS_module_because_0_does_not_have_field_type_1460", `File is CommonJS module because '{0}' does not have field "type"`),
  File_is_CommonJS_module_because_package_json_was_not_found: diag(1461, 3 /* Message */, "File_is_CommonJS_module_because_package_json_was_not_found_1461", "File is CommonJS module because 'package.json' was not found"),
  resolution_mode_is_the_only_valid_key_for_type_import_attributes: diag(1463, 1 /* Error */, "resolution_mode_is_the_only_valid_key_for_type_import_attributes_1463", "'resolution-mode' is the only valid key for type import attributes."),
  Type_import_attributes_should_have_exactly_one_key_resolution_mode_with_value_import_or_require: diag(1464, 1 /* Error */, "Type_import_attributes_should_have_exactly_one_key_resolution_mode_with_value_import_or_require_1464", "Type import attributes should have exactly one key - 'resolution-mode' - with value 'import' or 'require'."),
  The_import_meta_meta_property_is_not_allowed_in_files_which_will_build_into_CommonJS_output: diag(1470, 1 /* Error */, "The_import_meta_meta_property_is_not_allowed_in_files_which_will_build_into_CommonJS_output_1470", "The 'import.meta' meta-property is not allowed in files which will build into CommonJS output."),
  Module_0_cannot_be_imported_using_this_construct_The_specifier_only_resolves_to_an_ES_module_which_cannot_be_imported_with_require_Use_an_ECMAScript_import_instead: diag(1471, 1 /* Error */, "Module_0_cannot_be_imported_using_this_construct_The_specifier_only_resolves_to_an_ES_module_which_c_1471", "Module '{0}' cannot be imported using this construct. The specifier only resolves to an ES module, which cannot be imported with 'require'. Use an ECMAScript import instead."),
  catch_or_finally_expected: diag(1472, 1 /* Error */, "catch_or_finally_expected_1472", "'catch' or 'finally' expected."),
  An_import_declaration_can_only_be_used_at_the_top_level_of_a_module: diag(1473, 1 /* Error */, "An_import_declaration_can_only_be_used_at_the_top_level_of_a_module_1473", "An import declaration can only be used at the top level of a module."),
  An_export_declaration_can_only_be_used_at_the_top_level_of_a_module: diag(1474, 1 /* Error */, "An_export_declaration_can_only_be_used_at_the_top_level_of_a_module_1474", "An export declaration can only be used at the top level of a module."),
  Control_what_method_is_used_to_detect_module_format_JS_files: diag(1475, 3 /* Message */, "Control_what_method_is_used_to_detect_module_format_JS_files_1475", "Control what method is used to detect module-format JS files."),
  auto_Colon_Treat_files_with_imports_exports_import_meta_jsx_with_jsx_Colon_react_jsx_or_esm_format_with_module_Colon_node16_as_modules: diag(1476, 3 /* Message */, "auto_Colon_Treat_files_with_imports_exports_import_meta_jsx_with_jsx_Colon_react_jsx_or_esm_format_w_1476", '"auto": Treat files with imports, exports, import.meta, jsx (with jsx: react-jsx), or esm format (with module: node16+) as modules.'),
  An_instantiation_expression_cannot_be_followed_by_a_property_access: diag(1477, 1 /* Error */, "An_instantiation_expression_cannot_be_followed_by_a_property_access_1477", "An instantiation expression cannot be followed by a property access."),
  Identifier_or_string_literal_expected: diag(1478, 1 /* Error */, "Identifier_or_string_literal_expected_1478", "Identifier or string literal expected."),
  The_current_file_is_a_CommonJS_module_whose_imports_will_produce_require_calls_however_the_referenced_file_is_an_ECMAScript_module_and_cannot_be_imported_with_require_Consider_writing_a_dynamic_import_0_call_instead: diag(1479, 1 /* Error */, "The_current_file_is_a_CommonJS_module_whose_imports_will_produce_require_calls_however_the_reference_1479", `The current file is a CommonJS module whose imports will produce 'require' calls; however, the referenced file is an ECMAScript module and cannot be imported with 'require'. Consider writing a dynamic 'import("{0}")' call instead.`),
  To_convert_this_file_to_an_ECMAScript_module_change_its_file_extension_to_0_or_create_a_local_package_json_file_with_type_Colon_module: diag(1480, 3 /* Message */, "To_convert_this_file_to_an_ECMAScript_module_change_its_file_extension_to_0_or_create_a_local_packag_1480", 'To convert this file to an ECMAScript module, change its file extension to \'{0}\' or create a local package.json file with `{ "type": "module" }`.'),
  To_convert_this_file_to_an_ECMAScript_module_change_its_file_extension_to_0_or_add_the_field_type_Colon_module_to_1: diag(1481, 3 /* Message */, "To_convert_this_file_to_an_ECMAScript_module_change_its_file_extension_to_0_or_add_the_field_type_Co_1481", `To convert this file to an ECMAScript module, change its file extension to '{0}', or add the field \`"type": "module"\` to '{1}'.`),
  To_convert_this_file_to_an_ECMAScript_module_add_the_field_type_Colon_module_to_0: diag(1482, 3 /* Message */, "To_convert_this_file_to_an_ECMAScript_module_add_the_field_type_Colon_module_to_0_1482", 'To convert this file to an ECMAScript module, add the field `"type": "module"` to \'{0}\'.'),
  To_convert_this_file_to_an_ECMAScript_module_create_a_local_package_json_file_with_type_Colon_module: diag(1483, 3 /* Message */, "To_convert_this_file_to_an_ECMAScript_module_create_a_local_package_json_file_with_type_Colon_module_1483", 'To convert this file to an ECMAScript module, create a local package.json file with `{ "type": "module" }`.'),
  _0_is_a_type_and_must_be_imported_using_a_type_only_import_when_verbatimModuleSyntax_is_enabled: diag(1484, 1 /* Error */, "_0_is_a_type_and_must_be_imported_using_a_type_only_import_when_verbatimModuleSyntax_is_enabled_1484", "'{0}' is a type and must be imported using a type-only import when 'verbatimModuleSyntax' is enabled."),
  _0_resolves_to_a_type_only_declaration_and_must_be_imported_using_a_type_only_import_when_verbatimModuleSyntax_is_enabled: diag(1485, 1 /* Error */, "_0_resolves_to_a_type_only_declaration_and_must_be_imported_using_a_type_only_import_when_verbatimMo_1485", "'{0}' resolves to a type-only declaration and must be imported using a type-only import when 'verbatimModuleSyntax' is enabled."),
  Decorator_used_before_export_here: diag(1486, 1 /* Error */, "Decorator_used_before_export_here_1486", "Decorator used before 'export' here."),
  Octal_escape_sequences_are_not_allowed_Use_the_syntax_0: diag(1487, 1 /* Error */, "Octal_escape_sequences_are_not_allowed_Use_the_syntax_0_1487", "Octal escape sequences are not allowed. Use the syntax '{0}'."),
  Escape_sequence_0_is_not_allowed: diag(1488, 1 /* Error */, "Escape_sequence_0_is_not_allowed_1488", "Escape sequence '{0}' is not allowed."),
  Decimals_with_leading_zeros_are_not_allowed: diag(1489, 1 /* Error */, "Decimals_with_leading_zeros_are_not_allowed_1489", "Decimals with leading zeros are not allowed."),
  File_appears_to_be_binary: diag(1490, 1 /* Error */, "File_appears_to_be_binary_1490", "File appears to be binary."),
  _0_modifier_cannot_appear_on_a_using_declaration: diag(1491, 1 /* Error */, "_0_modifier_cannot_appear_on_a_using_declaration_1491", "'{0}' modifier cannot appear on a 'using' declaration."),
  _0_declarations_may_not_have_binding_patterns: diag(1492, 1 /* Error */, "_0_declarations_may_not_have_binding_patterns_1492", "'{0}' declarations may not have binding patterns."),
  The_left_hand_side_of_a_for_in_statement_cannot_be_a_using_declaration: diag(1493, 1 /* Error */, "The_left_hand_side_of_a_for_in_statement_cannot_be_a_using_declaration_1493", "The left-hand side of a 'for...in' statement cannot be a 'using' declaration."),
  The_left_hand_side_of_a_for_in_statement_cannot_be_an_await_using_declaration: diag(1494, 1 /* Error */, "The_left_hand_side_of_a_for_in_statement_cannot_be_an_await_using_declaration_1494", "The left-hand side of a 'for...in' statement cannot be an 'await using' declaration."),
  _0_modifier_cannot_appear_on_an_await_using_declaration: diag(1495, 1 /* Error */, "_0_modifier_cannot_appear_on_an_await_using_declaration_1495", "'{0}' modifier cannot appear on an 'await using' declaration."),
  Identifier_string_literal_or_number_literal_expected: diag(1496, 1 /* Error */, "Identifier_string_literal_or_number_literal_expected_1496", "Identifier, string literal, or number literal expected."),
  Expression_must_be_enclosed_in_parentheses_to_be_used_as_a_decorator: diag(1497, 1 /* Error */, "Expression_must_be_enclosed_in_parentheses_to_be_used_as_a_decorator_1497", "Expression must be enclosed in parentheses to be used as a decorator."),
  Invalid_syntax_in_decorator: diag(1498, 1 /* Error */, "Invalid_syntax_in_decorator_1498", "Invalid syntax in decorator."),
  Unknown_regular_expression_flag: diag(1499, 1 /* Error */, "Unknown_regular_expression_flag_1499", "Unknown regular expression flag."),
  Duplicate_regular_expression_flag: diag(1500, 1 /* Error */, "Duplicate_regular_expression_flag_1500", "Duplicate regular expression flag."),
  This_regular_expression_flag_is_only_available_when_targeting_0_or_later: diag(1501, 1 /* Error */, "This_regular_expression_flag_is_only_available_when_targeting_0_or_later_1501", "This regular expression flag is only available when targeting '{0}' or later."),
  The_Unicode_u_flag_and_the_Unicode_Sets_v_flag_cannot_be_set_simultaneously: diag(1502, 1 /* Error */, "The_Unicode_u_flag_and_the_Unicode_Sets_v_flag_cannot_be_set_simultaneously_1502", "The Unicode (u) flag and the Unicode Sets (v) flag cannot be set simultaneously."),
  Named_capturing_groups_are_only_available_when_targeting_ES2018_or_later: diag(1503, 1 /* Error */, "Named_capturing_groups_are_only_available_when_targeting_ES2018_or_later_1503", "Named capturing groups are only available when targeting 'ES2018' or later."),
  Subpattern_flags_must_be_present_when_there_is_a_minus_sign: diag(1504, 1 /* Error */, "Subpattern_flags_must_be_present_when_there_is_a_minus_sign_1504", "Subpattern flags must be present when there is a minus sign."),
  Incomplete_quantifier_Digit_expected: diag(1505, 1 /* Error */, "Incomplete_quantifier_Digit_expected_1505", "Incomplete quantifier. Digit expected."),
  Numbers_out_of_order_in_quantifier: diag(1506, 1 /* Error */, "Numbers_out_of_order_in_quantifier_1506", "Numbers out of order in quantifier."),
  There_is_nothing_available_for_repetition: diag(1507, 1 /* Error */, "There_is_nothing_available_for_repetition_1507", "There is nothing available for repetition."),
  Unexpected_0_Did_you_mean_to_escape_it_with_backslash: diag(1508, 1 /* Error */, "Unexpected_0_Did_you_mean_to_escape_it_with_backslash_1508", "Unexpected '{0}'. Did you mean to escape it with backslash?"),
  This_regular_expression_flag_cannot_be_toggled_within_a_subpattern: diag(1509, 1 /* Error */, "This_regular_expression_flag_cannot_be_toggled_within_a_subpattern_1509", "This regular expression flag cannot be toggled within a subpattern."),
  k_must_be_followed_by_a_capturing_group_name_enclosed_in_angle_brackets: diag(1510, 1 /* Error */, "k_must_be_followed_by_a_capturing_group_name_enclosed_in_angle_brackets_1510", "'\\k' must be followed by a capturing group name enclosed in angle brackets."),
  q_is_only_available_inside_character_class: diag(1511, 1 /* Error */, "q_is_only_available_inside_character_class_1511", "'\\q' is only available inside character class."),
  c_must_be_followed_by_an_ASCII_letter: diag(1512, 1 /* Error */, "c_must_be_followed_by_an_ASCII_letter_1512", "'\\c' must be followed by an ASCII letter."),
  Undetermined_character_escape: diag(1513, 1 /* Error */, "Undetermined_character_escape_1513", "Undetermined character escape."),
  Expected_a_capturing_group_name: diag(1514, 1 /* Error */, "Expected_a_capturing_group_name_1514", "Expected a capturing group name."),
  Named_capturing_groups_with_the_same_name_must_be_mutually_exclusive_to_each_other: diag(1515, 1 /* Error */, "Named_capturing_groups_with_the_same_name_must_be_mutually_exclusive_to_each_other_1515", "Named capturing groups with the same name must be mutually exclusive to each other."),
  A_character_class_range_must_not_be_bounded_by_another_character_class: diag(1516, 1 /* Error */, "A_character_class_range_must_not_be_bounded_by_another_character_class_1516", "A character class range must not be bounded by another character class."),
  Range_out_of_order_in_character_class: diag(1517, 1 /* Error */, "Range_out_of_order_in_character_class_1517", "Range out of order in character class."),
  Anything_that_would_possibly_match_more_than_a_single_character_is_invalid_inside_a_negated_character_class: diag(1518, 1 /* Error */, "Anything_that_would_possibly_match_more_than_a_single_character_is_invalid_inside_a_negated_characte_1518", "Anything that would possibly match more than a single character is invalid inside a negated character class."),
  Operators_must_not_be_mixed_within_a_character_class_Wrap_it_in_a_nested_class_instead: diag(1519, 1 /* Error */, "Operators_must_not_be_mixed_within_a_character_class_Wrap_it_in_a_nested_class_instead_1519", "Operators must not be mixed within a character class. Wrap it in a nested class instead."),
  Expected_a_class_set_operand: diag(1520, 1 /* Error */, "Expected_a_class_set_operand_1520", "Expected a class set operand."),
  q_must_be_followed_by_string_alternatives_enclosed_in_braces: diag(1521, 1 /* Error */, "q_must_be_followed_by_string_alternatives_enclosed_in_braces_1521", "'\\q' must be followed by string alternatives enclosed in braces."),
  A_character_class_must_not_contain_a_reserved_double_punctuator_Did_you_mean_to_escape_it_with_backslash: diag(1522, 1 /* Error */, "A_character_class_must_not_contain_a_reserved_double_punctuator_Did_you_mean_to_escape_it_with_backs_1522", "A character class must not contain a reserved double punctuator. Did you mean to escape it with backslash?"),
  Expected_a_Unicode_property_name: diag(1523, 1 /* Error */, "Expected_a_Unicode_property_name_1523", "Expected a Unicode property name."),
  Unknown_Unicode_property_name: diag(1524, 1 /* Error */, "Unknown_Unicode_property_name_1524", "Unknown Unicode property name."),
  Expected_a_Unicode_property_value: diag(1525, 1 /* Error */, "Expected_a_Unicode_property_value_1525", "Expected a Unicode property value."),
  Unknown_Unicode_property_value: diag(1526, 1 /* Error */, "Unknown_Unicode_property_value_1526", "Unknown Unicode property value."),
  Expected_a_Unicode_property_name_or_value: diag(1527, 1 /* Error */, "Expected_a_Unicode_property_name_or_value_1527", "Expected a Unicode property name or value."),
  Any_Unicode_property_that_would_possibly_match_more_than_a_single_character_is_only_available_when_the_Unicode_Sets_v_flag_is_set: diag(1528, 1 /* Error */, "Any_Unicode_property_that_would_possibly_match_more_than_a_single_character_is_only_available_when_t_1528", "Any Unicode property that would possibly match more than a single character is only available when the Unicode Sets (v) flag is set."),
  Unknown_Unicode_property_name_or_value: diag(1529, 1 /* Error */, "Unknown_Unicode_property_name_or_value_1529", "Unknown Unicode property name or value."),
  Unicode_property_value_expressions_are_only_available_when_the_Unicode_u_flag_or_the_Unicode_Sets_v_flag_is_set: diag(1530, 1 /* Error */, "Unicode_property_value_expressions_are_only_available_when_the_Unicode_u_flag_or_the_Unicode_Sets_v__1530", "Unicode property value expressions are only available when the Unicode (u) flag or the Unicode Sets (v) flag is set."),
  _0_must_be_followed_by_a_Unicode_property_value_expression_enclosed_in_braces: diag(1531, 1 /* Error */, "_0_must_be_followed_by_a_Unicode_property_value_expression_enclosed_in_braces_1531", "'\\{0}' must be followed by a Unicode property value expression enclosed in braces."),
  There_is_no_capturing_group_named_0_in_this_regular_expression: diag(1532, 1 /* Error */, "There_is_no_capturing_group_named_0_in_this_regular_expression_1532", "There is no capturing group named '{0}' in this regular expression."),
  This_backreference_refers_to_a_group_that_does_not_exist_There_are_only_0_capturing_groups_in_this_regular_expression: diag(1533, 1 /* Error */, "This_backreference_refers_to_a_group_that_does_not_exist_There_are_only_0_capturing_groups_in_this_r_1533", "This backreference refers to a group that does not exist. There are only {0} capturing groups in this regular expression."),
  This_backreference_refers_to_a_group_that_does_not_exist_There_are_no_capturing_groups_in_this_regular_expression: diag(1534, 1 /* Error */, "This_backreference_refers_to_a_group_that_does_not_exist_There_are_no_capturing_groups_in_this_regul_1534", "This backreference refers to a group that does not exist. There are no capturing groups in this regular expression."),
  This_character_cannot_be_escaped_in_a_regular_expression: diag(1535, 1 /* Error */, "This_character_cannot_be_escaped_in_a_regular_expression_1535", "This character cannot be escaped in a regular expression."),
  Octal_escape_sequences_and_backreferences_are_not_allowed_in_a_character_class_If_this_was_intended_as_an_escape_sequence_use_the_syntax_0_instead: diag(1536, 1 /* Error */, "Octal_escape_sequences_and_backreferences_are_not_allowed_in_a_character_class_If_this_was_intended__1536", "Octal escape sequences and backreferences are not allowed in a character class. If this was intended as an escape sequence, use the syntax '{0}' instead."),
  Decimal_escape_sequences_and_backreferences_are_not_allowed_in_a_character_class: diag(1537, 1 /* Error */, "Decimal_escape_sequences_and_backreferences_are_not_allowed_in_a_character_class_1537", "Decimal escape sequences and backreferences are not allowed in a character class."),
  Unicode_escape_sequences_are_only_available_when_the_Unicode_u_flag_or_the_Unicode_Sets_v_flag_is_set: diag(1538, 1 /* Error */, "Unicode_escape_sequences_are_only_available_when_the_Unicode_u_flag_or_the_Unicode_Sets_v_flag_is_se_1538", "Unicode escape sequences are only available when the Unicode (u) flag or the Unicode Sets (v) flag is set."),
  A_bigint_literal_cannot_be_used_as_a_property_name: diag(1539, 1 /* Error */, "A_bigint_literal_cannot_be_used_as_a_property_name_1539", "A 'bigint' literal cannot be used as a property name."),
  A_namespace_declaration_should_not_be_declared_using_the_module_keyword_Please_use_the_namespace_keyword_instead: diag(
    1540,
    2 /* Suggestion */,
    "A_namespace_declaration_should_not_be_declared_using_the_module_keyword_Please_use_the_namespace_key_1540",
    "A 'namespace' declaration should not be declared using the 'module' keyword. Please use the 'namespace' keyword instead.",
    /*reportsUnnecessary*/
    void 0,
    /*elidedInCompatabilityPyramid*/
    void 0,
    /*reportsDeprecated*/
    true
  ),
  Type_only_import_of_an_ECMAScript_module_from_a_CommonJS_module_must_have_a_resolution_mode_attribute: diag(1541, 1 /* Error */, "Type_only_import_of_an_ECMAScript_module_from_a_CommonJS_module_must_have_a_resolution_mode_attribut_1541", "Type-only import of an ECMAScript module from a CommonJS module must have a 'resolution-mode' attribute."),
  Type_import_of_an_ECMAScript_module_from_a_CommonJS_module_must_have_a_resolution_mode_attribute: diag(1542, 1 /* Error */, "Type_import_of_an_ECMAScript_module_from_a_CommonJS_module_must_have_a_resolution_mode_attribute_1542", "Type import of an ECMAScript module from a CommonJS module must have a 'resolution-mode' attribute."),
  Importing_a_JSON_file_into_an_ECMAScript_module_requires_a_type_Colon_json_import_attribute_when_module_is_set_to_0: diag(1543, 1 /* Error */, "Importing_a_JSON_file_into_an_ECMAScript_module_requires_a_type_Colon_json_import_attribute_when_mod_1543", `Importing a JSON file into an ECMAScript module requires a 'type: "json"' import attribute when 'module' is set to '{0}'.`),
  Named_imports_from_a_JSON_file_into_an_ECMAScript_module_are_not_allowed_when_module_is_set_to_0: diag(1544, 1 /* Error */, "Named_imports_from_a_JSON_file_into_an_ECMAScript_module_are_not_allowed_when_module_is_set_to_0_1544", "Named imports from a JSON file into an ECMAScript module are not allowed when 'module' is set to '{0}'."),
  The_types_of_0_are_incompatible_between_these_types: diag(2200, 1 /* Error */, "The_types_of_0_are_incompatible_between_these_types_2200", "The types of '{0}' are incompatible between these types."),
  The_types_returned_by_0_are_incompatible_between_these_types: diag(2201, 1 /* Error */, "The_types_returned_by_0_are_incompatible_between_these_types_2201", "The types returned by '{0}' are incompatible between these types."),
  Call_signature_return_types_0_and_1_are_incompatible: diag(
    2202,
    1 /* Error */,
    "Call_signature_return_types_0_and_1_are_incompatible_2202",
    "Call signature return types '{0}' and '{1}' are incompatible.",
    /*reportsUnnecessary*/
    void 0,
    /*elidedInCompatabilityPyramid*/
    true
  ),
  Construct_signature_return_types_0_and_1_are_incompatible: diag(
    2203,
    1 /* Error */,
    "Construct_signature_return_types_0_and_1_are_incompatible_2203",
    "Construct signature return types '{0}' and '{1}' are incompatible.",
    /*reportsUnnecessary*/
    void 0,
    /*elidedInCompatabilityPyramid*/
    true
  ),
  Call_signatures_with_no_arguments_have_incompatible_return_types_0_and_1: diag(
    2204,
    1 /* Error */,
    "Call_signatures_with_no_arguments_have_incompatible_return_types_0_and_1_2204",
    "Call signatures with no arguments have incompatible return types '{0}' and '{1}'.",
    /*reportsUnnecessary*/
    void 0,
    /*elidedInCompatabilityPyramid*/
    true
  ),
  Construct_signatures_with_no_arguments_have_incompatible_return_types_0_and_1: diag(
    2205,
    1 /* Error */,
    "Construct_signatures_with_no_arguments_have_incompatible_return_types_0_and_1_2205",
    "Construct signatures with no arguments have incompatible return types '{0}' and '{1}'.",
    /*reportsUnnecessary*/
    void 0,
    /*elidedInCompatabilityPyramid*/
    true
  ),
  The_type_modifier_cannot_be_used_on_a_named_import_when_import_type_is_used_on_its_import_statement: diag(2206, 1 /* Error */, "The_type_modifier_cannot_be_used_on_a_named_import_when_import_type_is_used_on_its_import_statement_2206", "The 'type' modifier cannot be used on a named import when 'import type' is used on its import statement."),
  The_type_modifier_cannot_be_used_on_a_named_export_when_export_type_is_used_on_its_export_statement: diag(2207, 1 /* Error */, "The_type_modifier_cannot_be_used_on_a_named_export_when_export_type_is_used_on_its_export_statement_2207", "The 'type' modifier cannot be used on a named export when 'export type' is used on its export statement."),
  This_type_parameter_might_need_an_extends_0_constraint: diag(2208, 1 /* Error */, "This_type_parameter_might_need_an_extends_0_constraint_2208", "This type parameter might need an `extends {0}` constraint."),
  The_project_root_is_ambiguous_but_is_required_to_resolve_export_map_entry_0_in_file_1_Supply_the_rootDir_compiler_option_to_disambiguate: diag(2209, 1 /* Error */, "The_project_root_is_ambiguous_but_is_required_to_resolve_export_map_entry_0_in_file_1_Supply_the_roo_2209", "The project root is ambiguous, but is required to resolve export map entry '{0}' in file '{1}'. Supply the `rootDir` compiler option to disambiguate."),
  The_project_root_is_ambiguous_but_is_required_to_resolve_import_map_entry_0_in_file_1_Supply_the_rootDir_compiler_option_to_disambiguate: diag(2210, 1 /* Error */, "The_project_root_is_ambiguous_but_is_required_to_resolve_import_map_entry_0_in_file_1_Supply_the_roo_2210", "The project root is ambiguous, but is required to resolve import map entry '{0}' in file '{1}'. Supply the `rootDir` compiler option to disambiguate."),
  Add_extends_constraint: diag(2211, 3 /* Message */, "Add_extends_constraint_2211", "Add `extends` constraint."),
  Add_extends_constraint_to_all_type_parameters: diag(2212, 3 /* Message */, "Add_extends_constraint_to_all_type_parameters_2212", "Add `extends` constraint to all type parameters"),
  Duplicate_identifier_0: diag(2300, 1 /* Error */, "Duplicate_identifier_0_2300", "Duplicate identifier '{0}'."),
  Initializer_of_instance_member_variable_0_cannot_reference_identifier_1_declared_in_the_constructor: diag(2301, 1 /* Error */, "Initializer_of_instance_member_variable_0_cannot_reference_identifier_1_declared_in_the_constructor_2301", "Initializer of instance member variable '{0}' cannot reference identifier '{1}' declared in the constructor."),
  Static_members_cannot_reference_class_type_parameters: diag(2302, 1 /* Error */, "Static_members_cannot_reference_class_type_parameters_2302", "Static members cannot reference class type parameters."),
  Circular_definition_of_import_alias_0: diag(2303, 1 /* Error */, "Circular_definition_of_import_alias_0_2303", "Circular definition of import alias '{0}'."),
  Cannot_find_name_0: diag(2304, 1 /* Error */, "Cannot_find_name_0_2304", "Cannot find name '{0}'."),
  Module_0_has_no_exported_member_1: diag(2305, 1 /* Error */, "Module_0_has_no_exported_member_1_2305", "Module '{0}' has no exported member '{1}'."),
  File_0_is_not_a_module: diag(2306, 1 /* Error */, "File_0_is_not_a_module_2306", "File '{0}' is not a module."),
  Cannot_find_module_0_or_its_corresponding_type_declarations: diag(2307, 1 /* Error */, "Cannot_find_module_0_or_its_corresponding_type_declarations_2307", "Cannot find module '{0}' or its corresponding type declarations."),
  Module_0_has_already_exported_a_member_named_1_Consider_explicitly_re_exporting_to_resolve_the_ambiguity: diag(2308, 1 /* Error */, "Module_0_has_already_exported_a_member_named_1_Consider_explicitly_re_exporting_to_resolve_the_ambig_2308", "Module {0} has already exported a member named '{1}'. Consider explicitly re-exporting to resolve the ambiguity."),
  An_export_assignment_cannot_be_used_in_a_module_with_other_exported_elements: diag(2309, 1 /* Error */, "An_export_assignment_cannot_be_used_in_a_module_with_other_exported_elements_2309", "An export assignment cannot be used in a module with other exported elements."),
  Type_0_recursively_references_itself_as_a_base_type: diag(2310, 1 /* Error */, "Type_0_recursively_references_itself_as_a_base_type_2310", "Type '{0}' recursively references itself as a base type."),
  Cannot_find_name_0_Did_you_mean_to_write_this_in_an_async_function: diag(2311, 1 /* Error */, "Cannot_find_name_0_Did_you_mean_to_write_this_in_an_async_function_2311", "Cannot find name '{0}'. Did you mean to write this in an async function?"),
  An_interface_can_only_extend_an_object_type_or_intersection_of_object_types_with_statically_known_members: diag(2312, 1 /* Error */, "An_interface_can_only_extend_an_object_type_or_intersection_of_object_types_with_statically_known_me_2312", "An interface can only extend an object type or intersection of object types with statically known members."),
  Type_parameter_0_has_a_circular_constraint: diag(2313, 1 /* Error */, "Type_parameter_0_has_a_circular_constraint_2313", "Type parameter '{0}' has a circular constraint."),
  Generic_type_0_requires_1_type_argument_s: diag(2314, 1 /* Error */, "Generic_type_0_requires_1_type_argument_s_2314", "Generic type '{0}' requires {1} type argument(s)."),
  Type_0_is_not_generic: diag(2315, 1 /* Error */, "Type_0_is_not_generic_2315", "Type '{0}' is not generic."),
  Global_type_0_must_be_a_class_or_interface_type: diag(2316, 1 /* Error */, "Global_type_0_must_be_a_class_or_interface_type_2316", "Global type '{0}' must be a class or interface type."),
  Global_type_0_must_have_1_type_parameter_s: diag(2317, 1 /* Error */, "Global_type_0_must_have_1_type_parameter_s_2317", "Global type '{0}' must have {1} type parameter(s)."),
  Cannot_find_global_type_0: diag(2318, 1 /* Error */, "Cannot_find_global_type_0_2318", "Cannot find global type '{0}'."),
  Named_property_0_of_types_1_and_2_are_not_identical: diag(2319, 1 /* Error */, "Named_property_0_of_types_1_and_2_are_not_identical_2319", "Named property '{0}' of types '{1}' and '{2}' are not identical."),
  Interface_0_cannot_simultaneously_extend_types_1_and_2: diag(2320, 1 /* Error */, "Interface_0_cannot_simultaneously_extend_types_1_and_2_2320", "Interface '{0}' cannot simultaneously extend types '{1}' and '{2}'."),
  Excessive_stack_depth_comparing_types_0_and_1: diag(2321, 1 /* Error */, "Excessive_stack_depth_comparing_types_0_and_1_2321", "Excessive stack depth comparing types '{0}' and '{1}'."),
  Type_0_is_not_assignable_to_type_1: diag(2322, 1 /* Error */, "Type_0_is_not_assignable_to_type_1_2322", "Type '{0}' is not assignable to type '{1}'."),
  Cannot_redeclare_exported_variable_0: diag(2323, 1 /* Error */, "Cannot_redeclare_exported_variable_0_2323", "Cannot redeclare exported variable '{0}'."),
  Property_0_is_missing_in_type_1: diag(2324, 1 /* Error */, "Property_0_is_missing_in_type_1_2324", "Property '{0}' is missing in type '{1}'."),
  Property_0_is_private_in_type_1_but_not_in_type_2: diag(2325, 1 /* Error */, "Property_0_is_private_in_type_1_but_not_in_type_2_2325", "Property '{0}' is private in type '{1}' but not in type '{2}'."),
  Types_of_property_0_are_incompatible: diag(2326, 1 /* Error */, "Types_of_property_0_are_incompatible_2326", "Types of property '{0}' are incompatible."),
  Property_0_is_optional_in_type_1_but_required_in_type_2: diag(2327, 1 /* Error */, "Property_0_is_optional_in_type_1_but_required_in_type_2_2327", "Property '{0}' is optional in type '{1}' but required in type '{2}'."),
  Types_of_parameters_0_and_1_are_incompatible: diag(2328, 1 /* Error */, "Types_of_parameters_0_and_1_are_incompatible_2328", "Types of parameters '{0}' and '{1}' are incompatible."),
  Index_signature_for_type_0_is_missing_in_type_1: diag(2329, 1 /* Error */, "Index_signature_for_type_0_is_missing_in_type_1_2329", "Index signature for type '{0}' is missing in type '{1}'."),
  _0_and_1_index_signatures_are_incompatible: diag(2330, 1 /* Error */, "_0_and_1_index_signatures_are_incompatible_2330", "'{0}' and '{1}' index signatures are incompatible."),
  this_cannot_be_referenced_in_a_module_or_namespace_body: diag(2331, 1 /* Error */, "this_cannot_be_referenced_in_a_module_or_namespace_body_2331", "'this' cannot be referenced in a module or namespace body."),
  this_cannot_be_referenced_in_current_location: diag(2332, 1 /* Error */, "this_cannot_be_referenced_in_current_location_2332", "'this' cannot be referenced in current location."),
  this_cannot_be_referenced_in_a_static_property_initializer: diag(2334, 1 /* Error */, "this_cannot_be_referenced_in_a_static_property_initializer_2334", "'this' cannot be referenced in a static property initializer."),
  super_can_only_be_referenced_in_a_derived_class: diag(2335, 1 /* Error */, "super_can_only_be_referenced_in_a_derived_class_2335", "'super' can only be referenced in a derived class."),
  super_cannot_be_referenced_in_constructor_arguments: diag(2336, 1 /* Error */, "super_cannot_be_referenced_in_constructor_arguments_2336", "'super' cannot be referenced in constructor arguments."),
  Super_calls_are_not_permitted_outside_constructors_or_in_nested_functions_inside_constructors: diag(2337, 1 /* Error */, "Super_calls_are_not_permitted_outside_constructors_or_in_nested_functions_inside_constructors_2337", "Super calls are not permitted outside constructors or in nested functions inside constructors."),
  super_property_access_is_permitted_only_in_a_constructor_member_function_or_member_accessor_of_a_derived_class: diag(2338, 1 /* Error */, "super_property_access_is_permitted_only_in_a_constructor_member_function_or_member_accessor_of_a_der_2338", "'super' property access is permitted only in a constructor, member function, or member accessor of a derived class."),
  Property_0_does_not_exist_on_type_1: diag(2339, 1 /* Error */, "Property_0_does_not_exist_on_type_1_2339", "Property '{0}' does not exist on type '{1}'."),
  Only_public_and_protected_methods_of_the_base_class_are_accessible_via_the_super_keyword: diag(2340, 1 /* Error */, "Only_public_and_protected_methods_of_the_base_class_are_accessible_via_the_super_keyword_2340", "Only public and protected methods of the base class are accessible via the 'super' keyword."),
  Property_0_is_private_and_only_accessible_within_class_1: diag(2341, 1 /* Error */, "Property_0_is_private_and_only_accessible_within_class_1_2341", "Property '{0}' is private and only accessible within class '{1}'."),
  This_syntax_requires_an_imported_helper_named_1_which_does_not_exist_in_0_Consider_upgrading_your_version_of_0: diag(2343, 1 /* Error */, "This_syntax_requires_an_imported_helper_named_1_which_does_not_exist_in_0_Consider_upgrading_your_ve_2343", "This syntax requires an imported helper named '{1}' which does not exist in '{0}'. Consider upgrading your version of '{0}'."),
  Type_0_does_not_satisfy_the_constraint_1: diag(2344, 1 /* Error */, "Type_0_does_not_satisfy_the_constraint_1_2344", "Type '{0}' does not satisfy the constraint '{1}'."),
  Argument_of_type_0_is_not_assignable_to_parameter_of_type_1: diag(2345, 1 /* Error */, "Argument_of_type_0_is_not_assignable_to_parameter_of_type_1_2345", "Argument of type '{0}' is not assignable to parameter of type '{1}'."),
  Untyped_function_calls_may_not_accept_type_arguments: diag(2347, 1 /* Error */, "Untyped_function_calls_may_not_accept_type_arguments_2347", "Untyped function calls may not accept type arguments."),
  Value_of_type_0_is_not_callable_Did_you_mean_to_include_new: diag(2348, 1 /* Error */, "Value_of_type_0_is_not_callable_Did_you_mean_to_include_new_2348", "Value of type '{0}' is not callable. Did you mean to include 'new'?"),
  This_expression_is_not_callable: diag(2349, 1 /* Error */, "This_expression_is_not_callable_2349", "This expression is not callable."),
  Only_a_void_function_can_be_called_with_the_new_keyword: diag(2350, 1 /* Error */, "Only_a_void_function_can_be_called_with_the_new_keyword_2350", "Only a void function can be called with the 'new' keyword."),
  This_expression_is_not_constructable: diag(2351, 1 /* Error */, "This_expression_is_not_constructable_2351", "This expression is not constructable."),
  Conversion_of_type_0_to_type_1_may_be_a_mistake_because_neither_type_sufficiently_overlaps_with_the_other_If_this_was_intentional_convert_the_expression_to_unknown_first: diag(2352, 1 /* Error */, "Conversion_of_type_0_to_type_1_may_be_a_mistake_because_neither_type_sufficiently_overlaps_with_the__2352", "Conversion of type '{0}' to type '{1}' may be a mistake because neither type sufficiently overlaps with the other. If this was intentional, convert the expression to 'unknown' first."),
  Object_literal_may_only_specify_known_properties_and_0_does_not_exist_in_type_1: diag(2353, 1 /* Error */, "Object_literal_may_only_specify_known_properties_and_0_does_not_exist_in_type_1_2353", "Object literal may only specify known properties, and '{0}' does not exist in type '{1}'."),
  This_syntax_requires_an_imported_helper_but_module_0_cannot_be_found: diag(2354, 1 /* Error */, "This_syntax_requires_an_imported_helper_but_module_0_cannot_be_found_2354", "This syntax requires an imported helper but module '{0}' cannot be found."),
  A_function_whose_declared_type_is_neither_undefined_void_nor_any_must_return_a_value: diag(2355, 1 /* Error */, "A_function_whose_declared_type_is_neither_undefined_void_nor_any_must_return_a_value_2355", "A function whose declared type is neither 'undefined', 'void', nor 'any' must return a value."),
  An_arithmetic_operand_must_be_of_type_any_number_bigint_or_an_enum_type: diag(2356, 1 /* Error */, "An_arithmetic_operand_must_be_of_type_any_number_bigint_or_an_enum_type_2356", "An arithmetic operand must be of type 'any', 'number', 'bigint' or an enum type."),
  The_operand_of_an_increment_or_decrement_operator_must_be_a_variable_or_a_property_access: diag(2357, 1 /* Error */, "The_operand_of_an_increment_or_decrement_operator_must_be_a_variable_or_a_property_access_2357", "The operand of an increment or decrement operator must be a variable or a property access."),
  The_left_hand_side_of_an_instanceof_expression_must_be_of_type_any_an_object_type_or_a_type_parameter: diag(2358, 1 /* Error */, "The_left_hand_side_of_an_instanceof_expression_must_be_of_type_any_an_object_type_or_a_type_paramete_2358", "The left-hand side of an 'instanceof' expression must be of type 'any', an object type or a type parameter."),
  The_right_hand_side_of_an_instanceof_expression_must_be_either_of_type_any_a_class_function_or_other_type_assignable_to_the_Function_interface_type_or_an_object_type_with_a_Symbol_hasInstance_method: diag(2359, 1 /* Error */, "The_right_hand_side_of_an_instanceof_expression_must_be_either_of_type_any_a_class_function_or_other_2359", "The right-hand side of an 'instanceof' expression must be either of type 'any', a class, function, or other type assignable to the 'Function' interface type, or an object type with a 'Symbol.hasInstance' method."),
  The_left_hand_side_of_an_arithmetic_operation_must_be_of_type_any_number_bigint_or_an_enum_type: diag(2362, 1 /* Error */, "The_left_hand_side_of_an_arithmetic_operation_must_be_of_type_any_number_bigint_or_an_enum_type_2362", "The left-hand side of an arithmetic operation must be of type 'any', 'number', 'bigint' or an enum type."),
  The_right_hand_side_of_an_arithmetic_operation_must_be_of_type_any_number_bigint_or_an_enum_type: diag(2363, 1 /* Error */, "The_right_hand_side_of_an_arithmetic_operation_must_be_of_type_any_number_bigint_or_an_enum_type_2363", "The right-hand side of an arithmetic operation must be of type 'any', 'number', 'bigint' or an enum type."),
  The_left_hand_side_of_an_assignment_expression_must_be_a_variable_or_a_property_access: diag(2364, 1 /* Error */, "The_left_hand_side_of_an_assignment_expression_must_be_a_variable_or_a_property_access_2364", "The left-hand side of an assignment expression must be a variable or a property access."),
  Operator_0_cannot_be_applied_to_types_1_and_2: diag(2365, 1 /* Error */, "Operator_0_cannot_be_applied_to_types_1_and_2_2365", "Operator '{0}' cannot be applied to types '{1}' and '{2}'."),
  Function_lacks_ending_return_statement_and_return_type_does_not_include_undefined: diag(2366, 1 /* Error */, "Function_lacks_ending_return_statement_and_return_type_does_not_include_undefined_2366", "Function lacks ending return statement and return type does not include 'undefined'."),
  This_comparison_appears_to_be_unintentional_because_the_types_0_and_1_have_no_overlap: diag(2367, 1 /* Error */, "This_comparison_appears_to_be_unintentional_because_the_types_0_and_1_have_no_overlap_2367", "This comparison appears to be unintentional because the types '{0}' and '{1}' have no overlap."),
  Type_parameter_name_cannot_be_0: diag(2368, 1 /* Error */, "Type_parameter_name_cannot_be_0_2368", "Type parameter name cannot be '{0}'."),
  A_parameter_property_is_only_allowed_in_a_constructor_implementation: diag(2369, 1 /* Error */, "A_parameter_property_is_only_allowed_in_a_constructor_implementation_2369", "A parameter property is only allowed in a constructor implementation."),
  A_rest_parameter_must_be_of_an_array_type: diag(2370, 1 /* Error */, "A_rest_parameter_must_be_of_an_array_type_2370", "A rest parameter must be of an array type."),
  A_parameter_initializer_is_only_allowed_in_a_function_or_constructor_implementation: diag(2371, 1 /* Error */, "A_parameter_initializer_is_only_allowed_in_a_function_or_constructor_implementation_2371", "A parameter initializer is only allowed in a function or constructor implementation."),
  Parameter_0_cannot_reference_itself: diag(2372, 1 /* Error */, "Parameter_0_cannot_reference_itself_2372", "Parameter '{0}' cannot reference itself."),
  Parameter_0_cannot_reference_identifier_1_declared_after_it: diag(2373, 1 /* Error */, "Parameter_0_cannot_reference_identifier_1_declared_after_it_2373", "Parameter '{0}' cannot reference identifier '{1}' declared after it."),
  Duplicate_index_signature_for_type_0: diag(2374, 1 /* Error */, "Duplicate_index_signature_for_type_0_2374", "Duplicate index signature for type '{0}'."),
  Type_0_is_not_assignable_to_type_1_with_exactOptionalPropertyTypes_Colon_true_Consider_adding_undefined_to_the_types_of_the_target_s_properties: diag(2375, 1 /* Error */, "Type_0_is_not_assignable_to_type_1_with_exactOptionalPropertyTypes_Colon_true_Consider_adding_undefi_2375", "Type '{0}' is not assignable to type '{1}' with 'exactOptionalPropertyTypes: true'. Consider adding 'undefined' to the types of the target's properties."),
  A_super_call_must_be_the_first_statement_in_the_constructor_to_refer_to_super_or_this_when_a_derived_class_contains_initialized_properties_parameter_properties_or_private_identifiers: diag(2376, 1 /* Error */, "A_super_call_must_be_the_first_statement_in_the_constructor_to_refer_to_super_or_this_when_a_derived_2376", "A 'super' call must be the first statement in the constructor to refer to 'super' or 'this' when a derived class contains initialized properties, parameter properties, or private identifiers."),
  Constructors_for_derived_classes_must_contain_a_super_call: diag(2377, 1 /* Error */, "Constructors_for_derived_classes_must_contain_a_super_call_2377", "Constructors for derived classes must contain a 'super' call."),
  A_get_accessor_must_return_a_value: diag(2378, 1 /* Error */, "A_get_accessor_must_return_a_value_2378", "A 'get' accessor must return a value."),
  Argument_of_type_0_is_not_assignable_to_parameter_of_type_1_with_exactOptionalPropertyTypes_Colon_true_Consider_adding_undefined_to_the_types_of_the_target_s_properties: diag(2379, 1 /* Error */, "Argument_of_type_0_is_not_assignable_to_parameter_of_type_1_with_exactOptionalPropertyTypes_Colon_tr_2379", "Argument of type '{0}' is not assignable to parameter of type '{1}' with 'exactOptionalPropertyTypes: true'. Consider adding 'undefined' to the types of the target's properties."),
  Overload_signatures_must_all_be_exported_or_non_exported: diag(2383, 1 /* Error */, "Overload_signatures_must_all_be_exported_or_non_exported_2383", "Overload signatures must all be exported or non-exported."),
  Overload_signatures_must_all_be_ambient_or_non_ambient: diag(2384, 1 /* Error */, "Overload_signatures_must_all_be_ambient_or_non_ambient_2384", "Overload signatures must all be ambient or non-ambient."),
  Overload_signatures_must_all_be_public_private_or_protected: diag(2385, 1 /* Error */, "Overload_signatures_must_all_be_public_private_or_protected_2385", "Overload signatures must all be public, private or protected."),
  Overload_signatures_must_all_be_optional_or_required: diag(2386, 1 /* Error */, "Overload_signatures_must_all_be_optional_or_required_2386", "Overload signatures must all be optional or required."),
  Function_overload_must_be_static: diag(2387, 1 /* Error */, "Function_overload_must_be_static_2387", "Function overload must be static."),
  Function_overload_must_not_be_static: diag(2388, 1 /* Error */, "Function_overload_must_not_be_static_2388", "Function overload must not be static."),
  Function_implementation_name_must_be_0: diag(2389, 1 /* Error */, "Function_implementation_name_must_be_0_2389", "Function implementation name must be '{0}'."),
  Constructor_implementation_is_missing: diag(2390, 1 /* Error */, "Constructor_implementation_is_missing_2390", "Constructor implementation is missing."),
  Function_implementation_is_missing_or_not_immediately_following_the_declaration: diag(2391, 1 /* Error */, "Function_implementation_is_missing_or_not_immediately_following_the_declaration_2391", "Function implementation is missing or not immediately following the declaration."),
  Multiple_constructor_implementations_are_not_allowed: diag(2392, 1 /* Error */, "Multiple_constructor_implementations_are_not_allowed_2392", "Multiple constructor implementations are not allowed."),
  Duplicate_function_implementation: diag(2393, 1 /* Error */, "Duplicate_function_implementation_2393", "Duplicate function implementation."),
  This_overload_signature_is_not_compatible_with_its_implementation_signature: diag(2394, 1 /* Error */, "This_overload_signature_is_not_compatible_with_its_implementation_signature_2394", "This overload signature is not compatible with its implementation signature."),
  Individual_declarations_in_merged_declaration_0_must_be_all_exported_or_all_local: diag(2395, 1 /* Error */, "Individual_declarations_in_merged_declaration_0_must_be_all_exported_or_all_local_2395", "Individual declarations in merged declaration '{0}' must be all exported or all local."),
  Duplicate_identifier_arguments_Compiler_uses_arguments_to_initialize_rest_parameters: diag(2396, 1 /* Error */, "Duplicate_identifier_arguments_Compiler_uses_arguments_to_initialize_rest_parameters_2396", "Duplicate identifier 'arguments'. Compiler uses 'arguments' to initialize rest parameters."),
  Declaration_name_conflicts_with_built_in_global_identifier_0: diag(2397, 1 /* Error */, "Declaration_name_conflicts_with_built_in_global_identifier_0_2397", "Declaration name conflicts with built-in global identifier '{0}'."),
  constructor_cannot_be_used_as_a_parameter_property_name: diag(2398, 1 /* Error */, "constructor_cannot_be_used_as_a_parameter_property_name_2398", "'constructor' cannot be used as a parameter property name."),
  Duplicate_identifier_this_Compiler_uses_variable_declaration_this_to_capture_this_reference: diag(2399, 1 /* Error */, "Duplicate_identifier_this_Compiler_uses_variable_declaration_this_to_capture_this_reference_2399", "Duplicate identifier '_this'. Compiler uses variable declaration '_this' to capture 'this' reference."),
  Expression_resolves_to_variable_declaration_this_that_compiler_uses_to_capture_this_reference: diag(2400, 1 /* Error */, "Expression_resolves_to_variable_declaration_this_that_compiler_uses_to_capture_this_reference_2400", "Expression resolves to variable declaration '_this' that compiler uses to capture 'this' reference."),
  A_super_call_must_be_a_root_level_statement_within_a_constructor_of_a_derived_class_that_contains_initialized_properties_parameter_properties_or_private_identifiers: diag(2401, 1 /* Error */, "A_super_call_must_be_a_root_level_statement_within_a_constructor_of_a_derived_class_that_contains_in_2401", "A 'super' call must be a root-level statement within a constructor of a derived class that contains initialized properties, parameter properties, or private identifiers."),
  Expression_resolves_to_super_that_compiler_uses_to_capture_base_class_reference: diag(2402, 1 /* Error */, "Expression_resolves_to_super_that_compiler_uses_to_capture_base_class_reference_2402", "Expression resolves to '_super' that compiler uses to capture base class reference."),
  Subsequent_variable_declarations_must_have_the_same_type_Variable_0_must_be_of_type_1_but_here_has_type_2: diag(2403, 1 /* Error */, "Subsequent_variable_declarations_must_have_the_same_type_Variable_0_must_be_of_type_1_but_here_has_t_2403", "Subsequent variable declarations must have the same type.  Variable '{0}' must be of type '{1}', but here has type '{2}'."),
  The_left_hand_side_of_a_for_in_statement_cannot_use_a_type_annotation: diag(2404, 1 /* Error */, "The_left_hand_side_of_a_for_in_statement_cannot_use_a_type_annotation_2404", "The left-hand side of a 'for...in' statement cannot use a type annotation."),
  The_left_hand_side_of_a_for_in_statement_must_be_of_type_string_or_any: diag(2405, 1 /* Error */, "The_left_hand_side_of_a_for_in_statement_must_be_of_type_string_or_any_2405", "The left-hand side of a 'for...in' statement must be of type 'string' or 'any'."),
  The_left_hand_side_of_a_for_in_statement_must_be_a_variable_or_a_property_access: diag(2406, 1 /* Error */, "The_left_hand_side_of_a_for_in_statement_must_be_a_variable_or_a_property_access_2406", "The left-hand side of a 'for...in' statement must be a variable or a property access."),
  The_right_hand_side_of_a_for_in_statement_must_be_of_type_any_an_object_type_or_a_type_parameter_but_here_has_type_0: diag(2407, 1 /* Error */, "The_right_hand_side_of_a_for_in_statement_must_be_of_type_any_an_object_type_or_a_type_parameter_but_2407", "The right-hand side of a 'for...in' statement must be of type 'any', an object type or a type parameter, but here has type '{0}'."),
  Setters_cannot_return_a_value: diag(2408, 1 /* Error */, "Setters_cannot_return_a_value_2408", "Setters cannot return a value."),
  Return_type_of_constructor_signature_must_be_assignable_to_the_instance_type_of_the_class: diag(2409, 1 /* Error */, "Return_type_of_constructor_signature_must_be_assignable_to_the_instance_type_of_the_class_2409", "Return type of constructor signature must be assignable to the instance type of the class."),
  The_with_statement_is_not_supported_All_symbols_in_a_with_block_will_have_type_any: diag(2410, 1 /* Error */, "The_with_statement_is_not_supported_All_symbols_in_a_with_block_will_have_type_any_2410", "The 'with' statement is not supported. All symbols in a 'with' block will have type 'any'."),
  Type_0_is_not_assignable_to_type_1_with_exactOptionalPropertyTypes_Colon_true_Consider_adding_undefined_to_the_type_of_the_target: diag(2412, 1 /* Error */, "Type_0_is_not_assignable_to_type_1_with_exactOptionalPropertyTypes_Colon_true_Consider_adding_undefi_2412", "Type '{0}' is not assignable to type '{1}' with 'exactOptionalPropertyTypes: true'. Consider adding 'undefined' to the type of the target."),
  Property_0_of_type_1_is_not_assignable_to_2_index_type_3: diag(2411, 1 /* Error */, "Property_0_of_type_1_is_not_assignable_to_2_index_type_3_2411", "Property '{0}' of type '{1}' is not assignable to '{2}' index type '{3}'."),
  _0_index_type_1_is_not_assignable_to_2_index_type_3: diag(2413, 1 /* Error */, "_0_index_type_1_is_not_assignable_to_2_index_type_3_2413", "'{0}' index type '{1}' is not assignable to '{2}' index type '{3}'."),
  Class_name_cannot_be_0: diag(2414, 1 /* Error */, "Class_name_cannot_be_0_2414", "Class name cannot be '{0}'."),
  Class_0_incorrectly_extends_base_class_1: diag(2415, 1 /* Error */, "Class_0_incorrectly_extends_base_class_1_2415", "Class '{0}' incorrectly extends base class '{1}'."),
  Property_0_in_type_1_is_not_assignable_to_the_same_property_in_base_type_2: diag(2416, 1 /* Error */, "Property_0_in_type_1_is_not_assignable_to_the_same_property_in_base_type_2_2416", "Property '{0}' in type '{1}' is not assignable to the same property in base type '{2}'."),
  Class_static_side_0_incorrectly_extends_base_class_static_side_1: diag(2417, 1 /* Error */, "Class_static_side_0_incorrectly_extends_base_class_static_side_1_2417", "Class static side '{0}' incorrectly extends base class static side '{1}'."),
  Type_of_computed_property_s_value_is_0_which_is_not_assignable_to_type_1: diag(2418, 1 /* Error */, "Type_of_computed_property_s_value_is_0_which_is_not_assignable_to_type_1_2418", "Type of computed property's value is '{0}', which is not assignable to type '{1}'."),
  Types_of_construct_signatures_are_incompatible: diag(2419, 1 /* Error */, "Types_of_construct_signatures_are_incompatible_2419", "Types of construct signatures are incompatible."),
  Class_0_incorrectly_implements_interface_1: diag(2420, 1 /* Error */, "Class_0_incorrectly_implements_interface_1_2420", "Class '{0}' incorrectly implements interface '{1}'."),
  A_class_can_only_implement_an_object_type_or_intersection_of_object_types_with_statically_known_members: diag(2422, 1 /* Error */, "A_class_can_only_implement_an_object_type_or_intersection_of_object_types_with_statically_known_memb_2422", "A class can only implement an object type or intersection of object types with statically known members."),
  Class_0_defines_instance_member_function_1_but_extended_class_2_defines_it_as_instance_member_accessor: diag(2423, 1 /* Error */, "Class_0_defines_instance_member_function_1_but_extended_class_2_defines_it_as_instance_member_access_2423", "Class '{0}' defines instance member function '{1}', but extended class '{2}' defines it as instance member accessor."),
  Class_0_defines_instance_member_property_1_but_extended_class_2_defines_it_as_instance_member_function: diag(2425, 1 /* Error */, "Class_0_defines_instance_member_property_1_but_extended_class_2_defines_it_as_instance_member_functi_2425", "Class '{0}' defines instance member property '{1}', but extended class '{2}' defines it as instance member function."),
  Class_0_defines_instance_member_accessor_1_but_extended_class_2_defines_it_as_instance_member_function: diag(2426, 1 /* Error */, "Class_0_defines_instance_member_accessor_1_but_extended_class_2_defines_it_as_instance_member_functi_2426", "Class '{0}' defines instance member accessor '{1}', but extended class '{2}' defines it as instance member function."),
  Interface_name_cannot_be_0: diag(2427, 1 /* Error */, "Interface_name_cannot_be_0_2427", "Interface name cannot be '{0}'."),
  All_declarations_of_0_must_have_identical_type_parameters: diag(2428, 1 /* Error */, "All_declarations_of_0_must_have_identical_type_parameters_2428", "All declarations of '{0}' must have identical type parameters."),
  Interface_0_incorrectly_extends_interface_1: diag(2430, 1 /* Error */, "Interface_0_incorrectly_extends_interface_1_2430", "Interface '{0}' incorrectly extends interface '{1}'."),
  Enum_name_cannot_be_0: diag(2431, 1 /* Error */, "Enum_name_cannot_be_0_2431", "Enum name cannot be '{0}'."),
  In_an_enum_with_multiple_declarations_only_one_declaration_can_omit_an_initializer_for_its_first_enum_element: diag(2432, 1 /* Error */, "In_an_enum_with_multiple_declarations_only_one_declaration_can_omit_an_initializer_for_its_first_enu_2432", "In an enum with multiple declarations, only one declaration can omit an initializer for its first enum element."),
  A_namespace_declaration_cannot_be_in_a_different_file_from_a_class_or_function_with_which_it_is_merged: diag(2433, 1 /* Error */, "A_namespace_declaration_cannot_be_in_a_different_file_from_a_class_or_function_with_which_it_is_merg_2433", "A namespace declaration cannot be in a different file from a class or function with which it is merged."),
  A_namespace_declaration_cannot_be_located_prior_to_a_class_or_function_with_which_it_is_merged: diag(2434, 1 /* Error */, "A_namespace_declaration_cannot_be_located_prior_to_a_class_or_function_with_which_it_is_merged_2434", "A namespace declaration cannot be located prior to a class or function with which it is merged."),
  Ambient_modules_cannot_be_nested_in_other_modules_or_namespaces: diag(2435, 1 /* Error */, "Ambient_modules_cannot_be_nested_in_other_modules_or_namespaces_2435", "Ambient modules cannot be nested in other modules or namespaces."),
  Ambient_module_declaration_cannot_specify_relative_module_name: diag(2436, 1 /* Error */, "Ambient_module_declaration_cannot_specify_relative_module_name_2436", "Ambient module declaration cannot specify relative module name."),
  Module_0_is_hidden_by_a_local_declaration_with_the_same_name: diag(2437, 1 /* Error */, "Module_0_is_hidden_by_a_local_declaration_with_the_same_name_2437", "Module '{0}' is hidden by a local declaration with the same name."),
  Import_name_cannot_be_0: diag(2438, 1 /* Error */, "Import_name_cannot_be_0_2438", "Import name cannot be '{0}'."),
  Import_or_export_declaration_in_an_ambient_module_declaration_cannot_reference_module_through_relative_module_name: diag(2439, 1 /* Error */, "Import_or_export_declaration_in_an_ambient_module_declaration_cannot_reference_module_through_relati_2439", "Import or export declaration in an ambient module declaration cannot reference module through relative module name."),
  Import_declaration_conflicts_with_local_declaration_of_0: diag(2440, 1 /* Error */, "Import_declaration_conflicts_with_local_declaration_of_0_2440", "Import declaration conflicts with local declaration of '{0}'."),
  Duplicate_identifier_0_Compiler_reserves_name_1_in_top_level_scope_of_a_module: diag(2441, 1 /* Error */, "Duplicate_identifier_0_Compiler_reserves_name_1_in_top_level_scope_of_a_module_2441", "Duplicate identifier '{0}'. Compiler reserves name '{1}' in top level scope of a module."),
  Types_have_separate_declarations_of_a_private_property_0: diag(2442, 1 /* Error */, "Types_have_separate_declarations_of_a_private_property_0_2442", "Types have separate declarations of a private property '{0}'."),
  Property_0_is_protected_but_type_1_is_not_a_class_derived_from_2: diag(2443, 1 /* Error */, "Property_0_is_protected_but_type_1_is_not_a_class_derived_from_2_2443", "Property '{0}' is protected but type '{1}' is not a class derived from '{2}'."),
  Property_0_is_protected_in_type_1_but_public_in_type_2: diag(2444, 1 /* Error */, "Property_0_is_protected_in_type_1_but_public_in_type_2_2444", "Property '{0}' is protected in type '{1}' but public in type '{2}'."),
  Property_0_is_protected_and_only_accessible_within_class_1_and_its_subclasses: diag(2445, 1 /* Error */, "Property_0_is_protected_and_only_accessible_within_class_1_and_its_subclasses_2445", "Property '{0}' is protected and only accessible within class '{1}' and its subclasses."),
  Property_0_is_protected_and_only_accessible_through_an_instance_of_class_1_This_is_an_instance_of_class_2: diag(2446, 1 /* Error */, "Property_0_is_protected_and_only_accessible_through_an_instance_of_class_1_This_is_an_instance_of_cl_2446", "Property '{0}' is protected and only accessible through an instance of class '{1}'. This is an instance of class '{2}'."),
  The_0_operator_is_not_allowed_for_boolean_types_Consider_using_1_instead: diag(2447, 1 /* Error */, "The_0_operator_is_not_allowed_for_boolean_types_Consider_using_1_instead_2447", "The '{0}' operator is not allowed for boolean types. Consider using '{1}' instead."),
  Block_scoped_variable_0_used_before_its_declaration: diag(2448, 1 /* Error */, "Block_scoped_variable_0_used_before_its_declaration_2448", "Block-scoped variable '{0}' used before its declaration."),
  Class_0_used_before_its_declaration: diag(2449, 1 /* Error */, "Class_0_used_before_its_declaration_2449", "Class '{0}' used before its declaration."),
  Enum_0_used_before_its_declaration: diag(2450, 1 /* Error */, "Enum_0_used_before_its_declaration_2450", "Enum '{0}' used before its declaration."),
  Cannot_redeclare_block_scoped_variable_0: diag(2451, 1 /* Error */, "Cannot_redeclare_block_scoped_variable_0_2451", "Cannot redeclare block-scoped variable '{0}'."),
  An_enum_member_cannot_have_a_numeric_name: diag(2452, 1 /* Error */, "An_enum_member_cannot_have_a_numeric_name_2452", "An enum member cannot have a numeric name."),
  Variable_0_is_used_before_being_assigned: diag(2454, 1 /* Error */, "Variable_0_is_used_before_being_assigned_2454", "Variable '{0}' is used before being assigned."),
  Type_alias_0_circularly_references_itself: diag(2456, 1 /* Error */, "Type_alias_0_circularly_references_itself_2456", "Type alias '{0}' circularly references itself."),
  Type_alias_name_cannot_be_0: diag(2457, 1 /* Error */, "Type_alias_name_cannot_be_0_2457", "Type alias name cannot be '{0}'."),
  An_AMD_module_cannot_have_multiple_name_assignments: diag(2458, 1 /* Error */, "An_AMD_module_cannot_have_multiple_name_assignments_2458", "An AMD module cannot have multiple name assignments."),
  Module_0_declares_1_locally_but_it_is_not_exported: diag(2459, 1 /* Error */, "Module_0_declares_1_locally_but_it_is_not_exported_2459", "Module '{0}' declares '{1}' locally, but it is not exported."),
  Module_0_declares_1_locally_but_it_is_exported_as_2: diag(2460, 1 /* Error */, "Module_0_declares_1_locally_but_it_is_exported_as_2_2460", "Module '{0}' declares '{1}' locally, but it is exported as '{2}'."),
  Type_0_is_not_an_array_type: diag(2461, 1 /* Error */, "Type_0_is_not_an_array_type_2461", "Type '{0}' is not an array type."),
  A_rest_element_must_be_last_in_a_destructuring_pattern: diag(2462, 1 /* Error */, "A_rest_element_must_be_last_in_a_destructuring_pattern_2462", "A rest element must be last in a destructuring pattern."),
  A_binding_pattern_parameter_cannot_be_optional_in_an_implementation_signature: diag(2463, 1 /* Error */, "A_binding_pattern_parameter_cannot_be_optional_in_an_implementation_signature_2463", "A binding pattern parameter cannot be optional in an implementation signature."),
  A_computed_property_name_must_be_of_type_string_number_symbol_or_any: diag(2464, 1 /* Error */, "A_computed_property_name_must_be_of_type_string_number_symbol_or_any_2464", "A computed property name must be of type 'string', 'number', 'symbol', or 'any'."),
  this_cannot_be_referenced_in_a_computed_property_name: diag(2465, 1 /* Error */, "this_cannot_be_referenced_in_a_computed_property_name_2465", "'this' cannot be referenced in a computed property name."),
  super_cannot_be_referenced_in_a_computed_property_name: diag(2466, 1 /* Error */, "super_cannot_be_referenced_in_a_computed_property_name_2466", "'super' cannot be referenced in a computed property name."),
  A_computed_property_name_cannot_reference_a_type_parameter_from_its_containing_type: diag(2467, 1 /* Error */, "A_computed_property_name_cannot_reference_a_type_parameter_from_its_containing_type_2467", "A computed property name cannot reference a type parameter from its containing type."),
  Cannot_find_global_value_0: diag(2468, 1 /* Error */, "Cannot_find_global_value_0_2468", "Cannot find global value '{0}'."),
  The_0_operator_cannot_be_applied_to_type_symbol: diag(2469, 1 /* Error */, "The_0_operator_cannot_be_applied_to_type_symbol_2469", "The '{0}' operator cannot be applied to type 'symbol'."),
  Spread_operator_in_new_expressions_is_only_available_when_targeting_ECMAScript_5_and_higher: diag(2472, 1 /* Error */, "Spread_operator_in_new_expressions_is_only_available_when_targeting_ECMAScript_5_and_higher_2472", "Spread operator in 'new' expressions is only available when targeting ECMAScript 5 and higher."),
  Enum_declarations_must_all_be_const_or_non_const: diag(2473, 1 /* Error */, "Enum_declarations_must_all_be_const_or_non_const_2473", "Enum declarations must all be const or non-const."),
  const_enum_member_initializers_must_be_constant_expressions: diag(2474, 1 /* Error */, "const_enum_member_initializers_must_be_constant_expressions_2474", "const enum member initializers must be constant expressions."),
  const_enums_can_only_be_used_in_property_or_index_access_expressions_or_the_right_hand_side_of_an_import_declaration_or_export_assignment_or_type_query: diag(2475, 1 /* Error */, "const_enums_can_only_be_used_in_property_or_index_access_expressions_or_the_right_hand_side_of_an_im_2475", "'const' enums can only be used in property or index access expressions or the right hand side of an import declaration or export assignment or type query."),
  A_const_enum_member_can_only_be_accessed_using_a_string_literal: diag(2476, 1 /* Error */, "A_const_enum_member_can_only_be_accessed_using_a_string_literal_2476", "A const enum member can only be accessed using a string literal."),
  const_enum_member_initializer_was_evaluated_to_a_non_finite_value: diag(2477, 1 /* Error */, "const_enum_member_initializer_was_evaluated_to_a_non_finite_value_2477", "'const' enum member initializer was evaluated to a non-finite value."),
  const_enum_member_initializer_was_evaluated_to_disallowed_value_NaN: diag(2478, 1 /* Error */, "const_enum_member_initializer_was_evaluated_to_disallowed_value_NaN_2478", "'const' enum member initializer was evaluated to disallowed value 'NaN'."),
  let_is_not_allowed_to_be_used_as_a_name_in_let_or_const_declarations: diag(2480, 1 /* Error */, "let_is_not_allowed_to_be_used_as_a_name_in_let_or_const_declarations_2480", "'let' is not allowed to be used as a name in 'let' or 'const' declarations."),
  Cannot_initialize_outer_scoped_variable_0_in_the_same_scope_as_block_scoped_declaration_1: diag(2481, 1 /* Error */, "Cannot_initialize_outer_scoped_variable_0_in_the_same_scope_as_block_scoped_declaration_1_2481", "Cannot initialize outer scoped variable '{0}' in the same scope as block scoped declaration '{1}'."),
  The_left_hand_side_of_a_for_of_statement_cannot_use_a_type_annotation: diag(2483, 1 /* Error */, "The_left_hand_side_of_a_for_of_statement_cannot_use_a_type_annotation_2483", "The left-hand side of a 'for...of' statement cannot use a type annotation."),
  Export_declaration_conflicts_with_exported_declaration_of_0: diag(2484, 1 /* Error */, "Export_declaration_conflicts_with_exported_declaration_of_0_2484", "Export declaration conflicts with exported declaration of '{0}'."),
  The_left_hand_side_of_a_for_of_statement_must_be_a_variable_or_a_property_access: diag(2487, 1 /* Error */, "The_left_hand_side_of_a_for_of_statement_must_be_a_variable_or_a_property_access_2487", "The left-hand side of a 'for...of' statement must be a variable or a property access."),
  Type_0_must_have_a_Symbol_iterator_method_that_returns_an_iterator: diag(2488, 1 /* Error */, "Type_0_must_have_a_Symbol_iterator_method_that_returns_an_iterator_2488", "Type '{0}' must have a '[Symbol.iterator]()' method that returns an iterator."),
  An_iterator_must_have_a_next_method: diag(2489, 1 /* Error */, "An_iterator_must_have_a_next_method_2489", "An iterator must have a 'next()' method."),
  The_type_returned_by_the_0_method_of_an_iterator_must_have_a_value_property: diag(2490, 1 /* Error */, "The_type_returned_by_the_0_method_of_an_iterator_must_have_a_value_property_2490", "The type returned by the '{0}()' method of an iterator must have a 'value' property."),
  The_left_hand_side_of_a_for_in_statement_cannot_be_a_destructuring_pattern: diag(2491, 1 /* Error */, "The_left_hand_side_of_a_for_in_statement_cannot_be_a_destructuring_pattern_2491", "The left-hand side of a 'for...in' statement cannot be a destructuring pattern."),
  Cannot_redeclare_identifier_0_in_catch_clause: diag(2492, 1 /* Error */, "Cannot_redeclare_identifier_0_in_catch_clause_2492", "Cannot redeclare identifier '{0}' in catch clause."),
  Tuple_type_0_of_length_1_has_no_element_at_index_2: diag(2493, 1 /* Error */, "Tuple_type_0_of_length_1_has_no_element_at_index_2_2493", "Tuple type '{0}' of length '{1}' has no element at index '{2}'."),
  Using_a_string_in_a_for_of_statement_is_only_supported_in_ECMAScript_5_and_higher: diag(2494, 1 /* Error */, "Using_a_string_in_a_for_of_statement_is_only_supported_in_ECMAScript_5_and_higher_2494", "Using a string in a 'for...of' statement is only supported in ECMAScript 5 and higher."),
  Type_0_is_not_an_array_type_or_a_string_type: diag(2495, 1 /* Error */, "Type_0_is_not_an_array_type_or_a_string_type_2495", "Type '{0}' is not an array type or a string type."),
  The_arguments_object_cannot_be_referenced_in_an_arrow_function_in_ES5_Consider_using_a_standard_function_expression: diag(2496, 1 /* Error */, "The_arguments_object_cannot_be_referenced_in_an_arrow_function_in_ES5_Consider_using_a_standard_func_2496", "The 'arguments' object cannot be referenced in an arrow function in ES5. Consider using a standard function expression."),
  This_module_can_only_be_referenced_with_ECMAScript_imports_Slashexports_by_turning_on_the_0_flag_and_referencing_its_default_export: diag(2497, 1 /* Error */, "This_module_can_only_be_referenced_with_ECMAScript_imports_Slashexports_by_turning_on_the_0_flag_and_2497", "This module can only be referenced with ECMAScript imports/exports by turning on the '{0}' flag and referencing its default export."),
  Module_0_uses_export_and_cannot_be_used_with_export_Asterisk: diag(2498, 1 /* Error */, "Module_0_uses_export_and_cannot_be_used_with_export_Asterisk_2498", "Module '{0}' uses 'export =' and cannot be used with 'export *'."),
  An_interface_can_only_extend_an_identifier_Slashqualified_name_with_optional_type_arguments: diag(2499, 1 /* Error */, "An_interface_can_only_extend_an_identifier_Slashqualified_name_with_optional_type_arguments_2499", "An interface can only extend an identifier/qualified-name with optional type arguments."),
  A_class_can_only_implement_an_identifier_Slashqualified_name_with_optional_type_arguments: diag(2500, 1 /* Error */, "A_class_can_only_implement_an_identifier_Slashqualified_name_with_optional_type_arguments_2500", "A class can only implement an identifier/qualified-name with optional type arguments."),
  A_rest_element_cannot_contain_a_binding_pattern: diag(2501, 1 /* Error */, "A_rest_element_cannot_contain_a_binding_pattern_2501", "A rest element cannot contain a binding pattern."),
  _0_is_referenced_directly_or_indirectly_in_its_own_type_annotation: diag(2502, 1 /* Error */, "_0_is_referenced_directly_or_indirectly_in_its_own_type_annotation_2502", "'{0}' is referenced directly or indirectly in its own type annotation."),
  Cannot_find_namespace_0: diag(2503, 1 /* Error */, "Cannot_find_namespace_0_2503", "Cannot find namespace '{0}'."),
  Type_0_must_have_a_Symbol_asyncIterator_method_that_returns_an_async_iterator: diag(2504, 1 /* Error */, "Type_0_must_have_a_Symbol_asyncIterator_method_that_returns_an_async_iterator_2504", "Type '{0}' must have a '[Symbol.asyncIterator]()' method that returns an async iterator."),
  A_generator_cannot_have_a_void_type_annotation: diag(2505, 1 /* Error */, "A_generator_cannot_have_a_void_type_annotation_2505", "A generator cannot have a 'void' type annotation."),
  _0_is_referenced_directly_or_indirectly_in_its_own_base_expression: diag(2506, 1 /* Error */, "_0_is_referenced_directly_or_indirectly_in_its_own_base_expression_2506", "'{0}' is referenced directly or indirectly in its own base expression."),
  Type_0_is_not_a_constructor_function_type: diag(2507, 1 /* Error */, "Type_0_is_not_a_constructor_function_type_2507", "Type '{0}' is not a constructor function type."),
  No_base_constructor_has_the_specified_number_of_type_arguments: diag(2508, 1 /* Error */, "No_base_constructor_has_the_specified_number_of_type_arguments_2508", "No base constructor has the specified number of type arguments."),
  Base_constructor_return_type_0_is_not_an_object_type_or_intersection_of_object_types_with_statically_known_members: diag(2509, 1 /* Error */, "Base_constructor_return_type_0_is_not_an_object_type_or_intersection_of_object_types_with_statically_2509", "Base constructor return type '{0}' is not an object type or intersection of object types with statically known members."),
  Base_constructors_must_all_have_the_same_return_type: diag(2510, 1 /* Error */, "Base_constructors_must_all_have_the_same_return_type_2510", "Base constructors must all have the same return type."),
  Cannot_create_an_instance_of_an_abstract_class: diag(2511, 1 /* Error */, "Cannot_create_an_instance_of_an_abstract_class_2511", "Cannot create an instance of an abstract class."),
  Overload_signatures_must_all_be_abstract_or_non_abstract: diag(2512, 1 /* Error */, "Overload_signatures_must_all_be_abstract_or_non_abstract_2512", "Overload signatures must all be abstract or non-abstract."),
  Abstract_method_0_in_class_1_cannot_be_accessed_via_super_expression: diag(2513, 1 /* Error */, "Abstract_method_0_in_class_1_cannot_be_accessed_via_super_expression_2513", "Abstract method '{0}' in class '{1}' cannot be accessed via super expression."),
  A_tuple_type_cannot_be_indexed_with_a_negative_value: diag(2514, 1 /* Error */, "A_tuple_type_cannot_be_indexed_with_a_negative_value_2514", "A tuple type cannot be indexed with a negative value."),
  Non_abstract_class_0_does_not_implement_inherited_abstract_member_1_from_class_2: diag(2515, 1 /* Error */, "Non_abstract_class_0_does_not_implement_inherited_abstract_member_1_from_class_2_2515", "Non-abstract class '{0}' does not implement inherited abstract member {1} from class '{2}'."),
  All_declarations_of_an_abstract_method_must_be_consecutive: diag(2516, 1 /* Error */, "All_declarations_of_an_abstract_method_must_be_consecutive_2516", "All declarations of an abstract method must be consecutive."),
  Cannot_assign_an_abstract_constructor_type_to_a_non_abstract_constructor_type: diag(2517, 1 /* Error */, "Cannot_assign_an_abstract_constructor_type_to_a_non_abstract_constructor_type_2517", "Cannot assign an abstract constructor type to a non-abstract constructor type."),
  A_this_based_type_guard_is_not_compatible_with_a_parameter_based_type_guard: diag(2518, 1 /* Error */, "A_this_based_type_guard_is_not_compatible_with_a_parameter_based_type_guard_2518", "A 'this'-based type guard is not compatible with a parameter-based type guard."),
  An_async_iterator_must_have_a_next_method: diag(2519, 1 /* Error */, "An_async_iterator_must_have_a_next_method_2519", "An async iterator must have a 'next()' method."),
  Duplicate_identifier_0_Compiler_uses_declaration_1_to_support_async_functions: diag(2520, 1 /* Error */, "Duplicate_identifier_0_Compiler_uses_declaration_1_to_support_async_functions_2520", "Duplicate identifier '{0}'. Compiler uses declaration '{1}' to support async functions."),
  The_arguments_object_cannot_be_referenced_in_an_async_function_or_method_in_ES5_Consider_using_a_standard_function_or_method: diag(2522, 1 /* Error */, "The_arguments_object_cannot_be_referenced_in_an_async_function_or_method_in_ES5_Consider_using_a_sta_2522", "The 'arguments' object cannot be referenced in an async function or method in ES5. Consider using a standard function or method."),
  yield_expressions_cannot_be_used_in_a_parameter_initializer: diag(2523, 1 /* Error */, "yield_expressions_cannot_be_used_in_a_parameter_initializer_2523", "'yield' expressions cannot be used in a parameter initializer."),
  await_expressions_cannot_be_used_in_a_parameter_initializer: diag(2524, 1 /* Error */, "await_expressions_cannot_be_used_in_a_parameter_initializer_2524", "'await' expressions cannot be used in a parameter initializer."),
  A_this_type_is_available_only_in_a_non_static_member_of_a_class_or_interface: diag(2526, 1 /* Error */, "A_this_type_is_available_only_in_a_non_static_member_of_a_class_or_interface_2526", "A 'this' type is available only in a non-static member of a class or interface."),
  The_inferred_type_of_0_references_an_inaccessible_1_type_A_type_annotation_is_necessary: diag(2527, 1 /* Error */, "The_inferred_type_of_0_references_an_inaccessible_1_type_A_type_annotation_is_necessary_2527", "The inferred type of '{0}' references an inaccessible '{1}' type. A type annotation is necessary."),
  A_module_cannot_have_multiple_default_exports: diag(2528, 1 /* Error */, "A_module_cannot_have_multiple_default_exports_2528", "A module cannot have multiple default exports."),
  Duplicate_identifier_0_Compiler_reserves_name_1_in_top_level_scope_of_a_module_containing_async_functions: diag(2529, 1 /* Error */, "Duplicate_identifier_0_Compiler_reserves_name_1_in_top_level_scope_of_a_module_containing_async_func_2529", "Duplicate identifier '{0}'. Compiler reserves name '{1}' in top level scope of a module containing async functions."),
  Property_0_is_incompatible_with_index_signature: diag(2530, 1 /* Error */, "Property_0_is_incompatible_with_index_signature_2530", "Property '{0}' is incompatible with index signature."),
  Object_is_possibly_null: diag(2531, 1 /* Error */, "Object_is_possibly_null_2531", "Object is possibly 'null'."),
  Object_is_possibly_undefined: diag(2532, 1 /* Error */, "Object_is_possibly_undefined_2532", "Object is possibly 'undefined'."),
  Object_is_possibly_null_or_undefined: diag(2533, 1 /* Error */, "Object_is_possibly_null_or_undefined_2533", "Object is possibly 'null' or 'undefined'."),
  A_function_returning_never_cannot_have_a_reachable_end_point: diag(2534, 1 /* Error */, "A_function_returning_never_cannot_have_a_reachable_end_point_2534", "A function returning 'never' cannot have a reachable end point."),
  Type_0_cannot_be_used_to_index_type_1: diag(2536, 1 /* Error */, "Type_0_cannot_be_used_to_index_type_1_2536", "Type '{0}' cannot be used to index type '{1}'."),
  Type_0_has_no_matching_index_signature_for_type_1: diag(2537, 1 /* Error */, "Type_0_has_no_matching_index_signature_for_type_1_2537", "Type '{0}' has no matching index signature for type '{1}'."),
  Type_0_cannot_be_used_as_an_index_type: diag(2538, 1 /* Error */, "Type_0_cannot_be_used_as_an_index_type_2538", "Type '{0}' cannot be used as an index type."),
  Cannot_assign_to_0_because_it_is_not_a_variable: diag(2539, 1 /* Error */, "Cannot_assign_to_0_because_it_is_not_a_variable_2539", "Cannot assign to '{0}' because it is not a variable."),
  Cannot_assign_to_0_because_it_is_a_read_only_property: diag(2540, 1 /* Error */, "Cannot_assign_to_0_because_it_is_a_read_only_property_2540", "Cannot assign to '{0}' because it is a read-only property."),
  Index_signature_in_type_0_only_permits_reading: diag(2542, 1 /* Error */, "Index_signature_in_type_0_only_permits_reading_2542", "Index signature in type '{0}' only permits reading."),
  Duplicate_identifier_newTarget_Compiler_uses_variable_declaration_newTarget_to_capture_new_target_meta_property_reference: diag(2543, 1 /* Error */, "Duplicate_identifier_newTarget_Compiler_uses_variable_declaration_newTarget_to_capture_new_target_me_2543", "Duplicate identifier '_newTarget'. Compiler uses variable declaration '_newTarget' to capture 'new.target' meta-property reference."),
  Expression_resolves_to_variable_declaration_newTarget_that_compiler_uses_to_capture_new_target_meta_property_reference: diag(2544, 1 /* Error */, "Expression_resolves_to_variable_declaration_newTarget_that_compiler_uses_to_capture_new_target_meta__2544", "Expression resolves to variable declaration '_newTarget' that compiler uses to capture 'new.target' meta-property reference."),
  A_mixin_class_must_have_a_constructor_with_a_single_rest_parameter_of_type_any: diag(2545, 1 /* Error */, "A_mixin_class_must_have_a_constructor_with_a_single_rest_parameter_of_type_any_2545", "A mixin class must have a constructor with a single rest parameter of type 'any[]'."),
  The_type_returned_by_the_0_method_of_an_async_iterator_must_be_a_promise_for_a_type_with_a_value_property: diag(2547, 1 /* Error */, "The_type_returned_by_the_0_method_of_an_async_iterator_must_be_a_promise_for_a_type_with_a_value_pro_2547", "The type returned by the '{0}()' method of an async iterator must be a promise for a type with a 'value' property."),
  Type_0_is_not_an_array_type_or_does_not_have_a_Symbol_iterator_method_that_returns_an_iterator: diag(2548, 1 /* Error */, "Type_0_is_not_an_array_type_or_does_not_have_a_Symbol_iterator_method_that_returns_an_iterator_2548", "Type '{0}' is not an array type or does not have a '[Symbol.iterator]()' method that returns an iterator."),
  Type_0_is_not_an_array_type_or_a_string_type_or_does_not_have_a_Symbol_iterator_method_that_returns_an_iterator: diag(2549, 1 /* Error */, "Type_0_is_not_an_array_type_or_a_string_type_or_does_not_have_a_Symbol_iterator_method_that_returns__2549", "Type '{0}' is not an array type or a string type or does not have a '[Symbol.iterator]()' method that returns an iterator."),
  Property_0_does_not_exist_on_type_1_Do_you_need_to_change_your_target_library_Try_changing_the_lib_compiler_option_to_2_or_later: diag(2550, 1 /* Error */, "Property_0_does_not_exist_on_type_1_Do_you_need_to_change_your_target_library_Try_changing_the_lib_c_2550", "Property '{0}' does not exist on type '{1}'. Do you need to change your target library? Try changing the 'lib' compiler option to '{2}' or later."),
  Property_0_does_not_exist_on_type_1_Did_you_mean_2: diag(2551, 1 /* Error */, "Property_0_does_not_exist_on_type_1_Did_you_mean_2_2551", "Property '{0}' does not exist on type '{1}'. Did you mean '{2}'?"),
  Cannot_find_name_0_Did_you_mean_1: diag(2552, 1 /* Error */, "Cannot_find_name_0_Did_you_mean_1_2552", "Cannot find name '{0}'. Did you mean '{1}'?"),
  Computed_values_are_not_permitted_in_an_enum_with_string_valued_members: diag(2553, 1 /* Error */, "Computed_values_are_not_permitted_in_an_enum_with_string_valued_members_2553", "Computed values are not permitted in an enum with string valued members."),
  Expected_0_arguments_but_got_1: diag(2554, 1 /* Error */, "Expected_0_arguments_but_got_1_2554", "Expected {0} arguments, but got {1}."),
  Expected_at_least_0_arguments_but_got_1: diag(2555, 1 /* Error */, "Expected_at_least_0_arguments_but_got_1_2555", "Expected at least {0} arguments, but got {1}."),
  A_spread_argument_must_either_have_a_tuple_type_or_be_passed_to_a_rest_parameter: diag(2556, 1 /* Error */, "A_spread_argument_must_either_have_a_tuple_type_or_be_passed_to_a_rest_parameter_2556", "A spread argument must either have a tuple type or be passed to a rest parameter."),
  Expected_0_type_arguments_but_got_1: diag(2558, 1 /* Error */, "Expected_0_type_arguments_but_got_1_2558", "Expected {0} type arguments, but got {1}."),
  Type_0_has_no_properties_in_common_with_type_1: diag(2559, 1 /* Error */, "Type_0_has_no_properties_in_common_with_type_1_2559", "Type '{0}' has no properties in common with type '{1}'."),
  Value_of_type_0_has_no_properties_in_common_with_type_1_Did_you_mean_to_call_it: diag(2560, 1 /* Error */, "Value_of_type_0_has_no_properties_in_common_with_type_1_Did_you_mean_to_call_it_2560", "Value of type '{0}' has no properties in common with type '{1}'. Did you mean to call it?"),
  Object_literal_may_only_specify_known_properties_but_0_does_not_exist_in_type_1_Did_you_mean_to_write_2: diag(2561, 1 /* Error */, "Object_literal_may_only_specify_known_properties_but_0_does_not_exist_in_type_1_Did_you_mean_to_writ_2561", "Object literal may only specify known properties, but '{0}' does not exist in type '{1}'. Did you mean to write '{2}'?"),
  Base_class_expressions_cannot_reference_class_type_parameters: diag(2562, 1 /* Error */, "Base_class_expressions_cannot_reference_class_type_parameters_2562", "Base class expressions cannot reference class type parameters."),
  The_containing_function_or_module_body_is_too_large_for_control_flow_analysis: diag(2563, 1 /* Error */, "The_containing_function_or_module_body_is_too_large_for_control_flow_analysis_2563", "The containing function or module body is too large for control flow analysis."),
  Property_0_has_no_initializer_and_is_not_definitely_assigned_in_the_constructor: diag(2564, 1 /* Error */, "Property_0_has_no_initializer_and_is_not_definitely_assigned_in_the_constructor_2564", "Property '{0}' has no initializer and is not definitely assigned in the constructor."),
  Property_0_is_used_before_being_assigned: diag(2565, 1 /* Error */, "Property_0_is_used_before_being_assigned_2565", "Property '{0}' is used before being assigned."),
  A_rest_element_cannot_have_a_property_name: diag(2566, 1 /* Error */, "A_rest_element_cannot_have_a_property_name_2566", "A rest element cannot have a property name."),
  Enum_declarations_can_only_merge_with_namespace_or_other_enum_declarations: diag(2567, 1 /* Error */, "Enum_declarations_can_only_merge_with_namespace_or_other_enum_declarations_2567", "Enum declarations can only merge with namespace or other enum declarations."),
  Property_0_may_not_exist_on_type_1_Did_you_mean_2: diag(2568, 1 /* Error */, "Property_0_may_not_exist_on_type_1_Did_you_mean_2_2568", "Property '{0}' may not exist on type '{1}'. Did you mean '{2}'?"),
  Could_not_find_name_0_Did_you_mean_1: diag(2570, 1 /* Error */, "Could_not_find_name_0_Did_you_mean_1_2570", "Could not find name '{0}'. Did you mean '{1}'?"),
  Object_is_of_type_unknown: diag(2571, 1 /* Error */, "Object_is_of_type_unknown_2571", "Object is of type 'unknown'."),
  A_rest_element_type_must_be_an_array_type: diag(2574, 1 /* Error */, "A_rest_element_type_must_be_an_array_type_2574", "A rest element type must be an array type."),
  No_overload_expects_0_arguments_but_overloads_do_exist_that_expect_either_1_or_2_arguments: diag(2575, 1 /* Error */, "No_overload_expects_0_arguments_but_overloads_do_exist_that_expect_either_1_or_2_arguments_2575", "No overload expects {0} arguments, but overloads do exist that expect either {1} or {2} arguments."),
  Property_0_does_not_exist_on_type_1_Did_you_mean_to_access_the_static_member_2_instead: diag(2576, 1 /* Error */, "Property_0_does_not_exist_on_type_1_Did_you_mean_to_access_the_static_member_2_instead_2576", "Property '{0}' does not exist on type '{1}'. Did you mean to access the static member '{2}' instead?"),
  Return_type_annotation_circularly_references_itself: diag(2577, 1 /* Error */, "Return_type_annotation_circularly_references_itself_2577", "Return type annotation circularly references itself."),
  Unused_ts_expect_error_directive: diag(2578, 1 /* Error */, "Unused_ts_expect_error_directive_2578", "Unused '@ts-expect-error' directive."),
  Cannot_find_name_0_Do_you_need_to_install_type_definitions_for_node_Try_npm_i_save_dev_types_Slashnode: diag(2580, 1 /* Error */, "Cannot_find_name_0_Do_you_need_to_install_type_definitions_for_node_Try_npm_i_save_dev_types_Slashno_2580", "Cannot find name '{0}'. Do you need to install type definitions for node? Try `npm i --save-dev @types/node`."),
  Cannot_find_name_0_Do_you_need_to_install_type_definitions_for_jQuery_Try_npm_i_save_dev_types_Slashjquery: diag(2581, 1 /* Error */, "Cannot_find_name_0_Do_you_need_to_install_type_definitions_for_jQuery_Try_npm_i_save_dev_types_Slash_2581", "Cannot find name '{0}'. Do you need to install type definitions for jQuery? Try `npm i --save-dev @types/jquery`."),
  Cannot_find_name_0_Do_you_need_to_install_type_definitions_for_a_test_runner_Try_npm_i_save_dev_types_Slashjest_or_npm_i_save_dev_types_Slashmocha: diag(2582, 1 /* Error */, "Cannot_find_name_0_Do_you_need_to_install_type_definitions_for_a_test_runner_Try_npm_i_save_dev_type_2582", "Cannot find name '{0}'. Do you need to install type definitions for a test runner? Try `npm i --save-dev @types/jest` or `npm i --save-dev @types/mocha`."),
  Cannot_find_name_0_Do_you_need_to_change_your_target_library_Try_changing_the_lib_compiler_option_to_1_or_later: diag(2583, 1 /* Error */, "Cannot_find_name_0_Do_you_need_to_change_your_target_library_Try_changing_the_lib_compiler_option_to_2583", "Cannot find name '{0}'. Do you need to change your target library? Try changing the 'lib' compiler option to '{1}' or later."),
  Cannot_find_name_0_Do_you_need_to_change_your_target_library_Try_changing_the_lib_compiler_option_to_include_dom: diag(2584, 1 /* Error */, "Cannot_find_name_0_Do_you_need_to_change_your_target_library_Try_changing_the_lib_compiler_option_to_2584", "Cannot find name '{0}'. Do you need to change your target library? Try changing the 'lib' compiler option to include 'dom'."),
  _0_only_refers_to_a_type_but_is_being_used_as_a_value_here_Do_you_need_to_change_your_target_library_Try_changing_the_lib_compiler_option_to_es2015_or_later: diag(2585, 1 /* Error */, "_0_only_refers_to_a_type_but_is_being_used_as_a_value_here_Do_you_need_to_change_your_target_library_2585", "'{0}' only refers to a type, but is being used as a value here. Do you need to change your target library? Try changing the 'lib' compiler option to es2015 or later."),
  Cannot_assign_to_0_because_it_is_a_constant: diag(2588, 1 /* Error */, "Cannot_assign_to_0_because_it_is_a_constant_2588", "Cannot assign to '{0}' because it is a constant."),
  Type_instantiation_is_excessively_deep_and_possibly_infinite: diag(2589, 1 /* Error */, "Type_instantiation_is_excessively_deep_and_possibly_infinite_2589", "Type instantiation is excessively deep and possibly infinite."),
  Expression_produces_a_union_type_that_is_too_complex_to_represent: diag(2590, 1 /* Error */, "Expression_produces_a_union_type_that_is_too_complex_to_represent_2590", "Expression produces a union type that is too complex to represent."),
  Cannot_find_name_0_Do_you_need_to_install_type_definitions_for_node_Try_npm_i_save_dev_types_Slashnode_and_then_add_node_to_the_types_field_in_your_tsconfig: diag(2591, 1 /* Error */, "Cannot_find_name_0_Do_you_need_to_install_type_definitions_for_node_Try_npm_i_save_dev_types_Slashno_2591", "Cannot find name '{0}'. Do you need to install type definitions for node? Try `npm i --save-dev @types/node` and then add 'node' to the types field in your tsconfig."),
  Cannot_find_name_0_Do_you_need_to_install_type_definitions_for_jQuery_Try_npm_i_save_dev_types_Slashjquery_and_then_add_jquery_to_the_types_field_in_your_tsconfig: diag(2592, 1 /* Error */, "Cannot_find_name_0_Do_you_need_to_install_type_definitions_for_jQuery_Try_npm_i_save_dev_types_Slash_2592", "Cannot find name '{0}'. Do you need to install type definitions for jQuery? Try `npm i --save-dev @types/jquery` and then add 'jquery' to the types field in your tsconfig."),
  Cannot_find_name_0_Do_you_need_to_install_type_definitions_for_a_test_runner_Try_npm_i_save_dev_types_Slashjest_or_npm_i_save_dev_types_Slashmocha_and_then_add_jest_or_mocha_to_the_types_field_in_your_tsconfig: diag(2593, 1 /* Error */, "Cannot_find_name_0_Do_you_need_to_install_type_definitions_for_a_test_runner_Try_npm_i_save_dev_type_2593", "Cannot find name '{0}'. Do you need to install type definitions for a test runner? Try `npm i --save-dev @types/jest` or `npm i --save-dev @types/mocha` and then add 'jest' or 'mocha' to the types field in your tsconfig."),
  This_module_is_declared_with_export_and_can_only_be_used_with_a_default_import_when_using_the_0_flag: diag(2594, 1 /* Error */, "This_module_is_declared_with_export_and_can_only_be_used_with_a_default_import_when_using_the_0_flag_2594", "This module is declared with 'export =', and can only be used with a default import when using the '{0}' flag."),
  _0_can_only_be_imported_by_using_a_default_import: diag(2595, 1 /* Error */, "_0_can_only_be_imported_by_using_a_default_import_2595", "'{0}' can only be imported by using a default import."),
  _0_can_only_be_imported_by_turning_on_the_esModuleInterop_flag_and_using_a_default_import: diag(2596, 1 /* Error */, "_0_can_only_be_imported_by_turning_on_the_esModuleInterop_flag_and_using_a_default_import_2596", "'{0}' can only be imported by turning on the 'esModuleInterop' flag and using a default import."),
  _0_can_only_be_imported_by_using_a_require_call_or_by_using_a_default_import: diag(2597, 1 /* Error */, "_0_can_only_be_imported_by_using_a_require_call_or_by_using_a_default_import_2597", "'{0}' can only be imported by using a 'require' call or by using a default import."),
  _0_can_only_be_imported_by_using_a_require_call_or_by_turning_on_the_esModuleInterop_flag_and_using_a_default_import: diag(2598, 1 /* Error */, "_0_can_only_be_imported_by_using_a_require_call_or_by_turning_on_the_esModuleInterop_flag_and_using__2598", "'{0}' can only be imported by using a 'require' call or by turning on the 'esModuleInterop' flag and using a default import."),
  JSX_element_implicitly_has_type_any_because_the_global_type_JSX_Element_does_not_exist: diag(2602, 1 /* Error */, "JSX_element_implicitly_has_type_any_because_the_global_type_JSX_Element_does_not_exist_2602", "JSX element implicitly has type 'any' because the global type 'JSX.Element' does not exist."),
  Property_0_in_type_1_is_not_assignable_to_type_2: diag(2603, 1 /* Error */, "Property_0_in_type_1_is_not_assignable_to_type_2_2603", "Property '{0}' in type '{1}' is not assignable to type '{2}'."),
  JSX_element_type_0_does_not_have_any_construct_or_call_signatures: diag(2604, 1 /* Error */, "JSX_element_type_0_does_not_have_any_construct_or_call_signatures_2604", "JSX element type '{0}' does not have any construct or call signatures."),
  Property_0_of_JSX_spread_attribute_is_not_assignable_to_target_property: diag(2606, 1 /* Error */, "Property_0_of_JSX_spread_attribute_is_not_assignable_to_target_property_2606", "Property '{0}' of JSX spread attribute is not assignable to target property."),
  JSX_element_class_does_not_support_attributes_because_it_does_not_have_a_0_property: diag(2607, 1 /* Error */, "JSX_element_class_does_not_support_attributes_because_it_does_not_have_a_0_property_2607", "JSX element class does not support attributes because it does not have a '{0}' property."),
  The_global_type_JSX_0_may_not_have_more_than_one_property: diag(2608, 1 /* Error */, "The_global_type_JSX_0_may_not_have_more_than_one_property_2608", "The global type 'JSX.{0}' may not have more than one property."),
  JSX_spread_child_must_be_an_array_type: diag(2609, 1 /* Error */, "JSX_spread_child_must_be_an_array_type_2609", "JSX spread child must be an array type."),
  _0_is_defined_as_an_accessor_in_class_1_but_is_overridden_here_in_2_as_an_instance_property: diag(2610, 1 /* Error */, "_0_is_defined_as_an_accessor_in_class_1_but_is_overridden_here_in_2_as_an_instance_property_2610", "'{0}' is defined as an accessor in class '{1}', but is overridden here in '{2}' as an instance property."),
  _0_is_defined_as_a_property_in_class_1_but_is_overridden_here_in_2_as_an_accessor: diag(2611, 1 /* Error */, "_0_is_defined_as_a_property_in_class_1_but_is_overridden_here_in_2_as_an_accessor_2611", "'{0}' is defined as a property in class '{1}', but is overridden here in '{2}' as an accessor."),
  Property_0_will_overwrite_the_base_property_in_1_If_this_is_intentional_add_an_initializer_Otherwise_add_a_declare_modifier_or_remove_the_redundant_declaration: diag(2612, 1 /* Error */, "Property_0_will_overwrite_the_base_property_in_1_If_this_is_intentional_add_an_initializer_Otherwise_2612", "Property '{0}' will overwrite the base property in '{1}'. If this is intentional, add an initializer. Otherwise, add a 'declare' modifier or remove the redundant declaration."),
  Module_0_has_no_default_export_Did_you_mean_to_use_import_1_from_0_instead: diag(2613, 1 /* Error */, "Module_0_has_no_default_export_Did_you_mean_to_use_import_1_from_0_instead_2613", "Module '{0}' has no default export. Did you mean to use 'import { {1} } from {0}' instead?"),
  Module_0_has_no_exported_member_1_Did_you_mean_to_use_import_1_from_0_instead: diag(2614, 1 /* Error */, "Module_0_has_no_exported_member_1_Did_you_mean_to_use_import_1_from_0_instead_2614", "Module '{0}' has no exported member '{1}'. Did you mean to use 'import {1} from {0}' instead?"),
  Type_of_property_0_circularly_references_itself_in_mapped_type_1: diag(2615, 1 /* Error */, "Type_of_property_0_circularly_references_itself_in_mapped_type_1_2615", "Type of property '{0}' circularly references itself in mapped type '{1}'."),
  _0_can_only_be_imported_by_using_import_1_require_2_or_a_default_import: diag(2616, 1 /* Error */, "_0_can_only_be_imported_by_using_import_1_require_2_or_a_default_import_2616", "'{0}' can only be imported by using 'import {1} = require({2})' or a default import."),
  _0_can_only_be_imported_by_using_import_1_require_2_or_by_turning_on_the_esModuleInterop_flag_and_using_a_default_import: diag(2617, 1 /* Error */, "_0_can_only_be_imported_by_using_import_1_require_2_or_by_turning_on_the_esModuleInterop_flag_and_us_2617", "'{0}' can only be imported by using 'import {1} = require({2})' or by turning on the 'esModuleInterop' flag and using a default import."),
  Source_has_0_element_s_but_target_requires_1: diag(2618, 1 /* Error */, "Source_has_0_element_s_but_target_requires_1_2618", "Source has {0} element(s) but target requires {1}."),
  Source_has_0_element_s_but_target_allows_only_1: diag(2619, 1 /* Error */, "Source_has_0_element_s_but_target_allows_only_1_2619", "Source has {0} element(s) but target allows only {1}."),
  Target_requires_0_element_s_but_source_may_have_fewer: diag(2620, 1 /* Error */, "Target_requires_0_element_s_but_source_may_have_fewer_2620", "Target requires {0} element(s) but source may have fewer."),
  Target_allows_only_0_element_s_but_source_may_have_more: diag(2621, 1 /* Error */, "Target_allows_only_0_element_s_but_source_may_have_more_2621", "Target allows only {0} element(s) but source may have more."),
  Source_provides_no_match_for_required_element_at_position_0_in_target: diag(2623, 1 /* Error */, "Source_provides_no_match_for_required_element_at_position_0_in_target_2623", "Source provides no match for required element at position {0} in target."),
  Source_provides_no_match_for_variadic_element_at_position_0_in_target: diag(2624, 1 /* Error */, "Source_provides_no_match_for_variadic_element_at_position_0_in_target_2624", "Source provides no match for variadic element at position {0} in target."),
  Variadic_element_at_position_0_in_source_does_not_match_element_at_position_1_in_target: diag(2625, 1 /* Error */, "Variadic_element_at_position_0_in_source_does_not_match_element_at_position_1_in_target_2625", "Variadic element at position {0} in source does not match element at position {1} in target."),
  Type_at_position_0_in_source_is_not_compatible_with_type_at_position_1_in_target: diag(2626, 1 /* Error */, "Type_at_position_0_in_source_is_not_compatible_with_type_at_position_1_in_target_2626", "Type at position {0} in source is not compatible with type at position {1} in target."),
  Type_at_positions_0_through_1_in_source_is_not_compatible_with_type_at_position_2_in_target: diag(2627, 1 /* Error */, "Type_at_positions_0_through_1_in_source_is_not_compatible_with_type_at_position_2_in_target_2627", "Type at positions {0} through {1} in source is not compatible with type at position {2} in target."),
  Cannot_assign_to_0_because_it_is_an_enum: diag(2628, 1 /* Error */, "Cannot_assign_to_0_because_it_is_an_enum_2628", "Cannot assign to '{0}' because it is an enum."),
  Cannot_assign_to_0_because_it_is_a_class: diag(2629, 1 /* Error */, "Cannot_assign_to_0_because_it_is_a_class_2629", "Cannot assign to '{0}' because it is a class."),
  Cannot_assign_to_0_because_it_is_a_function: diag(2630, 1 /* Error */, "Cannot_assign_to_0_because_it_is_a_function_2630", "Cannot assign to '{0}' because it is a function."),
  Cannot_assign_to_0_because_it_is_a_namespace: diag(2631, 1 /* Error */, "Cannot_assign_to_0_because_it_is_a_namespace_2631", "Cannot assign to '{0}' because it is a namespace."),
  Cannot_assign_to_0_because_it_is_an_import: diag(2632, 1 /* Error */, "Cannot_assign_to_0_because_it_is_an_import_2632", "Cannot assign to '{0}' because it is an import."),
  JSX_property_access_expressions_cannot_include_JSX_namespace_names: diag(2633, 1 /* Error */, "JSX_property_access_expressions_cannot_include_JSX_namespace_names_2633", "JSX property access expressions cannot include JSX namespace names"),
  _0_index_signatures_are_incompatible: diag(2634, 1 /* Error */, "_0_index_signatures_are_incompatible_2634", "'{0}' index signatures are incompatible."),
  Type_0_has_no_signatures_for_which_the_type_argument_list_is_applicable: diag(2635, 1 /* Error */, "Type_0_has_no_signatures_for_which_the_type_argument_list_is_applicable_2635", "Type '{0}' has no signatures for which the type argument list is applicable."),
  Type_0_is_not_assignable_to_type_1_as_implied_by_variance_annotation: diag(2636, 1 /* Error */, "Type_0_is_not_assignable_to_type_1_as_implied_by_variance_annotation_2636", "Type '{0}' is not assignable to type '{1}' as implied by variance annotation."),
  Variance_annotations_are_only_supported_in_type_aliases_for_object_function_constructor_and_mapped_types: diag(2637, 1 /* Error */, "Variance_annotations_are_only_supported_in_type_aliases_for_object_function_constructor_and_mapped_t_2637", "Variance annotations are only supported in type aliases for object, function, constructor, and mapped types."),
  Type_0_may_represent_a_primitive_value_which_is_not_permitted_as_the_right_operand_of_the_in_operator: diag(2638, 1 /* Error */, "Type_0_may_represent_a_primitive_value_which_is_not_permitted_as_the_right_operand_of_the_in_operato_2638", "Type '{0}' may represent a primitive value, which is not permitted as the right operand of the 'in' operator."),
  React_components_cannot_include_JSX_namespace_names: diag(2639, 1 /* Error */, "React_components_cannot_include_JSX_namespace_names_2639", "React components cannot include JSX namespace names"),
  Cannot_augment_module_0_with_value_exports_because_it_resolves_to_a_non_module_entity: diag(2649, 1 /* Error */, "Cannot_augment_module_0_with_value_exports_because_it_resolves_to_a_non_module_entity_2649", "Cannot augment module '{0}' with value exports because it resolves to a non-module entity."),
  Non_abstract_class_expression_is_missing_implementations_for_the_following_members_of_0_Colon_1_and_2_more: diag(2650, 1 /* Error */, "Non_abstract_class_expression_is_missing_implementations_for_the_following_members_of_0_Colon_1_and__2650", "Non-abstract class expression is missing implementations for the following members of '{0}': {1} and {2} more."),
  A_member_initializer_in_a_enum_declaration_cannot_reference_members_declared_after_it_including_members_defined_in_other_enums: diag(2651, 1 /* Error */, "A_member_initializer_in_a_enum_declaration_cannot_reference_members_declared_after_it_including_memb_2651", "A member initializer in a enum declaration cannot reference members declared after it, including members defined in other enums."),
  Merged_declaration_0_cannot_include_a_default_export_declaration_Consider_adding_a_separate_export_default_0_declaration_instead: diag(2652, 1 /* Error */, "Merged_declaration_0_cannot_include_a_default_export_declaration_Consider_adding_a_separate_export_d_2652", "Merged declaration '{0}' cannot include a default export declaration. Consider adding a separate 'export default {0}' declaration instead."),
  Non_abstract_class_expression_does_not_implement_inherited_abstract_member_0_from_class_1: diag(2653, 1 /* Error */, "Non_abstract_class_expression_does_not_implement_inherited_abstract_member_0_from_class_1_2653", "Non-abstract class expression does not implement inherited abstract member '{0}' from class '{1}'."),
  Non_abstract_class_0_is_missing_implementations_for_the_following_members_of_1_Colon_2: diag(2654, 1 /* Error */, "Non_abstract_class_0_is_missing_implementations_for_the_following_members_of_1_Colon_2_2654", "Non-abstract class '{0}' is missing implementations for the following members of '{1}': {2}."),
  Non_abstract_class_0_is_missing_implementations_for_the_following_members_of_1_Colon_2_and_3_more: diag(2655, 1 /* Error */, "Non_abstract_class_0_is_missing_implementations_for_the_following_members_of_1_Colon_2_and_3_more_2655", "Non-abstract class '{0}' is missing implementations for the following members of '{1}': {2} and {3} more."),
  Non_abstract_class_expression_is_missing_implementations_for_the_following_members_of_0_Colon_1: diag(2656, 1 /* Error */, "Non_abstract_class_expression_is_missing_implementations_for_the_following_members_of_0_Colon_1_2656", "Non-abstract class expression is missing implementations for the following members of '{0}': {1}."),
  JSX_expressions_must_have_one_parent_element: diag(2657, 1 /* Error */, "JSX_expressions_must_have_one_parent_element_2657", "JSX expressions must have one parent element."),
  Type_0_provides_no_match_for_the_signature_1: diag(2658, 1 /* Error */, "Type_0_provides_no_match_for_the_signature_1_2658", "Type '{0}' provides no match for the signature '{1}'."),
  super_is_only_allowed_in_members_of_object_literal_expressions_when_option_target_is_ES2015_or_higher: diag(2659, 1 /* Error */, "super_is_only_allowed_in_members_of_object_literal_expressions_when_option_target_is_ES2015_or_highe_2659", "'super' is only allowed in members of object literal expressions when option 'target' is 'ES2015' or higher."),
  super_can_only_be_referenced_in_members_of_derived_classes_or_object_literal_expressions: diag(2660, 1 /* Error */, "super_can_only_be_referenced_in_members_of_derived_classes_or_object_literal_expressions_2660", "'super' can only be referenced in members of derived classes or object literal expressions."),
  Cannot_export_0_Only_local_declarations_can_be_exported_from_a_module: diag(2661, 1 /* Error */, "Cannot_export_0_Only_local_declarations_can_be_exported_from_a_module_2661", "Cannot export '{0}'. Only local declarations can be exported from a module."),
  Cannot_find_name_0_Did_you_mean_the_static_member_1_0: diag(2662, 1 /* Error */, "Cannot_find_name_0_Did_you_mean_the_static_member_1_0_2662", "Cannot find name '{0}'. Did you mean the static member '{1}.{0}'?"),
  Cannot_find_name_0_Did_you_mean_the_instance_member_this_0: diag(2663, 1 /* Error */, "Cannot_find_name_0_Did_you_mean_the_instance_member_this_0_2663", "Cannot find name '{0}'. Did you mean the instance member 'this.{0}'?"),
  Invalid_module_name_in_augmentation_module_0_cannot_be_found: diag(2664, 1 /* Error */, "Invalid_module_name_in_augmentation_module_0_cannot_be_found_2664", "Invalid module name in augmentation, module '{0}' cannot be found."),
  Invalid_module_name_in_augmentation_Module_0_resolves_to_an_untyped_module_at_1_which_cannot_be_augmented: diag(2665, 1 /* Error */, "Invalid_module_name_in_augmentation_Module_0_resolves_to_an_untyped_module_at_1_which_cannot_be_augm_2665", "Invalid module name in augmentation. Module '{0}' resolves to an untyped module at '{1}', which cannot be augmented."),
  Exports_and_export_assignments_are_not_permitted_in_module_augmentations: diag(2666, 1 /* Error */, "Exports_and_export_assignments_are_not_permitted_in_module_augmentations_2666", "Exports and export assignments are not permitted in module augmentations."),
  Imports_are_not_permitted_in_module_augmentations_Consider_moving_them_to_the_enclosing_external_module: diag(2667, 1 /* Error */, "Imports_are_not_permitted_in_module_augmentations_Consider_moving_them_to_the_enclosing_external_mod_2667", "Imports are not permitted in module augmentations. Consider moving them to the enclosing external module."),
  export_modifier_cannot_be_applied_to_ambient_modules_and_module_augmentations_since_they_are_always_visible: diag(2668, 1 /* Error */, "export_modifier_cannot_be_applied_to_ambient_modules_and_module_augmentations_since_they_are_always__2668", "'export' modifier cannot be applied to ambient modules and module augmentations since they are always visible."),
  Augmentations_for_the_global_scope_can_only_be_directly_nested_in_external_modules_or_ambient_module_declarations: diag(2669, 1 /* Error */, "Augmentations_for_the_global_scope_can_only_be_directly_nested_in_external_modules_or_ambient_module_2669", "Augmentations for the global scope can only be directly nested in external modules or ambient module declarations."),
  Augmentations_for_the_global_scope_should_have_declare_modifier_unless_they_appear_in_already_ambient_context: diag(2670, 1 /* Error */, "Augmentations_for_the_global_scope_should_have_declare_modifier_unless_they_appear_in_already_ambien_2670", "Augmentations for the global scope should have 'declare' modifier unless they appear in already ambient context."),
  Cannot_augment_module_0_because_it_resolves_to_a_non_module_entity: diag(2671, 1 /* Error */, "Cannot_augment_module_0_because_it_resolves_to_a_non_module_entity_2671", "Cannot augment module '{0}' because it resolves to a non-module entity."),
  Cannot_assign_a_0_constructor_type_to_a_1_constructor_type: diag(2672, 1 /* Error */, "Cannot_assign_a_0_constructor_type_to_a_1_constructor_type_2672", "Cannot assign a '{0}' constructor type to a '{1}' constructor type."),
  Constructor_of_class_0_is_private_and_only_accessible_within_the_class_declaration: diag(2673, 1 /* Error */, "Constructor_of_class_0_is_private_and_only_accessible_within_the_class_declaration_2673", "Constructor of class '{0}' is private and only accessible within the class declaration."),
  Constructor_of_class_0_is_protected_and_only_accessible_within_the_class_declaration: diag(2674, 1 /* Error */, "Constructor_of_class_0_is_protected_and_only_accessible_within_the_class_declaration_2674", "Constructor of class '{0}' is protected and only accessible within the class declaration."),
  Cannot_extend_a_class_0_Class_constructor_is_marked_as_private: diag(2675, 1 /* Error */, "Cannot_extend_a_class_0_Class_constructor_is_marked_as_private_2675", "Cannot extend a class '{0}'. Class constructor is marked as private."),
  Accessors_must_both_be_abstract_or_non_abstract: diag(2676, 1 /* Error */, "Accessors_must_both_be_abstract_or_non_abstract_2676", "Accessors must both be abstract or non-abstract."),
  A_type_predicate_s_type_must_be_assignable_to_its_parameter_s_type: diag(2677, 1 /* Error */, "A_type_predicate_s_type_must_be_assignable_to_its_parameter_s_type_2677", "A type predicate's type must be assignable to its parameter's type."),
  Type_0_is_not_comparable_to_type_1: diag(2678, 1 /* Error */, "Type_0_is_not_comparable_to_type_1_2678", "Type '{0}' is not comparable to type '{1}'."),
  A_function_that_is_called_with_the_new_keyword_cannot_have_a_this_type_that_is_void: diag(2679, 1 /* Error */, "A_function_that_is_called_with_the_new_keyword_cannot_have_a_this_type_that_is_void_2679", "A function that is called with the 'new' keyword cannot have a 'this' type that is 'void'."),
  A_0_parameter_must_be_the_first_parameter: diag(2680, 1 /* Error */, "A_0_parameter_must_be_the_first_parameter_2680", "A '{0}' parameter must be the first parameter."),
  A_constructor_cannot_have_a_this_parameter: diag(2681, 1 /* Error */, "A_constructor_cannot_have_a_this_parameter_2681", "A constructor cannot have a 'this' parameter."),
  this_implicitly_has_type_any_because_it_does_not_have_a_type_annotation: diag(2683, 1 /* Error */, "this_implicitly_has_type_any_because_it_does_not_have_a_type_annotation_2683", "'this' implicitly has type 'any' because it does not have a type annotation."),
  The_this_context_of_type_0_is_not_assignable_to_method_s_this_of_type_1: diag(2684, 1 /* Error */, "The_this_context_of_type_0_is_not_assignable_to_method_s_this_of_type_1_2684", "The 'this' context of type '{0}' is not assignable to method's 'this' of type '{1}'."),
  The_this_types_of_each_signature_are_incompatible: diag(2685, 1 /* Error */, "The_this_types_of_each_signature_are_incompatible_2685", "The 'this' types of each signature are incompatible."),
  _0_refers_to_a_UMD_global_but_the_current_file_is_a_module_Consider_adding_an_import_instead: diag(2686, 1 /* Error */, "_0_refers_to_a_UMD_global_but_the_current_file_is_a_module_Consider_adding_an_import_instead_2686", "'{0}' refers to a UMD global, but the current file is a module. Consider adding an import instead."),
  All_declarations_of_0_must_have_identical_modifiers: diag(2687, 1 /* Error */, "All_declarations_of_0_must_have_identical_modifiers_2687", "All declarations of '{0}' must have identical modifiers."),
  Cannot_find_type_definition_file_for_0: diag(2688, 1 /* Error */, "Cannot_find_type_definition_file_for_0_2688", "Cannot find type definition file for '{0}'."),
  Cannot_extend_an_interface_0_Did_you_mean_implements: diag(2689, 1 /* Error */, "Cannot_extend_an_interface_0_Did_you_mean_implements_2689", "Cannot extend an interface '{0}'. Did you mean 'implements'?"),
  _0_only_refers_to_a_type_but_is_being_used_as_a_value_here_Did_you_mean_to_use_1_in_0: diag(2690, 1 /* Error */, "_0_only_refers_to_a_type_but_is_being_used_as_a_value_here_Did_you_mean_to_use_1_in_0_2690", "'{0}' only refers to a type, but is being used as a value here. Did you mean to use '{1} in {0}'?"),
  _0_is_a_primitive_but_1_is_a_wrapper_object_Prefer_using_0_when_possible: diag(2692, 1 /* Error */, "_0_is_a_primitive_but_1_is_a_wrapper_object_Prefer_using_0_when_possible_2692", "'{0}' is a primitive, but '{1}' is a wrapper object. Prefer using '{0}' when possible."),
  _0_only_refers_to_a_type_but_is_being_used_as_a_value_here: diag(2693, 1 /* Error */, "_0_only_refers_to_a_type_but_is_being_used_as_a_value_here_2693", "'{0}' only refers to a type, but is being used as a value here."),
  Namespace_0_has_no_exported_member_1: diag(2694, 1 /* Error */, "Namespace_0_has_no_exported_member_1_2694", "Namespace '{0}' has no exported member '{1}'."),
  Left_side_of_comma_operator_is_unused_and_has_no_side_effects: diag(
    2695,
    1 /* Error */,
    "Left_side_of_comma_operator_is_unused_and_has_no_side_effects_2695",
    "Left side of comma operator is unused and has no side effects.",
    /*reportsUnnecessary*/
    true
  ),
  The_Object_type_is_assignable_to_very_few_other_types_Did_you_mean_to_use_the_any_type_instead: diag(2696, 1 /* Error */, "The_Object_type_is_assignable_to_very_few_other_types_Did_you_mean_to_use_the_any_type_instead_2696", "The 'Object' type is assignable to very few other types. Did you mean to use the 'any' type instead?"),
  An_async_function_or_method_must_return_a_Promise_Make_sure_you_have_a_declaration_for_Promise_or_include_ES2015_in_your_lib_option: diag(2697, 1 /* Error */, "An_async_function_or_method_must_return_a_Promise_Make_sure_you_have_a_declaration_for_Promise_or_in_2697", "An async function or method must return a 'Promise'. Make sure you have a declaration for 'Promise' or include 'ES2015' in your '--lib' option."),
  Spread_types_may_only_be_created_from_object_types: diag(2698, 1 /* Error */, "Spread_types_may_only_be_created_from_object_types_2698", "Spread types may only be created from object types."),
  Static_property_0_conflicts_with_built_in_property_Function_0_of_constructor_function_1: diag(2699, 1 /* Error */, "Static_property_0_conflicts_with_built_in_property_Function_0_of_constructor_function_1_2699", "Static property '{0}' conflicts with built-in property 'Function.{0}' of constructor function '{1}'."),
  Rest_types_may_only_be_created_from_object_types: diag(2700, 1 /* Error */, "Rest_types_may_only_be_created_from_object_types_2700", "Rest types may only be created from object types."),
  The_target_of_an_object_rest_assignment_must_be_a_variable_or_a_property_access: diag(2701, 1 /* Error */, "The_target_of_an_object_rest_assignment_must_be_a_variable_or_a_property_access_2701", "The target of an object rest assignment must be a variable or a property access."),
  _0_only_refers_to_a_type_but_is_being_used_as_a_namespace_here: diag(2702, 1 /* Error */, "_0_only_refers_to_a_type_but_is_being_used_as_a_namespace_here_2702", "'{0}' only refers to a type, but is being used as a namespace here."),
  The_operand_of_a_delete_operator_must_be_a_property_reference: diag(2703, 1 /* Error */, "The_operand_of_a_delete_operator_must_be_a_property_reference_2703", "The operand of a 'delete' operator must be a property reference."),
  The_operand_of_a_delete_operator_cannot_be_a_read_only_property: diag(2704, 1 /* Error */, "The_operand_of_a_delete_operator_cannot_be_a_read_only_property_2704", "The operand of a 'delete' operator cannot be a read-only property."),
  An_async_function_or_method_in_ES5_requires_the_Promise_constructor_Make_sure_you_have_a_declaration_for_the_Promise_constructor_or_include_ES2015_in_your_lib_option: diag(2705, 1 /* Error */, "An_async_function_or_method_in_ES5_requires_the_Promise_constructor_Make_sure_you_have_a_declaration_2705", "An async function or method in ES5 requires the 'Promise' constructor.  Make sure you have a declaration for the 'Promise' constructor or include 'ES2015' in your '--lib' option."),
  Required_type_parameters_may_not_follow_optional_type_parameters: diag(2706, 1 /* Error */, "Required_type_parameters_may_not_follow_optional_type_parameters_2706", "Required type parameters may not follow optional type parameters."),
  Generic_type_0_requires_between_1_and_2_type_arguments: diag(2707, 1 /* Error */, "Generic_type_0_requires_between_1_and_2_type_arguments_2707", "Generic type '{0}' requires between {1} and {2} type arguments."),
  Cannot_use_namespace_0_as_a_value: diag(2708, 1 /* Error */, "Cannot_use_namespace_0_as_a_value_2708", "Cannot use namespace '{0}' as a value."),
  Cannot_use_namespace_0_as_a_type: diag(2709, 1 /* Error */, "Cannot_use_namespace_0_as_a_type_2709", "Cannot use namespace '{0}' as a type."),
  _0_are_specified_twice_The_attribute_named_0_will_be_overwritten: diag(2710, 1 /* Error */, "_0_are_specified_twice_The_attribute_named_0_will_be_overwritten_2710", "'{0}' are specified twice. The attribute named '{0}' will be overwritten."),
  A_dynamic_import_call_returns_a_Promise_Make_sure_you_have_a_declaration_for_Promise_or_include_ES2015_in_your_lib_option: diag(2711, 1 /* Error */, "A_dynamic_import_call_returns_a_Promise_Make_sure_you_have_a_declaration_for_Promise_or_include_ES20_2711", "A dynamic import call returns a 'Promise'. Make sure you have a declaration for 'Promise' or include 'ES2015' in your '--lib' option."),
  A_dynamic_import_call_in_ES5_requires_the_Promise_constructor_Make_sure_you_have_a_declaration_for_the_Promise_constructor_or_include_ES2015_in_your_lib_option: diag(2712, 1 /* Error */, "A_dynamic_import_call_in_ES5_requires_the_Promise_constructor_Make_sure_you_have_a_declaration_for_t_2712", "A dynamic import call in ES5 requires the 'Promise' constructor.  Make sure you have a declaration for the 'Promise' constructor or include 'ES2015' in your '--lib' option."),
  Cannot_access_0_1_because_0_is_a_type_but_not_a_namespace_Did_you_mean_to_retrieve_the_type_of_the_property_1_in_0_with_0_1: diag(2713, 1 /* Error */, "Cannot_access_0_1_because_0_is_a_type_but_not_a_namespace_Did_you_mean_to_retrieve_the_type_of_the_p_2713", `Cannot access '{0}.{1}' because '{0}' is a type, but not a namespace. Did you mean to retrieve the type of the property '{1}' in '{0}' with '{0}["{1}"]'?`),
  The_expression_of_an_export_assignment_must_be_an_identifier_or_qualified_name_in_an_ambient_context: diag(2714, 1 /* Error */, "The_expression_of_an_export_assignment_must_be_an_identifier_or_qualified_name_in_an_ambient_context_2714", "The expression of an export assignment must be an identifier or qualified name in an ambient context."),
  Abstract_property_0_in_class_1_cannot_be_accessed_in_the_constructor: diag(2715, 1 /* Error */, "Abstract_property_0_in_class_1_cannot_be_accessed_in_the_constructor_2715", "Abstract property '{0}' in class '{1}' cannot be accessed in the constructor."),
  Type_parameter_0_has_a_circular_default: diag(2716, 1 /* Error */, "Type_parameter_0_has_a_circular_default_2716", "Type parameter '{0}' has a circular default."),
  Subsequent_property_declarations_must_have_the_same_type_Property_0_must_be_of_type_1_but_here_has_type_2: diag(2717, 1 /* Error */, "Subsequent_property_declarations_must_have_the_same_type_Property_0_must_be_of_type_1_but_here_has_t_2717", "Subsequent property declarations must have the same type.  Property '{0}' must be of type '{1}', but here has type '{2}'."),
  Duplicate_property_0: diag(2718, 1 /* Error */, "Duplicate_property_0_2718", "Duplicate property '{0}'."),
  Type_0_is_not_assignable_to_type_1_Two_different_types_with_this_name_exist_but_they_are_unrelated: diag(2719, 1 /* Error */, "Type_0_is_not_assignable_to_type_1_Two_different_types_with_this_name_exist_but_they_are_unrelated_2719", "Type '{0}' is not assignable to type '{1}'. Two different types with this name exist, but they are unrelated."),
  Class_0_incorrectly_implements_class_1_Did_you_mean_to_extend_1_and_inherit_its_members_as_a_subclass: diag(2720, 1 /* Error */, "Class_0_incorrectly_implements_class_1_Did_you_mean_to_extend_1_and_inherit_its_members_as_a_subclas_2720", "Class '{0}' incorrectly implements class '{1}'. Did you mean to extend '{1}' and inherit its members as a subclass?"),
  Cannot_invoke_an_object_which_is_possibly_null: diag(2721, 1 /* Error */, "Cannot_invoke_an_object_which_is_possibly_null_2721", "Cannot invoke an object which is possibly 'null'."),
  Cannot_invoke_an_object_which_is_possibly_undefined: diag(2722, 1 /* Error */, "Cannot_invoke_an_object_which_is_possibly_undefined_2722", "Cannot invoke an object which is possibly 'undefined'."),
  Cannot_invoke_an_object_which_is_possibly_null_or_undefined: diag(2723, 1 /* Error */, "Cannot_invoke_an_object_which_is_possibly_null_or_undefined_2723", "Cannot invoke an object which is possibly 'null' or 'undefined'."),
  _0_has_no_exported_member_named_1_Did_you_mean_2: diag(2724, 1 /* Error */, "_0_has_no_exported_member_named_1_Did_you_mean_2_2724", "'{0}' has no exported member named '{1}'. Did you mean '{2}'?"),
  Class_name_cannot_be_Object_when_targeting_ES5_with_module_0: diag(2725, 1 /* Error */, "Class_name_cannot_be_Object_when_targeting_ES5_with_module_0_2725", "Class name cannot be 'Object' when targeting ES5 with module {0}."),
  Cannot_find_lib_definition_for_0: diag(2726, 1 /* Error */, "Cannot_find_lib_definition_for_0_2726", "Cannot find lib definition for '{0}'."),
  Cannot_find_lib_definition_for_0_Did_you_mean_1: diag(2727, 1 /* Error */, "Cannot_find_lib_definition_for_0_Did_you_mean_1_2727", "Cannot find lib definition for '{0}'. Did you mean '{1}'?"),
  _0_is_declared_here: diag(2728, 3 /* Message */, "_0_is_declared_here_2728", "'{0}' is declared here."),
  Property_0_is_used_before_its_initialization: diag(2729, 1 /* Error */, "Property_0_is_used_before_its_initialization_2729", "Property '{0}' is used before its initialization."),
  An_arrow_function_cannot_have_a_this_parameter: diag(2730, 1 /* Error */, "An_arrow_function_cannot_have_a_this_parameter_2730", "An arrow function cannot have a 'this' parameter."),
  Implicit_conversion_of_a_symbol_to_a_string_will_fail_at_runtime_Consider_wrapping_this_expression_in_String: diag(2731, 1 /* Error */, "Implicit_conversion_of_a_symbol_to_a_string_will_fail_at_runtime_Consider_wrapping_this_expression_i_2731", "Implicit conversion of a 'symbol' to a 'string' will fail at runtime. Consider wrapping this expression in 'String(...)'."),
  Cannot_find_module_0_Consider_using_resolveJsonModule_to_import_module_with_json_extension: diag(2732, 1 /* Error */, "Cannot_find_module_0_Consider_using_resolveJsonModule_to_import_module_with_json_extension_2732", "Cannot find module '{0}'. Consider using '--resolveJsonModule' to import module with '.json' extension."),
  Property_0_was_also_declared_here: diag(2733, 1 /* Error */, "Property_0_was_also_declared_here_2733", "Property '{0}' was also declared here."),
  Are_you_missing_a_semicolon: diag(2734, 1 /* Error */, "Are_you_missing_a_semicolon_2734", "Are you missing a semicolon?"),
  Did_you_mean_for_0_to_be_constrained_to_type_new_args_Colon_any_1: diag(2735, 1 /* Error */, "Did_you_mean_for_0_to_be_constrained_to_type_new_args_Colon_any_1_2735", "Did you mean for '{0}' to be constrained to type 'new (...args: any[]) => {1}'?"),
  Operator_0_cannot_be_applied_to_type_1: diag(2736, 1 /* Error */, "Operator_0_cannot_be_applied_to_type_1_2736", "Operator '{0}' cannot be applied to type '{1}'."),
  BigInt_literals_are_not_available_when_targeting_lower_than_ES2020: diag(2737, 1 /* Error */, "BigInt_literals_are_not_available_when_targeting_lower_than_ES2020_2737", "BigInt literals are not available when targeting lower than ES2020."),
  An_outer_value_of_this_is_shadowed_by_this_container: diag(2738, 3 /* Message */, "An_outer_value_of_this_is_shadowed_by_this_container_2738", "An outer value of 'this' is shadowed by this container."),
  Type_0_is_missing_the_following_properties_from_type_1_Colon_2: diag(2739, 1 /* Error */, "Type_0_is_missing_the_following_properties_from_type_1_Colon_2_2739", "Type '{0}' is missing the following properties from type '{1}': {2}"),
  Type_0_is_missing_the_following_properties_from_type_1_Colon_2_and_3_more: diag(2740, 1 /* Error */, "Type_0_is_missing_the_following_properties_from_type_1_Colon_2_and_3_more_2740", "Type '{0}' is missing the following properties from type '{1}': {2}, and {3} more."),
  Property_0_is_missing_in_type_1_but_required_in_type_2: diag(2741, 1 /* Error */, "Property_0_is_missing_in_type_1_but_required_in_type_2_2741", "Property '{0}' is missing in type '{1}' but required in type '{2}'."),
  The_inferred_type_of_0_cannot_be_named_without_a_reference_to_1_This_is_likely_not_portable_A_type_annotation_is_necessary: diag(2742, 1 /* Error */, "The_inferred_type_of_0_cannot_be_named_without_a_reference_to_1_This_is_likely_not_portable_A_type_a_2742", "The inferred type of '{0}' cannot be named without a reference to '{1}'. This is likely not portable. A type annotation is necessary."),
  No_overload_expects_0_type_arguments_but_overloads_do_exist_that_expect_either_1_or_2_type_arguments: diag(2743, 1 /* Error */, "No_overload_expects_0_type_arguments_but_overloads_do_exist_that_expect_either_1_or_2_type_arguments_2743", "No overload expects {0} type arguments, but overloads do exist that expect either {1} or {2} type arguments."),
  Type_parameter_defaults_can_only_reference_previously_declared_type_parameters: diag(2744, 1 /* Error */, "Type_parameter_defaults_can_only_reference_previously_declared_type_parameters_2744", "Type parameter defaults can only reference previously declared type parameters."),
  This_JSX_tag_s_0_prop_expects_type_1_which_requires_multiple_children_but_only_a_single_child_was_provided: diag(2745, 1 /* Error */, "This_JSX_tag_s_0_prop_expects_type_1_which_requires_multiple_children_but_only_a_single_child_was_pr_2745", "This JSX tag's '{0}' prop expects type '{1}' which requires multiple children, but only a single child was provided."),
  This_JSX_tag_s_0_prop_expects_a_single_child_of_type_1_but_multiple_children_were_provided: diag(2746, 1 /* Error */, "This_JSX_tag_s_0_prop_expects_a_single_child_of_type_1_but_multiple_children_were_provided_2746", "This JSX tag's '{0}' prop expects a single child of type '{1}', but multiple children were provided."),
  _0_components_don_t_accept_text_as_child_elements_Text_in_JSX_has_the_type_string_but_the_expected_type_of_1_is_2: diag(2747, 1 /* Error */, "_0_components_don_t_accept_text_as_child_elements_Text_in_JSX_has_the_type_string_but_the_expected_t_2747", "'{0}' components don't accept text as child elements. Text in JSX has the type 'string', but the expected type of '{1}' is '{2}'."),
  Cannot_access_ambient_const_enums_when_0_is_enabled: diag(2748, 1 /* Error */, "Cannot_access_ambient_const_enums_when_0_is_enabled_2748", "Cannot access ambient const enums when '{0}' is enabled."),
  _0_refers_to_a_value_but_is_being_used_as_a_type_here_Did_you_mean_typeof_0: diag(2749, 1 /* Error */, "_0_refers_to_a_value_but_is_being_used_as_a_type_here_Did_you_mean_typeof_0_2749", "'{0}' refers to a value, but is being used as a type here. Did you mean 'typeof {0}'?"),
  The_implementation_signature_is_declared_here: diag(2750, 1 /* Error */, "The_implementation_signature_is_declared_here_2750", "The implementation signature is declared here."),
  Circularity_originates_in_type_at_this_location: diag(2751, 1 /* Error */, "Circularity_originates_in_type_at_this_location_2751", "Circularity originates in type at this location."),
  The_first_export_default_is_here: diag(2752, 1 /* Error */, "The_first_export_default_is_here_2752", "The first export default is here."),
  Another_export_default_is_here: diag(2753, 1 /* Error */, "Another_export_default_is_here_2753", "Another export default is here."),
  super_may_not_use_type_arguments: diag(2754, 1 /* Error */, "super_may_not_use_type_arguments_2754", "'super' may not use type arguments."),
  No_constituent_of_type_0_is_callable: diag(2755, 1 /* Error */, "No_constituent_of_type_0_is_callable_2755", "No constituent of type '{0}' is callable."),
  Not_all_constituents_of_type_0_are_callable: diag(2756, 1 /* Error */, "Not_all_constituents_of_type_0_are_callable_2756", "Not all constituents of type '{0}' are callable."),
  Type_0_has_no_call_signatures: diag(2757, 1 /* Error */, "Type_0_has_no_call_signatures_2757", "Type '{0}' has no call signatures."),
  Each_member_of_the_union_type_0_has_signatures_but_none_of_those_signatures_are_compatible_with_each_other: diag(2758, 1 /* Error */, "Each_member_of_the_union_type_0_has_signatures_but_none_of_those_signatures_are_compatible_with_each_2758", "Each member of the union type '{0}' has signatures, but none of those signatures are compatible with each other."),
  No_constituent_of_type_0_is_constructable: diag(2759, 1 /* Error */, "No_constituent_of_type_0_is_constructable_2759", "No constituent of type '{0}' is constructable."),
  Not_all_constituents_of_type_0_are_constructable: diag(2760, 1 /* Error */, "Not_all_constituents_of_type_0_are_constructable_2760", "Not all constituents of type '{0}' are constructable."),
  Type_0_has_no_construct_signatures: diag(2761, 1 /* Error */, "Type_0_has_no_construct_signatures_2761", "Type '{0}' has no construct signatures."),
  Each_member_of_the_union_type_0_has_construct_signatures_but_none_of_those_signatures_are_compatible_with_each_other: diag(2762, 1 /* Error */, "Each_member_of_the_union_type_0_has_construct_signatures_but_none_of_those_signatures_are_compatible_2762", "Each member of the union type '{0}' has construct signatures, but none of those signatures are compatible with each other."),
  Cannot_iterate_value_because_the_next_method_of_its_iterator_expects_type_1_but_for_of_will_always_send_0: diag(2763, 1 /* Error */, "Cannot_iterate_value_because_the_next_method_of_its_iterator_expects_type_1_but_for_of_will_always_s_2763", "Cannot iterate value because the 'next' method of its iterator expects type '{1}', but for-of will always send '{0}'."),
  Cannot_iterate_value_because_the_next_method_of_its_iterator_expects_type_1_but_array_spread_will_always_send_0: diag(2764, 1 /* Error */, "Cannot_iterate_value_because_the_next_method_of_its_iterator_expects_type_1_but_array_spread_will_al_2764", "Cannot iterate value because the 'next' method of its iterator expects type '{1}', but array spread will always send '{0}'."),
  Cannot_iterate_value_because_the_next_method_of_its_iterator_expects_type_1_but_array_destructuring_will_always_send_0: diag(2765, 1 /* Error */, "Cannot_iterate_value_because_the_next_method_of_its_iterator_expects_type_1_but_array_destructuring__2765", "Cannot iterate value because the 'next' method of its iterator expects type '{1}', but array destructuring will always send '{0}'."),
  Cannot_delegate_iteration_to_value_because_the_next_method_of_its_iterator_expects_type_1_but_the_containing_generator_will_always_send_0: diag(2766, 1 /* Error */, "Cannot_delegate_iteration_to_value_because_the_next_method_of_its_iterator_expects_type_1_but_the_co_2766", "Cannot delegate iteration to value because the 'next' method of its iterator expects type '{1}', but the containing generator will always send '{0}'."),
  The_0_property_of_an_iterator_must_be_a_method: diag(2767, 1 /* Error */, "The_0_property_of_an_iterator_must_be_a_method_2767", "The '{0}' property of an iterator must be a method."),
  The_0_property_of_an_async_iterator_must_be_a_method: diag(2768, 1 /* Error */, "The_0_property_of_an_async_iterator_must_be_a_method_2768", "The '{0}' property of an async iterator must be a method."),
  No_overload_matches_this_call: diag(2769, 1 /* Error */, "No_overload_matches_this_call_2769", "No overload matches this call."),
  The_last_overload_gave_the_following_error: diag(2770, 1 /* Error */, "The_last_overload_gave_the_following_error_2770", "The last overload gave the following error."),
  The_last_overload_is_declared_here: diag(2771, 1 /* Error */, "The_last_overload_is_declared_here_2771", "The last overload is declared here."),
  Overload_0_of_1_2_gave_the_following_error: diag(2772, 1 /* Error */, "Overload_0_of_1_2_gave_the_following_error_2772", "Overload {0} of {1}, '{2}', gave the following error."),
  Did_you_forget_to_use_await: diag(2773, 1 /* Error */, "Did_you_forget_to_use_await_2773", "Did you forget to use 'await'?"),
  This_condition_will_always_return_true_since_this_function_is_always_defined_Did_you_mean_to_call_it_instead: diag(2774, 1 /* Error */, "This_condition_will_always_return_true_since_this_function_is_always_defined_Did_you_mean_to_call_it_2774", "This condition will always return true since this function is always defined. Did you mean to call it instead?"),
  Assertions_require_every_name_in_the_call_target_to_be_declared_with_an_explicit_type_annotation: diag(2775, 1 /* Error */, "Assertions_require_every_name_in_the_call_target_to_be_declared_with_an_explicit_type_annotation_2775", "Assertions require every name in the call target to be declared with an explicit type annotation."),
  Assertions_require_the_call_target_to_be_an_identifier_or_qualified_name: diag(2776, 1 /* Error */, "Assertions_require_the_call_target_to_be_an_identifier_or_qualified_name_2776", "Assertions require the call target to be an identifier or qualified name."),
  The_operand_of_an_increment_or_decrement_operator_may_not_be_an_optional_property_access: diag(2777, 1 /* Error */, "The_operand_of_an_increment_or_decrement_operator_may_not_be_an_optional_property_access_2777", "The operand of an increment or decrement operator may not be an optional property access."),
  The_target_of_an_object_rest_assignment_may_not_be_an_optional_property_access: diag(2778, 1 /* Error */, "The_target_of_an_object_rest_assignment_may_not_be_an_optional_property_access_2778", "The target of an object rest assignment may not be an optional property access."),
  The_left_hand_side_of_an_assignment_expression_may_not_be_an_optional_property_access: diag(2779, 1 /* Error */, "The_left_hand_side_of_an_assignment_expression_may_not_be_an_optional_property_access_2779", "The left-hand side of an assignment expression may not be an optional property access."),
  The_left_hand_side_of_a_for_in_statement_may_not_be_an_optional_property_access: diag(2780, 1 /* Error */, "The_left_hand_side_of_a_for_in_statement_may_not_be_an_optional_property_access_2780", "The left-hand side of a 'for...in' statement may not be an optional property access."),
  The_left_hand_side_of_a_for_of_statement_may_not_be_an_optional_property_access: diag(2781, 1 /* Error */, "The_left_hand_side_of_a_for_of_statement_may_not_be_an_optional_property_access_2781", "The left-hand side of a 'for...of' statement may not be an optional property access."),
  _0_needs_an_explicit_type_annotation: diag(2782, 3 /* Message */, "_0_needs_an_explicit_type_annotation_2782", "'{0}' needs an explicit type annotation."),
  _0_is_specified_more_than_once_so_this_usage_will_be_overwritten: diag(2783, 1 /* Error */, "_0_is_specified_more_than_once_so_this_usage_will_be_overwritten_2783", "'{0}' is specified more than once, so this usage will be overwritten."),
  get_and_set_accessors_cannot_declare_this_parameters: diag(2784, 1 /* Error */, "get_and_set_accessors_cannot_declare_this_parameters_2784", "'get' and 'set' accessors cannot declare 'this' parameters."),
  This_spread_always_overwrites_this_property: diag(2785, 1 /* Error */, "This_spread_always_overwrites_this_property_2785", "This spread always overwrites this property."),
  _0_cannot_be_used_as_a_JSX_component: diag(2786, 1 /* Error */, "_0_cannot_be_used_as_a_JSX_component_2786", "'{0}' cannot be used as a JSX component."),
  Its_return_type_0_is_not_a_valid_JSX_element: diag(2787, 1 /* Error */, "Its_return_type_0_is_not_a_valid_JSX_element_2787", "Its return type '{0}' is not a valid JSX element."),
  Its_instance_type_0_is_not_a_valid_JSX_element: diag(2788, 1 /* Error */, "Its_instance_type_0_is_not_a_valid_JSX_element_2788", "Its instance type '{0}' is not a valid JSX element."),
  Its_element_type_0_is_not_a_valid_JSX_element: diag(2789, 1 /* Error */, "Its_element_type_0_is_not_a_valid_JSX_element_2789", "Its element type '{0}' is not a valid JSX element."),
  The_operand_of_a_delete_operator_must_be_optional: diag(2790, 1 /* Error */, "The_operand_of_a_delete_operator_must_be_optional_2790", "The operand of a 'delete' operator must be optional."),
  Exponentiation_cannot_be_performed_on_bigint_values_unless_the_target_option_is_set_to_es2016_or_later: diag(2791, 1 /* Error */, "Exponentiation_cannot_be_performed_on_bigint_values_unless_the_target_option_is_set_to_es2016_or_lat_2791", "Exponentiation cannot be performed on 'bigint' values unless the 'target' option is set to 'es2016' or later."),
  Cannot_find_module_0_Did_you_mean_to_set_the_moduleResolution_option_to_nodenext_or_to_add_aliases_to_the_paths_option: diag(2792, 1 /* Error */, "Cannot_find_module_0_Did_you_mean_to_set_the_moduleResolution_option_to_nodenext_or_to_add_aliases_t_2792", "Cannot find module '{0}'. Did you mean to set the 'moduleResolution' option to 'nodenext', or to add aliases to the 'paths' option?"),
  The_call_would_have_succeeded_against_this_implementation_but_implementation_signatures_of_overloads_are_not_externally_visible: diag(2793, 1 /* Error */, "The_call_would_have_succeeded_against_this_implementation_but_implementation_signatures_of_overloads_2793", "The call would have succeeded against this implementation, but implementation signatures of overloads are not externally visible."),
  Expected_0_arguments_but_got_1_Did_you_forget_to_include_void_in_your_type_argument_to_Promise: diag(2794, 1 /* Error */, "Expected_0_arguments_but_got_1_Did_you_forget_to_include_void_in_your_type_argument_to_Promise_2794", "Expected {0} arguments, but got {1}. Did you forget to include 'void' in your type argument to 'Promise'?"),
  The_intrinsic_keyword_can_only_be_used_to_declare_compiler_provided_intrinsic_types: diag(2795, 1 /* Error */, "The_intrinsic_keyword_can_only_be_used_to_declare_compiler_provided_intrinsic_types_2795", "The 'intrinsic' keyword can only be used to declare compiler provided intrinsic types."),
  It_is_likely_that_you_are_missing_a_comma_to_separate_these_two_template_expressions_They_form_a_tagged_template_expression_which_cannot_be_invoked: diag(2796, 1 /* Error */, "It_is_likely_that_you_are_missing_a_comma_to_separate_these_two_template_expressions_They_form_a_tag_2796", "It is likely that you are missing a comma to separate these two template expressions. They form a tagged template expression which cannot be invoked."),
  A_mixin_class_that_extends_from_a_type_variable_containing_an_abstract_construct_signature_must_also_be_declared_abstract: diag(2797, 1 /* Error */, "A_mixin_class_that_extends_from_a_type_variable_containing_an_abstract_construct_signature_must_also_2797", "A mixin class that extends from a type variable containing an abstract construct signature must also be declared 'abstract'."),
  The_declaration_was_marked_as_deprecated_here: diag(2798, 1 /* Error */, "The_declaration_was_marked_as_deprecated_here_2798", "The declaration was marked as deprecated here."),
  Type_produces_a_tuple_type_that_is_too_large_to_represent: diag(2799, 1 /* Error */, "Type_produces_a_tuple_type_that_is_too_large_to_represent_2799", "Type produces a tuple type that is too large to represent."),
  Expression_produces_a_tuple_type_that_is_too_large_to_represent: diag(2800, 1 /* Error */, "Expression_produces_a_tuple_type_that_is_too_large_to_represent_2800", "Expression produces a tuple type that is too large to represent."),
  This_condition_will_always_return_true_since_this_0_is_always_defined: diag(2801, 1 /* Error */, "This_condition_will_always_return_true_since_this_0_is_always_defined_2801", "This condition will always return true since this '{0}' is always defined."),
  Type_0_can_only_be_iterated_through_when_using_the_downlevelIteration_flag_or_with_a_target_of_es2015_or_higher: diag(2802, 1 /* Error */, "Type_0_can_only_be_iterated_through_when_using_the_downlevelIteration_flag_or_with_a_target_of_es201_2802", "Type '{0}' can only be iterated through when using the '--downlevelIteration' flag or with a '--target' of 'es2015' or higher."),
  Cannot_assign_to_private_method_0_Private_methods_are_not_writable: diag(2803, 1 /* Error */, "Cannot_assign_to_private_method_0_Private_methods_are_not_writable_2803", "Cannot assign to private method '{0}'. Private methods are not writable."),
  Duplicate_identifier_0_Static_and_instance_elements_cannot_share_the_same_private_name: diag(2804, 1 /* Error */, "Duplicate_identifier_0_Static_and_instance_elements_cannot_share_the_same_private_name_2804", "Duplicate identifier '{0}'. Static and instance elements cannot share the same private name."),
  Private_accessor_was_defined_without_a_getter: diag(2806, 1 /* Error */, "Private_accessor_was_defined_without_a_getter_2806", "Private accessor was defined without a getter."),
  This_syntax_requires_an_imported_helper_named_1_with_2_parameters_which_is_not_compatible_with_the_one_in_0_Consider_upgrading_your_version_of_0: diag(2807, 1 /* Error */, "This_syntax_requires_an_imported_helper_named_1_with_2_parameters_which_is_not_compatible_with_the_o_2807", "This syntax requires an imported helper named '{1}' with {2} parameters, which is not compatible with the one in '{0}'. Consider upgrading your version of '{0}'."),
  A_get_accessor_must_be_at_least_as_accessible_as_the_setter: diag(2808, 1 /* Error */, "A_get_accessor_must_be_at_least_as_accessible_as_the_setter_2808", "A get accessor must be at least as accessible as the setter"),
  Declaration_or_statement_expected_This_follows_a_block_of_statements_so_if_you_intended_to_write_a_destructuring_assignment_you_might_need_to_wrap_the_whole_assignment_in_parentheses: diag(2809, 1 /* Error */, "Declaration_or_statement_expected_This_follows_a_block_of_statements_so_if_you_intended_to_write_a_d_2809", "Declaration or statement expected. This '=' follows a block of statements, so if you intended to write a destructuring assignment, you might need to wrap the whole assignment in parentheses."),
  Expected_1_argument_but_got_0_new_Promise_needs_a_JSDoc_hint_to_produce_a_resolve_that_can_be_called_without_arguments: diag(2810, 1 /* Error */, "Expected_1_argument_but_got_0_new_Promise_needs_a_JSDoc_hint_to_produce_a_resolve_that_can_be_called_2810", "Expected 1 argument, but got 0. 'new Promise()' needs a JSDoc hint to produce a 'resolve' that can be called without arguments."),
  Initializer_for_property_0: diag(2811, 1 /* Error */, "Initializer_for_property_0_2811", "Initializer for property '{0}'"),
  Property_0_does_not_exist_on_type_1_Try_changing_the_lib_compiler_option_to_include_dom: diag(2812, 1 /* Error */, "Property_0_does_not_exist_on_type_1_Try_changing_the_lib_compiler_option_to_include_dom_2812", "Property '{0}' does not exist on type '{1}'. Try changing the 'lib' compiler option to include 'dom'."),
  Class_declaration_cannot_implement_overload_list_for_0: diag(2813, 1 /* Error */, "Class_declaration_cannot_implement_overload_list_for_0_2813", "Class declaration cannot implement overload list for '{0}'."),
  Function_with_bodies_can_only_merge_with_classes_that_are_ambient: diag(2814, 1 /* Error */, "Function_with_bodies_can_only_merge_with_classes_that_are_ambient_2814", "Function with bodies can only merge with classes that are ambient."),
  arguments_cannot_be_referenced_in_property_initializers: diag(2815, 1 /* Error */, "arguments_cannot_be_referenced_in_property_initializers_2815", "'arguments' cannot be referenced in property initializers."),
  Cannot_use_this_in_a_static_property_initializer_of_a_decorated_class: diag(2816, 1 /* Error */, "Cannot_use_this_in_a_static_property_initializer_of_a_decorated_class_2816", "Cannot use 'this' in a static property initializer of a decorated class."),
  Property_0_has_no_initializer_and_is_not_definitely_assigned_in_a_class_static_block: diag(2817, 1 /* Error */, "Property_0_has_no_initializer_and_is_not_definitely_assigned_in_a_class_static_block_2817", "Property '{0}' has no initializer and is not definitely assigned in a class static block."),
  Duplicate_identifier_0_Compiler_reserves_name_1_when_emitting_super_references_in_static_initializers: diag(2818, 1 /* Error */, "Duplicate_identifier_0_Compiler_reserves_name_1_when_emitting_super_references_in_static_initializer_2818", "Duplicate identifier '{0}'. Compiler reserves name '{1}' when emitting 'super' references in static initializers."),
  Namespace_name_cannot_be_0: diag(2819, 1 /* Error */, "Namespace_name_cannot_be_0_2819", "Namespace name cannot be '{0}'."),
  Type_0_is_not_assignable_to_type_1_Did_you_mean_2: diag(2820, 1 /* Error */, "Type_0_is_not_assignable_to_type_1_Did_you_mean_2_2820", "Type '{0}' is not assignable to type '{1}'. Did you mean '{2}'?"),
  Import_assertions_are_only_supported_when_the_module_option_is_set_to_esnext_nodenext_or_preserve: diag(2821, 1 /* Error */, "Import_assertions_are_only_supported_when_the_module_option_is_set_to_esnext_nodenext_or_preserve_2821", "Import assertions are only supported when the '--module' option is set to 'esnext', 'nodenext', or 'preserve'."),
  Import_assertions_cannot_be_used_with_type_only_imports_or_exports: diag(2822, 1 /* Error */, "Import_assertions_cannot_be_used_with_type_only_imports_or_exports_2822", "Import assertions cannot be used with type-only imports or exports."),
  Import_attributes_are_only_supported_when_the_module_option_is_set_to_esnext_nodenext_or_preserve: diag(2823, 1 /* Error */, "Import_attributes_are_only_supported_when_the_module_option_is_set_to_esnext_nodenext_or_preserve_2823", "Import attributes are only supported when the '--module' option is set to 'esnext', 'nodenext', or 'preserve'."),
  Cannot_find_namespace_0_Did_you_mean_1: diag(2833, 1 /* Error */, "Cannot_find_namespace_0_Did_you_mean_1_2833", "Cannot find namespace '{0}'. Did you mean '{1}'?"),
  Relative_import_paths_need_explicit_file_extensions_in_ECMAScript_imports_when_moduleResolution_is_node16_or_nodenext_Consider_adding_an_extension_to_the_import_path: diag(2834, 1 /* Error */, "Relative_import_paths_need_explicit_file_extensions_in_ECMAScript_imports_when_moduleResolution_is_n_2834", "Relative import paths need explicit file extensions in ECMAScript imports when '--moduleResolution' is 'node16' or 'nodenext'. Consider adding an extension to the import path."),
  Relative_import_paths_need_explicit_file_extensions_in_ECMAScript_imports_when_moduleResolution_is_node16_or_nodenext_Did_you_mean_0: diag(2835, 1 /* Error */, "Relative_import_paths_need_explicit_file_extensions_in_ECMAScript_imports_when_moduleResolution_is_n_2835", "Relative import paths need explicit file extensions in ECMAScript imports when '--moduleResolution' is 'node16' or 'nodenext'. Did you mean '{0}'?"),
  Import_assertions_are_not_allowed_on_statements_that_compile_to_CommonJS_require_calls: diag(2836, 1 /* Error */, "Import_assertions_are_not_allowed_on_statements_that_compile_to_CommonJS_require_calls_2836", "Import assertions are not allowed on statements that compile to CommonJS 'require' calls."),
  Import_assertion_values_must_be_string_literal_expressions: diag(2837, 1 /* Error */, "Import_assertion_values_must_be_string_literal_expressions_2837", "Import assertion values must be string literal expressions."),
  All_declarations_of_0_must_have_identical_constraints: diag(2838, 1 /* Error */, "All_declarations_of_0_must_have_identical_constraints_2838", "All declarations of '{0}' must have identical constraints."),
  This_condition_will_always_return_0_since_JavaScript_compares_objects_by_reference_not_value: diag(2839, 1 /* Error */, "This_condition_will_always_return_0_since_JavaScript_compares_objects_by_reference_not_value_2839", "This condition will always return '{0}' since JavaScript compares objects by reference, not value."),
  An_interface_cannot_extend_a_primitive_type_like_0_It_can_only_extend_other_named_object_types: diag(2840, 1 /* Error */, "An_interface_cannot_extend_a_primitive_type_like_0_It_can_only_extend_other_named_object_types_2840", "An interface cannot extend a primitive type like '{0}'. It can only extend other named object types."),
  _0_is_an_unused_renaming_of_1_Did_you_intend_to_use_it_as_a_type_annotation: diag(2842, 1 /* Error */, "_0_is_an_unused_renaming_of_1_Did_you_intend_to_use_it_as_a_type_annotation_2842", "'{0}' is an unused renaming of '{1}'. Did you intend to use it as a type annotation?"),
  We_can_only_write_a_type_for_0_by_adding_a_type_for_the_entire_parameter_here: diag(2843, 1 /* Error */, "We_can_only_write_a_type_for_0_by_adding_a_type_for_the_entire_parameter_here_2843", "We can only write a type for '{0}' by adding a type for the entire parameter here."),
  Type_of_instance_member_variable_0_cannot_reference_identifier_1_declared_in_the_constructor: diag(2844, 1 /* Error */, "Type_of_instance_member_variable_0_cannot_reference_identifier_1_declared_in_the_constructor_2844", "Type of instance member variable '{0}' cannot reference identifier '{1}' declared in the constructor."),
  This_condition_will_always_return_0: diag(2845, 1 /* Error */, "This_condition_will_always_return_0_2845", "This condition will always return '{0}'."),
  A_declaration_file_cannot_be_imported_without_import_type_Did_you_mean_to_import_an_implementation_file_0_instead: diag(2846, 1 /* Error */, "A_declaration_file_cannot_be_imported_without_import_type_Did_you_mean_to_import_an_implementation_f_2846", "A declaration file cannot be imported without 'import type'. Did you mean to import an implementation file '{0}' instead?"),
  The_right_hand_side_of_an_instanceof_expression_must_not_be_an_instantiation_expression: diag(2848, 1 /* Error */, "The_right_hand_side_of_an_instanceof_expression_must_not_be_an_instantiation_expression_2848", "The right-hand side of an 'instanceof' expression must not be an instantiation expression."),
  Target_signature_provides_too_few_arguments_Expected_0_or_more_but_got_1: diag(2849, 1 /* Error */, "Target_signature_provides_too_few_arguments_Expected_0_or_more_but_got_1_2849", "Target signature provides too few arguments. Expected {0} or more, but got {1}."),
  The_initializer_of_a_using_declaration_must_be_either_an_object_with_a_Symbol_dispose_method_or_be_null_or_undefined: diag(2850, 1 /* Error */, "The_initializer_of_a_using_declaration_must_be_either_an_object_with_a_Symbol_dispose_method_or_be_n_2850", "The initializer of a 'using' declaration must be either an object with a '[Symbol.dispose]()' method, or be 'null' or 'undefined'."),
  The_initializer_of_an_await_using_declaration_must_be_either_an_object_with_a_Symbol_asyncDispose_or_Symbol_dispose_method_or_be_null_or_undefined: diag(2851, 1 /* Error */, "The_initializer_of_an_await_using_declaration_must_be_either_an_object_with_a_Symbol_asyncDispose_or_2851", "The initializer of an 'await using' declaration must be either an object with a '[Symbol.asyncDispose]()' or '[Symbol.dispose]()' method, or be 'null' or 'undefined'."),
  await_using_statements_are_only_allowed_within_async_functions_and_at_the_top_levels_of_modules: diag(2852, 1 /* Error */, "await_using_statements_are_only_allowed_within_async_functions_and_at_the_top_levels_of_modules_2852", "'await using' statements are only allowed within async functions and at the top levels of modules."),
  await_using_statements_are_only_allowed_at_the_top_level_of_a_file_when_that_file_is_a_module_but_this_file_has_no_imports_or_exports_Consider_adding_an_empty_export_to_make_this_file_a_module: diag(2853, 1 /* Error */, "await_using_statements_are_only_allowed_at_the_top_level_of_a_file_when_that_file_is_a_module_but_th_2853", "'await using' statements are only allowed at the top level of a file when that file is a module, but this file has no imports or exports. Consider adding an empty 'export {}' to make this file a module."),
  Top_level_await_using_statements_are_only_allowed_when_the_module_option_is_set_to_es2022_esnext_system_node16_nodenext_or_preserve_and_the_target_option_is_set_to_es2017_or_higher: diag(2854, 1 /* Error */, "Top_level_await_using_statements_are_only_allowed_when_the_module_option_is_set_to_es2022_esnext_sys_2854", "Top-level 'await using' statements are only allowed when the 'module' option is set to 'es2022', 'esnext', 'system', 'node16', 'nodenext', or 'preserve', and the 'target' option is set to 'es2017' or higher."),
  Class_field_0_defined_by_the_parent_class_is_not_accessible_in_the_child_class_via_super: diag(2855, 1 /* Error */, "Class_field_0_defined_by_the_parent_class_is_not_accessible_in_the_child_class_via_super_2855", "Class field '{0}' defined by the parent class is not accessible in the child class via super."),
  Import_attributes_are_not_allowed_on_statements_that_compile_to_CommonJS_require_calls: diag(2856, 1 /* Error */, "Import_attributes_are_not_allowed_on_statements_that_compile_to_CommonJS_require_calls_2856", "Import attributes are not allowed on statements that compile to CommonJS 'require' calls."),
  Import_attributes_cannot_be_used_with_type_only_imports_or_exports: diag(2857, 1 /* Error */, "Import_attributes_cannot_be_used_with_type_only_imports_or_exports_2857", "Import attributes cannot be used with type-only imports or exports."),
  Import_attribute_values_must_be_string_literal_expressions: diag(2858, 1 /* Error */, "Import_attribute_values_must_be_string_literal_expressions_2858", "Import attribute values must be string literal expressions."),
  Excessive_complexity_comparing_types_0_and_1: diag(2859, 1 /* Error */, "Excessive_complexity_comparing_types_0_and_1_2859", "Excessive complexity comparing types '{0}' and '{1}'."),
  The_left_hand_side_of_an_instanceof_expression_must_be_assignable_to_the_first_argument_of_the_right_hand_side_s_Symbol_hasInstance_method: diag(2860, 1 /* Error */, "The_left_hand_side_of_an_instanceof_expression_must_be_assignable_to_the_first_argument_of_the_right_2860", "The left-hand side of an 'instanceof' expression must be assignable to the first argument of the right-hand side's '[Symbol.hasInstance]' method."),
  An_object_s_Symbol_hasInstance_method_must_return_a_boolean_value_for_it_to_be_used_on_the_right_hand_side_of_an_instanceof_expression: diag(2861, 1 /* Error */, "An_object_s_Symbol_hasInstance_method_must_return_a_boolean_value_for_it_to_be_used_on_the_right_han_2861", "An object's '[Symbol.hasInstance]' method must return a boolean value for it to be used on the right-hand side of an 'instanceof' expression."),
  Type_0_is_generic_and_can_only_be_indexed_for_reading: diag(2862, 1 /* Error */, "Type_0_is_generic_and_can_only_be_indexed_for_reading_2862", "Type '{0}' is generic and can only be indexed for reading."),
  A_class_cannot_extend_a_primitive_type_like_0_Classes_can_only_extend_constructable_values: diag(2863, 1 /* Error */, "A_class_cannot_extend_a_primitive_type_like_0_Classes_can_only_extend_constructable_values_2863", "A class cannot extend a primitive type like '{0}'. Classes can only extend constructable values."),
  A_class_cannot_implement_a_primitive_type_like_0_It_can_only_implement_other_named_object_types: diag(2864, 1 /* Error */, "A_class_cannot_implement_a_primitive_type_like_0_It_can_only_implement_other_named_object_types_2864", "A class cannot implement a primitive type like '{0}'. It can only implement other named object types."),
  Import_0_conflicts_with_local_value_so_must_be_declared_with_a_type_only_import_when_isolatedModules_is_enabled: diag(2865, 1 /* Error */, "Import_0_conflicts_with_local_value_so_must_be_declared_with_a_type_only_import_when_isolatedModules_2865", "Import '{0}' conflicts with local value, so must be declared with a type-only import when 'isolatedModules' is enabled."),
  Import_0_conflicts_with_global_value_used_in_this_file_so_must_be_declared_with_a_type_only_import_when_isolatedModules_is_enabled: diag(2866, 1 /* Error */, "Import_0_conflicts_with_global_value_used_in_this_file_so_must_be_declared_with_a_type_only_import_w_2866", "Import '{0}' conflicts with global value used in this file, so must be declared with a type-only import when 'isolatedModules' is enabled."),
  Cannot_find_name_0_Do_you_need_to_install_type_definitions_for_Bun_Try_npm_i_save_dev_types_Slashbun: diag(2867, 1 /* Error */, "Cannot_find_name_0_Do_you_need_to_install_type_definitions_for_Bun_Try_npm_i_save_dev_types_Slashbun_2867", "Cannot find name '{0}'. Do you need to install type definitions for Bun? Try `npm i --save-dev @types/bun`."),
  Cannot_find_name_0_Do_you_need_to_install_type_definitions_for_Bun_Try_npm_i_save_dev_types_Slashbun_and_then_add_bun_to_the_types_field_in_your_tsconfig: diag(2868, 1 /* Error */, "Cannot_find_name_0_Do_you_need_to_install_type_definitions_for_Bun_Try_npm_i_save_dev_types_Slashbun_2868", "Cannot find name '{0}'. Do you need to install type definitions for Bun? Try `npm i --save-dev @types/bun` and then add 'bun' to the types field in your tsconfig."),
  Right_operand_of_is_unreachable_because_the_left_operand_is_never_nullish: diag(2869, 1 /* Error */, "Right_operand_of_is_unreachable_because_the_left_operand_is_never_nullish_2869", "Right operand of ?? is unreachable because the left operand is never nullish."),
  This_binary_expression_is_never_nullish_Are_you_missing_parentheses: diag(2870, 1 /* Error */, "This_binary_expression_is_never_nullish_Are_you_missing_parentheses_2870", "This binary expression is never nullish. Are you missing parentheses?"),
  This_expression_is_always_nullish: diag(2871, 1 /* Error */, "This_expression_is_always_nullish_2871", "This expression is always nullish."),
  This_kind_of_expression_is_always_truthy: diag(2872, 1 /* Error */, "This_kind_of_expression_is_always_truthy_2872", "This kind of expression is always truthy."),
  This_kind_of_expression_is_always_falsy: diag(2873, 1 /* Error */, "This_kind_of_expression_is_always_falsy_2873", "This kind of expression is always falsy."),
  This_JSX_tag_requires_0_to_be_in_scope_but_it_could_not_be_found: diag(2874, 1 /* Error */, "This_JSX_tag_requires_0_to_be_in_scope_but_it_could_not_be_found_2874", "This JSX tag requires '{0}' to be in scope, but it could not be found."),
  This_JSX_tag_requires_the_module_path_0_to_exist_but_none_could_be_found_Make_sure_you_have_types_for_the_appropriate_package_installed: diag(2875, 1 /* Error */, "This_JSX_tag_requires_the_module_path_0_to_exist_but_none_could_be_found_Make_sure_you_have_types_fo_2875", "This JSX tag requires the module path '{0}' to exist, but none could be found. Make sure you have types for the appropriate package installed."),
  This_relative_import_path_is_unsafe_to_rewrite_because_it_looks_like_a_file_name_but_actually_resolves_to_0: diag(2876, 1 /* Error */, "This_relative_import_path_is_unsafe_to_rewrite_because_it_looks_like_a_file_name_but_actually_resolv_2876", 'This relative import path is unsafe to rewrite because it looks like a file name, but actually resolves to "{0}".'),
  This_import_uses_a_0_extension_to_resolve_to_an_input_TypeScript_file_but_will_not_be_rewritten_during_emit_because_it_is_not_a_relative_path: diag(2877, 1 /* Error */, "This_import_uses_a_0_extension_to_resolve_to_an_input_TypeScript_file_but_will_not_be_rewritten_duri_2877", "This import uses a '{0}' extension to resolve to an input TypeScript file, but will not be rewritten during emit because it is not a relative path."),
  This_import_path_is_unsafe_to_rewrite_because_it_resolves_to_another_project_and_the_relative_path_between_the_projects_output_files_is_not_the_same_as_the_relative_path_between_its_input_files: diag(2878, 1 /* Error */, "This_import_path_is_unsafe_to_rewrite_because_it_resolves_to_another_project_and_the_relative_path_b_2878", "This import path is unsafe to rewrite because it resolves to another project, and the relative path between the projects' output files is not the same as the relative path between its input files."),
  Using_JSX_fragments_requires_fragment_factory_0_to_be_in_scope_but_it_could_not_be_found: diag(2879, 1 /* Error */, "Using_JSX_fragments_requires_fragment_factory_0_to_be_in_scope_but_it_could_not_be_found_2879", "Using JSX fragments requires fragment factory '{0}' to be in scope, but it could not be found."),
  Import_declaration_0_is_using_private_name_1: diag(4e3, 1 /* Error */, "Import_declaration_0_is_using_private_name_1_4000", "Import declaration '{0}' is using private name '{1}'."),
  Type_parameter_0_of_exported_class_has_or_is_using_private_name_1: diag(4002, 1 /* Error */, "Type_parameter_0_of_exported_class_has_or_is_using_private_name_1_4002", "Type parameter '{0}' of exported class has or is using private name '{1}'."),
  Type_parameter_0_of_exported_interface_has_or_is_using_private_name_1: diag(4004, 1 /* Error */, "Type_parameter_0_of_exported_interface_has_or_is_using_private_name_1_4004", "Type parameter '{0}' of exported interface has or is using private name '{1}'."),
  Type_parameter_0_of_constructor_signature_from_exported_interface_has_or_is_using_private_name_1: diag(4006, 1 /* Error */, "Type_parameter_0_of_constructor_signature_from_exported_interface_has_or_is_using_private_name_1_4006", "Type parameter '{0}' of constructor signature from exported interface has or is using private name '{1}'."),
  Type_parameter_0_of_call_signature_from_exported_interface_has_or_is_using_private_name_1: diag(4008, 1 /* Error */, "Type_parameter_0_of_call_signature_from_exported_interface_has_or_is_using_private_name_1_4008", "Type parameter '{0}' of call signature from exported interface has or is using private name '{1}'."),
  Type_parameter_0_of_public_static_method_from_exported_class_has_or_is_using_private_name_1: diag(4010, 1 /* Error */, "Type_parameter_0_of_public_static_method_from_exported_class_has_or_is_using_private_name_1_4010", "Type parameter '{0}' of public static method from exported class has or is using private name '{1}'."),
  Type_parameter_0_of_public_method_from_exported_class_has_or_is_using_private_name_1: diag(4012, 1 /* Error */, "Type_parameter_0_of_public_method_from_exported_class_has_or_is_using_private_name_1_4012", "Type parameter '{0}' of public method from exported class has or is using private name '{1}'."),
  Type_parameter_0_of_method_from_exported_interface_has_or_is_using_private_name_1: diag(4014, 1 /* Error */, "Type_parameter_0_of_method_from_exported_interface_has_or_is_using_private_name_1_4014", "Type parameter '{0}' of method from exported interface has or is using private name '{1}'."),
  Type_parameter_0_of_exported_function_has_or_is_using_private_name_1: diag(4016, 1 /* Error */, "Type_parameter_0_of_exported_function_has_or_is_using_private_name_1_4016", "Type parameter '{0}' of exported function has or is using private name '{1}'."),
  Implements_clause_of_exported_class_0_has_or_is_using_private_name_1: diag(4019, 1 /* Error */, "Implements_clause_of_exported_class_0_has_or_is_using_private_name_1_4019", "Implements clause of exported class '{0}' has or is using private name '{1}'."),
  extends_clause_of_exported_class_0_has_or_is_using_private_name_1: diag(4020, 1 /* Error */, "extends_clause_of_exported_class_0_has_or_is_using_private_name_1_4020", "'extends' clause of exported class '{0}' has or is using private name '{1}'."),
  extends_clause_of_exported_class_has_or_is_using_private_name_0: diag(4021, 1 /* Error */, "extends_clause_of_exported_class_has_or_is_using_private_name_0_4021", "'extends' clause of exported class has or is using private name '{0}'."),
  extends_clause_of_exported_interface_0_has_or_is_using_private_name_1: diag(4022, 1 /* Error */, "extends_clause_of_exported_interface_0_has_or_is_using_private_name_1_4022", "'extends' clause of exported interface '{0}' has or is using private name '{1}'."),
  Exported_variable_0_has_or_is_using_name_1_from_external_module_2_but_cannot_be_named: diag(4023, 1 /* Error */, "Exported_variable_0_has_or_is_using_name_1_from_external_module_2_but_cannot_be_named_4023", "Exported variable '{0}' has or is using name '{1}' from external module {2} but cannot be named."),
  Exported_variable_0_has_or_is_using_name_1_from_private_module_2: diag(4024, 1 /* Error */, "Exported_variable_0_has_or_is_using_name_1_from_private_module_2_4024", "Exported variable '{0}' has or is using name '{1}' from private module '{2}'."),
  Exported_variable_0_has_or_is_using_private_name_1: diag(4025, 1 /* Error */, "Exported_variable_0_has_or_is_using_private_name_1_4025", "Exported variable '{0}' has or is using private name '{1}'."),
  Public_static_property_0_of_exported_class_has_or_is_using_name_1_from_external_module_2_but_cannot_be_named: diag(4026, 1 /* Error */, "Public_static_property_0_of_exported_class_has_or_is_using_name_1_from_external_module_2_but_cannot__4026", "Public static property '{0}' of exported class has or is using name '{1}' from external module {2} but cannot be named."),
  Public_static_property_0_of_exported_class_has_or_is_using_name_1_from_private_module_2: diag(4027, 1 /* Error */, "Public_static_property_0_of_exported_class_has_or_is_using_name_1_from_private_module_2_4027", "Public static property '{0}' of exported class has or is using name '{1}' from private module '{2}'."),
  Public_static_property_0_of_exported_class_has_or_is_using_private_name_1: diag(4028, 1 /* Error */, "Public_static_property_0_of_exported_class_has_or_is_using_private_name_1_4028", "Public static property '{0}' of exported class has or is using private name '{1}'."),
  Public_property_0_of_exported_class_has_or_is_using_name_1_from_external_module_2_but_cannot_be_named: diag(4029, 1 /* Error */, "Public_property_0_of_exported_class_has_or_is_using_name_1_from_external_module_2_but_cannot_be_name_4029", "Public property '{0}' of exported class has or is using name '{1}' from external module {2} but cannot be named."),
  Public_property_0_of_exported_class_has_or_is_using_name_1_from_private_module_2: diag(4030, 1 /* Error */, "Public_property_0_of_exported_class_has_or_is_using_name_1_from_private_module_2_4030", "Public property '{0}' of exported class has or is using name '{1}' from private module '{2}'."),
  Public_property_0_of_exported_class_has_or_is_using_private_name_1: diag(4031, 1 /* Error */, "Public_property_0_of_exported_class_has_or_is_using_private_name_1_4031", "Public property '{0}' of exported class has or is using private name '{1}'."),
  Property_0_of_exported_interface_has_or_is_using_name_1_from_private_module_2: diag(4032, 1 /* Error */, "Property_0_of_exported_interface_has_or_is_using_name_1_from_private_module_2_4032", "Property '{0}' of exported interface has or is using name '{1}' from private module '{2}'."),
  Property_0_of_exported_interface_has_or_is_using_private_name_1: diag(4033, 1 /* Error */, "Property_0_of_exported_interface_has_or_is_using_private_name_1_4033", "Property '{0}' of exported interface has or is using private name '{1}'."),
  Parameter_type_of_public_static_setter_0_from_exported_class_has_or_is_using_name_1_from_private_module_2: diag(4034, 1 /* Error */, "Parameter_type_of_public_static_setter_0_from_exported_class_has_or_is_using_name_1_from_private_mod_4034", "Parameter type of public static setter '{0}' from exported class has or is using name '{1}' from private module '{2}'."),
  Parameter_type_of_public_static_setter_0_from_exported_class_has_or_is_using_private_name_1: diag(4035, 1 /* Error */, "Parameter_type_of_public_static_setter_0_from_exported_class_has_or_is_using_private_name_1_4035", "Parameter type of public static setter '{0}' from exported class has or is using private name '{1}'."),
  Parameter_type_of_public_setter_0_from_exported_class_has_or_is_using_name_1_from_private_module_2: diag(4036, 1 /* Error */, "Parameter_type_of_public_setter_0_from_exported_class_has_or_is_using_name_1_from_private_module_2_4036", "Parameter type of public setter '{0}' from exported class has or is using name '{1}' from private module '{2}'."),
  Parameter_type_of_public_setter_0_from_exported_class_has_or_is_using_private_name_1: diag(4037, 1 /* Error */, "Parameter_type_of_public_setter_0_from_exported_class_has_or_is_using_private_name_1_4037", "Parameter type of public setter '{0}' from exported class has or is using private name '{1}'."),
  Return_type_of_public_static_getter_0_from_exported_class_has_or_is_using_name_1_from_external_module_2_but_cannot_be_named: diag(4038, 1 /* Error */, "Return_type_of_public_static_getter_0_from_exported_class_has_or_is_using_name_1_from_external_modul_4038", "Return type of public static getter '{0}' from exported class has or is using name '{1}' from external module {2} but cannot be named."),
  Return_type_of_public_static_getter_0_from_exported_class_has_or_is_using_name_1_from_private_module_2: diag(4039, 1 /* Error */, "Return_type_of_public_static_getter_0_from_exported_class_has_or_is_using_name_1_from_private_module_4039", "Return type of public static getter '{0}' from exported class has or is using name '{1}' from private module '{2}'."),
  Return_type_of_public_static_getter_0_from_exported_class_has_or_is_using_private_name_1: diag(4040, 1 /* Error */, "Return_type_of_public_static_getter_0_from_exported_class_has_or_is_using_private_name_1_4040", "Return type of public static getter '{0}' from exported class has or is using private name '{1}'."),
  Return_type_of_public_getter_0_from_exported_class_has_or_is_using_name_1_from_external_module_2_but_cannot_be_named: diag(4041, 1 /* Error */, "Return_type_of_public_getter_0_from_exported_class_has_or_is_using_name_1_from_external_module_2_but_4041", "Return type of public getter '{0}' from exported class has or is using name '{1}' from external module {2} but cannot be named."),
  Return_type_of_public_getter_0_from_exported_class_has_or_is_using_name_1_from_private_module_2: diag(4042, 1 /* Error */, "Return_type_of_public_getter_0_from_exported_class_has_or_is_using_name_1_from_private_module_2_4042", "Return type of public getter '{0}' from exported class has or is using name '{1}' from private module '{2}'."),
  Return_type_of_public_getter_0_from_exported_class_has_or_is_using_private_name_1: diag(4043, 1 /* Error */, "Return_type_of_public_getter_0_from_exported_class_has_or_is_using_private_name_1_4043", "Return type of public getter '{0}' from exported class has or is using private name '{1}'."),
  Return_type_of_constructor_signature_from_exported_interface_has_or_is_using_name_0_from_private_module_1: diag(4044, 1 /* Error */, "Return_type_of_constructor_signature_from_exported_interface_has_or_is_using_name_0_from_private_mod_4044", "Return type of constructor signature from exported interface has or is using name '{0}' from private module '{1}'."),
  Return_type_of_constructor_signature_from_exported_interface_has_or_is_using_private_name_0: diag(4045, 1 /* Error */, "Return_type_of_constructor_signature_from_exported_interface_has_or_is_using_private_name_0_4045", "Return type of constructor signature from exported interface has or is using private name '{0}'."),
  Return_type_of_call_signature_from_exported_interface_has_or_is_using_name_0_from_private_module_1: diag(4046, 1 /* Error */, "Return_type_of_call_signature_from_exported_interface_has_or_is_using_name_0_from_private_module_1_4046", "Return type of call signature from exported interface has or is using name '{0}' from private module '{1}'."),
  Return_type_of_call_signature_from_exported_interface_has_or_is_using_private_name_0: diag(4047, 1 /* Error */, "Return_type_of_call_signature_from_exported_interface_has_or_is_using_private_name_0_4047", "Return type of call signature from exported interface has or is using private name '{0}'."),
  Return_type_of_index_signature_from_exported_interface_has_or_is_using_name_0_from_private_module_1: diag(4048, 1 /* Error */, "Return_type_of_index_signature_from_exported_interface_has_or_is_using_name_0_from_private_module_1_4048", "Return type of index signature from exported interface has or is using name '{0}' from private module '{1}'."),
  Return_type_of_index_signature_from_exported_interface_has_or_is_using_private_name_0: diag(4049, 1 /* Error */, "Return_type_of_index_signature_from_exported_interface_has_or_is_using_private_name_0_4049", "Return type of index signature from exported interface has or is using private name '{0}'."),
  Return_type_of_public_static_method_from_exported_class_has_or_is_using_name_0_from_external_module_1_but_cannot_be_named: diag(4050, 1 /* Error */, "Return_type_of_public_static_method_from_exported_class_has_or_is_using_name_0_from_external_module__4050", "Return type of public static method from exported class has or is using name '{0}' from external module {1} but cannot be named."),
  Return_type_of_public_static_method_from_exported_class_has_or_is_using_name_0_from_private_module_1: diag(4051, 1 /* Error */, "Return_type_of_public_static_method_from_exported_class_has_or_is_using_name_0_from_private_module_1_4051", "Return type of public static method from exported class has or is using name '{0}' from private module '{1}'."),
  Return_type_of_public_static_method_from_exported_class_has_or_is_using_private_name_0: diag(4052, 1 /* Error */, "Return_type_of_public_static_method_from_exported_class_has_or_is_using_private_name_0_4052", "Return type of public static method from exported class has or is using private name '{0}'."),
  Return_type_of_public_method_from_exported_class_has_or_is_using_name_0_from_external_module_1_but_cannot_be_named: diag(4053, 1 /* Error */, "Return_type_of_public_method_from_exported_class_has_or_is_using_name_0_from_external_module_1_but_c_4053", "Return type of public method from exported class has or is using name '{0}' from external module {1} but cannot be named."),
  Return_type_of_public_method_from_exported_class_has_or_is_using_name_0_from_private_module_1: diag(4054, 1 /* Error */, "Return_type_of_public_method_from_exported_class_has_or_is_using_name_0_from_private_module_1_4054", "Return type of public method from exported class has or is using name '{0}' from private module '{1}'."),
  Return_type_of_public_method_from_exported_class_has_or_is_using_private_name_0: diag(4055, 1 /* Error */, "Return_type_of_public_method_from_exported_class_has_or_is_using_private_name_0_4055", "Return type of public method from exported class has or is using private name '{0}'."),
  Return_type_of_method_from_exported_interface_has_or_is_using_name_0_from_private_module_1: diag(4056, 1 /* Error */, "Return_type_of_method_from_exported_interface_has_or_is_using_name_0_from_private_module_1_4056", "Return type of method from exported interface has or is using name '{0}' from private module '{1}'."),
  Return_type_of_method_from_exported_interface_has_or_is_using_private_name_0: diag(4057, 1 /* Error */, "Return_type_of_method_from_exported_interface_has_or_is_using_private_name_0_4057", "Return type of method from exported interface has or is using private name '{0}'."),
  Return_type_of_exported_function_has_or_is_using_name_0_from_external_module_1_but_cannot_be_named: diag(4058, 1 /* Error */, "Return_type_of_exported_function_has_or_is_using_name_0_from_external_module_1_but_cannot_be_named_4058", "Return type of exported function has or is using name '{0}' from external module {1} but cannot be named."),
  Return_type_of_exported_function_has_or_is_using_name_0_from_private_module_1: diag(4059, 1 /* Error */, "Return_type_of_exported_function_has_or_is_using_name_0_from_private_module_1_4059", "Return type of exported function has or is using name '{0}' from private module '{1}'."),
  Return_type_of_exported_function_has_or_is_using_private_name_0: diag(4060, 1 /* Error */, "Return_type_of_exported_function_has_or_is_using_private_name_0_4060", "Return type of exported function has or is using private name '{0}'."),
  Parameter_0_of_constructor_from_exported_class_has_or_is_using_name_1_from_external_module_2_but_cannot_be_named: diag(4061, 1 /* Error */, "Parameter_0_of_constructor_from_exported_class_has_or_is_using_name_1_from_external_module_2_but_can_4061", "Parameter '{0}' of constructor from exported class has or is using name '{1}' from external module {2} but cannot be named."),
  Parameter_0_of_constructor_from_exported_class_has_or_is_using_name_1_from_private_module_2: diag(4062, 1 /* Error */, "Parameter_0_of_constructor_from_exported_class_has_or_is_using_name_1_from_private_module_2_4062", "Parameter '{0}' of constructor from exported class has or is using name '{1}' from private module '{2}'."),
  Parameter_0_of_constructor_from_exported_class_has_or_is_using_private_name_1: diag(4063, 1 /* Error */, "Parameter_0_of_constructor_from_exported_class_has_or_is_using_private_name_1_4063", "Parameter '{0}' of constructor from exported class has or is using private name '{1}'."),
  Parameter_0_of_constructor_signature_from_exported_interface_has_or_is_using_name_1_from_private_module_2: diag(4064, 1 /* Error */, "Parameter_0_of_constructor_signature_from_exported_interface_has_or_is_using_name_1_from_private_mod_4064", "Parameter '{0}' of constructor signature from exported interface has or is using name '{1}' from private module '{2}'."),
  Parameter_0_of_constructor_signature_from_exported_interface_has_or_is_using_private_name_1: diag(4065, 1 /* Error */, "Parameter_0_of_constructor_signature_from_exported_interface_has_or_is_using_private_name_1_4065", "Parameter '{0}' of constructor signature from exported interface has or is using private name '{1}'."),
  Parameter_0_of_call_signature_from_exported_interface_has_or_is_using_name_1_from_private_module_2: diag(4066, 1 /* Error */, "Parameter_0_of_call_signature_from_exported_interface_has_or_is_using_name_1_from_private_module_2_4066", "Parameter '{0}' of call signature from exported interface has or is using name '{1}' from private module '{2}'."),
  Parameter_0_of_call_signature_from_exported_interface_has_or_is_using_private_name_1: diag(4067, 1 /* Error */, "Parameter_0_of_call_signature_from_exported_interface_has_or_is_using_private_name_1_4067", "Parameter '{0}' of call signature from exported interface has or is using private name '{1}'."),
  Parameter_0_of_public_static_method_from_exported_class_has_or_is_using_name_1_from_external_module_2_but_cannot_be_named: diag(4068, 1 /* Error */, "Parameter_0_of_public_static_method_from_exported_class_has_or_is_using_name_1_from_external_module__4068", "Parameter '{0}' of public static method from exported class has or is using name '{1}' from external module {2} but cannot be named."),
  Parameter_0_of_public_static_method_from_exported_class_has_or_is_using_name_1_from_private_module_2: diag(4069, 1 /* Error */, "Parameter_0_of_public_static_method_from_exported_class_has_or_is_using_name_1_from_private_module_2_4069", "Parameter '{0}' of public static method from exported class has or is using name '{1}' from private module '{2}'."),
  Parameter_0_of_public_static_method_from_exported_class_has_or_is_using_private_name_1: diag(4070, 1 /* Error */, "Parameter_0_of_public_static_method_from_exported_class_has_or_is_using_private_name_1_4070", "Parameter '{0}' of public static method from exported class has or is using private name '{1}'."),
  Parameter_0_of_public_method_from_exported_class_has_or_is_using_name_1_from_external_module_2_but_cannot_be_named: diag(4071, 1 /* Error */, "Parameter_0_of_public_method_from_exported_class_has_or_is_using_name_1_from_external_module_2_but_c_4071", "Parameter '{0}' of public method from exported class has or is using name '{1}' from external module {2} but cannot be named."),
  Parameter_0_of_public_method_from_exported_class_has_or_is_using_name_1_from_private_module_2: diag(4072, 1 /* Error */, "Parameter_0_of_public_method_from_exported_class_has_or_is_using_name_1_from_private_module_2_4072", "Parameter '{0}' of public method from exported class has or is using name '{1}' from private module '{2}'."),
  Parameter_0_of_public_method_from_exported_class_has_or_is_using_private_name_1: diag(4073, 1 /* Error */, "Parameter_0_of_public_method_from_exported_class_has_or_is_using_private_name_1_4073", "Parameter '{0}' of public method from exported class has or is using private name '{1}'."),
  Parameter_0_of_method_from_exported_interface_has_or_is_using_name_1_from_private_module_2: diag(4074, 1 /* Error */, "Parameter_0_of_method_from_exported_interface_has_or_is_using_name_1_from_private_module_2_4074", "Parameter '{0}' of method from exported interface has or is using name '{1}' from private module '{2}'."),
  Parameter_0_of_method_from_exported_interface_has_or_is_using_private_name_1: diag(4075, 1 /* Error */, "Parameter_0_of_method_from_exported_interface_has_or_is_using_private_name_1_4075", "Parameter '{0}' of method from exported interface has or is using private name '{1}'."),
  Parameter_0_of_exported_function_has_or_is_using_name_1_from_external_module_2_but_cannot_be_named: diag(4076, 1 /* Error */, "Parameter_0_of_exported_function_has_or_is_using_name_1_from_external_module_2_but_cannot_be_named_4076", "Parameter '{0}' of exported function has or is using name '{1}' from external module {2} but cannot be named."),
  Parameter_0_of_exported_function_has_or_is_using_name_1_from_private_module_2: diag(4077, 1 /* Error */, "Parameter_0_of_exported_function_has_or_is_using_name_1_from_private_module_2_4077", "Parameter '{0}' of exported function has or is using name '{1}' from private module '{2}'."),
  Parameter_0_of_exported_function_has_or_is_using_private_name_1: diag(4078, 1 /* Error */, "Parameter_0_of_exported_function_has_or_is_using_private_name_1_4078", "Parameter '{0}' of exported function has or is using private name '{1}'."),
  Exported_type_alias_0_has_or_is_using_private_name_1: diag(4081, 1 /* Error */, "Exported_type_alias_0_has_or_is_using_private_name_1_4081", "Exported type alias '{0}' has or is using private name '{1}'."),
  Default_export_of_the_module_has_or_is_using_private_name_0: diag(4082, 1 /* Error */, "Default_export_of_the_module_has_or_is_using_private_name_0_4082", "Default export of the module has or is using private name '{0}'."),
  Type_parameter_0_of_exported_type_alias_has_or_is_using_private_name_1: diag(4083, 1 /* Error */, "Type_parameter_0_of_exported_type_alias_has_or_is_using_private_name_1_4083", "Type parameter '{0}' of exported type alias has or is using private name '{1}'."),
  Exported_type_alias_0_has_or_is_using_private_name_1_from_module_2: diag(4084, 1 /* Error */, "Exported_type_alias_0_has_or_is_using_private_name_1_from_module_2_4084", "Exported type alias '{0}' has or is using private name '{1}' from module {2}."),
  Extends_clause_for_inferred_type_0_has_or_is_using_private_name_1: diag(4085, 1 /* Error */, "Extends_clause_for_inferred_type_0_has_or_is_using_private_name_1_4085", "Extends clause for inferred type '{0}' has or is using private name '{1}'."),
  Parameter_0_of_index_signature_from_exported_interface_has_or_is_using_name_1_from_private_module_2: diag(4091, 1 /* Error */, "Parameter_0_of_index_signature_from_exported_interface_has_or_is_using_name_1_from_private_module_2_4091", "Parameter '{0}' of index signature from exported interface has or is using name '{1}' from private module '{2}'."),
  Parameter_0_of_index_signature_from_exported_interface_has_or_is_using_private_name_1: diag(4092, 1 /* Error */, "Parameter_0_of_index_signature_from_exported_interface_has_or_is_using_private_name_1_4092", "Parameter '{0}' of index signature from exported interface has or is using private name '{1}'."),
  Property_0_of_exported_anonymous_class_type_may_not_be_private_or_protected: diag(4094, 1 /* Error */, "Property_0_of_exported_anonymous_class_type_may_not_be_private_or_protected_4094", "Property '{0}' of exported anonymous class type may not be private or protected."),
  Public_static_method_0_of_exported_class_has_or_is_using_name_1_from_external_module_2_but_cannot_be_named: diag(4095, 1 /* Error */, "Public_static_method_0_of_exported_class_has_or_is_using_name_1_from_external_module_2_but_cannot_be_4095", "Public static method '{0}' of exported class has or is using name '{1}' from external module {2} but cannot be named."),
  Public_static_method_0_of_exported_class_has_or_is_using_name_1_from_private_module_2: diag(4096, 1 /* Error */, "Public_static_method_0_of_exported_class_has_or_is_using_name_1_from_private_module_2_4096", "Public static method '{0}' of exported class has or is using name '{1}' from private module '{2}'."),
  Public_static_method_0_of_exported_class_has_or_is_using_private_name_1: diag(4097, 1 /* Error */, "Public_static_method_0_of_exported_class_has_or_is_using_private_name_1_4097", "Public static method '{0}' of exported class has or is using private name '{1}'."),
  Public_method_0_of_exported_class_has_or_is_using_name_1_from_external_module_2_but_cannot_be_named: diag(4098, 1 /* Error */, "Public_method_0_of_exported_class_has_or_is_using_name_1_from_external_module_2_but_cannot_be_named_4098", "Public method '{0}' of exported class has or is using name '{1}' from external module {2} but cannot be named."),
  Public_method_0_of_exported_class_has_or_is_using_name_1_from_private_module_2: diag(4099, 1 /* Error */, "Public_method_0_of_exported_class_has_or_is_using_name_1_from_private_module_2_4099", "Public method '{0}' of exported class has or is using name '{1}' from private module '{2}'."),
  Public_method_0_of_exported_class_has_or_is_using_private_name_1: diag(4100, 1 /* Error */, "Public_method_0_of_exported_class_has_or_is_using_private_name_1_4100", "Public method '{0}' of exported class has or is using private name '{1}'."),
  Method_0_of_exported_interface_has_or_is_using_name_1_from_private_module_2: diag(4101, 1 /* Error */, "Method_0_of_exported_interface_has_or_is_using_name_1_from_private_module_2_4101", "Method '{0}' of exported interface has or is using name '{1}' from private module '{2}'."),
  Method_0_of_exported_interface_has_or_is_using_private_name_1: diag(4102, 1 /* Error */, "Method_0_of_exported_interface_has_or_is_using_private_name_1_4102", "Method '{0}' of exported interface has or is using private name '{1}'."),
  Type_parameter_0_of_exported_mapped_object_type_is_using_private_name_1: diag(4103, 1 /* Error */, "Type_parameter_0_of_exported_mapped_object_type_is_using_private_name_1_4103", "Type parameter '{0}' of exported mapped object type is using private name '{1}'."),
  The_type_0_is_readonly_and_cannot_be_assigned_to_the_mutable_type_1: diag(4104, 1 /* Error */, "The_type_0_is_readonly_and_cannot_be_assigned_to_the_mutable_type_1_4104", "The type '{0}' is 'readonly' and cannot be assigned to the mutable type '{1}'."),
  Private_or_protected_member_0_cannot_be_accessed_on_a_type_parameter: diag(4105, 1 /* Error */, "Private_or_protected_member_0_cannot_be_accessed_on_a_type_parameter_4105", "Private or protected member '{0}' cannot be accessed on a type parameter."),
  Parameter_0_of_accessor_has_or_is_using_private_name_1: diag(4106, 1 /* Error */, "Parameter_0_of_accessor_has_or_is_using_private_name_1_4106", "Parameter '{0}' of accessor has or is using private name '{1}'."),
  Parameter_0_of_accessor_has_or_is_using_name_1_from_private_module_2: diag(4107, 1 /* Error */, "Parameter_0_of_accessor_has_or_is_using_name_1_from_private_module_2_4107", "Parameter '{0}' of accessor has or is using name '{1}' from private module '{2}'."),
  Parameter_0_of_accessor_has_or_is_using_name_1_from_external_module_2_but_cannot_be_named: diag(4108, 1 /* Error */, "Parameter_0_of_accessor_has_or_is_using_name_1_from_external_module_2_but_cannot_be_named_4108", "Parameter '{0}' of accessor has or is using name '{1}' from external module '{2}' but cannot be named."),
  Type_arguments_for_0_circularly_reference_themselves: diag(4109, 1 /* Error */, "Type_arguments_for_0_circularly_reference_themselves_4109", "Type arguments for '{0}' circularly reference themselves."),
  Tuple_type_arguments_circularly_reference_themselves: diag(4110, 1 /* Error */, "Tuple_type_arguments_circularly_reference_themselves_4110", "Tuple type arguments circularly reference themselves."),
  Property_0_comes_from_an_index_signature_so_it_must_be_accessed_with_0: diag(4111, 1 /* Error */, "Property_0_comes_from_an_index_signature_so_it_must_be_accessed_with_0_4111", "Property '{0}' comes from an index signature, so it must be accessed with ['{0}']."),
  This_member_cannot_have_an_override_modifier_because_its_containing_class_0_does_not_extend_another_class: diag(4112, 1 /* Error */, "This_member_cannot_have_an_override_modifier_because_its_containing_class_0_does_not_extend_another__4112", "This member cannot have an 'override' modifier because its containing class '{0}' does not extend another class."),
  This_member_cannot_have_an_override_modifier_because_it_is_not_declared_in_the_base_class_0: diag(4113, 1 /* Error */, "This_member_cannot_have_an_override_modifier_because_it_is_not_declared_in_the_base_class_0_4113", "This member cannot have an 'override' modifier because it is not declared in the base class '{0}'."),
  This_member_must_have_an_override_modifier_because_it_overrides_a_member_in_the_base_class_0: diag(4114, 1 /* Error */, "This_member_must_have_an_override_modifier_because_it_overrides_a_member_in_the_base_class_0_4114", "This member must have an 'override' modifier because it overrides a member in the base class '{0}'."),
  This_parameter_property_must_have_an_override_modifier_because_it_overrides_a_member_in_base_class_0: diag(4115, 1 /* Error */, "This_parameter_property_must_have_an_override_modifier_because_it_overrides_a_member_in_base_class_0_4115", "This parameter property must have an 'override' modifier because it overrides a member in base class '{0}'."),
  This_member_must_have_an_override_modifier_because_it_overrides_an_abstract_method_that_is_declared_in_the_base_class_0: diag(4116, 1 /* Error */, "This_member_must_have_an_override_modifier_because_it_overrides_an_abstract_method_that_is_declared__4116", "This member must have an 'override' modifier because it overrides an abstract method that is declared in the base class '{0}'."),
  This_member_cannot_have_an_override_modifier_because_it_is_not_declared_in_the_base_class_0_Did_you_mean_1: diag(4117, 1 /* Error */, "This_member_cannot_have_an_override_modifier_because_it_is_not_declared_in_the_base_class_0_Did_you__4117", "This member cannot have an 'override' modifier because it is not declared in the base class '{0}'. Did you mean '{1}'?"),
  The_type_of_this_node_cannot_be_serialized_because_its_property_0_cannot_be_serialized: diag(4118, 1 /* Error */, "The_type_of_this_node_cannot_be_serialized_because_its_property_0_cannot_be_serialized_4118", "The type of this node cannot be serialized because its property '{0}' cannot be serialized."),
  This_member_must_have_a_JSDoc_comment_with_an_override_tag_because_it_overrides_a_member_in_the_base_class_0: diag(4119, 1 /* Error */, "This_member_must_have_a_JSDoc_comment_with_an_override_tag_because_it_overrides_a_member_in_the_base_4119", "This member must have a JSDoc comment with an '@override' tag because it overrides a member in the base class '{0}'."),
  This_parameter_property_must_have_a_JSDoc_comment_with_an_override_tag_because_it_overrides_a_member_in_the_base_class_0: diag(4120, 1 /* Error */, "This_parameter_property_must_have_a_JSDoc_comment_with_an_override_tag_because_it_overrides_a_member_4120", "This parameter property must have a JSDoc comment with an '@override' tag because it overrides a member in the base class '{0}'."),
  This_member_cannot_have_a_JSDoc_comment_with_an_override_tag_because_its_containing_class_0_does_not_extend_another_class: diag(4121, 1 /* Error */, "This_member_cannot_have_a_JSDoc_comment_with_an_override_tag_because_its_containing_class_0_does_not_4121", "This member cannot have a JSDoc comment with an '@override' tag because its containing class '{0}' does not extend another class."),
  This_member_cannot_have_a_JSDoc_comment_with_an_override_tag_because_it_is_not_declared_in_the_base_class_0: diag(4122, 1 /* Error */, "This_member_cannot_have_a_JSDoc_comment_with_an_override_tag_because_it_is_not_declared_in_the_base__4122", "This member cannot have a JSDoc comment with an '@override' tag because it is not declared in the base class '{0}'."),
  This_member_cannot_have_a_JSDoc_comment_with_an_override_tag_because_it_is_not_declared_in_the_base_class_0_Did_you_mean_1: diag(4123, 1 /* Error */, "This_member_cannot_have_a_JSDoc_comment_with_an_override_tag_because_it_is_not_declared_in_the_base__4123", "This member cannot have a JSDoc comment with an 'override' tag because it is not declared in the base class '{0}'. Did you mean '{1}'?"),
  Compiler_option_0_of_value_1_is_unstable_Use_nightly_TypeScript_to_silence_this_error_Try_updating_with_npm_install_D_typescript_next: diag(4124, 1 /* Error */, "Compiler_option_0_of_value_1_is_unstable_Use_nightly_TypeScript_to_silence_this_error_Try_updating_w_4124", "Compiler option '{0}' of value '{1}' is unstable. Use nightly TypeScript to silence this error. Try updating with 'npm install -D typescript@next'."),
  Each_declaration_of_0_1_differs_in_its_value_where_2_was_expected_but_3_was_given: diag(4125, 1 /* Error */, "Each_declaration_of_0_1_differs_in_its_value_where_2_was_expected_but_3_was_given_4125", "Each declaration of '{0}.{1}' differs in its value, where '{2}' was expected but '{3}' was given."),
  One_value_of_0_1_is_the_string_2_and_the_other_is_assumed_to_be_an_unknown_numeric_value: diag(4126, 1 /* Error */, "One_value_of_0_1_is_the_string_2_and_the_other_is_assumed_to_be_an_unknown_numeric_value_4126", "One value of '{0}.{1}' is the string '{2}', and the other is assumed to be an unknown numeric value."),
  The_current_host_does_not_support_the_0_option: diag(5001, 1 /* Error */, "The_current_host_does_not_support_the_0_option_5001", "The current host does not support the '{0}' option."),
  Cannot_find_the_common_subdirectory_path_for_the_input_files: diag(5009, 1 /* Error */, "Cannot_find_the_common_subdirectory_path_for_the_input_files_5009", "Cannot find the common subdirectory path for the input files."),
  File_specification_cannot_end_in_a_recursive_directory_wildcard_Asterisk_Asterisk_Colon_0: diag(5010, 1 /* Error */, "File_specification_cannot_end_in_a_recursive_directory_wildcard_Asterisk_Asterisk_Colon_0_5010", "File specification cannot end in a recursive directory wildcard ('**'): '{0}'."),
  Cannot_read_file_0_Colon_1: diag(5012, 1 /* Error */, "Cannot_read_file_0_Colon_1_5012", "Cannot read file '{0}': {1}."),
  Unknown_compiler_option_0: diag(5023, 1 /* Error */, "Unknown_compiler_option_0_5023", "Unknown compiler option '{0}'."),
  Compiler_option_0_requires_a_value_of_type_1: diag(5024, 1 /* Error */, "Compiler_option_0_requires_a_value_of_type_1_5024", "Compiler option '{0}' requires a value of type {1}."),
  Unknown_compiler_option_0_Did_you_mean_1: diag(5025, 1 /* Error */, "Unknown_compiler_option_0_Did_you_mean_1_5025", "Unknown compiler option '{0}'. Did you mean '{1}'?"),
  Could_not_write_file_0_Colon_1: diag(5033, 1 /* Error */, "Could_not_write_file_0_Colon_1_5033", "Could not write file '{0}': {1}."),
  Option_project_cannot_be_mixed_with_source_files_on_a_command_line: diag(5042, 1 /* Error */, "Option_project_cannot_be_mixed_with_source_files_on_a_command_line_5042", "Option 'project' cannot be mixed with source files on a command line."),
  Option_isolatedModules_can_only_be_used_when_either_option_module_is_provided_or_option_target_is_ES2015_or_higher: diag(5047, 1 /* Error */, "Option_isolatedModules_can_only_be_used_when_either_option_module_is_provided_or_option_target_is_ES_5047", "Option 'isolatedModules' can only be used when either option '--module' is provided or option 'target' is 'ES2015' or higher."),
  Option_0_can_only_be_used_when_either_option_inlineSourceMap_or_option_sourceMap_is_provided: diag(5051, 1 /* Error */, "Option_0_can_only_be_used_when_either_option_inlineSourceMap_or_option_sourceMap_is_provided_5051", "Option '{0} can only be used when either option '--inlineSourceMap' or option '--sourceMap' is provided."),
  Option_0_cannot_be_specified_without_specifying_option_1: diag(5052, 1 /* Error */, "Option_0_cannot_be_specified_without_specifying_option_1_5052", "Option '{0}' cannot be specified without specifying option '{1}'."),
  Option_0_cannot_be_specified_with_option_1: diag(5053, 1 /* Error */, "Option_0_cannot_be_specified_with_option_1_5053", "Option '{0}' cannot be specified with option '{1}'."),
  A_tsconfig_json_file_is_already_defined_at_Colon_0: diag(5054, 1 /* Error */, "A_tsconfig_json_file_is_already_defined_at_Colon_0_5054", "A 'tsconfig.json' file is already defined at: '{0}'."),
  Cannot_write_file_0_because_it_would_overwrite_input_file: diag(5055, 1 /* Error */, "Cannot_write_file_0_because_it_would_overwrite_input_file_5055", "Cannot write file '{0}' because it would overwrite input file."),
  Cannot_write_file_0_because_it_would_be_overwritten_by_multiple_input_files: diag(5056, 1 /* Error */, "Cannot_write_file_0_because_it_would_be_overwritten_by_multiple_input_files_5056", "Cannot write file '{0}' because it would be overwritten by multiple input files."),
  Cannot_find_a_tsconfig_json_file_at_the_specified_directory_Colon_0: diag(5057, 1 /* Error */, "Cannot_find_a_tsconfig_json_file_at_the_specified_directory_Colon_0_5057", "Cannot find a tsconfig.json file at the specified directory: '{0}'."),
  The_specified_path_does_not_exist_Colon_0: diag(5058, 1 /* Error */, "The_specified_path_does_not_exist_Colon_0_5058", "The specified path does not exist: '{0}'."),
  Invalid_value_for_reactNamespace_0_is_not_a_valid_identifier: diag(5059, 1 /* Error */, "Invalid_value_for_reactNamespace_0_is_not_a_valid_identifier_5059", "Invalid value for '--reactNamespace'. '{0}' is not a valid identifier."),
  Pattern_0_can_have_at_most_one_Asterisk_character: diag(5061, 1 /* Error */, "Pattern_0_can_have_at_most_one_Asterisk_character_5061", "Pattern '{0}' can have at most one '*' character."),
  Substitution_0_in_pattern_1_can_have_at_most_one_Asterisk_character: diag(5062, 1 /* Error */, "Substitution_0_in_pattern_1_can_have_at_most_one_Asterisk_character_5062", "Substitution '{0}' in pattern '{1}' can have at most one '*' character."),
  Substitutions_for_pattern_0_should_be_an_array: diag(5063, 1 /* Error */, "Substitutions_for_pattern_0_should_be_an_array_5063", "Substitutions for pattern '{0}' should be an array."),
  Substitution_0_for_pattern_1_has_incorrect_type_expected_string_got_2: diag(5064, 1 /* Error */, "Substitution_0_for_pattern_1_has_incorrect_type_expected_string_got_2_5064", "Substitution '{0}' for pattern '{1}' has incorrect type, expected 'string', got '{2}'."),
  File_specification_cannot_contain_a_parent_directory_that_appears_after_a_recursive_directory_wildcard_Asterisk_Asterisk_Colon_0: diag(5065, 1 /* Error */, "File_specification_cannot_contain_a_parent_directory_that_appears_after_a_recursive_directory_wildca_5065", "File specification cannot contain a parent directory ('..') that appears after a recursive directory wildcard ('**'): '{0}'."),
  Substitutions_for_pattern_0_shouldn_t_be_an_empty_array: diag(5066, 1 /* Error */, "Substitutions_for_pattern_0_shouldn_t_be_an_empty_array_5066", "Substitutions for pattern '{0}' shouldn't be an empty array."),
  Invalid_value_for_jsxFactory_0_is_not_a_valid_identifier_or_qualified_name: diag(5067, 1 /* Error */, "Invalid_value_for_jsxFactory_0_is_not_a_valid_identifier_or_qualified_name_5067", "Invalid value for 'jsxFactory'. '{0}' is not a valid identifier or qualified-name."),
  Adding_a_tsconfig_json_file_will_help_organize_projects_that_contain_both_TypeScript_and_JavaScript_files_Learn_more_at_https_Colon_Slash_Slashaka_ms_Slashtsconfig: diag(5068, 1 /* Error */, "Adding_a_tsconfig_json_file_will_help_organize_projects_that_contain_both_TypeScript_and_JavaScript__5068", "Adding a tsconfig.json file will help organize projects that contain both TypeScript and JavaScript files. Learn more at https://aka.ms/tsconfig."),
  Option_0_cannot_be_specified_without_specifying_option_1_or_option_2: diag(5069, 1 /* Error */, "Option_0_cannot_be_specified_without_specifying_option_1_or_option_2_5069", "Option '{0}' cannot be specified without specifying option '{1}' or option '{2}'."),
  Option_resolveJsonModule_cannot_be_specified_when_moduleResolution_is_set_to_classic: diag(5070, 1 /* Error */, "Option_resolveJsonModule_cannot_be_specified_when_moduleResolution_is_set_to_classic_5070", "Option '--resolveJsonModule' cannot be specified when 'moduleResolution' is set to 'classic'."),
  Option_resolveJsonModule_cannot_be_specified_when_module_is_set_to_none_system_or_umd: diag(5071, 1 /* Error */, "Option_resolveJsonModule_cannot_be_specified_when_module_is_set_to_none_system_or_umd_5071", "Option '--resolveJsonModule' cannot be specified when 'module' is set to 'none', 'system', or 'umd'."),
  Unknown_build_option_0: diag(5072, 1 /* Error */, "Unknown_build_option_0_5072", "Unknown build option '{0}'."),
  Build_option_0_requires_a_value_of_type_1: diag(5073, 1 /* Error */, "Build_option_0_requires_a_value_of_type_1_5073", "Build option '{0}' requires a value of type {1}."),
  Option_incremental_can_only_be_specified_using_tsconfig_emitting_to_single_file_or_when_option_tsBuildInfoFile_is_specified: diag(5074, 1 /* Error */, "Option_incremental_can_only_be_specified_using_tsconfig_emitting_to_single_file_or_when_option_tsBui_5074", "Option '--incremental' can only be specified using tsconfig, emitting to single file or when option '--tsBuildInfoFile' is specified."),
  _0_is_assignable_to_the_constraint_of_type_1_but_1_could_be_instantiated_with_a_different_subtype_of_constraint_2: diag(5075, 1 /* Error */, "_0_is_assignable_to_the_constraint_of_type_1_but_1_could_be_instantiated_with_a_different_subtype_of_5075", "'{0}' is assignable to the constraint of type '{1}', but '{1}' could be instantiated with a different subtype of constraint '{2}'."),
  _0_and_1_operations_cannot_be_mixed_without_parentheses: diag(5076, 1 /* Error */, "_0_and_1_operations_cannot_be_mixed_without_parentheses_5076", "'{0}' and '{1}' operations cannot be mixed without parentheses."),
  Unknown_build_option_0_Did_you_mean_1: diag(5077, 1 /* Error */, "Unknown_build_option_0_Did_you_mean_1_5077", "Unknown build option '{0}'. Did you mean '{1}'?"),
  Unknown_watch_option_0: diag(5078, 1 /* Error */, "Unknown_watch_option_0_5078", "Unknown watch option '{0}'."),
  Unknown_watch_option_0_Did_you_mean_1: diag(5079, 1 /* Error */, "Unknown_watch_option_0_Did_you_mean_1_5079", "Unknown watch option '{0}'. Did you mean '{1}'?"),
  Watch_option_0_requires_a_value_of_type_1: diag(5080, 1 /* Error */, "Watch_option_0_requires_a_value_of_type_1_5080", "Watch option '{0}' requires a value of type {1}."),
  Cannot_find_a_tsconfig_json_file_at_the_current_directory_Colon_0: diag(5081, 1 /* Error */, "Cannot_find_a_tsconfig_json_file_at_the_current_directory_Colon_0_5081", "Cannot find a tsconfig.json file at the current directory: {0}."),
  _0_could_be_instantiated_with_an_arbitrary_type_which_could_be_unrelated_to_1: diag(5082, 1 /* Error */, "_0_could_be_instantiated_with_an_arbitrary_type_which_could_be_unrelated_to_1_5082", "'{0}' could be instantiated with an arbitrary type which could be unrelated to '{1}'."),
  Cannot_read_file_0: diag(5083, 1 /* Error */, "Cannot_read_file_0_5083", "Cannot read file '{0}'."),
  A_tuple_member_cannot_be_both_optional_and_rest: diag(5085, 1 /* Error */, "A_tuple_member_cannot_be_both_optional_and_rest_5085", "A tuple member cannot be both optional and rest."),
  A_labeled_tuple_element_is_declared_as_optional_with_a_question_mark_after_the_name_and_before_the_colon_rather_than_after_the_type: diag(5086, 1 /* Error */, "A_labeled_tuple_element_is_declared_as_optional_with_a_question_mark_after_the_name_and_before_the_c_5086", "A labeled tuple element is declared as optional with a question mark after the name and before the colon, rather than after the type."),
  A_labeled_tuple_element_is_declared_as_rest_with_a_before_the_name_rather_than_before_the_type: diag(5087, 1 /* Error */, "A_labeled_tuple_element_is_declared_as_rest_with_a_before_the_name_rather_than_before_the_type_5087", "A labeled tuple element is declared as rest with a '...' before the name, rather than before the type."),
  The_inferred_type_of_0_references_a_type_with_a_cyclic_structure_which_cannot_be_trivially_serialized_A_type_annotation_is_necessary: diag(5088, 1 /* Error */, "The_inferred_type_of_0_references_a_type_with_a_cyclic_structure_which_cannot_be_trivially_serialize_5088", "The inferred type of '{0}' references a type with a cyclic structure which cannot be trivially serialized. A type annotation is necessary."),
  Option_0_cannot_be_specified_when_option_jsx_is_1: diag(5089, 1 /* Error */, "Option_0_cannot_be_specified_when_option_jsx_is_1_5089", "Option '{0}' cannot be specified when option 'jsx' is '{1}'."),
  Non_relative_paths_are_not_allowed_when_baseUrl_is_not_set_Did_you_forget_a_leading_Slash: diag(5090, 1 /* Error */, "Non_relative_paths_are_not_allowed_when_baseUrl_is_not_set_Did_you_forget_a_leading_Slash_5090", "Non-relative paths are not allowed when 'baseUrl' is not set. Did you forget a leading './'?"),
  Option_preserveConstEnums_cannot_be_disabled_when_0_is_enabled: diag(5091, 1 /* Error */, "Option_preserveConstEnums_cannot_be_disabled_when_0_is_enabled_5091", "Option 'preserveConstEnums' cannot be disabled when '{0}' is enabled."),
  The_root_value_of_a_0_file_must_be_an_object: diag(5092, 1 /* Error */, "The_root_value_of_a_0_file_must_be_an_object_5092", "The root value of a '{0}' file must be an object."),
  Compiler_option_0_may_only_be_used_with_build: diag(5093, 1 /* Error */, "Compiler_option_0_may_only_be_used_with_build_5093", "Compiler option '--{0}' may only be used with '--build'."),
  Compiler_option_0_may_not_be_used_with_build: diag(5094, 1 /* Error */, "Compiler_option_0_may_not_be_used_with_build_5094", "Compiler option '--{0}' may not be used with '--build'."),
  Option_0_can_only_be_used_when_module_is_set_to_preserve_or_to_es2015_or_later: diag(5095, 1 /* Error */, "Option_0_can_only_be_used_when_module_is_set_to_preserve_or_to_es2015_or_later_5095", "Option '{0}' can only be used when 'module' is set to 'preserve' or to 'es2015' or later."),
  Option_allowImportingTsExtensions_can_only_be_used_when_either_noEmit_or_emitDeclarationOnly_is_set: diag(5096, 1 /* Error */, "Option_allowImportingTsExtensions_can_only_be_used_when_either_noEmit_or_emitDeclarationOnly_is_set_5096", "Option 'allowImportingTsExtensions' can only be used when either 'noEmit' or 'emitDeclarationOnly' is set."),
  An_import_path_can_only_end_with_a_0_extension_when_allowImportingTsExtensions_is_enabled: diag(5097, 1 /* Error */, "An_import_path_can_only_end_with_a_0_extension_when_allowImportingTsExtensions_is_enabled_5097", "An import path can only end with a '{0}' extension when 'allowImportingTsExtensions' is enabled."),
  Option_0_can_only_be_used_when_moduleResolution_is_set_to_node16_nodenext_or_bundler: diag(5098, 1 /* Error */, "Option_0_can_only_be_used_when_moduleResolution_is_set_to_node16_nodenext_or_bundler_5098", "Option '{0}' can only be used when 'moduleResolution' is set to 'node16', 'nodenext', or 'bundler'."),
  Option_0_is_deprecated_and_will_stop_functioning_in_TypeScript_1_Specify_compilerOption_ignoreDeprecations_Colon_2_to_silence_this_error: diag(5101, 1 /* Error */, "Option_0_is_deprecated_and_will_stop_functioning_in_TypeScript_1_Specify_compilerOption_ignoreDeprec_5101", `Option '{0}' is deprecated and will stop functioning in TypeScript {1}. Specify compilerOption '"ignoreDeprecations": "{2}"' to silence this error.`),
  Option_0_has_been_removed_Please_remove_it_from_your_configuration: diag(5102, 1 /* Error */, "Option_0_has_been_removed_Please_remove_it_from_your_configuration_5102", "Option '{0}' has been removed. Please remove it from your configuration."),
  Invalid_value_for_ignoreDeprecations: diag(5103, 1 /* Error */, "Invalid_value_for_ignoreDeprecations_5103", "Invalid value for '--ignoreDeprecations'."),
  Option_0_is_redundant_and_cannot_be_specified_with_option_1: diag(5104, 1 /* Error */, "Option_0_is_redundant_and_cannot_be_specified_with_option_1_5104", "Option '{0}' is redundant and cannot be specified with option '{1}'."),
  Option_verbatimModuleSyntax_cannot_be_used_when_module_is_set_to_UMD_AMD_or_System: diag(5105, 1 /* Error */, "Option_verbatimModuleSyntax_cannot_be_used_when_module_is_set_to_UMD_AMD_or_System_5105", "Option 'verbatimModuleSyntax' cannot be used when 'module' is set to 'UMD', 'AMD', or 'System'."),
  Use_0_instead: diag(5106, 3 /* Message */, "Use_0_instead_5106", "Use '{0}' instead."),
  Option_0_1_is_deprecated_and_will_stop_functioning_in_TypeScript_2_Specify_compilerOption_ignoreDeprecations_Colon_3_to_silence_this_error: diag(5107, 1 /* Error */, "Option_0_1_is_deprecated_and_will_stop_functioning_in_TypeScript_2_Specify_compilerOption_ignoreDepr_5107", `Option '{0}={1}' is deprecated and will stop functioning in TypeScript {2}. Specify compilerOption '"ignoreDeprecations": "{3}"' to silence this error.`),
  Option_0_1_has_been_removed_Please_remove_it_from_your_configuration: diag(5108, 1 /* Error */, "Option_0_1_has_been_removed_Please_remove_it_from_your_configuration_5108", "Option '{0}={1}' has been removed. Please remove it from your configuration."),
  Option_moduleResolution_must_be_set_to_0_or_left_unspecified_when_option_module_is_set_to_1: diag(5109, 1 /* Error */, "Option_moduleResolution_must_be_set_to_0_or_left_unspecified_when_option_module_is_set_to_1_5109", "Option 'moduleResolution' must be set to '{0}' (or left unspecified) when option 'module' is set to '{1}'."),
  Option_module_must_be_set_to_0_when_option_moduleResolution_is_set_to_1: diag(5110, 1 /* Error */, "Option_module_must_be_set_to_0_when_option_moduleResolution_is_set_to_1_5110", "Option 'module' must be set to '{0}' when option 'moduleResolution' is set to '{1}'."),
  Generates_a_sourcemap_for_each_corresponding_d_ts_file: diag(6e3, 3 /* Message */, "Generates_a_sourcemap_for_each_corresponding_d_ts_file_6000", "Generates a sourcemap for each corresponding '.d.ts' file."),
  Concatenate_and_emit_output_to_single_file: diag(6001, 3 /* Message */, "Concatenate_and_emit_output_to_single_file_6001", "Concatenate and emit output to single file."),
  Generates_corresponding_d_ts_file: diag(6002, 3 /* Message */, "Generates_corresponding_d_ts_file_6002", "Generates corresponding '.d.ts' file."),
  Specify_the_location_where_debugger_should_locate_TypeScript_files_instead_of_source_locations: diag(6004, 3 /* Message */, "Specify_the_location_where_debugger_should_locate_TypeScript_files_instead_of_source_locations_6004", "Specify the location where debugger should locate TypeScript files instead of source locations."),
  Watch_input_files: diag(6005, 3 /* Message */, "Watch_input_files_6005", "Watch input files."),
  Redirect_output_structure_to_the_directory: diag(6006, 3 /* Message */, "Redirect_output_structure_to_the_directory_6006", "Redirect output structure to the directory."),
  Do_not_erase_const_enum_declarations_in_generated_code: diag(6007, 3 /* Message */, "Do_not_erase_const_enum_declarations_in_generated_code_6007", "Do not erase const enum declarations in generated code."),
  Do_not_emit_outputs_if_any_errors_were_reported: diag(6008, 3 /* Message */, "Do_not_emit_outputs_if_any_errors_were_reported_6008", "Do not emit outputs if any errors were reported."),
  Do_not_emit_comments_to_output: diag(6009, 3 /* Message */, "Do_not_emit_comments_to_output_6009", "Do not emit comments to output."),
  Do_not_emit_outputs: diag(6010, 3 /* Message */, "Do_not_emit_outputs_6010", "Do not emit outputs."),
  Allow_default_imports_from_modules_with_no_default_export_This_does_not_affect_code_emit_just_typechecking: diag(6011, 3 /* Message */, "Allow_default_imports_from_modules_with_no_default_export_This_does_not_affect_code_emit_just_typech_6011", "Allow default imports from modules with no default export. This does not affect code emit, just typechecking."),
  Skip_type_checking_of_declaration_files: diag(6012, 3 /* Message */, "Skip_type_checking_of_declaration_files_6012", "Skip type checking of declaration files."),
  Do_not_resolve_the_real_path_of_symlinks: diag(6013, 3 /* Message */, "Do_not_resolve_the_real_path_of_symlinks_6013", "Do not resolve the real path of symlinks."),
  Only_emit_d_ts_declaration_files: diag(6014, 3 /* Message */, "Only_emit_d_ts_declaration_files_6014", "Only emit '.d.ts' declaration files."),
  Specify_ECMAScript_target_version: diag(6015, 3 /* Message */, "Specify_ECMAScript_target_version_6015", "Specify ECMAScript target version."),
  Specify_module_code_generation: diag(6016, 3 /* Message */, "Specify_module_code_generation_6016", "Specify module code generation."),
  Print_this_message: diag(6017, 3 /* Message */, "Print_this_message_6017", "Print this message."),
  Print_the_compiler_s_version: diag(6019, 3 /* Message */, "Print_the_compiler_s_version_6019", "Print the compiler's version."),
  Compile_the_project_given_the_path_to_its_configuration_file_or_to_a_folder_with_a_tsconfig_json: diag(6020, 3 /* Message */, "Compile_the_project_given_the_path_to_its_configuration_file_or_to_a_folder_with_a_tsconfig_json_6020", "Compile the project given the path to its configuration file, or to a folder with a 'tsconfig.json'."),
  Syntax_Colon_0: diag(6023, 3 /* Message */, "Syntax_Colon_0_6023", "Syntax: {0}"),
  options: diag(6024, 3 /* Message */, "options_6024", "options"),
  file: diag(6025, 3 /* Message */, "file_6025", "file"),
  Examples_Colon_0: diag(6026, 3 /* Message */, "Examples_Colon_0_6026", "Examples: {0}"),
  Options_Colon: diag(6027, 3 /* Message */, "Options_Colon_6027", "Options:"),
  Version_0: diag(6029, 3 /* Message */, "Version_0_6029", "Version {0}"),
  Insert_command_line_options_and_files_from_a_file: diag(6030, 3 /* Message */, "Insert_command_line_options_and_files_from_a_file_6030", "Insert command line options and files from a file."),
  Starting_compilation_in_watch_mode: diag(6031, 3 /* Message */, "Starting_compilation_in_watch_mode_6031", "Starting compilation in watch mode..."),
  File_change_detected_Starting_incremental_compilation: diag(6032, 3 /* Message */, "File_change_detected_Starting_incremental_compilation_6032", "File change detected. Starting incremental compilation..."),
  KIND: diag(6034, 3 /* Message */, "KIND_6034", "KIND"),
  FILE: diag(6035, 3 /* Message */, "FILE_6035", "FILE"),
  VERSION: diag(6036, 3 /* Message */, "VERSION_6036", "VERSION"),
  LOCATION: diag(6037, 3 /* Message */, "LOCATION_6037", "LOCATION"),
  DIRECTORY: diag(6038, 3 /* Message */, "DIRECTORY_6038", "DIRECTORY"),
  STRATEGY: diag(6039, 3 /* Message */, "STRATEGY_6039", "STRATEGY"),
  FILE_OR_DIRECTORY: diag(6040, 3 /* Message */, "FILE_OR_DIRECTORY_6040", "FILE OR DIRECTORY"),
  Errors_Files: diag(6041, 3 /* Message */, "Errors_Files_6041", "Errors  Files"),
  Generates_corresponding_map_file: diag(6043, 3 /* Message */, "Generates_corresponding_map_file_6043", "Generates corresponding '.map' file."),
  Compiler_option_0_expects_an_argument: diag(6044, 1 /* Error */, "Compiler_option_0_expects_an_argument_6044", "Compiler option '{0}' expects an argument."),
  Unterminated_quoted_string_in_response_file_0: diag(6045, 1 /* Error */, "Unterminated_quoted_string_in_response_file_0_6045", "Unterminated quoted string in response file '{0}'."),
  Argument_for_0_option_must_be_Colon_1: diag(6046, 1 /* Error */, "Argument_for_0_option_must_be_Colon_1_6046", "Argument for '{0}' option must be: {1}."),
  Locale_must_be_of_the_form_language_or_language_territory_For_example_0_or_1: diag(6048, 1 /* Error */, "Locale_must_be_of_the_form_language_or_language_territory_For_example_0_or_1_6048", "Locale must be of the form <language> or <language>-<territory>. For example '{0}' or '{1}'."),
  Unable_to_open_file_0: diag(6050, 1 /* Error */, "Unable_to_open_file_0_6050", "Unable to open file '{0}'."),
  Corrupted_locale_file_0: diag(6051, 1 /* Error */, "Corrupted_locale_file_0_6051", "Corrupted locale file {0}."),
  Raise_error_on_expressions_and_declarations_with_an_implied_any_type: diag(6052, 3 /* Message */, "Raise_error_on_expressions_and_declarations_with_an_implied_any_type_6052", "Raise error on expressions and declarations with an implied 'any' type."),
  File_0_not_found: diag(6053, 1 /* Error */, "File_0_not_found_6053", "File '{0}' not found."),
  File_0_has_an_unsupported_extension_The_only_supported_extensions_are_1: diag(6054, 1 /* Error */, "File_0_has_an_unsupported_extension_The_only_supported_extensions_are_1_6054", "File '{0}' has an unsupported extension. The only supported extensions are {1}."),
  Suppress_noImplicitAny_errors_for_indexing_objects_lacking_index_signatures: diag(6055, 3 /* Message */, "Suppress_noImplicitAny_errors_for_indexing_objects_lacking_index_signatures_6055", "Suppress noImplicitAny errors for indexing objects lacking index signatures."),
  Do_not_emit_declarations_for_code_that_has_an_internal_annotation: diag(6056, 3 /* Message */, "Do_not_emit_declarations_for_code_that_has_an_internal_annotation_6056", "Do not emit declarations for code that has an '@internal' annotation."),
  Specify_the_root_directory_of_input_files_Use_to_control_the_output_directory_structure_with_outDir: diag(6058, 3 /* Message */, "Specify_the_root_directory_of_input_files_Use_to_control_the_output_directory_structure_with_outDir_6058", "Specify the root directory of input files. Use to control the output directory structure with --outDir."),
  File_0_is_not_under_rootDir_1_rootDir_is_expected_to_contain_all_source_files: diag(6059, 1 /* Error */, "File_0_is_not_under_rootDir_1_rootDir_is_expected_to_contain_all_source_files_6059", "File '{0}' is not under 'rootDir' '{1}'. 'rootDir' is expected to contain all source files."),
  Specify_the_end_of_line_sequence_to_be_used_when_emitting_files_Colon_CRLF_dos_or_LF_unix: diag(6060, 3 /* Message */, "Specify_the_end_of_line_sequence_to_be_used_when_emitting_files_Colon_CRLF_dos_or_LF_unix_6060", "Specify the end of line sequence to be used when emitting files: 'CRLF' (dos) or 'LF' (unix)."),
  NEWLINE: diag(6061, 3 /* Message */, "NEWLINE_6061", "NEWLINE"),
  Option_0_can_only_be_specified_in_tsconfig_json_file_or_set_to_null_on_command_line: diag(6064, 1 /* Error */, "Option_0_can_only_be_specified_in_tsconfig_json_file_or_set_to_null_on_command_line_6064", "Option '{0}' can only be specified in 'tsconfig.json' file or set to 'null' on command line."),
  Enables_experimental_support_for_ES7_decorators: diag(6065, 3 /* Message */, "Enables_experimental_support_for_ES7_decorators_6065", "Enables experimental support for ES7 decorators."),
  Enables_experimental_support_for_emitting_type_metadata_for_decorators: diag(6066, 3 /* Message */, "Enables_experimental_support_for_emitting_type_metadata_for_decorators_6066", "Enables experimental support for emitting type metadata for decorators."),
  Initializes_a_TypeScript_project_and_creates_a_tsconfig_json_file: diag(6070, 3 /* Message */, "Initializes_a_TypeScript_project_and_creates_a_tsconfig_json_file_6070", "Initializes a TypeScript project and creates a tsconfig.json file."),
  Successfully_created_a_tsconfig_json_file: diag(6071, 3 /* Message */, "Successfully_created_a_tsconfig_json_file_6071", "Successfully created a tsconfig.json file."),
  Suppress_excess_property_checks_for_object_literals: diag(6072, 3 /* Message */, "Suppress_excess_property_checks_for_object_literals_6072", "Suppress excess property checks for object literals."),
  Stylize_errors_and_messages_using_color_and_context_experimental: diag(6073, 3 /* Message */, "Stylize_errors_and_messages_using_color_and_context_experimental_6073", "Stylize errors and messages using color and context (experimental)."),
  Do_not_report_errors_on_unused_labels: diag(6074, 3 /* Message */, "Do_not_report_errors_on_unused_labels_6074", "Do not report errors on unused labels."),
  Report_error_when_not_all_code_paths_in_function_return_a_value: diag(6075, 3 /* Message */, "Report_error_when_not_all_code_paths_in_function_return_a_value_6075", "Report error when not all code paths in function return a value."),
  Report_errors_for_fallthrough_cases_in_switch_statement: diag(6076, 3 /* Message */, "Report_errors_for_fallthrough_cases_in_switch_statement_6076", "Report errors for fallthrough cases in switch statement."),
  Do_not_report_errors_on_unreachable_code: diag(6077, 3 /* Message */, "Do_not_report_errors_on_unreachable_code_6077", "Do not report errors on unreachable code."),
  Disallow_inconsistently_cased_references_to_the_same_file: diag(6078, 3 /* Message */, "Disallow_inconsistently_cased_references_to_the_same_file_6078", "Disallow inconsistently-cased references to the same file."),
  Specify_library_files_to_be_included_in_the_compilation: diag(6079, 3 /* Message */, "Specify_library_files_to_be_included_in_the_compilation_6079", "Specify library files to be included in the compilation."),
  Specify_JSX_code_generation: diag(6080, 3 /* Message */, "Specify_JSX_code_generation_6080", "Specify JSX code generation."),
  Only_amd_and_system_modules_are_supported_alongside_0: diag(6082, 1 /* Error */, "Only_amd_and_system_modules_are_supported_alongside_0_6082", "Only 'amd' and 'system' modules are supported alongside --{0}."),
  Base_directory_to_resolve_non_absolute_module_names: diag(6083, 3 /* Message */, "Base_directory_to_resolve_non_absolute_module_names_6083", "Base directory to resolve non-absolute module names."),
  Deprecated_Use_jsxFactory_instead_Specify_the_object_invoked_for_createElement_when_targeting_react_JSX_emit: diag(6084, 3 /* Message */, "Deprecated_Use_jsxFactory_instead_Specify_the_object_invoked_for_createElement_when_targeting_react__6084", "[Deprecated] Use '--jsxFactory' instead. Specify the object invoked for createElement when targeting 'react' JSX emit"),
  Enable_tracing_of_the_name_resolution_process: diag(6085, 3 /* Message */, "Enable_tracing_of_the_name_resolution_process_6085", "Enable tracing of the name resolution process."),
  Resolving_module_0_from_1: diag(6086, 3 /* Message */, "Resolving_module_0_from_1_6086", "======== Resolving module '{0}' from '{1}'. ========"),
  Explicitly_specified_module_resolution_kind_Colon_0: diag(6087, 3 /* Message */, "Explicitly_specified_module_resolution_kind_Colon_0_6087", "Explicitly specified module resolution kind: '{0}'."),
  Module_resolution_kind_is_not_specified_using_0: diag(6088, 3 /* Message */, "Module_resolution_kind_is_not_specified_using_0_6088", "Module resolution kind is not specified, using '{0}'."),
  Module_name_0_was_successfully_resolved_to_1: diag(6089, 3 /* Message */, "Module_name_0_was_successfully_resolved_to_1_6089", "======== Module name '{0}' was successfully resolved to '{1}'. ========"),
  Module_name_0_was_not_resolved: diag(6090, 3 /* Message */, "Module_name_0_was_not_resolved_6090", "======== Module name '{0}' was not resolved. ========"),
  paths_option_is_specified_looking_for_a_pattern_to_match_module_name_0: diag(6091, 3 /* Message */, "paths_option_is_specified_looking_for_a_pattern_to_match_module_name_0_6091", "'paths' option is specified, looking for a pattern to match module name '{0}'."),
  Module_name_0_matched_pattern_1: diag(6092, 3 /* Message */, "Module_name_0_matched_pattern_1_6092", "Module name '{0}', matched pattern '{1}'."),
  Trying_substitution_0_candidate_module_location_Colon_1: diag(6093, 3 /* Message */, "Trying_substitution_0_candidate_module_location_Colon_1_6093", "Trying substitution '{0}', candidate module location: '{1}'."),
  Resolving_module_name_0_relative_to_base_url_1_2: diag(6094, 3 /* Message */, "Resolving_module_name_0_relative_to_base_url_1_2_6094", "Resolving module name '{0}' relative to base url '{1}' - '{2}'."),
  Loading_module_as_file_Slash_folder_candidate_module_location_0_target_file_types_Colon_1: diag(6095, 3 /* Message */, "Loading_module_as_file_Slash_folder_candidate_module_location_0_target_file_types_Colon_1_6095", "Loading module as file / folder, candidate module location '{0}', target file types: {1}."),
  File_0_does_not_exist: diag(6096, 3 /* Message */, "File_0_does_not_exist_6096", "File '{0}' does not exist."),
  File_0_exists_use_it_as_a_name_resolution_result: diag(6097, 3 /* Message */, "File_0_exists_use_it_as_a_name_resolution_result_6097", "File '{0}' exists - use it as a name resolution result."),
  Loading_module_0_from_node_modules_folder_target_file_types_Colon_1: diag(6098, 3 /* Message */, "Loading_module_0_from_node_modules_folder_target_file_types_Colon_1_6098", "Loading module '{0}' from 'node_modules' folder, target file types: {1}."),
  Found_package_json_at_0: diag(6099, 3 /* Message */, "Found_package_json_at_0_6099", "Found 'package.json' at '{0}'."),
  package_json_does_not_have_a_0_field: diag(6100, 3 /* Message */, "package_json_does_not_have_a_0_field_6100", "'package.json' does not have a '{0}' field."),
  package_json_has_0_field_1_that_references_2: diag(6101, 3 /* Message */, "package_json_has_0_field_1_that_references_2_6101", "'package.json' has '{0}' field '{1}' that references '{2}'."),
  Allow_javascript_files_to_be_compiled: diag(6102, 3 /* Message */, "Allow_javascript_files_to_be_compiled_6102", "Allow javascript files to be compiled."),
  Checking_if_0_is_the_longest_matching_prefix_for_1_2: diag(6104, 3 /* Message */, "Checking_if_0_is_the_longest_matching_prefix_for_1_2_6104", "Checking if '{0}' is the longest matching prefix for '{1}' - '{2}'."),
  Expected_type_of_0_field_in_package_json_to_be_1_got_2: diag(6105, 3 /* Message */, "Expected_type_of_0_field_in_package_json_to_be_1_got_2_6105", "Expected type of '{0}' field in 'package.json' to be '{1}', got '{2}'."),
  baseUrl_option_is_set_to_0_using_this_value_to_resolve_non_relative_module_name_1: diag(6106, 3 /* Message */, "baseUrl_option_is_set_to_0_using_this_value_to_resolve_non_relative_module_name_1_6106", "'baseUrl' option is set to '{0}', using this value to resolve non-relative module name '{1}'."),
  rootDirs_option_is_set_using_it_to_resolve_relative_module_name_0: diag(6107, 3 /* Message */, "rootDirs_option_is_set_using_it_to_resolve_relative_module_name_0_6107", "'rootDirs' option is set, using it to resolve relative module name '{0}'."),
  Longest_matching_prefix_for_0_is_1: diag(6108, 3 /* Message */, "Longest_matching_prefix_for_0_is_1_6108", "Longest matching prefix for '{0}' is '{1}'."),
  Loading_0_from_the_root_dir_1_candidate_location_2: diag(6109, 3 /* Message */, "Loading_0_from_the_root_dir_1_candidate_location_2_6109", "Loading '{0}' from the root dir '{1}', candidate location '{2}'."),
  Trying_other_entries_in_rootDirs: diag(6110, 3 /* Message */, "Trying_other_entries_in_rootDirs_6110", "Trying other entries in 'rootDirs'."),
  Module_resolution_using_rootDirs_has_failed: diag(6111, 3 /* Message */, "Module_resolution_using_rootDirs_has_failed_6111", "Module resolution using 'rootDirs' has failed."),
  Do_not_emit_use_strict_directives_in_module_output: diag(6112, 3 /* Message */, "Do_not_emit_use_strict_directives_in_module_output_6112", "Do not emit 'use strict' directives in module output."),
  Enable_strict_null_checks: diag(6113, 3 /* Message */, "Enable_strict_null_checks_6113", "Enable strict null checks."),
  Unknown_option_excludes_Did_you_mean_exclude: diag(6114, 1 /* Error */, "Unknown_option_excludes_Did_you_mean_exclude_6114", "Unknown option 'excludes'. Did you mean 'exclude'?"),
  Raise_error_on_this_expressions_with_an_implied_any_type: diag(6115, 3 /* Message */, "Raise_error_on_this_expressions_with_an_implied_any_type_6115", "Raise error on 'this' expressions with an implied 'any' type."),
  Resolving_type_reference_directive_0_containing_file_1_root_directory_2: diag(6116, 3 /* Message */, "Resolving_type_reference_directive_0_containing_file_1_root_directory_2_6116", "======== Resolving type reference directive '{0}', containing file '{1}', root directory '{2}'. ========"),
  Type_reference_directive_0_was_successfully_resolved_to_1_primary_Colon_2: diag(6119, 3 /* Message */, "Type_reference_directive_0_was_successfully_resolved_to_1_primary_Colon_2_6119", "======== Type reference directive '{0}' was successfully resolved to '{1}', primary: {2}. ========"),
  Type_reference_directive_0_was_not_resolved: diag(6120, 3 /* Message */, "Type_reference_directive_0_was_not_resolved_6120", "======== Type reference directive '{0}' was not resolved. ========"),
  Resolving_with_primary_search_path_0: diag(6121, 3 /* Message */, "Resolving_with_primary_search_path_0_6121", "Resolving with primary search path '{0}'."),
  Root_directory_cannot_be_determined_skipping_primary_search_paths: diag(6122, 3 /* Message */, "Root_directory_cannot_be_determined_skipping_primary_search_paths_6122", "Root directory cannot be determined, skipping primary search paths."),
  Resolving_type_reference_directive_0_containing_file_1_root_directory_not_set: diag(6123, 3 /* Message */, "Resolving_type_reference_directive_0_containing_file_1_root_directory_not_set_6123", "======== Resolving type reference directive '{0}', containing file '{1}', root directory not set. ========"),
  Type_declaration_files_to_be_included_in_compilation: diag(6124, 3 /* Message */, "Type_declaration_files_to_be_included_in_compilation_6124", "Type declaration files to be included in compilation."),
  Looking_up_in_node_modules_folder_initial_location_0: diag(6125, 3 /* Message */, "Looking_up_in_node_modules_folder_initial_location_0_6125", "Looking up in 'node_modules' folder, initial location '{0}'."),
  Containing_file_is_not_specified_and_root_directory_cannot_be_determined_skipping_lookup_in_node_modules_folder: diag(6126, 3 /* Message */, "Containing_file_is_not_specified_and_root_directory_cannot_be_determined_skipping_lookup_in_node_mod_6126", "Containing file is not specified and root directory cannot be determined, skipping lookup in 'node_modules' folder."),
  Resolving_type_reference_directive_0_containing_file_not_set_root_directory_1: diag(6127, 3 /* Message */, "Resolving_type_reference_directive_0_containing_file_not_set_root_directory_1_6127", "======== Resolving type reference directive '{0}', containing file not set, root directory '{1}'. ========"),
  Resolving_type_reference_directive_0_containing_file_not_set_root_directory_not_set: diag(6128, 3 /* Message */, "Resolving_type_reference_directive_0_containing_file_not_set_root_directory_not_set_6128", "======== Resolving type reference directive '{0}', containing file not set, root directory not set. ========"),
  Resolving_real_path_for_0_result_1: diag(6130, 3 /* Message */, "Resolving_real_path_for_0_result_1_6130", "Resolving real path for '{0}', result '{1}'."),
  Cannot_compile_modules_using_option_0_unless_the_module_flag_is_amd_or_system: diag(6131, 1 /* Error */, "Cannot_compile_modules_using_option_0_unless_the_module_flag_is_amd_or_system_6131", "Cannot compile modules using option '{0}' unless the '--module' flag is 'amd' or 'system'."),
  File_name_0_has_a_1_extension_stripping_it: diag(6132, 3 /* Message */, "File_name_0_has_a_1_extension_stripping_it_6132", "File name '{0}' has a '{1}' extension - stripping it."),
  _0_is_declared_but_its_value_is_never_read: diag(
    6133,
    1 /* Error */,
    "_0_is_declared_but_its_value_is_never_read_6133",
    "'{0}' is declared but its value is never read.",
    /*reportsUnnecessary*/
    true
  ),
  Report_errors_on_unused_locals: diag(6134, 3 /* Message */, "Report_errors_on_unused_locals_6134", "Report errors on unused locals."),
  Report_errors_on_unused_parameters: diag(6135, 3 /* Message */, "Report_errors_on_unused_parameters_6135", "Report errors on unused parameters."),
  The_maximum_dependency_depth_to_search_under_node_modules_and_load_JavaScript_files: diag(6136, 3 /* Message */, "The_maximum_dependency_depth_to_search_under_node_modules_and_load_JavaScript_files_6136", "The maximum dependency depth to search under node_modules and load JavaScript files."),
  Cannot_import_type_declaration_files_Consider_importing_0_instead_of_1: diag(6137, 1 /* Error */, "Cannot_import_type_declaration_files_Consider_importing_0_instead_of_1_6137", "Cannot import type declaration files. Consider importing '{0}' instead of '{1}'."),
  Property_0_is_declared_but_its_value_is_never_read: diag(
    6138,
    1 /* Error */,
    "Property_0_is_declared_but_its_value_is_never_read_6138",
    "Property '{0}' is declared but its value is never read.",
    /*reportsUnnecessary*/
    true
  ),
  Import_emit_helpers_from_tslib: diag(6139, 3 /* Message */, "Import_emit_helpers_from_tslib_6139", "Import emit helpers from 'tslib'."),
  Auto_discovery_for_typings_is_enabled_in_project_0_Running_extra_resolution_pass_for_module_1_using_cache_location_2: diag(6140, 1 /* Error */, "Auto_discovery_for_typings_is_enabled_in_project_0_Running_extra_resolution_pass_for_module_1_using__6140", "Auto discovery for typings is enabled in project '{0}'. Running extra resolution pass for module '{1}' using cache location '{2}'."),
  Parse_in_strict_mode_and_emit_use_strict_for_each_source_file: diag(6141, 3 /* Message */, "Parse_in_strict_mode_and_emit_use_strict_for_each_source_file_6141", 'Parse in strict mode and emit "use strict" for each source file.'),
  Module_0_was_resolved_to_1_but_jsx_is_not_set: diag(6142, 1 /* Error */, "Module_0_was_resolved_to_1_but_jsx_is_not_set_6142", "Module '{0}' was resolved to '{1}', but '--jsx' is not set."),
  Module_0_was_resolved_as_locally_declared_ambient_module_in_file_1: diag(6144, 3 /* Message */, "Module_0_was_resolved_as_locally_declared_ambient_module_in_file_1_6144", "Module '{0}' was resolved as locally declared ambient module in file '{1}'."),
  Specify_the_JSX_factory_function_to_use_when_targeting_react_JSX_emit_e_g_React_createElement_or_h: diag(6146, 3 /* Message */, "Specify_the_JSX_factory_function_to_use_when_targeting_react_JSX_emit_e_g_React_createElement_or_h_6146", "Specify the JSX factory function to use when targeting 'react' JSX emit, e.g. 'React.createElement' or 'h'."),
  Resolution_for_module_0_was_found_in_cache_from_location_1: diag(6147, 3 /* Message */, "Resolution_for_module_0_was_found_in_cache_from_location_1_6147", "Resolution for module '{0}' was found in cache from location '{1}'."),
  Directory_0_does_not_exist_skipping_all_lookups_in_it: diag(6148, 3 /* Message */, "Directory_0_does_not_exist_skipping_all_lookups_in_it_6148", "Directory '{0}' does not exist, skipping all lookups in it."),
  Show_diagnostic_information: diag(6149, 3 /* Message */, "Show_diagnostic_information_6149", "Show diagnostic information."),
  Show_verbose_diagnostic_information: diag(6150, 3 /* Message */, "Show_verbose_diagnostic_information_6150", "Show verbose diagnostic information."),
  Emit_a_single_file_with_source_maps_instead_of_having_a_separate_file: diag(6151, 3 /* Message */, "Emit_a_single_file_with_source_maps_instead_of_having_a_separate_file_6151", "Emit a single file with source maps instead of having a separate file."),
  Emit_the_source_alongside_the_sourcemaps_within_a_single_file_requires_inlineSourceMap_or_sourceMap_to_be_set: diag(6152, 3 /* Message */, "Emit_the_source_alongside_the_sourcemaps_within_a_single_file_requires_inlineSourceMap_or_sourceMap__6152", "Emit the source alongside the sourcemaps within a single file; requires '--inlineSourceMap' or '--sourceMap' to be set."),
  Transpile_each_file_as_a_separate_module_similar_to_ts_transpileModule: diag(6153, 3 /* Message */, "Transpile_each_file_as_a_separate_module_similar_to_ts_transpileModule_6153", "Transpile each file as a separate module (similar to 'ts.transpileModule')."),
  Print_names_of_generated_files_part_of_the_compilation: diag(6154, 3 /* Message */, "Print_names_of_generated_files_part_of_the_compilation_6154", "Print names of generated files part of the compilation."),
  Print_names_of_files_part_of_the_compilation: diag(6155, 3 /* Message */, "Print_names_of_files_part_of_the_compilation_6155", "Print names of files part of the compilation."),
  The_locale_used_when_displaying_messages_to_the_user_e_g_en_us: diag(6156, 3 /* Message */, "The_locale_used_when_displaying_messages_to_the_user_e_g_en_us_6156", "The locale used when displaying messages to the user (e.g. 'en-us')"),
  Do_not_generate_custom_helper_functions_like_extends_in_compiled_output: diag(6157, 3 /* Message */, "Do_not_generate_custom_helper_functions_like_extends_in_compiled_output_6157", "Do not generate custom helper functions like '__extends' in compiled output."),
  Do_not_include_the_default_library_file_lib_d_ts: diag(6158, 3 /* Message */, "Do_not_include_the_default_library_file_lib_d_ts_6158", "Do not include the default library file (lib.d.ts)."),
  Do_not_add_triple_slash_references_or_imported_modules_to_the_list_of_compiled_files: diag(6159, 3 /* Message */, "Do_not_add_triple_slash_references_or_imported_modules_to_the_list_of_compiled_files_6159", "Do not add triple-slash references or imported modules to the list of compiled files."),
  Deprecated_Use_skipLibCheck_instead_Skip_type_checking_of_default_library_declaration_files: diag(6160, 3 /* Message */, "Deprecated_Use_skipLibCheck_instead_Skip_type_checking_of_default_library_declaration_files_6160", "[Deprecated] Use '--skipLibCheck' instead. Skip type checking of default library declaration files."),
  List_of_folders_to_include_type_definitions_from: diag(6161, 3 /* Message */, "List_of_folders_to_include_type_definitions_from_6161", "List of folders to include type definitions from."),
  Disable_size_limitations_on_JavaScript_projects: diag(6162, 3 /* Message */, "Disable_size_limitations_on_JavaScript_projects_6162", "Disable size limitations on JavaScript projects."),
  The_character_set_of_the_input_files: diag(6163, 3 /* Message */, "The_character_set_of_the_input_files_6163", "The character set of the input files."),
  Skipping_module_0_that_looks_like_an_absolute_URI_target_file_types_Colon_1: diag(6164, 3 /* Message */, "Skipping_module_0_that_looks_like_an_absolute_URI_target_file_types_Colon_1_6164", "Skipping module '{0}' that looks like an absolute URI, target file types: {1}."),
  Do_not_truncate_error_messages: diag(6165, 3 /* Message */, "Do_not_truncate_error_messages_6165", "Do not truncate error messages."),
  Output_directory_for_generated_declaration_files: diag(6166, 3 /* Message */, "Output_directory_for_generated_declaration_files_6166", "Output directory for generated declaration files."),
  A_series_of_entries_which_re_map_imports_to_lookup_locations_relative_to_the_baseUrl: diag(6167, 3 /* Message */, "A_series_of_entries_which_re_map_imports_to_lookup_locations_relative_to_the_baseUrl_6167", "A series of entries which re-map imports to lookup locations relative to the 'baseUrl'."),
  List_of_root_folders_whose_combined_content_represents_the_structure_of_the_project_at_runtime: diag(6168, 3 /* Message */, "List_of_root_folders_whose_combined_content_represents_the_structure_of_the_project_at_runtime_6168", "List of root folders whose combined content represents the structure of the project at runtime."),
  Show_all_compiler_options: diag(6169, 3 /* Message */, "Show_all_compiler_options_6169", "Show all compiler options."),
  Deprecated_Use_outFile_instead_Concatenate_and_emit_output_to_single_file: diag(6170, 3 /* Message */, "Deprecated_Use_outFile_instead_Concatenate_and_emit_output_to_single_file_6170", "[Deprecated] Use '--outFile' instead. Concatenate and emit output to single file"),
  Command_line_Options: diag(6171, 3 /* Message */, "Command_line_Options_6171", "Command-line Options"),
  Provide_full_support_for_iterables_in_for_of_spread_and_destructuring_when_targeting_ES5: diag(6179, 3 /* Message */, "Provide_full_support_for_iterables_in_for_of_spread_and_destructuring_when_targeting_ES5_6179", "Provide full support for iterables in 'for-of', spread, and destructuring when targeting 'ES5'."),
  Enable_all_strict_type_checking_options: diag(6180, 3 /* Message */, "Enable_all_strict_type_checking_options_6180", "Enable all strict type-checking options."),
  Scoped_package_detected_looking_in_0: diag(6182, 3 /* Message */, "Scoped_package_detected_looking_in_0_6182", "Scoped package detected, looking in '{0}'"),
  Reusing_resolution_of_module_0_from_1_of_old_program_it_was_successfully_resolved_to_2: diag(6183, 3 /* Message */, "Reusing_resolution_of_module_0_from_1_of_old_program_it_was_successfully_resolved_to_2_6183", "Reusing resolution of module '{0}' from '{1}' of old program, it was successfully resolved to '{2}'."),
  Reusing_resolution_of_module_0_from_1_of_old_program_it_was_successfully_resolved_to_2_with_Package_ID_3: diag(6184, 3 /* Message */, "Reusing_resolution_of_module_0_from_1_of_old_program_it_was_successfully_resolved_to_2_with_Package__6184", "Reusing resolution of module '{0}' from '{1}' of old program, it was successfully resolved to '{2}' with Package ID '{3}'."),
  Enable_strict_checking_of_function_types: diag(6186, 3 /* Message */, "Enable_strict_checking_of_function_types_6186", "Enable strict checking of function types."),
  Enable_strict_checking_of_property_initialization_in_classes: diag(6187, 3 /* Message */, "Enable_strict_checking_of_property_initialization_in_classes_6187", "Enable strict checking of property initialization in classes."),
  Numeric_separators_are_not_allowed_here: diag(6188, 1 /* Error */, "Numeric_separators_are_not_allowed_here_6188", "Numeric separators are not allowed here."),
  Multiple_consecutive_numeric_separators_are_not_permitted: diag(6189, 1 /* Error */, "Multiple_consecutive_numeric_separators_are_not_permitted_6189", "Multiple consecutive numeric separators are not permitted."),
  Whether_to_keep_outdated_console_output_in_watch_mode_instead_of_clearing_the_screen: diag(6191, 3 /* Message */, "Whether_to_keep_outdated_console_output_in_watch_mode_instead_of_clearing_the_screen_6191", "Whether to keep outdated console output in watch mode instead of clearing the screen."),
  All_imports_in_import_declaration_are_unused: diag(
    6192,
    1 /* Error */,
    "All_imports_in_import_declaration_are_unused_6192",
    "All imports in import declaration are unused.",
    /*reportsUnnecessary*/
    true
  ),
  Found_1_error_Watching_for_file_changes: diag(6193, 3 /* Message */, "Found_1_error_Watching_for_file_changes_6193", "Found 1 error. Watching for file changes."),
  Found_0_errors_Watching_for_file_changes: diag(6194, 3 /* Message */, "Found_0_errors_Watching_for_file_changes_6194", "Found {0} errors. Watching for file changes."),
  Resolve_keyof_to_string_valued_property_names_only_no_numbers_or_symbols: diag(6195, 3 /* Message */, "Resolve_keyof_to_string_valued_property_names_only_no_numbers_or_symbols_6195", "Resolve 'keyof' to string valued property names only (no numbers or symbols)."),
  _0_is_declared_but_never_used: diag(
    6196,
    1 /* Error */,
    "_0_is_declared_but_never_used_6196",
    "'{0}' is declared but never used.",
    /*reportsUnnecessary*/
    true
  ),
  Include_modules_imported_with_json_extension: diag(6197, 3 /* Message */, "Include_modules_imported_with_json_extension_6197", "Include modules imported with '.json' extension"),
  All_destructured_elements_are_unused: diag(
    6198,
    1 /* Error */,
    "All_destructured_elements_are_unused_6198",
    "All destructured elements are unused.",
    /*reportsUnnecessary*/
    true
  ),
  All_variables_are_unused: diag(
    6199,
    1 /* Error */,
    "All_variables_are_unused_6199",
    "All variables are unused.",
    /*reportsUnnecessary*/
    true
  ),
  Definitions_of_the_following_identifiers_conflict_with_those_in_another_file_Colon_0: diag(6200, 1 /* Error */, "Definitions_of_the_following_identifiers_conflict_with_those_in_another_file_Colon_0_6200", "Definitions of the following identifiers conflict with those in another file: {0}"),
  Conflicts_are_in_this_file: diag(6201, 3 /* Message */, "Conflicts_are_in_this_file_6201", "Conflicts are in this file."),
  Project_references_may_not_form_a_circular_graph_Cycle_detected_Colon_0: diag(6202, 1 /* Error */, "Project_references_may_not_form_a_circular_graph_Cycle_detected_Colon_0_6202", "Project references may not form a circular graph. Cycle detected: {0}"),
  _0_was_also_declared_here: diag(6203, 3 /* Message */, "_0_was_also_declared_here_6203", "'{0}' was also declared here."),
  and_here: diag(6204, 3 /* Message */, "and_here_6204", "and here."),
  All_type_parameters_are_unused: diag(6205, 1 /* Error */, "All_type_parameters_are_unused_6205", "All type parameters are unused."),
  package_json_has_a_typesVersions_field_with_version_specific_path_mappings: diag(6206, 3 /* Message */, "package_json_has_a_typesVersions_field_with_version_specific_path_mappings_6206", "'package.json' has a 'typesVersions' field with version-specific path mappings."),
  package_json_does_not_have_a_typesVersions_entry_that_matches_version_0: diag(6207, 3 /* Message */, "package_json_does_not_have_a_typesVersions_entry_that_matches_version_0_6207", "'package.json' does not have a 'typesVersions' entry that matches version '{0}'."),
  package_json_has_a_typesVersions_entry_0_that_matches_compiler_version_1_looking_for_a_pattern_to_match_module_name_2: diag(6208, 3 /* Message */, "package_json_has_a_typesVersions_entry_0_that_matches_compiler_version_1_looking_for_a_pattern_to_ma_6208", "'package.json' has a 'typesVersions' entry '{0}' that matches compiler version '{1}', looking for a pattern to match module name '{2}'."),
  package_json_has_a_typesVersions_entry_0_that_is_not_a_valid_semver_range: diag(6209, 3 /* Message */, "package_json_has_a_typesVersions_entry_0_that_is_not_a_valid_semver_range_6209", "'package.json' has a 'typesVersions' entry '{0}' that is not a valid semver range."),
  An_argument_for_0_was_not_provided: diag(6210, 3 /* Message */, "An_argument_for_0_was_not_provided_6210", "An argument for '{0}' was not provided."),
  An_argument_matching_this_binding_pattern_was_not_provided: diag(6211, 3 /* Message */, "An_argument_matching_this_binding_pattern_was_not_provided_6211", "An argument matching this binding pattern was not provided."),
  Did_you_mean_to_call_this_expression: diag(6212, 3 /* Message */, "Did_you_mean_to_call_this_expression_6212", "Did you mean to call this expression?"),
  Did_you_mean_to_use_new_with_this_expression: diag(6213, 3 /* Message */, "Did_you_mean_to_use_new_with_this_expression_6213", "Did you mean to use 'new' with this expression?"),
  Enable_strict_bind_call_and_apply_methods_on_functions: diag(6214, 3 /* Message */, "Enable_strict_bind_call_and_apply_methods_on_functions_6214", "Enable strict 'bind', 'call', and 'apply' methods on functions."),
  Using_compiler_options_of_project_reference_redirect_0: diag(6215, 3 /* Message */, "Using_compiler_options_of_project_reference_redirect_0_6215", "Using compiler options of project reference redirect '{0}'."),
  Found_1_error: diag(6216, 3 /* Message */, "Found_1_error_6216", "Found 1 error."),
  Found_0_errors: diag(6217, 3 /* Message */, "Found_0_errors_6217", "Found {0} errors."),
  Module_name_0_was_successfully_resolved_to_1_with_Package_ID_2: diag(6218, 3 /* Message */, "Module_name_0_was_successfully_resolved_to_1_with_Package_ID_2_6218", "======== Module name '{0}' was successfully resolved to '{1}' with Package ID '{2}'. ========"),
  Type_reference_directive_0_was_successfully_resolved_to_1_with_Package_ID_2_primary_Colon_3: diag(6219, 3 /* Message */, "Type_reference_directive_0_was_successfully_resolved_to_1_with_Package_ID_2_primary_Colon_3_6219", "======== Type reference directive '{0}' was successfully resolved to '{1}' with Package ID '{2}', primary: {3}. ========"),
  package_json_had_a_falsy_0_field: diag(6220, 3 /* Message */, "package_json_had_a_falsy_0_field_6220", "'package.json' had a falsy '{0}' field."),
  Disable_use_of_source_files_instead_of_declaration_files_from_referenced_projects: diag(6221, 3 /* Message */, "Disable_use_of_source_files_instead_of_declaration_files_from_referenced_projects_6221", "Disable use of source files instead of declaration files from referenced projects."),
  Emit_class_fields_with_Define_instead_of_Set: diag(6222, 3 /* Message */, "Emit_class_fields_with_Define_instead_of_Set_6222", "Emit class fields with Define instead of Set."),
  Generates_a_CPU_profile: diag(6223, 3 /* Message */, "Generates_a_CPU_profile_6223", "Generates a CPU profile."),
  Disable_solution_searching_for_this_project: diag(6224, 3 /* Message */, "Disable_solution_searching_for_this_project_6224", "Disable solution searching for this project."),
  Specify_strategy_for_watching_file_Colon_FixedPollingInterval_default_PriorityPollingInterval_DynamicPriorityPolling_FixedChunkSizePolling_UseFsEvents_UseFsEventsOnParentDirectory: diag(6225, 3 /* Message */, "Specify_strategy_for_watching_file_Colon_FixedPollingInterval_default_PriorityPollingInterval_Dynami_6225", "Specify strategy for watching file: 'FixedPollingInterval' (default), 'PriorityPollingInterval', 'DynamicPriorityPolling', 'FixedChunkSizePolling', 'UseFsEvents', 'UseFsEventsOnParentDirectory'."),
  Specify_strategy_for_watching_directory_on_platforms_that_don_t_support_recursive_watching_natively_Colon_UseFsEvents_default_FixedPollingInterval_DynamicPriorityPolling_FixedChunkSizePolling: diag(6226, 3 /* Message */, "Specify_strategy_for_watching_directory_on_platforms_that_don_t_support_recursive_watching_natively__6226", "Specify strategy for watching directory on platforms that don't support recursive watching natively: 'UseFsEvents' (default), 'FixedPollingInterval', 'DynamicPriorityPolling', 'FixedChunkSizePolling'."),
  Specify_strategy_for_creating_a_polling_watch_when_it_fails_to_create_using_file_system_events_Colon_FixedInterval_default_PriorityInterval_DynamicPriority_FixedChunkSize: diag(6227, 3 /* Message */, "Specify_strategy_for_creating_a_polling_watch_when_it_fails_to_create_using_file_system_events_Colon_6227", "Specify strategy for creating a polling watch when it fails to create using file system events: 'FixedInterval' (default), 'PriorityInterval', 'DynamicPriority', 'FixedChunkSize'."),
  Tag_0_expects_at_least_1_arguments_but_the_JSX_factory_2_provides_at_most_3: diag(6229, 1 /* Error */, "Tag_0_expects_at_least_1_arguments_but_the_JSX_factory_2_provides_at_most_3_6229", "Tag '{0}' expects at least '{1}' arguments, but the JSX factory '{2}' provides at most '{3}'."),
  Option_0_can_only_be_specified_in_tsconfig_json_file_or_set_to_false_or_null_on_command_line: diag(6230, 1 /* Error */, "Option_0_can_only_be_specified_in_tsconfig_json_file_or_set_to_false_or_null_on_command_line_6230", "Option '{0}' can only be specified in 'tsconfig.json' file or set to 'false' or 'null' on command line."),
  Could_not_resolve_the_path_0_with_the_extensions_Colon_1: diag(6231, 1 /* Error */, "Could_not_resolve_the_path_0_with_the_extensions_Colon_1_6231", "Could not resolve the path '{0}' with the extensions: {1}."),
  Declaration_augments_declaration_in_another_file_This_cannot_be_serialized: diag(6232, 1 /* Error */, "Declaration_augments_declaration_in_another_file_This_cannot_be_serialized_6232", "Declaration augments declaration in another file. This cannot be serialized."),
  This_is_the_declaration_being_augmented_Consider_moving_the_augmenting_declaration_into_the_same_file: diag(6233, 1 /* Error */, "This_is_the_declaration_being_augmented_Consider_moving_the_augmenting_declaration_into_the_same_fil_6233", "This is the declaration being augmented. Consider moving the augmenting declaration into the same file."),
  This_expression_is_not_callable_because_it_is_a_get_accessor_Did_you_mean_to_use_it_without: diag(6234, 1 /* Error */, "This_expression_is_not_callable_because_it_is_a_get_accessor_Did_you_mean_to_use_it_without_6234", "This expression is not callable because it is a 'get' accessor. Did you mean to use it without '()'?"),
  Disable_loading_referenced_projects: diag(6235, 3 /* Message */, "Disable_loading_referenced_projects_6235", "Disable loading referenced projects."),
  Arguments_for_the_rest_parameter_0_were_not_provided: diag(6236, 1 /* Error */, "Arguments_for_the_rest_parameter_0_were_not_provided_6236", "Arguments for the rest parameter '{0}' were not provided."),
  Generates_an_event_trace_and_a_list_of_types: diag(6237, 3 /* Message */, "Generates_an_event_trace_and_a_list_of_types_6237", "Generates an event trace and a list of types."),
  Specify_the_module_specifier_to_be_used_to_import_the_jsx_and_jsxs_factory_functions_from_eg_react: diag(6238, 1 /* Error */, "Specify_the_module_specifier_to_be_used_to_import_the_jsx_and_jsxs_factory_functions_from_eg_react_6238", "Specify the module specifier to be used to import the 'jsx' and 'jsxs' factory functions from. eg, react"),
  File_0_exists_according_to_earlier_cached_lookups: diag(6239, 3 /* Message */, "File_0_exists_according_to_earlier_cached_lookups_6239", "File '{0}' exists according to earlier cached lookups."),
  File_0_does_not_exist_according_to_earlier_cached_lookups: diag(6240, 3 /* Message */, "File_0_does_not_exist_according_to_earlier_cached_lookups_6240", "File '{0}' does not exist according to earlier cached lookups."),
  Resolution_for_type_reference_directive_0_was_found_in_cache_from_location_1: diag(6241, 3 /* Message */, "Resolution_for_type_reference_directive_0_was_found_in_cache_from_location_1_6241", "Resolution for type reference directive '{0}' was found in cache from location '{1}'."),
  Resolving_type_reference_directive_0_containing_file_1: diag(6242, 3 /* Message */, "Resolving_type_reference_directive_0_containing_file_1_6242", "======== Resolving type reference directive '{0}', containing file '{1}'. ========"),
  Interpret_optional_property_types_as_written_rather_than_adding_undefined: diag(6243, 3 /* Message */, "Interpret_optional_property_types_as_written_rather_than_adding_undefined_6243", "Interpret optional property types as written, rather than adding 'undefined'."),
  Modules: diag(6244, 3 /* Message */, "Modules_6244", "Modules"),
  File_Management: diag(6245, 3 /* Message */, "File_Management_6245", "File Management"),
  Emit: diag(6246, 3 /* Message */, "Emit_6246", "Emit"),
  JavaScript_Support: diag(6247, 3 /* Message */, "JavaScript_Support_6247", "JavaScript Support"),
  Type_Checking: diag(6248, 3 /* Message */, "Type_Checking_6248", "Type Checking"),
  Editor_Support: diag(6249, 3 /* Message */, "Editor_Support_6249", "Editor Support"),
  Watch_and_Build_Modes: diag(6250, 3 /* Message */, "Watch_and_Build_Modes_6250", "Watch and Build Modes"),
  Compiler_Diagnostics: diag(6251, 3 /* Message */, "Compiler_Diagnostics_6251", "Compiler Diagnostics"),
  Interop_Constraints: diag(6252, 3 /* Message */, "Interop_Constraints_6252", "Interop Constraints"),
  Backwards_Compatibility: diag(6253, 3 /* Message */, "Backwards_Compatibility_6253", "Backwards Compatibility"),
  Language_and_Environment: diag(6254, 3 /* Message */, "Language_and_Environment_6254", "Language and Environment"),
  Projects: diag(6255, 3 /* Message */, "Projects_6255", "Projects"),
  Output_Formatting: diag(6256, 3 /* Message */, "Output_Formatting_6256", "Output Formatting"),
  Completeness: diag(6257, 3 /* Message */, "Completeness_6257", "Completeness"),
  _0_should_be_set_inside_the_compilerOptions_object_of_the_config_json_file: diag(6258, 1 /* Error */, "_0_should_be_set_inside_the_compilerOptions_object_of_the_config_json_file_6258", "'{0}' should be set inside the 'compilerOptions' object of the config json file"),
  Found_1_error_in_0: diag(6259, 3 /* Message */, "Found_1_error_in_0_6259", "Found 1 error in {0}"),
  Found_0_errors_in_the_same_file_starting_at_Colon_1: diag(6260, 3 /* Message */, "Found_0_errors_in_the_same_file_starting_at_Colon_1_6260", "Found {0} errors in the same file, starting at: {1}"),
  Found_0_errors_in_1_files: diag(6261, 3 /* Message */, "Found_0_errors_in_1_files_6261", "Found {0} errors in {1} files."),
  File_name_0_has_a_1_extension_looking_up_2_instead: diag(6262, 3 /* Message */, "File_name_0_has_a_1_extension_looking_up_2_instead_6262", "File name '{0}' has a '{1}' extension - looking up '{2}' instead."),
  Module_0_was_resolved_to_1_but_allowArbitraryExtensions_is_not_set: diag(6263, 1 /* Error */, "Module_0_was_resolved_to_1_but_allowArbitraryExtensions_is_not_set_6263", "Module '{0}' was resolved to '{1}', but '--allowArbitraryExtensions' is not set."),
  Enable_importing_files_with_any_extension_provided_a_declaration_file_is_present: diag(6264, 3 /* Message */, "Enable_importing_files_with_any_extension_provided_a_declaration_file_is_present_6264", "Enable importing files with any extension, provided a declaration file is present."),
  Resolving_type_reference_directive_for_program_that_specifies_custom_typeRoots_skipping_lookup_in_node_modules_folder: diag(6265, 3 /* Message */, "Resolving_type_reference_directive_for_program_that_specifies_custom_typeRoots_skipping_lookup_in_no_6265", "Resolving type reference directive for program that specifies custom typeRoots, skipping lookup in 'node_modules' folder."),
  Option_0_can_only_be_specified_on_command_line: diag(6266, 1 /* Error */, "Option_0_can_only_be_specified_on_command_line_6266", "Option '{0}' can only be specified on command line."),
  Directory_0_has_no_containing_package_json_scope_Imports_will_not_resolve: diag(6270, 3 /* Message */, "Directory_0_has_no_containing_package_json_scope_Imports_will_not_resolve_6270", "Directory '{0}' has no containing package.json scope. Imports will not resolve."),
  Import_specifier_0_does_not_exist_in_package_json_scope_at_path_1: diag(6271, 3 /* Message */, "Import_specifier_0_does_not_exist_in_package_json_scope_at_path_1_6271", "Import specifier '{0}' does not exist in package.json scope at path '{1}'."),
  Invalid_import_specifier_0_has_no_possible_resolutions: diag(6272, 3 /* Message */, "Invalid_import_specifier_0_has_no_possible_resolutions_6272", "Invalid import specifier '{0}' has no possible resolutions."),
  package_json_scope_0_has_no_imports_defined: diag(6273, 3 /* Message */, "package_json_scope_0_has_no_imports_defined_6273", "package.json scope '{0}' has no imports defined."),
  package_json_scope_0_explicitly_maps_specifier_1_to_null: diag(6274, 3 /* Message */, "package_json_scope_0_explicitly_maps_specifier_1_to_null_6274", "package.json scope '{0}' explicitly maps specifier '{1}' to null."),
  package_json_scope_0_has_invalid_type_for_target_of_specifier_1: diag(6275, 3 /* Message */, "package_json_scope_0_has_invalid_type_for_target_of_specifier_1_6275", "package.json scope '{0}' has invalid type for target of specifier '{1}'"),
  Export_specifier_0_does_not_exist_in_package_json_scope_at_path_1: diag(6276, 3 /* Message */, "Export_specifier_0_does_not_exist_in_package_json_scope_at_path_1_6276", "Export specifier '{0}' does not exist in package.json scope at path '{1}'."),
  Resolution_of_non_relative_name_failed_trying_with_modern_Node_resolution_features_disabled_to_see_if_npm_library_needs_configuration_update: diag(6277, 3 /* Message */, "Resolution_of_non_relative_name_failed_trying_with_modern_Node_resolution_features_disabled_to_see_i_6277", "Resolution of non-relative name failed; trying with modern Node resolution features disabled to see if npm library needs configuration update."),
  There_are_types_at_0_but_this_result_could_not_be_resolved_when_respecting_package_json_exports_The_1_library_may_need_to_update_its_package_json_or_typings: diag(6278, 3 /* Message */, "There_are_types_at_0_but_this_result_could_not_be_resolved_when_respecting_package_json_exports_The__6278", `There are types at '{0}', but this result could not be resolved when respecting package.json "exports". The '{1}' library may need to update its package.json or typings.`),
  Resolution_of_non_relative_name_failed_trying_with_moduleResolution_bundler_to_see_if_project_may_need_configuration_update: diag(6279, 3 /* Message */, "Resolution_of_non_relative_name_failed_trying_with_moduleResolution_bundler_to_see_if_project_may_ne_6279", "Resolution of non-relative name failed; trying with '--moduleResolution bundler' to see if project may need configuration update."),
  There_are_types_at_0_but_this_result_could_not_be_resolved_under_your_current_moduleResolution_setting_Consider_updating_to_node16_nodenext_or_bundler: diag(6280, 3 /* Message */, "There_are_types_at_0_but_this_result_could_not_be_resolved_under_your_current_moduleResolution_setti_6280", "There are types at '{0}', but this result could not be resolved under your current 'moduleResolution' setting. Consider updating to 'node16', 'nodenext', or 'bundler'."),
  package_json_has_a_peerDependencies_field: diag(6281, 3 /* Message */, "package_json_has_a_peerDependencies_field_6281", "'package.json' has a 'peerDependencies' field."),
  Found_peerDependency_0_with_1_version: diag(6282, 3 /* Message */, "Found_peerDependency_0_with_1_version_6282", "Found peerDependency '{0}' with '{1}' version."),
  Failed_to_find_peerDependency_0: diag(6283, 3 /* Message */, "Failed_to_find_peerDependency_0_6283", "Failed to find peerDependency '{0}'."),
  Enable_project_compilation: diag(6302, 3 /* Message */, "Enable_project_compilation_6302", "Enable project compilation"),
  Composite_projects_may_not_disable_declaration_emit: diag(6304, 1 /* Error */, "Composite_projects_may_not_disable_declaration_emit_6304", "Composite projects may not disable declaration emit."),
  Output_file_0_has_not_been_built_from_source_file_1: diag(6305, 1 /* Error */, "Output_file_0_has_not_been_built_from_source_file_1_6305", "Output file '{0}' has not been built from source file '{1}'."),
  Referenced_project_0_must_have_setting_composite_Colon_true: diag(6306, 1 /* Error */, "Referenced_project_0_must_have_setting_composite_Colon_true_6306", `Referenced project '{0}' must have setting "composite": true.`),
  File_0_is_not_listed_within_the_file_list_of_project_1_Projects_must_list_all_files_or_use_an_include_pattern: diag(6307, 1 /* Error */, "File_0_is_not_listed_within_the_file_list_of_project_1_Projects_must_list_all_files_or_use_an_includ_6307", "File '{0}' is not listed within the file list of project '{1}'. Projects must list all files or use an 'include' pattern."),
  Referenced_project_0_may_not_disable_emit: diag(6310, 1 /* Error */, "Referenced_project_0_may_not_disable_emit_6310", "Referenced project '{0}' may not disable emit."),
  Project_0_is_out_of_date_because_output_1_is_older_than_input_2: diag(6350, 3 /* Message */, "Project_0_is_out_of_date_because_output_1_is_older_than_input_2_6350", "Project '{0}' is out of date because output '{1}' is older than input '{2}'"),
  Project_0_is_up_to_date_because_newest_input_1_is_older_than_output_2: diag(6351, 3 /* Message */, "Project_0_is_up_to_date_because_newest_input_1_is_older_than_output_2_6351", "Project '{0}' is up to date because newest input '{1}' is older than output '{2}'"),
  Project_0_is_out_of_date_because_output_file_1_does_not_exist: diag(6352, 3 /* Message */, "Project_0_is_out_of_date_because_output_file_1_does_not_exist_6352", "Project '{0}' is out of date because output file '{1}' does not exist"),
  Project_0_is_out_of_date_because_its_dependency_1_is_out_of_date: diag(6353, 3 /* Message */, "Project_0_is_out_of_date_because_its_dependency_1_is_out_of_date_6353", "Project '{0}' is out of date because its dependency '{1}' is out of date"),
  Project_0_is_up_to_date_with_d_ts_files_from_its_dependencies: diag(6354, 3 /* Message */, "Project_0_is_up_to_date_with_d_ts_files_from_its_dependencies_6354", "Project '{0}' is up to date with .d.ts files from its dependencies"),
  Projects_in_this_build_Colon_0: diag(6355, 3 /* Message */, "Projects_in_this_build_Colon_0_6355", "Projects in this build: {0}"),
  A_non_dry_build_would_delete_the_following_files_Colon_0: diag(6356, 3 /* Message */, "A_non_dry_build_would_delete_the_following_files_Colon_0_6356", "A non-dry build would delete the following files: {0}"),
  A_non_dry_build_would_build_project_0: diag(6357, 3 /* Message */, "A_non_dry_build_would_build_project_0_6357", "A non-dry build would build project '{0}'"),
  Building_project_0: diag(6358, 3 /* Message */, "Building_project_0_6358", "Building project '{0}'..."),
  Updating_output_timestamps_of_project_0: diag(6359, 3 /* Message */, "Updating_output_timestamps_of_project_0_6359", "Updating output timestamps of project '{0}'..."),
  Project_0_is_up_to_date: diag(6361, 3 /* Message */, "Project_0_is_up_to_date_6361", "Project '{0}' is up to date"),
  Skipping_build_of_project_0_because_its_dependency_1_has_errors: diag(6362, 3 /* Message */, "Skipping_build_of_project_0_because_its_dependency_1_has_errors_6362", "Skipping build of project '{0}' because its dependency '{1}' has errors"),
  Project_0_can_t_be_built_because_its_dependency_1_has_errors: diag(6363, 3 /* Message */, "Project_0_can_t_be_built_because_its_dependency_1_has_errors_6363", "Project '{0}' can't be built because its dependency '{1}' has errors"),
  Build_one_or_more_projects_and_their_dependencies_if_out_of_date: diag(6364, 3 /* Message */, "Build_one_or_more_projects_and_their_dependencies_if_out_of_date_6364", "Build one or more projects and their dependencies, if out of date"),
  Delete_the_outputs_of_all_projects: diag(6365, 3 /* Message */, "Delete_the_outputs_of_all_projects_6365", "Delete the outputs of all projects."),
  Show_what_would_be_built_or_deleted_if_specified_with_clean: diag(6367, 3 /* Message */, "Show_what_would_be_built_or_deleted_if_specified_with_clean_6367", "Show what would be built (or deleted, if specified with '--clean')"),
  Option_build_must_be_the_first_command_line_argument: diag(6369, 1 /* Error */, "Option_build_must_be_the_first_command_line_argument_6369", "Option '--build' must be the first command line argument."),
  Options_0_and_1_cannot_be_combined: diag(6370, 1 /* Error */, "Options_0_and_1_cannot_be_combined_6370", "Options '{0}' and '{1}' cannot be combined."),
  Updating_unchanged_output_timestamps_of_project_0: diag(6371, 3 /* Message */, "Updating_unchanged_output_timestamps_of_project_0_6371", "Updating unchanged output timestamps of project '{0}'..."),
  A_non_dry_build_would_update_timestamps_for_output_of_project_0: diag(6374, 3 /* Message */, "A_non_dry_build_would_update_timestamps_for_output_of_project_0_6374", "A non-dry build would update timestamps for output of project '{0}'"),
  Cannot_write_file_0_because_it_will_overwrite_tsbuildinfo_file_generated_by_referenced_project_1: diag(6377, 1 /* Error */, "Cannot_write_file_0_because_it_will_overwrite_tsbuildinfo_file_generated_by_referenced_project_1_6377", "Cannot write file '{0}' because it will overwrite '.tsbuildinfo' file generated by referenced project '{1}'"),
  Composite_projects_may_not_disable_incremental_compilation: diag(6379, 1 /* Error */, "Composite_projects_may_not_disable_incremental_compilation_6379", "Composite projects may not disable incremental compilation."),
  Specify_file_to_store_incremental_compilation_information: diag(6380, 3 /* Message */, "Specify_file_to_store_incremental_compilation_information_6380", "Specify file to store incremental compilation information"),
  Project_0_is_out_of_date_because_output_for_it_was_generated_with_version_1_that_differs_with_current_version_2: diag(6381, 3 /* Message */, "Project_0_is_out_of_date_because_output_for_it_was_generated_with_version_1_that_differs_with_curren_6381", "Project '{0}' is out of date because output for it was generated with version '{1}' that differs with current version '{2}'"),
  Skipping_build_of_project_0_because_its_dependency_1_was_not_built: diag(6382, 3 /* Message */, "Skipping_build_of_project_0_because_its_dependency_1_was_not_built_6382", "Skipping build of project '{0}' because its dependency '{1}' was not built"),
  Project_0_can_t_be_built_because_its_dependency_1_was_not_built: diag(6383, 3 /* Message */, "Project_0_can_t_be_built_because_its_dependency_1_was_not_built_6383", "Project '{0}' can't be built because its dependency '{1}' was not built"),
  Have_recompiles_in_incremental_and_watch_assume_that_changes_within_a_file_will_only_affect_files_directly_depending_on_it: diag(6384, 3 /* Message */, "Have_recompiles_in_incremental_and_watch_assume_that_changes_within_a_file_will_only_affect_files_di_6384", "Have recompiles in '--incremental' and '--watch' assume that changes within a file will only affect files directly depending on it."),
  _0_is_deprecated: diag(
    6385,
    2 /* Suggestion */,
    "_0_is_deprecated_6385",
    "'{0}' is deprecated.",
    /*reportsUnnecessary*/
    void 0,
    /*elidedInCompatabilityPyramid*/
    void 0,
    /*reportsDeprecated*/
    true
  ),
  Performance_timings_for_diagnostics_or_extendedDiagnostics_are_not_available_in_this_session_A_native_implementation_of_the_Web_Performance_API_could_not_be_found: diag(6386, 3 /* Message */, "Performance_timings_for_diagnostics_or_extendedDiagnostics_are_not_available_in_this_session_A_nativ_6386", "Performance timings for '--diagnostics' or '--extendedDiagnostics' are not available in this session. A native implementation of the Web Performance API could not be found."),
  The_signature_0_of_1_is_deprecated: diag(
    6387,
    2 /* Suggestion */,
    "The_signature_0_of_1_is_deprecated_6387",
    "The signature '{0}' of '{1}' is deprecated.",
    /*reportsUnnecessary*/
    void 0,
    /*elidedInCompatabilityPyramid*/
    void 0,
    /*reportsDeprecated*/
    true
  ),
  Project_0_is_being_forcibly_rebuilt: diag(6388, 3 /* Message */, "Project_0_is_being_forcibly_rebuilt_6388", "Project '{0}' is being forcibly rebuilt"),
  Reusing_resolution_of_module_0_from_1_of_old_program_it_was_not_resolved: diag(6389, 3 /* Message */, "Reusing_resolution_of_module_0_from_1_of_old_program_it_was_not_resolved_6389", "Reusing resolution of module '{0}' from '{1}' of old program, it was not resolved."),
  Reusing_resolution_of_type_reference_directive_0_from_1_of_old_program_it_was_successfully_resolved_to_2: diag(6390, 3 /* Message */, "Reusing_resolution_of_type_reference_directive_0_from_1_of_old_program_it_was_successfully_resolved__6390", "Reusing resolution of type reference directive '{0}' from '{1}' of old program, it was successfully resolved to '{2}'."),
  Reusing_resolution_of_type_reference_directive_0_from_1_of_old_program_it_was_successfully_resolved_to_2_with_Package_ID_3: diag(6391, 3 /* Message */, "Reusing_resolution_of_type_reference_directive_0_from_1_of_old_program_it_was_successfully_resolved__6391", "Reusing resolution of type reference directive '{0}' from '{1}' of old program, it was successfully resolved to '{2}' with Package ID '{3}'."),
  Reusing_resolution_of_type_reference_directive_0_from_1_of_old_program_it_was_not_resolved: diag(6392, 3 /* Message */, "Reusing_resolution_of_type_reference_directive_0_from_1_of_old_program_it_was_not_resolved_6392", "Reusing resolution of type reference directive '{0}' from '{1}' of old program, it was not resolved."),
  Reusing_resolution_of_module_0_from_1_found_in_cache_from_location_2_it_was_successfully_resolved_to_3: diag(6393, 3 /* Message */, "Reusing_resolution_of_module_0_from_1_found_in_cache_from_location_2_it_was_successfully_resolved_to_6393", "Reusing resolution of module '{0}' from '{1}' found in cache from location '{2}', it was successfully resolved to '{3}'."),
  Reusing_resolution_of_module_0_from_1_found_in_cache_from_location_2_it_was_successfully_resolved_to_3_with_Package_ID_4: diag(6394, 3 /* Message */, "Reusing_resolution_of_module_0_from_1_found_in_cache_from_location_2_it_was_successfully_resolved_to_6394", "Reusing resolution of module '{0}' from '{1}' found in cache from location '{2}', it was successfully resolved to '{3}' with Package ID '{4}'."),
  Reusing_resolution_of_module_0_from_1_found_in_cache_from_location_2_it_was_not_resolved: diag(6395, 3 /* Message */, "Reusing_resolution_of_module_0_from_1_found_in_cache_from_location_2_it_was_not_resolved_6395", "Reusing resolution of module '{0}' from '{1}' found in cache from location '{2}', it was not resolved."),
  Reusing_resolution_of_type_reference_directive_0_from_1_found_in_cache_from_location_2_it_was_successfully_resolved_to_3: diag(6396, 3 /* Message */, "Reusing_resolution_of_type_reference_directive_0_from_1_found_in_cache_from_location_2_it_was_succes_6396", "Reusing resolution of type reference directive '{0}' from '{1}' found in cache from location '{2}', it was successfully resolved to '{3}'."),
  Reusing_resolution_of_type_reference_directive_0_from_1_found_in_cache_from_location_2_it_was_successfully_resolved_to_3_with_Package_ID_4: diag(6397, 3 /* Message */, "Reusing_resolution_of_type_reference_directive_0_from_1_found_in_cache_from_location_2_it_was_succes_6397", "Reusing resolution of type reference directive '{0}' from '{1}' found in cache from location '{2}', it was successfully resolved to '{3}' with Package ID '{4}'."),
  Reusing_resolution_of_type_reference_directive_0_from_1_found_in_cache_from_location_2_it_was_not_resolved: diag(6398, 3 /* Message */, "Reusing_resolution_of_type_reference_directive_0_from_1_found_in_cache_from_location_2_it_was_not_re_6398", "Reusing resolution of type reference directive '{0}' from '{1}' found in cache from location '{2}', it was not resolved."),
  Project_0_is_out_of_date_because_buildinfo_file_1_indicates_that_some_of_the_changes_were_not_emitted: diag(6399, 3 /* Message */, "Project_0_is_out_of_date_because_buildinfo_file_1_indicates_that_some_of_the_changes_were_not_emitte_6399", "Project '{0}' is out of date because buildinfo file '{1}' indicates that some of the changes were not emitted"),
  Project_0_is_up_to_date_but_needs_to_update_timestamps_of_output_files_that_are_older_than_input_files: diag(6400, 3 /* Message */, "Project_0_is_up_to_date_but_needs_to_update_timestamps_of_output_files_that_are_older_than_input_fil_6400", "Project '{0}' is up to date but needs to update timestamps of output files that are older than input files"),
  Project_0_is_out_of_date_because_there_was_error_reading_file_1: diag(6401, 3 /* Message */, "Project_0_is_out_of_date_because_there_was_error_reading_file_1_6401", "Project '{0}' is out of date because there was error reading file '{1}'"),
  Resolving_in_0_mode_with_conditions_1: diag(6402, 3 /* Message */, "Resolving_in_0_mode_with_conditions_1_6402", "Resolving in {0} mode with conditions {1}."),
  Matched_0_condition_1: diag(6403, 3 /* Message */, "Matched_0_condition_1_6403", "Matched '{0}' condition '{1}'."),
  Using_0_subpath_1_with_target_2: diag(6404, 3 /* Message */, "Using_0_subpath_1_with_target_2_6404", "Using '{0}' subpath '{1}' with target '{2}'."),
  Saw_non_matching_condition_0: diag(6405, 3 /* Message */, "Saw_non_matching_condition_0_6405", "Saw non-matching condition '{0}'."),
  Project_0_is_out_of_date_because_buildinfo_file_1_indicates_there_is_change_in_compilerOptions: diag(6406, 3 /* Message */, "Project_0_is_out_of_date_because_buildinfo_file_1_indicates_there_is_change_in_compilerOptions_6406", "Project '{0}' is out of date because buildinfo file '{1}' indicates there is change in compilerOptions"),
  Allow_imports_to_include_TypeScript_file_extensions_Requires_moduleResolution_bundler_and_either_noEmit_or_emitDeclarationOnly_to_be_set: diag(6407, 3 /* Message */, "Allow_imports_to_include_TypeScript_file_extensions_Requires_moduleResolution_bundler_and_either_noE_6407", "Allow imports to include TypeScript file extensions. Requires '--moduleResolution bundler' and either '--noEmit' or '--emitDeclarationOnly' to be set."),
  Use_the_package_json_exports_field_when_resolving_package_imports: diag(6408, 3 /* Message */, "Use_the_package_json_exports_field_when_resolving_package_imports_6408", "Use the package.json 'exports' field when resolving package imports."),
  Use_the_package_json_imports_field_when_resolving_imports: diag(6409, 3 /* Message */, "Use_the_package_json_imports_field_when_resolving_imports_6409", "Use the package.json 'imports' field when resolving imports."),
  Conditions_to_set_in_addition_to_the_resolver_specific_defaults_when_resolving_imports: diag(6410, 3 /* Message */, "Conditions_to_set_in_addition_to_the_resolver_specific_defaults_when_resolving_imports_6410", "Conditions to set in addition to the resolver-specific defaults when resolving imports."),
  true_when_moduleResolution_is_node16_nodenext_or_bundler_otherwise_false: diag(6411, 3 /* Message */, "true_when_moduleResolution_is_node16_nodenext_or_bundler_otherwise_false_6411", "`true` when 'moduleResolution' is 'node16', 'nodenext', or 'bundler'; otherwise `false`."),
  Project_0_is_out_of_date_because_buildinfo_file_1_indicates_that_file_2_was_root_file_of_compilation_but_not_any_more: diag(6412, 3 /* Message */, "Project_0_is_out_of_date_because_buildinfo_file_1_indicates_that_file_2_was_root_file_of_compilation_6412", "Project '{0}' is out of date because buildinfo file '{1}' indicates that file '{2}' was root file of compilation but not any more."),
  Entering_conditional_exports: diag(6413, 3 /* Message */, "Entering_conditional_exports_6413", "Entering conditional exports."),
  Resolved_under_condition_0: diag(6414, 3 /* Message */, "Resolved_under_condition_0_6414", "Resolved under condition '{0}'."),
  Failed_to_resolve_under_condition_0: diag(6415, 3 /* Message */, "Failed_to_resolve_under_condition_0_6415", "Failed to resolve under condition '{0}'."),
  Exiting_conditional_exports: diag(6416, 3 /* Message */, "Exiting_conditional_exports_6416", "Exiting conditional exports."),
  Searching_all_ancestor_node_modules_directories_for_preferred_extensions_Colon_0: diag(6417, 3 /* Message */, "Searching_all_ancestor_node_modules_directories_for_preferred_extensions_Colon_0_6417", "Searching all ancestor node_modules directories for preferred extensions: {0}."),
  Searching_all_ancestor_node_modules_directories_for_fallback_extensions_Colon_0: diag(6418, 3 /* Message */, "Searching_all_ancestor_node_modules_directories_for_fallback_extensions_Colon_0_6418", "Searching all ancestor node_modules directories for fallback extensions: {0}."),
  Project_0_is_out_of_date_because_buildinfo_file_1_indicates_that_program_needs_to_report_errors: diag(6419, 3 /* Message */, "Project_0_is_out_of_date_because_buildinfo_file_1_indicates_that_program_needs_to_report_errors_6419", "Project '{0}' is out of date because buildinfo file '{1}' indicates that program needs to report errors."),
  Project_0_is_out_of_date_because_1: diag(6420, 3 /* Message */, "Project_0_is_out_of_date_because_1_6420", "Project '{0}' is out of date because {1}."),
  Rewrite_ts_tsx_mts_and_cts_file_extensions_in_relative_import_paths_to_their_JavaScript_equivalent_in_output_files: diag(6421, 3 /* Message */, "Rewrite_ts_tsx_mts_and_cts_file_extensions_in_relative_import_paths_to_their_JavaScript_equivalent_i_6421", "Rewrite '.ts', '.tsx', '.mts', and '.cts' file extensions in relative import paths to their JavaScript equivalent in output files."),
  The_expected_type_comes_from_property_0_which_is_declared_here_on_type_1: diag(6500, 3 /* Message */, "The_expected_type_comes_from_property_0_which_is_declared_here_on_type_1_6500", "The expected type comes from property '{0}' which is declared here on type '{1}'"),
  The_expected_type_comes_from_this_index_signature: diag(6501, 3 /* Message */, "The_expected_type_comes_from_this_index_signature_6501", "The expected type comes from this index signature."),
  The_expected_type_comes_from_the_return_type_of_this_signature: diag(6502, 3 /* Message */, "The_expected_type_comes_from_the_return_type_of_this_signature_6502", "The expected type comes from the return type of this signature."),
  Print_names_of_files_that_are_part_of_the_compilation_and_then_stop_processing: diag(6503, 3 /* Message */, "Print_names_of_files_that_are_part_of_the_compilation_and_then_stop_processing_6503", "Print names of files that are part of the compilation and then stop processing."),
  File_0_is_a_JavaScript_file_Did_you_mean_to_enable_the_allowJs_option: diag(6504, 1 /* Error */, "File_0_is_a_JavaScript_file_Did_you_mean_to_enable_the_allowJs_option_6504", "File '{0}' is a JavaScript file. Did you mean to enable the 'allowJs' option?"),
  Print_names_of_files_and_the_reason_they_are_part_of_the_compilation: diag(6505, 3 /* Message */, "Print_names_of_files_and_the_reason_they_are_part_of_the_compilation_6505", "Print names of files and the reason they are part of the compilation."),
  Consider_adding_a_declare_modifier_to_this_class: diag(6506, 3 /* Message */, "Consider_adding_a_declare_modifier_to_this_class_6506", "Consider adding a 'declare' modifier to this class."),
  Allow_JavaScript_files_to_be_a_part_of_your_program_Use_the_checkJS_option_to_get_errors_from_these_files: diag(6600, 3 /* Message */, "Allow_JavaScript_files_to_be_a_part_of_your_program_Use_the_checkJS_option_to_get_errors_from_these__6600", "Allow JavaScript files to be a part of your program. Use the 'checkJS' option to get errors from these files."),
  Allow_import_x_from_y_when_a_module_doesn_t_have_a_default_export: diag(6601, 3 /* Message */, "Allow_import_x_from_y_when_a_module_doesn_t_have_a_default_export_6601", "Allow 'import x from y' when a module doesn't have a default export."),
  Allow_accessing_UMD_globals_from_modules: diag(6602, 3 /* Message */, "Allow_accessing_UMD_globals_from_modules_6602", "Allow accessing UMD globals from modules."),
  Disable_error_reporting_for_unreachable_code: diag(6603, 3 /* Message */, "Disable_error_reporting_for_unreachable_code_6603", "Disable error reporting for unreachable code."),
  Disable_error_reporting_for_unused_labels: diag(6604, 3 /* Message */, "Disable_error_reporting_for_unused_labels_6604", "Disable error reporting for unused labels."),
  Ensure_use_strict_is_always_emitted: diag(6605, 3 /* Message */, "Ensure_use_strict_is_always_emitted_6605", "Ensure 'use strict' is always emitted."),
  Have_recompiles_in_projects_that_use_incremental_and_watch_mode_assume_that_changes_within_a_file_will_only_affect_files_directly_depending_on_it: diag(6606, 3 /* Message */, "Have_recompiles_in_projects_that_use_incremental_and_watch_mode_assume_that_changes_within_a_file_wi_6606", "Have recompiles in projects that use 'incremental' and 'watch' mode assume that changes within a file will only affect files directly depending on it."),
  Specify_the_base_directory_to_resolve_non_relative_module_names: diag(6607, 3 /* Message */, "Specify_the_base_directory_to_resolve_non_relative_module_names_6607", "Specify the base directory to resolve non-relative module names."),
  No_longer_supported_In_early_versions_manually_set_the_text_encoding_for_reading_files: diag(6608, 3 /* Message */, "No_longer_supported_In_early_versions_manually_set_the_text_encoding_for_reading_files_6608", "No longer supported. In early versions, manually set the text encoding for reading files."),
  Enable_error_reporting_in_type_checked_JavaScript_files: diag(6609, 3 /* Message */, "Enable_error_reporting_in_type_checked_JavaScript_files_6609", "Enable error reporting in type-checked JavaScript files."),
  Enable_constraints_that_allow_a_TypeScript_project_to_be_used_with_project_references: diag(6611, 3 /* Message */, "Enable_constraints_that_allow_a_TypeScript_project_to_be_used_with_project_references_6611", "Enable constraints that allow a TypeScript project to be used with project references."),
  Generate_d_ts_files_from_TypeScript_and_JavaScript_files_in_your_project: diag(6612, 3 /* Message */, "Generate_d_ts_files_from_TypeScript_and_JavaScript_files_in_your_project_6612", "Generate .d.ts files from TypeScript and JavaScript files in your project."),
  Specify_the_output_directory_for_generated_declaration_files: diag(6613, 3 /* Message */, "Specify_the_output_directory_for_generated_declaration_files_6613", "Specify the output directory for generated declaration files."),
  Create_sourcemaps_for_d_ts_files: diag(6614, 3 /* Message */, "Create_sourcemaps_for_d_ts_files_6614", "Create sourcemaps for d.ts files."),
  Output_compiler_performance_information_after_building: diag(6615, 3 /* Message */, "Output_compiler_performance_information_after_building_6615", "Output compiler performance information after building."),
  Disables_inference_for_type_acquisition_by_looking_at_filenames_in_a_project: diag(6616, 3 /* Message */, "Disables_inference_for_type_acquisition_by_looking_at_filenames_in_a_project_6616", "Disables inference for type acquisition by looking at filenames in a project."),
  Reduce_the_number_of_projects_loaded_automatically_by_TypeScript: diag(6617, 3 /* Message */, "Reduce_the_number_of_projects_loaded_automatically_by_TypeScript_6617", "Reduce the number of projects loaded automatically by TypeScript."),
  Remove_the_20mb_cap_on_total_source_code_size_for_JavaScript_files_in_the_TypeScript_language_server: diag(6618, 3 /* Message */, "Remove_the_20mb_cap_on_total_source_code_size_for_JavaScript_files_in_the_TypeScript_language_server_6618", "Remove the 20mb cap on total source code size for JavaScript files in the TypeScript language server."),
  Opt_a_project_out_of_multi_project_reference_checking_when_editing: diag(6619, 3 /* Message */, "Opt_a_project_out_of_multi_project_reference_checking_when_editing_6619", "Opt a project out of multi-project reference checking when editing."),
  Disable_preferring_source_files_instead_of_declaration_files_when_referencing_composite_projects: diag(6620, 3 /* Message */, "Disable_preferring_source_files_instead_of_declaration_files_when_referencing_composite_projects_6620", "Disable preferring source files instead of declaration files when referencing composite projects."),
  Emit_more_compliant_but_verbose_and_less_performant_JavaScript_for_iteration: diag(6621, 3 /* Message */, "Emit_more_compliant_but_verbose_and_less_performant_JavaScript_for_iteration_6621", "Emit more compliant, but verbose and less performant JavaScript for iteration."),
  Emit_a_UTF_8_Byte_Order_Mark_BOM_in_the_beginning_of_output_files: diag(6622, 3 /* Message */, "Emit_a_UTF_8_Byte_Order_Mark_BOM_in_the_beginning_of_output_files_6622", "Emit a UTF-8 Byte Order Mark (BOM) in the beginning of output files."),
  Only_output_d_ts_files_and_not_JavaScript_files: diag(6623, 3 /* Message */, "Only_output_d_ts_files_and_not_JavaScript_files_6623", "Only output d.ts files and not JavaScript files."),
  Emit_design_type_metadata_for_decorated_declarations_in_source_files: diag(6624, 3 /* Message */, "Emit_design_type_metadata_for_decorated_declarations_in_source_files_6624", "Emit design-type metadata for decorated declarations in source files."),
  Disable_the_type_acquisition_for_JavaScript_projects: diag(6625, 3 /* Message */, "Disable_the_type_acquisition_for_JavaScript_projects_6625", "Disable the type acquisition for JavaScript projects"),
  Emit_additional_JavaScript_to_ease_support_for_importing_CommonJS_modules_This_enables_allowSyntheticDefaultImports_for_type_compatibility: diag(6626, 3 /* Message */, "Emit_additional_JavaScript_to_ease_support_for_importing_CommonJS_modules_This_enables_allowSyntheti_6626", "Emit additional JavaScript to ease support for importing CommonJS modules. This enables 'allowSyntheticDefaultImports' for type compatibility."),
  Filters_results_from_the_include_option: diag(6627, 3 /* Message */, "Filters_results_from_the_include_option_6627", "Filters results from the `include` option."),
  Remove_a_list_of_directories_from_the_watch_process: diag(6628, 3 /* Message */, "Remove_a_list_of_directories_from_the_watch_process_6628", "Remove a list of directories from the watch process."),
  Remove_a_list_of_files_from_the_watch_mode_s_processing: diag(6629, 3 /* Message */, "Remove_a_list_of_files_from_the_watch_mode_s_processing_6629", "Remove a list of files from the watch mode's processing."),
  Enable_experimental_support_for_legacy_experimental_decorators: diag(6630, 3 /* Message */, "Enable_experimental_support_for_legacy_experimental_decorators_6630", "Enable experimental support for legacy experimental decorators."),
  Print_files_read_during_the_compilation_including_why_it_was_included: diag(6631, 3 /* Message */, "Print_files_read_during_the_compilation_including_why_it_was_included_6631", "Print files read during the compilation including why it was included."),
  Output_more_detailed_compiler_performance_information_after_building: diag(6632, 3 /* Message */, "Output_more_detailed_compiler_performance_information_after_building_6632", "Output more detailed compiler performance information after building."),
  Specify_one_or_more_path_or_node_module_references_to_base_configuration_files_from_which_settings_are_inherited: diag(6633, 3 /* Message */, "Specify_one_or_more_path_or_node_module_references_to_base_configuration_files_from_which_settings_a_6633", "Specify one or more path or node module references to base configuration files from which settings are inherited."),
  Specify_what_approach_the_watcher_should_use_if_the_system_runs_out_of_native_file_watchers: diag(6634, 3 /* Message */, "Specify_what_approach_the_watcher_should_use_if_the_system_runs_out_of_native_file_watchers_6634", "Specify what approach the watcher should use if the system runs out of native file watchers."),
  Include_a_list_of_files_This_does_not_support_glob_patterns_as_opposed_to_include: diag(6635, 3 /* Message */, "Include_a_list_of_files_This_does_not_support_glob_patterns_as_opposed_to_include_6635", "Include a list of files. This does not support glob patterns, as opposed to `include`."),
  Build_all_projects_including_those_that_appear_to_be_up_to_date: diag(6636, 3 /* Message */, "Build_all_projects_including_those_that_appear_to_be_up_to_date_6636", "Build all projects, including those that appear to be up to date."),
  Ensure_that_casing_is_correct_in_imports: diag(6637, 3 /* Message */, "Ensure_that_casing_is_correct_in_imports_6637", "Ensure that casing is correct in imports."),
  Emit_a_v8_CPU_profile_of_the_compiler_run_for_debugging: diag(6638, 3 /* Message */, "Emit_a_v8_CPU_profile_of_the_compiler_run_for_debugging_6638", "Emit a v8 CPU profile of the compiler run for debugging."),
  Allow_importing_helper_functions_from_tslib_once_per_project_instead_of_including_them_per_file: diag(6639, 3 /* Message */, "Allow_importing_helper_functions_from_tslib_once_per_project_instead_of_including_them_per_file_6639", "Allow importing helper functions from tslib once per project, instead of including them per-file."),
  Skip_building_downstream_projects_on_error_in_upstream_project: diag(6640, 3 /* Message */, "Skip_building_downstream_projects_on_error_in_upstream_project_6640", "Skip building downstream projects on error in upstream project."),
  Specify_a_list_of_glob_patterns_that_match_files_to_be_included_in_compilation: diag(6641, 3 /* Message */, "Specify_a_list_of_glob_patterns_that_match_files_to_be_included_in_compilation_6641", "Specify a list of glob patterns that match files to be included in compilation."),
  Save_tsbuildinfo_files_to_allow_for_incremental_compilation_of_projects: diag(6642, 3 /* Message */, "Save_tsbuildinfo_files_to_allow_for_incremental_compilation_of_projects_6642", "Save .tsbuildinfo files to allow for incremental compilation of projects."),
  Include_sourcemap_files_inside_the_emitted_JavaScript: diag(6643, 3 /* Message */, "Include_sourcemap_files_inside_the_emitted_JavaScript_6643", "Include sourcemap files inside the emitted JavaScript."),
  Include_source_code_in_the_sourcemaps_inside_the_emitted_JavaScript: diag(6644, 3 /* Message */, "Include_source_code_in_the_sourcemaps_inside_the_emitted_JavaScript_6644", "Include source code in the sourcemaps inside the emitted JavaScript."),
  Ensure_that_each_file_can_be_safely_transpiled_without_relying_on_other_imports: diag(6645, 3 /* Message */, "Ensure_that_each_file_can_be_safely_transpiled_without_relying_on_other_imports_6645", "Ensure that each file can be safely transpiled without relying on other imports."),
  Specify_what_JSX_code_is_generated: diag(6646, 3 /* Message */, "Specify_what_JSX_code_is_generated_6646", "Specify what JSX code is generated."),
  Specify_the_JSX_factory_function_used_when_targeting_React_JSX_emit_e_g_React_createElement_or_h: diag(6647, 3 /* Message */, "Specify_the_JSX_factory_function_used_when_targeting_React_JSX_emit_e_g_React_createElement_or_h_6647", "Specify the JSX factory function used when targeting React JSX emit, e.g. 'React.createElement' or 'h'."),
  Specify_the_JSX_Fragment_reference_used_for_fragments_when_targeting_React_JSX_emit_e_g_React_Fragment_or_Fragment: diag(6648, 3 /* Message */, "Specify_the_JSX_Fragment_reference_used_for_fragments_when_targeting_React_JSX_emit_e_g_React_Fragme_6648", "Specify the JSX Fragment reference used for fragments when targeting React JSX emit e.g. 'React.Fragment' or 'Fragment'."),
  Specify_module_specifier_used_to_import_the_JSX_factory_functions_when_using_jsx_Colon_react_jsx_Asterisk: diag(6649, 3 /* Message */, "Specify_module_specifier_used_to_import_the_JSX_factory_functions_when_using_jsx_Colon_react_jsx_Ast_6649", "Specify module specifier used to import the JSX factory functions when using 'jsx: react-jsx*'."),
  Make_keyof_only_return_strings_instead_of_string_numbers_or_symbols_Legacy_option: diag(6650, 3 /* Message */, "Make_keyof_only_return_strings_instead_of_string_numbers_or_symbols_Legacy_option_6650", "Make keyof only return strings instead of string, numbers or symbols. Legacy option."),
  Specify_a_set_of_bundled_library_declaration_files_that_describe_the_target_runtime_environment: diag(6651, 3 /* Message */, "Specify_a_set_of_bundled_library_declaration_files_that_describe_the_target_runtime_environment_6651", "Specify a set of bundled library declaration files that describe the target runtime environment."),
  Print_the_names_of_emitted_files_after_a_compilation: diag(6652, 3 /* Message */, "Print_the_names_of_emitted_files_after_a_compilation_6652", "Print the names of emitted files after a compilation."),
  Print_all_of_the_files_read_during_the_compilation: diag(6653, 3 /* Message */, "Print_all_of_the_files_read_during_the_compilation_6653", "Print all of the files read during the compilation."),
  Set_the_language_of_the_messaging_from_TypeScript_This_does_not_affect_emit: diag(6654, 3 /* Message */, "Set_the_language_of_the_messaging_from_TypeScript_This_does_not_affect_emit_6654", "Set the language of the messaging from TypeScript. This does not affect emit."),
  Specify_the_location_where_debugger_should_locate_map_files_instead_of_generated_locations: diag(6655, 3 /* Message */, "Specify_the_location_where_debugger_should_locate_map_files_instead_of_generated_locations_6655", "Specify the location where debugger should locate map files instead of generated locations."),
  Specify_the_maximum_folder_depth_used_for_checking_JavaScript_files_from_node_modules_Only_applicable_with_allowJs: diag(6656, 3 /* Message */, "Specify_the_maximum_folder_depth_used_for_checking_JavaScript_files_from_node_modules_Only_applicabl_6656", "Specify the maximum folder depth used for checking JavaScript files from 'node_modules'. Only applicable with 'allowJs'."),
  Specify_what_module_code_is_generated: diag(6657, 3 /* Message */, "Specify_what_module_code_is_generated_6657", "Specify what module code is generated."),
  Specify_how_TypeScript_looks_up_a_file_from_a_given_module_specifier: diag(6658, 3 /* Message */, "Specify_how_TypeScript_looks_up_a_file_from_a_given_module_specifier_6658", "Specify how TypeScript looks up a file from a given module specifier."),
  Set_the_newline_character_for_emitting_files: diag(6659, 3 /* Message */, "Set_the_newline_character_for_emitting_files_6659", "Set the newline character for emitting files."),
  Disable_emitting_files_from_a_compilation: diag(6660, 3 /* Message */, "Disable_emitting_files_from_a_compilation_6660", "Disable emitting files from a compilation."),
  Disable_generating_custom_helper_functions_like_extends_in_compiled_output: diag(6661, 3 /* Message */, "Disable_generating_custom_helper_functions_like_extends_in_compiled_output_6661", "Disable generating custom helper functions like '__extends' in compiled output."),
  Disable_emitting_files_if_any_type_checking_errors_are_reported: diag(6662, 3 /* Message */, "Disable_emitting_files_if_any_type_checking_errors_are_reported_6662", "Disable emitting files if any type checking errors are reported."),
  Disable_truncating_types_in_error_messages: diag(6663, 3 /* Message */, "Disable_truncating_types_in_error_messages_6663", "Disable truncating types in error messages."),
  Enable_error_reporting_for_fallthrough_cases_in_switch_statements: diag(6664, 3 /* Message */, "Enable_error_reporting_for_fallthrough_cases_in_switch_statements_6664", "Enable error reporting for fallthrough cases in switch statements."),
  Enable_error_reporting_for_expressions_and_declarations_with_an_implied_any_type: diag(6665, 3 /* Message */, "Enable_error_reporting_for_expressions_and_declarations_with_an_implied_any_type_6665", "Enable error reporting for expressions and declarations with an implied 'any' type."),
  Ensure_overriding_members_in_derived_classes_are_marked_with_an_override_modifier: diag(6666, 3 /* Message */, "Ensure_overriding_members_in_derived_classes_are_marked_with_an_override_modifier_6666", "Ensure overriding members in derived classes are marked with an override modifier."),
  Enable_error_reporting_for_codepaths_that_do_not_explicitly_return_in_a_function: diag(6667, 3 /* Message */, "Enable_error_reporting_for_codepaths_that_do_not_explicitly_return_in_a_function_6667", "Enable error reporting for codepaths that do not explicitly return in a function."),
  Enable_error_reporting_when_this_is_given_the_type_any: diag(6668, 3 /* Message */, "Enable_error_reporting_when_this_is_given_the_type_any_6668", "Enable error reporting when 'this' is given the type 'any'."),
  Disable_adding_use_strict_directives_in_emitted_JavaScript_files: diag(6669, 3 /* Message */, "Disable_adding_use_strict_directives_in_emitted_JavaScript_files_6669", "Disable adding 'use strict' directives in emitted JavaScript files."),
  Disable_including_any_library_files_including_the_default_lib_d_ts: diag(6670, 3 /* Message */, "Disable_including_any_library_files_including_the_default_lib_d_ts_6670", "Disable including any library files, including the default lib.d.ts."),
  Enforces_using_indexed_accessors_for_keys_declared_using_an_indexed_type: diag(6671, 3 /* Message */, "Enforces_using_indexed_accessors_for_keys_declared_using_an_indexed_type_6671", "Enforces using indexed accessors for keys declared using an indexed type."),
  Disallow_import_s_require_s_or_reference_s_from_expanding_the_number_of_files_TypeScript_should_add_to_a_project: diag(6672, 3 /* Message */, "Disallow_import_s_require_s_or_reference_s_from_expanding_the_number_of_files_TypeScript_should_add__6672", "Disallow 'import's, 'require's or '<reference>'s from expanding the number of files TypeScript should add to a project."),
  Disable_strict_checking_of_generic_signatures_in_function_types: diag(6673, 3 /* Message */, "Disable_strict_checking_of_generic_signatures_in_function_types_6673", "Disable strict checking of generic signatures in function types."),
  Add_undefined_to_a_type_when_accessed_using_an_index: diag(6674, 3 /* Message */, "Add_undefined_to_a_type_when_accessed_using_an_index_6674", "Add 'undefined' to a type when accessed using an index."),
  Enable_error_reporting_when_local_variables_aren_t_read: diag(6675, 3 /* Message */, "Enable_error_reporting_when_local_variables_aren_t_read_6675", "Enable error reporting when local variables aren't read."),
  Raise_an_error_when_a_function_parameter_isn_t_read: diag(6676, 3 /* Message */, "Raise_an_error_when_a_function_parameter_isn_t_read_6676", "Raise an error when a function parameter isn't read."),
  Deprecated_setting_Use_outFile_instead: diag(6677, 3 /* Message */, "Deprecated_setting_Use_outFile_instead_6677", "Deprecated setting. Use 'outFile' instead."),
  Specify_an_output_folder_for_all_emitted_files: diag(6678, 3 /* Message */, "Specify_an_output_folder_for_all_emitted_files_6678", "Specify an output folder for all emitted files."),
  Specify_a_file_that_bundles_all_outputs_into_one_JavaScript_file_If_declaration_is_true_also_designates_a_file_that_bundles_all_d_ts_output: diag(6679, 3 /* Message */, "Specify_a_file_that_bundles_all_outputs_into_one_JavaScript_file_If_declaration_is_true_also_designa_6679", "Specify a file that bundles all outputs into one JavaScript file. If 'declaration' is true, also designates a file that bundles all .d.ts output."),
  Specify_a_set_of_entries_that_re_map_imports_to_additional_lookup_locations: diag(6680, 3 /* Message */, "Specify_a_set_of_entries_that_re_map_imports_to_additional_lookup_locations_6680", "Specify a set of entries that re-map imports to additional lookup locations."),
  Specify_a_list_of_language_service_plugins_to_include: diag(6681, 3 /* Message */, "Specify_a_list_of_language_service_plugins_to_include_6681", "Specify a list of language service plugins to include."),
  Disable_erasing_const_enum_declarations_in_generated_code: diag(6682, 3 /* Message */, "Disable_erasing_const_enum_declarations_in_generated_code_6682", "Disable erasing 'const enum' declarations in generated code."),
  Disable_resolving_symlinks_to_their_realpath_This_correlates_to_the_same_flag_in_node: diag(6683, 3 /* Message */, "Disable_resolving_symlinks_to_their_realpath_This_correlates_to_the_same_flag_in_node_6683", "Disable resolving symlinks to their realpath. This correlates to the same flag in node."),
  Disable_wiping_the_console_in_watch_mode: diag(6684, 3 /* Message */, "Disable_wiping_the_console_in_watch_mode_6684", "Disable wiping the console in watch mode."),
  Enable_color_and_formatting_in_TypeScript_s_output_to_make_compiler_errors_easier_to_read: diag(6685, 3 /* Message */, "Enable_color_and_formatting_in_TypeScript_s_output_to_make_compiler_errors_easier_to_read_6685", "Enable color and formatting in TypeScript's output to make compiler errors easier to read."),
  Specify_the_object_invoked_for_createElement_This_only_applies_when_targeting_react_JSX_emit: diag(6686, 3 /* Message */, "Specify_the_object_invoked_for_createElement_This_only_applies_when_targeting_react_JSX_emit_6686", "Specify the object invoked for 'createElement'. This only applies when targeting 'react' JSX emit."),
  Specify_an_array_of_objects_that_specify_paths_for_projects_Used_in_project_references: diag(6687, 3 /* Message */, "Specify_an_array_of_objects_that_specify_paths_for_projects_Used_in_project_references_6687", "Specify an array of objects that specify paths for projects. Used in project references."),
  Disable_emitting_comments: diag(6688, 3 /* Message */, "Disable_emitting_comments_6688", "Disable emitting comments."),
  Enable_importing_json_files: diag(6689, 3 /* Message */, "Enable_importing_json_files_6689", "Enable importing .json files."),
  Specify_the_root_folder_within_your_source_files: diag(6690, 3 /* Message */, "Specify_the_root_folder_within_your_source_files_6690", "Specify the root folder within your source files."),
  Allow_multiple_folders_to_be_treated_as_one_when_resolving_modules: diag(6691, 3 /* Message */, "Allow_multiple_folders_to_be_treated_as_one_when_resolving_modules_6691", "Allow multiple folders to be treated as one when resolving modules."),
  Skip_type_checking_d_ts_files_that_are_included_with_TypeScript: diag(6692, 3 /* Message */, "Skip_type_checking_d_ts_files_that_are_included_with_TypeScript_6692", "Skip type checking .d.ts files that are included with TypeScript."),
  Skip_type_checking_all_d_ts_files: diag(6693, 3 /* Message */, "Skip_type_checking_all_d_ts_files_6693", "Skip type checking all .d.ts files."),
  Create_source_map_files_for_emitted_JavaScript_files: diag(6694, 3 /* Message */, "Create_source_map_files_for_emitted_JavaScript_files_6694", "Create source map files for emitted JavaScript files."),
  Specify_the_root_path_for_debuggers_to_find_the_reference_source_code: diag(6695, 3 /* Message */, "Specify_the_root_path_for_debuggers_to_find_the_reference_source_code_6695", "Specify the root path for debuggers to find the reference source code."),
  Check_that_the_arguments_for_bind_call_and_apply_methods_match_the_original_function: diag(6697, 3 /* Message */, "Check_that_the_arguments_for_bind_call_and_apply_methods_match_the_original_function_6697", "Check that the arguments for 'bind', 'call', and 'apply' methods match the original function."),
  When_assigning_functions_check_to_ensure_parameters_and_the_return_values_are_subtype_compatible: diag(6698, 3 /* Message */, "When_assigning_functions_check_to_ensure_parameters_and_the_return_values_are_subtype_compatible_6698", "When assigning functions, check to ensure parameters and the return values are subtype-compatible."),
  When_type_checking_take_into_account_null_and_undefined: diag(6699, 3 /* Message */, "When_type_checking_take_into_account_null_and_undefined_6699", "When type checking, take into account 'null' and 'undefined'."),
  Check_for_class_properties_that_are_declared_but_not_set_in_the_constructor: diag(6700, 3 /* Message */, "Check_for_class_properties_that_are_declared_but_not_set_in_the_constructor_6700", "Check for class properties that are declared but not set in the constructor."),
  Disable_emitting_declarations_that_have_internal_in_their_JSDoc_comments: diag(6701, 3 /* Message */, "Disable_emitting_declarations_that_have_internal_in_their_JSDoc_comments_6701", "Disable emitting declarations that have '@internal' in their JSDoc comments."),
  Disable_reporting_of_excess_property_errors_during_the_creation_of_object_literals: diag(6702, 3 /* Message */, "Disable_reporting_of_excess_property_errors_during_the_creation_of_object_literals_6702", "Disable reporting of excess property errors during the creation of object literals."),
  Suppress_noImplicitAny_errors_when_indexing_objects_that_lack_index_signatures: diag(6703, 3 /* Message */, "Suppress_noImplicitAny_errors_when_indexing_objects_that_lack_index_signatures_6703", "Suppress 'noImplicitAny' errors when indexing objects that lack index signatures."),
  Synchronously_call_callbacks_and_update_the_state_of_directory_watchers_on_platforms_that_don_t_support_recursive_watching_natively: diag(6704, 3 /* Message */, "Synchronously_call_callbacks_and_update_the_state_of_directory_watchers_on_platforms_that_don_t_supp_6704", "Synchronously call callbacks and update the state of directory watchers on platforms that don`t support recursive watching natively."),
  Set_the_JavaScript_language_version_for_emitted_JavaScript_and_include_compatible_library_declarations: diag(6705, 3 /* Message */, "Set_the_JavaScript_language_version_for_emitted_JavaScript_and_include_compatible_library_declaratio_6705", "Set the JavaScript language version for emitted JavaScript and include compatible library declarations."),
  Log_paths_used_during_the_moduleResolution_process: diag(6706, 3 /* Message */, "Log_paths_used_during_the_moduleResolution_process_6706", "Log paths used during the 'moduleResolution' process."),
  Specify_the_path_to_tsbuildinfo_incremental_compilation_file: diag(6707, 3 /* Message */, "Specify_the_path_to_tsbuildinfo_incremental_compilation_file_6707", "Specify the path to .tsbuildinfo incremental compilation file."),
  Specify_options_for_automatic_acquisition_of_declaration_files: diag(6709, 3 /* Message */, "Specify_options_for_automatic_acquisition_of_declaration_files_6709", "Specify options for automatic acquisition of declaration files."),
  Specify_multiple_folders_that_act_like_Slashnode_modules_Slash_types: diag(6710, 3 /* Message */, "Specify_multiple_folders_that_act_like_Slashnode_modules_Slash_types_6710", "Specify multiple folders that act like './node_modules/@types'."),
  Specify_type_package_names_to_be_included_without_being_referenced_in_a_source_file: diag(6711, 3 /* Message */, "Specify_type_package_names_to_be_included_without_being_referenced_in_a_source_file_6711", "Specify type package names to be included without being referenced in a source file."),
  Emit_ECMAScript_standard_compliant_class_fields: diag(6712, 3 /* Message */, "Emit_ECMAScript_standard_compliant_class_fields_6712", "Emit ECMAScript-standard-compliant class fields."),
  Enable_verbose_logging: diag(6713, 3 /* Message */, "Enable_verbose_logging_6713", "Enable verbose logging."),
  Specify_how_directories_are_watched_on_systems_that_lack_recursive_file_watching_functionality: diag(6714, 3 /* Message */, "Specify_how_directories_are_watched_on_systems_that_lack_recursive_file_watching_functionality_6714", "Specify how directories are watched on systems that lack recursive file-watching functionality."),
  Specify_how_the_TypeScript_watch_mode_works: diag(6715, 3 /* Message */, "Specify_how_the_TypeScript_watch_mode_works_6715", "Specify how the TypeScript watch mode works."),
  Require_undeclared_properties_from_index_signatures_to_use_element_accesses: diag(6717, 3 /* Message */, "Require_undeclared_properties_from_index_signatures_to_use_element_accesses_6717", "Require undeclared properties from index signatures to use element accesses."),
  Specify_emit_Slashchecking_behavior_for_imports_that_are_only_used_for_types: diag(6718, 3 /* Message */, "Specify_emit_Slashchecking_behavior_for_imports_that_are_only_used_for_types_6718", "Specify emit/checking behavior for imports that are only used for types."),
  Require_sufficient_annotation_on_exports_so_other_tools_can_trivially_generate_declaration_files: diag(6719, 3 /* Message */, "Require_sufficient_annotation_on_exports_so_other_tools_can_trivially_generate_declaration_files_6719", "Require sufficient annotation on exports so other tools can trivially generate declaration files."),
  Built_in_iterators_are_instantiated_with_a_TReturn_type_of_undefined_instead_of_any: diag(6720, 3 /* Message */, "Built_in_iterators_are_instantiated_with_a_TReturn_type_of_undefined_instead_of_any_6720", "Built-in iterators are instantiated with a 'TReturn' type of 'undefined' instead of 'any'."),
  Default_catch_clause_variables_as_unknown_instead_of_any: diag(6803, 3 /* Message */, "Default_catch_clause_variables_as_unknown_instead_of_any_6803", "Default catch clause variables as 'unknown' instead of 'any'."),
  Do_not_transform_or_elide_any_imports_or_exports_not_marked_as_type_only_ensuring_they_are_written_in_the_output_file_s_format_based_on_the_module_setting: diag(6804, 3 /* Message */, "Do_not_transform_or_elide_any_imports_or_exports_not_marked_as_type_only_ensuring_they_are_written_i_6804", "Do not transform or elide any imports or exports not marked as type-only, ensuring they are written in the output file's format based on the 'module' setting."),
  Disable_full_type_checking_only_critical_parse_and_emit_errors_will_be_reported: diag(6805, 3 /* Message */, "Disable_full_type_checking_only_critical_parse_and_emit_errors_will_be_reported_6805", "Disable full type checking (only critical parse and emit errors will be reported)."),
  Check_side_effect_imports: diag(6806, 3 /* Message */, "Check_side_effect_imports_6806", "Check side effect imports."),
  This_operation_can_be_simplified_This_shift_is_identical_to_0_1_2: diag(6807, 1 /* Error */, "This_operation_can_be_simplified_This_shift_is_identical_to_0_1_2_6807", "This operation can be simplified. This shift is identical to `{0} {1} {2}`."),
  one_of_Colon: diag(6900, 3 /* Message */, "one_of_Colon_6900", "one of:"),
  one_or_more_Colon: diag(6901, 3 /* Message */, "one_or_more_Colon_6901", "one or more:"),
  type_Colon: diag(6902, 3 /* Message */, "type_Colon_6902", "type:"),
  default_Colon: diag(6903, 3 /* Message */, "default_Colon_6903", "default:"),
  module_system_or_esModuleInterop: diag(6904, 3 /* Message */, "module_system_or_esModuleInterop_6904", 'module === "system" or esModuleInterop'),
  false_unless_strict_is_set: diag(6905, 3 /* Message */, "false_unless_strict_is_set_6905", "`false`, unless `strict` is set"),
  false_unless_composite_is_set: diag(6906, 3 /* Message */, "false_unless_composite_is_set_6906", "`false`, unless `composite` is set"),
  node_modules_bower_components_jspm_packages_plus_the_value_of_outDir_if_one_is_specified: diag(6907, 3 /* Message */, "node_modules_bower_components_jspm_packages_plus_the_value_of_outDir_if_one_is_specified_6907", '`["node_modules", "bower_components", "jspm_packages"]`, plus the value of `outDir` if one is specified.'),
  if_files_is_specified_otherwise_Asterisk_Asterisk_Slash_Asterisk: diag(6908, 3 /* Message */, "if_files_is_specified_otherwise_Asterisk_Asterisk_Slash_Asterisk_6908", '`[]` if `files` is specified, otherwise `["**/*"]`'),
  true_if_composite_false_otherwise: diag(6909, 3 /* Message */, "true_if_composite_false_otherwise_6909", "`true` if `composite`, `false` otherwise"),
  module_AMD_or_UMD_or_System_or_ES6_then_Classic_Otherwise_Node: diag(69010, 3 /* Message */, "module_AMD_or_UMD_or_System_or_ES6_then_Classic_Otherwise_Node_69010", "module === `AMD` or `UMD` or `System` or `ES6`, then `Classic`, Otherwise `Node`"),
  Computed_from_the_list_of_input_files: diag(6911, 3 /* Message */, "Computed_from_the_list_of_input_files_6911", "Computed from the list of input files"),
  Platform_specific: diag(6912, 3 /* Message */, "Platform_specific_6912", "Platform specific"),
  You_can_learn_about_all_of_the_compiler_options_at_0: diag(6913, 3 /* Message */, "You_can_learn_about_all_of_the_compiler_options_at_0_6913", "You can learn about all of the compiler options at {0}"),
  Including_watch_w_will_start_watching_the_current_project_for_the_file_changes_Once_set_you_can_config_watch_mode_with_Colon: diag(6914, 3 /* Message */, "Including_watch_w_will_start_watching_the_current_project_for_the_file_changes_Once_set_you_can_conf_6914", "Including --watch, -w will start watching the current project for the file changes. Once set, you can config watch mode with:"),
  Using_build_b_will_make_tsc_behave_more_like_a_build_orchestrator_than_a_compiler_This_is_used_to_trigger_building_composite_projects_which_you_can_learn_more_about_at_0: diag(6915, 3 /* Message */, "Using_build_b_will_make_tsc_behave_more_like_a_build_orchestrator_than_a_compiler_This_is_used_to_tr_6915", "Using --build, -b will make tsc behave more like a build orchestrator than a compiler. This is used to trigger building composite projects which you can learn more about at {0}"),
  COMMON_COMMANDS: diag(6916, 3 /* Message */, "COMMON_COMMANDS_6916", "COMMON COMMANDS"),
  ALL_COMPILER_OPTIONS: diag(6917, 3 /* Message */, "ALL_COMPILER_OPTIONS_6917", "ALL COMPILER OPTIONS"),
  WATCH_OPTIONS: diag(6918, 3 /* Message */, "WATCH_OPTIONS_6918", "WATCH OPTIONS"),
  BUILD_OPTIONS: diag(6919, 3 /* Message */, "BUILD_OPTIONS_6919", "BUILD OPTIONS"),
  COMMON_COMPILER_OPTIONS: diag(6920, 3 /* Message */, "COMMON_COMPILER_OPTIONS_6920", "COMMON COMPILER OPTIONS"),
  COMMAND_LINE_FLAGS: diag(6921, 3 /* Message */, "COMMAND_LINE_FLAGS_6921", "COMMAND LINE FLAGS"),
  tsc_Colon_The_TypeScript_Compiler: diag(6922, 3 /* Message */, "tsc_Colon_The_TypeScript_Compiler_6922", "tsc: The TypeScript Compiler"),
  Compiles_the_current_project_tsconfig_json_in_the_working_directory: diag(6923, 3 /* Message */, "Compiles_the_current_project_tsconfig_json_in_the_working_directory_6923", "Compiles the current project (tsconfig.json in the working directory.)"),
  Ignoring_tsconfig_json_compiles_the_specified_files_with_default_compiler_options: diag(6924, 3 /* Message */, "Ignoring_tsconfig_json_compiles_the_specified_files_with_default_compiler_options_6924", "Ignoring tsconfig.json, compiles the specified files with default compiler options."),
  Build_a_composite_project_in_the_working_directory: diag(6925, 3 /* Message */, "Build_a_composite_project_in_the_working_directory_6925", "Build a composite project in the working directory."),
  Creates_a_tsconfig_json_with_the_recommended_settings_in_the_working_directory: diag(6926, 3 /* Message */, "Creates_a_tsconfig_json_with_the_recommended_settings_in_the_working_directory_6926", "Creates a tsconfig.json with the recommended settings in the working directory."),
  Compiles_the_TypeScript_project_located_at_the_specified_path: diag(6927, 3 /* Message */, "Compiles_the_TypeScript_project_located_at_the_specified_path_6927", "Compiles the TypeScript project located at the specified path."),
  An_expanded_version_of_this_information_showing_all_possible_compiler_options: diag(6928, 3 /* Message */, "An_expanded_version_of_this_information_showing_all_possible_compiler_options_6928", "An expanded version of this information, showing all possible compiler options"),
  Compiles_the_current_project_with_additional_settings: diag(6929, 3 /* Message */, "Compiles_the_current_project_with_additional_settings_6929", "Compiles the current project, with additional settings."),
  true_for_ES2022_and_above_including_ESNext: diag(6930, 3 /* Message */, "true_for_ES2022_and_above_including_ESNext_6930", "`true` for ES2022 and above, including ESNext."),
  List_of_file_name_suffixes_to_search_when_resolving_a_module: diag(6931, 1 /* Error */, "List_of_file_name_suffixes_to_search_when_resolving_a_module_6931", "List of file name suffixes to search when resolving a module."),
  Variable_0_implicitly_has_an_1_type: diag(7005, 1 /* Error */, "Variable_0_implicitly_has_an_1_type_7005", "Variable '{0}' implicitly has an '{1}' type."),
  Parameter_0_implicitly_has_an_1_type: diag(7006, 1 /* Error */, "Parameter_0_implicitly_has_an_1_type_7006", "Parameter '{0}' implicitly has an '{1}' type."),
  Member_0_implicitly_has_an_1_type: diag(7008, 1 /* Error */, "Member_0_implicitly_has_an_1_type_7008", "Member '{0}' implicitly has an '{1}' type."),
  new_expression_whose_target_lacks_a_construct_signature_implicitly_has_an_any_type: diag(7009, 1 /* Error */, "new_expression_whose_target_lacks_a_construct_signature_implicitly_has_an_any_type_7009", "'new' expression, whose target lacks a construct signature, implicitly has an 'any' type."),
  _0_which_lacks_return_type_annotation_implicitly_has_an_1_return_type: diag(7010, 1 /* Error */, "_0_which_lacks_return_type_annotation_implicitly_has_an_1_return_type_7010", "'{0}', which lacks return-type annotation, implicitly has an '{1}' return type."),
  Function_expression_which_lacks_return_type_annotation_implicitly_has_an_0_return_type: diag(7011, 1 /* Error */, "Function_expression_which_lacks_return_type_annotation_implicitly_has_an_0_return_type_7011", "Function expression, which lacks return-type annotation, implicitly has an '{0}' return type."),
  This_overload_implicitly_returns_the_type_0_because_it_lacks_a_return_type_annotation: diag(7012, 1 /* Error */, "This_overload_implicitly_returns_the_type_0_because_it_lacks_a_return_type_annotation_7012", "This overload implicitly returns the type '{0}' because it lacks a return type annotation."),
  Construct_signature_which_lacks_return_type_annotation_implicitly_has_an_any_return_type: diag(7013, 1 /* Error */, "Construct_signature_which_lacks_return_type_annotation_implicitly_has_an_any_return_type_7013", "Construct signature, which lacks return-type annotation, implicitly has an 'any' return type."),
  Function_type_which_lacks_return_type_annotation_implicitly_has_an_0_return_type: diag(7014, 1 /* Error */, "Function_type_which_lacks_return_type_annotation_implicitly_has_an_0_return_type_7014", "Function type, which lacks return-type annotation, implicitly has an '{0}' return type."),
  Element_implicitly_has_an_any_type_because_index_expression_is_not_of_type_number: diag(7015, 1 /* Error */, "Element_implicitly_has_an_any_type_because_index_expression_is_not_of_type_number_7015", "Element implicitly has an 'any' type because index expression is not of type 'number'."),
  Could_not_find_a_declaration_file_for_module_0_1_implicitly_has_an_any_type: diag(7016, 1 /* Error */, "Could_not_find_a_declaration_file_for_module_0_1_implicitly_has_an_any_type_7016", "Could not find a declaration file for module '{0}'. '{1}' implicitly has an 'any' type."),
  Element_implicitly_has_an_any_type_because_type_0_has_no_index_signature: diag(7017, 1 /* Error */, "Element_implicitly_has_an_any_type_because_type_0_has_no_index_signature_7017", "Element implicitly has an 'any' type because type '{0}' has no index signature."),
  Object_literal_s_property_0_implicitly_has_an_1_type: diag(7018, 1 /* Error */, "Object_literal_s_property_0_implicitly_has_an_1_type_7018", "Object literal's property '{0}' implicitly has an '{1}' type."),
  Rest_parameter_0_implicitly_has_an_any_type: diag(7019, 1 /* Error */, "Rest_parameter_0_implicitly_has_an_any_type_7019", "Rest parameter '{0}' implicitly has an 'any[]' type."),
  Call_signature_which_lacks_return_type_annotation_implicitly_has_an_any_return_type: diag(7020, 1 /* Error */, "Call_signature_which_lacks_return_type_annotation_implicitly_has_an_any_return_type_7020", "Call signature, which lacks return-type annotation, implicitly has an 'any' return type."),
  _0_implicitly_has_type_any_because_it_does_not_have_a_type_annotation_and_is_referenced_directly_or_indirectly_in_its_own_initializer: diag(7022, 1 /* Error */, "_0_implicitly_has_type_any_because_it_does_not_have_a_type_annotation_and_is_referenced_directly_or__7022", "'{0}' implicitly has type 'any' because it does not have a type annotation and is referenced directly or indirectly in its own initializer."),
  _0_implicitly_has_return_type_any_because_it_does_not_have_a_return_type_annotation_and_is_referenced_directly_or_indirectly_in_one_of_its_return_expressions: diag(7023, 1 /* Error */, "_0_implicitly_has_return_type_any_because_it_does_not_have_a_return_type_annotation_and_is_reference_7023", "'{0}' implicitly has return type 'any' because it does not have a return type annotation and is referenced directly or indirectly in one of its return expressions."),
  Function_implicitly_has_return_type_any_because_it_does_not_have_a_return_type_annotation_and_is_referenced_directly_or_indirectly_in_one_of_its_return_expressions: diag(7024, 1 /* Error */, "Function_implicitly_has_return_type_any_because_it_does_not_have_a_return_type_annotation_and_is_ref_7024", "Function implicitly has return type 'any' because it does not have a return type annotation and is referenced directly or indirectly in one of its return expressions."),
  Generator_implicitly_has_yield_type_0_Consider_supplying_a_return_type_annotation: diag(7025, 1 /* Error */, "Generator_implicitly_has_yield_type_0_Consider_supplying_a_return_type_annotation_7025", "Generator implicitly has yield type '{0}'. Consider supplying a return type annotation."),
  JSX_element_implicitly_has_type_any_because_no_interface_JSX_0_exists: diag(7026, 1 /* Error */, "JSX_element_implicitly_has_type_any_because_no_interface_JSX_0_exists_7026", "JSX element implicitly has type 'any' because no interface 'JSX.{0}' exists."),
  Unreachable_code_detected: diag(
    7027,
    1 /* Error */,
    "Unreachable_code_detected_7027",
    "Unreachable code detected.",
    /*reportsUnnecessary*/
    true
  ),
  Unused_label: diag(
    7028,
    1 /* Error */,
    "Unused_label_7028",
    "Unused label.",
    /*reportsUnnecessary*/
    true
  ),
  Fallthrough_case_in_switch: diag(7029, 1 /* Error */, "Fallthrough_case_in_switch_7029", "Fallthrough case in switch."),
  Not_all_code_paths_return_a_value: diag(7030, 1 /* Error */, "Not_all_code_paths_return_a_value_7030", "Not all code paths return a value."),
  Binding_element_0_implicitly_has_an_1_type: diag(7031, 1 /* Error */, "Binding_element_0_implicitly_has_an_1_type_7031", "Binding element '{0}' implicitly has an '{1}' type."),
  Property_0_implicitly_has_type_any_because_its_set_accessor_lacks_a_parameter_type_annotation: diag(7032, 1 /* Error */, "Property_0_implicitly_has_type_any_because_its_set_accessor_lacks_a_parameter_type_annotation_7032", "Property '{0}' implicitly has type 'any', because its set accessor lacks a parameter type annotation."),
  Property_0_implicitly_has_type_any_because_its_get_accessor_lacks_a_return_type_annotation: diag(7033, 1 /* Error */, "Property_0_implicitly_has_type_any_because_its_get_accessor_lacks_a_return_type_annotation_7033", "Property '{0}' implicitly has type 'any', because its get accessor lacks a return type annotation."),
  Variable_0_implicitly_has_type_1_in_some_locations_where_its_type_cannot_be_determined: diag(7034, 1 /* Error */, "Variable_0_implicitly_has_type_1_in_some_locations_where_its_type_cannot_be_determined_7034", "Variable '{0}' implicitly has type '{1}' in some locations where its type cannot be determined."),
  Try_npm_i_save_dev_types_Slash_1_if_it_exists_or_add_a_new_declaration_d_ts_file_containing_declare_module_0: diag(7035, 1 /* Error */, "Try_npm_i_save_dev_types_Slash_1_if_it_exists_or_add_a_new_declaration_d_ts_file_containing_declare__7035", "Try `npm i --save-dev @types/{1}` if it exists or add a new declaration (.d.ts) file containing `declare module '{0}';`"),
  Dynamic_import_s_specifier_must_be_of_type_string_but_here_has_type_0: diag(7036, 1 /* Error */, "Dynamic_import_s_specifier_must_be_of_type_string_but_here_has_type_0_7036", "Dynamic import's specifier must be of type 'string', but here has type '{0}'."),
  Enables_emit_interoperability_between_CommonJS_and_ES_Modules_via_creation_of_namespace_objects_for_all_imports_Implies_allowSyntheticDefaultImports: diag(7037, 3 /* Message */, "Enables_emit_interoperability_between_CommonJS_and_ES_Modules_via_creation_of_namespace_objects_for__7037", "Enables emit interoperability between CommonJS and ES Modules via creation of namespace objects for all imports. Implies 'allowSyntheticDefaultImports'."),
  Type_originates_at_this_import_A_namespace_style_import_cannot_be_called_or_constructed_and_will_cause_a_failure_at_runtime_Consider_using_a_default_import_or_import_require_here_instead: diag(7038, 3 /* Message */, "Type_originates_at_this_import_A_namespace_style_import_cannot_be_called_or_constructed_and_will_cau_7038", "Type originates at this import. A namespace-style import cannot be called or constructed, and will cause a failure at runtime. Consider using a default import or import require here instead."),
  Mapped_object_type_implicitly_has_an_any_template_type: diag(7039, 1 /* Error */, "Mapped_object_type_implicitly_has_an_any_template_type_7039", "Mapped object type implicitly has an 'any' template type."),
  If_the_0_package_actually_exposes_this_module_consider_sending_a_pull_request_to_amend_https_Colon_Slash_Slashgithub_com_SlashDefinitelyTyped_SlashDefinitelyTyped_Slashtree_Slashmaster_Slashtypes_Slash_1: diag(7040, 1 /* Error */, "If_the_0_package_actually_exposes_this_module_consider_sending_a_pull_request_to_amend_https_Colon_S_7040", "If the '{0}' package actually exposes this module, consider sending a pull request to amend 'https://github.com/DefinitelyTyped/DefinitelyTyped/tree/master/types/{1}'"),
  The_containing_arrow_function_captures_the_global_value_of_this: diag(7041, 1 /* Error */, "The_containing_arrow_function_captures_the_global_value_of_this_7041", "The containing arrow function captures the global value of 'this'."),
  Module_0_was_resolved_to_1_but_resolveJsonModule_is_not_used: diag(7042, 1 /* Error */, "Module_0_was_resolved_to_1_but_resolveJsonModule_is_not_used_7042", "Module '{0}' was resolved to '{1}', but '--resolveJsonModule' is not used."),
  Variable_0_implicitly_has_an_1_type_but_a_better_type_may_be_inferred_from_usage: diag(7043, 2 /* Suggestion */, "Variable_0_implicitly_has_an_1_type_but_a_better_type_may_be_inferred_from_usage_7043", "Variable '{0}' implicitly has an '{1}' type, but a better type may be inferred from usage."),
  Parameter_0_implicitly_has_an_1_type_but_a_better_type_may_be_inferred_from_usage: diag(7044, 2 /* Suggestion */, "Parameter_0_implicitly_has_an_1_type_but_a_better_type_may_be_inferred_from_usage_7044", "Parameter '{0}' implicitly has an '{1}' type, but a better type may be inferred from usage."),
  Member_0_implicitly_has_an_1_type_but_a_better_type_may_be_inferred_from_usage: diag(7045, 2 /* Suggestion */, "Member_0_implicitly_has_an_1_type_but_a_better_type_may_be_inferred_from_usage_7045", "Member '{0}' implicitly has an '{1}' type, but a better type may be inferred from usage."),
  Variable_0_implicitly_has_type_1_in_some_locations_but_a_better_type_may_be_inferred_from_usage: diag(7046, 2 /* Suggestion */, "Variable_0_implicitly_has_type_1_in_some_locations_but_a_better_type_may_be_inferred_from_usage_7046", "Variable '{0}' implicitly has type '{1}' in some locations, but a better type may be inferred from usage."),
  Rest_parameter_0_implicitly_has_an_any_type_but_a_better_type_may_be_inferred_from_usage: diag(7047, 2 /* Suggestion */, "Rest_parameter_0_implicitly_has_an_any_type_but_a_better_type_may_be_inferred_from_usage_7047", "Rest parameter '{0}' implicitly has an 'any[]' type, but a better type may be inferred from usage."),
  Property_0_implicitly_has_type_any_but_a_better_type_for_its_get_accessor_may_be_inferred_from_usage: diag(7048, 2 /* Suggestion */, "Property_0_implicitly_has_type_any_but_a_better_type_for_its_get_accessor_may_be_inferred_from_usage_7048", "Property '{0}' implicitly has type 'any', but a better type for its get accessor may be inferred from usage."),
  Property_0_implicitly_has_type_any_but_a_better_type_for_its_set_accessor_may_be_inferred_from_usage: diag(7049, 2 /* Suggestion */, "Property_0_implicitly_has_type_any_but_a_better_type_for_its_set_accessor_may_be_inferred_from_usage_7049", "Property '{0}' implicitly has type 'any', but a better type for its set accessor may be inferred from usage."),
  _0_implicitly_has_an_1_return_type_but_a_better_type_may_be_inferred_from_usage: diag(7050, 2 /* Suggestion */, "_0_implicitly_has_an_1_return_type_but_a_better_type_may_be_inferred_from_usage_7050", "'{0}' implicitly has an '{1}' return type, but a better type may be inferred from usage."),
  Parameter_has_a_name_but_no_type_Did_you_mean_0_Colon_1: diag(7051, 1 /* Error */, "Parameter_has_a_name_but_no_type_Did_you_mean_0_Colon_1_7051", "Parameter has a name but no type. Did you mean '{0}: {1}'?"),
  Element_implicitly_has_an_any_type_because_type_0_has_no_index_signature_Did_you_mean_to_call_1: diag(7052, 1 /* Error */, "Element_implicitly_has_an_any_type_because_type_0_has_no_index_signature_Did_you_mean_to_call_1_7052", "Element implicitly has an 'any' type because type '{0}' has no index signature. Did you mean to call '{1}'?"),
  Element_implicitly_has_an_any_type_because_expression_of_type_0_can_t_be_used_to_index_type_1: diag(7053, 1 /* Error */, "Element_implicitly_has_an_any_type_because_expression_of_type_0_can_t_be_used_to_index_type_1_7053", "Element implicitly has an 'any' type because expression of type '{0}' can't be used to index type '{1}'."),
  No_index_signature_with_a_parameter_of_type_0_was_found_on_type_1: diag(7054, 1 /* Error */, "No_index_signature_with_a_parameter_of_type_0_was_found_on_type_1_7054", "No index signature with a parameter of type '{0}' was found on type '{1}'."),
  _0_which_lacks_return_type_annotation_implicitly_has_an_1_yield_type: diag(7055, 1 /* Error */, "_0_which_lacks_return_type_annotation_implicitly_has_an_1_yield_type_7055", "'{0}', which lacks return-type annotation, implicitly has an '{1}' yield type."),
  The_inferred_type_of_this_node_exceeds_the_maximum_length_the_compiler_will_serialize_An_explicit_type_annotation_is_needed: diag(7056, 1 /* Error */, "The_inferred_type_of_this_node_exceeds_the_maximum_length_the_compiler_will_serialize_An_explicit_ty_7056", "The inferred type of this node exceeds the maximum length the compiler will serialize. An explicit type annotation is needed."),
  yield_expression_implicitly_results_in_an_any_type_because_its_containing_generator_lacks_a_return_type_annotation: diag(7057, 1 /* Error */, "yield_expression_implicitly_results_in_an_any_type_because_its_containing_generator_lacks_a_return_t_7057", "'yield' expression implicitly results in an 'any' type because its containing generator lacks a return-type annotation."),
  If_the_0_package_actually_exposes_this_module_try_adding_a_new_declaration_d_ts_file_containing_declare_module_1: diag(7058, 1 /* Error */, "If_the_0_package_actually_exposes_this_module_try_adding_a_new_declaration_d_ts_file_containing_decl_7058", "If the '{0}' package actually exposes this module, try adding a new declaration (.d.ts) file containing `declare module '{1}';`"),
  This_syntax_is_reserved_in_files_with_the_mts_or_cts_extension_Use_an_as_expression_instead: diag(7059, 1 /* Error */, "This_syntax_is_reserved_in_files_with_the_mts_or_cts_extension_Use_an_as_expression_instead_7059", "This syntax is reserved in files with the .mts or .cts extension. Use an `as` expression instead."),
  This_syntax_is_reserved_in_files_with_the_mts_or_cts_extension_Add_a_trailing_comma_or_explicit_constraint: diag(7060, 1 /* Error */, "This_syntax_is_reserved_in_files_with_the_mts_or_cts_extension_Add_a_trailing_comma_or_explicit_cons_7060", "This syntax is reserved in files with the .mts or .cts extension. Add a trailing comma or explicit constraint."),
  A_mapped_type_may_not_declare_properties_or_methods: diag(7061, 1 /* Error */, "A_mapped_type_may_not_declare_properties_or_methods_7061", "A mapped type may not declare properties or methods."),
  You_cannot_rename_this_element: diag(8e3, 1 /* Error */, "You_cannot_rename_this_element_8000", "You cannot rename this element."),
  You_cannot_rename_elements_that_are_defined_in_the_standard_TypeScript_library: diag(8001, 1 /* Error */, "You_cannot_rename_elements_that_are_defined_in_the_standard_TypeScript_library_8001", "You cannot rename elements that are defined in the standard TypeScript library."),
  import_can_only_be_used_in_TypeScript_files: diag(8002, 1 /* Error */, "import_can_only_be_used_in_TypeScript_files_8002", "'import ... =' can only be used in TypeScript files."),
  export_can_only_be_used_in_TypeScript_files: diag(8003, 1 /* Error */, "export_can_only_be_used_in_TypeScript_files_8003", "'export =' can only be used in TypeScript files."),
  Type_parameter_declarations_can_only_be_used_in_TypeScript_files: diag(8004, 1 /* Error */, "Type_parameter_declarations_can_only_be_used_in_TypeScript_files_8004", "Type parameter declarations can only be used in TypeScript files."),
  implements_clauses_can_only_be_used_in_TypeScript_files: diag(8005, 1 /* Error */, "implements_clauses_can_only_be_used_in_TypeScript_files_8005", "'implements' clauses can only be used in TypeScript files."),
  _0_declarations_can_only_be_used_in_TypeScript_files: diag(8006, 1 /* Error */, "_0_declarations_can_only_be_used_in_TypeScript_files_8006", "'{0}' declarations can only be used in TypeScript files."),
  Type_aliases_can_only_be_used_in_TypeScript_files: diag(8008, 1 /* Error */, "Type_aliases_can_only_be_used_in_TypeScript_files_8008", "Type aliases can only be used in TypeScript files."),
  The_0_modifier_can_only_be_used_in_TypeScript_files: diag(8009, 1 /* Error */, "The_0_modifier_can_only_be_used_in_TypeScript_files_8009", "The '{0}' modifier can only be used in TypeScript files."),
  Type_annotations_can_only_be_used_in_TypeScript_files: diag(8010, 1 /* Error */, "Type_annotations_can_only_be_used_in_TypeScript_files_8010", "Type annotations can only be used in TypeScript files."),
  Type_arguments_can_only_be_used_in_TypeScript_files: diag(8011, 1 /* Error */, "Type_arguments_can_only_be_used_in_TypeScript_files_8011", "Type arguments can only be used in TypeScript files."),
  Parameter_modifiers_can_only_be_used_in_TypeScript_files: diag(8012, 1 /* Error */, "Parameter_modifiers_can_only_be_used_in_TypeScript_files_8012", "Parameter modifiers can only be used in TypeScript files."),
  Non_null_assertions_can_only_be_used_in_TypeScript_files: diag(8013, 1 /* Error */, "Non_null_assertions_can_only_be_used_in_TypeScript_files_8013", "Non-null assertions can only be used in TypeScript files."),
  Type_assertion_expressions_can_only_be_used_in_TypeScript_files: diag(8016, 1 /* Error */, "Type_assertion_expressions_can_only_be_used_in_TypeScript_files_8016", "Type assertion expressions can only be used in TypeScript files."),
  Signature_declarations_can_only_be_used_in_TypeScript_files: diag(8017, 1 /* Error */, "Signature_declarations_can_only_be_used_in_TypeScript_files_8017", "Signature declarations can only be used in TypeScript files."),
  Report_errors_in_js_files: diag(8019, 3 /* Message */, "Report_errors_in_js_files_8019", "Report errors in .js files."),
  JSDoc_types_can_only_be_used_inside_documentation_comments: diag(8020, 1 /* Error */, "JSDoc_types_can_only_be_used_inside_documentation_comments_8020", "JSDoc types can only be used inside documentation comments."),
  JSDoc_typedef_tag_should_either_have_a_type_annotation_or_be_followed_by_property_or_member_tags: diag(8021, 1 /* Error */, "JSDoc_typedef_tag_should_either_have_a_type_annotation_or_be_followed_by_property_or_member_tags_8021", "JSDoc '@typedef' tag should either have a type annotation or be followed by '@property' or '@member' tags."),
  JSDoc_0_is_not_attached_to_a_class: diag(8022, 1 /* Error */, "JSDoc_0_is_not_attached_to_a_class_8022", "JSDoc '@{0}' is not attached to a class."),
  JSDoc_0_1_does_not_match_the_extends_2_clause: diag(8023, 1 /* Error */, "JSDoc_0_1_does_not_match_the_extends_2_clause_8023", "JSDoc '@{0} {1}' does not match the 'extends {2}' clause."),
  JSDoc_param_tag_has_name_0_but_there_is_no_parameter_with_that_name: diag(8024, 1 /* Error */, "JSDoc_param_tag_has_name_0_but_there_is_no_parameter_with_that_name_8024", "JSDoc '@param' tag has name '{0}', but there is no parameter with that name."),
  Class_declarations_cannot_have_more_than_one_augments_or_extends_tag: diag(8025, 1 /* Error */, "Class_declarations_cannot_have_more_than_one_augments_or_extends_tag_8025", "Class declarations cannot have more than one '@augments' or '@extends' tag."),
  Expected_0_type_arguments_provide_these_with_an_extends_tag: diag(8026, 1 /* Error */, "Expected_0_type_arguments_provide_these_with_an_extends_tag_8026", "Expected {0} type arguments; provide these with an '@extends' tag."),
  Expected_0_1_type_arguments_provide_these_with_an_extends_tag: diag(8027, 1 /* Error */, "Expected_0_1_type_arguments_provide_these_with_an_extends_tag_8027", "Expected {0}-{1} type arguments; provide these with an '@extends' tag."),
  JSDoc_may_only_appear_in_the_last_parameter_of_a_signature: diag(8028, 1 /* Error */, "JSDoc_may_only_appear_in_the_last_parameter_of_a_signature_8028", "JSDoc '...' may only appear in the last parameter of a signature."),
  JSDoc_param_tag_has_name_0_but_there_is_no_parameter_with_that_name_It_would_match_arguments_if_it_had_an_array_type: diag(8029, 1 /* Error */, "JSDoc_param_tag_has_name_0_but_there_is_no_parameter_with_that_name_It_would_match_arguments_if_it_h_8029", "JSDoc '@param' tag has name '{0}', but there is no parameter with that name. It would match 'arguments' if it had an array type."),
  The_type_of_a_function_declaration_must_match_the_function_s_signature: diag(8030, 1 /* Error */, "The_type_of_a_function_declaration_must_match_the_function_s_signature_8030", "The type of a function declaration must match the function's signature."),
  You_cannot_rename_a_module_via_a_global_import: diag(8031, 1 /* Error */, "You_cannot_rename_a_module_via_a_global_import_8031", "You cannot rename a module via a global import."),
  Qualified_name_0_is_not_allowed_without_a_leading_param_object_1: diag(8032, 1 /* Error */, "Qualified_name_0_is_not_allowed_without_a_leading_param_object_1_8032", "Qualified name '{0}' is not allowed without a leading '@param {object} {1}'."),
  A_JSDoc_typedef_comment_may_not_contain_multiple_type_tags: diag(8033, 1 /* Error */, "A_JSDoc_typedef_comment_may_not_contain_multiple_type_tags_8033", "A JSDoc '@typedef' comment may not contain multiple '@type' tags."),
  The_tag_was_first_specified_here: diag(8034, 1 /* Error */, "The_tag_was_first_specified_here_8034", "The tag was first specified here."),
  You_cannot_rename_elements_that_are_defined_in_a_node_modules_folder: diag(8035, 1 /* Error */, "You_cannot_rename_elements_that_are_defined_in_a_node_modules_folder_8035", "You cannot rename elements that are defined in a 'node_modules' folder."),
  You_cannot_rename_elements_that_are_defined_in_another_node_modules_folder: diag(8036, 1 /* Error */, "You_cannot_rename_elements_that_are_defined_in_another_node_modules_folder_8036", "You cannot rename elements that are defined in another 'node_modules' folder."),
  Type_satisfaction_expressions_can_only_be_used_in_TypeScript_files: diag(8037, 1 /* Error */, "Type_satisfaction_expressions_can_only_be_used_in_TypeScript_files_8037", "Type satisfaction expressions can only be used in TypeScript files."),
  Decorators_may_not_appear_after_export_or_export_default_if_they_also_appear_before_export: diag(8038, 1 /* Error */, "Decorators_may_not_appear_after_export_or_export_default_if_they_also_appear_before_export_8038", "Decorators may not appear after 'export' or 'export default' if they also appear before 'export'."),
  A_JSDoc_template_tag_may_not_follow_a_typedef_callback_or_overload_tag: diag(8039, 1 /* Error */, "A_JSDoc_template_tag_may_not_follow_a_typedef_callback_or_overload_tag_8039", "A JSDoc '@template' tag may not follow a '@typedef', '@callback', or '@overload' tag"),
  Declaration_emit_for_this_file_requires_using_private_name_0_An_explicit_type_annotation_may_unblock_declaration_emit: diag(9005, 1 /* Error */, "Declaration_emit_for_this_file_requires_using_private_name_0_An_explicit_type_annotation_may_unblock_9005", "Declaration emit for this file requires using private name '{0}'. An explicit type annotation may unblock declaration emit."),
  Declaration_emit_for_this_file_requires_using_private_name_0_from_module_1_An_explicit_type_annotation_may_unblock_declaration_emit: diag(9006, 1 /* Error */, "Declaration_emit_for_this_file_requires_using_private_name_0_from_module_1_An_explicit_type_annotati_9006", "Declaration emit for this file requires using private name '{0}' from module '{1}'. An explicit type annotation may unblock declaration emit."),
  Function_must_have_an_explicit_return_type_annotation_with_isolatedDeclarations: diag(9007, 1 /* Error */, "Function_must_have_an_explicit_return_type_annotation_with_isolatedDeclarations_9007", "Function must have an explicit return type annotation with --isolatedDeclarations."),
  Method_must_have_an_explicit_return_type_annotation_with_isolatedDeclarations: diag(9008, 1 /* Error */, "Method_must_have_an_explicit_return_type_annotation_with_isolatedDeclarations_9008", "Method must have an explicit return type annotation with --isolatedDeclarations."),
  At_least_one_accessor_must_have_an_explicit_type_annotation_with_isolatedDeclarations: diag(9009, 1 /* Error */, "At_least_one_accessor_must_have_an_explicit_type_annotation_with_isolatedDeclarations_9009", "At least one accessor must have an explicit type annotation with --isolatedDeclarations."),
  Variable_must_have_an_explicit_type_annotation_with_isolatedDeclarations: diag(9010, 1 /* Error */, "Variable_must_have_an_explicit_type_annotation_with_isolatedDeclarations_9010", "Variable must have an explicit type annotation with --isolatedDeclarations."),
  Parameter_must_have_an_explicit_type_annotation_with_isolatedDeclarations: diag(9011, 1 /* Error */, "Parameter_must_have_an_explicit_type_annotation_with_isolatedDeclarations_9011", "Parameter must have an explicit type annotation with --isolatedDeclarations."),
  Property_must_have_an_explicit_type_annotation_with_isolatedDeclarations: diag(9012, 1 /* Error */, "Property_must_have_an_explicit_type_annotation_with_isolatedDeclarations_9012", "Property must have an explicit type annotation with --isolatedDeclarations."),
  Expression_type_can_t_be_inferred_with_isolatedDeclarations: diag(9013, 1 /* Error */, "Expression_type_can_t_be_inferred_with_isolatedDeclarations_9013", "Expression type can't be inferred with --isolatedDeclarations."),
  Computed_properties_must_be_number_or_string_literals_variables_or_dotted_expressions_with_isolatedDeclarations: diag(9014, 1 /* Error */, "Computed_properties_must_be_number_or_string_literals_variables_or_dotted_expressions_with_isolatedD_9014", "Computed properties must be number or string literals, variables or dotted expressions with --isolatedDeclarations."),
  Objects_that_contain_spread_assignments_can_t_be_inferred_with_isolatedDeclarations: diag(9015, 1 /* Error */, "Objects_that_contain_spread_assignments_can_t_be_inferred_with_isolatedDeclarations_9015", "Objects that contain spread assignments can't be inferred with --isolatedDeclarations."),
  Objects_that_contain_shorthand_properties_can_t_be_inferred_with_isolatedDeclarations: diag(9016, 1 /* Error */, "Objects_that_contain_shorthand_properties_can_t_be_inferred_with_isolatedDeclarations_9016", "Objects that contain shorthand properties can't be inferred with --isolatedDeclarations."),
  Only_const_arrays_can_be_inferred_with_isolatedDeclarations: diag(9017, 1 /* Error */, "Only_const_arrays_can_be_inferred_with_isolatedDeclarations_9017", "Only const arrays can be inferred with --isolatedDeclarations."),
  Arrays_with_spread_elements_can_t_inferred_with_isolatedDeclarations: diag(9018, 1 /* Error */, "Arrays_with_spread_elements_can_t_inferred_with_isolatedDeclarations_9018", "Arrays with spread elements can't inferred with --isolatedDeclarations."),
  Binding_elements_can_t_be_exported_directly_with_isolatedDeclarations: diag(9019, 1 /* Error */, "Binding_elements_can_t_be_exported_directly_with_isolatedDeclarations_9019", "Binding elements can't be exported directly with --isolatedDeclarations."),
  Enum_member_initializers_must_be_computable_without_references_to_external_symbols_with_isolatedDeclarations: diag(9020, 1 /* Error */, "Enum_member_initializers_must_be_computable_without_references_to_external_symbols_with_isolatedDecl_9020", "Enum member initializers must be computable without references to external symbols with --isolatedDeclarations."),
  Extends_clause_can_t_contain_an_expression_with_isolatedDeclarations: diag(9021, 1 /* Error */, "Extends_clause_can_t_contain_an_expression_with_isolatedDeclarations_9021", "Extends clause can't contain an expression with --isolatedDeclarations."),
  Inference_from_class_expressions_is_not_supported_with_isolatedDeclarations: diag(9022, 1 /* Error */, "Inference_from_class_expressions_is_not_supported_with_isolatedDeclarations_9022", "Inference from class expressions is not supported with --isolatedDeclarations."),
  Assigning_properties_to_functions_without_declaring_them_is_not_supported_with_isolatedDeclarations_Add_an_explicit_declaration_for_the_properties_assigned_to_this_function: diag(9023, 1 /* Error */, "Assigning_properties_to_functions_without_declaring_them_is_not_supported_with_isolatedDeclarations__9023", "Assigning properties to functions without declaring them is not supported with --isolatedDeclarations. Add an explicit declaration for the properties assigned to this function."),
  Declaration_emit_for_this_parameter_requires_implicitly_adding_undefined_to_its_type_This_is_not_supported_with_isolatedDeclarations: diag(9025, 1 /* Error */, "Declaration_emit_for_this_parameter_requires_implicitly_adding_undefined_to_its_type_This_is_not_sup_9025", "Declaration emit for this parameter requires implicitly adding undefined to its type. This is not supported with --isolatedDeclarations."),
  Declaration_emit_for_this_file_requires_preserving_this_import_for_augmentations_This_is_not_supported_with_isolatedDeclarations: diag(9026, 1 /* Error */, "Declaration_emit_for_this_file_requires_preserving_this_import_for_augmentations_This_is_not_support_9026", "Declaration emit for this file requires preserving this import for augmentations. This is not supported with --isolatedDeclarations."),
  Add_a_type_annotation_to_the_variable_0: diag(9027, 1 /* Error */, "Add_a_type_annotation_to_the_variable_0_9027", "Add a type annotation to the variable {0}."),
  Add_a_type_annotation_to_the_parameter_0: diag(9028, 1 /* Error */, "Add_a_type_annotation_to_the_parameter_0_9028", "Add a type annotation to the parameter {0}."),
  Add_a_type_annotation_to_the_property_0: diag(9029, 1 /* Error */, "Add_a_type_annotation_to_the_property_0_9029", "Add a type annotation to the property {0}."),
  Add_a_return_type_to_the_function_expression: diag(9030, 1 /* Error */, "Add_a_return_type_to_the_function_expression_9030", "Add a return type to the function expression."),
  Add_a_return_type_to_the_function_declaration: diag(9031, 1 /* Error */, "Add_a_return_type_to_the_function_declaration_9031", "Add a return type to the function declaration."),
  Add_a_return_type_to_the_get_accessor_declaration: diag(9032, 1 /* Error */, "Add_a_return_type_to_the_get_accessor_declaration_9032", "Add a return type to the get accessor declaration."),
  Add_a_type_to_parameter_of_the_set_accessor_declaration: diag(9033, 1 /* Error */, "Add_a_type_to_parameter_of_the_set_accessor_declaration_9033", "Add a type to parameter of the set accessor declaration."),
  Add_a_return_type_to_the_method: diag(9034, 1 /* Error */, "Add_a_return_type_to_the_method_9034", "Add a return type to the method"),
  Add_satisfies_and_a_type_assertion_to_this_expression_satisfies_T_as_T_to_make_the_type_explicit: diag(9035, 1 /* Error */, "Add_satisfies_and_a_type_assertion_to_this_expression_satisfies_T_as_T_to_make_the_type_explicit_9035", "Add satisfies and a type assertion to this expression (satisfies T as T) to make the type explicit."),
  Move_the_expression_in_default_export_to_a_variable_and_add_a_type_annotation_to_it: diag(9036, 1 /* Error */, "Move_the_expression_in_default_export_to_a_variable_and_add_a_type_annotation_to_it_9036", "Move the expression in default export to a variable and add a type annotation to it."),
  Default_exports_can_t_be_inferred_with_isolatedDeclarations: diag(9037, 1 /* Error */, "Default_exports_can_t_be_inferred_with_isolatedDeclarations_9037", "Default exports can't be inferred with --isolatedDeclarations."),
  Computed_property_names_on_class_or_object_literals_cannot_be_inferred_with_isolatedDeclarations: diag(9038, 1 /* Error */, "Computed_property_names_on_class_or_object_literals_cannot_be_inferred_with_isolatedDeclarations_9038", "Computed property names on class or object literals cannot be inferred with --isolatedDeclarations."),
  Type_containing_private_name_0_can_t_be_used_with_isolatedDeclarations: diag(9039, 1 /* Error */, "Type_containing_private_name_0_can_t_be_used_with_isolatedDeclarations_9039", "Type containing private name '{0}' can't be used with --isolatedDeclarations."),
  JSX_attributes_must_only_be_assigned_a_non_empty_expression: diag(17e3, 1 /* Error */, "JSX_attributes_must_only_be_assigned_a_non_empty_expression_17000", "JSX attributes must only be assigned a non-empty 'expression'."),
  JSX_elements_cannot_have_multiple_attributes_with_the_same_name: diag(17001, 1 /* Error */, "JSX_elements_cannot_have_multiple_attributes_with_the_same_name_17001", "JSX elements cannot have multiple attributes with the same name."),
  Expected_corresponding_JSX_closing_tag_for_0: diag(17002, 1 /* Error */, "Expected_corresponding_JSX_closing_tag_for_0_17002", "Expected corresponding JSX closing tag for '{0}'."),
  Cannot_use_JSX_unless_the_jsx_flag_is_provided: diag(17004, 1 /* Error */, "Cannot_use_JSX_unless_the_jsx_flag_is_provided_17004", "Cannot use JSX unless the '--jsx' flag is provided."),
  A_constructor_cannot_contain_a_super_call_when_its_class_extends_null: diag(17005, 1 /* Error */, "A_constructor_cannot_contain_a_super_call_when_its_class_extends_null_17005", "A constructor cannot contain a 'super' call when its class extends 'null'."),
  An_unary_expression_with_the_0_operator_is_not_allowed_in_the_left_hand_side_of_an_exponentiation_expression_Consider_enclosing_the_expression_in_parentheses: diag(17006, 1 /* Error */, "An_unary_expression_with_the_0_operator_is_not_allowed_in_the_left_hand_side_of_an_exponentiation_ex_17006", "An unary expression with the '{0}' operator is not allowed in the left-hand side of an exponentiation expression. Consider enclosing the expression in parentheses."),
  A_type_assertion_expression_is_not_allowed_in_the_left_hand_side_of_an_exponentiation_expression_Consider_enclosing_the_expression_in_parentheses: diag(17007, 1 /* Error */, "A_type_assertion_expression_is_not_allowed_in_the_left_hand_side_of_an_exponentiation_expression_Con_17007", "A type assertion expression is not allowed in the left-hand side of an exponentiation expression. Consider enclosing the expression in parentheses."),
  JSX_element_0_has_no_corresponding_closing_tag: diag(17008, 1 /* Error */, "JSX_element_0_has_no_corresponding_closing_tag_17008", "JSX element '{0}' has no corresponding closing tag."),
  super_must_be_called_before_accessing_this_in_the_constructor_of_a_derived_class: diag(17009, 1 /* Error */, "super_must_be_called_before_accessing_this_in_the_constructor_of_a_derived_class_17009", "'super' must be called before accessing 'this' in the constructor of a derived class."),
  Unknown_type_acquisition_option_0: diag(17010, 1 /* Error */, "Unknown_type_acquisition_option_0_17010", "Unknown type acquisition option '{0}'."),
  super_must_be_called_before_accessing_a_property_of_super_in_the_constructor_of_a_derived_class: diag(17011, 1 /* Error */, "super_must_be_called_before_accessing_a_property_of_super_in_the_constructor_of_a_derived_class_17011", "'super' must be called before accessing a property of 'super' in the constructor of a derived class."),
  _0_is_not_a_valid_meta_property_for_keyword_1_Did_you_mean_2: diag(17012, 1 /* Error */, "_0_is_not_a_valid_meta_property_for_keyword_1_Did_you_mean_2_17012", "'{0}' is not a valid meta-property for keyword '{1}'. Did you mean '{2}'?"),
  Meta_property_0_is_only_allowed_in_the_body_of_a_function_declaration_function_expression_or_constructor: diag(17013, 1 /* Error */, "Meta_property_0_is_only_allowed_in_the_body_of_a_function_declaration_function_expression_or_constru_17013", "Meta-property '{0}' is only allowed in the body of a function declaration, function expression, or constructor."),
  JSX_fragment_has_no_corresponding_closing_tag: diag(17014, 1 /* Error */, "JSX_fragment_has_no_corresponding_closing_tag_17014", "JSX fragment has no corresponding closing tag."),
  Expected_corresponding_closing_tag_for_JSX_fragment: diag(17015, 1 /* Error */, "Expected_corresponding_closing_tag_for_JSX_fragment_17015", "Expected corresponding closing tag for JSX fragment."),
  The_jsxFragmentFactory_compiler_option_must_be_provided_to_use_JSX_fragments_with_the_jsxFactory_compiler_option: diag(17016, 1 /* Error */, "The_jsxFragmentFactory_compiler_option_must_be_provided_to_use_JSX_fragments_with_the_jsxFactory_com_17016", "The 'jsxFragmentFactory' compiler option must be provided to use JSX fragments with the 'jsxFactory' compiler option."),
  An_jsxFrag_pragma_is_required_when_using_an_jsx_pragma_with_JSX_fragments: diag(17017, 1 /* Error */, "An_jsxFrag_pragma_is_required_when_using_an_jsx_pragma_with_JSX_fragments_17017", "An @jsxFrag pragma is required when using an @jsx pragma with JSX fragments."),
  Unknown_type_acquisition_option_0_Did_you_mean_1: diag(17018, 1 /* Error */, "Unknown_type_acquisition_option_0_Did_you_mean_1_17018", "Unknown type acquisition option '{0}'. Did you mean '{1}'?"),
  _0_at_the_end_of_a_type_is_not_valid_TypeScript_syntax_Did_you_mean_to_write_1: diag(17019, 1 /* Error */, "_0_at_the_end_of_a_type_is_not_valid_TypeScript_syntax_Did_you_mean_to_write_1_17019", "'{0}' at the end of a type is not valid TypeScript syntax. Did you mean to write '{1}'?"),
  _0_at_the_start_of_a_type_is_not_valid_TypeScript_syntax_Did_you_mean_to_write_1: diag(17020, 1 /* Error */, "_0_at_the_start_of_a_type_is_not_valid_TypeScript_syntax_Did_you_mean_to_write_1_17020", "'{0}' at the start of a type is not valid TypeScript syntax. Did you mean to write '{1}'?"),
  Unicode_escape_sequence_cannot_appear_here: diag(17021, 1 /* Error */, "Unicode_escape_sequence_cannot_appear_here_17021", "Unicode escape sequence cannot appear here."),
  Circularity_detected_while_resolving_configuration_Colon_0: diag(18e3, 1 /* Error */, "Circularity_detected_while_resolving_configuration_Colon_0_18000", "Circularity detected while resolving configuration: {0}"),
  The_files_list_in_config_file_0_is_empty: diag(18002, 1 /* Error */, "The_files_list_in_config_file_0_is_empty_18002", "The 'files' list in config file '{0}' is empty."),
  No_inputs_were_found_in_config_file_0_Specified_include_paths_were_1_and_exclude_paths_were_2: diag(18003, 1 /* Error */, "No_inputs_were_found_in_config_file_0_Specified_include_paths_were_1_and_exclude_paths_were_2_18003", "No inputs were found in config file '{0}'. Specified 'include' paths were '{1}' and 'exclude' paths were '{2}'."),
  File_is_a_CommonJS_module_it_may_be_converted_to_an_ES_module: diag(80001, 2 /* Suggestion */, "File_is_a_CommonJS_module_it_may_be_converted_to_an_ES_module_80001", "File is a CommonJS module; it may be converted to an ES module."),
  This_constructor_function_may_be_converted_to_a_class_declaration: diag(80002, 2 /* Suggestion */, "This_constructor_function_may_be_converted_to_a_class_declaration_80002", "This constructor function may be converted to a class declaration."),
  Import_may_be_converted_to_a_default_import: diag(80003, 2 /* Suggestion */, "Import_may_be_converted_to_a_default_import_80003", "Import may be converted to a default import."),
  JSDoc_types_may_be_moved_to_TypeScript_types: diag(80004, 2 /* Suggestion */, "JSDoc_types_may_be_moved_to_TypeScript_types_80004", "JSDoc types may be moved to TypeScript types."),
  require_call_may_be_converted_to_an_import: diag(80005, 2 /* Suggestion */, "require_call_may_be_converted_to_an_import_80005", "'require' call may be converted to an import."),
  This_may_be_converted_to_an_async_function: diag(80006, 2 /* Suggestion */, "This_may_be_converted_to_an_async_function_80006", "This may be converted to an async function."),
  await_has_no_effect_on_the_type_of_this_expression: diag(80007, 2 /* Suggestion */, "await_has_no_effect_on_the_type_of_this_expression_80007", "'await' has no effect on the type of this expression."),
  Numeric_literals_with_absolute_values_equal_to_2_53_or_greater_are_too_large_to_be_represented_accurately_as_integers: diag(80008, 2 /* Suggestion */, "Numeric_literals_with_absolute_values_equal_to_2_53_or_greater_are_too_large_to_be_represented_accur_80008", "Numeric literals with absolute values equal to 2^53 or greater are too large to be represented accurately as integers."),
  JSDoc_typedef_may_be_converted_to_TypeScript_type: diag(80009, 2 /* Suggestion */, "JSDoc_typedef_may_be_converted_to_TypeScript_type_80009", "JSDoc typedef may be converted to TypeScript type."),
  JSDoc_typedefs_may_be_converted_to_TypeScript_types: diag(80010, 2 /* Suggestion */, "JSDoc_typedefs_may_be_converted_to_TypeScript_types_80010", "JSDoc typedefs may be converted to TypeScript types."),
  Add_missing_super_call: diag(90001, 3 /* Message */, "Add_missing_super_call_90001", "Add missing 'super()' call"),
  Make_super_call_the_first_statement_in_the_constructor: diag(90002, 3 /* Message */, "Make_super_call_the_first_statement_in_the_constructor_90002", "Make 'super()' call the first statement in the constructor"),
  Change_extends_to_implements: diag(90003, 3 /* Message */, "Change_extends_to_implements_90003", "Change 'extends' to 'implements'"),
  Remove_unused_declaration_for_Colon_0: diag(90004, 3 /* Message */, "Remove_unused_declaration_for_Colon_0_90004", "Remove unused declaration for: '{0}'"),
  Remove_import_from_0: diag(90005, 3 /* Message */, "Remove_import_from_0_90005", "Remove import from '{0}'"),
  Implement_interface_0: diag(90006, 3 /* Message */, "Implement_interface_0_90006", "Implement interface '{0}'"),
  Implement_inherited_abstract_class: diag(90007, 3 /* Message */, "Implement_inherited_abstract_class_90007", "Implement inherited abstract class"),
  Add_0_to_unresolved_variable: diag(90008, 3 /* Message */, "Add_0_to_unresolved_variable_90008", "Add '{0}.' to unresolved variable"),
  Remove_variable_statement: diag(90010, 3 /* Message */, "Remove_variable_statement_90010", "Remove variable statement"),
  Remove_template_tag: diag(90011, 3 /* Message */, "Remove_template_tag_90011", "Remove template tag"),
  Remove_type_parameters: diag(90012, 3 /* Message */, "Remove_type_parameters_90012", "Remove type parameters"),
  Import_0_from_1: diag(90013, 3 /* Message */, "Import_0_from_1_90013", `Import '{0}' from "{1}"`),
  Change_0_to_1: diag(90014, 3 /* Message */, "Change_0_to_1_90014", "Change '{0}' to '{1}'"),
  Declare_property_0: diag(90016, 3 /* Message */, "Declare_property_0_90016", "Declare property '{0}'"),
  Add_index_signature_for_property_0: diag(90017, 3 /* Message */, "Add_index_signature_for_property_0_90017", "Add index signature for property '{0}'"),
  Disable_checking_for_this_file: diag(90018, 3 /* Message */, "Disable_checking_for_this_file_90018", "Disable checking for this file"),
  Ignore_this_error_message: diag(90019, 3 /* Message */, "Ignore_this_error_message_90019", "Ignore this error message"),
  Initialize_property_0_in_the_constructor: diag(90020, 3 /* Message */, "Initialize_property_0_in_the_constructor_90020", "Initialize property '{0}' in the constructor"),
  Initialize_static_property_0: diag(90021, 3 /* Message */, "Initialize_static_property_0_90021", "Initialize static property '{0}'"),
  Change_spelling_to_0: diag(90022, 3 /* Message */, "Change_spelling_to_0_90022", "Change spelling to '{0}'"),
  Declare_method_0: diag(90023, 3 /* Message */, "Declare_method_0_90023", "Declare method '{0}'"),
  Declare_static_method_0: diag(90024, 3 /* Message */, "Declare_static_method_0_90024", "Declare static method '{0}'"),
  Prefix_0_with_an_underscore: diag(90025, 3 /* Message */, "Prefix_0_with_an_underscore_90025", "Prefix '{0}' with an underscore"),
  Rewrite_as_the_indexed_access_type_0: diag(90026, 3 /* Message */, "Rewrite_as_the_indexed_access_type_0_90026", "Rewrite as the indexed access type '{0}'"),
  Declare_static_property_0: diag(90027, 3 /* Message */, "Declare_static_property_0_90027", "Declare static property '{0}'"),
  Call_decorator_expression: diag(90028, 3 /* Message */, "Call_decorator_expression_90028", "Call decorator expression"),
  Add_async_modifier_to_containing_function: diag(90029, 3 /* Message */, "Add_async_modifier_to_containing_function_90029", "Add async modifier to containing function"),
  Replace_infer_0_with_unknown: diag(90030, 3 /* Message */, "Replace_infer_0_with_unknown_90030", "Replace 'infer {0}' with 'unknown'"),
  Replace_all_unused_infer_with_unknown: diag(90031, 3 /* Message */, "Replace_all_unused_infer_with_unknown_90031", "Replace all unused 'infer' with 'unknown'"),
  Add_parameter_name: diag(90034, 3 /* Message */, "Add_parameter_name_90034", "Add parameter name"),
  Declare_private_property_0: diag(90035, 3 /* Message */, "Declare_private_property_0_90035", "Declare private property '{0}'"),
  Replace_0_with_Promise_1: diag(90036, 3 /* Message */, "Replace_0_with_Promise_1_90036", "Replace '{0}' with 'Promise<{1}>'"),
  Fix_all_incorrect_return_type_of_an_async_functions: diag(90037, 3 /* Message */, "Fix_all_incorrect_return_type_of_an_async_functions_90037", "Fix all incorrect return type of an async functions"),
  Declare_private_method_0: diag(90038, 3 /* Message */, "Declare_private_method_0_90038", "Declare private method '{0}'"),
  Remove_unused_destructuring_declaration: diag(90039, 3 /* Message */, "Remove_unused_destructuring_declaration_90039", "Remove unused destructuring declaration"),
  Remove_unused_declarations_for_Colon_0: diag(90041, 3 /* Message */, "Remove_unused_declarations_for_Colon_0_90041", "Remove unused declarations for: '{0}'"),
  Declare_a_private_field_named_0: diag(90053, 3 /* Message */, "Declare_a_private_field_named_0_90053", "Declare a private field named '{0}'."),
  Includes_imports_of_types_referenced_by_0: diag(90054, 3 /* Message */, "Includes_imports_of_types_referenced_by_0_90054", "Includes imports of types referenced by '{0}'"),
  Remove_type_from_import_declaration_from_0: diag(90055, 3 /* Message */, "Remove_type_from_import_declaration_from_0_90055", `Remove 'type' from import declaration from "{0}"`),
  Remove_type_from_import_of_0_from_1: diag(90056, 3 /* Message */, "Remove_type_from_import_of_0_from_1_90056", `Remove 'type' from import of '{0}' from "{1}"`),
  Add_import_from_0: diag(90057, 3 /* Message */, "Add_import_from_0_90057", 'Add import from "{0}"'),
  Update_import_from_0: diag(90058, 3 /* Message */, "Update_import_from_0_90058", 'Update import from "{0}"'),
  Export_0_from_module_1: diag(90059, 3 /* Message */, "Export_0_from_module_1_90059", "Export '{0}' from module '{1}'"),
  Export_all_referenced_locals: diag(90060, 3 /* Message */, "Export_all_referenced_locals_90060", "Export all referenced locals"),
  Update_modifiers_of_0: diag(90061, 3 /* Message */, "Update_modifiers_of_0_90061", "Update modifiers of '{0}'"),
  Add_annotation_of_type_0: diag(90062, 3 /* Message */, "Add_annotation_of_type_0_90062", "Add annotation of type '{0}'"),
  Add_return_type_0: diag(90063, 3 /* Message */, "Add_return_type_0_90063", "Add return type '{0}'"),
  Extract_base_class_to_variable: diag(90064, 3 /* Message */, "Extract_base_class_to_variable_90064", "Extract base class to variable"),
  Extract_default_export_to_variable: diag(90065, 3 /* Message */, "Extract_default_export_to_variable_90065", "Extract default export to variable"),
  Extract_binding_expressions_to_variable: diag(90066, 3 /* Message */, "Extract_binding_expressions_to_variable_90066", "Extract binding expressions to variable"),
  Add_all_missing_type_annotations: diag(90067, 3 /* Message */, "Add_all_missing_type_annotations_90067", "Add all missing type annotations"),
  Add_satisfies_and_an_inline_type_assertion_with_0: diag(90068, 3 /* Message */, "Add_satisfies_and_an_inline_type_assertion_with_0_90068", "Add satisfies and an inline type assertion with '{0}'"),
  Extract_to_variable_and_replace_with_0_as_typeof_0: diag(90069, 3 /* Message */, "Extract_to_variable_and_replace_with_0_as_typeof_0_90069", "Extract to variable and replace with '{0} as typeof {0}'"),
  Mark_array_literal_as_const: diag(90070, 3 /* Message */, "Mark_array_literal_as_const_90070", "Mark array literal as const"),
  Annotate_types_of_properties_expando_function_in_a_namespace: diag(90071, 3 /* Message */, "Annotate_types_of_properties_expando_function_in_a_namespace_90071", "Annotate types of properties expando function in a namespace"),
  Convert_function_to_an_ES2015_class: diag(95001, 3 /* Message */, "Convert_function_to_an_ES2015_class_95001", "Convert function to an ES2015 class"),
  Convert_0_to_1_in_0: diag(95003, 3 /* Message */, "Convert_0_to_1_in_0_95003", "Convert '{0}' to '{1} in {0}'"),
  Extract_to_0_in_1: diag(95004, 3 /* Message */, "Extract_to_0_in_1_95004", "Extract to {0} in {1}"),
  Extract_function: diag(95005, 3 /* Message */, "Extract_function_95005", "Extract function"),
  Extract_constant: diag(95006, 3 /* Message */, "Extract_constant_95006", "Extract constant"),
  Extract_to_0_in_enclosing_scope: diag(95007, 3 /* Message */, "Extract_to_0_in_enclosing_scope_95007", "Extract to {0} in enclosing scope"),
  Extract_to_0_in_1_scope: diag(95008, 3 /* Message */, "Extract_to_0_in_1_scope_95008", "Extract to {0} in {1} scope"),
  Annotate_with_type_from_JSDoc: diag(95009, 3 /* Message */, "Annotate_with_type_from_JSDoc_95009", "Annotate with type from JSDoc"),
  Infer_type_of_0_from_usage: diag(95011, 3 /* Message */, "Infer_type_of_0_from_usage_95011", "Infer type of '{0}' from usage"),
  Infer_parameter_types_from_usage: diag(95012, 3 /* Message */, "Infer_parameter_types_from_usage_95012", "Infer parameter types from usage"),
  Convert_to_default_import: diag(95013, 3 /* Message */, "Convert_to_default_import_95013", "Convert to default import"),
  Install_0: diag(95014, 3 /* Message */, "Install_0_95014", "Install '{0}'"),
  Replace_import_with_0: diag(95015, 3 /* Message */, "Replace_import_with_0_95015", "Replace import with '{0}'."),
  Use_synthetic_default_member: diag(95016, 3 /* Message */, "Use_synthetic_default_member_95016", "Use synthetic 'default' member."),
  Convert_to_ES_module: diag(95017, 3 /* Message */, "Convert_to_ES_module_95017", "Convert to ES module"),
  Add_undefined_type_to_property_0: diag(95018, 3 /* Message */, "Add_undefined_type_to_property_0_95018", "Add 'undefined' type to property '{0}'"),
  Add_initializer_to_property_0: diag(95019, 3 /* Message */, "Add_initializer_to_property_0_95019", "Add initializer to property '{0}'"),
  Add_definite_assignment_assertion_to_property_0: diag(95020, 3 /* Message */, "Add_definite_assignment_assertion_to_property_0_95020", "Add definite assignment assertion to property '{0}'"),
  Convert_all_type_literals_to_mapped_type: diag(95021, 3 /* Message */, "Convert_all_type_literals_to_mapped_type_95021", "Convert all type literals to mapped type"),
  Add_all_missing_members: diag(95022, 3 /* Message */, "Add_all_missing_members_95022", "Add all missing members"),
  Infer_all_types_from_usage: diag(95023, 3 /* Message */, "Infer_all_types_from_usage_95023", "Infer all types from usage"),
  Delete_all_unused_declarations: diag(95024, 3 /* Message */, "Delete_all_unused_declarations_95024", "Delete all unused declarations"),
  Prefix_all_unused_declarations_with_where_possible: diag(95025, 3 /* Message */, "Prefix_all_unused_declarations_with_where_possible_95025", "Prefix all unused declarations with '_' where possible"),
  Fix_all_detected_spelling_errors: diag(95026, 3 /* Message */, "Fix_all_detected_spelling_errors_95026", "Fix all detected spelling errors"),
  Add_initializers_to_all_uninitialized_properties: diag(95027, 3 /* Message */, "Add_initializers_to_all_uninitialized_properties_95027", "Add initializers to all uninitialized properties"),
  Add_definite_assignment_assertions_to_all_uninitialized_properties: diag(95028, 3 /* Message */, "Add_definite_assignment_assertions_to_all_uninitialized_properties_95028", "Add definite assignment assertions to all uninitialized properties"),
  Add_undefined_type_to_all_uninitialized_properties: diag(95029, 3 /* Message */, "Add_undefined_type_to_all_uninitialized_properties_95029", "Add undefined type to all uninitialized properties"),
  Change_all_jsdoc_style_types_to_TypeScript: diag(95030, 3 /* Message */, "Change_all_jsdoc_style_types_to_TypeScript_95030", "Change all jsdoc-style types to TypeScript"),
  Change_all_jsdoc_style_types_to_TypeScript_and_add_undefined_to_nullable_types: diag(95031, 3 /* Message */, "Change_all_jsdoc_style_types_to_TypeScript_and_add_undefined_to_nullable_types_95031", "Change all jsdoc-style types to TypeScript (and add '| undefined' to nullable types)"),
  Implement_all_unimplemented_interfaces: diag(95032, 3 /* Message */, "Implement_all_unimplemented_interfaces_95032", "Implement all unimplemented interfaces"),
  Install_all_missing_types_packages: diag(95033, 3 /* Message */, "Install_all_missing_types_packages_95033", "Install all missing types packages"),
  Rewrite_all_as_indexed_access_types: diag(95034, 3 /* Message */, "Rewrite_all_as_indexed_access_types_95034", "Rewrite all as indexed access types"),
  Convert_all_to_default_imports: diag(95035, 3 /* Message */, "Convert_all_to_default_imports_95035", "Convert all to default imports"),
  Make_all_super_calls_the_first_statement_in_their_constructor: diag(95036, 3 /* Message */, "Make_all_super_calls_the_first_statement_in_their_constructor_95036", "Make all 'super()' calls the first statement in their constructor"),
  Add_qualifier_to_all_unresolved_variables_matching_a_member_name: diag(95037, 3 /* Message */, "Add_qualifier_to_all_unresolved_variables_matching_a_member_name_95037", "Add qualifier to all unresolved variables matching a member name"),
  Change_all_extended_interfaces_to_implements: diag(95038, 3 /* Message */, "Change_all_extended_interfaces_to_implements_95038", "Change all extended interfaces to 'implements'"),
  Add_all_missing_super_calls: diag(95039, 3 /* Message */, "Add_all_missing_super_calls_95039", "Add all missing super calls"),
  Implement_all_inherited_abstract_classes: diag(95040, 3 /* Message */, "Implement_all_inherited_abstract_classes_95040", "Implement all inherited abstract classes"),
  Add_all_missing_async_modifiers: diag(95041, 3 /* Message */, "Add_all_missing_async_modifiers_95041", "Add all missing 'async' modifiers"),
  Add_ts_ignore_to_all_error_messages: diag(95042, 3 /* Message */, "Add_ts_ignore_to_all_error_messages_95042", "Add '@ts-ignore' to all error messages"),
  Annotate_everything_with_types_from_JSDoc: diag(95043, 3 /* Message */, "Annotate_everything_with_types_from_JSDoc_95043", "Annotate everything with types from JSDoc"),
  Add_to_all_uncalled_decorators: diag(95044, 3 /* Message */, "Add_to_all_uncalled_decorators_95044", "Add '()' to all uncalled decorators"),
  Convert_all_constructor_functions_to_classes: diag(95045, 3 /* Message */, "Convert_all_constructor_functions_to_classes_95045", "Convert all constructor functions to classes"),
  Generate_get_and_set_accessors: diag(95046, 3 /* Message */, "Generate_get_and_set_accessors_95046", "Generate 'get' and 'set' accessors"),
  Convert_require_to_import: diag(95047, 3 /* Message */, "Convert_require_to_import_95047", "Convert 'require' to 'import'"),
  Convert_all_require_to_import: diag(95048, 3 /* Message */, "Convert_all_require_to_import_95048", "Convert all 'require' to 'import'"),
  Move_to_a_new_file: diag(95049, 3 /* Message */, "Move_to_a_new_file_95049", "Move to a new file"),
  Remove_unreachable_code: diag(95050, 3 /* Message */, "Remove_unreachable_code_95050", "Remove unreachable code"),
  Remove_all_unreachable_code: diag(95051, 3 /* Message */, "Remove_all_unreachable_code_95051", "Remove all unreachable code"),
  Add_missing_typeof: diag(95052, 3 /* Message */, "Add_missing_typeof_95052", "Add missing 'typeof'"),
  Remove_unused_label: diag(95053, 3 /* Message */, "Remove_unused_label_95053", "Remove unused label"),
  Remove_all_unused_labels: diag(95054, 3 /* Message */, "Remove_all_unused_labels_95054", "Remove all unused labels"),
  Convert_0_to_mapped_object_type: diag(95055, 3 /* Message */, "Convert_0_to_mapped_object_type_95055", "Convert '{0}' to mapped object type"),
  Convert_namespace_import_to_named_imports: diag(95056, 3 /* Message */, "Convert_namespace_import_to_named_imports_95056", "Convert namespace import to named imports"),
  Convert_named_imports_to_namespace_import: diag(95057, 3 /* Message */, "Convert_named_imports_to_namespace_import_95057", "Convert named imports to namespace import"),
  Add_or_remove_braces_in_an_arrow_function: diag(95058, 3 /* Message */, "Add_or_remove_braces_in_an_arrow_function_95058", "Add or remove braces in an arrow function"),
  Add_braces_to_arrow_function: diag(95059, 3 /* Message */, "Add_braces_to_arrow_function_95059", "Add braces to arrow function"),
  Remove_braces_from_arrow_function: diag(95060, 3 /* Message */, "Remove_braces_from_arrow_function_95060", "Remove braces from arrow function"),
  Convert_default_export_to_named_export: diag(95061, 3 /* Message */, "Convert_default_export_to_named_export_95061", "Convert default export to named export"),
  Convert_named_export_to_default_export: diag(95062, 3 /* Message */, "Convert_named_export_to_default_export_95062", "Convert named export to default export"),
  Add_missing_enum_member_0: diag(95063, 3 /* Message */, "Add_missing_enum_member_0_95063", "Add missing enum member '{0}'"),
  Add_all_missing_imports: diag(95064, 3 /* Message */, "Add_all_missing_imports_95064", "Add all missing imports"),
  Convert_to_async_function: diag(95065, 3 /* Message */, "Convert_to_async_function_95065", "Convert to async function"),
  Convert_all_to_async_functions: diag(95066, 3 /* Message */, "Convert_all_to_async_functions_95066", "Convert all to async functions"),
  Add_missing_call_parentheses: diag(95067, 3 /* Message */, "Add_missing_call_parentheses_95067", "Add missing call parentheses"),
  Add_all_missing_call_parentheses: diag(95068, 3 /* Message */, "Add_all_missing_call_parentheses_95068", "Add all missing call parentheses"),
  Add_unknown_conversion_for_non_overlapping_types: diag(95069, 3 /* Message */, "Add_unknown_conversion_for_non_overlapping_types_95069", "Add 'unknown' conversion for non-overlapping types"),
  Add_unknown_to_all_conversions_of_non_overlapping_types: diag(95070, 3 /* Message */, "Add_unknown_to_all_conversions_of_non_overlapping_types_95070", "Add 'unknown' to all conversions of non-overlapping types"),
  Add_missing_new_operator_to_call: diag(95071, 3 /* Message */, "Add_missing_new_operator_to_call_95071", "Add missing 'new' operator to call"),
  Add_missing_new_operator_to_all_calls: diag(95072, 3 /* Message */, "Add_missing_new_operator_to_all_calls_95072", "Add missing 'new' operator to all calls"),
  Add_names_to_all_parameters_without_names: diag(95073, 3 /* Message */, "Add_names_to_all_parameters_without_names_95073", "Add names to all parameters without names"),
  Enable_the_experimentalDecorators_option_in_your_configuration_file: diag(95074, 3 /* Message */, "Enable_the_experimentalDecorators_option_in_your_configuration_file_95074", "Enable the 'experimentalDecorators' option in your configuration file"),
  Convert_parameters_to_destructured_object: diag(95075, 3 /* Message */, "Convert_parameters_to_destructured_object_95075", "Convert parameters to destructured object"),
  Extract_type: diag(95077, 3 /* Message */, "Extract_type_95077", "Extract type"),
  Extract_to_type_alias: diag(95078, 3 /* Message */, "Extract_to_type_alias_95078", "Extract to type alias"),
  Extract_to_typedef: diag(95079, 3 /* Message */, "Extract_to_typedef_95079", "Extract to typedef"),
  Infer_this_type_of_0_from_usage: diag(95080, 3 /* Message */, "Infer_this_type_of_0_from_usage_95080", "Infer 'this' type of '{0}' from usage"),
  Add_const_to_unresolved_variable: diag(95081, 3 /* Message */, "Add_const_to_unresolved_variable_95081", "Add 'const' to unresolved variable"),
  Add_const_to_all_unresolved_variables: diag(95082, 3 /* Message */, "Add_const_to_all_unresolved_variables_95082", "Add 'const' to all unresolved variables"),
  Add_await: diag(95083, 3 /* Message */, "Add_await_95083", "Add 'await'"),
  Add_await_to_initializer_for_0: diag(95084, 3 /* Message */, "Add_await_to_initializer_for_0_95084", "Add 'await' to initializer for '{0}'"),
  Fix_all_expressions_possibly_missing_await: diag(95085, 3 /* Message */, "Fix_all_expressions_possibly_missing_await_95085", "Fix all expressions possibly missing 'await'"),
  Remove_unnecessary_await: diag(95086, 3 /* Message */, "Remove_unnecessary_await_95086", "Remove unnecessary 'await'"),
  Remove_all_unnecessary_uses_of_await: diag(95087, 3 /* Message */, "Remove_all_unnecessary_uses_of_await_95087", "Remove all unnecessary uses of 'await'"),
  Enable_the_jsx_flag_in_your_configuration_file: diag(95088, 3 /* Message */, "Enable_the_jsx_flag_in_your_configuration_file_95088", "Enable the '--jsx' flag in your configuration file"),
  Add_await_to_initializers: diag(95089, 3 /* Message */, "Add_await_to_initializers_95089", "Add 'await' to initializers"),
  Extract_to_interface: diag(95090, 3 /* Message */, "Extract_to_interface_95090", "Extract to interface"),
  Convert_to_a_bigint_numeric_literal: diag(95091, 3 /* Message */, "Convert_to_a_bigint_numeric_literal_95091", "Convert to a bigint numeric literal"),
  Convert_all_to_bigint_numeric_literals: diag(95092, 3 /* Message */, "Convert_all_to_bigint_numeric_literals_95092", "Convert all to bigint numeric literals"),
  Convert_const_to_let: diag(95093, 3 /* Message */, "Convert_const_to_let_95093", "Convert 'const' to 'let'"),
  Prefix_with_declare: diag(95094, 3 /* Message */, "Prefix_with_declare_95094", "Prefix with 'declare'"),
  Prefix_all_incorrect_property_declarations_with_declare: diag(95095, 3 /* Message */, "Prefix_all_incorrect_property_declarations_with_declare_95095", "Prefix all incorrect property declarations with 'declare'"),
  Convert_to_template_string: diag(95096, 3 /* Message */, "Convert_to_template_string_95096", "Convert to template string"),
  Add_export_to_make_this_file_into_a_module: diag(95097, 3 /* Message */, "Add_export_to_make_this_file_into_a_module_95097", "Add 'export {}' to make this file into a module"),
  Set_the_target_option_in_your_configuration_file_to_0: diag(95098, 3 /* Message */, "Set_the_target_option_in_your_configuration_file_to_0_95098", "Set the 'target' option in your configuration file to '{0}'"),
  Set_the_module_option_in_your_configuration_file_to_0: diag(95099, 3 /* Message */, "Set_the_module_option_in_your_configuration_file_to_0_95099", "Set the 'module' option in your configuration file to '{0}'"),
  Convert_invalid_character_to_its_html_entity_code: diag(95100, 3 /* Message */, "Convert_invalid_character_to_its_html_entity_code_95100", "Convert invalid character to its html entity code"),
  Convert_all_invalid_characters_to_HTML_entity_code: diag(95101, 3 /* Message */, "Convert_all_invalid_characters_to_HTML_entity_code_95101", "Convert all invalid characters to HTML entity code"),
  Convert_all_const_to_let: diag(95102, 3 /* Message */, "Convert_all_const_to_let_95102", "Convert all 'const' to 'let'"),
  Convert_function_expression_0_to_arrow_function: diag(95105, 3 /* Message */, "Convert_function_expression_0_to_arrow_function_95105", "Convert function expression '{0}' to arrow function"),
  Convert_function_declaration_0_to_arrow_function: diag(95106, 3 /* Message */, "Convert_function_declaration_0_to_arrow_function_95106", "Convert function declaration '{0}' to arrow function"),
  Fix_all_implicit_this_errors: diag(95107, 3 /* Message */, "Fix_all_implicit_this_errors_95107", "Fix all implicit-'this' errors"),
  Wrap_invalid_character_in_an_expression_container: diag(95108, 3 /* Message */, "Wrap_invalid_character_in_an_expression_container_95108", "Wrap invalid character in an expression container"),
  Wrap_all_invalid_characters_in_an_expression_container: diag(95109, 3 /* Message */, "Wrap_all_invalid_characters_in_an_expression_container_95109", "Wrap all invalid characters in an expression container"),
  Visit_https_Colon_Slash_Slashaka_ms_Slashtsconfig_to_read_more_about_this_file: diag(95110, 3 /* Message */, "Visit_https_Colon_Slash_Slashaka_ms_Slashtsconfig_to_read_more_about_this_file_95110", "Visit https://aka.ms/tsconfig to read more about this file"),
  Add_a_return_statement: diag(95111, 3 /* Message */, "Add_a_return_statement_95111", "Add a return statement"),
  Remove_braces_from_arrow_function_body: diag(95112, 3 /* Message */, "Remove_braces_from_arrow_function_body_95112", "Remove braces from arrow function body"),
  Wrap_the_following_body_with_parentheses_which_should_be_an_object_literal: diag(95113, 3 /* Message */, "Wrap_the_following_body_with_parentheses_which_should_be_an_object_literal_95113", "Wrap the following body with parentheses which should be an object literal"),
  Add_all_missing_return_statement: diag(95114, 3 /* Message */, "Add_all_missing_return_statement_95114", "Add all missing return statement"),
  Remove_braces_from_all_arrow_function_bodies_with_relevant_issues: diag(95115, 3 /* Message */, "Remove_braces_from_all_arrow_function_bodies_with_relevant_issues_95115", "Remove braces from all arrow function bodies with relevant issues"),
  Wrap_all_object_literal_with_parentheses: diag(95116, 3 /* Message */, "Wrap_all_object_literal_with_parentheses_95116", "Wrap all object literal with parentheses"),
  Move_labeled_tuple_element_modifiers_to_labels: diag(95117, 3 /* Message */, "Move_labeled_tuple_element_modifiers_to_labels_95117", "Move labeled tuple element modifiers to labels"),
  Convert_overload_list_to_single_signature: diag(95118, 3 /* Message */, "Convert_overload_list_to_single_signature_95118", "Convert overload list to single signature"),
  Generate_get_and_set_accessors_for_all_overriding_properties: diag(95119, 3 /* Message */, "Generate_get_and_set_accessors_for_all_overriding_properties_95119", "Generate 'get' and 'set' accessors for all overriding properties"),
  Wrap_in_JSX_fragment: diag(95120, 3 /* Message */, "Wrap_in_JSX_fragment_95120", "Wrap in JSX fragment"),
  Wrap_all_unparented_JSX_in_JSX_fragment: diag(95121, 3 /* Message */, "Wrap_all_unparented_JSX_in_JSX_fragment_95121", "Wrap all unparented JSX in JSX fragment"),
  Convert_arrow_function_or_function_expression: diag(95122, 3 /* Message */, "Convert_arrow_function_or_function_expression_95122", "Convert arrow function or function expression"),
  Convert_to_anonymous_function: diag(95123, 3 /* Message */, "Convert_to_anonymous_function_95123", "Convert to anonymous function"),
  Convert_to_named_function: diag(95124, 3 /* Message */, "Convert_to_named_function_95124", "Convert to named function"),
  Convert_to_arrow_function: diag(95125, 3 /* Message */, "Convert_to_arrow_function_95125", "Convert to arrow function"),
  Remove_parentheses: diag(95126, 3 /* Message */, "Remove_parentheses_95126", "Remove parentheses"),
  Could_not_find_a_containing_arrow_function: diag(95127, 3 /* Message */, "Could_not_find_a_containing_arrow_function_95127", "Could not find a containing arrow function"),
  Containing_function_is_not_an_arrow_function: diag(95128, 3 /* Message */, "Containing_function_is_not_an_arrow_function_95128", "Containing function is not an arrow function"),
  Could_not_find_export_statement: diag(95129, 3 /* Message */, "Could_not_find_export_statement_95129", "Could not find export statement"),
  This_file_already_has_a_default_export: diag(95130, 3 /* Message */, "This_file_already_has_a_default_export_95130", "This file already has a default export"),
  Could_not_find_import_clause: diag(95131, 3 /* Message */, "Could_not_find_import_clause_95131", "Could not find import clause"),
  Could_not_find_namespace_import_or_named_imports: diag(95132, 3 /* Message */, "Could_not_find_namespace_import_or_named_imports_95132", "Could not find namespace import or named imports"),
  Selection_is_not_a_valid_type_node: diag(95133, 3 /* Message */, "Selection_is_not_a_valid_type_node_95133", "Selection is not a valid type node"),
  No_type_could_be_extracted_from_this_type_node: diag(95134, 3 /* Message */, "No_type_could_be_extracted_from_this_type_node_95134", "No type could be extracted from this type node"),
  Could_not_find_property_for_which_to_generate_accessor: diag(95135, 3 /* Message */, "Could_not_find_property_for_which_to_generate_accessor_95135", "Could not find property for which to generate accessor"),
  Name_is_not_valid: diag(95136, 3 /* Message */, "Name_is_not_valid_95136", "Name is not valid"),
  Can_only_convert_property_with_modifier: diag(95137, 3 /* Message */, "Can_only_convert_property_with_modifier_95137", "Can only convert property with modifier"),
  Switch_each_misused_0_to_1: diag(95138, 3 /* Message */, "Switch_each_misused_0_to_1_95138", "Switch each misused '{0}' to '{1}'"),
  Convert_to_optional_chain_expression: diag(95139, 3 /* Message */, "Convert_to_optional_chain_expression_95139", "Convert to optional chain expression"),
  Could_not_find_convertible_access_expression: diag(95140, 3 /* Message */, "Could_not_find_convertible_access_expression_95140", "Could not find convertible access expression"),
  Could_not_find_matching_access_expressions: diag(95141, 3 /* Message */, "Could_not_find_matching_access_expressions_95141", "Could not find matching access expressions"),
  Can_only_convert_logical_AND_access_chains: diag(95142, 3 /* Message */, "Can_only_convert_logical_AND_access_chains_95142", "Can only convert logical AND access chains"),
  Add_void_to_Promise_resolved_without_a_value: diag(95143, 3 /* Message */, "Add_void_to_Promise_resolved_without_a_value_95143", "Add 'void' to Promise resolved without a value"),
  Add_void_to_all_Promises_resolved_without_a_value: diag(95144, 3 /* Message */, "Add_void_to_all_Promises_resolved_without_a_value_95144", "Add 'void' to all Promises resolved without a value"),
  Use_element_access_for_0: diag(95145, 3 /* Message */, "Use_element_access_for_0_95145", "Use element access for '{0}'"),
  Use_element_access_for_all_undeclared_properties: diag(95146, 3 /* Message */, "Use_element_access_for_all_undeclared_properties_95146", "Use element access for all undeclared properties."),
  Delete_all_unused_imports: diag(95147, 3 /* Message */, "Delete_all_unused_imports_95147", "Delete all unused imports"),
  Infer_function_return_type: diag(95148, 3 /* Message */, "Infer_function_return_type_95148", "Infer function return type"),
  Return_type_must_be_inferred_from_a_function: diag(95149, 3 /* Message */, "Return_type_must_be_inferred_from_a_function_95149", "Return type must be inferred from a function"),
  Could_not_determine_function_return_type: diag(95150, 3 /* Message */, "Could_not_determine_function_return_type_95150", "Could not determine function return type"),
  Could_not_convert_to_arrow_function: diag(95151, 3 /* Message */, "Could_not_convert_to_arrow_function_95151", "Could not convert to arrow function"),
  Could_not_convert_to_named_function: diag(95152, 3 /* Message */, "Could_not_convert_to_named_function_95152", "Could not convert to named function"),
  Could_not_convert_to_anonymous_function: diag(95153, 3 /* Message */, "Could_not_convert_to_anonymous_function_95153", "Could not convert to anonymous function"),
  Can_only_convert_string_concatenations_and_string_literals: diag(95154, 3 /* Message */, "Can_only_convert_string_concatenations_and_string_literals_95154", "Can only convert string concatenations and string literals"),
  Selection_is_not_a_valid_statement_or_statements: diag(95155, 3 /* Message */, "Selection_is_not_a_valid_statement_or_statements_95155", "Selection is not a valid statement or statements"),
  Add_missing_function_declaration_0: diag(95156, 3 /* Message */, "Add_missing_function_declaration_0_95156", "Add missing function declaration '{0}'"),
  Add_all_missing_function_declarations: diag(95157, 3 /* Message */, "Add_all_missing_function_declarations_95157", "Add all missing function declarations"),
  Method_not_implemented: diag(95158, 3 /* Message */, "Method_not_implemented_95158", "Method not implemented."),
  Function_not_implemented: diag(95159, 3 /* Message */, "Function_not_implemented_95159", "Function not implemented."),
  Add_override_modifier: diag(95160, 3 /* Message */, "Add_override_modifier_95160", "Add 'override' modifier"),
  Remove_override_modifier: diag(95161, 3 /* Message */, "Remove_override_modifier_95161", "Remove 'override' modifier"),
  Add_all_missing_override_modifiers: diag(95162, 3 /* Message */, "Add_all_missing_override_modifiers_95162", "Add all missing 'override' modifiers"),
  Remove_all_unnecessary_override_modifiers: diag(95163, 3 /* Message */, "Remove_all_unnecessary_override_modifiers_95163", "Remove all unnecessary 'override' modifiers"),
  Can_only_convert_named_export: diag(95164, 3 /* Message */, "Can_only_convert_named_export_95164", "Can only convert named export"),
  Add_missing_properties: diag(95165, 3 /* Message */, "Add_missing_properties_95165", "Add missing properties"),
  Add_all_missing_properties: diag(95166, 3 /* Message */, "Add_all_missing_properties_95166", "Add all missing properties"),
  Add_missing_attributes: diag(95167, 3 /* Message */, "Add_missing_attributes_95167", "Add missing attributes"),
  Add_all_missing_attributes: diag(95168, 3 /* Message */, "Add_all_missing_attributes_95168", "Add all missing attributes"),
  Add_undefined_to_optional_property_type: diag(95169, 3 /* Message */, "Add_undefined_to_optional_property_type_95169", "Add 'undefined' to optional property type"),
  Convert_named_imports_to_default_import: diag(95170, 3 /* Message */, "Convert_named_imports_to_default_import_95170", "Convert named imports to default import"),
  Delete_unused_param_tag_0: diag(95171, 3 /* Message */, "Delete_unused_param_tag_0_95171", "Delete unused '@param' tag '{0}'"),
  Delete_all_unused_param_tags: diag(95172, 3 /* Message */, "Delete_all_unused_param_tags_95172", "Delete all unused '@param' tags"),
  Rename_param_tag_name_0_to_1: diag(95173, 3 /* Message */, "Rename_param_tag_name_0_to_1_95173", "Rename '@param' tag name '{0}' to '{1}'"),
  Use_0: diag(95174, 3 /* Message */, "Use_0_95174", "Use `{0}`."),
  Use_Number_isNaN_in_all_conditions: diag(95175, 3 /* Message */, "Use_Number_isNaN_in_all_conditions_95175", "Use `Number.isNaN` in all conditions."),
  Convert_typedef_to_TypeScript_type: diag(95176, 3 /* Message */, "Convert_typedef_to_TypeScript_type_95176", "Convert typedef to TypeScript type."),
  Convert_all_typedef_to_TypeScript_types: diag(95177, 3 /* Message */, "Convert_all_typedef_to_TypeScript_types_95177", "Convert all typedef to TypeScript types."),
  Move_to_file: diag(95178, 3 /* Message */, "Move_to_file_95178", "Move to file"),
  Cannot_move_to_file_selected_file_is_invalid: diag(95179, 3 /* Message */, "Cannot_move_to_file_selected_file_is_invalid_95179", "Cannot move to file, selected file is invalid"),
  Use_import_type: diag(95180, 3 /* Message */, "Use_import_type_95180", "Use 'import type'"),
  Use_type_0: diag(95181, 3 /* Message */, "Use_type_0_95181", "Use 'type {0}'"),
  Fix_all_with_type_only_imports: diag(95182, 3 /* Message */, "Fix_all_with_type_only_imports_95182", "Fix all with type-only imports"),
  Cannot_move_statements_to_the_selected_file: diag(95183, 3 /* Message */, "Cannot_move_statements_to_the_selected_file_95183", "Cannot move statements to the selected file"),
  Inline_variable: diag(95184, 3 /* Message */, "Inline_variable_95184", "Inline variable"),
  Could_not_find_variable_to_inline: diag(95185, 3 /* Message */, "Could_not_find_variable_to_inline_95185", "Could not find variable to inline."),
  Variables_with_multiple_declarations_cannot_be_inlined: diag(95186, 3 /* Message */, "Variables_with_multiple_declarations_cannot_be_inlined_95186", "Variables with multiple declarations cannot be inlined."),
  Add_missing_comma_for_object_member_completion_0: diag(95187, 3 /* Message */, "Add_missing_comma_for_object_member_completion_0_95187", "Add missing comma for object member completion '{0}'."),
  Add_missing_parameter_to_0: diag(95188, 3 /* Message */, "Add_missing_parameter_to_0_95188", "Add missing parameter to '{0}'"),
  Add_missing_parameters_to_0: diag(95189, 3 /* Message */, "Add_missing_parameters_to_0_95189", "Add missing parameters to '{0}'"),
  Add_all_missing_parameters: diag(95190, 3 /* Message */, "Add_all_missing_parameters_95190", "Add all missing parameters"),
  Add_optional_parameter_to_0: diag(95191, 3 /* Message */, "Add_optional_parameter_to_0_95191", "Add optional parameter to '{0}'"),
  Add_optional_parameters_to_0: diag(95192, 3 /* Message */, "Add_optional_parameters_to_0_95192", "Add optional parameters to '{0}'"),
  Add_all_optional_parameters: diag(95193, 3 /* Message */, "Add_all_optional_parameters_95193", "Add all optional parameters"),
  Wrap_in_parentheses: diag(95194, 3 /* Message */, "Wrap_in_parentheses_95194", "Wrap in parentheses"),
  Wrap_all_invalid_decorator_expressions_in_parentheses: diag(95195, 3 /* Message */, "Wrap_all_invalid_decorator_expressions_in_parentheses_95195", "Wrap all invalid decorator expressions in parentheses"),
  Add_resolution_mode_import_attribute: diag(95196, 3 /* Message */, "Add_resolution_mode_import_attribute_95196", "Add 'resolution-mode' import attribute"),
  Add_resolution_mode_import_attribute_to_all_type_only_imports_that_need_it: diag(95197, 3 /* Message */, "Add_resolution_mode_import_attribute_to_all_type_only_imports_that_need_it_95197", "Add 'resolution-mode' import attribute to all type-only imports that need it"),
  No_value_exists_in_scope_for_the_shorthand_property_0_Either_declare_one_or_provide_an_initializer: diag(18004, 1 /* Error */, "No_value_exists_in_scope_for_the_shorthand_property_0_Either_declare_one_or_provide_an_initializer_18004", "No value exists in scope for the shorthand property '{0}'. Either declare one or provide an initializer."),
  Classes_may_not_have_a_field_named_constructor: diag(18006, 1 /* Error */, "Classes_may_not_have_a_field_named_constructor_18006", "Classes may not have a field named 'constructor'."),
  JSX_expressions_may_not_use_the_comma_operator_Did_you_mean_to_write_an_array: diag(18007, 1 /* Error */, "JSX_expressions_may_not_use_the_comma_operator_Did_you_mean_to_write_an_array_18007", "JSX expressions may not use the comma operator. Did you mean to write an array?"),
  Private_identifiers_cannot_be_used_as_parameters: diag(18009, 1 /* Error */, "Private_identifiers_cannot_be_used_as_parameters_18009", "Private identifiers cannot be used as parameters."),
  An_accessibility_modifier_cannot_be_used_with_a_private_identifier: diag(18010, 1 /* Error */, "An_accessibility_modifier_cannot_be_used_with_a_private_identifier_18010", "An accessibility modifier cannot be used with a private identifier."),
  The_operand_of_a_delete_operator_cannot_be_a_private_identifier: diag(18011, 1 /* Error */, "The_operand_of_a_delete_operator_cannot_be_a_private_identifier_18011", "The operand of a 'delete' operator cannot be a private identifier."),
  constructor_is_a_reserved_word: diag(18012, 1 /* Error */, "constructor_is_a_reserved_word_18012", "'#constructor' is a reserved word."),
  Property_0_is_not_accessible_outside_class_1_because_it_has_a_private_identifier: diag(18013, 1 /* Error */, "Property_0_is_not_accessible_outside_class_1_because_it_has_a_private_identifier_18013", "Property '{0}' is not accessible outside class '{1}' because it has a private identifier."),
  The_property_0_cannot_be_accessed_on_type_1_within_this_class_because_it_is_shadowed_by_another_private_identifier_with_the_same_spelling: diag(18014, 1 /* Error */, "The_property_0_cannot_be_accessed_on_type_1_within_this_class_because_it_is_shadowed_by_another_priv_18014", "The property '{0}' cannot be accessed on type '{1}' within this class because it is shadowed by another private identifier with the same spelling."),
  Property_0_in_type_1_refers_to_a_different_member_that_cannot_be_accessed_from_within_type_2: diag(18015, 1 /* Error */, "Property_0_in_type_1_refers_to_a_different_member_that_cannot_be_accessed_from_within_type_2_18015", "Property '{0}' in type '{1}' refers to a different member that cannot be accessed from within type '{2}'."),
  Private_identifiers_are_not_allowed_outside_class_bodies: diag(18016, 1 /* Error */, "Private_identifiers_are_not_allowed_outside_class_bodies_18016", "Private identifiers are not allowed outside class bodies."),
  The_shadowing_declaration_of_0_is_defined_here: diag(18017, 1 /* Error */, "The_shadowing_declaration_of_0_is_defined_here_18017", "The shadowing declaration of '{0}' is defined here"),
  The_declaration_of_0_that_you_probably_intended_to_use_is_defined_here: diag(18018, 1 /* Error */, "The_declaration_of_0_that_you_probably_intended_to_use_is_defined_here_18018", "The declaration of '{0}' that you probably intended to use is defined here"),
  _0_modifier_cannot_be_used_with_a_private_identifier: diag(18019, 1 /* Error */, "_0_modifier_cannot_be_used_with_a_private_identifier_18019", "'{0}' modifier cannot be used with a private identifier."),
  An_enum_member_cannot_be_named_with_a_private_identifier: diag(18024, 1 /* Error */, "An_enum_member_cannot_be_named_with_a_private_identifier_18024", "An enum member cannot be named with a private identifier."),
  can_only_be_used_at_the_start_of_a_file: diag(18026, 1 /* Error */, "can_only_be_used_at_the_start_of_a_file_18026", "'#!' can only be used at the start of a file."),
  Compiler_reserves_name_0_when_emitting_private_identifier_downlevel: diag(18027, 1 /* Error */, "Compiler_reserves_name_0_when_emitting_private_identifier_downlevel_18027", "Compiler reserves name '{0}' when emitting private identifier downlevel."),
  Private_identifiers_are_only_available_when_targeting_ECMAScript_2015_and_higher: diag(18028, 1 /* Error */, "Private_identifiers_are_only_available_when_targeting_ECMAScript_2015_and_higher_18028", "Private identifiers are only available when targeting ECMAScript 2015 and higher."),
  Private_identifiers_are_not_allowed_in_variable_declarations: diag(18029, 1 /* Error */, "Private_identifiers_are_not_allowed_in_variable_declarations_18029", "Private identifiers are not allowed in variable declarations."),
  An_optional_chain_cannot_contain_private_identifiers: diag(18030, 1 /* Error */, "An_optional_chain_cannot_contain_private_identifiers_18030", "An optional chain cannot contain private identifiers."),
  The_intersection_0_was_reduced_to_never_because_property_1_has_conflicting_types_in_some_constituents: diag(18031, 1 /* Error */, "The_intersection_0_was_reduced_to_never_because_property_1_has_conflicting_types_in_some_constituent_18031", "The intersection '{0}' was reduced to 'never' because property '{1}' has conflicting types in some constituents."),
  The_intersection_0_was_reduced_to_never_because_property_1_exists_in_multiple_constituents_and_is_private_in_some: diag(18032, 1 /* Error */, "The_intersection_0_was_reduced_to_never_because_property_1_exists_in_multiple_constituents_and_is_pr_18032", "The intersection '{0}' was reduced to 'never' because property '{1}' exists in multiple constituents and is private in some."),
  Type_0_is_not_assignable_to_type_1_as_required_for_computed_enum_member_values: diag(18033, 1 /* Error */, "Type_0_is_not_assignable_to_type_1_as_required_for_computed_enum_member_values_18033", "Type '{0}' is not assignable to type '{1}' as required for computed enum member values."),
  Specify_the_JSX_fragment_factory_function_to_use_when_targeting_react_JSX_emit_with_jsxFactory_compiler_option_is_specified_e_g_Fragment: diag(18034, 3 /* Message */, "Specify_the_JSX_fragment_factory_function_to_use_when_targeting_react_JSX_emit_with_jsxFactory_compi_18034", "Specify the JSX fragment factory function to use when targeting 'react' JSX emit with 'jsxFactory' compiler option is specified, e.g. 'Fragment'."),
  Invalid_value_for_jsxFragmentFactory_0_is_not_a_valid_identifier_or_qualified_name: diag(18035, 1 /* Error */, "Invalid_value_for_jsxFragmentFactory_0_is_not_a_valid_identifier_or_qualified_name_18035", "Invalid value for 'jsxFragmentFactory'. '{0}' is not a valid identifier or qualified-name."),
  Class_decorators_can_t_be_used_with_static_private_identifier_Consider_removing_the_experimental_decorator: diag(18036, 1 /* Error */, "Class_decorators_can_t_be_used_with_static_private_identifier_Consider_removing_the_experimental_dec_18036", "Class decorators can't be used with static private identifier. Consider removing the experimental decorator."),
  await_expression_cannot_be_used_inside_a_class_static_block: diag(18037, 1 /* Error */, "await_expression_cannot_be_used_inside_a_class_static_block_18037", "'await' expression cannot be used inside a class static block."),
  for_await_loops_cannot_be_used_inside_a_class_static_block: diag(18038, 1 /* Error */, "for_await_loops_cannot_be_used_inside_a_class_static_block_18038", "'for await' loops cannot be used inside a class static block."),
  Invalid_use_of_0_It_cannot_be_used_inside_a_class_static_block: diag(18039, 1 /* Error */, "Invalid_use_of_0_It_cannot_be_used_inside_a_class_static_block_18039", "Invalid use of '{0}'. It cannot be used inside a class static block."),
  A_return_statement_cannot_be_used_inside_a_class_static_block: diag(18041, 1 /* Error */, "A_return_statement_cannot_be_used_inside_a_class_static_block_18041", "A 'return' statement cannot be used inside a class static block."),
  _0_is_a_type_and_cannot_be_imported_in_JavaScript_files_Use_1_in_a_JSDoc_type_annotation: diag(18042, 1 /* Error */, "_0_is_a_type_and_cannot_be_imported_in_JavaScript_files_Use_1_in_a_JSDoc_type_annotation_18042", "'{0}' is a type and cannot be imported in JavaScript files. Use '{1}' in a JSDoc type annotation."),
  Types_cannot_appear_in_export_declarations_in_JavaScript_files: diag(18043, 1 /* Error */, "Types_cannot_appear_in_export_declarations_in_JavaScript_files_18043", "Types cannot appear in export declarations in JavaScript files."),
  _0_is_automatically_exported_here: diag(18044, 3 /* Message */, "_0_is_automatically_exported_here_18044", "'{0}' is automatically exported here."),
  Properties_with_the_accessor_modifier_are_only_available_when_targeting_ECMAScript_2015_and_higher: diag(18045, 1 /* Error */, "Properties_with_the_accessor_modifier_are_only_available_when_targeting_ECMAScript_2015_and_higher_18045", "Properties with the 'accessor' modifier are only available when targeting ECMAScript 2015 and higher."),
  _0_is_of_type_unknown: diag(18046, 1 /* Error */, "_0_is_of_type_unknown_18046", "'{0}' is of type 'unknown'."),
  _0_is_possibly_null: diag(18047, 1 /* Error */, "_0_is_possibly_null_18047", "'{0}' is possibly 'null'."),
  _0_is_possibly_undefined: diag(18048, 1 /* Error */, "_0_is_possibly_undefined_18048", "'{0}' is possibly 'undefined'."),
  _0_is_possibly_null_or_undefined: diag(18049, 1 /* Error */, "_0_is_possibly_null_or_undefined_18049", "'{0}' is possibly 'null' or 'undefined'."),
  The_value_0_cannot_be_used_here: diag(18050, 1 /* Error */, "The_value_0_cannot_be_used_here_18050", "The value '{0}' cannot be used here."),
  Compiler_option_0_cannot_be_given_an_empty_string: diag(18051, 1 /* Error */, "Compiler_option_0_cannot_be_given_an_empty_string_18051", "Compiler option '{0}' cannot be given an empty string."),
  Its_type_0_is_not_a_valid_JSX_element_type: diag(18053, 1 /* Error */, "Its_type_0_is_not_a_valid_JSX_element_type_18053", "Its type '{0}' is not a valid JSX element type."),
  await_using_statements_cannot_be_used_inside_a_class_static_block: diag(18054, 1 /* Error */, "await_using_statements_cannot_be_used_inside_a_class_static_block_18054", "'await using' statements cannot be used inside a class static block."),
  _0_has_a_string_type_but_must_have_syntactically_recognizable_string_syntax_when_isolatedModules_is_enabled: diag(18055, 1 /* Error */, "_0_has_a_string_type_but_must_have_syntactically_recognizable_string_syntax_when_isolatedModules_is__18055", "'{0}' has a string type, but must have syntactically recognizable string syntax when 'isolatedModules' is enabled."),
  Enum_member_following_a_non_literal_numeric_member_must_have_an_initializer_when_isolatedModules_is_enabled: diag(18056, 1 /* Error */, "Enum_member_following_a_non_literal_numeric_member_must_have_an_initializer_when_isolatedModules_is__18056", "Enum member following a non-literal numeric member must have an initializer when 'isolatedModules' is enabled."),
  String_literal_import_and_export_names_are_not_supported_when_the_module_flag_is_set_to_es2015_or_es2020: diag(18057, 1 /* Error */, "String_literal_import_and_export_names_are_not_supported_when_the_module_flag_is_set_to_es2015_or_es_18057", "String literal import and export names are not supported when the '--module' flag is set to 'es2015' or 'es2020'.")
};

// src/compiler/scanner.ts
function tokenIsIdentifierOrKeyword(token) {
  return token >= 80 /* Identifier */;
}
function tokenIsIdentifierOrKeywordOrGreaterThan(token) {
  return token === 32 /* GreaterThanToken */ || tokenIsIdentifierOrKeyword(token);
}
var textToKeywordObj = {
  abstract: 128 /* AbstractKeyword */,
  accessor: 129 /* AccessorKeyword */,
  any: 133 /* AnyKeyword */,
  as: 130 /* AsKeyword */,
  asserts: 131 /* AssertsKeyword */,
  assert: 132 /* AssertKeyword */,
  bigint: 163 /* BigIntKeyword */,
  boolean: 136 /* BooleanKeyword */,
  break: 83 /* BreakKeyword */,
  case: 84 /* CaseKeyword */,
  catch: 85 /* CatchKeyword */,
  class: 86 /* ClassKeyword */,
  continue: 88 /* ContinueKeyword */,
  const: 87 /* ConstKeyword */,
  ["constructor"]: 137 /* ConstructorKeyword */,
  debugger: 89 /* DebuggerKeyword */,
  declare: 138 /* DeclareKeyword */,
  default: 90 /* DefaultKeyword */,
  delete: 91 /* DeleteKeyword */,
  do: 92 /* DoKeyword */,
  else: 93 /* ElseKeyword */,
  enum: 94 /* EnumKeyword */,
  export: 95 /* ExportKeyword */,
  extends: 96 /* ExtendsKeyword */,
  false: 97 /* FalseKeyword */,
  finally: 98 /* FinallyKeyword */,
  for: 99 /* ForKeyword */,
  from: 161 /* FromKeyword */,
  function: 100 /* FunctionKeyword */,
  get: 139 /* GetKeyword */,
  if: 101 /* IfKeyword */,
  implements: 119 /* ImplementsKeyword */,
  import: 102 /* ImportKeyword */,
  in: 103 /* InKeyword */,
  infer: 140 /* InferKeyword */,
  instanceof: 104 /* InstanceOfKeyword */,
  interface: 120 /* InterfaceKeyword */,
  intrinsic: 141 /* IntrinsicKeyword */,
  is: 142 /* IsKeyword */,
  keyof: 143 /* KeyOfKeyword */,
  let: 121 /* LetKeyword */,
  module: 144 /* ModuleKeyword */,
  namespace: 145 /* NamespaceKeyword */,
  never: 146 /* NeverKeyword */,
  new: 105 /* NewKeyword */,
  null: 106 /* NullKeyword */,
  number: 150 /* NumberKeyword */,
  object: 151 /* ObjectKeyword */,
  package: 122 /* PackageKeyword */,
  private: 123 /* PrivateKeyword */,
  protected: 124 /* ProtectedKeyword */,
  public: 125 /* PublicKeyword */,
  override: 164 /* OverrideKeyword */,
  out: 147 /* OutKeyword */,
  readonly: 148 /* ReadonlyKeyword */,
  require: 149 /* RequireKeyword */,
  global: 162 /* GlobalKeyword */,
  return: 107 /* ReturnKeyword */,
  satisfies: 152 /* SatisfiesKeyword */,
  set: 153 /* SetKeyword */,
  static: 126 /* StaticKeyword */,
  string: 154 /* StringKeyword */,
  super: 108 /* SuperKeyword */,
  switch: 109 /* SwitchKeyword */,
  symbol: 155 /* SymbolKeyword */,
  this: 110 /* ThisKeyword */,
  throw: 111 /* ThrowKeyword */,
  true: 112 /* TrueKeyword */,
  try: 113 /* TryKeyword */,
  type: 156 /* TypeKeyword */,
  typeof: 114 /* TypeOfKeyword */,
  undefined: 157 /* UndefinedKeyword */,
  unique: 158 /* UniqueKeyword */,
  unknown: 159 /* UnknownKeyword */,
  using: 160 /* UsingKeyword */,
  var: 115 /* VarKeyword */,
  void: 116 /* VoidKeyword */,
  while: 117 /* WhileKeyword */,
  with: 118 /* WithKeyword */,
  yield: 127 /* YieldKeyword */,
  async: 134 /* AsyncKeyword */,
  await: 135 /* AwaitKeyword */,
  of: 165 /* OfKeyword */
};
var textToKeyword = new Map(Object.entries(textToKeywordObj));
var textToToken = new Map(Object.entries({
  ...textToKeywordObj,
  "{": 19 /* OpenBraceToken */,
  "}": 20 /* CloseBraceToken */,
  "(": 21 /* OpenParenToken */,
  ")": 22 /* CloseParenToken */,
  "[": 23 /* OpenBracketToken */,
  "]": 24 /* CloseBracketToken */,
  ".": 25 /* DotToken */,
  "...": 26 /* DotDotDotToken */,
  ";": 27 /* SemicolonToken */,
  ",": 28 /* CommaToken */,
  "<": 30 /* LessThanToken */,
  ">": 32 /* GreaterThanToken */,
  "<=": 33 /* LessThanEqualsToken */,
  ">=": 34 /* GreaterThanEqualsToken */,
  "==": 35 /* EqualsEqualsToken */,
  "!=": 36 /* ExclamationEqualsToken */,
  "===": 37 /* EqualsEqualsEqualsToken */,
  "!==": 38 /* ExclamationEqualsEqualsToken */,
  "=>": 39 /* EqualsGreaterThanToken */,
  "+": 40 /* PlusToken */,
  "-": 41 /* MinusToken */,
  "**": 43 /* AsteriskAsteriskToken */,
  "*": 42 /* AsteriskToken */,
  "/": 44 /* SlashToken */,
  "%": 45 /* PercentToken */,
  "++": 46 /* PlusPlusToken */,
  "--": 47 /* MinusMinusToken */,
  "<<": 48 /* LessThanLessThanToken */,
  "</": 31 /* LessThanSlashToken */,
  ">>": 49 /* GreaterThanGreaterThanToken */,
  ">>>": 50 /* GreaterThanGreaterThanGreaterThanToken */,
  "&": 51 /* AmpersandToken */,
  "|": 52 /* BarToken */,
  "^": 53 /* CaretToken */,
  "!": 54 /* ExclamationToken */,
  "~": 55 /* TildeToken */,
  "&&": 56 /* AmpersandAmpersandToken */,
  "||": 57 /* BarBarToken */,
  "?": 58 /* QuestionToken */,
  "??": 61 /* QuestionQuestionToken */,
  "?.": 29 /* QuestionDotToken */,
  ":": 59 /* ColonToken */,
  "=": 64 /* EqualsToken */,
  "+=": 65 /* PlusEqualsToken */,
  "-=": 66 /* MinusEqualsToken */,
  "*=": 67 /* AsteriskEqualsToken */,
  "**=": 68 /* AsteriskAsteriskEqualsToken */,
  "/=": 69 /* SlashEqualsToken */,
  "%=": 70 /* PercentEqualsToken */,
  "<<=": 71 /* LessThanLessThanEqualsToken */,
  ">>=": 72 /* GreaterThanGreaterThanEqualsToken */,
  ">>>=": 73 /* GreaterThanGreaterThanGreaterThanEqualsToken */,
  "&=": 74 /* AmpersandEqualsToken */,
  "|=": 75 /* BarEqualsToken */,
  "^=": 79 /* CaretEqualsToken */,
  "||=": 76 /* BarBarEqualsToken */,
  "&&=": 77 /* AmpersandAmpersandEqualsToken */,
  "??=": 78 /* QuestionQuestionEqualsToken */,
  "@": 60 /* AtToken */,
  "#": 63 /* HashToken */,
  "`": 62 /* BacktickToken */
}));
var charCodeToRegExpFlag = /* @__PURE__ */ new Map([
  [100 /* d */, 1 /* HasIndices */],
  [103 /* g */, 2 /* Global */],
  [105 /* i */, 4 /* IgnoreCase */],
  [109 /* m */, 8 /* Multiline */],
  [115 /* s */, 16 /* DotAll */],
  [117 /* u */, 32 /* Unicode */],
  [118 /* v */, 64 /* UnicodeSets */],
  [121 /* y */, 128 /* Sticky */]
]);
var regExpFlagToFirstAvailableLanguageVersion = /* @__PURE__ */ new Map([
  [1 /* HasIndices */, LanguageFeatureMinimumTarget.RegularExpressionFlagsHasIndices],
  [16 /* DotAll */, LanguageFeatureMinimumTarget.RegularExpressionFlagsDotAll],
  [32 /* Unicode */, LanguageFeatureMinimumTarget.RegularExpressionFlagsUnicode],
  [64 /* UnicodeSets */, LanguageFeatureMinimumTarget.RegularExpressionFlagsUnicodeSets],
  [128 /* Sticky */, LanguageFeatureMinimumTarget.RegularExpressionFlagsSticky]
]);
var unicodeES5IdentifierStart = [170, 170, 181, 181, 186, 186, 192, 214, 216, 246, 248, 705, 710, 721, 736, 740, 748, 748, 750, 750, 880, 884, 886, 887, 890, 893, 902, 902, 904, 906, 908, 908, 910, 929, 931, 1013, 1015, 1153, 1162, 1319, 1329, 1366, 1369, 1369, 1377, 1415, 1488, 1514, 1520, 1522, 1568, 1610, 1646, 1647, 1649, 1747, 1749, 1749, 1765, 1766, 1774, 1775, 1786, 1788, 1791, 1791, 1808, 1808, 1810, 1839, 1869, 1957, 1969, 1969, 1994, 2026, 2036, 2037, 2042, 2042, 2048, 2069, 2074, 2074, 2084, 2084, 2088, 2088, 2112, 2136, 2208, 2208, 2210, 2220, 2308, 2361, 2365, 2365, 2384, 2384, 2392, 2401, 2417, 2423, 2425, 2431, 2437, 2444, 2447, 2448, 2451, 2472, 2474, 2480, 2482, 2482, 2486, 2489, 2493, 2493, 2510, 2510, 2524, 2525, 2527, 2529, 2544, 2545, 2565, 2570, 2575, 2576, 2579, 2600, 2602, 2608, 2610, 2611, 2613, 2614, 2616, 2617, 2649, 2652, 2654, 2654, 2674, 2676, 2693, 2701, 2703, 2705, 2707, 2728, 2730, 2736, 2738, 2739, 2741, 2745, 2749, 2749, 2768, 2768, 2784, 2785, 2821, 2828, 2831, 2832, 2835, 2856, 2858, 2864, 2866, 2867, 2869, 2873, 2877, 2877, 2908, 2909, 2911, 2913, 2929, 2929, 2947, 2947, 2949, 2954, 2958, 2960, 2962, 2965, 2969, 2970, 2972, 2972, 2974, 2975, 2979, 2980, 2984, 2986, 2990, 3001, 3024, 3024, 3077, 3084, 3086, 3088, 3090, 3112, 3114, 3123, 3125, 3129, 3133, 3133, 3160, 3161, 3168, 3169, 3205, 3212, 3214, 3216, 3218, 3240, 3242, 3251, 3253, 3257, 3261, 3261, 3294, 3294, 3296, 3297, 3313, 3314, 3333, 3340, 3342, 3344, 3346, 3386, 3389, 3389, 3406, 3406, 3424, 3425, 3450, 3455, 3461, 3478, 3482, 3505, 3507, 3515, 3517, 3517, 3520, 3526, 3585, 3632, 3634, 3635, 3648, 3654, 3713, 3714, 3716, 3716, 3719, 3720, 3722, 3722, 3725, 3725, 3732, 3735, 3737, 3743, 3745, 3747, 3749, 3749, 3751, 3751, 3754, 3755, 3757, 3760, 3762, 3763, 3773, 3773, 3776, 3780, 3782, 3782, 3804, 3807, 3840, 3840, 3904, 3911, 3913, 3948, 3976, 3980, 4096, 4138, 4159, 4159, 4176, 4181, 4186, 4189, 4193, 4193, 4197, 4198, 4206, 4208, 4213, 4225, 4238, 4238, 4256, 4293, 4295, 4295, 4301, 4301, 4304, 4346, 4348, 4680, 4682, 4685, 4688, 4694, 4696, 4696, 4698, 4701, 4704, 4744, 4746, 4749, 4752, 4784, 4786, 4789, 4792, 4798, 4800, 4800, 4802, 4805, 4808, 4822, 4824, 4880, 4882, 4885, 4888, 4954, 4992, 5007, 5024, 5108, 5121, 5740, 5743, 5759, 5761, 5786, 5792, 5866, 5870, 5872, 5888, 5900, 5902, 5905, 5920, 5937, 5952, 5969, 5984, 5996, 5998, 6e3, 6016, 6067, 6103, 6103, 6108, 6108, 6176, 6263, 6272, 6312, 6314, 6314, 6320, 6389, 6400, 6428, 6480, 6509, 6512, 6516, 6528, 6571, 6593, 6599, 6656, 6678, 6688, 6740, 6823, 6823, 6917, 6963, 6981, 6987, 7043, 7072, 7086, 7087, 7098, 7141, 7168, 7203, 7245, 7247, 7258, 7293, 7401, 7404, 7406, 7409, 7413, 7414, 7424, 7615, 7680, 7957, 7960, 7965, 7968, 8005, 8008, 8013, 8016, 8023, 8025, 8025, 8027, 8027, 8029, 8029, 8031, 8061, 8064, 8116, 8118, 8124, 8126, 8126, 8130, 8132, 8134, 8140, 8144, 8147, 8150, 8155, 8160, 8172, 8178, 8180, 8182, 8188, 8305, 8305, 8319, 8319, 8336, 8348, 8450, 8450, 8455, 8455, 8458, 8467, 8469, 8469, 8473, 8477, 8484, 8484, 8486, 8486, 8488, 8488, 8490, 8493, 8495, 8505, 8508, 8511, 8517, 8521, 8526, 8526, 8544, 8584, 11264, 11310, 11312, 11358, 11360, 11492, 11499, 11502, 11506, 11507, 11520, 11557, 11559, 11559, 11565, 11565, 11568, 11623, 11631, 11631, 11648, 11670, 11680, 11686, 11688, 11694, 11696, 11702, 11704, 11710, 11712, 11718, 11720, 11726, 11728, 11734, 11736, 11742, 11823, 11823, 12293, 12295, 12321, 12329, 12337, 12341, 12344, 12348, 12353, 12438, 12445, 12447, 12449, 12538, 12540, 12543, 12549, 12589, 12593, 12686, 12704, 12730, 12784, 12799, 13312, 19893, 19968, 40908, 40960, 42124, 42192, 42237, 42240, 42508, 42512, 42527, 42538, 42539, 42560, 42606, 42623, 42647, 42656, 42735, 42775, 42783, 42786, 42888, 42891, 42894, 42896, 42899, 42912, 42922, 43e3, 43009, 43011, 43013, 43015, 43018, 43020, 43042, 43072, 43123, 43138, 43187, 43250, 43255, 43259, 43259, 43274, 43301, 43312, 43334, 43360, 43388, 43396, 43442, 43471, 43471, 43520, 43560, 43584, 43586, 43588, 43595, 43616, 43638, 43642, 43642, 43648, 43695, 43697, 43697, 43701, 43702, 43705, 43709, 43712, 43712, 43714, 43714, 43739, 43741, 43744, 43754, 43762, 43764, 43777, 43782, 43785, 43790, 43793, 43798, 43808, 43814, 43816, 43822, 43968, 44002, 44032, 55203, 55216, 55238, 55243, 55291, 63744, 64109, 64112, 64217, 64256, 64262, 64275, 64279, 64285, 64285, 64287, 64296, 64298, 64310, 64312, 64316, 64318, 64318, 64320, 64321, 64323, 64324, 64326, 64433, 64467, 64829, 64848, 64911, 64914, 64967, 65008, 65019, 65136, 65140, 65142, 65276, 65313, 65338, 65345, 65370, 65382, 65470, 65474, 65479, 65482, 65487, 65490, 65495, 65498, 65500];
var unicodeES5IdentifierPart = [170, 170, 181, 181, 186, 186, 192, 214, 216, 246, 248, 705, 710, 721, 736, 740, 748, 748, 750, 750, 768, 884, 886, 887, 890, 893, 902, 902, 904, 906, 908, 908, 910, 929, 931, 1013, 1015, 1153, 1155, 1159, 1162, 1319, 1329, 1366, 1369, 1369, 1377, 1415, 1425, 1469, 1471, 1471, 1473, 1474, 1476, 1477, 1479, 1479, 1488, 1514, 1520, 1522, 1552, 1562, 1568, 1641, 1646, 1747, 1749, 1756, 1759, 1768, 1770, 1788, 1791, 1791, 1808, 1866, 1869, 1969, 1984, 2037, 2042, 2042, 2048, 2093, 2112, 2139, 2208, 2208, 2210, 2220, 2276, 2302, 2304, 2403, 2406, 2415, 2417, 2423, 2425, 2431, 2433, 2435, 2437, 2444, 2447, 2448, 2451, 2472, 2474, 2480, 2482, 2482, 2486, 2489, 2492, 2500, 2503, 2504, 2507, 2510, 2519, 2519, 2524, 2525, 2527, 2531, 2534, 2545, 2561, 2563, 2565, 2570, 2575, 2576, 2579, 2600, 2602, 2608, 2610, 2611, 2613, 2614, 2616, 2617, 2620, 2620, 2622, 2626, 2631, 2632, 2635, 2637, 2641, 2641, 2649, 2652, 2654, 2654, 2662, 2677, 2689, 2691, 2693, 2701, 2703, 2705, 2707, 2728, 2730, 2736, 2738, 2739, 2741, 2745, 2748, 2757, 2759, 2761, 2763, 2765, 2768, 2768, 2784, 2787, 2790, 2799, 2817, 2819, 2821, 2828, 2831, 2832, 2835, 2856, 2858, 2864, 2866, 2867, 2869, 2873, 2876, 2884, 2887, 2888, 2891, 2893, 2902, 2903, 2908, 2909, 2911, 2915, 2918, 2927, 2929, 2929, 2946, 2947, 2949, 2954, 2958, 2960, 2962, 2965, 2969, 2970, 2972, 2972, 2974, 2975, 2979, 2980, 2984, 2986, 2990, 3001, 3006, 3010, 3014, 3016, 3018, 3021, 3024, 3024, 3031, 3031, 3046, 3055, 3073, 3075, 3077, 3084, 3086, 3088, 3090, 3112, 3114, 3123, 3125, 3129, 3133, 3140, 3142, 3144, 3146, 3149, 3157, 3158, 3160, 3161, 3168, 3171, 3174, 3183, 3202, 3203, 3205, 3212, 3214, 3216, 3218, 3240, 3242, 3251, 3253, 3257, 3260, 3268, 3270, 3272, 3274, 3277, 3285, 3286, 3294, 3294, 3296, 3299, 3302, 3311, 3313, 3314, 3330, 3331, 3333, 3340, 3342, 3344, 3346, 3386, 3389, 3396, 3398, 3400, 3402, 3406, 3415, 3415, 3424, 3427, 3430, 3439, 3450, 3455, 3458, 3459, 3461, 3478, 3482, 3505, 3507, 3515, 3517, 3517, 3520, 3526, 3530, 3530, 3535, 3540, 3542, 3542, 3544, 3551, 3570, 3571, 3585, 3642, 3648, 3662, 3664, 3673, 3713, 3714, 3716, 3716, 3719, 3720, 3722, 3722, 3725, 3725, 3732, 3735, 3737, 3743, 3745, 3747, 3749, 3749, 3751, 3751, 3754, 3755, 3757, 3769, 3771, 3773, 3776, 3780, 3782, 3782, 3784, 3789, 3792, 3801, 3804, 3807, 3840, 3840, 3864, 3865, 3872, 3881, 3893, 3893, 3895, 3895, 3897, 3897, 3902, 3911, 3913, 3948, 3953, 3972, 3974, 3991, 3993, 4028, 4038, 4038, 4096, 4169, 4176, 4253, 4256, 4293, 4295, 4295, 4301, 4301, 4304, 4346, 4348, 4680, 4682, 4685, 4688, 4694, 4696, 4696, 4698, 4701, 4704, 4744, 4746, 4749, 4752, 4784, 4786, 4789, 4792, 4798, 4800, 4800, 4802, 4805, 4808, 4822, 4824, 4880, 4882, 4885, 4888, 4954, 4957, 4959, 4992, 5007, 5024, 5108, 5121, 5740, 5743, 5759, 5761, 5786, 5792, 5866, 5870, 5872, 5888, 5900, 5902, 5908, 5920, 5940, 5952, 5971, 5984, 5996, 5998, 6e3, 6002, 6003, 6016, 6099, 6103, 6103, 6108, 6109, 6112, 6121, 6155, 6157, 6160, 6169, 6176, 6263, 6272, 6314, 6320, 6389, 6400, 6428, 6432, 6443, 6448, 6459, 6470, 6509, 6512, 6516, 6528, 6571, 6576, 6601, 6608, 6617, 6656, 6683, 6688, 6750, 6752, 6780, 6783, 6793, 6800, 6809, 6823, 6823, 6912, 6987, 6992, 7001, 7019, 7027, 7040, 7155, 7168, 7223, 7232, 7241, 7245, 7293, 7376, 7378, 7380, 7414, 7424, 7654, 7676, 7957, 7960, 7965, 7968, 8005, 8008, 8013, 8016, 8023, 8025, 8025, 8027, 8027, 8029, 8029, 8031, 8061, 8064, 8116, 8118, 8124, 8126, 8126, 8130, 8132, 8134, 8140, 8144, 8147, 8150, 8155, 8160, 8172, 8178, 8180, 8182, 8188, 8204, 8205, 8255, 8256, 8276, 8276, 8305, 8305, 8319, 8319, 8336, 8348, 8400, 8412, 8417, 8417, 8421, 8432, 8450, 8450, 8455, 8455, 8458, 8467, 8469, 8469, 8473, 8477, 8484, 8484, 8486, 8486, 8488, 8488, 8490, 8493, 8495, 8505, 8508, 8511, 8517, 8521, 8526, 8526, 8544, 8584, 11264, 11310, 11312, 11358, 11360, 11492, 11499, 11507, 11520, 11557, 11559, 11559, 11565, 11565, 11568, 11623, 11631, 11631, 11647, 11670, 11680, 11686, 11688, 11694, 11696, 11702, 11704, 11710, 11712, 11718, 11720, 11726, 11728, 11734, 11736, 11742, 11744, 11775, 11823, 11823, 12293, 12295, 12321, 12335, 12337, 12341, 12344, 12348, 12353, 12438, 12441, 12442, 12445, 12447, 12449, 12538, 12540, 12543, 12549, 12589, 12593, 12686, 12704, 12730, 12784, 12799, 13312, 19893, 19968, 40908, 40960, 42124, 42192, 42237, 42240, 42508, 42512, 42539, 42560, 42607, 42612, 42621, 42623, 42647, 42655, 42737, 42775, 42783, 42786, 42888, 42891, 42894, 42896, 42899, 42912, 42922, 43e3, 43047, 43072, 43123, 43136, 43204, 43216, 43225, 43232, 43255, 43259, 43259, 43264, 43309, 43312, 43347, 43360, 43388, 43392, 43456, 43471, 43481, 43520, 43574, 43584, 43597, 43600, 43609, 43616, 43638, 43642, 43643, 43648, 43714, 43739, 43741, 43744, 43759, 43762, 43766, 43777, 43782, 43785, 43790, 43793, 43798, 43808, 43814, 43816, 43822, 43968, 44010, 44012, 44013, 44016, 44025, 44032, 55203, 55216, 55238, 55243, 55291, 63744, 64109, 64112, 64217, 64256, 64262, 64275, 64279, 64285, 64296, 64298, 64310, 64312, 64316, 64318, 64318, 64320, 64321, 64323, 64324, 64326, 64433, 64467, 64829, 64848, 64911, 64914, 64967, 65008, 65019, 65024, 65039, 65056, 65062, 65075, 65076, 65101, 65103, 65136, 65140, 65142, 65276, 65296, 65305, 65313, 65338, 65343, 65343, 65345, 65370, 65382, 65470, 65474, 65479, 65482, 65487, 65490, 65495, 65498, 65500];
var unicodeESNextIdentifierStart = [65, 90, 97, 122, 170, 170, 181, 181, 186, 186, 192, 214, 216, 246, 248, 705, 710, 721, 736, 740, 748, 748, 750, 750, 880, 884, 886, 887, 890, 893, 895, 895, 902, 902, 904, 906, 908, 908, 910, 929, 931, 1013, 1015, 1153, 1162, 1327, 1329, 1366, 1369, 1369, 1376, 1416, 1488, 1514, 1519, 1522, 1568, 1610, 1646, 1647, 1649, 1747, 1749, 1749, 1765, 1766, 1774, 1775, 1786, 1788, 1791, 1791, 1808, 1808, 1810, 1839, 1869, 1957, 1969, 1969, 1994, 2026, 2036, 2037, 2042, 2042, 2048, 2069, 2074, 2074, 2084, 2084, 2088, 2088, 2112, 2136, 2144, 2154, 2160, 2183, 2185, 2190, 2208, 2249, 2308, 2361, 2365, 2365, 2384, 2384, 2392, 2401, 2417, 2432, 2437, 2444, 2447, 2448, 2451, 2472, 2474, 2480, 2482, 2482, 2486, 2489, 2493, 2493, 2510, 2510, 2524, 2525, 2527, 2529, 2544, 2545, 2556, 2556, 2565, 2570, 2575, 2576, 2579, 2600, 2602, 2608, 2610, 2611, 2613, 2614, 2616, 2617, 2649, 2652, 2654, 2654, 2674, 2676, 2693, 2701, 2703, 2705, 2707, 2728, 2730, 2736, 2738, 2739, 2741, 2745, 2749, 2749, 2768, 2768, 2784, 2785, 2809, 2809, 2821, 2828, 2831, 2832, 2835, 2856, 2858, 2864, 2866, 2867, 2869, 2873, 2877, 2877, 2908, 2909, 2911, 2913, 2929, 2929, 2947, 2947, 2949, 2954, 2958, 2960, 2962, 2965, 2969, 2970, 2972, 2972, 2974, 2975, 2979, 2980, 2984, 2986, 2990, 3001, 3024, 3024, 3077, 3084, 3086, 3088, 3090, 3112, 3114, 3129, 3133, 3133, 3160, 3162, 3165, 3165, 3168, 3169, 3200, 3200, 3205, 3212, 3214, 3216, 3218, 3240, 3242, 3251, 3253, 3257, 3261, 3261, 3293, 3294, 3296, 3297, 3313, 3314, 3332, 3340, 3342, 3344, 3346, 3386, 3389, 3389, 3406, 3406, 3412, 3414, 3423, 3425, 3450, 3455, 3461, 3478, 3482, 3505, 3507, 3515, 3517, 3517, 3520, 3526, 3585, 3632, 3634, 3635, 3648, 3654, 3713, 3714, 3716, 3716, 3718, 3722, 3724, 3747, 3749, 3749, 3751, 3760, 3762, 3763, 3773, 3773, 3776, 3780, 3782, 3782, 3804, 3807, 3840, 3840, 3904, 3911, 3913, 3948, 3976, 3980, 4096, 4138, 4159, 4159, 4176, 4181, 4186, 4189, 4193, 4193, 4197, 4198, 4206, 4208, 4213, 4225, 4238, 4238, 4256, 4293, 4295, 4295, 4301, 4301, 4304, 4346, 4348, 4680, 4682, 4685, 4688, 4694, 4696, 4696, 4698, 4701, 4704, 4744, 4746, 4749, 4752, 4784, 4786, 4789, 4792, 4798, 4800, 4800, 4802, 4805, 4808, 4822, 4824, 4880, 4882, 4885, 4888, 4954, 4992, 5007, 5024, 5109, 5112, 5117, 5121, 5740, 5743, 5759, 5761, 5786, 5792, 5866, 5870, 5880, 5888, 5905, 5919, 5937, 5952, 5969, 5984, 5996, 5998, 6e3, 6016, 6067, 6103, 6103, 6108, 6108, 6176, 6264, 6272, 6312, 6314, 6314, 6320, 6389, 6400, 6430, 6480, 6509, 6512, 6516, 6528, 6571, 6576, 6601, 6656, 6678, 6688, 6740, 6823, 6823, 6917, 6963, 6981, 6988, 7043, 7072, 7086, 7087, 7098, 7141, 7168, 7203, 7245, 7247, 7258, 7293, 7296, 7304, 7312, 7354, 7357, 7359, 7401, 7404, 7406, 7411, 7413, 7414, 7418, 7418, 7424, 7615, 7680, 7957, 7960, 7965, 7968, 8005, 8008, 8013, 8016, 8023, 8025, 8025, 8027, 8027, 8029, 8029, 8031, 8061, 8064, 8116, 8118, 8124, 8126, 8126, 8130, 8132, 8134, 8140, 8144, 8147, 8150, 8155, 8160, 8172, 8178, 8180, 8182, 8188, 8305, 8305, 8319, 8319, 8336, 8348, 8450, 8450, 8455, 8455, 8458, 8467, 8469, 8469, 8472, 8477, 8484, 8484, 8486, 8486, 8488, 8488, 8490, 8505, 8508, 8511, 8517, 8521, 8526, 8526, 8544, 8584, 11264, 11492, 11499, 11502, 11506, 11507, 11520, 11557, 11559, 11559, 11565, 11565, 11568, 11623, 11631, 11631, 11648, 11670, 11680, 11686, 11688, 11694, 11696, 11702, 11704, 11710, 11712, 11718, 11720, 11726, 11728, 11734, 11736, 11742, 12293, 12295, 12321, 12329, 12337, 12341, 12344, 12348, 12353, 12438, 12443, 12447, 12449, 12538, 12540, 12543, 12549, 12591, 12593, 12686, 12704, 12735, 12784, 12799, 13312, 19903, 19968, 42124, 42192, 42237, 42240, 42508, 42512, 42527, 42538, 42539, 42560, 42606, 42623, 42653, 42656, 42735, 42775, 42783, 42786, 42888, 42891, 42954, 42960, 42961, 42963, 42963, 42965, 42969, 42994, 43009, 43011, 43013, 43015, 43018, 43020, 43042, 43072, 43123, 43138, 43187, 43250, 43255, 43259, 43259, 43261, 43262, 43274, 43301, 43312, 43334, 43360, 43388, 43396, 43442, 43471, 43471, 43488, 43492, 43494, 43503, 43514, 43518, 43520, 43560, 43584, 43586, 43588, 43595, 43616, 43638, 43642, 43642, 43646, 43695, 43697, 43697, 43701, 43702, 43705, 43709, 43712, 43712, 43714, 43714, 43739, 43741, 43744, 43754, 43762, 43764, 43777, 43782, 43785, 43790, 43793, 43798, 43808, 43814, 43816, 43822, 43824, 43866, 43868, 43881, 43888, 44002, 44032, 55203, 55216, 55238, 55243, 55291, 63744, 64109, 64112, 64217, 64256, 64262, 64275, 64279, 64285, 64285, 64287, 64296, 64298, 64310, 64312, 64316, 64318, 64318, 64320, 64321, 64323, 64324, 64326, 64433, 64467, 64829, 64848, 64911, 64914, 64967, 65008, 65019, 65136, 65140, 65142, 65276, 65313, 65338, 65345, 65370, 65382, 65470, 65474, 65479, 65482, 65487, 65490, 65495, 65498, 65500, 65536, 65547, 65549, 65574, 65576, 65594, 65596, 65597, 65599, 65613, 65616, 65629, 65664, 65786, 65856, 65908, 66176, 66204, 66208, 66256, 66304, 66335, 66349, 66378, 66384, 66421, 66432, 66461, 66464, 66499, 66504, 66511, 66513, 66517, 66560, 66717, 66736, 66771, 66776, 66811, 66816, 66855, 66864, 66915, 66928, 66938, 66940, 66954, 66956, 66962, 66964, 66965, 66967, 66977, 66979, 66993, 66995, 67001, 67003, 67004, 67072, 67382, 67392, 67413, 67424, 67431, 67456, 67461, 67463, 67504, 67506, 67514, 67584, 67589, 67592, 67592, 67594, 67637, 67639, 67640, 67644, 67644, 67647, 67669, 67680, 67702, 67712, 67742, 67808, 67826, 67828, 67829, 67840, 67861, 67872, 67897, 67968, 68023, 68030, 68031, 68096, 68096, 68112, 68115, 68117, 68119, 68121, 68149, 68192, 68220, 68224, 68252, 68288, 68295, 68297, 68324, 68352, 68405, 68416, 68437, 68448, 68466, 68480, 68497, 68608, 68680, 68736, 68786, 68800, 68850, 68864, 68899, 69248, 69289, 69296, 69297, 69376, 69404, 69415, 69415, 69424, 69445, 69488, 69505, 69552, 69572, 69600, 69622, 69635, 69687, 69745, 69746, 69749, 69749, 69763, 69807, 69840, 69864, 69891, 69926, 69956, 69956, 69959, 69959, 69968, 70002, 70006, 70006, 70019, 70066, 70081, 70084, 70106, 70106, 70108, 70108, 70144, 70161, 70163, 70187, 70207, 70208, 70272, 70278, 70280, 70280, 70282, 70285, 70287, 70301, 70303, 70312, 70320, 70366, 70405, 70412, 70415, 70416, 70419, 70440, 70442, 70448, 70450, 70451, 70453, 70457, 70461, 70461, 70480, 70480, 70493, 70497, 70656, 70708, 70727, 70730, 70751, 70753, 70784, 70831, 70852, 70853, 70855, 70855, 71040, 71086, 71128, 71131, 71168, 71215, 71236, 71236, 71296, 71338, 71352, 71352, 71424, 71450, 71488, 71494, 71680, 71723, 71840, 71903, 71935, 71942, 71945, 71945, 71948, 71955, 71957, 71958, 71960, 71983, 71999, 71999, 72001, 72001, 72096, 72103, 72106, 72144, 72161, 72161, 72163, 72163, 72192, 72192, 72203, 72242, 72250, 72250, 72272, 72272, 72284, 72329, 72349, 72349, 72368, 72440, 72704, 72712, 72714, 72750, 72768, 72768, 72818, 72847, 72960, 72966, 72968, 72969, 72971, 73008, 73030, 73030, 73056, 73061, 73063, 73064, 73066, 73097, 73112, 73112, 73440, 73458, 73474, 73474, 73476, 73488, 73490, 73523, 73648, 73648, 73728, 74649, 74752, 74862, 74880, 75075, 77712, 77808, 77824, 78895, 78913, 78918, 82944, 83526, 92160, 92728, 92736, 92766, 92784, 92862, 92880, 92909, 92928, 92975, 92992, 92995, 93027, 93047, 93053, 93071, 93760, 93823, 93952, 94026, 94032, 94032, 94099, 94111, 94176, 94177, 94179, 94179, 94208, 100343, 100352, 101589, 101632, 101640, 110576, 110579, 110581, 110587, 110589, 110590, 110592, 110882, 110898, 110898, 110928, 110930, 110933, 110933, 110948, 110951, 110960, 111355, 113664, 113770, 113776, 113788, 113792, 113800, 113808, 113817, 119808, 119892, 119894, 119964, 119966, 119967, 119970, 119970, 119973, 119974, 119977, 119980, 119982, 119993, 119995, 119995, 119997, 120003, 120005, 120069, 120071, 120074, 120077, 120084, 120086, 120092, 120094, 120121, 120123, 120126, 120128, 120132, 120134, 120134, 120138, 120144, 120146, 120485, 120488, 120512, 120514, 120538, 120540, 120570, 120572, 120596, 120598, 120628, 120630, 120654, 120656, 120686, 120688, 120712, 120714, 120744, 120746, 120770, 120772, 120779, 122624, 122654, 122661, 122666, 122928, 122989, 123136, 123180, 123191, 123197, 123214, 123214, 123536, 123565, 123584, 123627, 124112, 124139, 124896, 124902, 124904, 124907, 124909, 124910, 124912, 124926, 124928, 125124, 125184, 125251, 125259, 125259, 126464, 126467, 126469, 126495, 126497, 126498, 126500, 126500, 126503, 126503, 126505, 126514, 126516, 126519, 126521, 126521, 126523, 126523, 126530, 126530, 126535, 126535, 126537, 126537, 126539, 126539, 126541, 126543, 126545, 126546, 126548, 126548, 126551, 126551, 126553, 126553, 126555, 126555, 126557, 126557, 126559, 126559, 126561, 126562, 126564, 126564, 126567, 126570, 126572, 126578, 126580, 126583, 126585, 126588, 126590, 126590, 126592, 126601, 126603, 126619, 126625, 126627, 126629, 126633, 126635, 126651, 131072, 173791, 173824, 177977, 177984, 178205, 178208, 183969, 183984, 191456, 191472, 192093, 194560, 195101, 196608, 201546, 201552, 205743];
var unicodeESNextIdentifierPart = [48, 57, 65, 90, 95, 95, 97, 122, 170, 170, 181, 181, 183, 183, 186, 186, 192, 214, 216, 246, 248, 705, 710, 721, 736, 740, 748, 748, 750, 750, 768, 884, 886, 887, 890, 893, 895, 895, 902, 906, 908, 908, 910, 929, 931, 1013, 1015, 1153, 1155, 1159, 1162, 1327, 1329, 1366, 1369, 1369, 1376, 1416, 1425, 1469, 1471, 1471, 1473, 1474, 1476, 1477, 1479, 1479, 1488, 1514, 1519, 1522, 1552, 1562, 1568, 1641, 1646, 1747, 1749, 1756, 1759, 1768, 1770, 1788, 1791, 1791, 1808, 1866, 1869, 1969, 1984, 2037, 2042, 2042, 2045, 2045, 2048, 2093, 2112, 2139, 2144, 2154, 2160, 2183, 2185, 2190, 2200, 2273, 2275, 2403, 2406, 2415, 2417, 2435, 2437, 2444, 2447, 2448, 2451, 2472, 2474, 2480, 2482, 2482, 2486, 2489, 2492, 2500, 2503, 2504, 2507, 2510, 2519, 2519, 2524, 2525, 2527, 2531, 2534, 2545, 2556, 2556, 2558, 2558, 2561, 2563, 2565, 2570, 2575, 2576, 2579, 2600, 2602, 2608, 2610, 2611, 2613, 2614, 2616, 2617, 2620, 2620, 2622, 2626, 2631, 2632, 2635, 2637, 2641, 2641, 2649, 2652, 2654, 2654, 2662, 2677, 2689, 2691, 2693, 2701, 2703, 2705, 2707, 2728, 2730, 2736, 2738, 2739, 2741, 2745, 2748, 2757, 2759, 2761, 2763, 2765, 2768, 2768, 2784, 2787, 2790, 2799, 2809, 2815, 2817, 2819, 2821, 2828, 2831, 2832, 2835, 2856, 2858, 2864, 2866, 2867, 2869, 2873, 2876, 2884, 2887, 2888, 2891, 2893, 2901, 2903, 2908, 2909, 2911, 2915, 2918, 2927, 2929, 2929, 2946, 2947, 2949, 2954, 2958, 2960, 2962, 2965, 2969, 2970, 2972, 2972, 2974, 2975, 2979, 2980, 2984, 2986, 2990, 3001, 3006, 3010, 3014, 3016, 3018, 3021, 3024, 3024, 3031, 3031, 3046, 3055, 3072, 3084, 3086, 3088, 3090, 3112, 3114, 3129, 3132, 3140, 3142, 3144, 3146, 3149, 3157, 3158, 3160, 3162, 3165, 3165, 3168, 3171, 3174, 3183, 3200, 3203, 3205, 3212, 3214, 3216, 3218, 3240, 3242, 3251, 3253, 3257, 3260, 3268, 3270, 3272, 3274, 3277, 3285, 3286, 3293, 3294, 3296, 3299, 3302, 3311, 3313, 3315, 3328, 3340, 3342, 3344, 3346, 3396, 3398, 3400, 3402, 3406, 3412, 3415, 3423, 3427, 3430, 3439, 3450, 3455, 3457, 3459, 3461, 3478, 3482, 3505, 3507, 3515, 3517, 3517, 3520, 3526, 3530, 3530, 3535, 3540, 3542, 3542, 3544, 3551, 3558, 3567, 3570, 3571, 3585, 3642, 3648, 3662, 3664, 3673, 3713, 3714, 3716, 3716, 3718, 3722, 3724, 3747, 3749, 3749, 3751, 3773, 3776, 3780, 3782, 3782, 3784, 3790, 3792, 3801, 3804, 3807, 3840, 3840, 3864, 3865, 3872, 3881, 3893, 3893, 3895, 3895, 3897, 3897, 3902, 3911, 3913, 3948, 3953, 3972, 3974, 3991, 3993, 4028, 4038, 4038, 4096, 4169, 4176, 4253, 4256, 4293, 4295, 4295, 4301, 4301, 4304, 4346, 4348, 4680, 4682, 4685, 4688, 4694, 4696, 4696, 4698, 4701, 4704, 4744, 4746, 4749, 4752, 4784, 4786, 4789, 4792, 4798, 4800, 4800, 4802, 4805, 4808, 4822, 4824, 4880, 4882, 4885, 4888, 4954, 4957, 4959, 4969, 4977, 4992, 5007, 5024, 5109, 5112, 5117, 5121, 5740, 5743, 5759, 5761, 5786, 5792, 5866, 5870, 5880, 5888, 5909, 5919, 5940, 5952, 5971, 5984, 5996, 5998, 6e3, 6002, 6003, 6016, 6099, 6103, 6103, 6108, 6109, 6112, 6121, 6155, 6157, 6159, 6169, 6176, 6264, 6272, 6314, 6320, 6389, 6400, 6430, 6432, 6443, 6448, 6459, 6470, 6509, 6512, 6516, 6528, 6571, 6576, 6601, 6608, 6618, 6656, 6683, 6688, 6750, 6752, 6780, 6783, 6793, 6800, 6809, 6823, 6823, 6832, 6845, 6847, 6862, 6912, 6988, 6992, 7001, 7019, 7027, 7040, 7155, 7168, 7223, 7232, 7241, 7245, 7293, 7296, 7304, 7312, 7354, 7357, 7359, 7376, 7378, 7380, 7418, 7424, 7957, 7960, 7965, 7968, 8005, 8008, 8013, 8016, 8023, 8025, 8025, 8027, 8027, 8029, 8029, 8031, 8061, 8064, 8116, 8118, 8124, 8126, 8126, 8130, 8132, 8134, 8140, 8144, 8147, 8150, 8155, 8160, 8172, 8178, 8180, 8182, 8188, 8204, 8205, 8255, 8256, 8276, 8276, 8305, 8305, 8319, 8319, 8336, 8348, 8400, 8412, 8417, 8417, 8421, 8432, 8450, 8450, 8455, 8455, 8458, 8467, 8469, 8469, 8472, 8477, 8484, 8484, 8486, 8486, 8488, 8488, 8490, 8505, 8508, 8511, 8517, 8521, 8526, 8526, 8544, 8584, 11264, 11492, 11499, 11507, 11520, 11557, 11559, 11559, 11565, 11565, 11568, 11623, 11631, 11631, 11647, 11670, 11680, 11686, 11688, 11694, 11696, 11702, 11704, 11710, 11712, 11718, 11720, 11726, 11728, 11734, 11736, 11742, 11744, 11775, 12293, 12295, 12321, 12335, 12337, 12341, 12344, 12348, 12353, 12438, 12441, 12447, 12449, 12543, 12549, 12591, 12593, 12686, 12704, 12735, 12784, 12799, 13312, 19903, 19968, 42124, 42192, 42237, 42240, 42508, 42512, 42539, 42560, 42607, 42612, 42621, 42623, 42737, 42775, 42783, 42786, 42888, 42891, 42954, 42960, 42961, 42963, 42963, 42965, 42969, 42994, 43047, 43052, 43052, 43072, 43123, 43136, 43205, 43216, 43225, 43232, 43255, 43259, 43259, 43261, 43309, 43312, 43347, 43360, 43388, 43392, 43456, 43471, 43481, 43488, 43518, 43520, 43574, 43584, 43597, 43600, 43609, 43616, 43638, 43642, 43714, 43739, 43741, 43744, 43759, 43762, 43766, 43777, 43782, 43785, 43790, 43793, 43798, 43808, 43814, 43816, 43822, 43824, 43866, 43868, 43881, 43888, 44010, 44012, 44013, 44016, 44025, 44032, 55203, 55216, 55238, 55243, 55291, 63744, 64109, 64112, 64217, 64256, 64262, 64275, 64279, 64285, 64296, 64298, 64310, 64312, 64316, 64318, 64318, 64320, 64321, 64323, 64324, 64326, 64433, 64467, 64829, 64848, 64911, 64914, 64967, 65008, 65019, 65024, 65039, 65056, 65071, 65075, 65076, 65101, 65103, 65136, 65140, 65142, 65276, 65296, 65305, 65313, 65338, 65343, 65343, 65345, 65370, 65381, 65470, 65474, 65479, 65482, 65487, 65490, 65495, 65498, 65500, 65536, 65547, 65549, 65574, 65576, 65594, 65596, 65597, 65599, 65613, 65616, 65629, 65664, 65786, 65856, 65908, 66045, 66045, 66176, 66204, 66208, 66256, 66272, 66272, 66304, 66335, 66349, 66378, 66384, 66426, 66432, 66461, 66464, 66499, 66504, 66511, 66513, 66517, 66560, 66717, 66720, 66729, 66736, 66771, 66776, 66811, 66816, 66855, 66864, 66915, 66928, 66938, 66940, 66954, 66956, 66962, 66964, 66965, 66967, 66977, 66979, 66993, 66995, 67001, 67003, 67004, 67072, 67382, 67392, 67413, 67424, 67431, 67456, 67461, 67463, 67504, 67506, 67514, 67584, 67589, 67592, 67592, 67594, 67637, 67639, 67640, 67644, 67644, 67647, 67669, 67680, 67702, 67712, 67742, 67808, 67826, 67828, 67829, 67840, 67861, 67872, 67897, 67968, 68023, 68030, 68031, 68096, 68099, 68101, 68102, 68108, 68115, 68117, 68119, 68121, 68149, 68152, 68154, 68159, 68159, 68192, 68220, 68224, 68252, 68288, 68295, 68297, 68326, 68352, 68405, 68416, 68437, 68448, 68466, 68480, 68497, 68608, 68680, 68736, 68786, 68800, 68850, 68864, 68903, 68912, 68921, 69248, 69289, 69291, 69292, 69296, 69297, 69373, 69404, 69415, 69415, 69424, 69456, 69488, 69509, 69552, 69572, 69600, 69622, 69632, 69702, 69734, 69749, 69759, 69818, 69826, 69826, 69840, 69864, 69872, 69881, 69888, 69940, 69942, 69951, 69956, 69959, 69968, 70003, 70006, 70006, 70016, 70084, 70089, 70092, 70094, 70106, 70108, 70108, 70144, 70161, 70163, 70199, 70206, 70209, 70272, 70278, 70280, 70280, 70282, 70285, 70287, 70301, 70303, 70312, 70320, 70378, 70384, 70393, 70400, 70403, 70405, 70412, 70415, 70416, 70419, 70440, 70442, 70448, 70450, 70451, 70453, 70457, 70459, 70468, 70471, 70472, 70475, 70477, 70480, 70480, 70487, 70487, 70493, 70499, 70502, 70508, 70512, 70516, 70656, 70730, 70736, 70745, 70750, 70753, 70784, 70853, 70855, 70855, 70864, 70873, 71040, 71093, 71096, 71104, 71128, 71133, 71168, 71232, 71236, 71236, 71248, 71257, 71296, 71352, 71360, 71369, 71424, 71450, 71453, 71467, 71472, 71481, 71488, 71494, 71680, 71738, 71840, 71913, 71935, 71942, 71945, 71945, 71948, 71955, 71957, 71958, 71960, 71989, 71991, 71992, 71995, 72003, 72016, 72025, 72096, 72103, 72106, 72151, 72154, 72161, 72163, 72164, 72192, 72254, 72263, 72263, 72272, 72345, 72349, 72349, 72368, 72440, 72704, 72712, 72714, 72758, 72760, 72768, 72784, 72793, 72818, 72847, 72850, 72871, 72873, 72886, 72960, 72966, 72968, 72969, 72971, 73014, 73018, 73018, 73020, 73021, 73023, 73031, 73040, 73049, 73056, 73061, 73063, 73064, 73066, 73102, 73104, 73105, 73107, 73112, 73120, 73129, 73440, 73462, 73472, 73488, 73490, 73530, 73534, 73538, 73552, 73561, 73648, 73648, 73728, 74649, 74752, 74862, 74880, 75075, 77712, 77808, 77824, 78895, 78912, 78933, 82944, 83526, 92160, 92728, 92736, 92766, 92768, 92777, 92784, 92862, 92864, 92873, 92880, 92909, 92912, 92916, 92928, 92982, 92992, 92995, 93008, 93017, 93027, 93047, 93053, 93071, 93760, 93823, 93952, 94026, 94031, 94087, 94095, 94111, 94176, 94177, 94179, 94180, 94192, 94193, 94208, 100343, 100352, 101589, 101632, 101640, 110576, 110579, 110581, 110587, 110589, 110590, 110592, 110882, 110898, 110898, 110928, 110930, 110933, 110933, 110948, 110951, 110960, 111355, 113664, 113770, 113776, 113788, 113792, 113800, 113808, 113817, 113821, 113822, 118528, 118573, 118576, 118598, 119141, 119145, 119149, 119154, 119163, 119170, 119173, 119179, 119210, 119213, 119362, 119364, 119808, 119892, 119894, 119964, 119966, 119967, 119970, 119970, 119973, 119974, 119977, 119980, 119982, 119993, 119995, 119995, 119997, 120003, 120005, 120069, 120071, 120074, 120077, 120084, 120086, 120092, 120094, 120121, 120123, 120126, 120128, 120132, 120134, 120134, 120138, 120144, 120146, 120485, 120488, 120512, 120514, 120538, 120540, 120570, 120572, 120596, 120598, 120628, 120630, 120654, 120656, 120686, 120688, 120712, 120714, 120744, 120746, 120770, 120772, 120779, 120782, 120831, 121344, 121398, 121403, 121452, 121461, 121461, 121476, 121476, 121499, 121503, 121505, 121519, 122624, 122654, 122661, 122666, 122880, 122886, 122888, 122904, 122907, 122913, 122915, 122916, 122918, 122922, 122928, 122989, 123023, 123023, 123136, 123180, 123184, 123197, 123200, 123209, 123214, 123214, 123536, 123566, 123584, 123641, 124112, 124153, 124896, 124902, 124904, 124907, 124909, 124910, 124912, 124926, 124928, 125124, 125136, 125142, 125184, 125259, 125264, 125273, 126464, 126467, 126469, 126495, 126497, 126498, 126500, 126500, 126503, 126503, 126505, 126514, 126516, 126519, 126521, 126521, 126523, 126523, 126530, 126530, 126535, 126535, 126537, 126537, 126539, 126539, 126541, 126543, 126545, 126546, 126548, 126548, 126551, 126551, 126553, 126553, 126555, 126555, 126557, 126557, 126559, 126559, 126561, 126562, 126564, 126564, 126567, 126570, 126572, 126578, 126580, 126583, 126585, 126588, 126590, 126590, 126592, 126601, 126603, 126619, 126625, 126627, 126629, 126633, 126635, 126651, 130032, 130041, 131072, 173791, 173824, 177977, 177984, 178205, 178208, 183969, 183984, 191456, 191472, 192093, 194560, 195101, 196608, 201546, 201552, 205743, 917760, 917999];
var commentDirectiveRegExSingleLine = /^\/\/\/?\s*@(ts-expect-error|ts-ignore)/;
var commentDirectiveRegExMultiLine = /^(?:\/|\*)*\s*@(ts-expect-error|ts-ignore)/;
var jsDocSeeOrLink = /@(?:see|link)/i;
function lookupInUnicodeMap(code, map2) {
  if (code < map2[0]) {
    return false;
  }
  let lo = 0;
  let hi = map2.length;
  let mid;
  while (lo + 1 < hi) {
    mid = lo + (hi - lo) / 2;
    mid -= mid % 2;
    if (map2[mid] <= code && code <= map2[mid + 1]) {
      return true;
    }
    if (code < map2[mid]) {
      hi = mid;
    } else {
      lo = mid + 2;
    }
  }
  return false;
}
function isUnicodeIdentifierStart(code, languageVersion) {
  return languageVersion >= 2 /* ES2015 */ ? lookupInUnicodeMap(code, unicodeESNextIdentifierStart) : lookupInUnicodeMap(code, unicodeES5IdentifierStart);
}
function isUnicodeIdentifierPart(code, languageVersion) {
  return languageVersion >= 2 /* ES2015 */ ? lookupInUnicodeMap(code, unicodeESNextIdentifierPart) : lookupInUnicodeMap(code, unicodeES5IdentifierPart);
}
function makeReverseMap(source) {
  const result = [];
  source.forEach((value, name) => {
    result[value] = name;
  });
  return result;
}
var tokenStrings = makeReverseMap(textToToken);
function tokenToString(t) {
  return tokenStrings[t];
}
function stringToToken(s) {
  return textToToken.get(s);
}
var regExpFlagCharCodes = makeReverseMap(charCodeToRegExpFlag);
function characterCodeToRegularExpressionFlag(ch) {
  return charCodeToRegExpFlag.get(ch);
}
function computeLineStarts(text) {
  const result = [];
  let pos = 0;
  let lineStart = 0;
  while (pos < text.length) {
    const ch = text.charCodeAt(pos);
    pos++;
    switch (ch) {
      case 13 /* carriageReturn */:
        if (text.charCodeAt(pos) === 10 /* lineFeed */) {
          pos++;
        }
      // falls through
      case 10 /* lineFeed */:
        result.push(lineStart);
        lineStart = pos;
        break;
      default:
        if (ch > 127 /* maxAsciiCharacter */ && isLineBreak(ch)) {
          result.push(lineStart);
          lineStart = pos;
        }
        break;
    }
  }
  result.push(lineStart);
  return result;
}
function getPositionOfLineAndCharacter(sourceFile, line, character, allowEdits) {
  return sourceFile.getPositionOfLineAndCharacter ? sourceFile.getPositionOfLineAndCharacter(line, character, allowEdits) : computePositionOfLineAndCharacter(getLineStarts(sourceFile), line, character, sourceFile.text, allowEdits);
}
function computePositionOfLineAndCharacter(lineStarts, line, character, debugText, allowEdits) {
  if (line < 0 || line >= lineStarts.length) {
    if (allowEdits) {
      line = line < 0 ? 0 : line >= lineStarts.length ? lineStarts.length - 1 : line;
    } else {
      Debug.fail(`Bad line number. Line: ${line}, lineStarts.length: ${lineStarts.length} , line map is correct? ${debugText !== void 0 ? arrayIsEqualTo(lineStarts, computeLineStarts(debugText)) : "unknown"}`);
    }
  }
  const res = lineStarts[line] + character;
  if (allowEdits) {
    return res > lineStarts[line + 1] ? lineStarts[line + 1] : typeof debugText === "string" && res > debugText.length ? debugText.length : res;
  }
  if (line < lineStarts.length - 1) {
    Debug.assert(res < lineStarts[line + 1]);
  } else if (debugText !== void 0) {
    Debug.assert(res <= debugText.length);
  }
  return res;
}
function getLineStarts(sourceFile) {
  return sourceFile.lineMap || (sourceFile.lineMap = computeLineStarts(sourceFile.text));
}
function computeLineAndCharacterOfPosition(lineStarts, position) {
  const lineNumber = computeLineOfPosition(lineStarts, position);
  return {
    line: lineNumber,
    character: position - lineStarts[lineNumber]
  };
}
function computeLineOfPosition(lineStarts, position, lowerBound) {
  let lineNumber = binarySearch(lineStarts, position, identity, compareValues, lowerBound);
  if (lineNumber < 0) {
    lineNumber = ~lineNumber - 1;
    Debug.assert(lineNumber !== -1, "position cannot precede the beginning of the file");
  }
  return lineNumber;
}
function getLinesBetweenPositions(sourceFile, pos1, pos2) {
  if (pos1 === pos2) return 0;
  const lineStarts = getLineStarts(sourceFile);
  const lower = Math.min(pos1, pos2);
  const isNegative = lower === pos2;
  const upper = isNegative ? pos1 : pos2;
  const lowerLine = computeLineOfPosition(lineStarts, lower);
  const upperLine = computeLineOfPosition(lineStarts, upper, lowerLine);
  return isNegative ? lowerLine - upperLine : upperLine - lowerLine;
}
function getLineAndCharacterOfPosition(sourceFile, position) {
  return computeLineAndCharacterOfPosition(getLineStarts(sourceFile), position);
}
function isWhiteSpaceLike(ch) {
  return isWhiteSpaceSingleLine(ch) || isLineBreak(ch);
}
function isWhiteSpaceSingleLine(ch) {
  return ch === 32 /* space */ || ch === 9 /* tab */ || ch === 11 /* verticalTab */ || ch === 12 /* formFeed */ || ch === 160 /* nonBreakingSpace */ || ch === 133 /* nextLine */ || ch === 5760 /* ogham */ || ch >= 8192 /* enQuad */ && ch <= 8203 /* zeroWidthSpace */ || ch === 8239 /* narrowNoBreakSpace */ || ch === 8287 /* mathematicalSpace */ || ch === 12288 /* ideographicSpace */ || ch === 65279 /* byteOrderMark */;
}
function isLineBreak(ch) {
  return ch === 10 /* lineFeed */ || ch === 13 /* carriageReturn */ || ch === 8232 /* lineSeparator */ || ch === 8233 /* paragraphSeparator */;
}
function isDigit(ch) {
  return ch >= 48 /* _0 */ && ch <= 57 /* _9 */;
}
function isHexDigit(ch) {
  return isDigit(ch) || ch >= 65 /* A */ && ch <= 70 /* F */ || ch >= 97 /* a */ && ch <= 102 /* f */;
}
function isASCIILetter(ch) {
  return ch >= 65 /* A */ && ch <= 90 /* Z */ || ch >= 97 /* a */ && ch <= 122 /* z */;
}
function isWordCharacter(ch) {
  return isASCIILetter(ch) || isDigit(ch) || ch === 95 /* _ */;
}
function isOctalDigit(ch) {
  return ch >= 48 /* _0 */ && ch <= 55 /* _7 */;
}
function skipTrivia(text, pos, stopAfterLineBreak, stopAtComments, inJSDoc) {
  if (positionIsSynthesized(pos)) {
    return pos;
  }
  let canConsumeStar = false;
  while (true) {
    const ch = text.charCodeAt(pos);
    switch (ch) {
      case 13 /* carriageReturn */:
        if (text.charCodeAt(pos + 1) === 10 /* lineFeed */) {
          pos++;
        }
      // falls through
      case 10 /* lineFeed */:
        pos++;
        if (stopAfterLineBreak) {
          return pos;
        }
        canConsumeStar = !!inJSDoc;
        continue;
      case 9 /* tab */:
      case 11 /* verticalTab */:
      case 12 /* formFeed */:
      case 32 /* space */:
        pos++;
        continue;
      case 47 /* slash */:
        if (stopAtComments) {
          break;
        }
        if (text.charCodeAt(pos + 1) === 47 /* slash */) {
          pos += 2;
          while (pos < text.length) {
            if (isLineBreak(text.charCodeAt(pos))) {
              break;
            }
            pos++;
          }
          canConsumeStar = false;
          continue;
        }
        if (text.charCodeAt(pos + 1) === 42 /* asterisk */) {
          pos += 2;
          while (pos < text.length) {
            if (text.charCodeAt(pos) === 42 /* asterisk */ && text.charCodeAt(pos + 1) === 47 /* slash */) {
              pos += 2;
              break;
            }
            pos++;
          }
          canConsumeStar = false;
          continue;
        }
        break;
      case 60 /* lessThan */:
      case 124 /* bar */:
      case 61 /* equals */:
      case 62 /* greaterThan */:
        if (isConflictMarkerTrivia(text, pos)) {
          pos = scanConflictMarkerTrivia(text, pos);
          canConsumeStar = false;
          continue;
        }
        break;
      case 35 /* hash */:
        if (pos === 0 && isShebangTrivia(text, pos)) {
          pos = scanShebangTrivia(text, pos);
          canConsumeStar = false;
          continue;
        }
        break;
      case 42 /* asterisk */:
        if (canConsumeStar) {
          pos++;
          canConsumeStar = false;
          continue;
        }
        break;
      default:
        if (ch > 127 /* maxAsciiCharacter */ && isWhiteSpaceLike(ch)) {
          pos++;
          continue;
        }
        break;
    }
    return pos;
  }
}
var mergeConflictMarkerLength = "<<<<<<<".length;
function isConflictMarkerTrivia(text, pos) {
  Debug.assert(pos >= 0);
  if (pos === 0 || isLineBreak(text.charCodeAt(pos - 1))) {
    const ch = text.charCodeAt(pos);
    if (pos + mergeConflictMarkerLength < text.length) {
      for (let i = 0; i < mergeConflictMarkerLength; i++) {
        if (text.charCodeAt(pos + i) !== ch) {
          return false;
        }
      }
      return ch === 61 /* equals */ || text.charCodeAt(pos + mergeConflictMarkerLength) === 32 /* space */;
    }
  }
  return false;
}
function scanConflictMarkerTrivia(text, pos, error) {
  if (error) {
    error(Diagnostics.Merge_conflict_marker_encountered, pos, mergeConflictMarkerLength);
  }
  const ch = text.charCodeAt(pos);
  const len = text.length;
  if (ch === 60 /* lessThan */ || ch === 62 /* greaterThan */) {
    while (pos < len && !isLineBreak(text.charCodeAt(pos))) {
      pos++;
    }
  } else {
    Debug.assert(ch === 124 /* bar */ || ch === 61 /* equals */);
    while (pos < len) {
      const currentChar = text.charCodeAt(pos);
      if ((currentChar === 61 /* equals */ || currentChar === 62 /* greaterThan */) && currentChar !== ch && isConflictMarkerTrivia(text, pos)) {
        break;
      }
      pos++;
    }
  }
  return pos;
}
var shebangTriviaRegex = /^#!.*/;
function isShebangTrivia(text, pos) {
  Debug.assert(pos === 0);
  return shebangTriviaRegex.test(text);
}
function scanShebangTrivia(text, pos) {
  const shebang = shebangTriviaRegex.exec(text)[0];
  pos = pos + shebang.length;
  return pos;
}
function iterateCommentRanges(reduce, text, pos, trailing, cb, state, initial) {
  let pendingPos;
  let pendingEnd;
  let pendingKind;
  let pendingHasTrailingNewLine;
  let hasPendingCommentRange = false;
  let collecting = trailing;
  let accumulator = initial;
  if (pos === 0) {
    collecting = true;
    const shebang = getShebang(text);
    if (shebang) {
      pos = shebang.length;
    }
  }
  scan:
    while (pos >= 0 && pos < text.length) {
      const ch = text.charCodeAt(pos);
      switch (ch) {
        case 13 /* carriageReturn */:
          if (text.charCodeAt(pos + 1) === 10 /* lineFeed */) {
            pos++;
          }
        // falls through
        case 10 /* lineFeed */:
          pos++;
          if (trailing) {
            break scan;
          }
          collecting = true;
          if (hasPendingCommentRange) {
            pendingHasTrailingNewLine = true;
          }
          continue;
        case 9 /* tab */:
        case 11 /* verticalTab */:
        case 12 /* formFeed */:
        case 32 /* space */:
          pos++;
          continue;
        case 47 /* slash */:
          const nextChar = text.charCodeAt(pos + 1);
          let hasTrailingNewLine = false;
          if (nextChar === 47 /* slash */ || nextChar === 42 /* asterisk */) {
            const kind = nextChar === 47 /* slash */ ? 2 /* SingleLineCommentTrivia */ : 3 /* MultiLineCommentTrivia */;
            const startPos = pos;
            pos += 2;
            if (nextChar === 47 /* slash */) {
              while (pos < text.length) {
                if (isLineBreak(text.charCodeAt(pos))) {
                  hasTrailingNewLine = true;
                  break;
                }
                pos++;
              }
            } else {
              while (pos < text.length) {
                if (text.charCodeAt(pos) === 42 /* asterisk */ && text.charCodeAt(pos + 1) === 47 /* slash */) {
                  pos += 2;
                  break;
                }
                pos++;
              }
            }
            if (collecting) {
              if (hasPendingCommentRange) {
                accumulator = cb(pendingPos, pendingEnd, pendingKind, pendingHasTrailingNewLine, state, accumulator);
                if (!reduce && accumulator) {
                  return accumulator;
                }
              }
              pendingPos = startPos;
              pendingEnd = pos;
              pendingKind = kind;
              pendingHasTrailingNewLine = hasTrailingNewLine;
              hasPendingCommentRange = true;
            }
            continue;
          }
          break scan;
        default:
          if (ch > 127 /* maxAsciiCharacter */ && isWhiteSpaceLike(ch)) {
            if (hasPendingCommentRange && isLineBreak(ch)) {
              pendingHasTrailingNewLine = true;
            }
            pos++;
            continue;
          }
          break scan;
      }
    }
  if (hasPendingCommentRange) {
    accumulator = cb(pendingPos, pendingEnd, pendingKind, pendingHasTrailingNewLine, state, accumulator);
  }
  return accumulator;
}
function forEachLeadingCommentRange(text, pos, cb, state) {
  return iterateCommentRanges(
    /*reduce*/
    false,
    text,
    pos,
    /*trailing*/
    false,
    cb,
    state
  );
}
function forEachTrailingCommentRange(text, pos, cb, state) {
  return iterateCommentRanges(
    /*reduce*/
    false,
    text,
    pos,
    /*trailing*/
    true,
    cb,
    state
  );
}
function reduceEachLeadingCommentRange(text, pos, cb, state, initial) {
  return iterateCommentRanges(
    /*reduce*/
    true,
    text,
    pos,
    /*trailing*/
    false,
    cb,
    state,
    initial
  );
}
function reduceEachTrailingCommentRange(text, pos, cb, state, initial) {
  return iterateCommentRanges(
    /*reduce*/
    true,
    text,
    pos,
    /*trailing*/
    true,
    cb,
    state,
    initial
  );
}
function appendCommentRange(pos, end, kind, hasTrailingNewLine, _state, comments = []) {
  comments.push({ kind, pos, end, hasTrailingNewLine });
  return comments;
}
function getLeadingCommentRanges(text, pos) {
  return reduceEachLeadingCommentRange(
    text,
    pos,
    appendCommentRange,
    /*state*/
    void 0,
    /*initial*/
    void 0
  );
}
function getTrailingCommentRanges(text, pos) {
  return reduceEachTrailingCommentRange(
    text,
    pos,
    appendCommentRange,
    /*state*/
    void 0,
    /*initial*/
    void 0
  );
}
function getShebang(text) {
  const match = shebangTriviaRegex.exec(text);
  if (match) {
    return match[0];
  }
}
function isIdentifierStart(ch, languageVersion) {
  return isASCIILetter(ch) || ch === 36 /* $ */ || ch === 95 /* _ */ || ch > 127 /* maxAsciiCharacter */ && isUnicodeIdentifierStart(ch, languageVersion);
}
function isIdentifierPart(ch, languageVersion, identifierVariant) {
  return isWordCharacter(ch) || ch === 36 /* $ */ || // "-" and ":" are valid in JSX Identifiers
  (identifierVariant === 1 /* JSX */ ? ch === 45 /* minus */ || ch === 58 /* colon */ : false) || ch > 127 /* maxAsciiCharacter */ && isUnicodeIdentifierPart(ch, languageVersion);
}
function isIdentifierText(name, languageVersion, identifierVariant) {
  let ch = codePointAt(name, 0);
  if (!isIdentifierStart(ch, languageVersion)) {
    return false;
  }
  for (let i = charSize(ch); i < name.length; i += charSize(ch)) {
    if (!isIdentifierPart(ch = codePointAt(name, i), languageVersion, identifierVariant)) {
      return false;
    }
  }
  return true;
}
function createScanner(languageVersion, skipTrivia2, languageVariant = 0 /* Standard */, textInitial, onError, start, length2) {
  var text = textInitial;
  var pos;
  var end;
  var fullStartPos;
  var tokenStart;
  var token;
  var tokenValue;
  var tokenFlags;
  var commentDirectives;
  var skipJsDocLeadingAsterisks = 0;
  var scriptKind = 0 /* Unknown */;
  var jsDocParsingMode = 0 /* ParseAll */;
  setText(text, start, length2);
  var scanner = {
    getTokenFullStart: () => fullStartPos,
    getStartPos: () => fullStartPos,
    getTokenEnd: () => pos,
    getTextPos: () => pos,
    getToken: () => token,
    getTokenStart: () => tokenStart,
    getTokenPos: () => tokenStart,
    getTokenText: () => text.substring(tokenStart, pos),
    getTokenValue: () => tokenValue,
    hasUnicodeEscape: () => (tokenFlags & 1024 /* UnicodeEscape */) !== 0,
    hasExtendedUnicodeEscape: () => (tokenFlags & 8 /* ExtendedUnicodeEscape */) !== 0,
    hasPrecedingLineBreak: () => (tokenFlags & 1 /* PrecedingLineBreak */) !== 0,
    hasPrecedingJSDocComment: () => (tokenFlags & 2 /* PrecedingJSDocComment */) !== 0,
    hasPrecedingJSDocLeadingAsterisks: () => (tokenFlags & 32768 /* PrecedingJSDocLeadingAsterisks */) !== 0,
    isIdentifier: () => token === 80 /* Identifier */ || token > 118 /* LastReservedWord */,
    isReservedWord: () => token >= 83 /* FirstReservedWord */ && token <= 118 /* LastReservedWord */,
    isUnterminated: () => (tokenFlags & 4 /* Unterminated */) !== 0,
    getCommentDirectives: () => commentDirectives,
    getNumericLiteralFlags: () => tokenFlags & 25584 /* NumericLiteralFlags */,
    getTokenFlags: () => tokenFlags,
    reScanGreaterToken,
    reScanAsteriskEqualsToken,
    reScanSlashToken,
    reScanTemplateToken,
    reScanTemplateHeadOrNoSubstitutionTemplate,
    scanJsxIdentifier,
    scanJsxAttributeValue,
    reScanJsxAttributeValue,
    reScanJsxToken,
    reScanLessThanToken,
    reScanHashToken,
    reScanQuestionToken,
    reScanInvalidIdentifier,
    scanJsxToken,
    scanJsDocToken,
    scanJSDocCommentTextToken,
    scan,
    getText,
    clearCommentDirectives,
    setText,
    setScriptTarget,
    setLanguageVariant,
    setScriptKind,
    setJSDocParsingMode,
    setOnError,
    resetTokenState,
    setTextPos: resetTokenState,
    setSkipJsDocLeadingAsterisks,
    tryScan,
    lookAhead,
    scanRange
  };
  if (Debug.isDebugging) {
    Object.defineProperty(scanner, "__debugShowCurrentPositionInText", {
      get: () => {
        const text2 = scanner.getText();
        return text2.slice(0, scanner.getTokenFullStart()) + "\u2551" + text2.slice(scanner.getTokenFullStart());
      }
    });
  }
  return scanner;
  function codePointUnchecked(pos2) {
    return codePointAt(text, pos2);
  }
  function codePointChecked(pos2) {
    return pos2 >= 0 && pos2 < end ? codePointUnchecked(pos2) : -1 /* EOF */;
  }
  function charCodeUnchecked(pos2) {
    return text.charCodeAt(pos2);
  }
  function charCodeChecked(pos2) {
    return pos2 >= 0 && pos2 < end ? charCodeUnchecked(pos2) : -1 /* EOF */;
  }
  function error(message, errPos = pos, length3, arg0) {
    if (onError) {
      const oldPos = pos;
      pos = errPos;
      onError(message, length3 || 0, arg0);
      pos = oldPos;
    }
  }
  function scanNumberFragment() {
    let start2 = pos;
    let allowSeparator = false;
    let isPreviousTokenSeparator = false;
    let result = "";
    while (true) {
      const ch = charCodeUnchecked(pos);
      if (ch === 95 /* _ */) {
        tokenFlags |= 512 /* ContainsSeparator */;
        if (allowSeparator) {
          allowSeparator = false;
          isPreviousTokenSeparator = true;
          result += text.substring(start2, pos);
        } else {
          tokenFlags |= 16384 /* ContainsInvalidSeparator */;
          if (isPreviousTokenSeparator) {
            error(Diagnostics.Multiple_consecutive_numeric_separators_are_not_permitted, pos, 1);
          } else {
            error(Diagnostics.Numeric_separators_are_not_allowed_here, pos, 1);
          }
        }
        pos++;
        start2 = pos;
        continue;
      }
      if (isDigit(ch)) {
        allowSeparator = true;
        isPreviousTokenSeparator = false;
        pos++;
        continue;
      }
      break;
    }
    if (charCodeUnchecked(pos - 1) === 95 /* _ */) {
      tokenFlags |= 16384 /* ContainsInvalidSeparator */;
      error(Diagnostics.Numeric_separators_are_not_allowed_here, pos - 1, 1);
    }
    return result + text.substring(start2, pos);
  }
  function scanNumber() {
    let start2 = pos;
    let mainFragment;
    if (charCodeUnchecked(pos) === 48 /* _0 */) {
      pos++;
      if (charCodeUnchecked(pos) === 95 /* _ */) {
        tokenFlags |= 512 /* ContainsSeparator */ | 16384 /* ContainsInvalidSeparator */;
        error(Diagnostics.Numeric_separators_are_not_allowed_here, pos, 1);
        pos--;
        mainFragment = scanNumberFragment();
      } else if (!scanDigits()) {
        tokenFlags |= 8192 /* ContainsLeadingZero */;
        mainFragment = "" + +tokenValue;
      } else if (!tokenValue) {
        mainFragment = "0";
      } else {
        tokenValue = "" + parseInt(tokenValue, 8);
        tokenFlags |= 32 /* Octal */;
        const withMinus = token === 41 /* MinusToken */;
        const literal = (withMinus ? "-" : "") + "0o" + (+tokenValue).toString(8);
        if (withMinus) start2--;
        error(Diagnostics.Octal_literals_are_not_allowed_Use_the_syntax_0, start2, pos - start2, literal);
        return 9 /* NumericLiteral */;
      }
    } else {
      mainFragment = scanNumberFragment();
    }
    let decimalFragment;
    let scientificFragment;
    if (charCodeUnchecked(pos) === 46 /* dot */) {
      pos++;
      decimalFragment = scanNumberFragment();
    }
    let end2 = pos;
    if (charCodeUnchecked(pos) === 69 /* E */ || charCodeUnchecked(pos) === 101 /* e */) {
      pos++;
      tokenFlags |= 16 /* Scientific */;
      if (charCodeUnchecked(pos) === 43 /* plus */ || charCodeUnchecked(pos) === 45 /* minus */) pos++;
      const preNumericPart = pos;
      const finalFragment = scanNumberFragment();
      if (!finalFragment) {
        error(Diagnostics.Digit_expected);
      } else {
        scientificFragment = text.substring(end2, preNumericPart) + finalFragment;
        end2 = pos;
      }
    }
    let result;
    if (tokenFlags & 512 /* ContainsSeparator */) {
      result = mainFragment;
      if (decimalFragment) {
        result += "." + decimalFragment;
      }
      if (scientificFragment) {
        result += scientificFragment;
      }
    } else {
      result = text.substring(start2, end2);
    }
    if (tokenFlags & 8192 /* ContainsLeadingZero */) {
      error(Diagnostics.Decimals_with_leading_zeros_are_not_allowed, start2, end2 - start2);
      tokenValue = "" + +result;
      return 9 /* NumericLiteral */;
    }
    if (decimalFragment !== void 0 || tokenFlags & 16 /* Scientific */) {
      checkForIdentifierStartAfterNumericLiteral(start2, decimalFragment === void 0 && !!(tokenFlags & 16 /* Scientific */));
      tokenValue = "" + +result;
      return 9 /* NumericLiteral */;
    } else {
      tokenValue = result;
      const type = checkBigIntSuffix();
      checkForIdentifierStartAfterNumericLiteral(start2);
      return type;
    }
  }
  function checkForIdentifierStartAfterNumericLiteral(numericStart, isScientific) {
    if (!isIdentifierStart(codePointUnchecked(pos), languageVersion)) {
      return;
    }
    const identifierStart = pos;
    const { length: length3 } = scanIdentifierParts();
    if (length3 === 1 && text[identifierStart] === "n") {
      if (isScientific) {
        error(Diagnostics.A_bigint_literal_cannot_use_exponential_notation, numericStart, identifierStart - numericStart + 1);
      } else {
        error(Diagnostics.A_bigint_literal_must_be_an_integer, numericStart, identifierStart - numericStart + 1);
      }
    } else {
      error(Diagnostics.An_identifier_or_keyword_cannot_immediately_follow_a_numeric_literal, identifierStart, length3);
      pos = identifierStart;
    }
  }
  function scanDigits() {
    const start2 = pos;
    let isOctal = true;
    while (isDigit(charCodeChecked(pos))) {
      if (!isOctalDigit(charCodeUnchecked(pos))) {
        isOctal = false;
      }
      pos++;
    }
    tokenValue = text.substring(start2, pos);
    return isOctal;
  }
  function scanExactNumberOfHexDigits(count, canHaveSeparators) {
    const valueString = scanHexDigits(
      /*minCount*/
      count,
      /*scanAsManyAsPossible*/
      false,
      canHaveSeparators
    );
    return valueString ? parseInt(valueString, 16) : -1;
  }
  function scanMinimumNumberOfHexDigits(count, canHaveSeparators) {
    return scanHexDigits(
      /*minCount*/
      count,
      /*scanAsManyAsPossible*/
      true,
      canHaveSeparators
    );
  }
  function scanHexDigits(minCount, scanAsManyAsPossible, canHaveSeparators) {
    let valueChars = [];
    let allowSeparator = false;
    let isPreviousTokenSeparator = false;
    while (valueChars.length < minCount || scanAsManyAsPossible) {
      let ch = charCodeUnchecked(pos);
      if (canHaveSeparators && ch === 95 /* _ */) {
        tokenFlags |= 512 /* ContainsSeparator */;
        if (allowSeparator) {
          allowSeparator = false;
          isPreviousTokenSeparator = true;
        } else if (isPreviousTokenSeparator) {
          error(Diagnostics.Multiple_consecutive_numeric_separators_are_not_permitted, pos, 1);
        } else {
          error(Diagnostics.Numeric_separators_are_not_allowed_here, pos, 1);
        }
        pos++;
        continue;
      }
      allowSeparator = canHaveSeparators;
      if (ch >= 65 /* A */ && ch <= 70 /* F */) {
        ch += 97 /* a */ - 65 /* A */;
      } else if (!(ch >= 48 /* _0 */ && ch <= 57 /* _9 */ || ch >= 97 /* a */ && ch <= 102 /* f */)) {
        break;
      }
      valueChars.push(ch);
      pos++;
      isPreviousTokenSeparator = false;
    }
    if (valueChars.length < minCount) {
      valueChars = [];
    }
    if (charCodeUnchecked(pos - 1) === 95 /* _ */) {
      error(Diagnostics.Numeric_separators_are_not_allowed_here, pos - 1, 1);
    }
    return String.fromCharCode(...valueChars);
  }
  function scanString(jsxAttributeString = false) {
    const quote = charCodeUnchecked(pos);
    pos++;
    let result = "";
    let start2 = pos;
    while (true) {
      if (pos >= end) {
        result += text.substring(start2, pos);
        tokenFlags |= 4 /* Unterminated */;
        error(Diagnostics.Unterminated_string_literal);
        break;
      }
      const ch = charCodeUnchecked(pos);
      if (ch === quote) {
        result += text.substring(start2, pos);
        pos++;
        break;
      }
      if (ch === 92 /* backslash */ && !jsxAttributeString) {
        result += text.substring(start2, pos);
        result += scanEscapeSequence(1 /* String */ | 2 /* ReportErrors */);
        start2 = pos;
        continue;
      }
      if ((ch === 10 /* lineFeed */ || ch === 13 /* carriageReturn */) && !jsxAttributeString) {
        result += text.substring(start2, pos);
        tokenFlags |= 4 /* Unterminated */;
        error(Diagnostics.Unterminated_string_literal);
        break;
      }
      pos++;
    }
    return result;
  }
  function scanTemplateAndSetTokenValue(shouldEmitInvalidEscapeError) {
    const startedWithBacktick = charCodeUnchecked(pos) === 96 /* backtick */;
    pos++;
    let start2 = pos;
    let contents = "";
    let resultingToken;
    while (true) {
      if (pos >= end) {
        contents += text.substring(start2, pos);
        tokenFlags |= 4 /* Unterminated */;
        error(Diagnostics.Unterminated_template_literal);
        resultingToken = startedWithBacktick ? 15 /* NoSubstitutionTemplateLiteral */ : 18 /* TemplateTail */;
        break;
      }
      const currChar = charCodeUnchecked(pos);
      if (currChar === 96 /* backtick */) {
        contents += text.substring(start2, pos);
        pos++;
        resultingToken = startedWithBacktick ? 15 /* NoSubstitutionTemplateLiteral */ : 18 /* TemplateTail */;
        break;
      }
      if (currChar === 36 /* $ */ && pos + 1 < end && charCodeUnchecked(pos + 1) === 123 /* openBrace */) {
        contents += text.substring(start2, pos);
        pos += 2;
        resultingToken = startedWithBacktick ? 16 /* TemplateHead */ : 17 /* TemplateMiddle */;
        break;
      }
      if (currChar === 92 /* backslash */) {
        contents += text.substring(start2, pos);
        contents += scanEscapeSequence(1 /* String */ | (shouldEmitInvalidEscapeError ? 2 /* ReportErrors */ : 0));
        start2 = pos;
        continue;
      }
      if (currChar === 13 /* carriageReturn */) {
        contents += text.substring(start2, pos);
        pos++;
        if (pos < end && charCodeUnchecked(pos) === 10 /* lineFeed */) {
          pos++;
        }
        contents += "\n";
        start2 = pos;
        continue;
      }
      pos++;
    }
    Debug.assert(resultingToken !== void 0);
    tokenValue = contents;
    return resultingToken;
  }
  function scanEscapeSequence(flags) {
    const start2 = pos;
    pos++;
    if (pos >= end) {
      error(Diagnostics.Unexpected_end_of_text);
      return "";
    }
    const ch = charCodeUnchecked(pos);
    pos++;
    switch (ch) {
      case 48 /* _0 */:
        if (pos >= end || !isDigit(charCodeUnchecked(pos))) {
          return "\0";
        }
      // '\01', '\011'
      // falls through
      case 49 /* _1 */:
      case 50 /* _2 */:
      case 51 /* _3 */:
        if (pos < end && isOctalDigit(charCodeUnchecked(pos))) {
          pos++;
        }
      // '\17', '\177'
      // falls through
      case 52 /* _4 */:
      case 53 /* _5 */:
      case 54 /* _6 */:
      case 55 /* _7 */:
        if (pos < end && isOctalDigit(charCodeUnchecked(pos))) {
          pos++;
        }
        tokenFlags |= 2048 /* ContainsInvalidEscape */;
        if (flags & 6 /* ReportInvalidEscapeErrors */) {
          const code = parseInt(text.substring(start2 + 1, pos), 8);
          if (flags & 4 /* RegularExpression */ && !(flags & 32 /* AtomEscape */) && ch !== 48 /* _0 */) {
            error(Diagnostics.Octal_escape_sequences_and_backreferences_are_not_allowed_in_a_character_class_If_this_was_intended_as_an_escape_sequence_use_the_syntax_0_instead, start2, pos - start2, "\\x" + code.toString(16).padStart(2, "0"));
          } else {
            error(Diagnostics.Octal_escape_sequences_are_not_allowed_Use_the_syntax_0, start2, pos - start2, "\\x" + code.toString(16).padStart(2, "0"));
          }
          return String.fromCharCode(code);
        }
        return text.substring(start2, pos);
      case 56 /* _8 */:
      case 57 /* _9 */:
        tokenFlags |= 2048 /* ContainsInvalidEscape */;
        if (flags & 6 /* ReportInvalidEscapeErrors */) {
          if (flags & 4 /* RegularExpression */ && !(flags & 32 /* AtomEscape */)) {
            error(Diagnostics.Decimal_escape_sequences_and_backreferences_are_not_allowed_in_a_character_class, start2, pos - start2);
          } else {
            error(Diagnostics.Escape_sequence_0_is_not_allowed, start2, pos - start2, text.substring(start2, pos));
          }
          return String.fromCharCode(ch);
        }
        return text.substring(start2, pos);
      case 98 /* b */:
        return "\b";
      case 116 /* t */:
        return "	";
      case 110 /* n */:
        return "\n";
      case 118 /* v */:
        return "\v";
      case 102 /* f */:
        return "\f";
      case 114 /* r */:
        return "\r";
      case 39 /* singleQuote */:
        return "'";
      case 34 /* doubleQuote */:
        return '"';
      case 117 /* u */:
        if (pos < end && charCodeUnchecked(pos) === 123 /* openBrace */) {
          pos -= 2;
          const result = scanExtendedUnicodeEscape(!!(flags & 6 /* ReportInvalidEscapeErrors */));
          if (!(flags & 17 /* AllowExtendedUnicodeEscape */)) {
            tokenFlags |= 2048 /* ContainsInvalidEscape */;
            if (flags & 6 /* ReportInvalidEscapeErrors */) {
              error(Diagnostics.Unicode_escape_sequences_are_only_available_when_the_Unicode_u_flag_or_the_Unicode_Sets_v_flag_is_set, start2, pos - start2);
            }
          }
          return result;
        }
        for (; pos < start2 + 6; pos++) {
          if (!(pos < end && isHexDigit(charCodeUnchecked(pos)))) {
            tokenFlags |= 2048 /* ContainsInvalidEscape */;
            if (flags & 6 /* ReportInvalidEscapeErrors */) {
              error(Diagnostics.Hexadecimal_digit_expected);
            }
            return text.substring(start2, pos);
          }
        }
        tokenFlags |= 1024 /* UnicodeEscape */;
        const escapedValue = parseInt(text.substring(start2 + 2, pos), 16);
        const escapedValueString = String.fromCharCode(escapedValue);
        if (flags & 16 /* AnyUnicodeMode */ && escapedValue >= 55296 && escapedValue <= 56319 && pos + 6 < end && text.substring(pos, pos + 2) === "\\u" && charCodeUnchecked(pos + 2) !== 123 /* openBrace */) {
          const nextStart = pos;
          let nextPos = pos + 2;
          for (; nextPos < nextStart + 6; nextPos++) {
            if (!isHexDigit(charCodeUnchecked(nextPos))) {
              return escapedValueString;
            }
          }
          const nextEscapedValue = parseInt(text.substring(nextStart + 2, nextPos), 16);
          if (nextEscapedValue >= 56320 && nextEscapedValue <= 57343) {
            pos = nextPos;
            return escapedValueString + String.fromCharCode(nextEscapedValue);
          }
        }
        return escapedValueString;
      case 120 /* x */:
        for (; pos < start2 + 4; pos++) {
          if (!(pos < end && isHexDigit(charCodeUnchecked(pos)))) {
            tokenFlags |= 2048 /* ContainsInvalidEscape */;
            if (flags & 6 /* ReportInvalidEscapeErrors */) {
              error(Diagnostics.Hexadecimal_digit_expected);
            }
            return text.substring(start2, pos);
          }
        }
        tokenFlags |= 4096 /* HexEscape */;
        return String.fromCharCode(parseInt(text.substring(start2 + 2, pos), 16));
      // when encountering a LineContinuation (i.e. a backslash and a line terminator sequence),
      // the line terminator is interpreted to be "the empty code unit sequence".
      case 13 /* carriageReturn */:
        if (pos < end && charCodeUnchecked(pos) === 10 /* lineFeed */) {
          pos++;
        }
      // falls through
      case 10 /* lineFeed */:
      case 8232 /* lineSeparator */:
      case 8233 /* paragraphSeparator */:
        return "";
      default:
        if (flags & 16 /* AnyUnicodeMode */ || flags & 4 /* RegularExpression */ && !(flags & 8 /* AnnexB */) && isIdentifierPart(ch, languageVersion)) {
          error(Diagnostics.This_character_cannot_be_escaped_in_a_regular_expression, pos - 2, 2);
        }
        return String.fromCharCode(ch);
    }
  }
  function scanExtendedUnicodeEscape(shouldEmitInvalidEscapeError) {
    const start2 = pos;
    pos += 3;
    const escapedStart = pos;
    const escapedValueString = scanMinimumNumberOfHexDigits(
      1,
      /*canHaveSeparators*/
      false
    );
    const escapedValue = escapedValueString ? parseInt(escapedValueString, 16) : -1;
    let isInvalidExtendedEscape = false;
    if (escapedValue < 0) {
      if (shouldEmitInvalidEscapeError) {
        error(Diagnostics.Hexadecimal_digit_expected);
      }
      isInvalidExtendedEscape = true;
    } else if (escapedValue > 1114111) {
      if (shouldEmitInvalidEscapeError) {
        error(Diagnostics.An_extended_Unicode_escape_value_must_be_between_0x0_and_0x10FFFF_inclusive, escapedStart, pos - escapedStart);
      }
      isInvalidExtendedEscape = true;
    }
    if (pos >= end) {
      if (shouldEmitInvalidEscapeError) {
        error(Diagnostics.Unexpected_end_of_text);
      }
      isInvalidExtendedEscape = true;
    } else if (charCodeUnchecked(pos) === 125 /* closeBrace */) {
      pos++;
    } else {
      if (shouldEmitInvalidEscapeError) {
        error(Diagnostics.Unterminated_Unicode_escape_sequence);
      }
      isInvalidExtendedEscape = true;
    }
    if (isInvalidExtendedEscape) {
      tokenFlags |= 2048 /* ContainsInvalidEscape */;
      return text.substring(start2, pos);
    }
    tokenFlags |= 8 /* ExtendedUnicodeEscape */;
    return utf16EncodeAsString(escapedValue);
  }
  function peekUnicodeEscape() {
    if (pos + 5 < end && charCodeUnchecked(pos + 1) === 117 /* u */) {
      const start2 = pos;
      pos += 2;
      const value = scanExactNumberOfHexDigits(
        4,
        /*canHaveSeparators*/
        false
      );
      pos = start2;
      return value;
    }
    return -1;
  }
  function peekExtendedUnicodeEscape() {
    if (codePointUnchecked(pos + 1) === 117 /* u */ && codePointUnchecked(pos + 2) === 123 /* openBrace */) {
      const start2 = pos;
      pos += 3;
      const escapedValueString = scanMinimumNumberOfHexDigits(
        1,
        /*canHaveSeparators*/
        false
      );
      const escapedValue = escapedValueString ? parseInt(escapedValueString, 16) : -1;
      pos = start2;
      return escapedValue;
    }
    return -1;
  }
  function scanIdentifierParts() {
    let result = "";
    let start2 = pos;
    while (pos < end) {
      let ch = codePointUnchecked(pos);
      if (isIdentifierPart(ch, languageVersion)) {
        pos += charSize(ch);
      } else if (ch === 92 /* backslash */) {
        ch = peekExtendedUnicodeEscape();
        if (ch >= 0 && isIdentifierPart(ch, languageVersion)) {
          result += scanExtendedUnicodeEscape(
            /*shouldEmitInvalidEscapeError*/
            true
          );
          start2 = pos;
          continue;
        }
        ch = peekUnicodeEscape();
        if (!(ch >= 0 && isIdentifierPart(ch, languageVersion))) {
          break;
        }
        tokenFlags |= 1024 /* UnicodeEscape */;
        result += text.substring(start2, pos);
        result += utf16EncodeAsString(ch);
        pos += 6;
        start2 = pos;
      } else {
        break;
      }
    }
    result += text.substring(start2, pos);
    return result;
  }
  function getIdentifierToken() {
    const len = tokenValue.length;
    if (len >= 2 && len <= 12) {
      const ch = tokenValue.charCodeAt(0);
      if (ch >= 97 /* a */ && ch <= 122 /* z */) {
        const keyword = textToKeyword.get(tokenValue);
        if (keyword !== void 0) {
          return token = keyword;
        }
      }
    }
    return token = 80 /* Identifier */;
  }
  function scanBinaryOrOctalDigits(base) {
    let value = "";
    let separatorAllowed = false;
    let isPreviousTokenSeparator = false;
    while (true) {
      const ch = charCodeUnchecked(pos);
      if (ch === 95 /* _ */) {
        tokenFlags |= 512 /* ContainsSeparator */;
        if (separatorAllowed) {
          separatorAllowed = false;
          isPreviousTokenSeparator = true;
        } else if (isPreviousTokenSeparator) {
          error(Diagnostics.Multiple_consecutive_numeric_separators_are_not_permitted, pos, 1);
        } else {
          error(Diagnostics.Numeric_separators_are_not_allowed_here, pos, 1);
        }
        pos++;
        continue;
      }
      separatorAllowed = true;
      if (!isDigit(ch) || ch - 48 /* _0 */ >= base) {
        break;
      }
      value += text[pos];
      pos++;
      isPreviousTokenSeparator = false;
    }
    if (charCodeUnchecked(pos - 1) === 95 /* _ */) {
      error(Diagnostics.Numeric_separators_are_not_allowed_here, pos - 1, 1);
    }
    return value;
  }
  function checkBigIntSuffix() {
    if (charCodeUnchecked(pos) === 110 /* n */) {
      tokenValue += "n";
      if (tokenFlags & 384 /* BinaryOrOctalSpecifier */) {
        tokenValue = parsePseudoBigInt(tokenValue) + "n";
      }
      pos++;
      return 10 /* BigIntLiteral */;
    } else {
      const numericValue = tokenFlags & 128 /* BinarySpecifier */ ? parseInt(tokenValue.slice(2), 2) : tokenFlags & 256 /* OctalSpecifier */ ? parseInt(tokenValue.slice(2), 8) : +tokenValue;
      tokenValue = "" + numericValue;
      return 9 /* NumericLiteral */;
    }
  }
  function scan() {
    fullStartPos = pos;
    tokenFlags = 0 /* None */;
    while (true) {
      tokenStart = pos;
      if (pos >= end) {
        return token = 1 /* EndOfFileToken */;
      }
      const ch = codePointUnchecked(pos);
      if (pos === 0) {
        if (ch === 35 /* hash */ && isShebangTrivia(text, pos)) {
          pos = scanShebangTrivia(text, pos);
          if (skipTrivia2) {
            continue;
          } else {
            return token = 6 /* ShebangTrivia */;
          }
        }
      }
      switch (ch) {
        case 10 /* lineFeed */:
        case 13 /* carriageReturn */:
          tokenFlags |= 1 /* PrecedingLineBreak */;
          if (skipTrivia2) {
            pos++;
            continue;
          } else {
            if (ch === 13 /* carriageReturn */ && pos + 1 < end && charCodeUnchecked(pos + 1) === 10 /* lineFeed */) {
              pos += 2;
            } else {
              pos++;
            }
            return token = 4 /* NewLineTrivia */;
          }
        case 9 /* tab */:
        case 11 /* verticalTab */:
        case 12 /* formFeed */:
        case 32 /* space */:
        case 160 /* nonBreakingSpace */:
        case 5760 /* ogham */:
        case 8192 /* enQuad */:
        case 8193 /* emQuad */:
        case 8194 /* enSpace */:
        case 8195 /* emSpace */:
        case 8196 /* threePerEmSpace */:
        case 8197 /* fourPerEmSpace */:
        case 8198 /* sixPerEmSpace */:
        case 8199 /* figureSpace */:
        case 8200 /* punctuationSpace */:
        case 8201 /* thinSpace */:
        case 8202 /* hairSpace */:
        case 8203 /* zeroWidthSpace */:
        case 8239 /* narrowNoBreakSpace */:
        case 8287 /* mathematicalSpace */:
        case 12288 /* ideographicSpace */:
        case 65279 /* byteOrderMark */:
          if (skipTrivia2) {
            pos++;
            continue;
          } else {
            while (pos < end && isWhiteSpaceSingleLine(charCodeUnchecked(pos))) {
              pos++;
            }
            return token = 5 /* WhitespaceTrivia */;
          }
        case 33 /* exclamation */:
          if (charCodeUnchecked(pos + 1) === 61 /* equals */) {
            if (charCodeUnchecked(pos + 2) === 61 /* equals */) {
              return pos += 3, token = 38 /* ExclamationEqualsEqualsToken */;
            }
            return pos += 2, token = 36 /* ExclamationEqualsToken */;
          }
          pos++;
          return token = 54 /* ExclamationToken */;
        case 34 /* doubleQuote */:
        case 39 /* singleQuote */:
          tokenValue = scanString();
          return token = 11 /* StringLiteral */;
        case 96 /* backtick */:
          return token = scanTemplateAndSetTokenValue(
            /*shouldEmitInvalidEscapeError*/
            false
          );
        case 37 /* percent */:
          if (charCodeUnchecked(pos + 1) === 61 /* equals */) {
            return pos += 2, token = 70 /* PercentEqualsToken */;
          }
          pos++;
          return token = 45 /* PercentToken */;
        case 38 /* ampersand */:
          if (charCodeUnchecked(pos + 1) === 38 /* ampersand */) {
            if (charCodeUnchecked(pos + 2) === 61 /* equals */) {
              return pos += 3, token = 77 /* AmpersandAmpersandEqualsToken */;
            }
            return pos += 2, token = 56 /* AmpersandAmpersandToken */;
          }
          if (charCodeUnchecked(pos + 1) === 61 /* equals */) {
            return pos += 2, token = 74 /* AmpersandEqualsToken */;
          }
          pos++;
          return token = 51 /* AmpersandToken */;
        case 40 /* openParen */:
          pos++;
          return token = 21 /* OpenParenToken */;
        case 41 /* closeParen */:
          pos++;
          return token = 22 /* CloseParenToken */;
        case 42 /* asterisk */:
          if (charCodeUnchecked(pos + 1) === 61 /* equals */) {
            return pos += 2, token = 67 /* AsteriskEqualsToken */;
          }
          if (charCodeUnchecked(pos + 1) === 42 /* asterisk */) {
            if (charCodeUnchecked(pos + 2) === 61 /* equals */) {
              return pos += 3, token = 68 /* AsteriskAsteriskEqualsToken */;
            }
            return pos += 2, token = 43 /* AsteriskAsteriskToken */;
          }
          pos++;
          if (skipJsDocLeadingAsterisks && (tokenFlags & 32768 /* PrecedingJSDocLeadingAsterisks */) === 0 && tokenFlags & 1 /* PrecedingLineBreak */) {
            tokenFlags |= 32768 /* PrecedingJSDocLeadingAsterisks */;
            continue;
          }
          return token = 42 /* AsteriskToken */;
        case 43 /* plus */:
          if (charCodeUnchecked(pos + 1) === 43 /* plus */) {
            return pos += 2, token = 46 /* PlusPlusToken */;
          }
          if (charCodeUnchecked(pos + 1) === 61 /* equals */) {
            return pos += 2, token = 65 /* PlusEqualsToken */;
          }
          pos++;
          return token = 40 /* PlusToken */;
        case 44 /* comma */:
          pos++;
          return token = 28 /* CommaToken */;
        case 45 /* minus */:
          if (charCodeUnchecked(pos + 1) === 45 /* minus */) {
            return pos += 2, token = 47 /* MinusMinusToken */;
          }
          if (charCodeUnchecked(pos + 1) === 61 /* equals */) {
            return pos += 2, token = 66 /* MinusEqualsToken */;
          }
          pos++;
          return token = 41 /* MinusToken */;
        case 46 /* dot */:
          if (isDigit(charCodeUnchecked(pos + 1))) {
            scanNumber();
            return token = 9 /* NumericLiteral */;
          }
          if (charCodeUnchecked(pos + 1) === 46 /* dot */ && charCodeUnchecked(pos + 2) === 46 /* dot */) {
            return pos += 3, token = 26 /* DotDotDotToken */;
          }
          pos++;
          return token = 25 /* DotToken */;
        case 47 /* slash */:
          if (charCodeUnchecked(pos + 1) === 47 /* slash */) {
            pos += 2;
            while (pos < end) {
              if (isLineBreak(charCodeUnchecked(pos))) {
                break;
              }
              pos++;
            }
            commentDirectives = appendIfCommentDirective(
              commentDirectives,
              text.slice(tokenStart, pos),
              commentDirectiveRegExSingleLine,
              tokenStart
            );
            if (skipTrivia2) {
              continue;
            } else {
              return token = 2 /* SingleLineCommentTrivia */;
            }
          }
          if (charCodeUnchecked(pos + 1) === 42 /* asterisk */) {
            pos += 2;
            const isJSDoc2 = charCodeUnchecked(pos) === 42 /* asterisk */ && charCodeUnchecked(pos + 1) !== 47 /* slash */;
            let commentClosed = false;
            let lastLineStart = tokenStart;
            while (pos < end) {
              const ch2 = charCodeUnchecked(pos);
              if (ch2 === 42 /* asterisk */ && charCodeUnchecked(pos + 1) === 47 /* slash */) {
                pos += 2;
                commentClosed = true;
                break;
              }
              pos++;
              if (isLineBreak(ch2)) {
                lastLineStart = pos;
                tokenFlags |= 1 /* PrecedingLineBreak */;
              }
            }
            if (isJSDoc2 && shouldParseJSDoc()) {
              tokenFlags |= 2 /* PrecedingJSDocComment */;
            }
            commentDirectives = appendIfCommentDirective(commentDirectives, text.slice(lastLineStart, pos), commentDirectiveRegExMultiLine, lastLineStart);
            if (!commentClosed) {
              error(Diagnostics.Asterisk_Slash_expected);
            }
            if (skipTrivia2) {
              continue;
            } else {
              if (!commentClosed) {
                tokenFlags |= 4 /* Unterminated */;
              }
              return token = 3 /* MultiLineCommentTrivia */;
            }
          }
          if (charCodeUnchecked(pos + 1) === 61 /* equals */) {
            return pos += 2, token = 69 /* SlashEqualsToken */;
          }
          pos++;
          return token = 44 /* SlashToken */;
        case 48 /* _0 */:
          if (pos + 2 < end && (charCodeUnchecked(pos + 1) === 88 /* X */ || charCodeUnchecked(pos + 1) === 120 /* x */)) {
            pos += 2;
            tokenValue = scanMinimumNumberOfHexDigits(
              1,
              /*canHaveSeparators*/
              true
            );
            if (!tokenValue) {
              error(Diagnostics.Hexadecimal_digit_expected);
              tokenValue = "0";
            }
            tokenValue = "0x" + tokenValue;
            tokenFlags |= 64 /* HexSpecifier */;
            return token = checkBigIntSuffix();
          } else if (pos + 2 < end && (charCodeUnchecked(pos + 1) === 66 /* B */ || charCodeUnchecked(pos + 1) === 98 /* b */)) {
            pos += 2;
            tokenValue = scanBinaryOrOctalDigits(
              /* base */
              2
            );
            if (!tokenValue) {
              error(Diagnostics.Binary_digit_expected);
              tokenValue = "0";
            }
            tokenValue = "0b" + tokenValue;
            tokenFlags |= 128 /* BinarySpecifier */;
            return token = checkBigIntSuffix();
          } else if (pos + 2 < end && (charCodeUnchecked(pos + 1) === 79 /* O */ || charCodeUnchecked(pos + 1) === 111 /* o */)) {
            pos += 2;
            tokenValue = scanBinaryOrOctalDigits(
              /* base */
              8
            );
            if (!tokenValue) {
              error(Diagnostics.Octal_digit_expected);
              tokenValue = "0";
            }
            tokenValue = "0o" + tokenValue;
            tokenFlags |= 256 /* OctalSpecifier */;
            return token = checkBigIntSuffix();
          }
        // falls through
        case 49 /* _1 */:
        case 50 /* _2 */:
        case 51 /* _3 */:
        case 52 /* _4 */:
        case 53 /* _5 */:
        case 54 /* _6 */:
        case 55 /* _7 */:
        case 56 /* _8 */:
        case 57 /* _9 */:
          return token = scanNumber();
        case 58 /* colon */:
          pos++;
          return token = 59 /* ColonToken */;
        case 59 /* semicolon */:
          pos++;
          return token = 27 /* SemicolonToken */;
        case 60 /* lessThan */:
          if (isConflictMarkerTrivia(text, pos)) {
            pos = scanConflictMarkerTrivia(text, pos, error);
            if (skipTrivia2) {
              continue;
            } else {
              return token = 7 /* ConflictMarkerTrivia */;
            }
          }
          if (charCodeUnchecked(pos + 1) === 60 /* lessThan */) {
            if (charCodeUnchecked(pos + 2) === 61 /* equals */) {
              return pos += 3, token = 71 /* LessThanLessThanEqualsToken */;
            }
            return pos += 2, token = 48 /* LessThanLessThanToken */;
          }
          if (charCodeUnchecked(pos + 1) === 61 /* equals */) {
            return pos += 2, token = 33 /* LessThanEqualsToken */;
          }
          if (languageVariant === 1 /* JSX */ && charCodeUnchecked(pos + 1) === 47 /* slash */ && charCodeUnchecked(pos + 2) !== 42 /* asterisk */) {
            return pos += 2, token = 31 /* LessThanSlashToken */;
          }
          pos++;
          return token = 30 /* LessThanToken */;
        case 61 /* equals */:
          if (isConflictMarkerTrivia(text, pos)) {
            pos = scanConflictMarkerTrivia(text, pos, error);
            if (skipTrivia2) {
              continue;
            } else {
              return token = 7 /* ConflictMarkerTrivia */;
            }
          }
          if (charCodeUnchecked(pos + 1) === 61 /* equals */) {
            if (charCodeUnchecked(pos + 2) === 61 /* equals */) {
              return pos += 3, token = 37 /* EqualsEqualsEqualsToken */;
            }
            return pos += 2, token = 35 /* EqualsEqualsToken */;
          }
          if (charCodeUnchecked(pos + 1) === 62 /* greaterThan */) {
            return pos += 2, token = 39 /* EqualsGreaterThanToken */;
          }
          pos++;
          return token = 64 /* EqualsToken */;
        case 62 /* greaterThan */:
          if (isConflictMarkerTrivia(text, pos)) {
            pos = scanConflictMarkerTrivia(text, pos, error);
            if (skipTrivia2) {
              continue;
            } else {
              return token = 7 /* ConflictMarkerTrivia */;
            }
          }
          pos++;
          return token = 32 /* GreaterThanToken */;
        case 63 /* question */:
          if (charCodeUnchecked(pos + 1) === 46 /* dot */ && !isDigit(charCodeUnchecked(pos + 2))) {
            return pos += 2, token = 29 /* QuestionDotToken */;
          }
          if (charCodeUnchecked(pos + 1) === 63 /* question */) {
            if (charCodeUnchecked(pos + 2) === 61 /* equals */) {
              return pos += 3, token = 78 /* QuestionQuestionEqualsToken */;
            }
            return pos += 2, token = 61 /* QuestionQuestionToken */;
          }
          pos++;
          return token = 58 /* QuestionToken */;
        case 91 /* openBracket */:
          pos++;
          return token = 23 /* OpenBracketToken */;
        case 93 /* closeBracket */:
          pos++;
          return token = 24 /* CloseBracketToken */;
        case 94 /* caret */:
          if (charCodeUnchecked(pos + 1) === 61 /* equals */) {
            return pos += 2, token = 79 /* CaretEqualsToken */;
          }
          pos++;
          return token = 53 /* CaretToken */;
        case 123 /* openBrace */:
          pos++;
          return token = 19 /* OpenBraceToken */;
        case 124 /* bar */:
          if (isConflictMarkerTrivia(text, pos)) {
            pos = scanConflictMarkerTrivia(text, pos, error);
            if (skipTrivia2) {
              continue;
            } else {
              return token = 7 /* ConflictMarkerTrivia */;
            }
          }
          if (charCodeUnchecked(pos + 1) === 124 /* bar */) {
            if (charCodeUnchecked(pos + 2) === 61 /* equals */) {
              return pos += 3, token = 76 /* BarBarEqualsToken */;
            }
            return pos += 2, token = 57 /* BarBarToken */;
          }
          if (charCodeUnchecked(pos + 1) === 61 /* equals */) {
            return pos += 2, token = 75 /* BarEqualsToken */;
          }
          pos++;
          return token = 52 /* BarToken */;
        case 125 /* closeBrace */:
          pos++;
          return token = 20 /* CloseBraceToken */;
        case 126 /* tilde */:
          pos++;
          return token = 55 /* TildeToken */;
        case 64 /* at */:
          pos++;
          return token = 60 /* AtToken */;
        case 92 /* backslash */:
          const extendedCookedChar = peekExtendedUnicodeEscape();
          if (extendedCookedChar >= 0 && isIdentifierStart(extendedCookedChar, languageVersion)) {
            tokenValue = scanExtendedUnicodeEscape(
              /*shouldEmitInvalidEscapeError*/
              true
            ) + scanIdentifierParts();
            return token = getIdentifierToken();
          }
          const cookedChar = peekUnicodeEscape();
          if (cookedChar >= 0 && isIdentifierStart(cookedChar, languageVersion)) {
            pos += 6;
            tokenFlags |= 1024 /* UnicodeEscape */;
            tokenValue = String.fromCharCode(cookedChar) + scanIdentifierParts();
            return token = getIdentifierToken();
          }
          error(Diagnostics.Invalid_character);
          pos++;
          return token = 0 /* Unknown */;
        case 35 /* hash */:
          if (pos !== 0 && text[pos + 1] === "!") {
            error(Diagnostics.can_only_be_used_at_the_start_of_a_file, pos, 2);
            pos++;
            return token = 0 /* Unknown */;
          }
          const charAfterHash = codePointUnchecked(pos + 1);
          if (charAfterHash === 92 /* backslash */) {
            pos++;
            const extendedCookedChar2 = peekExtendedUnicodeEscape();
            if (extendedCookedChar2 >= 0 && isIdentifierStart(extendedCookedChar2, languageVersion)) {
              tokenValue = "#" + scanExtendedUnicodeEscape(
                /*shouldEmitInvalidEscapeError*/
                true
              ) + scanIdentifierParts();
              return token = 81 /* PrivateIdentifier */;
            }
            const cookedChar2 = peekUnicodeEscape();
            if (cookedChar2 >= 0 && isIdentifierStart(cookedChar2, languageVersion)) {
              pos += 6;
              tokenFlags |= 1024 /* UnicodeEscape */;
              tokenValue = "#" + String.fromCharCode(cookedChar2) + scanIdentifierParts();
              return token = 81 /* PrivateIdentifier */;
            }
            pos--;
          }
          if (isIdentifierStart(charAfterHash, languageVersion)) {
            pos++;
            scanIdentifier(charAfterHash, languageVersion);
          } else {
            tokenValue = "#";
            error(Diagnostics.Invalid_character, pos++, charSize(ch));
          }
          return token = 81 /* PrivateIdentifier */;
        case 65533 /* replacementCharacter */:
          error(Diagnostics.File_appears_to_be_binary, 0, 0);
          pos = end;
          return token = 8 /* NonTextFileMarkerTrivia */;
        default:
          const identifierKind = scanIdentifier(ch, languageVersion);
          if (identifierKind) {
            return token = identifierKind;
          } else if (isWhiteSpaceSingleLine(ch)) {
            pos += charSize(ch);
            continue;
          } else if (isLineBreak(ch)) {
            tokenFlags |= 1 /* PrecedingLineBreak */;
            pos += charSize(ch);
            continue;
          }
          const size = charSize(ch);
          error(Diagnostics.Invalid_character, pos, size);
          pos += size;
          return token = 0 /* Unknown */;
      }
    }
  }
  function shouldParseJSDoc() {
    switch (jsDocParsingMode) {
      case 0 /* ParseAll */:
        return true;
      case 1 /* ParseNone */:
        return false;
    }
    if (scriptKind !== 3 /* TS */ && scriptKind !== 4 /* TSX */) {
      return true;
    }
    if (jsDocParsingMode === 3 /* ParseForTypeInfo */) {
      return false;
    }
    return jsDocSeeOrLink.test(text.slice(fullStartPos, pos));
  }
  function reScanInvalidIdentifier() {
    Debug.assert(token === 0 /* Unknown */, "'reScanInvalidIdentifier' should only be called when the current token is 'SyntaxKind.Unknown'.");
    pos = tokenStart = fullStartPos;
    tokenFlags = 0;
    const ch = codePointUnchecked(pos);
    const identifierKind = scanIdentifier(ch, 99 /* ESNext */);
    if (identifierKind) {
      return token = identifierKind;
    }
    pos += charSize(ch);
    return token;
  }
  function scanIdentifier(startCharacter, languageVersion2) {
    let ch = startCharacter;
    if (isIdentifierStart(ch, languageVersion2)) {
      pos += charSize(ch);
      while (pos < end && isIdentifierPart(ch = codePointUnchecked(pos), languageVersion2)) pos += charSize(ch);
      tokenValue = text.substring(tokenStart, pos);
      if (ch === 92 /* backslash */) {
        tokenValue += scanIdentifierParts();
      }
      return getIdentifierToken();
    }
  }
  function reScanGreaterToken() {
    if (token === 32 /* GreaterThanToken */) {
      if (charCodeUnchecked(pos) === 62 /* greaterThan */) {
        if (charCodeUnchecked(pos + 1) === 62 /* greaterThan */) {
          if (charCodeUnchecked(pos + 2) === 61 /* equals */) {
            return pos += 3, token = 73 /* GreaterThanGreaterThanGreaterThanEqualsToken */;
          }
          return pos += 2, token = 50 /* GreaterThanGreaterThanGreaterThanToken */;
        }
        if (charCodeUnchecked(pos + 1) === 61 /* equals */) {
          return pos += 2, token = 72 /* GreaterThanGreaterThanEqualsToken */;
        }
        pos++;
        return token = 49 /* GreaterThanGreaterThanToken */;
      }
      if (charCodeUnchecked(pos) === 61 /* equals */) {
        pos++;
        return token = 34 /* GreaterThanEqualsToken */;
      }
    }
    return token;
  }
  function reScanAsteriskEqualsToken() {
    Debug.assert(token === 67 /* AsteriskEqualsToken */, "'reScanAsteriskEqualsToken' should only be called on a '*='");
    pos = tokenStart + 1;
    return token = 64 /* EqualsToken */;
  }
  function reScanSlashToken(reportErrors2) {
    if (token === 44 /* SlashToken */ || token === 69 /* SlashEqualsToken */) {
      const startOfRegExpBody = tokenStart + 1;
      pos = startOfRegExpBody;
      let inEscape = false;
      let namedCaptureGroups = false;
      let inCharacterClass = false;
      while (true) {
        const ch = charCodeChecked(pos);
        if (ch === -1 /* EOF */ || isLineBreak(ch)) {
          tokenFlags |= 4 /* Unterminated */;
          break;
        }
        if (inEscape) {
          inEscape = false;
        } else if (ch === 47 /* slash */ && !inCharacterClass) {
          break;
        } else if (ch === 91 /* openBracket */) {
          inCharacterClass = true;
        } else if (ch === 92 /* backslash */) {
          inEscape = true;
        } else if (ch === 93 /* closeBracket */) {
          inCharacterClass = false;
        } else if (!inCharacterClass && ch === 40 /* openParen */ && charCodeChecked(pos + 1) === 63 /* question */ && charCodeChecked(pos + 2) === 60 /* lessThan */ && charCodeChecked(pos + 3) !== 61 /* equals */ && charCodeChecked(pos + 3) !== 33 /* exclamation */) {
          namedCaptureGroups = true;
        }
        pos++;
      }
      const endOfRegExpBody = pos;
      if (tokenFlags & 4 /* Unterminated */) {
        pos = startOfRegExpBody;
        inEscape = false;
        let characterClassDepth = 0;
        let inDecimalQuantifier = false;
        let groupDepth = 0;
        while (pos < endOfRegExpBody) {
          const ch = charCodeUnchecked(pos);
          if (inEscape) {
            inEscape = false;
          } else if (ch === 92 /* backslash */) {
            inEscape = true;
          } else if (ch === 91 /* openBracket */) {
            characterClassDepth++;
          } else if (ch === 93 /* closeBracket */ && characterClassDepth) {
            characterClassDepth--;
          } else if (!characterClassDepth) {
            if (ch === 123 /* openBrace */) {
              inDecimalQuantifier = true;
            } else if (ch === 125 /* closeBrace */ && inDecimalQuantifier) {
              inDecimalQuantifier = false;
            } else if (!inDecimalQuantifier) {
              if (ch === 40 /* openParen */) {
                groupDepth++;
              } else if (ch === 41 /* closeParen */ && groupDepth) {
                groupDepth--;
              } else if (ch === 41 /* closeParen */ || ch === 93 /* closeBracket */ || ch === 125 /* closeBrace */) {
                break;
              }
            }
          }
          pos++;
        }
        while (isWhiteSpaceLike(charCodeChecked(pos - 1)) || charCodeChecked(pos - 1) === 59 /* semicolon */) pos--;
        error(Diagnostics.Unterminated_regular_expression_literal, tokenStart, pos - tokenStart);
      } else {
        pos++;
        let regExpFlags = 0 /* None */;
        while (true) {
          const ch = codePointChecked(pos);
          if (ch === -1 /* EOF */ || !isIdentifierPart(ch, languageVersion)) {
            break;
          }
          const size = charSize(ch);
          if (reportErrors2) {
            const flag = characterCodeToRegularExpressionFlag(ch);
            if (flag === void 0) {
              error(Diagnostics.Unknown_regular_expression_flag, pos, size);
            } else if (regExpFlags & flag) {
              error(Diagnostics.Duplicate_regular_expression_flag, pos, size);
            } else if (((regExpFlags | flag) & 96 /* AnyUnicodeMode */) === 96 /* AnyUnicodeMode */) {
              error(Diagnostics.The_Unicode_u_flag_and_the_Unicode_Sets_v_flag_cannot_be_set_simultaneously, pos, size);
            } else {
              regExpFlags |= flag;
              checkRegularExpressionFlagAvailability(flag, size);
            }
          }
          pos += size;
        }
        if (reportErrors2) {
          scanRange(startOfRegExpBody, endOfRegExpBody - startOfRegExpBody, () => {
            scanRegularExpressionWorker(
              regExpFlags,
              /*annexB*/
              true,
              namedCaptureGroups
            );
          });
        }
      }
      tokenValue = text.substring(tokenStart, pos);
      token = 14 /* RegularExpressionLiteral */;
    }
    return token;
  }
  function scanRegularExpressionWorker(regExpFlags, annexB, namedCaptureGroups) {
    var unicodeSetsMode = !!(regExpFlags & 64 /* UnicodeSets */);
    var anyUnicodeMode = !!(regExpFlags & 96 /* AnyUnicodeMode */);
    var anyUnicodeModeOrNonAnnexB = anyUnicodeMode || !annexB;
    var mayContainStrings = false;
    var numberOfCapturingGroups = 0;
    var groupSpecifiers;
    var groupNameReferences;
    var decimalEscapes;
    var namedCapturingGroupsScopeStack = [];
    var topNamedCapturingGroupsScope;
    function scanDisjunction(isInGroup) {
      while (true) {
        namedCapturingGroupsScopeStack.push(topNamedCapturingGroupsScope);
        topNamedCapturingGroupsScope = void 0;
        scanAlternative(isInGroup);
        topNamedCapturingGroupsScope = namedCapturingGroupsScopeStack.pop();
        if (charCodeChecked(pos) !== 124 /* bar */) {
          return;
        }
        pos++;
      }
    }
    function scanAlternative(isInGroup) {
      let isPreviousTermQuantifiable = false;
      while (true) {
        const start2 = pos;
        const ch = charCodeChecked(pos);
        switch (ch) {
          case -1 /* EOF */:
            return;
          case 94 /* caret */:
          case 36 /* $ */:
            pos++;
            isPreviousTermQuantifiable = false;
            break;
          case 92 /* backslash */:
            pos++;
            switch (charCodeChecked(pos)) {
              case 98 /* b */:
              case 66 /* B */:
                pos++;
                isPreviousTermQuantifiable = false;
                break;
              default:
                scanAtomEscape();
                isPreviousTermQuantifiable = true;
                break;
            }
            break;
          case 40 /* openParen */:
            pos++;
            if (charCodeChecked(pos) === 63 /* question */) {
              pos++;
              switch (charCodeChecked(pos)) {
                case 61 /* equals */:
                case 33 /* exclamation */:
                  pos++;
                  isPreviousTermQuantifiable = !anyUnicodeModeOrNonAnnexB;
                  break;
                case 60 /* lessThan */:
                  const groupNameStart = pos;
                  pos++;
                  switch (charCodeChecked(pos)) {
                    case 61 /* equals */:
                    case 33 /* exclamation */:
                      pos++;
                      isPreviousTermQuantifiable = false;
                      break;
                    default:
                      scanGroupName(
                        /*isReference*/
                        false
                      );
                      scanExpectedChar(62 /* greaterThan */);
                      if (languageVersion < 5 /* ES2018 */) {
                        error(Diagnostics.Named_capturing_groups_are_only_available_when_targeting_ES2018_or_later, groupNameStart, pos - groupNameStart);
                      }
                      numberOfCapturingGroups++;
                      isPreviousTermQuantifiable = true;
                      break;
                  }
                  break;
                default:
                  const start3 = pos;
                  const setFlags = scanPatternModifiers(0 /* None */);
                  if (charCodeChecked(pos) === 45 /* minus */) {
                    pos++;
                    scanPatternModifiers(setFlags);
                    if (pos === start3 + 1) {
                      error(Diagnostics.Subpattern_flags_must_be_present_when_there_is_a_minus_sign, start3, pos - start3);
                    }
                  }
                  scanExpectedChar(58 /* colon */);
                  isPreviousTermQuantifiable = true;
                  break;
              }
            } else {
              numberOfCapturingGroups++;
              isPreviousTermQuantifiable = true;
            }
            scanDisjunction(
              /*isInGroup*/
              true
            );
            scanExpectedChar(41 /* closeParen */);
            break;
          case 123 /* openBrace */:
            pos++;
            const digitsStart = pos;
            scanDigits();
            const min2 = tokenValue;
            if (!anyUnicodeModeOrNonAnnexB && !min2) {
              isPreviousTermQuantifiable = true;
              break;
            }
            if (charCodeChecked(pos) === 44 /* comma */) {
              pos++;
              scanDigits();
              const max = tokenValue;
              if (!min2) {
                if (max || charCodeChecked(pos) === 125 /* closeBrace */) {
                  error(Diagnostics.Incomplete_quantifier_Digit_expected, digitsStart, 0);
                } else {
                  error(Diagnostics.Unexpected_0_Did_you_mean_to_escape_it_with_backslash, start2, 1, String.fromCharCode(ch));
                  isPreviousTermQuantifiable = true;
                  break;
                }
              } else if (max && Number.parseInt(min2) > Number.parseInt(max) && (anyUnicodeModeOrNonAnnexB || charCodeChecked(pos) === 125 /* closeBrace */)) {
                error(Diagnostics.Numbers_out_of_order_in_quantifier, digitsStart, pos - digitsStart);
              }
            } else if (!min2) {
              if (anyUnicodeModeOrNonAnnexB) {
                error(Diagnostics.Unexpected_0_Did_you_mean_to_escape_it_with_backslash, start2, 1, String.fromCharCode(ch));
              }
              isPreviousTermQuantifiable = true;
              break;
            }
            if (charCodeChecked(pos) !== 125 /* closeBrace */) {
              if (anyUnicodeModeOrNonAnnexB) {
                error(Diagnostics._0_expected, pos, 0, String.fromCharCode(125 /* closeBrace */));
                pos--;
              } else {
                isPreviousTermQuantifiable = true;
                break;
              }
            }
          // falls through
          case 42 /* asterisk */:
          case 43 /* plus */:
          case 63 /* question */:
            pos++;
            if (charCodeChecked(pos) === 63 /* question */) {
              pos++;
            }
            if (!isPreviousTermQuantifiable) {
              error(Diagnostics.There_is_nothing_available_for_repetition, start2, pos - start2);
            }
            isPreviousTermQuantifiable = false;
            break;
          case 46 /* dot */:
            pos++;
            isPreviousTermQuantifiable = true;
            break;
          case 91 /* openBracket */:
            pos++;
            if (unicodeSetsMode) {
              scanClassSetExpression();
            } else {
              scanClassRanges();
            }
            scanExpectedChar(93 /* closeBracket */);
            isPreviousTermQuantifiable = true;
            break;
          case 41 /* closeParen */:
            if (isInGroup) {
              return;
            }
          // falls through
          case 93 /* closeBracket */:
          case 125 /* closeBrace */:
            if (anyUnicodeModeOrNonAnnexB || ch === 41 /* closeParen */) {
              error(Diagnostics.Unexpected_0_Did_you_mean_to_escape_it_with_backslash, pos, 1, String.fromCharCode(ch));
            }
            pos++;
            isPreviousTermQuantifiable = true;
            break;
          case 47 /* slash */:
          case 124 /* bar */:
            return;
          default:
            scanSourceCharacter();
            isPreviousTermQuantifiable = true;
            break;
        }
      }
    }
    function scanPatternModifiers(currFlags) {
      while (true) {
        const ch = codePointChecked(pos);
        if (ch === -1 /* EOF */ || !isIdentifierPart(ch, languageVersion)) {
          break;
        }
        const size = charSize(ch);
        const flag = characterCodeToRegularExpressionFlag(ch);
        if (flag === void 0) {
          error(Diagnostics.Unknown_regular_expression_flag, pos, size);
        } else if (currFlags & flag) {
          error(Diagnostics.Duplicate_regular_expression_flag, pos, size);
        } else if (!(flag & 28 /* Modifiers */)) {
          error(Diagnostics.This_regular_expression_flag_cannot_be_toggled_within_a_subpattern, pos, size);
        } else {
          currFlags |= flag;
          checkRegularExpressionFlagAvailability(flag, size);
        }
        pos += size;
      }
      return currFlags;
    }
    function scanAtomEscape() {
      Debug.assertEqual(charCodeUnchecked(pos - 1), 92 /* backslash */);
      switch (charCodeChecked(pos)) {
        case 107 /* k */:
          pos++;
          if (charCodeChecked(pos) === 60 /* lessThan */) {
            pos++;
            scanGroupName(
              /*isReference*/
              true
            );
            scanExpectedChar(62 /* greaterThan */);
          } else if (anyUnicodeModeOrNonAnnexB || namedCaptureGroups) {
            error(Diagnostics.k_must_be_followed_by_a_capturing_group_name_enclosed_in_angle_brackets, pos - 2, 2);
          }
          break;
        case 113 /* q */:
          if (unicodeSetsMode) {
            pos++;
            error(Diagnostics.q_is_only_available_inside_character_class, pos - 2, 2);
            break;
          }
        // falls through
        default:
          Debug.assert(scanCharacterClassEscape() || scanDecimalEscape() || scanCharacterEscape(
            /*atomEscape*/
            true
          ));
          break;
      }
    }
    function scanDecimalEscape() {
      Debug.assertEqual(charCodeUnchecked(pos - 1), 92 /* backslash */);
      const ch = charCodeChecked(pos);
      if (ch >= 49 /* _1 */ && ch <= 57 /* _9 */) {
        const start2 = pos;
        scanDigits();
        decimalEscapes = append(decimalEscapes, { pos: start2, end: pos, value: +tokenValue });
        return true;
      }
      return false;
    }
    function scanCharacterEscape(atomEscape) {
      Debug.assertEqual(charCodeUnchecked(pos - 1), 92 /* backslash */);
      let ch = charCodeChecked(pos);
      switch (ch) {
        case -1 /* EOF */:
          error(Diagnostics.Undetermined_character_escape, pos - 1, 1);
          return "\\";
        case 99 /* c */:
          pos++;
          ch = charCodeChecked(pos);
          if (isASCIILetter(ch)) {
            pos++;
            return String.fromCharCode(ch & 31);
          }
          if (anyUnicodeModeOrNonAnnexB) {
            error(Diagnostics.c_must_be_followed_by_an_ASCII_letter, pos - 2, 2);
          } else if (atomEscape) {
            pos--;
            return "\\";
          }
          return String.fromCharCode(ch);
        case 94 /* caret */:
        case 36 /* $ */:
        case 47 /* slash */:
        case 92 /* backslash */:
        case 46 /* dot */:
        case 42 /* asterisk */:
        case 43 /* plus */:
        case 63 /* question */:
        case 40 /* openParen */:
        case 41 /* closeParen */:
        case 91 /* openBracket */:
        case 93 /* closeBracket */:
        case 123 /* openBrace */:
        case 125 /* closeBrace */:
        case 124 /* bar */:
          pos++;
          return String.fromCharCode(ch);
        default:
          pos--;
          return scanEscapeSequence(
            4 /* RegularExpression */ | (annexB ? 8 /* AnnexB */ : 0) | (anyUnicodeMode ? 16 /* AnyUnicodeMode */ : 0) | (atomEscape ? 32 /* AtomEscape */ : 0)
          );
      }
    }
    function scanGroupName(isReference) {
      Debug.assertEqual(charCodeUnchecked(pos - 1), 60 /* lessThan */);
      tokenStart = pos;
      scanIdentifier(codePointChecked(pos), languageVersion);
      if (pos === tokenStart) {
        error(Diagnostics.Expected_a_capturing_group_name);
      } else if (isReference) {
        groupNameReferences = append(groupNameReferences, { pos: tokenStart, end: pos, name: tokenValue });
      } else if ((topNamedCapturingGroupsScope == null ? void 0 : topNamedCapturingGroupsScope.has(tokenValue)) || namedCapturingGroupsScopeStack.some((group2) => group2 == null ? void 0 : group2.has(tokenValue))) {
        error(Diagnostics.Named_capturing_groups_with_the_same_name_must_be_mutually_exclusive_to_each_other, tokenStart, pos - tokenStart);
      } else {
        topNamedCapturingGroupsScope ?? (topNamedCapturingGroupsScope = /* @__PURE__ */ new Set());
        topNamedCapturingGroupsScope.add(tokenValue);
        groupSpecifiers ?? (groupSpecifiers = /* @__PURE__ */ new Set());
        groupSpecifiers.add(tokenValue);
      }
    }
    function isClassContentExit(ch) {
      return ch === 93 /* closeBracket */ || ch === -1 /* EOF */ || pos >= end;
    }
    function scanClassRanges() {
      Debug.assertEqual(charCodeUnchecked(pos - 1), 91 /* openBracket */);
      if (charCodeChecked(pos) === 94 /* caret */) {
        pos++;
      }
      while (true) {
        const ch = charCodeChecked(pos);
        if (isClassContentExit(ch)) {
          return;
        }
        const minStart = pos;
        const minCharacter = scanClassAtom();
        if (charCodeChecked(pos) === 45 /* minus */) {
          pos++;
          const ch2 = charCodeChecked(pos);
          if (isClassContentExit(ch2)) {
            return;
          }
          if (!minCharacter && anyUnicodeModeOrNonAnnexB) {
            error(Diagnostics.A_character_class_range_must_not_be_bounded_by_another_character_class, minStart, pos - 1 - minStart);
          }
          const maxStart = pos;
          const maxCharacter = scanClassAtom();
          if (!maxCharacter && anyUnicodeModeOrNonAnnexB) {
            error(Diagnostics.A_character_class_range_must_not_be_bounded_by_another_character_class, maxStart, pos - maxStart);
            continue;
          }
          if (!minCharacter) {
            continue;
          }
          const minCharacterValue = codePointAt(minCharacter, 0);
          const maxCharacterValue = codePointAt(maxCharacter, 0);
          if (minCharacter.length === charSize(minCharacterValue) && maxCharacter.length === charSize(maxCharacterValue) && minCharacterValue > maxCharacterValue) {
            error(Diagnostics.Range_out_of_order_in_character_class, minStart, pos - minStart);
          }
        }
      }
    }
    function scanClassSetExpression() {
      Debug.assertEqual(charCodeUnchecked(pos - 1), 91 /* openBracket */);
      let isCharacterComplement = false;
      if (charCodeChecked(pos) === 94 /* caret */) {
        pos++;
        isCharacterComplement = true;
      }
      let expressionMayContainStrings = false;
      let ch = charCodeChecked(pos);
      if (isClassContentExit(ch)) {
        return;
      }
      let start2 = pos;
      let operand;
      switch (text.slice(pos, pos + 2)) {
        // TODO: don't use slice
        case "--":
        case "&&":
          error(Diagnostics.Expected_a_class_set_operand);
          mayContainStrings = false;
          break;
        default:
          operand = scanClassSetOperand();
          break;
      }
      switch (charCodeChecked(pos)) {
        case 45 /* minus */:
          if (charCodeChecked(pos + 1) === 45 /* minus */) {
            if (isCharacterComplement && mayContainStrings) {
              error(Diagnostics.Anything_that_would_possibly_match_more_than_a_single_character_is_invalid_inside_a_negated_character_class, start2, pos - start2);
            }
            expressionMayContainStrings = mayContainStrings;
            scanClassSetSubExpression(3 /* ClassSubtraction */);
            mayContainStrings = !isCharacterComplement && expressionMayContainStrings;
            return;
          }
          break;
        case 38 /* ampersand */:
          if (charCodeChecked(pos + 1) === 38 /* ampersand */) {
            scanClassSetSubExpression(2 /* ClassIntersection */);
            if (isCharacterComplement && mayContainStrings) {
              error(Diagnostics.Anything_that_would_possibly_match_more_than_a_single_character_is_invalid_inside_a_negated_character_class, start2, pos - start2);
            }
            expressionMayContainStrings = mayContainStrings;
            mayContainStrings = !isCharacterComplement && expressionMayContainStrings;
            return;
          } else {
            error(Diagnostics.Unexpected_0_Did_you_mean_to_escape_it_with_backslash, pos, 1, String.fromCharCode(ch));
          }
          break;
        default:
          if (isCharacterComplement && mayContainStrings) {
            error(Diagnostics.Anything_that_would_possibly_match_more_than_a_single_character_is_invalid_inside_a_negated_character_class, start2, pos - start2);
          }
          expressionMayContainStrings = mayContainStrings;
          break;
      }
      while (true) {
        ch = charCodeChecked(pos);
        if (ch === -1 /* EOF */) {
          break;
        }
        switch (ch) {
          case 45 /* minus */:
            pos++;
            ch = charCodeChecked(pos);
            if (isClassContentExit(ch)) {
              mayContainStrings = !isCharacterComplement && expressionMayContainStrings;
              return;
            }
            if (ch === 45 /* minus */) {
              pos++;
              error(Diagnostics.Operators_must_not_be_mixed_within_a_character_class_Wrap_it_in_a_nested_class_instead, pos - 2, 2);
              start2 = pos - 2;
              operand = text.slice(start2, pos);
              continue;
            } else {
              if (!operand) {
                error(Diagnostics.A_character_class_range_must_not_be_bounded_by_another_character_class, start2, pos - 1 - start2);
              }
              const secondStart = pos;
              const secondOperand = scanClassSetOperand();
              if (isCharacterComplement && mayContainStrings) {
                error(Diagnostics.Anything_that_would_possibly_match_more_than_a_single_character_is_invalid_inside_a_negated_character_class, secondStart, pos - secondStart);
              }
              expressionMayContainStrings || (expressionMayContainStrings = mayContainStrings);
              if (!secondOperand) {
                error(Diagnostics.A_character_class_range_must_not_be_bounded_by_another_character_class, secondStart, pos - secondStart);
                break;
              }
              if (!operand) {
                break;
              }
              const minCharacterValue = codePointAt(operand, 0);
              const maxCharacterValue = codePointAt(secondOperand, 0);
              if (operand.length === charSize(minCharacterValue) && secondOperand.length === charSize(maxCharacterValue) && minCharacterValue > maxCharacterValue) {
                error(Diagnostics.Range_out_of_order_in_character_class, start2, pos - start2);
              }
            }
            break;
          case 38 /* ampersand */:
            start2 = pos;
            pos++;
            if (charCodeChecked(pos) === 38 /* ampersand */) {
              pos++;
              error(Diagnostics.Operators_must_not_be_mixed_within_a_character_class_Wrap_it_in_a_nested_class_instead, pos - 2, 2);
              if (charCodeChecked(pos) === 38 /* ampersand */) {
                error(Diagnostics.Unexpected_0_Did_you_mean_to_escape_it_with_backslash, pos, 1, String.fromCharCode(ch));
                pos++;
              }
            } else {
              error(Diagnostics.Unexpected_0_Did_you_mean_to_escape_it_with_backslash, pos - 1, 1, String.fromCharCode(ch));
            }
            operand = text.slice(start2, pos);
            continue;
        }
        if (isClassContentExit(charCodeChecked(pos))) {
          break;
        }
        start2 = pos;
        switch (text.slice(pos, pos + 2)) {
          // TODO: don't use slice
          case "--":
          case "&&":
            error(Diagnostics.Operators_must_not_be_mixed_within_a_character_class_Wrap_it_in_a_nested_class_instead, pos, 2);
            pos += 2;
            operand = text.slice(start2, pos);
            break;
          default:
            operand = scanClassSetOperand();
            break;
        }
      }
      mayContainStrings = !isCharacterComplement && expressionMayContainStrings;
    }
    function scanClassSetSubExpression(expressionType) {
      let expressionMayContainStrings = mayContainStrings;
      while (true) {
        let ch = charCodeChecked(pos);
        if (isClassContentExit(ch)) {
          break;
        }
        switch (ch) {
          case 45 /* minus */:
            pos++;
            if (charCodeChecked(pos) === 45 /* minus */) {
              pos++;
              if (expressionType !== 3 /* ClassSubtraction */) {
                error(Diagnostics.Operators_must_not_be_mixed_within_a_character_class_Wrap_it_in_a_nested_class_instead, pos - 2, 2);
              }
            } else {
              error(Diagnostics.Operators_must_not_be_mixed_within_a_character_class_Wrap_it_in_a_nested_class_instead, pos - 1, 1);
            }
            break;
          case 38 /* ampersand */:
            pos++;
            if (charCodeChecked(pos) === 38 /* ampersand */) {
              pos++;
              if (expressionType !== 2 /* ClassIntersection */) {
                error(Diagnostics.Operators_must_not_be_mixed_within_a_character_class_Wrap_it_in_a_nested_class_instead, pos - 2, 2);
              }
              if (charCodeChecked(pos) === 38 /* ampersand */) {
                error(Diagnostics.Unexpected_0_Did_you_mean_to_escape_it_with_backslash, pos, 1, String.fromCharCode(ch));
                pos++;
              }
            } else {
              error(Diagnostics.Unexpected_0_Did_you_mean_to_escape_it_with_backslash, pos - 1, 1, String.fromCharCode(ch));
            }
            break;
          default:
            switch (expressionType) {
              case 3 /* ClassSubtraction */:
                error(Diagnostics._0_expected, pos, 0, "--");
                break;
              case 2 /* ClassIntersection */:
                error(Diagnostics._0_expected, pos, 0, "&&");
                break;
              default:
                break;
            }
            break;
        }
        ch = charCodeChecked(pos);
        if (isClassContentExit(ch)) {
          error(Diagnostics.Expected_a_class_set_operand);
          break;
        }
        scanClassSetOperand();
        expressionMayContainStrings && (expressionMayContainStrings = mayContainStrings);
      }
      mayContainStrings = expressionMayContainStrings;
    }
    function scanClassSetOperand() {
      mayContainStrings = false;
      switch (charCodeChecked(pos)) {
        case -1 /* EOF */:
          return "";
        case 91 /* openBracket */:
          pos++;
          scanClassSetExpression();
          scanExpectedChar(93 /* closeBracket */);
          return "";
        case 92 /* backslash */:
          pos++;
          if (scanCharacterClassEscape()) {
            return "";
          } else if (charCodeChecked(pos) === 113 /* q */) {
            pos++;
            if (charCodeChecked(pos) === 123 /* openBrace */) {
              pos++;
              scanClassStringDisjunctionContents();
              scanExpectedChar(125 /* closeBrace */);
              return "";
            } else {
              error(Diagnostics.q_must_be_followed_by_string_alternatives_enclosed_in_braces, pos - 2, 2);
              return "q";
            }
          }
          pos--;
        // falls through
        default:
          return scanClassSetCharacter();
      }
    }
    function scanClassStringDisjunctionContents() {
      Debug.assertEqual(charCodeUnchecked(pos - 1), 123 /* openBrace */);
      let characterCount = 0;
      while (true) {
        const ch = charCodeChecked(pos);
        switch (ch) {
          case -1 /* EOF */:
            return;
          case 125 /* closeBrace */:
            if (characterCount !== 1) {
              mayContainStrings = true;
            }
            return;
          case 124 /* bar */:
            if (characterCount !== 1) {
              mayContainStrings = true;
            }
            pos++;
            start = pos;
            characterCount = 0;
            break;
          default:
            scanClassSetCharacter();
            characterCount++;
            break;
        }
      }
    }
    function scanClassSetCharacter() {
      const ch = charCodeChecked(pos);
      if (ch === -1 /* EOF */) {
        return "";
      }
      if (ch === 92 /* backslash */) {
        pos++;
        const ch2 = charCodeChecked(pos);
        switch (ch2) {
          case 98 /* b */:
            pos++;
            return "\b";
          case 38 /* ampersand */:
          case 45 /* minus */:
          case 33 /* exclamation */:
          case 35 /* hash */:
          case 37 /* percent */:
          case 44 /* comma */:
          case 58 /* colon */:
          case 59 /* semicolon */:
          case 60 /* lessThan */:
          case 61 /* equals */:
          case 62 /* greaterThan */:
          case 64 /* at */:
          case 96 /* backtick */:
          case 126 /* tilde */:
            pos++;
            return String.fromCharCode(ch2);
          default:
            return scanCharacterEscape(
              /*atomEscape*/
              false
            );
        }
      } else if (ch === charCodeChecked(pos + 1)) {
        switch (ch) {
          case 38 /* ampersand */:
          case 33 /* exclamation */:
          case 35 /* hash */:
          case 37 /* percent */:
          case 42 /* asterisk */:
          case 43 /* plus */:
          case 44 /* comma */:
          case 46 /* dot */:
          case 58 /* colon */:
          case 59 /* semicolon */:
          case 60 /* lessThan */:
          case 61 /* equals */:
          case 62 /* greaterThan */:
          case 63 /* question */:
          case 64 /* at */:
          case 96 /* backtick */:
          case 126 /* tilde */:
            error(Diagnostics.A_character_class_must_not_contain_a_reserved_double_punctuator_Did_you_mean_to_escape_it_with_backslash, pos, 2);
            pos += 2;
            return text.substring(pos - 2, pos);
        }
      }
      switch (ch) {
        case 47 /* slash */:
        case 40 /* openParen */:
        case 41 /* closeParen */:
        case 91 /* openBracket */:
        case 93 /* closeBracket */:
        case 123 /* openBrace */:
        case 125 /* closeBrace */:
        case 45 /* minus */:
        case 124 /* bar */:
          error(Diagnostics.Unexpected_0_Did_you_mean_to_escape_it_with_backslash, pos, 1, String.fromCharCode(ch));
          pos++;
          return String.fromCharCode(ch);
      }
      return scanSourceCharacter();
    }
    function scanClassAtom() {
      if (charCodeChecked(pos) === 92 /* backslash */) {
        pos++;
        const ch = charCodeChecked(pos);
        switch (ch) {
          case 98 /* b */:
            pos++;
            return "\b";
          case 45 /* minus */:
            pos++;
            return String.fromCharCode(ch);
          default:
            if (scanCharacterClassEscape()) {
              return "";
            }
            return scanCharacterEscape(
              /*atomEscape*/
              false
            );
        }
      } else {
        return scanSourceCharacter();
      }
    }
    function scanCharacterClassEscape() {
      Debug.assertEqual(charCodeUnchecked(pos - 1), 92 /* backslash */);
      let isCharacterComplement = false;
      const start2 = pos - 1;
      const ch = charCodeChecked(pos);
      switch (ch) {
        case 100 /* d */:
        case 68 /* D */:
        case 115 /* s */:
        case 83 /* S */:
        case 119 /* w */:
        case 87 /* W */:
          pos++;
          return true;
        case 80 /* P */:
          isCharacterComplement = true;
        // falls through
        case 112 /* p */:
          pos++;
          if (charCodeChecked(pos) === 123 /* openBrace */) {
            pos++;
            const propertyNameOrValueStart = pos;
            const propertyNameOrValue = scanWordCharacters();
            if (charCodeChecked(pos) === 61 /* equals */) {
              const propertyName = nonBinaryUnicodeProperties.get(propertyNameOrValue);
              if (pos === propertyNameOrValueStart) {
                error(Diagnostics.Expected_a_Unicode_property_name);
              } else if (propertyName === void 0) {
                error(Diagnostics.Unknown_Unicode_property_name, propertyNameOrValueStart, pos - propertyNameOrValueStart);
                const suggestion = getSpellingSuggestion(propertyNameOrValue, nonBinaryUnicodeProperties.keys(), identity);
                if (suggestion) {
                  error(Diagnostics.Did_you_mean_0, propertyNameOrValueStart, pos - propertyNameOrValueStart, suggestion);
                }
              }
              pos++;
              const propertyValueStart = pos;
              const propertyValue = scanWordCharacters();
              if (pos === propertyValueStart) {
                error(Diagnostics.Expected_a_Unicode_property_value);
              } else if (propertyName !== void 0 && !valuesOfNonBinaryUnicodeProperties[propertyName].has(propertyValue)) {
                error(Diagnostics.Unknown_Unicode_property_value, propertyValueStart, pos - propertyValueStart);
                const suggestion = getSpellingSuggestion(propertyValue, valuesOfNonBinaryUnicodeProperties[propertyName], identity);
                if (suggestion) {
                  error(Diagnostics.Did_you_mean_0, propertyValueStart, pos - propertyValueStart, suggestion);
                }
              }
            } else {
              if (pos === propertyNameOrValueStart) {
                error(Diagnostics.Expected_a_Unicode_property_name_or_value);
              } else if (binaryUnicodePropertiesOfStrings.has(propertyNameOrValue)) {
                if (!unicodeSetsMode) {
                  error(Diagnostics.Any_Unicode_property_that_would_possibly_match_more_than_a_single_character_is_only_available_when_the_Unicode_Sets_v_flag_is_set, propertyNameOrValueStart, pos - propertyNameOrValueStart);
                } else if (isCharacterComplement) {
                  error(Diagnostics.Anything_that_would_possibly_match_more_than_a_single_character_is_invalid_inside_a_negated_character_class, propertyNameOrValueStart, pos - propertyNameOrValueStart);
                } else {
                  mayContainStrings = true;
                }
              } else if (!valuesOfNonBinaryUnicodeProperties.General_Category.has(propertyNameOrValue) && !binaryUnicodeProperties.has(propertyNameOrValue)) {
                error(Diagnostics.Unknown_Unicode_property_name_or_value, propertyNameOrValueStart, pos - propertyNameOrValueStart);
                const suggestion = getSpellingSuggestion(propertyNameOrValue, [...valuesOfNonBinaryUnicodeProperties.General_Category, ...binaryUnicodeProperties, ...binaryUnicodePropertiesOfStrings], identity);
                if (suggestion) {
                  error(Diagnostics.Did_you_mean_0, propertyNameOrValueStart, pos - propertyNameOrValueStart, suggestion);
                }
              }
            }
            scanExpectedChar(125 /* closeBrace */);
            if (!anyUnicodeMode) {
              error(Diagnostics.Unicode_property_value_expressions_are_only_available_when_the_Unicode_u_flag_or_the_Unicode_Sets_v_flag_is_set, start2, pos - start2);
            }
          } else if (anyUnicodeModeOrNonAnnexB) {
            error(Diagnostics._0_must_be_followed_by_a_Unicode_property_value_expression_enclosed_in_braces, pos - 2, 2, String.fromCharCode(ch));
          } else {
            pos--;
            return false;
          }
          return true;
      }
      return false;
    }
    function scanWordCharacters() {
      let value = "";
      while (true) {
        const ch = charCodeChecked(pos);
        if (ch === -1 /* EOF */ || !isWordCharacter(ch)) {
          break;
        }
        value += String.fromCharCode(ch);
        pos++;
      }
      return value;
    }
    function scanSourceCharacter() {
      const size = anyUnicodeMode ? charSize(codePointChecked(pos)) : 1;
      pos += size;
      return size > 0 ? text.substring(pos - size, pos) : "";
    }
    function scanExpectedChar(ch) {
      if (charCodeChecked(pos) === ch) {
        pos++;
      } else {
        error(Diagnostics._0_expected, pos, 0, String.fromCharCode(ch));
      }
    }
    scanDisjunction(
      /*isInGroup*/
      false
    );
    forEach(groupNameReferences, (reference) => {
      if (!(groupSpecifiers == null ? void 0 : groupSpecifiers.has(reference.name))) {
        error(Diagnostics.There_is_no_capturing_group_named_0_in_this_regular_expression, reference.pos, reference.end - reference.pos, reference.name);
        if (groupSpecifiers) {
          const suggestion = getSpellingSuggestion(reference.name, groupSpecifiers, identity);
          if (suggestion) {
            error(Diagnostics.Did_you_mean_0, reference.pos, reference.end - reference.pos, suggestion);
          }
        }
      }
    });
    forEach(decimalEscapes, (escape) => {
      if (escape.value > numberOfCapturingGroups) {
        if (numberOfCapturingGroups) {
          error(Diagnostics.This_backreference_refers_to_a_group_that_does_not_exist_There_are_only_0_capturing_groups_in_this_regular_expression, escape.pos, escape.end - escape.pos, numberOfCapturingGroups);
        } else {
          error(Diagnostics.This_backreference_refers_to_a_group_that_does_not_exist_There_are_no_capturing_groups_in_this_regular_expression, escape.pos, escape.end - escape.pos);
        }
      }
    });
  }
  function checkRegularExpressionFlagAvailability(flag, size) {
    const availableFrom = regExpFlagToFirstAvailableLanguageVersion.get(flag);
    if (availableFrom && languageVersion < availableFrom) {
      error(Diagnostics.This_regular_expression_flag_is_only_available_when_targeting_0_or_later, pos, size, getNameOfScriptTarget(availableFrom));
    }
  }
  function appendIfCommentDirective(commentDirectives2, text2, commentDirectiveRegEx, lineStart) {
    const type = getDirectiveFromComment(text2.trimStart(), commentDirectiveRegEx);
    if (type === void 0) {
      return commentDirectives2;
    }
    return append(
      commentDirectives2,
      {
        range: { pos: lineStart, end: pos },
        type
      }
    );
  }
  function getDirectiveFromComment(text2, commentDirectiveRegEx) {
    const match = commentDirectiveRegEx.exec(text2);
    if (!match) {
      return void 0;
    }
    switch (match[1]) {
      case "ts-expect-error":
        return 0 /* ExpectError */;
      case "ts-ignore":
        return 1 /* Ignore */;
    }
    return void 0;
  }
  function reScanTemplateToken(isTaggedTemplate) {
    pos = tokenStart;
    return token = scanTemplateAndSetTokenValue(!isTaggedTemplate);
  }
  function reScanTemplateHeadOrNoSubstitutionTemplate() {
    pos = tokenStart;
    return token = scanTemplateAndSetTokenValue(
      /*shouldEmitInvalidEscapeError*/
      true
    );
  }
  function reScanJsxToken(allowMultilineJsxText = true) {
    pos = tokenStart = fullStartPos;
    return token = scanJsxToken(allowMultilineJsxText);
  }
  function reScanLessThanToken() {
    if (token === 48 /* LessThanLessThanToken */) {
      pos = tokenStart + 1;
      return token = 30 /* LessThanToken */;
    }
    return token;
  }
  function reScanHashToken() {
    if (token === 81 /* PrivateIdentifier */) {
      pos = tokenStart + 1;
      return token = 63 /* HashToken */;
    }
    return token;
  }
  function reScanQuestionToken() {
    Debug.assert(token === 61 /* QuestionQuestionToken */, "'reScanQuestionToken' should only be called on a '??'");
    pos = tokenStart + 1;
    return token = 58 /* QuestionToken */;
  }
  function scanJsxToken(allowMultilineJsxText = true) {
    fullStartPos = tokenStart = pos;
    if (pos >= end) {
      return token = 1 /* EndOfFileToken */;
    }
    let char = charCodeUnchecked(pos);
    if (char === 60 /* lessThan */) {
      if (charCodeUnchecked(pos + 1) === 47 /* slash */) {
        pos += 2;
        return token = 31 /* LessThanSlashToken */;
      }
      pos++;
      return token = 30 /* LessThanToken */;
    }
    if (char === 123 /* openBrace */) {
      pos++;
      return token = 19 /* OpenBraceToken */;
    }
    let firstNonWhitespace = 0;
    while (pos < end) {
      char = charCodeUnchecked(pos);
      if (char === 123 /* openBrace */) {
        break;
      }
      if (char === 60 /* lessThan */) {
        if (isConflictMarkerTrivia(text, pos)) {
          pos = scanConflictMarkerTrivia(text, pos, error);
          return token = 7 /* ConflictMarkerTrivia */;
        }
        break;
      }
      if (char === 62 /* greaterThan */) {
        error(Diagnostics.Unexpected_token_Did_you_mean_or_gt, pos, 1);
      }
      if (char === 125 /* closeBrace */) {
        error(Diagnostics.Unexpected_token_Did_you_mean_or_rbrace, pos, 1);
      }
      if (isLineBreak(char) && firstNonWhitespace === 0) {
        firstNonWhitespace = -1;
      } else if (!allowMultilineJsxText && isLineBreak(char) && firstNonWhitespace > 0) {
        break;
      } else if (!isWhiteSpaceLike(char)) {
        firstNonWhitespace = pos;
      }
      pos++;
    }
    tokenValue = text.substring(fullStartPos, pos);
    return firstNonWhitespace === -1 ? 13 /* JsxTextAllWhiteSpaces */ : 12 /* JsxText */;
  }
  function scanJsxIdentifier() {
    if (tokenIsIdentifierOrKeyword(token)) {
      while (pos < end) {
        const ch = charCodeUnchecked(pos);
        if (ch === 45 /* minus */) {
          tokenValue += "-";
          pos++;
          continue;
        }
        const oldPos = pos;
        tokenValue += scanIdentifierParts();
        if (pos === oldPos) {
          break;
        }
      }
      return getIdentifierToken();
    }
    return token;
  }
  function scanJsxAttributeValue() {
    fullStartPos = pos;
    switch (charCodeUnchecked(pos)) {
      case 34 /* doubleQuote */:
      case 39 /* singleQuote */:
        tokenValue = scanString(
          /*jsxAttributeString*/
          true
        );
        return token = 11 /* StringLiteral */;
      default:
        return scan();
    }
  }
  function reScanJsxAttributeValue() {
    pos = tokenStart = fullStartPos;
    return scanJsxAttributeValue();
  }
  function scanJSDocCommentTextToken(inBackticks) {
    fullStartPos = tokenStart = pos;
    tokenFlags = 0 /* None */;
    if (pos >= end) {
      return token = 1 /* EndOfFileToken */;
    }
    for (let ch = charCodeUnchecked(pos); pos < end && (!isLineBreak(ch) && ch !== 96 /* backtick */); ch = codePointUnchecked(++pos)) {
      if (!inBackticks) {
        if (ch === 123 /* openBrace */) {
          break;
        } else if (ch === 64 /* at */ && pos - 1 >= 0 && isWhiteSpaceSingleLine(charCodeUnchecked(pos - 1)) && !(pos + 1 < end && isWhiteSpaceLike(charCodeUnchecked(pos + 1)))) {
          break;
        }
      }
    }
    if (pos === tokenStart) {
      return scanJsDocToken();
    }
    tokenValue = text.substring(tokenStart, pos);
    return token = 82 /* JSDocCommentTextToken */;
  }
  function scanJsDocToken() {
    fullStartPos = tokenStart = pos;
    tokenFlags = 0 /* None */;
    if (pos >= end) {
      return token = 1 /* EndOfFileToken */;
    }
    const ch = codePointUnchecked(pos);
    pos += charSize(ch);
    switch (ch) {
      case 9 /* tab */:
      case 11 /* verticalTab */:
      case 12 /* formFeed */:
      case 32 /* space */:
        while (pos < end && isWhiteSpaceSingleLine(charCodeUnchecked(pos))) {
          pos++;
        }
        return token = 5 /* WhitespaceTrivia */;
      case 64 /* at */:
        return token = 60 /* AtToken */;
      case 13 /* carriageReturn */:
        if (charCodeUnchecked(pos) === 10 /* lineFeed */) {
          pos++;
        }
      // falls through
      case 10 /* lineFeed */:
        tokenFlags |= 1 /* PrecedingLineBreak */;
        return token = 4 /* NewLineTrivia */;
      case 42 /* asterisk */:
        return token = 42 /* AsteriskToken */;
      case 123 /* openBrace */:
        return token = 19 /* OpenBraceToken */;
      case 125 /* closeBrace */:
        return token = 20 /* CloseBraceToken */;
      case 91 /* openBracket */:
        return token = 23 /* OpenBracketToken */;
      case 93 /* closeBracket */:
        return token = 24 /* CloseBracketToken */;
      case 40 /* openParen */:
        return token = 21 /* OpenParenToken */;
      case 41 /* closeParen */:
        return token = 22 /* CloseParenToken */;
      case 60 /* lessThan */:
        return token = 30 /* LessThanToken */;
      case 62 /* greaterThan */:
        return token = 32 /* GreaterThanToken */;
      case 61 /* equals */:
        return token = 64 /* EqualsToken */;
      case 44 /* comma */:
        return token = 28 /* CommaToken */;
      case 46 /* dot */:
        return token = 25 /* DotToken */;
      case 96 /* backtick */:
        return token = 62 /* BacktickToken */;
      case 35 /* hash */:
        return token = 63 /* HashToken */;
      case 92 /* backslash */:
        pos--;
        const extendedCookedChar = peekExtendedUnicodeEscape();
        if (extendedCookedChar >= 0 && isIdentifierStart(extendedCookedChar, languageVersion)) {
          tokenValue = scanExtendedUnicodeEscape(
            /*shouldEmitInvalidEscapeError*/
            true
          ) + scanIdentifierParts();
          return token = getIdentifierToken();
        }
        const cookedChar = peekUnicodeEscape();
        if (cookedChar >= 0 && isIdentifierStart(cookedChar, languageVersion)) {
          pos += 6;
          tokenFlags |= 1024 /* UnicodeEscape */;
          tokenValue = String.fromCharCode(cookedChar) + scanIdentifierParts();
          return token = getIdentifierToken();
        }
        pos++;
        return token = 0 /* Unknown */;
    }
    if (isIdentifierStart(ch, languageVersion)) {
      let char = ch;
      while (pos < end && isIdentifierPart(char = codePointUnchecked(pos), languageVersion) || char === 45 /* minus */) pos += charSize(char);
      tokenValue = text.substring(tokenStart, pos);
      if (char === 92 /* backslash */) {
        tokenValue += scanIdentifierParts();
      }
      return token = getIdentifierToken();
    } else {
      return token = 0 /* Unknown */;
    }
  }
  function speculationHelper(callback, isLookahead) {
    const savePos = pos;
    const saveStartPos = fullStartPos;
    const saveTokenPos = tokenStart;
    const saveToken = token;
    const saveTokenValue = tokenValue;
    const saveTokenFlags = tokenFlags;
    const result = callback();
    if (!result || isLookahead) {
      pos = savePos;
      fullStartPos = saveStartPos;
      tokenStart = saveTokenPos;
      token = saveToken;
      tokenValue = saveTokenValue;
      tokenFlags = saveTokenFlags;
    }
    return result;
  }
  function scanRange(start2, length3, callback) {
    const saveEnd = end;
    const savePos = pos;
    const saveStartPos = fullStartPos;
    const saveTokenPos = tokenStart;
    const saveToken = token;
    const saveTokenValue = tokenValue;
    const saveTokenFlags = tokenFlags;
    const saveErrorExpectations = commentDirectives;
    setText(text, start2, length3);
    const result = callback();
    end = saveEnd;
    pos = savePos;
    fullStartPos = saveStartPos;
    tokenStart = saveTokenPos;
    token = saveToken;
    tokenValue = saveTokenValue;
    tokenFlags = saveTokenFlags;
    commentDirectives = saveErrorExpectations;
    return result;
  }
  function lookAhead(callback) {
    return speculationHelper(
      callback,
      /*isLookahead*/
      true
    );
  }
  function tryScan(callback) {
    return speculationHelper(
      callback,
      /*isLookahead*/
      false
    );
  }
  function getText() {
    return text;
  }
  function clearCommentDirectives() {
    commentDirectives = void 0;
  }
  function setText(newText, start2, length3) {
    text = newText || "";
    end = length3 === void 0 ? text.length : start2 + length3;
    resetTokenState(start2 || 0);
  }
  function setOnError(errorCallback) {
    onError = errorCallback;
  }
  function setScriptTarget(scriptTarget) {
    languageVersion = scriptTarget;
  }
  function setLanguageVariant(variant) {
    languageVariant = variant;
  }
  function setScriptKind(kind) {
    scriptKind = kind;
  }
  function setJSDocParsingMode(kind) {
    jsDocParsingMode = kind;
  }
  function resetTokenState(position) {
    Debug.assert(position >= 0);
    pos = position;
    fullStartPos = position;
    tokenStart = position;
    token = 0 /* Unknown */;
    tokenValue = void 0;
    tokenFlags = 0 /* None */;
  }
  function setSkipJsDocLeadingAsterisks(skip) {
    skipJsDocLeadingAsterisks += skip ? 1 : -1;
  }
}
function codePointAt(s, i) {
  return s.codePointAt(i);
}
function charSize(ch) {
  if (ch >= 65536) {
    return 2;
  }
  if (ch === -1 /* EOF */) {
    return 0;
  }
  return 1;
}
function utf16EncodeAsStringFallback(codePoint) {
  Debug.assert(0 <= codePoint && codePoint <= 1114111);
  if (codePoint <= 65535) {
    return String.fromCharCode(codePoint);
  }
  const codeUnit1 = Math.floor((codePoint - 65536) / 1024) + 55296;
  const codeUnit2 = (codePoint - 65536) % 1024 + 56320;
  return String.fromCharCode(codeUnit1, codeUnit2);
}
var utf16EncodeAsStringWorker = String.fromCodePoint ? (codePoint) => String.fromCodePoint(codePoint) : utf16EncodeAsStringFallback;
function utf16EncodeAsString(codePoint) {
  return utf16EncodeAsStringWorker(codePoint);
}
var nonBinaryUnicodeProperties = new Map(Object.entries({
  General_Category: "General_Category",
  gc: "General_Category",
  Script: "Script",
  sc: "Script",
  Script_Extensions: "Script_Extensions",
  scx: "Script_Extensions"
}));
var binaryUnicodeProperties = /* @__PURE__ */ new Set(["ASCII", "ASCII_Hex_Digit", "AHex", "Alphabetic", "Alpha", "Any", "Assigned", "Bidi_Control", "Bidi_C", "Bidi_Mirrored", "Bidi_M", "Case_Ignorable", "CI", "Cased", "Changes_When_Casefolded", "CWCF", "Changes_When_Casemapped", "CWCM", "Changes_When_Lowercased", "CWL", "Changes_When_NFKC_Casefolded", "CWKCF", "Changes_When_Titlecased", "CWT", "Changes_When_Uppercased", "CWU", "Dash", "Default_Ignorable_Code_Point", "DI", "Deprecated", "Dep", "Diacritic", "Dia", "Emoji", "Emoji_Component", "EComp", "Emoji_Modifier", "EMod", "Emoji_Modifier_Base", "EBase", "Emoji_Presentation", "EPres", "Extended_Pictographic", "ExtPict", "Extender", "Ext", "Grapheme_Base", "Gr_Base", "Grapheme_Extend", "Gr_Ext", "Hex_Digit", "Hex", "IDS_Binary_Operator", "IDSB", "IDS_Trinary_Operator", "IDST", "ID_Continue", "IDC", "ID_Start", "IDS", "Ideographic", "Ideo", "Join_Control", "Join_C", "Logical_Order_Exception", "LOE", "Lowercase", "Lower", "Math", "Noncharacter_Code_Point", "NChar", "Pattern_Syntax", "Pat_Syn", "Pattern_White_Space", "Pat_WS", "Quotation_Mark", "QMark", "Radical", "Regional_Indicator", "RI", "Sentence_Terminal", "STerm", "Soft_Dotted", "SD", "Terminal_Punctuation", "Term", "Unified_Ideograph", "UIdeo", "Uppercase", "Upper", "Variation_Selector", "VS", "White_Space", "space", "XID_Continue", "XIDC", "XID_Start", "XIDS"]);
var binaryUnicodePropertiesOfStrings = /* @__PURE__ */ new Set(["Basic_Emoji", "Emoji_Keycap_Sequence", "RGI_Emoji_Modifier_Sequence", "RGI_Emoji_Flag_Sequence", "RGI_Emoji_Tag_Sequence", "RGI_Emoji_ZWJ_Sequence", "RGI_Emoji"]);
var valuesOfNonBinaryUnicodeProperties = {
  General_Category: /* @__PURE__ */ new Set(["C", "Other", "Cc", "Control", "cntrl", "Cf", "Format", "Cn", "Unassigned", "Co", "Private_Use", "Cs", "Surrogate", "L", "Letter", "LC", "Cased_Letter", "Ll", "Lowercase_Letter", "Lm", "Modifier_Letter", "Lo", "Other_Letter", "Lt", "Titlecase_Letter", "Lu", "Uppercase_Letter", "M", "Mark", "Combining_Mark", "Mc", "Spacing_Mark", "Me", "Enclosing_Mark", "Mn", "Nonspacing_Mark", "N", "Number", "Nd", "Decimal_Number", "digit", "Nl", "Letter_Number", "No", "Other_Number", "P", "Punctuation", "punct", "Pc", "Connector_Punctuation", "Pd", "Dash_Punctuation", "Pe", "Close_Punctuation", "Pf", "Final_Punctuation", "Pi", "Initial_Punctuation", "Po", "Other_Punctuation", "Ps", "Open_Punctuation", "S", "Symbol", "Sc", "Currency_Symbol", "Sk", "Modifier_Symbol", "Sm", "Math_Symbol", "So", "Other_Symbol", "Z", "Separator", "Zl", "Line_Separator", "Zp", "Paragraph_Separator", "Zs", "Space_Separator"]),
  Script: /* @__PURE__ */ new Set(["Adlm", "Adlam", "Aghb", "Caucasian_Albanian", "Ahom", "Arab", "Arabic", "Armi", "Imperial_Aramaic", "Armn", "Armenian", "Avst", "Avestan", "Bali", "Balinese", "Bamu", "Bamum", "Bass", "Bassa_Vah", "Batk", "Batak", "Beng", "Bengali", "Bhks", "Bhaiksuki", "Bopo", "Bopomofo", "Brah", "Brahmi", "Brai", "Braille", "Bugi", "Buginese", "Buhd", "Buhid", "Cakm", "Chakma", "Cans", "Canadian_Aboriginal", "Cari", "Carian", "Cham", "Cher", "Cherokee", "Chrs", "Chorasmian", "Copt", "Coptic", "Qaac", "Cpmn", "Cypro_Minoan", "Cprt", "Cypriot", "Cyrl", "Cyrillic", "Deva", "Devanagari", "Diak", "Dives_Akuru", "Dogr", "Dogra", "Dsrt", "Deseret", "Dupl", "Duployan", "Egyp", "Egyptian_Hieroglyphs", "Elba", "Elbasan", "Elym", "Elymaic", "Ethi", "Ethiopic", "Geor", "Georgian", "Glag", "Glagolitic", "Gong", "Gunjala_Gondi", "Gonm", "Masaram_Gondi", "Goth", "Gothic", "Gran", "Grantha", "Grek", "Greek", "Gujr", "Gujarati", "Guru", "Gurmukhi", "Hang", "Hangul", "Hani", "Han", "Hano", "Hanunoo", "Hatr", "Hatran", "Hebr", "Hebrew", "Hira", "Hiragana", "Hluw", "Anatolian_Hieroglyphs", "Hmng", "Pahawh_Hmong", "Hmnp", "Nyiakeng_Puachue_Hmong", "Hrkt", "Katakana_Or_Hiragana", "Hung", "Old_Hungarian", "Ital", "Old_Italic", "Java", "Javanese", "Kali", "Kayah_Li", "Kana", "Katakana", "Kawi", "Khar", "Kharoshthi", "Khmr", "Khmer", "Khoj", "Khojki", "Kits", "Khitan_Small_Script", "Knda", "Kannada", "Kthi", "Kaithi", "Lana", "Tai_Tham", "Laoo", "Lao", "Latn", "Latin", "Lepc", "Lepcha", "Limb", "Limbu", "Lina", "Linear_A", "Linb", "Linear_B", "Lisu", "Lyci", "Lycian", "Lydi", "Lydian", "Mahj", "Mahajani", "Maka", "Makasar", "Mand", "Mandaic", "Mani", "Manichaean", "Marc", "Marchen", "Medf", "Medefaidrin", "Mend", "Mende_Kikakui", "Merc", "Meroitic_Cursive", "Mero", "Meroitic_Hieroglyphs", "Mlym", "Malayalam", "Modi", "Mong", "Mongolian", "Mroo", "Mro", "Mtei", "Meetei_Mayek", "Mult", "Multani", "Mymr", "Myanmar", "Nagm", "Nag_Mundari", "Nand", "Nandinagari", "Narb", "Old_North_Arabian", "Nbat", "Nabataean", "Newa", "Nkoo", "Nko", "Nshu", "Nushu", "Ogam", "Ogham", "Olck", "Ol_Chiki", "Orkh", "Old_Turkic", "Orya", "Oriya", "Osge", "Osage", "Osma", "Osmanya", "Ougr", "Old_Uyghur", "Palm", "Palmyrene", "Pauc", "Pau_Cin_Hau", "Perm", "Old_Permic", "Phag", "Phags_Pa", "Phli", "Inscriptional_Pahlavi", "Phlp", "Psalter_Pahlavi", "Phnx", "Phoenician", "Plrd", "Miao", "Prti", "Inscriptional_Parthian", "Rjng", "Rejang", "Rohg", "Hanifi_Rohingya", "Runr", "Runic", "Samr", "Samaritan", "Sarb", "Old_South_Arabian", "Saur", "Saurashtra", "Sgnw", "SignWriting", "Shaw", "Shavian", "Shrd", "Sharada", "Sidd", "Siddham", "Sind", "Khudawadi", "Sinh", "Sinhala", "Sogd", "Sogdian", "Sogo", "Old_Sogdian", "Sora", "Sora_Sompeng", "Soyo", "Soyombo", "Sund", "Sundanese", "Sylo", "Syloti_Nagri", "Syrc", "Syriac", "Tagb", "Tagbanwa", "Takr", "Takri", "Tale", "Tai_Le", "Talu", "New_Tai_Lue", "Taml", "Tamil", "Tang", "Tangut", "Tavt", "Tai_Viet", "Telu", "Telugu", "Tfng", "Tifinagh", "Tglg", "Tagalog", "Thaa", "Thaana", "Thai", "Tibt", "Tibetan", "Tirh", "Tirhuta", "Tnsa", "Tangsa", "Toto", "Ugar", "Ugaritic", "Vaii", "Vai", "Vith", "Vithkuqi", "Wara", "Warang_Citi", "Wcho", "Wancho", "Xpeo", "Old_Persian", "Xsux", "Cuneiform", "Yezi", "Yezidi", "Yiii", "Yi", "Zanb", "Zanabazar_Square", "Zinh", "Inherited", "Qaai", "Zyyy", "Common", "Zzzz", "Unknown"]),
  Script_Extensions: void 0
};
valuesOfNonBinaryUnicodeProperties.Script_Extensions = valuesOfNonBinaryUnicodeProperties.Script;

// src/compiler/utilitiesPublic.ts
function isExternalModuleNameRelative(moduleName) {
  return pathIsRelative(moduleName) || isRootedDiskPath(moduleName);
}
function sortAndDeduplicateDiagnostics(diagnostics) {
  return sortAndDeduplicate(diagnostics, compareDiagnostics, diagnosticsEqualityComparer);
}
var targetToLibMap = /* @__PURE__ */ new Map([
  [99 /* ESNext */, "lib.esnext.full.d.ts"],
  [11 /* ES2024 */, "lib.es2024.full.d.ts"],
  [10 /* ES2023 */, "lib.es2023.full.d.ts"],
  [9 /* ES2022 */, "lib.es2022.full.d.ts"],
  [8 /* ES2021 */, "lib.es2021.full.d.ts"],
  [7 /* ES2020 */, "lib.es2020.full.d.ts"],
  [6 /* ES2019 */, "lib.es2019.full.d.ts"],
  [5 /* ES2018 */, "lib.es2018.full.d.ts"],
  [4 /* ES2017 */, "lib.es2017.full.d.ts"],
  [3 /* ES2016 */, "lib.es2016.full.d.ts"],
  [2 /* ES2015 */, "lib.es6.d.ts"]
  // We don't use lib.es2015.full.d.ts due to breaking change.
]);
function getDefaultLibFileName(options) {
  const target = getEmitScriptTarget(options);
  switch (target) {
    case 99 /* ESNext */:
    case 11 /* ES2024 */:
    case 10 /* ES2023 */:
    case 9 /* ES2022 */:
    case 8 /* ES2021 */:
    case 7 /* ES2020 */:
    case 6 /* ES2019 */:
    case 5 /* ES2018 */:
    case 4 /* ES2017 */:
    case 3 /* ES2016 */:
    case 2 /* ES2015 */:
      return targetToLibMap.get(target);
    default:
      return "lib.d.ts";
  }
}
function textSpanEnd(span) {
  return span.start + span.length;
}
function textSpanIsEmpty(span) {
  return span.length === 0;
}
function textSpanContainsPosition(span, position) {
  return position >= span.start && position < textSpanEnd(span);
}
function textRangeContainsPositionInclusive(range, position) {
  return position >= range.pos && position <= range.end;
}
function createTextSpan(start, length2) {
  if (start < 0) {
    throw new Error("start < 0");
  }
  if (length2 < 0) {
    throw new Error("length < 0");
  }
  return { start, length: length2 };
}
function createTextSpanFromBounds(start, end) {
  return createTextSpan(start, end - start);
}
function textChangeRangeNewSpan(range) {
  return createTextSpan(range.span.start, range.newLength);
}
function textChangeRangeIsUnchanged(range) {
  return textSpanIsEmpty(range.span) && range.newLength === 0;
}
function createTextChangeRange(span, newLength) {
  if (newLength < 0) {
    throw new Error("newLength < 0");
  }
  return { span, newLength };
}
var unchangedTextChangeRange = createTextChangeRange(createTextSpan(0, 0), 0);
function isParameterPropertyDeclaration(node, parent) {
  return isParameter(node) && hasSyntacticModifier(node, 31 /* ParameterPropertyModifier */) && parent.kind === 176 /* Constructor */;
}
function walkUpBindingElementsAndPatterns(binding) {
  let node = binding.parent;
  while (isBindingElement(node.parent)) {
    node = node.parent.parent;
  }
  return node.parent;
}
function getCombinedFlags(node, getFlags) {
  if (isBindingElement(node)) {
    node = walkUpBindingElementsAndPatterns(node);
  }
  let flags = getFlags(node);
  if (node.kind === 260 /* VariableDeclaration */) {
    node = node.parent;
  }
  if (node && node.kind === 261 /* VariableDeclarationList */) {
    flags |= getFlags(node);
    node = node.parent;
  }
  if (node && node.kind === 243 /* VariableStatement */) {
    flags |= getFlags(node);
  }
  return flags;
}
function getCombinedModifierFlags(node) {
  return getCombinedFlags(node, getEffectiveModifierFlags);
}
function getCombinedNodeFlags(node) {
  return getCombinedFlags(node, getNodeFlags);
}
function getNodeFlags(node) {
  return node.flags;
}
var supportedLocaleDirectories = ["cs", "de", "es", "fr", "it", "ja", "ko", "pl", "pt-br", "ru", "tr", "zh-cn", "zh-tw"];
function validateLocaleAndSetLanguage(locale, sys2, errors) {
  const lowerCaseLocale = locale.toLowerCase();
  const matchResult = /^([a-z]+)(?:[_-]([a-z]+))?$/.exec(lowerCaseLocale);
  if (!matchResult) {
    if (errors) {
      errors.push(createCompilerDiagnostic(Diagnostics.Locale_must_be_of_the_form_language_or_language_territory_For_example_0_or_1, "en", "ja-jp"));
    }
    return;
  }
  const language = matchResult[1];
  const territory = matchResult[2];
  if (contains(supportedLocaleDirectories, lowerCaseLocale) && !trySetLanguageAndTerritory(language, territory, errors)) {
    trySetLanguageAndTerritory(
      language,
      /*territory*/
      void 0,
      errors
    );
  }
  setUILocale(locale);
  function trySetLanguageAndTerritory(language2, territory2, errors2) {
    const compilerFilePath = normalizePath(sys2.getExecutingFilePath());
    const containingDirectoryPath = getDirectoryPath(compilerFilePath);
    let filePath = combinePaths(containingDirectoryPath, language2);
    if (territory2) {
      filePath = filePath + "-" + territory2;
    }
    filePath = sys2.resolvePath(combinePaths(filePath, "diagnosticMessages.generated.json"));
    if (!sys2.fileExists(filePath)) {
      return false;
    }
    let fileContents = "";
    try {
      fileContents = sys2.readFile(filePath);
    } catch {
      if (errors2) {
        errors2.push(createCompilerDiagnostic(Diagnostics.Unable_to_open_file_0, filePath));
      }
      return false;
    }
    try {
      setLocalizedDiagnosticMessages(JSON.parse(fileContents));
    } catch {
      if (errors2) {
        errors2.push(createCompilerDiagnostic(Diagnostics.Corrupted_locale_file_0, filePath));
      }
      return false;
    }
    return true;
  }
}
function getOriginalNode(node, nodeTest) {
  if (node) {
    while (node.original !== void 0) {
      node = node.original;
    }
  }
  if (!node || !nodeTest) {
    return node;
  }
  return nodeTest(node) ? node : void 0;
}
function findAncestor(node, callback) {
  while (node) {
    const result = callback(node);
    if (result === "quit") {
      return void 0;
    } else if (result) {
      return node;
    }
    node = node.parent;
  }
  return void 0;
}
function isParseTreeNode(node) {
  return (node.flags & 16 /* Synthesized */) === 0;
}
function getParseTreeNode(node, nodeTest) {
  if (node === void 0 || isParseTreeNode(node)) {
    return node;
  }
  node = node.original;
  while (node) {
    if (isParseTreeNode(node)) {
      return !nodeTest || nodeTest(node) ? node : void 0;
    }
    node = node.original;
  }
}
function escapeLeadingUnderscores(identifier) {
  return identifier.length >= 2 && identifier.charCodeAt(0) === 95 /* _ */ && identifier.charCodeAt(1) === 95 /* _ */ ? "_" + identifier : identifier;
}
function unescapeLeadingUnderscores(identifier) {
  const id = identifier;
  return id.length >= 3 && id.charCodeAt(0) === 95 /* _ */ && id.charCodeAt(1) === 95 /* _ */ && id.charCodeAt(2) === 95 /* _ */ ? id.substr(1) : id;
}
function idText(identifierOrPrivateName) {
  return unescapeLeadingUnderscores(identifierOrPrivateName.escapedText);
}
function identifierToKeywordKind(node) {
  const token = stringToToken(node.escapedText);
  return token ? tryCast(token, isKeyword) : void 0;
}
function symbolName(symbol) {
  if (symbol.valueDeclaration && isPrivateIdentifierClassElementDeclaration(symbol.valueDeclaration)) {
    return idText(symbol.valueDeclaration.name);
  }
  return unescapeLeadingUnderscores(symbol.escapedName);
}
function nameForNamelessJSDocTypedef(declaration) {
  const hostNode = declaration.parent.parent;
  if (!hostNode) {
    return void 0;
  }
  if (isDeclaration(hostNode)) {
    return getDeclarationIdentifier(hostNode);
  }
  switch (hostNode.kind) {
    case 243 /* VariableStatement */:
      if (hostNode.declarationList && hostNode.declarationList.declarations[0]) {
        return getDeclarationIdentifier(hostNode.declarationList.declarations[0]);
      }
      break;
    case 244 /* ExpressionStatement */:
      let expr = hostNode.expression;
      if (expr.kind === 226 /* BinaryExpression */ && expr.operatorToken.kind === 64 /* EqualsToken */) {
        expr = expr.left;
      }
      switch (expr.kind) {
        case 211 /* PropertyAccessExpression */:
          return expr.name;
        case 212 /* ElementAccessExpression */:
          const arg = expr.argumentExpression;
          if (isIdentifier(arg)) {
            return arg;
          }
      }
      break;
    case 217 /* ParenthesizedExpression */: {
      return getDeclarationIdentifier(hostNode.expression);
    }
    case 256 /* LabeledStatement */: {
      if (isDeclaration(hostNode.statement) || isExpression(hostNode.statement)) {
        return getDeclarationIdentifier(hostNode.statement);
      }
      break;
    }
  }
}
function getDeclarationIdentifier(node) {
  const name = getNameOfDeclaration(node);
  return name && isIdentifier(name) ? name : void 0;
}
function nodeHasName(statement, name) {
  if (isNamedDeclaration(statement) && isIdentifier(statement.name) && idText(statement.name) === idText(name)) {
    return true;
  }
  if (isVariableStatement(statement) && some(statement.declarationList.declarations, (d) => nodeHasName(d, name))) {
    return true;
  }
  return false;
}
function getNameOfJSDocTypedef(declaration) {
  return declaration.name || nameForNamelessJSDocTypedef(declaration);
}
function isNamedDeclaration(node) {
  return !!node.name;
}
function getNonAssignedNameOfDeclaration(declaration) {
  switch (declaration.kind) {
    case 80 /* Identifier */:
      return declaration;
    case 348 /* JSDocPropertyTag */:
    case 341 /* JSDocParameterTag */: {
      const { name } = declaration;
      if (name.kind === 166 /* QualifiedName */) {
        return name.right;
      }
      break;
    }
    case 213 /* CallExpression */:
    case 226 /* BinaryExpression */: {
      const expr2 = declaration;
      switch (getAssignmentDeclarationKind(expr2)) {
        case 1 /* ExportsProperty */:
        case 4 /* ThisProperty */:
        case 5 /* Property */:
        case 3 /* PrototypeProperty */:
          return getElementOrPropertyAccessArgumentExpressionOrName(expr2.left);
        case 7 /* ObjectDefinePropertyValue */:
        case 8 /* ObjectDefinePropertyExports */:
        case 9 /* ObjectDefinePrototypeProperty */:
          return expr2.arguments[1];
        default:
          return void 0;
      }
    }
    case 346 /* JSDocTypedefTag */:
      return getNameOfJSDocTypedef(declaration);
    case 340 /* JSDocEnumTag */:
      return nameForNamelessJSDocTypedef(declaration);
    case 277 /* ExportAssignment */: {
      const { expression } = declaration;
      return isIdentifier(expression) ? expression : void 0;
    }
    case 212 /* ElementAccessExpression */:
      const expr = declaration;
      if (isBindableStaticElementAccessExpression(expr)) {
        return expr.argumentExpression;
      }
  }
  return declaration.name;
}
function getNameOfDeclaration(declaration) {
  if (declaration === void 0) return void 0;
  return getNonAssignedNameOfDeclaration(declaration) || (isFunctionExpression(declaration) || isArrowFunction(declaration) || isClassExpression(declaration) ? getAssignedName(declaration) : void 0);
}
function getAssignedName(node) {
  if (!node.parent) {
    return void 0;
  } else if (isPropertyAssignment(node.parent) || isBindingElement(node.parent)) {
    return node.parent.name;
  } else if (isBinaryExpression(node.parent) && node === node.parent.right) {
    if (isIdentifier(node.parent.left)) {
      return node.parent.left;
    } else if (isAccessExpression(node.parent.left)) {
      return getElementOrPropertyAccessArgumentExpressionOrName(node.parent.left);
    }
  } else if (isVariableDeclaration(node.parent) && isIdentifier(node.parent.name)) {
    return node.parent.name;
  }
}
function getDecorators(node) {
  if (hasDecorators(node)) {
    return filter(node.modifiers, isDecorator);
  }
}
function getModifiers(node) {
  if (hasSyntacticModifier(node, 98303 /* Modifier */)) {
    return filter(node.modifiers, isModifier);
  }
}
function getJSDocParameterTagsWorker(param, noCache) {
  if (param.name) {
    if (isIdentifier(param.name)) {
      const name = param.name.escapedText;
      return getJSDocTagsWorker(param.parent, noCache).filter((tag) => isJSDocParameterTag(tag) && isIdentifier(tag.name) && tag.name.escapedText === name);
    } else {
      const i = param.parent.parameters.indexOf(param);
      Debug.assert(i > -1, "Parameters should always be in their parents' parameter list");
      const paramTags = getJSDocTagsWorker(param.parent, noCache).filter(isJSDocParameterTag);
      if (i < paramTags.length) {
        return [paramTags[i]];
      }
    }
  }
  return emptyArray;
}
function getJSDocParameterTags(param) {
  return getJSDocParameterTagsWorker(
    param,
    /*noCache*/
    false
  );
}
function getJSDocParameterTagsNoCache(param) {
  return getJSDocParameterTagsWorker(
    param,
    /*noCache*/
    true
  );
}
function getJSDocTypeParameterTagsWorker(param, noCache) {
  const name = param.name.escapedText;
  return getJSDocTagsWorker(param.parent, noCache).filter((tag) => isJSDocTemplateTag(tag) && tag.typeParameters.some((tp) => tp.name.escapedText === name));
}
function getJSDocTypeParameterTags(param) {
  return getJSDocTypeParameterTagsWorker(
    param,
    /*noCache*/
    false
  );
}
function getJSDocTypeParameterTagsNoCache(param) {
  return getJSDocTypeParameterTagsWorker(
    param,
    /*noCache*/
    true
  );
}
function hasJSDocParameterTags(node) {
  return !!getFirstJSDocTag(node, isJSDocParameterTag);
}
function getJSDocAugmentsTag(node) {
  return getFirstJSDocTag(node, isJSDocAugmentsTag);
}
function getJSDocImplementsTags(node) {
  return getAllJSDocTags(node, isJSDocImplementsTag);
}
function getJSDocClassTag(node) {
  return getFirstJSDocTag(node, isJSDocClassTag);
}
function getJSDocPublicTagNoCache(node) {
  return getFirstJSDocTag(
    node,
    isJSDocPublicTag,
    /*noCache*/
    true
  );
}
function getJSDocPrivateTagNoCache(node) {
  return getFirstJSDocTag(
    node,
    isJSDocPrivateTag,
    /*noCache*/
    true
  );
}
function getJSDocProtectedTagNoCache(node) {
  return getFirstJSDocTag(
    node,
    isJSDocProtectedTag,
    /*noCache*/
    true
  );
}
function getJSDocReadonlyTagNoCache(node) {
  return getFirstJSDocTag(
    node,
    isJSDocReadonlyTag,
    /*noCache*/
    true
  );
}
function getJSDocOverrideTagNoCache(node) {
  return getFirstJSDocTag(
    node,
    isJSDocOverrideTag,
    /*noCache*/
    true
  );
}
function getJSDocDeprecatedTag(node) {
  return getFirstJSDocTag(node, isJSDocDeprecatedTag);
}
function getJSDocDeprecatedTagNoCache(node) {
  return getFirstJSDocTag(
    node,
    isJSDocDeprecatedTag,
    /*noCache*/
    true
  );
}
function getJSDocEnumTag(node) {
  return getFirstJSDocTag(node, isJSDocEnumTag);
}
function getJSDocThisTag(node) {
  return getFirstJSDocTag(node, isJSDocThisTag);
}
function getJSDocReturnTag(node) {
  return getFirstJSDocTag(node, isJSDocReturnTag);
}
function getJSDocSatisfiesTag(node) {
  return getFirstJSDocTag(node, isJSDocSatisfiesTag);
}
function getJSDocTypeTag(node) {
  const tag = getFirstJSDocTag(node, isJSDocTypeTag);
  if (tag && tag.typeExpression && tag.typeExpression.type) {
    return tag;
  }
  return void 0;
}
function getJSDocType(node) {
  let tag = getFirstJSDocTag(node, isJSDocTypeTag);
  if (!tag && isParameter(node)) {
    tag = find(getJSDocParameterTags(node), (tag2) => !!tag2.typeExpression);
  }
  return tag && tag.typeExpression && tag.typeExpression.type;
}
function getJSDocReturnType(node) {
  const returnTag = getJSDocReturnTag(node);
  if (returnTag && returnTag.typeExpression) {
    return returnTag.typeExpression.type;
  }
  const typeTag = getJSDocTypeTag(node);
  if (typeTag && typeTag.typeExpression) {
    const type = typeTag.typeExpression.type;
    if (isTypeLiteralNode(type)) {
      const sig = find(type.members, isCallSignatureDeclaration);
      return sig && sig.type;
    }
    if (isFunctionTypeNode(type) || isJSDocFunctionType(type)) {
      return type.type;
    }
  }
}
function getJSDocTagsWorker(node, noCache) {
  var _a;
  if (!canHaveJSDoc(node)) return emptyArray;
  let tags = (_a = node.jsDoc) == null ? void 0 : _a.jsDocCache;
  if (tags === void 0 || noCache) {
    const comments = getJSDocCommentsAndTags(node, noCache);
    Debug.assert(comments.length < 2 || comments[0] !== comments[1]);
    tags = flatMap(comments, (j) => isJSDoc(j) ? j.tags : j);
    if (!noCache) {
      node.jsDoc ?? (node.jsDoc = []);
      node.jsDoc.jsDocCache = tags;
    }
  }
  return tags;
}
function getJSDocTags(node) {
  return getJSDocTagsWorker(
    node,
    /*noCache*/
    false
  );
}
function getFirstJSDocTag(node, predicate, noCache) {
  return find(getJSDocTagsWorker(node, noCache), predicate);
}
function getAllJSDocTags(node, predicate) {
  return getJSDocTags(node).filter(predicate);
}
function getTextOfJSDocComment(comment) {
  return typeof comment === "string" ? comment : comment == null ? void 0 : comment.map((c) => c.kind === 321 /* JSDocText */ ? c.text : formatJSDocLink(c)).join("");
}
function formatJSDocLink(link) {
  const kind = link.kind === 324 /* JSDocLink */ ? "link" : link.kind === 325 /* JSDocLinkCode */ ? "linkcode" : "linkplain";
  const name = link.name ? entityNameToString(link.name) : "";
  const space = link.name && (link.text === "" || link.text.startsWith("://")) ? "" : " ";
  return `{@${kind} ${name}${space}${link.text}}`;
}
function getEffectiveTypeParameterDeclarations(node) {
  if (isJSDocSignature(node)) {
    if (isJSDocOverloadTag(node.parent)) {
      const jsDoc = getJSDocRoot(node.parent);
      if (jsDoc && length(jsDoc.tags)) {
        return flatMap(jsDoc.tags, (tag) => isJSDocTemplateTag(tag) ? tag.typeParameters : void 0);
      }
    }
    return emptyArray;
  }
  if (isJSDocTypeAlias(node)) {
    Debug.assert(node.parent.kind === 320 /* JSDoc */);
    return flatMap(node.parent.tags, (tag) => isJSDocTemplateTag(tag) ? tag.typeParameters : void 0);
  }
  if (node.typeParameters) {
    return node.typeParameters;
  }
  if (canHaveIllegalTypeParameters(node) && node.typeParameters) {
    return node.typeParameters;
  }
  if (isInJSFile(node)) {
    const decls = getJSDocTypeParameterDeclarations(node);
    if (decls.length) {
      return decls;
    }
    const typeTag = getJSDocType(node);
    if (typeTag && isFunctionTypeNode(typeTag) && typeTag.typeParameters) {
      return typeTag.typeParameters;
    }
  }
  return emptyArray;
}
function getEffectiveConstraintOfTypeParameter(node) {
  return node.constraint ? node.constraint : isJSDocTemplateTag(node.parent) && node === node.parent.typeParameters[0] ? node.parent.constraint : void 0;
}
function isMemberName(node) {
  return node.kind === 80 /* Identifier */ || node.kind === 81 /* PrivateIdentifier */;
}
function isGetOrSetAccessorDeclaration(node) {
  return node.kind === 178 /* SetAccessor */ || node.kind === 177 /* GetAccessor */;
}
function isPropertyAccessChain(node) {
  return isPropertyAccessExpression(node) && !!(node.flags & 64 /* OptionalChain */);
}
function isElementAccessChain(node) {
  return isElementAccessExpression(node) && !!(node.flags & 64 /* OptionalChain */);
}
function isCallChain(node) {
  return isCallExpression(node) && !!(node.flags & 64 /* OptionalChain */);
}
function isOptionalChain(node) {
  const kind = node.kind;
  return !!(node.flags & 64 /* OptionalChain */) && (kind === 211 /* PropertyAccessExpression */ || kind === 212 /* ElementAccessExpression */ || kind === 213 /* CallExpression */ || kind === 235 /* NonNullExpression */);
}
function isOptionalChainRoot(node) {
  return isOptionalChain(node) && !isNonNullExpression(node) && !!node.questionDotToken;
}
function isExpressionOfOptionalChainRoot(node) {
  return isOptionalChainRoot(node.parent) && node.parent.expression === node;
}
function isOutermostOptionalChain(node) {
  return !isOptionalChain(node.parent) || isOptionalChainRoot(node.parent) || node !== node.parent.expression;
}
function isNullishCoalesce(node) {
  return node.kind === 226 /* BinaryExpression */ && node.operatorToken.kind === 61 /* QuestionQuestionToken */;
}
function isConstTypeReference(node) {
  return isTypeReferenceNode(node) && isIdentifier(node.typeName) && node.typeName.escapedText === "const" && !node.typeArguments;
}
function skipPartiallyEmittedExpressions(node) {
  return skipOuterExpressions(node, 8 /* PartiallyEmittedExpressions */);
}
function isNonNullChain(node) {
  return isNonNullExpression(node) && !!(node.flags & 64 /* OptionalChain */);
}
function isNamedExportBindings(node) {
  return node.kind === 280 /* NamespaceExport */ || node.kind === 279 /* NamedExports */;
}
function isJSDocPropertyLikeTag(node) {
  return node.kind === 348 /* JSDocPropertyTag */ || node.kind === 341 /* JSDocParameterTag */;
}
function isNodeKind(kind) {
  return kind >= 166 /* FirstNode */;
}
function isTokenKind(kind) {
  return kind >= 0 /* FirstToken */ && kind <= 165 /* LastToken */;
}
function isNodeArray(array) {
  return hasProperty(array, "pos") && hasProperty(array, "end");
}
function isLiteralKind(kind) {
  return 9 /* FirstLiteralToken */ <= kind && kind <= 15 /* LastLiteralToken */;
}
function isLiteralExpression(node) {
  return isLiteralKind(node.kind);
}
function isLiteralExpressionOfObject(node) {
  switch (node.kind) {
    case 210 /* ObjectLiteralExpression */:
    case 209 /* ArrayLiteralExpression */:
    case 14 /* RegularExpressionLiteral */:
    case 218 /* FunctionExpression */:
    case 231 /* ClassExpression */:
      return true;
  }
  return false;
}
function isTemplateLiteralKind(kind) {
  return 15 /* FirstTemplateToken */ <= kind && kind <= 18 /* LastTemplateToken */;
}
function isTemplateMiddleOrTemplateTail(node) {
  const kind = node.kind;
  return kind === 17 /* TemplateMiddle */ || kind === 18 /* TemplateTail */;
}
function isImportOrExportSpecifier(node) {
  return isImportSpecifier(node) || isExportSpecifier(node);
}
function isTypeOnlyImportDeclaration(node) {
  switch (node.kind) {
    case 276 /* ImportSpecifier */:
      return node.isTypeOnly || node.parent.parent.isTypeOnly;
    case 274 /* NamespaceImport */:
      return node.parent.isTypeOnly;
    case 273 /* ImportClause */:
    case 271 /* ImportEqualsDeclaration */:
      return node.isTypeOnly;
  }
  return false;
}
function isTypeOnlyExportDeclaration(node) {
  switch (node.kind) {
    case 281 /* ExportSpecifier */:
      return node.isTypeOnly || node.parent.parent.isTypeOnly;
    case 278 /* ExportDeclaration */:
      return node.isTypeOnly && !!node.moduleSpecifier && !node.exportClause;
    case 280 /* NamespaceExport */:
      return node.parent.isTypeOnly;
  }
  return false;
}
function isTypeOnlyImportOrExportDeclaration(node) {
  return isTypeOnlyImportDeclaration(node) || isTypeOnlyExportDeclaration(node);
}
function isPartOfTypeOnlyImportOrExportDeclaration(node) {
  return findAncestor(node, isTypeOnlyImportOrExportDeclaration) !== void 0;
}
function isImportAttributeName(node) {
  return isStringLiteral(node) || isIdentifier(node);
}
function isGeneratedIdentifier(node) {
  var _a;
  return isIdentifier(node) && ((_a = node.emitNode) == null ? void 0 : _a.autoGenerate) !== void 0;
}
function isGeneratedPrivateIdentifier(node) {
  var _a;
  return isPrivateIdentifier(node) && ((_a = node.emitNode) == null ? void 0 : _a.autoGenerate) !== void 0;
}
function isFileLevelReservedGeneratedIdentifier(node) {
  const flags = node.emitNode.autoGenerate.flags;
  return !!(flags & 32 /* FileLevel */) && !!(flags & 16 /* Optimistic */) && !!(flags & 8 /* ReservedInNestedScopes */);
}
function isPrivateIdentifierClassElementDeclaration(node) {
  return (isPropertyDeclaration(node) || isMethodOrAccessor(node)) && isPrivateIdentifier(node.name);
}
function isPrivateIdentifierPropertyAccessExpression(node) {
  return isPropertyAccessExpression(node) && isPrivateIdentifier(node.name);
}
function isModifierKind(token) {
  switch (token) {
    case 128 /* AbstractKeyword */:
    case 129 /* AccessorKeyword */:
    case 134 /* AsyncKeyword */:
    case 87 /* ConstKeyword */:
    case 138 /* DeclareKeyword */:
    case 90 /* DefaultKeyword */:
    case 95 /* ExportKeyword */:
    case 103 /* InKeyword */:
    case 125 /* PublicKeyword */:
    case 123 /* PrivateKeyword */:
    case 124 /* ProtectedKeyword */:
    case 148 /* ReadonlyKeyword */:
    case 126 /* StaticKeyword */:
    case 147 /* OutKeyword */:
    case 164 /* OverrideKeyword */:
      return true;
  }
  return false;
}
function isParameterPropertyModifier(kind) {
  return !!(modifierToFlag(kind) & 31 /* ParameterPropertyModifier */);
}
function isClassMemberModifier(idToken) {
  return isParameterPropertyModifier(idToken) || idToken === 126 /* StaticKeyword */ || idToken === 164 /* OverrideKeyword */ || idToken === 129 /* AccessorKeyword */;
}
function isModifier(node) {
  return isModifierKind(node.kind);
}
function isEntityName(node) {
  const kind = node.kind;
  return kind === 166 /* QualifiedName */ || kind === 80 /* Identifier */;
}
function isPropertyName(node) {
  const kind = node.kind;
  return kind === 80 /* Identifier */ || kind === 81 /* PrivateIdentifier */ || kind === 11 /* StringLiteral */ || kind === 9 /* NumericLiteral */ || kind === 167 /* ComputedPropertyName */;
}
function isBindingName(node) {
  const kind = node.kind;
  return kind === 80 /* Identifier */ || kind === 206 /* ObjectBindingPattern */ || kind === 207 /* ArrayBindingPattern */;
}
function isFunctionLike(node) {
  return !!node && isFunctionLikeKind(node.kind);
}
function isFunctionLikeOrClassStaticBlockDeclaration(node) {
  return !!node && (isFunctionLikeKind(node.kind) || isClassStaticBlockDeclaration(node));
}
function isFunctionLikeDeclaration(node) {
  return node && isFunctionLikeDeclarationKind(node.kind);
}
function isBooleanLiteral(node) {
  return node.kind === 112 /* TrueKeyword */ || node.kind === 97 /* FalseKeyword */;
}
function isFunctionLikeDeclarationKind(kind) {
  switch (kind) {
    case 262 /* FunctionDeclaration */:
    case 174 /* MethodDeclaration */:
    case 176 /* Constructor */:
    case 177 /* GetAccessor */:
    case 178 /* SetAccessor */:
    case 218 /* FunctionExpression */:
    case 219 /* ArrowFunction */:
      return true;
    default:
      return false;
  }
}
function isFunctionLikeKind(kind) {
  switch (kind) {
    case 173 /* MethodSignature */:
    case 179 /* CallSignature */:
    case 323 /* JSDocSignature */:
    case 180 /* ConstructSignature */:
    case 181 /* IndexSignature */:
    case 184 /* FunctionType */:
    case 317 /* JSDocFunctionType */:
    case 185 /* ConstructorType */:
      return true;
    default:
      return isFunctionLikeDeclarationKind(kind);
  }
}
function isFunctionOrModuleBlock(node) {
  return isSourceFile(node) || isModuleBlock(node) || isBlock(node) && isFunctionLike(node.parent);
}
function isClassElement(node) {
  const kind = node.kind;
  return kind === 176 /* Constructor */ || kind === 172 /* PropertyDeclaration */ || kind === 174 /* MethodDeclaration */ || kind === 177 /* GetAccessor */ || kind === 178 /* SetAccessor */ || kind === 181 /* IndexSignature */ || kind === 175 /* ClassStaticBlockDeclaration */ || kind === 240 /* SemicolonClassElement */;
}
function isClassLike(node) {
  return node && (node.kind === 263 /* ClassDeclaration */ || node.kind === 231 /* ClassExpression */);
}
function isAccessor(node) {
  return node && (node.kind === 177 /* GetAccessor */ || node.kind === 178 /* SetAccessor */);
}
function isAutoAccessorPropertyDeclaration(node) {
  return isPropertyDeclaration(node) && hasAccessorModifier(node);
}
function isClassInstanceProperty(node) {
  if (isInJSFile(node) && isExpandoPropertyDeclaration(node)) {
    return (!isBindableStaticAccessExpression(node) || !isPrototypeAccess(node.expression)) && !isBindableStaticNameExpression(
      node,
      /*excludeThisKeyword*/
      true
    );
  }
  return node.parent && isClassLike(node.parent) && isPropertyDeclaration(node) && !hasAccessorModifier(node);
}
function isMethodOrAccessor(node) {
  switch (node.kind) {
    case 174 /* MethodDeclaration */:
    case 177 /* GetAccessor */:
    case 178 /* SetAccessor */:
      return true;
    default:
      return false;
  }
}
function isModifierLike(node) {
  return isModifier(node) || isDecorator(node);
}
function isTypeElement(node) {
  const kind = node.kind;
  return kind === 180 /* ConstructSignature */ || kind === 179 /* CallSignature */ || kind === 171 /* PropertySignature */ || kind === 173 /* MethodSignature */ || kind === 181 /* IndexSignature */ || kind === 177 /* GetAccessor */ || kind === 178 /* SetAccessor */ || kind === 354 /* NotEmittedTypeElement */;
}
function isObjectLiteralElementLike(node) {
  const kind = node.kind;
  return kind === 303 /* PropertyAssignment */ || kind === 304 /* ShorthandPropertyAssignment */ || kind === 305 /* SpreadAssignment */ || kind === 174 /* MethodDeclaration */ || kind === 177 /* GetAccessor */ || kind === 178 /* SetAccessor */;
}
function isTypeNode(node) {
  return isTypeNodeKind(node.kind);
}
function isFunctionOrConstructorTypeNode(node) {
  switch (node.kind) {
    case 184 /* FunctionType */:
    case 185 /* ConstructorType */:
      return true;
  }
  return false;
}
function isBindingPattern(node) {
  if (node) {
    const kind = node.kind;
    return kind === 207 /* ArrayBindingPattern */ || kind === 206 /* ObjectBindingPattern */;
  }
  return false;
}
function isAssignmentPattern(node) {
  const kind = node.kind;
  return kind === 209 /* ArrayLiteralExpression */ || kind === 210 /* ObjectLiteralExpression */;
}
function isArrayBindingElement(node) {
  const kind = node.kind;
  return kind === 208 /* BindingElement */ || kind === 232 /* OmittedExpression */;
}
function isDeclarationBindingElement(bindingElement) {
  switch (bindingElement.kind) {
    case 260 /* VariableDeclaration */:
    case 169 /* Parameter */:
    case 208 /* BindingElement */:
      return true;
  }
  return false;
}
function isBindingOrAssignmentElement(node) {
  return isVariableDeclaration(node) || isParameter(node) || isObjectBindingOrAssignmentElement(node) || isArrayBindingOrAssignmentElement(node);
}
function isBindingOrAssignmentPattern(node) {
  return isObjectBindingOrAssignmentPattern(node) || isArrayBindingOrAssignmentPattern(node);
}
function isObjectBindingOrAssignmentPattern(node) {
  switch (node.kind) {
    case 206 /* ObjectBindingPattern */:
    case 210 /* ObjectLiteralExpression */:
      return true;
  }
  return false;
}
function isObjectBindingOrAssignmentElement(node) {
  switch (node.kind) {
    case 208 /* BindingElement */:
    case 303 /* PropertyAssignment */:
    // AssignmentProperty
    case 304 /* ShorthandPropertyAssignment */:
    // AssignmentProperty
    case 305 /* SpreadAssignment */:
      return true;
  }
  return false;
}
function isArrayBindingOrAssignmentPattern(node) {
  switch (node.kind) {
    case 207 /* ArrayBindingPattern */:
    case 209 /* ArrayLiteralExpression */:
      return true;
  }
  return false;
}
function isArrayBindingOrAssignmentElement(node) {
  switch (node.kind) {
    case 208 /* BindingElement */:
    case 232 /* OmittedExpression */:
    // Elision
    case 230 /* SpreadElement */:
    // AssignmentRestElement
    case 209 /* ArrayLiteralExpression */:
    // ArrayAssignmentPattern
    case 210 /* ObjectLiteralExpression */:
    // ObjectAssignmentPattern
    case 80 /* Identifier */:
    // DestructuringAssignmentTarget
    case 211 /* PropertyAccessExpression */:
    // DestructuringAssignmentTarget
    case 212 /* ElementAccessExpression */:
      return true;
  }
  return isAssignmentExpression(
    node,
    /*excludeCompoundAssignment*/
    true
  );
}
function isPropertyAccessOrQualifiedNameOrImportTypeNode(node) {
  const kind = node.kind;
  return kind === 211 /* PropertyAccessExpression */ || kind === 166 /* QualifiedName */ || kind === 205 /* ImportType */;
}
function isPropertyAccessOrQualifiedName(node) {
  const kind = node.kind;
  return kind === 211 /* PropertyAccessExpression */ || kind === 166 /* QualifiedName */;
}
function isCallLikeOrFunctionLikeExpression(node) {
  return isCallLikeExpression(node) || isFunctionExpressionOrArrowFunction(node);
}
function isCallLikeExpression(node) {
  switch (node.kind) {
    case 213 /* CallExpression */:
    case 214 /* NewExpression */:
    case 215 /* TaggedTemplateExpression */:
    case 170 /* Decorator */:
    case 286 /* JsxOpeningElement */:
    case 285 /* JsxSelfClosingElement */:
    case 289 /* JsxOpeningFragment */:
      return true;
    case 226 /* BinaryExpression */:
      return node.operatorToken.kind === 104 /* InstanceOfKeyword */;
    default:
      return false;
  }
}
function isCallOrNewExpression(node) {
  return node.kind === 213 /* CallExpression */ || node.kind === 214 /* NewExpression */;
}
function isTemplateLiteral(node) {
  const kind = node.kind;
  return kind === 228 /* TemplateExpression */ || kind === 15 /* NoSubstitutionTemplateLiteral */;
}
function isLeftHandSideExpression(node) {
  return isLeftHandSideExpressionKind(skipPartiallyEmittedExpressions(node).kind);
}
function isLeftHandSideExpressionKind(kind) {
  switch (kind) {
    case 211 /* PropertyAccessExpression */:
    case 212 /* ElementAccessExpression */:
    case 214 /* NewExpression */:
    case 213 /* CallExpression */:
    case 284 /* JsxElement */:
    case 285 /* JsxSelfClosingElement */:
    case 288 /* JsxFragment */:
    case 215 /* TaggedTemplateExpression */:
    case 209 /* ArrayLiteralExpression */:
    case 217 /* ParenthesizedExpression */:
    case 210 /* ObjectLiteralExpression */:
    case 231 /* ClassExpression */:
    case 218 /* FunctionExpression */:
    case 80 /* Identifier */:
    case 81 /* PrivateIdentifier */:
    // technically this is only an Expression if it's in a `#field in expr` BinaryExpression
    case 14 /* RegularExpressionLiteral */:
    case 9 /* NumericLiteral */:
    case 10 /* BigIntLiteral */:
    case 11 /* StringLiteral */:
    case 15 /* NoSubstitutionTemplateLiteral */:
    case 228 /* TemplateExpression */:
    case 97 /* FalseKeyword */:
    case 106 /* NullKeyword */:
    case 110 /* ThisKeyword */:
    case 112 /* TrueKeyword */:
    case 108 /* SuperKeyword */:
    case 235 /* NonNullExpression */:
    case 233 /* ExpressionWithTypeArguments */:
    case 236 /* MetaProperty */:
    case 102 /* ImportKeyword */:
    // technically this is only an Expression if it's in a CallExpression
    case 282 /* MissingDeclaration */:
      return true;
    default:
      return false;
  }
}
function isUnaryExpression(node) {
  return isUnaryExpressionKind(skipPartiallyEmittedExpressions(node).kind);
}
function isUnaryExpressionKind(kind) {
  switch (kind) {
    case 224 /* PrefixUnaryExpression */:
    case 225 /* PostfixUnaryExpression */:
    case 220 /* DeleteExpression */:
    case 221 /* TypeOfExpression */:
    case 222 /* VoidExpression */:
    case 223 /* AwaitExpression */:
    case 216 /* TypeAssertionExpression */:
      return true;
    default:
      return isLeftHandSideExpressionKind(kind);
  }
}
function isLiteralTypeLiteral(node) {
  switch (node.kind) {
    case 106 /* NullKeyword */:
    case 112 /* TrueKeyword */:
    case 97 /* FalseKeyword */:
    case 224 /* PrefixUnaryExpression */:
      return true;
    default:
      return isLiteralExpression(node);
  }
}
function isExpression(node) {
  return isExpressionKind(skipPartiallyEmittedExpressions(node).kind);
}
function isExpressionKind(kind) {
  switch (kind) {
    case 227 /* ConditionalExpression */:
    case 229 /* YieldExpression */:
    case 219 /* ArrowFunction */:
    case 226 /* BinaryExpression */:
    case 230 /* SpreadElement */:
    case 234 /* AsExpression */:
    case 232 /* OmittedExpression */:
    case 356 /* CommaListExpression */:
    case 355 /* PartiallyEmittedExpression */:
    case 238 /* SatisfiesExpression */:
      return true;
    default:
      return isUnaryExpressionKind(kind);
  }
}
function isAssertionExpression(node) {
  const kind = node.kind;
  return kind === 216 /* TypeAssertionExpression */ || kind === 234 /* AsExpression */;
}
function isIterationStatement(node, lookInLabeledStatements) {
  switch (node.kind) {
    case 248 /* ForStatement */:
    case 249 /* ForInStatement */:
    case 250 /* ForOfStatement */:
    case 246 /* DoStatement */:
    case 247 /* WhileStatement */:
      return true;
    case 256 /* LabeledStatement */:
      return lookInLabeledStatements && isIterationStatement(node.statement, lookInLabeledStatements);
  }
  return false;
}
function isScopeMarker(node) {
  return isExportAssignment(node) || isExportDeclaration(node);
}
function hasScopeMarker(statements) {
  return some(statements, isScopeMarker);
}
function needsScopeMarker(result) {
  return !isAnyImportOrReExport(result) && !isExportAssignment(result) && !hasSyntacticModifier(result, 32 /* Export */) && !isAmbientModule(result);
}
function isExternalModuleIndicator(result) {
  return isAnyImportOrReExport(result) || isExportAssignment(result) || hasSyntacticModifier(result, 32 /* Export */);
}
function isForInOrOfStatement(node) {
  return node.kind === 249 /* ForInStatement */ || node.kind === 250 /* ForOfStatement */;
}
function isConciseBody(node) {
  return isBlock(node) || isExpression(node);
}
function isForInitializer(node) {
  return isVariableDeclarationList(node) || isExpression(node);
}
function isModuleBody(node) {
  const kind = node.kind;
  return kind === 268 /* ModuleBlock */ || kind === 267 /* ModuleDeclaration */ || kind === 80 /* Identifier */;
}
function isNamedImportBindings(node) {
  const kind = node.kind;
  return kind === 275 /* NamedImports */ || kind === 274 /* NamespaceImport */;
}
function isModuleOrEnumDeclaration(node) {
  return node.kind === 267 /* ModuleDeclaration */ || node.kind === 266 /* EnumDeclaration */;
}
function canHaveSymbol(node) {
  switch (node.kind) {
    case 219 /* ArrowFunction */:
    case 226 /* BinaryExpression */:
    case 208 /* BindingElement */:
    case 213 /* CallExpression */:
    case 179 /* CallSignature */:
    case 263 /* ClassDeclaration */:
    case 231 /* ClassExpression */:
    case 175 /* ClassStaticBlockDeclaration */:
    case 176 /* Constructor */:
    case 185 /* ConstructorType */:
    case 180 /* ConstructSignature */:
    case 212 /* ElementAccessExpression */:
    case 266 /* EnumDeclaration */:
    case 306 /* EnumMember */:
    case 277 /* ExportAssignment */:
    case 278 /* ExportDeclaration */:
    case 281 /* ExportSpecifier */:
    case 262 /* FunctionDeclaration */:
    case 218 /* FunctionExpression */:
    case 184 /* FunctionType */:
    case 177 /* GetAccessor */:
    case 80 /* Identifier */:
    case 273 /* ImportClause */:
    case 271 /* ImportEqualsDeclaration */:
    case 276 /* ImportSpecifier */:
    case 181 /* IndexSignature */:
    case 264 /* InterfaceDeclaration */:
    case 338 /* JSDocCallbackTag */:
    case 340 /* JSDocEnumTag */:
    case 317 /* JSDocFunctionType */:
    case 341 /* JSDocParameterTag */:
    case 348 /* JSDocPropertyTag */:
    case 323 /* JSDocSignature */:
    case 346 /* JSDocTypedefTag */:
    case 322 /* JSDocTypeLiteral */:
    case 291 /* JsxAttribute */:
    case 292 /* JsxAttributes */:
    case 293 /* JsxSpreadAttribute */:
    case 200 /* MappedType */:
    case 174 /* MethodDeclaration */:
    case 173 /* MethodSignature */:
    case 267 /* ModuleDeclaration */:
    case 202 /* NamedTupleMember */:
    case 280 /* NamespaceExport */:
    case 270 /* NamespaceExportDeclaration */:
    case 274 /* NamespaceImport */:
    case 214 /* NewExpression */:
    case 15 /* NoSubstitutionTemplateLiteral */:
    case 9 /* NumericLiteral */:
    case 210 /* ObjectLiteralExpression */:
    case 169 /* Parameter */:
    case 211 /* PropertyAccessExpression */:
    case 303 /* PropertyAssignment */:
    case 172 /* PropertyDeclaration */:
    case 171 /* PropertySignature */:
    case 178 /* SetAccessor */:
    case 304 /* ShorthandPropertyAssignment */:
    case 307 /* SourceFile */:
    case 305 /* SpreadAssignment */:
    case 11 /* StringLiteral */:
    case 265 /* TypeAliasDeclaration */:
    case 187 /* TypeLiteral */:
    case 168 /* TypeParameter */:
    case 260 /* VariableDeclaration */:
      return true;
    default:
      return false;
  }
}
function canHaveLocals(node) {
  switch (node.kind) {
    case 219 /* ArrowFunction */:
    case 241 /* Block */:
    case 179 /* CallSignature */:
    case 269 /* CaseBlock */:
    case 299 /* CatchClause */:
    case 175 /* ClassStaticBlockDeclaration */:
    case 194 /* ConditionalType */:
    case 176 /* Constructor */:
    case 185 /* ConstructorType */:
    case 180 /* ConstructSignature */:
    case 248 /* ForStatement */:
    case 249 /* ForInStatement */:
    case 250 /* ForOfStatement */:
    case 262 /* FunctionDeclaration */:
    case 218 /* FunctionExpression */:
    case 184 /* FunctionType */:
    case 177 /* GetAccessor */:
    case 181 /* IndexSignature */:
    case 338 /* JSDocCallbackTag */:
    case 340 /* JSDocEnumTag */:
    case 317 /* JSDocFunctionType */:
    case 323 /* JSDocSignature */:
    case 346 /* JSDocTypedefTag */:
    case 200 /* MappedType */:
    case 174 /* MethodDeclaration */:
    case 173 /* MethodSignature */:
    case 267 /* ModuleDeclaration */:
    case 178 /* SetAccessor */:
    case 307 /* SourceFile */:
    case 265 /* TypeAliasDeclaration */:
      return true;
    default:
      return false;
  }
}
function isDeclarationKind(kind) {
  return kind === 219 /* ArrowFunction */ || kind === 208 /* BindingElement */ || kind === 263 /* ClassDeclaration */ || kind === 231 /* ClassExpression */ || kind === 175 /* ClassStaticBlockDeclaration */ || kind === 176 /* Constructor */ || kind === 266 /* EnumDeclaration */ || kind === 306 /* EnumMember */ || kind === 281 /* ExportSpecifier */ || kind === 262 /* FunctionDeclaration */ || kind === 218 /* FunctionExpression */ || kind === 177 /* GetAccessor */ || kind === 273 /* ImportClause */ || kind === 271 /* ImportEqualsDeclaration */ || kind === 276 /* ImportSpecifier */ || kind === 264 /* InterfaceDeclaration */ || kind === 291 /* JsxAttribute */ || kind === 174 /* MethodDeclaration */ || kind === 173 /* MethodSignature */ || kind === 267 /* ModuleDeclaration */ || kind === 270 /* NamespaceExportDeclaration */ || kind === 274 /* NamespaceImport */ || kind === 280 /* NamespaceExport */ || kind === 169 /* Parameter */ || kind === 303 /* PropertyAssignment */ || kind === 172 /* PropertyDeclaration */ || kind === 171 /* PropertySignature */ || kind === 178 /* SetAccessor */ || kind === 304 /* ShorthandPropertyAssignment */ || kind === 265 /* TypeAliasDeclaration */ || kind === 168 /* TypeParameter */ || kind === 260 /* VariableDeclaration */ || kind === 346 /* JSDocTypedefTag */ || kind === 338 /* JSDocCallbackTag */ || kind === 348 /* JSDocPropertyTag */ || kind === 202 /* NamedTupleMember */;
}
function isDeclarationStatementKind(kind) {
  return kind === 262 /* FunctionDeclaration */ || kind === 282 /* MissingDeclaration */ || kind === 263 /* ClassDeclaration */ || kind === 264 /* InterfaceDeclaration */ || kind === 265 /* TypeAliasDeclaration */ || kind === 266 /* EnumDeclaration */ || kind === 267 /* ModuleDeclaration */ || kind === 272 /* ImportDeclaration */ || kind === 271 /* ImportEqualsDeclaration */ || kind === 278 /* ExportDeclaration */ || kind === 277 /* ExportAssignment */ || kind === 270 /* NamespaceExportDeclaration */;
}
function isStatementKindButNotDeclarationKind(kind) {
  return kind === 252 /* BreakStatement */ || kind === 251 /* ContinueStatement */ || kind === 259 /* DebuggerStatement */ || kind === 246 /* DoStatement */ || kind === 244 /* ExpressionStatement */ || kind === 242 /* EmptyStatement */ || kind === 249 /* ForInStatement */ || kind === 250 /* ForOfStatement */ || kind === 248 /* ForStatement */ || kind === 245 /* IfStatement */ || kind === 256 /* LabeledStatement */ || kind === 253 /* ReturnStatement */ || kind === 255 /* SwitchStatement */ || kind === 257 /* ThrowStatement */ || kind === 258 /* TryStatement */ || kind === 243 /* VariableStatement */ || kind === 247 /* WhileStatement */ || kind === 254 /* WithStatement */ || kind === 353 /* NotEmittedStatement */;
}
function isDeclaration(node) {
  if (node.kind === 168 /* TypeParameter */) {
    return node.parent && node.parent.kind !== 345 /* JSDocTemplateTag */ || isInJSFile(node);
  }
  return isDeclarationKind(node.kind);
}
function isDeclarationStatement(node) {
  return isDeclarationStatementKind(node.kind);
}
function isStatementButNotDeclaration(node) {
  return isStatementKindButNotDeclarationKind(node.kind);
}
function isStatement(node) {
  const kind = node.kind;
  return isStatementKindButNotDeclarationKind(kind) || isDeclarationStatementKind(kind) || isBlockStatement(node);
}
function isBlockStatement(node) {
  if (node.kind !== 241 /* Block */) return false;
  if (node.parent !== void 0) {
    if (node.parent.kind === 258 /* TryStatement */ || node.parent.kind === 299 /* CatchClause */) {
      return false;
    }
  }
  return !isFunctionBlock(node);
}
function isStatementOrBlock(node) {
  const kind = node.kind;
  return isStatementKindButNotDeclarationKind(kind) || isDeclarationStatementKind(kind) || kind === 241 /* Block */;
}
function isModuleReference(node) {
  const kind = node.kind;
  return kind === 283 /* ExternalModuleReference */ || kind === 166 /* QualifiedName */ || kind === 80 /* Identifier */;
}
function isJsxTagNameExpression(node) {
  const kind = node.kind;
  return kind === 110 /* ThisKeyword */ || kind === 80 /* Identifier */ || kind === 211 /* PropertyAccessExpression */ || kind === 295 /* JsxNamespacedName */;
}
function isJsxChild(node) {
  const kind = node.kind;
  return kind === 284 /* JsxElement */ || kind === 294 /* JsxExpression */ || kind === 285 /* JsxSelfClosingElement */ || kind === 12 /* JsxText */ || kind === 288 /* JsxFragment */;
}
function isJsxAttributeLike(node) {
  const kind = node.kind;
  return kind === 291 /* JsxAttribute */ || kind === 293 /* JsxSpreadAttribute */;
}
function isStringLiteralOrJsxExpression(node) {
  const kind = node.kind;
  return kind === 11 /* StringLiteral */ || kind === 294 /* JsxExpression */;
}
function isJsxOpeningLikeElement(node) {
  const kind = node.kind;
  return kind === 286 /* JsxOpeningElement */ || kind === 285 /* JsxSelfClosingElement */;
}
function isJsxCallLike(node) {
  const kind = node.kind;
  return kind === 286 /* JsxOpeningElement */ || kind === 285 /* JsxSelfClosingElement */ || kind === 289 /* JsxOpeningFragment */;
}
function isCaseOrDefaultClause(node) {
  const kind = node.kind;
  return kind === 296 /* CaseClause */ || kind === 297 /* DefaultClause */;
}
function isJSDocNode(node) {
  return node.kind >= 309 /* FirstJSDocNode */ && node.kind <= 351 /* LastJSDocNode */;
}
function isJSDocTag(node) {
  return node.kind >= 327 /* FirstJSDocTagNode */ && node.kind <= 351 /* LastJSDocTagNode */;
}
function isSetAccessor(node) {
  return node.kind === 178 /* SetAccessor */;
}
function isGetAccessor(node) {
  return node.kind === 177 /* GetAccessor */;
}
function hasJSDocNodes(node) {
  if (!canHaveJSDoc(node)) return false;
  const { jsDoc } = node;
  return !!jsDoc && jsDoc.length > 0;
}
function hasType(node) {
  return !!node.type;
}
function hasInitializer(node) {
  return !!node.initializer;
}
function hasOnlyExpressionInitializer(node) {
  switch (node.kind) {
    case 260 /* VariableDeclaration */:
    case 169 /* Parameter */:
    case 208 /* BindingElement */:
    case 172 /* PropertyDeclaration */:
    case 303 /* PropertyAssignment */:
    case 306 /* EnumMember */:
      return true;
    default:
      return false;
  }
}
function isTypeReferenceType(node) {
  return node.kind === 183 /* TypeReference */ || node.kind === 233 /* ExpressionWithTypeArguments */;
}
var MAX_SMI_X86 = 1073741823;
function guessIndentation(lines) {
  let indentation = MAX_SMI_X86;
  for (const line of lines) {
    if (!line.length) {
      continue;
    }
    let i = 0;
    for (; i < line.length && i < indentation; i++) {
      if (!isWhiteSpaceLike(line.charCodeAt(i))) {
        break;
      }
    }
    if (i < indentation) {
      indentation = i;
    }
    if (indentation === 0) {
      return 0;
    }
  }
  return indentation === MAX_SMI_X86 ? void 0 : indentation;
}
function isStringLiteralLike(node) {
  return node.kind === 11 /* StringLiteral */ || node.kind === 15 /* NoSubstitutionTemplateLiteral */;
}
function isJSDocLinkLike(node) {
  return node.kind === 324 /* JSDocLink */ || node.kind === 325 /* JSDocLinkCode */ || node.kind === 326 /* JSDocLinkPlain */;
}
function hasRestParameter(s) {
  const last2 = lastOrUndefined(s.parameters);
  return !!last2 && isRestParameter(last2);
}
function isRestParameter(node) {
  const type = isJSDocParameterTag(node) ? node.typeExpression && node.typeExpression.type : node.type;
  return node.dotDotDotToken !== void 0 || !!type && type.kind === 318 /* JSDocVariadicType */;
}
function hasInternalAnnotation(range, sourceFile) {
  const comment = sourceFile.text.substring(range.pos, range.end);
  return comment.includes("@internal");
}
function isInternalDeclaration(node, sourceFile) {
  sourceFile ?? (sourceFile = getSourceFileOfNode(node));
  const parseTreeNode = getParseTreeNode(node);
  if (parseTreeNode && parseTreeNode.kind === 169 /* Parameter */) {
    const paramIdx = parseTreeNode.parent.parameters.indexOf(parseTreeNode);
    const previousSibling = paramIdx > 0 ? parseTreeNode.parent.parameters[paramIdx - 1] : void 0;
    const text = sourceFile.text;
    const commentRanges = previousSibling ? concatenate(
      // to handle
      // ... parameters, /** @internal */
      // public param: string
      getTrailingCommentRanges(text, skipTrivia(
        text,
        previousSibling.end + 1,
        /*stopAfterLineBreak*/
        false,
        /*stopAtComments*/
        true
      )),
      getLeadingCommentRanges(text, node.pos)
    ) : getTrailingCommentRanges(text, skipTrivia(
      text,
      node.pos,
      /*stopAfterLineBreak*/
      false,
      /*stopAtComments*/
      true
    ));
    return some(commentRanges) && hasInternalAnnotation(last(commentRanges), sourceFile);
  }
  const leadingCommentRanges = parseTreeNode && getLeadingCommentRangesOfNode(parseTreeNode, sourceFile);
  return !!forEach(leadingCommentRanges, (range) => {
    return hasInternalAnnotation(range, sourceFile);
  });
}

// src/compiler/utilities.ts
var resolvingEmptyArray = [];
var externalHelpersModuleNameText = "tslib";
var defaultMaximumTruncationLength = 160;
var noTruncationMaximumTruncationLength = 1e6;
function getDeclarationOfKind(symbol, kind) {
  const declarations = symbol.declarations;
  if (declarations) {
    for (const declaration of declarations) {
      if (declaration.kind === kind) {
        return declaration;
      }
    }
  }
  return void 0;
}
function getDeclarationsOfKind(symbol, kind) {
  return filter(symbol.declarations || emptyArray, (d) => d.kind === kind);
}
function createSymbolTable(symbols) {
  const result = /* @__PURE__ */ new Map();
  if (symbols) {
    for (const symbol of symbols) {
      result.set(symbol.escapedName, symbol);
    }
  }
  return result;
}
function isTransientSymbol(symbol) {
  return (symbol.flags & 33554432 /* Transient */) !== 0;
}
function isExternalModuleSymbol(moduleSymbol) {
  return !!(moduleSymbol.flags & 1536 /* Module */) && moduleSymbol.escapedName.charCodeAt(0) === 34 /* doubleQuote */;
}
var stringWriter = createSingleLineStringWriter();
function createSingleLineStringWriter() {
  var str = "";
  const writeText = (text) => str += text;
  return {
    getText: () => str,
    write: writeText,
    rawWrite: writeText,
    writeKeyword: writeText,
    writeOperator: writeText,
    writePunctuation: writeText,
    writeSpace: writeText,
    writeStringLiteral: writeText,
    writeLiteral: writeText,
    writeParameter: writeText,
    writeProperty: writeText,
    writeSymbol: (s, _) => writeText(s),
    writeTrailingSemicolon: writeText,
    writeComment: writeText,
    getTextPos: () => str.length,
    getLine: () => 0,
    getColumn: () => 0,
    getIndent: () => 0,
    isAtStartOfLine: () => false,
    hasTrailingComment: () => false,
    hasTrailingWhitespace: () => !!str.length && isWhiteSpaceLike(str.charCodeAt(str.length - 1)),
    // Completely ignore indentation for string writers.  And map newlines to
    // a single space.
    writeLine: () => str += " ",
    increaseIndent: noop,
    decreaseIndent: noop,
    clear: () => str = ""
  };
}
function changesAffectModuleResolution(oldOptions, newOptions) {
  return oldOptions.configFilePath !== newOptions.configFilePath || optionsHaveModuleResolutionChanges(oldOptions, newOptions);
}
function optionsHaveModuleResolutionChanges(oldOptions, newOptions) {
  return optionsHaveChanges(oldOptions, newOptions, moduleResolutionOptionDeclarations);
}
function changesAffectingProgramStructure(oldOptions, newOptions) {
  return optionsHaveChanges(oldOptions, newOptions, optionsAffectingProgramStructure);
}
function optionsHaveChanges(oldOptions, newOptions, optionDeclarations2) {
  return oldOptions !== newOptions && optionDeclarations2.some((o) => !isJsonEqual(getCompilerOptionValue(oldOptions, o), getCompilerOptionValue(newOptions, o)));
}
function forEachEntry(map2, callback) {
  const iterator = map2.entries();
  for (const [key, value] of iterator) {
    const result = callback(value, key);
    if (result) {
      return result;
    }
  }
  return void 0;
}
function forEachKey(map2, callback) {
  const iterator = map2.keys();
  for (const key of iterator) {
    const result = callback(key);
    if (result) {
      return result;
    }
  }
  return void 0;
}
function copyEntries(source, target) {
  source.forEach((value, key) => {
    target.set(key, value);
  });
}
function usingSingleLineStringWriter(action) {
  const oldString = stringWriter.getText();
  try {
    action(stringWriter);
    return stringWriter.getText();
  } finally {
    stringWriter.clear();
    stringWriter.writeKeyword(oldString);
  }
}
function getFullWidth(node) {
  return node.end - node.pos;
}
function projectReferenceIsEqualTo(oldRef, newRef) {
  return oldRef.path === newRef.path && !oldRef.prepend === !newRef.prepend && !oldRef.circular === !newRef.circular;
}
function moduleResolutionIsEqualTo(oldResolution, newResolution) {
  return oldResolution === newResolution || oldResolution.resolvedModule === newResolution.resolvedModule || !!oldResolution.resolvedModule && !!newResolution.resolvedModule && oldResolution.resolvedModule.isExternalLibraryImport === newResolution.resolvedModule.isExternalLibraryImport && oldResolution.resolvedModule.extension === newResolution.resolvedModule.extension && oldResolution.resolvedModule.resolvedFileName === newResolution.resolvedModule.resolvedFileName && oldResolution.resolvedModule.originalPath === newResolution.resolvedModule.originalPath && packageIdIsEqual(oldResolution.resolvedModule.packageId, newResolution.resolvedModule.packageId) && oldResolution.alternateResult === newResolution.alternateResult;
}
function getResolvedModuleFromResolution(resolution) {
  return resolution.resolvedModule;
}
function getResolvedTypeReferenceDirectiveFromResolution(resolution) {
  return resolution.resolvedTypeReferenceDirective;
}
function createModuleNotFoundChain(sourceFile, host, moduleReference, mode, packageName) {
  var _a;
  const alternateResult = (_a = host.getResolvedModule(sourceFile, moduleReference, mode)) == null ? void 0 : _a.alternateResult;
  const alternateResultMessage = alternateResult && (getEmitModuleResolutionKind(host.getCompilerOptions()) === 2 /* Node10 */ ? [Diagnostics.There_are_types_at_0_but_this_result_could_not_be_resolved_under_your_current_moduleResolution_setting_Consider_updating_to_node16_nodenext_or_bundler, [alternateResult]] : [
    Diagnostics.There_are_types_at_0_but_this_result_could_not_be_resolved_when_respecting_package_json_exports_The_1_library_may_need_to_update_its_package_json_or_typings,
    [alternateResult, alternateResult.includes(nodeModulesPathPart + "@types/") ? `@types/${mangleScopedPackageName(packageName)}` : packageName]
  ]);
  const result = alternateResultMessage ? chainDiagnosticMessages(
    /*details*/
    void 0,
    alternateResultMessage[0],
    ...alternateResultMessage[1]
  ) : host.typesPackageExists(packageName) ? chainDiagnosticMessages(
    /*details*/
    void 0,
    Diagnostics.If_the_0_package_actually_exposes_this_module_consider_sending_a_pull_request_to_amend_https_Colon_Slash_Slashgithub_com_SlashDefinitelyTyped_SlashDefinitelyTyped_Slashtree_Slashmaster_Slashtypes_Slash_1,
    packageName,
    mangleScopedPackageName(packageName)
  ) : host.packageBundlesTypes(packageName) ? chainDiagnosticMessages(
    /*details*/
    void 0,
    Diagnostics.If_the_0_package_actually_exposes_this_module_try_adding_a_new_declaration_d_ts_file_containing_declare_module_1,
    packageName,
    moduleReference
  ) : chainDiagnosticMessages(
    /*details*/
    void 0,
    Diagnostics.Try_npm_i_save_dev_types_Slash_1_if_it_exists_or_add_a_new_declaration_d_ts_file_containing_declare_module_0,
    moduleReference,
    mangleScopedPackageName(packageName)
  );
  if (result) result.repopulateInfo = () => ({ moduleReference, mode, packageName: packageName === moduleReference ? void 0 : packageName });
  return result;
}
function createModeMismatchDetails(currentSourceFile) {
  const ext = tryGetExtensionFromPath2(currentSourceFile.fileName);
  const scope = currentSourceFile.packageJsonScope;
  const targetExt = ext === ".ts" /* Ts */ ? ".mts" /* Mts */ : ext === ".js" /* Js */ ? ".mjs" /* Mjs */ : void 0;
  const result = scope && !scope.contents.packageJsonContent.type ? targetExt ? chainDiagnosticMessages(
    /*details*/
    void 0,
    Diagnostics.To_convert_this_file_to_an_ECMAScript_module_change_its_file_extension_to_0_or_add_the_field_type_Colon_module_to_1,
    targetExt,
    combinePaths(scope.packageDirectory, "package.json")
  ) : chainDiagnosticMessages(
    /*details*/
    void 0,
    Diagnostics.To_convert_this_file_to_an_ECMAScript_module_add_the_field_type_Colon_module_to_0,
    combinePaths(scope.packageDirectory, "package.json")
  ) : targetExt ? chainDiagnosticMessages(
    /*details*/
    void 0,
    Diagnostics.To_convert_this_file_to_an_ECMAScript_module_change_its_file_extension_to_0_or_create_a_local_package_json_file_with_type_Colon_module,
    targetExt
  ) : chainDiagnosticMessages(
    /*details*/
    void 0,
    Diagnostics.To_convert_this_file_to_an_ECMAScript_module_create_a_local_package_json_file_with_type_Colon_module
  );
  result.repopulateInfo = () => true;
  return result;
}
function packageIdIsEqual(a, b) {
  return a === b || !!a && !!b && a.name === b.name && a.subModuleName === b.subModuleName && a.version === b.version && a.peerDependencies === b.peerDependencies;
}
function packageIdToPackageName({ name, subModuleName }) {
  return subModuleName ? `${name}/${subModuleName}` : name;
}
function packageIdToString(packageId) {
  return `${packageIdToPackageName(packageId)}@${packageId.version}${packageId.peerDependencies ?? ""}`;
}
function typeDirectiveIsEqualTo(oldResolution, newResolution) {
  return oldResolution === newResolution || oldResolution.resolvedTypeReferenceDirective === newResolution.resolvedTypeReferenceDirective || !!oldResolution.resolvedTypeReferenceDirective && !!newResolution.resolvedTypeReferenceDirective && oldResolution.resolvedTypeReferenceDirective.resolvedFileName === newResolution.resolvedTypeReferenceDirective.resolvedFileName && !!oldResolution.resolvedTypeReferenceDirective.primary === !!newResolution.resolvedTypeReferenceDirective.primary && oldResolution.resolvedTypeReferenceDirective.originalPath === newResolution.resolvedTypeReferenceDirective.originalPath;
}
function hasChangesInResolutions(names, newResolutions, getOldResolution, comparer) {
  Debug.assert(names.length === newResolutions.length);
  for (let i = 0; i < names.length; i++) {
    const newResolution = newResolutions[i];
    const entry = names[i];
    const oldResolution = getOldResolution(entry);
    const changed = oldResolution ? !newResolution || !comparer(oldResolution, newResolution) : newResolution;
    if (changed) {
      return true;
    }
  }
  return false;
}
function containsParseError(node) {
  aggregateChildData(node);
  return (node.flags & 1048576 /* ThisNodeOrAnySubNodesHasError */) !== 0;
}
function aggregateChildData(node) {
  if (!(node.flags & 2097152 /* HasAggregatedChildData */)) {
    const thisNodeOrAnySubNodesHasError = (node.flags & 262144 /* ThisNodeHasError */) !== 0 || forEachChild(node, containsParseError);
    if (thisNodeOrAnySubNodesHasError) {
      node.flags |= 1048576 /* ThisNodeOrAnySubNodesHasError */;
    }
    node.flags |= 2097152 /* HasAggregatedChildData */;
  }
}
function getSourceFileOfNode(node) {
  while (node && node.kind !== 307 /* SourceFile */) {
    node = node.parent;
  }
  return node;
}
function getSourceFileOfModule(module2) {
  return getSourceFileOfNode(module2.valueDeclaration || getNonAugmentationDeclaration(module2));
}
function isPlainJsFile(file, checkJs) {
  return !!file && (file.scriptKind === 1 /* JS */ || file.scriptKind === 2 /* JSX */) && !file.checkJsDirective && checkJs === void 0;
}
function isStatementWithLocals(node) {
  switch (node.kind) {
    case 241 /* Block */:
    case 269 /* CaseBlock */:
    case 248 /* ForStatement */:
    case 249 /* ForInStatement */:
    case 250 /* ForOfStatement */:
      return true;
  }
  return false;
}
function getEndLinePosition(line, sourceFile) {
  Debug.assert(line >= 0);
  const lineStarts = getLineStarts(sourceFile);
  const lineIndex = line;
  const sourceText = sourceFile.text;
  if (lineIndex + 1 === lineStarts.length) {
    return sourceText.length - 1;
  } else {
    const start = lineStarts[lineIndex];
    let pos = lineStarts[lineIndex + 1] - 1;
    Debug.assert(isLineBreak(sourceText.charCodeAt(pos)));
    while (start <= pos && isLineBreak(sourceText.charCodeAt(pos))) {
      pos--;
    }
    return pos;
  }
}
function isFileLevelUniqueName(sourceFile, name, hasGlobalName) {
  return !(hasGlobalName && hasGlobalName(name)) && !sourceFile.identifiers.has(name);
}
function nodeIsMissing(node) {
  if (node === void 0) {
    return true;
  }
  return node.pos === node.end && node.pos >= 0 && node.kind !== 1 /* EndOfFileToken */;
}
function nodeIsPresent(node) {
  return !nodeIsMissing(node);
}
function insertStatementsAfterPrologue(to, from, isPrologueDirective2) {
  if (from === void 0 || from.length === 0) return to;
  let statementIndex = 0;
  for (; statementIndex < to.length; ++statementIndex) {
    if (!isPrologueDirective2(to[statementIndex])) {
      break;
    }
  }
  to.splice(statementIndex, 0, ...from);
  return to;
}
function insertStatementAfterPrologue(to, statement, isPrologueDirective2) {
  if (statement === void 0) return to;
  let statementIndex = 0;
  for (; statementIndex < to.length; ++statementIndex) {
    if (!isPrologueDirective2(to[statementIndex])) {
      break;
    }
  }
  to.splice(statementIndex, 0, statement);
  return to;
}
function isAnyPrologueDirective(node) {
  return isPrologueDirective(node) || !!(getEmitFlags(node) & 2097152 /* CustomPrologue */);
}
function insertStatementsAfterStandardPrologue(to, from) {
  return insertStatementsAfterPrologue(to, from, isPrologueDirective);
}
function insertStatementsAfterCustomPrologue(to, from) {
  return insertStatementsAfterPrologue(to, from, isAnyPrologueDirective);
}
function insertStatementAfterCustomPrologue(to, statement) {
  return insertStatementAfterPrologue(to, statement, isAnyPrologueDirective);
}
function isRecognizedTripleSlashComment(text, commentPos, commentEnd) {
  if (text.charCodeAt(commentPos + 1) === 47 /* slash */ && commentPos + 2 < commentEnd && text.charCodeAt(commentPos + 2) === 47 /* slash */) {
    const textSubStr = text.substring(commentPos, commentEnd);
    return fullTripleSlashReferencePathRegEx.test(textSubStr) || fullTripleSlashAMDReferencePathRegEx.test(textSubStr) || fullTripleSlashAMDModuleRegEx.test(textSubStr) || fullTripleSlashReferenceTypeReferenceDirectiveRegEx.test(textSubStr) || fullTripleSlashLibReferenceRegEx.test(textSubStr) || defaultLibReferenceRegEx.test(textSubStr) ? true : false;
  }
  return false;
}
function isPinnedComment(text, start) {
  return text.charCodeAt(start + 1) === 42 /* asterisk */ && text.charCodeAt(start + 2) === 33 /* exclamation */;
}
function createCommentDirectivesMap(sourceFile, commentDirectives) {
  const directivesByLine = new Map(
    commentDirectives.map((commentDirective) => [
      `${getLineAndCharacterOfPosition(sourceFile, commentDirective.range.end).line}`,
      commentDirective
    ])
  );
  const usedLines = /* @__PURE__ */ new Map();
  return { getUnusedExpectations, markUsed };
  function getUnusedExpectations() {
    return arrayFrom(directivesByLine.entries()).filter(([line, directive]) => directive.type === 0 /* ExpectError */ && !usedLines.get(line)).map(([_, directive]) => directive);
  }
  function markUsed(line) {
    if (!directivesByLine.has(`${line}`)) {
      return false;
    }
    usedLines.set(`${line}`, true);
    return true;
  }
}
function getTokenPosOfNode(node, sourceFile, includeJsDoc) {
  if (nodeIsMissing(node)) {
    return node.pos;
  }
  if (isJSDocNode(node) || node.kind === 12 /* JsxText */) {
    return skipTrivia(
      (sourceFile ?? getSourceFileOfNode(node)).text,
      node.pos,
      /*stopAfterLineBreak*/
      false,
      /*stopAtComments*/
      true
    );
  }
  if (includeJsDoc && hasJSDocNodes(node)) {
    return getTokenPosOfNode(node.jsDoc[0], sourceFile);
  }
  if (node.kind === 352 /* SyntaxList */) {
    sourceFile ?? (sourceFile = getSourceFileOfNode(node));
    const first2 = firstOrUndefined(getNodeChildren(node, sourceFile));
    if (first2) {
      return getTokenPosOfNode(first2, sourceFile, includeJsDoc);
    }
  }
  return skipTrivia(
    (sourceFile ?? getSourceFileOfNode(node)).text,
    node.pos,
    /*stopAfterLineBreak*/
    false,
    /*stopAtComments*/
    false,
    isInJSDoc(node)
  );
}
function getNonModifierTokenPosOfNode(node, sourceFile) {
  const lastModifier = !nodeIsMissing(node) && canHaveModifiers(node) && node.modifiers ? last(node.modifiers) : void 0;
  if (!lastModifier) {
    return getTokenPosOfNode(node, sourceFile);
  }
  return skipTrivia((sourceFile || getSourceFileOfNode(node)).text, lastModifier.end);
}
function getSourceTextOfNodeFromSourceFile(sourceFile, node, includeTrivia = false) {
  return getTextOfNodeFromSourceText(sourceFile.text, node, includeTrivia);
}
function isJSDocTypeExpressionOrChild(node) {
  return !!findAncestor(node, isJSDocTypeExpression);
}
function isExportNamespaceAsDefaultDeclaration(node) {
  return !!(isExportDeclaration(node) && node.exportClause && isNamespaceExport(node.exportClause) && moduleExportNameIsDefault(node.exportClause.name));
}
function moduleExportNameTextUnescaped(node) {
  return node.kind === 11 /* StringLiteral */ ? node.text : unescapeLeadingUnderscores(node.escapedText);
}
function moduleExportNameTextEscaped(node) {
  return node.kind === 11 /* StringLiteral */ ? escapeLeadingUnderscores(node.text) : node.escapedText;
}
function moduleExportNameIsDefault(node) {
  return (node.kind === 11 /* StringLiteral */ ? node.text : node.escapedText) === "default" /* Default */;
}
function getTextOfNodeFromSourceText(sourceText, node, includeTrivia = false) {
  if (nodeIsMissing(node)) {
    return "";
  }
  let text = sourceText.substring(includeTrivia ? node.pos : skipTrivia(sourceText, node.pos), node.end);
  if (isJSDocTypeExpressionOrChild(node)) {
    text = text.split(/\r\n|\n|\r/).map((line) => line.replace(/^\s*\*/, "").trimStart()).join("\n");
  }
  return text;
}
function getTextOfNode(node, includeTrivia = false) {
  return getSourceTextOfNodeFromSourceFile(getSourceFileOfNode(node), node, includeTrivia);
}
function getPos(range) {
  return range.pos;
}
function indexOfNode(nodeArray, node) {
  return binarySearch(nodeArray, node, getPos, compareValues);
}
function getEmitFlags(node) {
  const emitNode = node.emitNode;
  return emitNode && emitNode.flags || 0;
}
function getInternalEmitFlags(node) {
  const emitNode = node.emitNode;
  return emitNode && emitNode.internalFlags || 0;
}
var getScriptTargetFeatures = /* @__PURE__ */ memoize(
  () => new Map(Object.entries({
    Array: new Map(Object.entries({
      es2015: [
        "find",
        "findIndex",
        "fill",
        "copyWithin",
        "entries",
        "keys",
        "values"
      ],
      es2016: [
        "includes"
      ],
      es2019: [
        "flat",
        "flatMap"
      ],
      es2022: [
        "at"
      ],
      es2023: [
        "findLastIndex",
        "findLast",
        "toReversed",
        "toSorted",
        "toSpliced",
        "with"
      ]
    })),
    Iterator: new Map(Object.entries({
      es2015: emptyArray
    })),
    AsyncIterator: new Map(Object.entries({
      es2015: emptyArray
    })),
    ArrayBuffer: new Map(Object.entries({
      es2024: [
        "maxByteLength",
        "resizable",
        "resize",
        "detached",
        "transfer",
        "transferToFixedLength"
      ]
    })),
    Atomics: new Map(Object.entries({
      es2017: [
        "add",
        "and",
        "compareExchange",
        "exchange",
        "isLockFree",
        "load",
        "or",
        "store",
        "sub",
        "wait",
        "notify",
        "xor"
      ],
      es2024: [
        "waitAsync"
      ]
    })),
    SharedArrayBuffer: new Map(Object.entries({
      es2017: [
        "byteLength",
        "slice"
      ],
      es2024: [
        "growable",
        "maxByteLength",
        "grow"
      ]
    })),
    AsyncIterable: new Map(Object.entries({
      es2018: emptyArray
    })),
    AsyncIterableIterator: new Map(Object.entries({
      es2018: emptyArray
    })),
    AsyncGenerator: new Map(Object.entries({
      es2018: emptyArray
    })),
    AsyncGeneratorFunction: new Map(Object.entries({
      es2018: emptyArray
    })),
    RegExp: new Map(Object.entries({
      es2015: [
        "flags",
        "sticky",
        "unicode"
      ],
      es2018: [
        "dotAll"
      ],
      es2024: [
        "unicodeSets"
      ]
    })),
    Reflect: new Map(Object.entries({
      es2015: [
        "apply",
        "construct",
        "defineProperty",
        "deleteProperty",
        "get",
        "getOwnPropertyDescriptor",
        "getPrototypeOf",
        "has",
        "isExtensible",
        "ownKeys",
        "preventExtensions",
        "set",
        "setPrototypeOf"
      ]
    })),
    ArrayConstructor: new Map(Object.entries({
      es2015: [
        "from",
        "of"
      ],
      esnext: [
        "fromAsync"
      ]
    })),
    ObjectConstructor: new Map(Object.entries({
      es2015: [
        "assign",
        "getOwnPropertySymbols",
        "keys",
        "is",
        "setPrototypeOf"
      ],
      es2017: [
        "values",
        "entries",
        "getOwnPropertyDescriptors"
      ],
      es2019: [
        "fromEntries"
      ],
      es2022: [
        "hasOwn"
      ],
      es2024: [
        "groupBy"
      ]
    })),
    NumberConstructor: new Map(Object.entries({
      es2015: [
        "isFinite",
        "isInteger",
        "isNaN",
        "isSafeInteger",
        "parseFloat",
        "parseInt"
      ]
    })),
    Math: new Map(Object.entries({
      es2015: [
        "clz32",
        "imul",
        "sign",
        "log10",
        "log2",
        "log1p",
        "expm1",
        "cosh",
        "sinh",
        "tanh",
        "acosh",
        "asinh",
        "atanh",
        "hypot",
        "trunc",
        "fround",
        "cbrt"
      ]
    })),
    Map: new Map(Object.entries({
      es2015: [
        "entries",
        "keys",
        "values"
      ]
    })),
    MapConstructor: new Map(Object.entries({
      es2024: [
        "groupBy"
      ]
    })),
    Set: new Map(Object.entries({
      es2015: [
        "entries",
        "keys",
        "values"
      ],
      esnext: [
        "union",
        "intersection",
        "difference",
        "symmetricDifference",
        "isSubsetOf",
        "isSupersetOf",
        "isDisjointFrom"
      ]
    })),
    PromiseConstructor: new Map(Object.entries({
      es2015: [
        "all",
        "race",
        "reject",
        "resolve"
      ],
      es2020: [
        "allSettled"
      ],
      es2021: [
        "any"
      ],
      es2024: [
        "withResolvers"
      ]
    })),
    Symbol: new Map(Object.entries({
      es2015: [
        "for",
        "keyFor"
      ],
      es2019: [
        "description"
      ]
    })),
    WeakMap: new Map(Object.entries({
      es2015: [
        "entries",
        "keys",
        "values"
      ]
    })),
    WeakSet: new Map(Object.entries({
      es2015: [
        "entries",
        "keys",
        "values"
      ]
    })),
    String: new Map(Object.entries({
      es2015: [
        "codePointAt",
        "includes",
        "endsWith",
        "normalize",
        "repeat",
        "startsWith",
        "anchor",
        "big",
        "blink",
        "bold",
        "fixed",
        "fontcolor",
        "fontsize",
        "italics",
        "link",
        "small",
        "strike",
        "sub",
        "sup"
      ],
      es2017: [
        "padStart",
        "padEnd"
      ],
      es2019: [
        "trimStart",
        "trimEnd",
        "trimLeft",
        "trimRight"
      ],
      es2020: [
        "matchAll"
      ],
      es2021: [
        "replaceAll"
      ],
      es2022: [
        "at"
      ],
      es2024: [
        "isWellFormed",
        "toWellFormed"
      ]
    })),
    StringConstructor: new Map(Object.entries({
      es2015: [
        "fromCodePoint",
        "raw"
      ]
    })),
    DateTimeFormat: new Map(Object.entries({
      es2017: [
        "formatToParts"
      ]
    })),
    Promise: new Map(Object.entries({
      es2015: emptyArray,
      es2018: [
        "finally"
      ]
    })),
    RegExpMatchArray: new Map(Object.entries({
      es2018: [
        "groups"
      ]
    })),
    RegExpExecArray: new Map(Object.entries({
      es2018: [
        "groups"
      ]
    })),
    Intl: new Map(Object.entries({
      es2018: [
        "PluralRules"
      ]
    })),
    NumberFormat: new Map(Object.entries({
      es2018: [
        "formatToParts"
      ]
    })),
    SymbolConstructor: new Map(Object.entries({
      es2020: [
        "matchAll"
      ],
      esnext: [
        "metadata",
        "dispose",
        "asyncDispose"
      ]
    })),
    DataView: new Map(Object.entries({
      es2020: [
        "setBigInt64",
        "setBigUint64",
        "getBigInt64",
        "getBigUint64"
      ]
    })),
    BigInt: new Map(Object.entries({
      es2020: emptyArray
    })),
    RelativeTimeFormat: new Map(Object.entries({
      es2020: [
        "format",
        "formatToParts",
        "resolvedOptions"
      ]
    })),
    Int8Array: new Map(Object.entries({
      es2022: [
        "at"
      ],
      es2023: [
        "findLastIndex",
        "findLast",
        "toReversed",
        "toSorted",
        "toSpliced",
        "with"
      ]
    })),
    Uint8Array: new Map(Object.entries({
      es2022: [
        "at"
      ],
      es2023: [
        "findLastIndex",
        "findLast",
        "toReversed",
        "toSorted",
        "toSpliced",
        "with"
      ]
    })),
    Uint8ClampedArray: new Map(Object.entries({
      es2022: [
        "at"
      ],
      es2023: [
        "findLastIndex",
        "findLast",
        "toReversed",
        "toSorted",
        "toSpliced",
        "with"
      ]
    })),
    Int16Array: new Map(Object.entries({
      es2022: [
        "at"
      ],
      es2023: [
        "findLastIndex",
        "findLast",
        "toReversed",
        "toSorted",
        "toSpliced",
        "with"
      ]
    })),
    Uint16Array: new Map(Object.entries({
      es2022: [
        "at"
      ],
      es2023: [
        "findLastIndex",
        "findLast",
        "toReversed",
        "toSorted",
        "toSpliced",
        "with"
      ]
    })),
    Int32Array: new Map(Object.entries({
      es2022: [
        "at"
      ],
      es2023: [
        "findLastIndex",
        "findLast",
        "toReversed",
        "toSorted",
        "toSpliced",
        "with"
      ]
    })),
    Uint32Array: new Map(Object.entries({
      es2022: [
        "at"
      ],
      es2023: [
        "findLastIndex",
        "findLast",
        "toReversed",
        "toSorted",
        "toSpliced",
        "with"
      ]
    })),
    Float32Array: new Map(Object.entries({
      es2022: [
        "at"
      ],
      es2023: [
        "findLastIndex",
        "findLast",
        "toReversed",
        "toSorted",
        "toSpliced",
        "with"
      ]
    })),
    Float64Array: new Map(Object.entries({
      es2022: [
        "at"
      ],
      es2023: [
        "findLastIndex",
        "findLast",
        "toReversed",
        "toSorted",
        "toSpliced",
        "with"
      ]
    })),
    BigInt64Array: new Map(Object.entries({
      es2020: emptyArray,
      es2022: [
        "at"
      ],
      es2023: [
        "findLastIndex",
        "findLast",
        "toReversed",
        "toSorted",
        "toSpliced",
        "with"
      ]
    })),
    BigUint64Array: new Map(Object.entries({
      es2020: emptyArray,
      es2022: [
        "at"
      ],
      es2023: [
        "findLastIndex",
        "findLast",
        "toReversed",
        "toSorted",
        "toSpliced",
        "with"
      ]
    })),
    Error: new Map(Object.entries({
      es2022: [
        "cause"
      ]
    }))
  }))
);
function getLiteralText(node, sourceFile, flags) {
  if (sourceFile && canUseOriginalText(node, flags)) {
    return getSourceTextOfNodeFromSourceFile(sourceFile, node);
  }
  switch (node.kind) {
    case 11 /* StringLiteral */: {
      const escapeText = flags & 2 /* JsxAttributeEscape */ ? escapeJsxAttributeString : flags & 1 /* NeverAsciiEscape */ || getEmitFlags(node) & 16777216 /* NoAsciiEscaping */ ? escapeString : escapeNonAsciiString;
      if (node.singleQuote) {
        return "'" + escapeText(node.text, 39 /* singleQuote */) + "'";
      } else {
        return '"' + escapeText(node.text, 34 /* doubleQuote */) + '"';
      }
    }
    case 15 /* NoSubstitutionTemplateLiteral */:
    case 16 /* TemplateHead */:
    case 17 /* TemplateMiddle */:
    case 18 /* TemplateTail */: {
      const escapeText = flags & 1 /* NeverAsciiEscape */ || getEmitFlags(node) & 16777216 /* NoAsciiEscaping */ ? escapeString : escapeNonAsciiString;
      const rawText = node.rawText ?? escapeTemplateSubstitution(escapeText(node.text, 96 /* backtick */));
      switch (node.kind) {
        case 15 /* NoSubstitutionTemplateLiteral */:
          return "`" + rawText + "`";
        case 16 /* TemplateHead */:
          return "`" + rawText + "${";
        case 17 /* TemplateMiddle */:
          return "}" + rawText + "${";
        case 18 /* TemplateTail */:
          return "}" + rawText + "`";
      }
      break;
    }
    case 9 /* NumericLiteral */:
    case 10 /* BigIntLiteral */:
      return node.text;
    case 14 /* RegularExpressionLiteral */:
      if (flags & 4 /* TerminateUnterminatedLiterals */ && node.isUnterminated) {
        return node.text + (node.text.charCodeAt(node.text.length - 1) === 92 /* backslash */ ? " /" : "/");
      }
      return node.text;
  }
  return Debug.fail(`Literal kind '${node.kind}' not accounted for.`);
}
function canUseOriginalText(node, flags) {
  if (nodeIsSynthesized(node) || !node.parent || flags & 4 /* TerminateUnterminatedLiterals */ && node.isUnterminated) {
    return false;
  }
  if (isNumericLiteral(node)) {
    if (node.numericLiteralFlags & 26656 /* IsInvalid */) {
      return false;
    }
    if (node.numericLiteralFlags & 512 /* ContainsSeparator */) {
      return !!(flags & 8 /* AllowNumericSeparator */);
    }
  }
  return !isBigIntLiteral(node);
}
function makeIdentifierFromModuleName(moduleName) {
  return getBaseFileName(moduleName).replace(/^(\d)/, "_$1").replace(/\W/g, "_");
}
function isBlockOrCatchScoped(declaration) {
  return (getCombinedNodeFlags(declaration) & 7 /* BlockScoped */) !== 0 || isCatchClauseVariableDeclarationOrBindingElement(declaration);
}
function isCatchClauseVariableDeclarationOrBindingElement(declaration) {
  const node = getRootDeclaration(declaration);
  return node.kind === 260 /* VariableDeclaration */ && node.parent.kind === 299 /* CatchClause */;
}
function isAmbientModule(node) {
  return isModuleDeclaration(node) && (node.name.kind === 11 /* StringLiteral */ || isGlobalScopeAugmentation(node));
}
function isModuleWithStringLiteralName(node) {
  return isModuleDeclaration(node) && node.name.kind === 11 /* StringLiteral */;
}
function isNonGlobalAmbientModule(node) {
  return isModuleDeclaration(node) && isStringLiteral(node.name);
}
function isEffectiveModuleDeclaration(node) {
  return isModuleDeclaration(node) || isIdentifier(node);
}
function isShorthandAmbientModuleSymbol(moduleSymbol) {
  return isShorthandAmbientModule(moduleSymbol.valueDeclaration);
}
function isShorthandAmbientModule(node) {
  return !!node && node.kind === 267 /* ModuleDeclaration */ && !node.body;
}
function isBlockScopedContainerTopLevel(node) {
  return node.kind === 307 /* SourceFile */ || node.kind === 267 /* ModuleDeclaration */ || isFunctionLikeOrClassStaticBlockDeclaration(node);
}
function isGlobalScopeAugmentation(module2) {
  return !!(module2.flags & 2048 /* GlobalAugmentation */);
}
function isExternalModuleAugmentation(node) {
  return isAmbientModule(node) && isModuleAugmentationExternal(node);
}
function isModuleAugmentationExternal(node) {
  switch (node.parent.kind) {
    case 307 /* SourceFile */:
      return isExternalModule(node.parent);
    case 268 /* ModuleBlock */:
      return isAmbientModule(node.parent.parent) && isSourceFile(node.parent.parent.parent) && !isExternalModule(node.parent.parent.parent);
  }
  return false;
}
function getNonAugmentationDeclaration(symbol) {
  var _a;
  return (_a = symbol.declarations) == null ? void 0 : _a.find((d) => !isExternalModuleAugmentation(d) && !(isModuleDeclaration(d) && isGlobalScopeAugmentation(d)));
}
function isCommonJSContainingModuleKind(kind) {
  return kind === 1 /* CommonJS */ || kind === 100 /* Node16 */ || kind === 199 /* NodeNext */;
}
function isEffectiveExternalModule(node, compilerOptions) {
  return isExternalModule(node) || isCommonJSContainingModuleKind(getEmitModuleKind(compilerOptions)) && !!node.commonJsModuleIndicator;
}
function isEffectiveStrictModeSourceFile(node, compilerOptions) {
  switch (node.scriptKind) {
    case 1 /* JS */:
    case 3 /* TS */:
    case 2 /* JSX */:
    case 4 /* TSX */:
      break;
    default:
      return false;
  }
  if (node.isDeclarationFile) {
    return false;
  }
  if (getStrictOptionValue(compilerOptions, "alwaysStrict")) {
    return true;
  }
  if (startsWithUseStrict(node.statements)) {
    return true;
  }
  if (isExternalModule(node) || getIsolatedModules(compilerOptions)) {
    return true;
  }
  return false;
}
function isAmbientPropertyDeclaration(node) {
  return !!(node.flags & 33554432 /* Ambient */) || hasSyntacticModifier(node, 128 /* Ambient */);
}
function isBlockScope(node, parentNode) {
  switch (node.kind) {
    case 307 /* SourceFile */:
    case 269 /* CaseBlock */:
    case 299 /* CatchClause */:
    case 267 /* ModuleDeclaration */:
    case 248 /* ForStatement */:
    case 249 /* ForInStatement */:
    case 250 /* ForOfStatement */:
    case 176 /* Constructor */:
    case 174 /* MethodDeclaration */:
    case 177 /* GetAccessor */:
    case 178 /* SetAccessor */:
    case 262 /* FunctionDeclaration */:
    case 218 /* FunctionExpression */:
    case 219 /* ArrowFunction */:
    case 172 /* PropertyDeclaration */:
    case 175 /* ClassStaticBlockDeclaration */:
      return true;
    case 241 /* Block */:
      return !isFunctionLikeOrClassStaticBlockDeclaration(parentNode);
  }
  return false;
}
function isAnyImportSyntax(node) {
  switch (node.kind) {
    case 272 /* ImportDeclaration */:
    case 271 /* ImportEqualsDeclaration */:
      return true;
    default:
      return false;
  }
}
function isLateVisibilityPaintedStatement(node) {
  switch (node.kind) {
    case 272 /* ImportDeclaration */:
    case 271 /* ImportEqualsDeclaration */:
    case 243 /* VariableStatement */:
    case 263 /* ClassDeclaration */:
    case 262 /* FunctionDeclaration */:
    case 267 /* ModuleDeclaration */:
    case 265 /* TypeAliasDeclaration */:
    case 264 /* InterfaceDeclaration */:
    case 266 /* EnumDeclaration */:
      return true;
    default:
      return false;
  }
}
function hasPossibleExternalModuleReference(node) {
  return isAnyImportOrReExport(node) || isModuleDeclaration(node) || isImportTypeNode(node) || isImportCall(node);
}
function isAnyImportOrReExport(node) {
  return isAnyImportSyntax(node) || isExportDeclaration(node);
}
function getEnclosingContainer(node) {
  return findAncestor(node.parent, (n) => !!(getContainerFlags(n) & 1 /* IsContainer */));
}
function getEnclosingBlockScopeContainer(node) {
  return findAncestor(node.parent, (current) => isBlockScope(current, current.parent));
}
function forEachEnclosingBlockScopeContainer(node, cb) {
  let container = getEnclosingBlockScopeContainer(node);
  while (container) {
    cb(container);
    container = getEnclosingBlockScopeContainer(container);
  }
}
function declarationNameToString(name) {
  return !name || getFullWidth(name) === 0 ? "(Missing)" : getTextOfNode(name);
}
function getNameFromIndexInfo(info) {
  return info.declaration ? declarationNameToString(info.declaration.parameters[0].name) : void 0;
}
function isComputedNonLiteralName(name) {
  return name.kind === 167 /* ComputedPropertyName */ && !isStringOrNumericLiteralLike(name.expression);
}
function tryGetTextOfPropertyName(name) {
  var _a;
  switch (name.kind) {
    case 80 /* Identifier */:
    case 81 /* PrivateIdentifier */:
      return ((_a = name.emitNode) == null ? void 0 : _a.autoGenerate) ? void 0 : name.escapedText;
    case 11 /* StringLiteral */:
    case 9 /* NumericLiteral */:
    case 10 /* BigIntLiteral */:
    case 15 /* NoSubstitutionTemplateLiteral */:
      return escapeLeadingUnderscores(name.text);
    case 167 /* ComputedPropertyName */:
      if (isStringOrNumericLiteralLike(name.expression)) return escapeLeadingUnderscores(name.expression.text);
      return void 0;
    case 295 /* JsxNamespacedName */:
      return getEscapedTextOfJsxNamespacedName(name);
    default:
      return Debug.assertNever(name);
  }
}
function getTextOfPropertyName(name) {
  return Debug.checkDefined(tryGetTextOfPropertyName(name));
}
function entityNameToString(name) {
  switch (name.kind) {
    case 110 /* ThisKeyword */:
      return "this";
    case 81 /* PrivateIdentifier */:
    case 80 /* Identifier */:
      return getFullWidth(name) === 0 ? idText(name) : getTextOfNode(name);
    case 166 /* QualifiedName */:
      return entityNameToString(name.left) + "." + entityNameToString(name.right);
    case 211 /* PropertyAccessExpression */:
      if (isIdentifier(name.name) || isPrivateIdentifier(name.name)) {
        return entityNameToString(name.expression) + "." + entityNameToString(name.name);
      } else {
        return Debug.assertNever(name.name);
      }
    case 311 /* JSDocMemberName */:
      return entityNameToString(name.left) + "#" + entityNameToString(name.right);
    case 295 /* JsxNamespacedName */:
      return entityNameToString(name.namespace) + ":" + entityNameToString(name.name);
    default:
      return Debug.assertNever(name);
  }
}
function createDiagnosticForNode(node, message, ...args) {
  const sourceFile = getSourceFileOfNode(node);
  return createDiagnosticForNodeInSourceFile(sourceFile, node, message, ...args);
}
function createDiagnosticForNodeArray(sourceFile, nodes, message, ...args) {
  const start = skipTrivia(sourceFile.text, nodes.pos);
  return createFileDiagnostic(sourceFile, start, nodes.end - start, message, ...args);
}
function createDiagnosticForNodeInSourceFile(sourceFile, node, message, ...args) {
  const span = getErrorSpanForNode(sourceFile, node);
  return createFileDiagnostic(sourceFile, span.start, span.length, message, ...args);
}
function createDiagnosticForNodeFromMessageChain(sourceFile, node, messageChain, relatedInformation) {
  const span = getErrorSpanForNode(sourceFile, node);
  return createFileDiagnosticFromMessageChain(sourceFile, span.start, span.length, messageChain, relatedInformation);
}
function createDiagnosticForNodeArrayFromMessageChain(sourceFile, nodes, messageChain, relatedInformation) {
  const start = skipTrivia(sourceFile.text, nodes.pos);
  return createFileDiagnosticFromMessageChain(sourceFile, start, nodes.end - start, messageChain, relatedInformation);
}
function assertDiagnosticLocation(sourceText, start, length2) {
  Debug.assertGreaterThanOrEqual(start, 0);
  Debug.assertGreaterThanOrEqual(length2, 0);
  Debug.assertLessThanOrEqual(start, sourceText.length);
  Debug.assertLessThanOrEqual(start + length2, sourceText.length);
}
function createFileDiagnosticFromMessageChain(file, start, length2, messageChain, relatedInformation) {
  assertDiagnosticLocation(file.text, start, length2);
  return {
    file,
    start,
    length: length2,
    code: messageChain.code,
    category: messageChain.category,
    messageText: messageChain.next ? messageChain : messageChain.messageText,
    relatedInformation,
    canonicalHead: messageChain.canonicalHead
  };
}
function createDiagnosticForFileFromMessageChain(sourceFile, messageChain, relatedInformation) {
  return {
    file: sourceFile,
    start: 0,
    length: 0,
    code: messageChain.code,
    category: messageChain.category,
    messageText: messageChain.next ? messageChain : messageChain.messageText,
    relatedInformation
  };
}
function createDiagnosticMessageChainFromDiagnostic(diagnostic) {
  return typeof diagnostic.messageText === "string" ? {
    code: diagnostic.code,
    category: diagnostic.category,
    messageText: diagnostic.messageText,
    next: diagnostic.next
  } : diagnostic.messageText;
}
function createDiagnosticForRange(sourceFile, range, message) {
  return {
    file: sourceFile,
    start: range.pos,
    length: range.end - range.pos,
    code: message.code,
    category: message.category,
    messageText: message.message
  };
}
function getCanonicalDiagnostic(message, ...args) {
  return {
    code: message.code,
    messageText: formatMessage(message, ...args)
  };
}
function getSpanOfTokenAtPosition(sourceFile, pos) {
  const scanner = createScanner(
    sourceFile.languageVersion,
    /*skipTrivia*/
    true,
    sourceFile.languageVariant,
    sourceFile.text,
    /*onError*/
    void 0,
    pos
  );
  scanner.scan();
  const start = scanner.getTokenStart();
  return createTextSpanFromBounds(start, scanner.getTokenEnd());
}
function scanTokenAtPosition(sourceFile, pos) {
  const scanner = createScanner(
    sourceFile.languageVersion,
    /*skipTrivia*/
    true,
    sourceFile.languageVariant,
    sourceFile.text,
    /*onError*/
    void 0,
    pos
  );
  scanner.scan();
  return scanner.getToken();
}
function getErrorSpanForArrowFunction(sourceFile, node) {
  const pos = skipTrivia(sourceFile.text, node.pos);
  if (node.body && node.body.kind === 241 /* Block */) {
    const { line: startLine } = getLineAndCharacterOfPosition(sourceFile, node.body.pos);
    const { line: endLine } = getLineAndCharacterOfPosition(sourceFile, node.body.end);
    if (startLine < endLine) {
      return createTextSpan(pos, getEndLinePosition(startLine, sourceFile) - pos + 1);
    }
  }
  return createTextSpanFromBounds(pos, node.end);
}
function getErrorSpanForNode(sourceFile, node) {
  let errorNode = node;
  switch (node.kind) {
    case 307 /* SourceFile */: {
      const pos2 = skipTrivia(
        sourceFile.text,
        0,
        /*stopAfterLineBreak*/
        false
      );
      if (pos2 === sourceFile.text.length) {
        return createTextSpan(0, 0);
      }
      return getSpanOfTokenAtPosition(sourceFile, pos2);
    }
    // This list is a work in progress. Add missing node kinds to improve their error
    // spans.
    case 260 /* VariableDeclaration */:
    case 208 /* BindingElement */:
    case 263 /* ClassDeclaration */:
    case 231 /* ClassExpression */:
    case 264 /* InterfaceDeclaration */:
    case 267 /* ModuleDeclaration */:
    case 266 /* EnumDeclaration */:
    case 306 /* EnumMember */:
    case 262 /* FunctionDeclaration */:
    case 218 /* FunctionExpression */:
    case 174 /* MethodDeclaration */:
    case 177 /* GetAccessor */:
    case 178 /* SetAccessor */:
    case 265 /* TypeAliasDeclaration */:
    case 172 /* PropertyDeclaration */:
    case 171 /* PropertySignature */:
    case 274 /* NamespaceImport */:
      errorNode = node.name;
      break;
    case 219 /* ArrowFunction */:
      return getErrorSpanForArrowFunction(sourceFile, node);
    case 296 /* CaseClause */:
    case 297 /* DefaultClause */: {
      const start = skipTrivia(sourceFile.text, node.pos);
      const end = node.statements.length > 0 ? node.statements[0].pos : node.end;
      return createTextSpanFromBounds(start, end);
    }
    case 253 /* ReturnStatement */:
    case 229 /* YieldExpression */: {
      const pos2 = skipTrivia(sourceFile.text, node.pos);
      return getSpanOfTokenAtPosition(sourceFile, pos2);
    }
    case 238 /* SatisfiesExpression */: {
      const pos2 = skipTrivia(sourceFile.text, node.expression.end);
      return getSpanOfTokenAtPosition(sourceFile, pos2);
    }
    case 350 /* JSDocSatisfiesTag */: {
      const pos2 = skipTrivia(sourceFile.text, node.tagName.pos);
      return getSpanOfTokenAtPosition(sourceFile, pos2);
    }
    case 176 /* Constructor */: {
      const constructorDeclaration = node;
      const start = skipTrivia(sourceFile.text, constructorDeclaration.pos);
      const scanner = createScanner(
        sourceFile.languageVersion,
        /*skipTrivia*/
        true,
        sourceFile.languageVariant,
        sourceFile.text,
        /*onError*/
        void 0,
        start
      );
      let token = scanner.scan();
      while (token !== 137 /* ConstructorKeyword */ && token !== 1 /* EndOfFileToken */) {
        token = scanner.scan();
      }
      const end = scanner.getTokenEnd();
      return createTextSpanFromBounds(start, end);
    }
  }
  if (errorNode === void 0) {
    return getSpanOfTokenAtPosition(sourceFile, node.pos);
  }
  Debug.assert(!isJSDoc(errorNode));
  const isMissing = nodeIsMissing(errorNode);
  const pos = isMissing || isJsxText(node) ? errorNode.pos : skipTrivia(sourceFile.text, errorNode.pos);
  if (isMissing) {
    Debug.assert(pos === errorNode.pos, "This failure could trigger https://github.com/Microsoft/TypeScript/issues/20809");
    Debug.assert(pos === errorNode.end, "This failure could trigger https://github.com/Microsoft/TypeScript/issues/20809");
  } else {
    Debug.assert(pos >= errorNode.pos, "This failure could trigger https://github.com/Microsoft/TypeScript/issues/20809");
    Debug.assert(pos <= errorNode.end, "This failure could trigger https://github.com/Microsoft/TypeScript/issues/20809");
  }
  return createTextSpanFromBounds(pos, errorNode.end);
}
function isGlobalSourceFile(node) {
  return node.kind === 307 /* SourceFile */ && !isExternalOrCommonJsModule(node);
}
function isExternalOrCommonJsModule(file) {
  return (file.externalModuleIndicator || file.commonJsModuleIndicator) !== void 0;
}
function isJsonSourceFile(file) {
  return file.scriptKind === 6 /* JSON */;
}
function isEnumConst(node) {
  return !!(getCombinedModifierFlags(node) & 4096 /* Const */);
}
function isDeclarationReadonly(declaration) {
  return !!(getCombinedModifierFlags(declaration) & 8 /* Readonly */ && !isParameterPropertyDeclaration(declaration, declaration.parent));
}
function isVarAwaitUsing(node) {
  return (getCombinedNodeFlags(node) & 7 /* BlockScoped */) === 6 /* AwaitUsing */;
}
function isVarUsing(node) {
  return (getCombinedNodeFlags(node) & 7 /* BlockScoped */) === 4 /* Using */;
}
function isVarConst(node) {
  return (getCombinedNodeFlags(node) & 7 /* BlockScoped */) === 2 /* Const */;
}
function isVarConstLike(node) {
  const blockScopeKind = getCombinedNodeFlags(node) & 7 /* BlockScoped */;
  return blockScopeKind === 2 /* Const */ || blockScopeKind === 4 /* Using */ || blockScopeKind === 6 /* AwaitUsing */;
}
function isLet(node) {
  return (getCombinedNodeFlags(node) & 7 /* BlockScoped */) === 1 /* Let */;
}
function isSuperCall(n) {
  return n.kind === 213 /* CallExpression */ && n.expression.kind === 108 /* SuperKeyword */;
}
function isImportCall(n) {
  return n.kind === 213 /* CallExpression */ && n.expression.kind === 102 /* ImportKeyword */;
}
function isImportMeta(n) {
  return isMetaProperty(n) && n.keywordToken === 102 /* ImportKeyword */ && n.name.escapedText === "meta";
}
function isLiteralImportTypeNode(n) {
  return isImportTypeNode(n) && isLiteralTypeNode(n.argument) && isStringLiteral(n.argument.literal);
}
function isPrologueDirective(node) {
  return node.kind === 244 /* ExpressionStatement */ && node.expression.kind === 11 /* StringLiteral */;
}
function isCustomPrologue(node) {
  return !!(getEmitFlags(node) & 2097152 /* CustomPrologue */);
}
function isHoistedFunction(node) {
  return isCustomPrologue(node) && isFunctionDeclaration(node);
}
function isHoistedVariable(node) {
  return isIdentifier(node.name) && !node.initializer;
}
function isHoistedVariableStatement(node) {
  return isCustomPrologue(node) && isVariableStatement(node) && every(node.declarationList.declarations, isHoistedVariable);
}
function getLeadingCommentRangesOfNode(node, sourceFileOfNode) {
  return node.kind !== 12 /* JsxText */ ? getLeadingCommentRanges(sourceFileOfNode.text, node.pos) : void 0;
}
function getJSDocCommentRanges(node, text) {
  const commentRanges = node.kind === 169 /* Parameter */ || node.kind === 168 /* TypeParameter */ || node.kind === 218 /* FunctionExpression */ || node.kind === 219 /* ArrowFunction */ || node.kind === 217 /* ParenthesizedExpression */ || node.kind === 260 /* VariableDeclaration */ || node.kind === 281 /* ExportSpecifier */ ? concatenate(getTrailingCommentRanges(text, node.pos), getLeadingCommentRanges(text, node.pos)) : getLeadingCommentRanges(text, node.pos);
  return filter(commentRanges, (comment) => comment.end <= node.end && // Due to parse errors sometime empty parameter may get comments assigned to it that end up not in parameter range
  text.charCodeAt(comment.pos + 1) === 42 /* asterisk */ && text.charCodeAt(comment.pos + 2) === 42 /* asterisk */ && text.charCodeAt(comment.pos + 3) !== 47 /* slash */);
}
var fullTripleSlashReferencePathRegEx = /^\/\/\/\s*<reference\s+path\s*=\s*(?:'[^']*'|"[^"]*").*?\/>/;
var fullTripleSlashReferenceTypeReferenceDirectiveRegEx = /^\/\/\/\s*<reference\s+types\s*=\s*(?:'[^']*'|"[^"]*").*?\/>/;
var fullTripleSlashLibReferenceRegEx = /^\/\/\/\s*<reference\s+lib\s*=\s*(?:'[^']*'|"[^"]*").*?\/>/;
var fullTripleSlashAMDReferencePathRegEx = /^\/\/\/\s*<amd-dependency\s+path\s*=\s*(?:'[^']*'|"[^"]*").*?\/>/;
var fullTripleSlashAMDModuleRegEx = /^\/\/\/\s*<amd-module\s+(?:\S.*?)??\/>/;
var defaultLibReferenceRegEx = /^\/\/\/\s*<reference\s+no-default-lib\s*=\s*(?:'[^']*'|"[^"]*")\s*\/>/;
function isPartOfTypeNode(node) {
  if (182 /* FirstTypeNode */ <= node.kind && node.kind <= 205 /* LastTypeNode */) {
    return true;
  }
  switch (node.kind) {
    case 133 /* AnyKeyword */:
    case 159 /* UnknownKeyword */:
    case 150 /* NumberKeyword */:
    case 163 /* BigIntKeyword */:
    case 154 /* StringKeyword */:
    case 136 /* BooleanKeyword */:
    case 155 /* SymbolKeyword */:
    case 151 /* ObjectKeyword */:
    case 157 /* UndefinedKeyword */:
    case 106 /* NullKeyword */:
    case 146 /* NeverKeyword */:
      return true;
    case 116 /* VoidKeyword */:
      return node.parent.kind !== 222 /* VoidExpression */;
    case 233 /* ExpressionWithTypeArguments */:
      return isPartOfTypeExpressionWithTypeArguments(node);
    case 168 /* TypeParameter */:
      return node.parent.kind === 200 /* MappedType */ || node.parent.kind === 195 /* InferType */;
    // Identifiers and qualified names may be type nodes, depending on their context. Climb
    // above them to find the lowest container
    case 80 /* Identifier */:
      if (node.parent.kind === 166 /* QualifiedName */ && node.parent.right === node) {
        node = node.parent;
      } else if (node.parent.kind === 211 /* PropertyAccessExpression */ && node.parent.name === node) {
        node = node.parent;
      }
      Debug.assert(node.kind === 80 /* Identifier */ || node.kind === 166 /* QualifiedName */ || node.kind === 211 /* PropertyAccessExpression */, "'node' was expected to be a qualified name, identifier or property access in 'isPartOfTypeNode'.");
    // falls through
    case 166 /* QualifiedName */:
    case 211 /* PropertyAccessExpression */:
    case 110 /* ThisKeyword */: {
      const { parent } = node;
      if (parent.kind === 186 /* TypeQuery */) {
        return false;
      }
      if (parent.kind === 205 /* ImportType */) {
        return !parent.isTypeOf;
      }
      if (182 /* FirstTypeNode */ <= parent.kind && parent.kind <= 205 /* LastTypeNode */) {
        return true;
      }
      switch (parent.kind) {
        case 233 /* ExpressionWithTypeArguments */:
          return isPartOfTypeExpressionWithTypeArguments(parent);
        case 168 /* TypeParameter */:
          return node === parent.constraint;
        case 345 /* JSDocTemplateTag */:
          return node === parent.constraint;
        case 172 /* PropertyDeclaration */:
        case 171 /* PropertySignature */:
        case 169 /* Parameter */:
        case 260 /* VariableDeclaration */:
          return node === parent.type;
        case 262 /* FunctionDeclaration */:
        case 218 /* FunctionExpression */:
        case 219 /* ArrowFunction */:
        case 176 /* Constructor */:
        case 174 /* MethodDeclaration */:
        case 173 /* MethodSignature */:
        case 177 /* GetAccessor */:
        case 178 /* SetAccessor */:
          return node === parent.type;
        case 179 /* CallSignature */:
        case 180 /* ConstructSignature */:
        case 181 /* IndexSignature */:
          return node === parent.type;
        case 216 /* TypeAssertionExpression */:
          return node === parent.type;
        case 213 /* CallExpression */:
        case 214 /* NewExpression */:
        case 215 /* TaggedTemplateExpression */:
          return contains(parent.typeArguments, node);
      }
    }
  }
  return false;
}
function isPartOfTypeExpressionWithTypeArguments(node) {
  return isJSDocImplementsTag(node.parent) || isJSDocAugmentsTag(node.parent) || isHeritageClause(node.parent) && !isExpressionWithTypeArgumentsInClassExtendsClause(node);
}
function forEachReturnStatement(body, visitor) {
  return traverse(body);
  function traverse(node) {
    switch (node.kind) {
      case 253 /* ReturnStatement */:
        return visitor(node);
      case 269 /* CaseBlock */:
      case 241 /* Block */:
      case 245 /* IfStatement */:
      case 246 /* DoStatement */:
      case 247 /* WhileStatement */:
      case 248 /* ForStatement */:
      case 249 /* ForInStatement */:
      case 250 /* ForOfStatement */:
      case 254 /* WithStatement */:
      case 255 /* SwitchStatement */:
      case 296 /* CaseClause */:
      case 297 /* DefaultClause */:
      case 256 /* LabeledStatement */:
      case 258 /* TryStatement */:
      case 299 /* CatchClause */:
        return forEachChild(node, traverse);
    }
  }
}
function forEachYieldExpression(body, visitor) {
  return traverse(body);
  function traverse(node) {
    switch (node.kind) {
      case 229 /* YieldExpression */:
        visitor(node);
        const operand = node.expression;
        if (operand) {
          traverse(operand);
        }
        return;
      case 266 /* EnumDeclaration */:
      case 264 /* InterfaceDeclaration */:
      case 267 /* ModuleDeclaration */:
      case 265 /* TypeAliasDeclaration */:
        return;
      default:
        if (isFunctionLike(node)) {
          if (node.name && node.name.kind === 167 /* ComputedPropertyName */) {
            traverse(node.name.expression);
            return;
          }
        } else if (!isPartOfTypeNode(node)) {
          forEachChild(node, traverse);
        }
    }
  }
}
function getRestParameterElementType(node) {
  if (node && node.kind === 188 /* ArrayType */) {
    return node.elementType;
  } else if (node && node.kind === 183 /* TypeReference */) {
    return singleOrUndefined(node.typeArguments);
  } else {
    return void 0;
  }
}
function getMembersOfDeclaration(node) {
  switch (node.kind) {
    case 264 /* InterfaceDeclaration */:
    case 263 /* ClassDeclaration */:
    case 231 /* ClassExpression */:
    case 187 /* TypeLiteral */:
      return node.members;
    case 210 /* ObjectLiteralExpression */:
      return node.properties;
  }
}
function isVariableLike(node) {
  if (node) {
    switch (node.kind) {
      case 208 /* BindingElement */:
      case 306 /* EnumMember */:
      case 169 /* Parameter */:
      case 303 /* PropertyAssignment */:
      case 172 /* PropertyDeclaration */:
      case 171 /* PropertySignature */:
      case 304 /* ShorthandPropertyAssignment */:
      case 260 /* VariableDeclaration */:
        return true;
    }
  }
  return false;
}
function isVariableDeclarationInVariableStatement(node) {
  return node.parent.kind === 261 /* VariableDeclarationList */ && node.parent.parent.kind === 243 /* VariableStatement */;
}
function isCommonJsExportedExpression(node) {
  if (!isInJSFile(node)) return false;
  return isObjectLiteralExpression(node.parent) && isBinaryExpression(node.parent.parent) && getAssignmentDeclarationKind(node.parent.parent) === 2 /* ModuleExports */ || isCommonJsExportPropertyAssignment(node.parent);
}
function isCommonJsExportPropertyAssignment(node) {
  if (!isInJSFile(node)) return false;
  return isBinaryExpression(node) && getAssignmentDeclarationKind(node) === 1 /* ExportsProperty */;
}
function isValidESSymbolDeclaration(node) {
  return (isVariableDeclaration(node) ? isVarConst(node) && isIdentifier(node.name) && isVariableDeclarationInVariableStatement(node) : isPropertyDeclaration(node) ? hasEffectiveReadonlyModifier(node) && hasStaticModifier(node) : isPropertySignature(node) && hasEffectiveReadonlyModifier(node)) || isCommonJsExportPropertyAssignment(node);
}
function introducesArgumentsExoticObject(node) {
  switch (node.kind) {
    case 174 /* MethodDeclaration */:
    case 173 /* MethodSignature */:
    case 176 /* Constructor */:
    case 177 /* GetAccessor */:
    case 178 /* SetAccessor */:
    case 262 /* FunctionDeclaration */:
    case 218 /* FunctionExpression */:
      return true;
  }
  return false;
}
function unwrapInnermostStatementOfLabel(node, beforeUnwrapLabelCallback) {
  while (true) {
    if (beforeUnwrapLabelCallback) {
      beforeUnwrapLabelCallback(node);
    }
    if (node.statement.kind !== 256 /* LabeledStatement */) {
      return node.statement;
    }
    node = node.statement;
  }
}
function isFunctionBlock(node) {
  return node && node.kind === 241 /* Block */ && isFunctionLike(node.parent);
}
function isObjectLiteralMethod(node) {
  return node && node.kind === 174 /* MethodDeclaration */ && node.parent.kind === 210 /* ObjectLiteralExpression */;
}
function isObjectLiteralOrClassExpressionMethodOrAccessor(node) {
  return (node.kind === 174 /* MethodDeclaration */ || node.kind === 177 /* GetAccessor */ || node.kind === 178 /* SetAccessor */) && (node.parent.kind === 210 /* ObjectLiteralExpression */ || node.parent.kind === 231 /* ClassExpression */);
}
function isIdentifierTypePredicate(predicate) {
  return predicate && predicate.kind === 1 /* Identifier */;
}
function isThisTypePredicate(predicate) {
  return predicate && predicate.kind === 0 /* This */;
}
function forEachPropertyAssignment(objectLiteral, key, callback, key2) {
  return forEach(objectLiteral == null ? void 0 : objectLiteral.properties, (property) => {
    if (!isPropertyAssignment(property)) return void 0;
    const propName = tryGetTextOfPropertyName(property.name);
    return key === propName || key2 && key2 === propName ? callback(property) : void 0;
  });
}
function getPropertyArrayElementValue(objectLiteral, propKey, elementValue) {
  return forEachPropertyAssignment(objectLiteral, propKey, (property) => isArrayLiteralExpression(property.initializer) ? find(property.initializer.elements, (element) => isStringLiteral(element) && element.text === elementValue) : void 0);
}
function getTsConfigObjectLiteralExpression(tsConfigSourceFile) {
  if (tsConfigSourceFile && tsConfigSourceFile.statements.length) {
    const expression = tsConfigSourceFile.statements[0].expression;
    return tryCast(expression, isObjectLiteralExpression);
  }
}
function getTsConfigPropArrayElementValue(tsConfigSourceFile, propKey, elementValue) {
  return forEachTsConfigPropArray(tsConfigSourceFile, propKey, (property) => isArrayLiteralExpression(property.initializer) ? find(property.initializer.elements, (element) => isStringLiteral(element) && element.text === elementValue) : void 0);
}
function forEachTsConfigPropArray(tsConfigSourceFile, propKey, callback) {
  return forEachPropertyAssignment(getTsConfigObjectLiteralExpression(tsConfigSourceFile), propKey, callback);
}
function getContainingFunction(node) {
  return findAncestor(node.parent, isFunctionLike);
}
function getContainingClass(node) {
  return findAncestor(node.parent, isClassLike);
}
function getContainingClassStaticBlock(node) {
  return findAncestor(node.parent, (n) => {
    if (isClassLike(n) || isFunctionLike(n)) {
      return "quit";
    }
    return isClassStaticBlockDeclaration(n);
  });
}
function getContainingFunctionOrClassStaticBlock(node) {
  return findAncestor(node.parent, isFunctionLikeOrClassStaticBlockDeclaration);
}
function getContainingClassExcludingClassDecorators(node) {
  const decorator = findAncestor(node.parent, (n) => isClassLike(n) ? "quit" : isDecorator(n));
  return decorator && isClassLike(decorator.parent) ? getContainingClass(decorator.parent) : getContainingClass(decorator ?? node);
}
function getThisContainer(node, includeArrowFunctions, includeClassComputedPropertyName) {
  Debug.assert(node.kind !== 307 /* SourceFile */);
  while (true) {
    node = node.parent;
    if (!node) {
      return Debug.fail();
    }
    switch (node.kind) {
      case 167 /* ComputedPropertyName */:
        if (includeClassComputedPropertyName && isClassLike(node.parent.parent)) {
          return node;
        }
        node = node.parent.parent;
        break;
      case 170 /* Decorator */:
        if (node.parent.kind === 169 /* Parameter */ && isClassElement(node.parent.parent)) {
          node = node.parent.parent;
        } else if (isClassElement(node.parent)) {
          node = node.parent;
        }
        break;
      case 219 /* ArrowFunction */:
        if (!includeArrowFunctions) {
          continue;
        }
      // falls through
      case 262 /* FunctionDeclaration */:
      case 218 /* FunctionExpression */:
      case 267 /* ModuleDeclaration */:
      case 175 /* ClassStaticBlockDeclaration */:
      case 172 /* PropertyDeclaration */:
      case 171 /* PropertySignature */:
      case 174 /* MethodDeclaration */:
      case 173 /* MethodSignature */:
      case 176 /* Constructor */:
      case 177 /* GetAccessor */:
      case 178 /* SetAccessor */:
      case 179 /* CallSignature */:
      case 180 /* ConstructSignature */:
      case 181 /* IndexSignature */:
      case 266 /* EnumDeclaration */:
      case 307 /* SourceFile */:
        return node;
    }
  }
}
function isThisContainerOrFunctionBlock(node) {
  switch (node.kind) {
    // Arrow functions use the same scope, but may do so in a "delayed" manner
    // For example, `const getThis = () => this` may be before a super() call in a derived constructor
    case 219 /* ArrowFunction */:
    case 262 /* FunctionDeclaration */:
    case 218 /* FunctionExpression */:
    case 172 /* PropertyDeclaration */:
      return true;
    case 241 /* Block */:
      switch (node.parent.kind) {
        case 176 /* Constructor */:
        case 174 /* MethodDeclaration */:
        case 177 /* GetAccessor */:
        case 178 /* SetAccessor */:
          return true;
        default:
          return false;
      }
    default:
      return false;
  }
}
function isInTopLevelContext(node) {
  if (isIdentifier(node) && (isClassDeclaration(node.parent) || isFunctionDeclaration(node.parent)) && node.parent.name === node) {
    node = node.parent;
  }
  const container = getThisContainer(
    node,
    /*includeArrowFunctions*/
    true,
    /*includeClassComputedPropertyName*/
    false
  );
  return isSourceFile(container);
}
function getNewTargetContainer(node) {
  const container = getThisContainer(
    node,
    /*includeArrowFunctions*/
    false,
    /*includeClassComputedPropertyName*/
    false
  );
  if (container) {
    switch (container.kind) {
      case 176 /* Constructor */:
      case 262 /* FunctionDeclaration */:
      case 218 /* FunctionExpression */:
        return container;
    }
  }
  return void 0;
}
function getSuperContainer(node, stopOnFunctions) {
  while (true) {
    node = node.parent;
    if (!node) {
      return void 0;
    }
    switch (node.kind) {
      case 167 /* ComputedPropertyName */:
        node = node.parent;
        break;
      case 262 /* FunctionDeclaration */:
      case 218 /* FunctionExpression */:
      case 219 /* ArrowFunction */:
        if (!stopOnFunctions) {
          continue;
        }
      // falls through
      case 172 /* PropertyDeclaration */:
      case 171 /* PropertySignature */:
      case 174 /* MethodDeclaration */:
      case 173 /* MethodSignature */:
      case 176 /* Constructor */:
      case 177 /* GetAccessor */:
      case 178 /* SetAccessor */:
      case 175 /* ClassStaticBlockDeclaration */:
        return node;
      case 170 /* Decorator */:
        if (node.parent.kind === 169 /* Parameter */ && isClassElement(node.parent.parent)) {
          node = node.parent.parent;
        } else if (isClassElement(node.parent)) {
          node = node.parent;
        }
        break;
    }
  }
}
function getImmediatelyInvokedFunctionExpression(func) {
  if (func.kind === 218 /* FunctionExpression */ || func.kind === 219 /* ArrowFunction */) {
    let prev = func;
    let parent = func.parent;
    while (parent.kind === 217 /* ParenthesizedExpression */) {
      prev = parent;
      parent = parent.parent;
    }
    if (parent.kind === 213 /* CallExpression */ && parent.expression === prev) {
      return parent;
    }
  }
}
function isSuperProperty(node) {
  const kind = node.kind;
  return (kind === 211 /* PropertyAccessExpression */ || kind === 212 /* ElementAccessExpression */) && node.expression.kind === 108 /* SuperKeyword */;
}
function isThisProperty(node) {
  const kind = node.kind;
  return (kind === 211 /* PropertyAccessExpression */ || kind === 212 /* ElementAccessExpression */) && node.expression.kind === 110 /* ThisKeyword */;
}
function isThisInitializedDeclaration(node) {
  var _a;
  return !!node && isVariableDeclaration(node) && ((_a = node.initializer) == null ? void 0 : _a.kind) === 110 /* ThisKeyword */;
}
function isThisInitializedObjectBindingExpression(node) {
  return !!node && (isShorthandPropertyAssignment(node) || isPropertyAssignment(node)) && isBinaryExpression(node.parent.parent) && node.parent.parent.operatorToken.kind === 64 /* EqualsToken */ && node.parent.parent.right.kind === 110 /* ThisKeyword */;
}
function getEntityNameFromTypeNode(node) {
  switch (node.kind) {
    case 183 /* TypeReference */:
      return node.typeName;
    case 233 /* ExpressionWithTypeArguments */:
      return isEntityNameExpression(node.expression) ? node.expression : void 0;
    // TODO(rbuckton): These aren't valid TypeNodes, but we treat them as such because of `isPartOfTypeNode`, which returns `true` for things that aren't `TypeNode`s.
    case 80 /* Identifier */:
    case 166 /* QualifiedName */:
      return node;
  }
  return void 0;
}
function getInvokedExpression(node) {
  switch (node.kind) {
    case 215 /* TaggedTemplateExpression */:
      return node.tag;
    case 286 /* JsxOpeningElement */:
    case 285 /* JsxSelfClosingElement */:
      return node.tagName;
    case 226 /* BinaryExpression */:
      return node.right;
    case 289 /* JsxOpeningFragment */:
      return node;
    default:
      return node.expression;
  }
}
function nodeCanBeDecorated(useLegacyDecorators, node, parent, grandparent) {
  if (useLegacyDecorators && isNamedDeclaration(node) && isPrivateIdentifier(node.name)) {
    return false;
  }
  switch (node.kind) {
    case 263 /* ClassDeclaration */:
      return true;
    case 231 /* ClassExpression */:
      return !useLegacyDecorators;
    case 172 /* PropertyDeclaration */:
      return parent !== void 0 && (useLegacyDecorators ? isClassDeclaration(parent) : isClassLike(parent) && !hasAbstractModifier(node) && !hasAmbientModifier(node));
    case 177 /* GetAccessor */:
    case 178 /* SetAccesso