/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.tools;

import com.fasterxml.jackson.databind.JsonMappingException;
import java.io.IOException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.stream.Collectors;
import joptsimple.AbstractOptionSpec;
import joptsimple.ArgumentAcceptingOptionSpec;
import joptsimple.OptionSpec;
import joptsimple.OptionSpecBuilder;
import joptsimple.ValueConverter;
import joptsimple.util.EnumConverter;
import org.apache.kafka.clients.admin.Admin;
import org.apache.kafka.common.ElectionType;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.ClusterAuthorizationException;
import org.apache.kafka.common.errors.ElectionNotNeededException;
import org.apache.kafka.common.errors.TimeoutException;
import org.apache.kafka.common.utils.Exit;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.server.util.CommandDefaultOptions;
import org.apache.kafka.server.util.CommandLineUtils;
import org.apache.kafka.server.util.Json;
import org.apache.kafka.server.util.json.DecodeJson;
import org.apache.kafka.server.util.json.JsonObject;
import org.apache.kafka.server.util.json.JsonValue;
import org.apache.kafka.tools.AdminCommandFailedException;
import org.apache.kafka.tools.AdminOperationException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LeaderElectionCommand {
    private static final Logger LOG = LoggerFactory.getLogger(LeaderElectionCommand.class);
    private static final DecodeJson.DecodeString STRING = new DecodeJson.DecodeString();
    private static final DecodeJson.DecodeInteger INT = new DecodeJson.DecodeInteger();

    public static void main(String ... args) {
        Exit.exit((int)LeaderElectionCommand.mainNoExit(args));
    }

    static int mainNoExit(String ... args) {
        try {
            LeaderElectionCommand.run(Duration.ofMillis(30000L), args);
            return 0;
        }
        catch (Throwable e) {
            System.err.println(e.getMessage());
            System.err.println(Utils.stackTrace((Throwable)e));
            return 1;
        }
    }

    static void run(Duration timeoutMs, String ... args) throws Exception {
        LeaderElectionCommandOptions commandOptions = new LeaderElectionCommandOptions(args);
        commandOptions.maybePrintHelpOrVersion();
        commandOptions.validate();
        ElectionType electionType = commandOptions.getElectionType();
        Optional<Set> jsonFileTopicPartitions = Optional.ofNullable(commandOptions.getPathToJsonFile()).map(LeaderElectionCommand::parseReplicaElectionData);
        Optional<String> topicOption = Optional.ofNullable(commandOptions.getTopic());
        Optional<Integer> partitionOption = Optional.ofNullable(commandOptions.getPartition());
        Optional singleTopicPartition = topicOption.isPresent() && partitionOption.isPresent() ? Optional.of(Collections.singleton(new TopicPartition(topicOption.get(), partitionOption.get().intValue()))) : Optional.empty();
        Optional<Set<TopicPartition>> topicPartitions = jsonFileTopicPartitions.or(() -> singleTopicPartition);
        Properties props = new Properties();
        if (commandOptions.hasAdminClientConfig()) {
            props.putAll((Map<?, ?>)Utils.loadProps((String)commandOptions.getAdminClientConfig()));
        }
        props.setProperty("bootstrap.servers", commandOptions.getBootstrapServer());
        if (!props.containsKey("default.api.timeout.ms")) {
            props.setProperty("default.api.timeout.ms", Integer.toString((int)timeoutMs.toMillis()));
        }
        if (!props.containsKey("request.timeout.ms")) {
            props.setProperty("request.timeout.ms", Integer.toString((int)(timeoutMs.toMillis() / 2L)));
        }
        try (Admin adminClient = Admin.create((Properties)props);){
            LeaderElectionCommand.electLeaders(adminClient, electionType, topicPartitions);
        }
    }

    private static void electLeaders(Admin client, ElectionType electionType, Optional<Set<TopicPartition>> partitions) {
        String partitionsAsString;
        Map electionResults;
        LOG.debug("Calling AdminClient.electLeaders({}, {})", (Object)electionType, partitions.orElse(null));
        try {
            electionResults = (Map)client.electLeaders(electionType, (Set)partitions.orElse(null)).partitions().get();
        }
        catch (ExecutionException e) {
            if (e.getCause() instanceof TimeoutException) {
                String message = "Timeout waiting for election results";
                System.out.println(message);
                throw new AdminCommandFailedException(message, e.getCause());
            }
            if (e.getCause() instanceof ClusterAuthorizationException) {
                String message = "Not authorized to perform leader election";
                System.out.println(message);
                throw new AdminCommandFailedException(message, e.getCause().getCause());
            }
            throw new RuntimeException(e);
        }
        catch (InterruptedException e) {
            System.out.println("Error while making request");
            throw new RuntimeException(e);
        }
        HashSet succeeded = new HashSet();
        HashSet noop = new HashSet();
        HashMap<TopicPartition, Throwable> failed = new HashMap<TopicPartition, Throwable>();
        electionResults.forEach((key, error) -> {
            if (error.isPresent()) {
                if (error.get() instanceof ElectionNotNeededException) {
                    noop.add(key);
                } else {
                    failed.put((TopicPartition)key, (Throwable)error.get());
                }
            } else {
                succeeded.add(key);
            }
        });
        if (!succeeded.isEmpty()) {
            partitionsAsString = succeeded.stream().map(TopicPartition::toString).collect(Collectors.joining(", "));
            System.out.println(String.format("Successfully completed leader election (%s) for partitions %s", electionType, partitionsAsString));
        }
        if (!noop.isEmpty()) {
            partitionsAsString = noop.stream().map(TopicPartition::toString).collect(Collectors.joining(", "));
            System.out.println(String.format("Valid replica already elected for partitions %s", partitionsAsString));
        }
        if (!failed.isEmpty()) {
            AdminCommandFailedException rootException = new AdminCommandFailedException(String.format("%s replica(s) could not be elected", failed.size()));
            failed.forEach((key, value) -> {
                System.out.println(String.format("Error completing leader election (%s) for partition: %s: %s", electionType, key, value));
                rootException.addSuppressed((Throwable)value);
            });
            throw rootException;
        }
    }

    private static Set<TopicPartition> parseReplicaElectionData(String path) {
        try {
            Optional jsonFile = Json.parseFull((String)Utils.readFileAsString((String)path));
            return jsonFile.map(js -> {
                try {
                    return LeaderElectionCommand.topicPartitions(js);
                }
                catch (JsonMappingException e) {
                    throw new RuntimeException(e);
                }
            }).orElseThrow(() -> new AdminOperationException("Replica election data is empty"));
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private static Set<TopicPartition> topicPartitions(JsonValue js) throws JsonMappingException {
        return js.asJsonObject().get("partitions").map(partitionsList -> {
            try {
                return LeaderElectionCommand.toTopicPartition(partitionsList);
            }
            catch (JsonMappingException e) {
                throw new RuntimeException(e);
            }
        }).orElseThrow(() -> new AdminOperationException("Replica election data is missing \"partitions\" field"));
    }

    private static Set<TopicPartition> toTopicPartition(JsonValue partitionsList) throws JsonMappingException {
        ArrayList<TopicPartition> partitions = new ArrayList<TopicPartition>();
        Iterator iterator = partitionsList.asJsonArray().iterator();
        while (iterator.hasNext()) {
            JsonObject partitionJs = ((JsonValue)iterator.next()).asJsonObject();
            String topic = (String)partitionJs.apply("topic").to((DecodeJson)STRING);
            int partition = (Integer)partitionJs.apply("partition").to((DecodeJson)INT);
            partitions.add(new TopicPartition(topic, partition));
        }
        Set duplicatePartitions = partitions.stream().filter(i -> Collections.frequency(partitions, i) > 1).collect(Collectors.toSet());
        if (duplicatePartitions.size() > 0) {
            throw new AdminOperationException(String.format("Replica election data contains duplicate partitions: %s", String.join((CharSequence)",", duplicatePartitions.toString())));
        }
        return new HashSet<TopicPartition>(partitions);
    }

    static class LeaderElectionCommandOptions
    extends CommandDefaultOptions {
        private final ArgumentAcceptingOptionSpec<String> bootstrapServer;
        private final ArgumentAcceptingOptionSpec<String> adminClientConfig;
        private final ArgumentAcceptingOptionSpec<String> pathToJsonFile;
        private final ArgumentAcceptingOptionSpec<String> topic;
        private final ArgumentAcceptingOptionSpec<Integer> partition;
        private final OptionSpecBuilder allTopicPartitions;
        private final ArgumentAcceptingOptionSpec<ElectionType> electionType;

        public LeaderElectionCommandOptions(String[] args) {
            super(args);
            this.bootstrapServer = this.parser.accepts("bootstrap-server", "A hostname and port for the broker to connect to, in the form host:port. Multiple comma separated URLs can be given. REQUIRED.").withRequiredArg().describedAs("host:port").ofType(String.class);
            this.adminClientConfig = this.parser.accepts("admin.config", "Configuration properties files to pass to the admin client").withRequiredArg().describedAs("config file").ofType(String.class);
            this.pathToJsonFile = this.parser.accepts("path-to-json-file", "The JSON file with the list  of partition for which leader elections should be performed. This is an example format. \n{\"partitions\":\n\t[{\"topic\": \"foo\", \"partition\": 1},\n\t {\"topic\": \"foobar\", \"partition\": 2}]\n}\nNot allowed if --all-topic-partitions or --topic flags are specified.").withRequiredArg().describedAs("Path to JSON file").ofType(String.class);
            this.topic = this.parser.accepts("topic", "Name of topic for which to perform an election. Not allowed if --path-to-json-file or --all-topic-partitions is specified.").withRequiredArg().describedAs("topic name").ofType(String.class);
            this.partition = this.parser.accepts("partition", "Partition id for which to perform an election. REQUIRED if --topic is specified.").withRequiredArg().describedAs("partition id").ofType(Integer.class);
            this.allTopicPartitions = this.parser.accepts("all-topic-partitions", "Perform election on all of the eligible topic partitions based on the type of election (see the --election-type flag). Not allowed if --topic or --path-to-json-file is specified.");
            this.electionType = this.parser.accepts("election-type", "Type of election to attempt. Possible values are \"preferred\" for preferred leader election or \"unclean\" for unclean leader election. If preferred election is selection, the election is only performed if the current leader is not the preferred leader for the topic partition. If unclean election is selected, the election is only performed if there are no leader for the topic partition. REQUIRED.").withRequiredArg().describedAs("election type").withValuesConvertedBy((ValueConverter)new ElectionTypeConverter());
            this.options = this.parser.parse(args);
        }

        public boolean hasAdminClientConfig() {
            return this.options.has(this.adminClientConfig);
        }

        public ElectionType getElectionType() {
            return (ElectionType)this.options.valueOf(this.electionType);
        }

        public String getPathToJsonFile() {
            return (String)this.options.valueOf(this.pathToJsonFile);
        }

        public String getBootstrapServer() {
            return (String)this.options.valueOf(this.bootstrapServer);
        }

        public String getAdminClientConfig() {
            return (String)this.options.valueOf(this.adminClientConfig);
        }

        public String getTopic() {
            return (String)this.options.valueOf(this.topic);
        }

        public Integer getPartition() {
            return (Integer)this.options.valueOf(this.partition);
        }

        public void validate() {
            ArrayList<String> missingOptions = new ArrayList<String>();
            if (!this.options.has(this.bootstrapServer)) {
                missingOptions.add((String)this.bootstrapServer.options().get(0));
            }
            if (!this.options.has(this.electionType)) {
                missingOptions.add((String)this.electionType.options().get(0));
            }
            if (!missingOptions.isEmpty()) {
                throw new AdminCommandFailedException("Missing required option(s): " + String.join((CharSequence)", ", missingOptions));
            }
            List<AbstractOptionSpec> mutuallyExclusiveOptions = Arrays.asList(this.topic, this.allTopicPartitions, this.pathToJsonFile);
            long mutuallyExclusiveOptionsCount = mutuallyExclusiveOptions.stream().filter(abstractOptionSpec -> this.options.has((OptionSpec)abstractOptionSpec)).count();
            if (mutuallyExclusiveOptionsCount != 1L) {
                throw new AdminCommandFailedException("One and only one of the following options is required: " + mutuallyExclusiveOptions.stream().map(opt -> (String)opt.options().get(0)).collect(Collectors.joining(", ")));
            }
            if (this.options.has(this.topic) && !this.options.has(this.partition)) {
                throw new AdminCommandFailedException(String.format("Missing required option(s): %s", this.partition.options().get(0)));
            }
            if (!this.options.has(this.topic) && this.options.has(this.partition)) {
                throw new AdminCommandFailedException(String.format("Option %s is only allowed if %s is used", this.partition.options().get(0), this.topic.options().get(0)));
            }
        }

        public void maybePrintHelpOrVersion() {
            CommandLineUtils.maybePrintHelpOrVersion((CommandDefaultOptions)this, (String)"This tool attempts to elect a new leader for a set of topic partitions. The type of elections supported are preferred replicas and unclean replicas.");
        }
    }

    static class ElectionTypeConverter
    extends EnumConverter<ElectionType> {
        public ElectionTypeConverter() {
            super(ElectionType.class);
        }
    }
}

