# combo_row.py
#
# Copyright 2025 Mitchell Winkelman (Heliguy)
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# SPDX-License-Identifier: GPL-3.0-or-later

from gi.repository import GObject, Adw, Gtk, Gio

from refine.compositions import RefineCompositions

from typing import Any, cast

import logging


class RefineComboRow(Adw.ComboRow):
    """
    Base combo row widget implementing the `schema-id` and `key` properties.
    It is assumed to derive from it, rather than using it directly.
    """

    __gtype_name__ = "RefineComboRow"

    is_valid_setting = GObject.Property(type=bool, default=True)
    settings = GObject.Property(type=Gio.Settings)
    restart_required = GObject.Property(type=bool, default=False)
    is_default = GObject.Property(type=bool, default=True)
    schema = GObject.Property(type=Gio.SettingsSchema)

    original_list: list[str]
    transformed_list: list[str]
    revert_button: Gtk.Button
    revealer: Gtk.Revealer

    def __init__(self, **kwargs: Any) -> None:
        super().__init__(**kwargs)

        self.bind_property("is-valid-setting", self, "sensitive")

        self.connect("notify::key", self.__key_set)

    def __key_set(self, *args: Any) -> None:
        self.revert_button.connect("clicked", self.__reset_button_clicked)
        self.bind_property(
            "is-default",
            self.revert_button,
            "sensitive",
            GObject.BindingFlags.SYNC_CREATE,
        )
        RefineCompositions.set_up_separator_revealer(
            self.revealer, self.revert_button, "sensitive"
        )
        self.connect("inconsistent", self.__inconsistent_set)
        self.settings.connect("changed", self.__settings_changed)

    def __settings_changed(self, *args: Any) -> None:
        row_string = self.original_list[self.get_selected()]
        settings_string = self.settings.get_string(self.key)

        if row_string != settings_string:
            self.emit("inconsistent")

    def __inconsistent_set(self, *args: Any) -> None:
        self.handler_block_by_func(self.selected_callback)  # type: ignore [no-untyped-call]

        string = self.settings.get_string(self.key)

        self.set_selected(self.original_list.index(string))
        self.set_is_default(cast(str, self.key))

        self.handler_unblock_by_func(self.selected_callback)  # type: ignore [no-untyped-call]

    def __reset_button_clicked(self, *args: Any) -> None:
        logging.debug("Revert button pressed")

        string_object = cast(
            Gtk.StringObject, self.settings.get_default_value(self.key)
        )
        if not string_object:
            return

        string = string_object.get_string()

        self.settings.reset(self.key)
        self.set_selected(self.original_list.index(string))
        self.set_is_default(cast(str, self.key))

    def set_is_default(self, key: str) -> None:
        """Update the `RefineComboRow.is-default` property."""

        default_string = self.settings.get_default_value(key).get_string()

        row_string = self.original_list[self.get_selected()]

        self.is_default = row_string != default_string

    def selected_callback(self, *args: Any) -> None:
        """Callback method for selecting an item."""

        row_string = self.original_list[self.get_selected()]

        self.set_is_default(cast(str, self.key))
        self.settings.set_string(self.key, row_string)
        self.value_set(row_string)

    @RefineCompositions.schema_id
    def schema_id(self, schema_id: str) -> None: ...

    @RefineCompositions.key
    def key(self, key: str) -> None: ...

    def value_set(self, value: Any) -> None:
        """Log key-value pair."""

        logging.debug(f"Set key “{self.key}” to “{value}”")

        if not self.restart_required:
            return

        toplevels = Gtk.Window.get_toplevels()

        if not (window := toplevels.get_item(0)):
            return

        window.emit("restart-items-changed", self)

    @GObject.Signal
    def inconsistent(self) -> None:
        """Log inconsistency signal."""

        logging.info(f"{self.__class__.__name__}::inconsistent signal emitted")
