# flags_switch_row.py
#
# Copyright 2024 Hari Rana (TheEvilSkeleton)
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# SPDX-License-Identifier: GPL-3.0-or-later


from typing import Any, cast

from gi.repository import GObject, Gtk, Gio


from refine.compositions import RefineCompositions
from refine.widgets.string_row import RefineStringRow

import logging


@Gtk.Template(resource_path="/page/tesk/Refine/widgets/flags-switch-row.ui")
class RefineFlagsSwitchRow(RefineStringRow):
    """A `RefineActionRow` to enable or disable a feature from a string list."""

    __gtype_name__ = "RefineFlagsSwitchRow"

    switch = Gtk.Template.Child()

    schema = GObject.Property(type=Gio.SettingsSchema)

    def __init__(self, **kwargs: Any) -> None:
        super().__init__(**kwargs)

    def get_string(self) -> list[str]:
        """Retrieve list from key."""

        return cast(list[str], self.settings.get_strv(self.key))

    def active_set(self, *args: Any) -> None:
        """Set whether @self is in its “on” or “off” position."""

        flags = self.get_string()

        value = cast(str, self.value)
        if self.active:
            flags.append(value)
        else:
            try:
                flags.remove(value)
            except ValueError:
                pass

        flags = list(set(flags))

        self.settings.set_strv(self.key, flags)
        self.value_set(flags)

    @RefineCompositions.value
    def value(self, value: str) -> None:
        schema_key = self.schema.get_key(self.key)

        try:
            key_range = schema_key.get_range()[1]

            if value not in key_range:
                raise TypeError
        except (IndexError, TypeError):
            self.is_valid_setting = False
            logging.warning(f"Value {value} does not exist. Skipping…")
            return
