# (c) Copyright 2022. CodeWeavers, Inc.

from gi.repository import Gdk
from gi.repository import GLib
from gi.repository import Gtk
from gi.repository import Pango

import iso8601

import bottleoperation
import cxguitools
import cxproduct
import cxutils
import distversion
import iconview
import installtask
import pyop
import ratingdialog

from cxutils import cxgettext as _


class BottleViewController(iconview.IconViewController):

    def __init__(self, parent_window):
        iconview.IconViewController.__init__(self)

        self.xml = Gtk.Builder()
        self.xml.set_translation_domain('crossover')
        self.xml.add_from_file(cxguitools.get_ui_path('bottleview'))
        self.xml.connect_signals(self)

        self.parent_window = parent_window
        self.bottle = None

        self.xml.get_object('IconView').set_model(self.launcher_list_store)
        self.setup_icon_view(self.xml.get_object('IconView'), False)

        self.installed_software_store = Gtk.ListStore(str, object)
        self.installed_software_store.set_sort_column_id(0, Gtk.SortType.ASCENDING)
        self.xml.get_object('InstalledSoftware').set_model(self.installed_software_store)

        label = Gtk.CellRendererText()
        label.set_property('ellipsize', Pango.EllipsizeMode.END)
        label.set_property('ellipsize-set', True)
        label.set_property('scale', 0.8)

        column = Gtk.TreeViewColumn(_('Name'), label, text=0)
        self.xml.get_object('InstalledSoftware').append_column(column)
        self.xml.get_object('InstalledSoftware').connect('button-press-event', self.on_InstalledSoftware_button_press_event)
        self.xml.get_object('InstalledSoftware').connect('popup-menu', self.on_InstalledSoftware_popup_menu)

        self.xml.get_object('Description').get_buffer().connect('changed', self.on_Description_changed)
        self.xml.get_object('Description').get_buffer().connect('insert-text', self.on_Description_insert_text)

        self.control_panel_ready = False
        self.control_panel_spinner = None
        self.control_panel_widget = None

        self.set_drag_destination()
        self.load_expander_states()

    def on_expander_activate(self, _widget):
        expander_states = []
        expander_states.append(self.xml.get_object('BottleActions').get_expanded())
        expander_states.append(self.xml.get_object('AdvancedSettings').get_expanded())
        expander_states.append(self.xml.get_object('ControlPanels').get_expanded())
        expander_states.append(self.xml.get_object('BottleDetails').get_expanded())

        cxproduct.set_config_string('OfficeSetup', 'ExpanderStates',
                                    '\0'.join(str(int(i)) for i in expander_states).encode('unicode_escape'))

    def load_expander_states(self):
        expander_states = cxproduct.get_config_string('OfficeSetup', 'ExpanderStates', '')
        if not expander_states:
            return

        expander_states = [bool(int(i)) for i in cxutils.string_to_utf8(expander_states).decode('unicode_escape').split('\0')]
        self.xml.get_object('BottleActions').set_expanded(expander_states[0])
        self.xml.get_object('AdvancedSettings').set_expanded(expander_states[1])
        self.xml.get_object('ControlPanels').set_expanded(expander_states[2])
        self.xml.get_object('BottleDetails').set_expanded(expander_states[3])

    def set_drag_destination(self):
        view = self.get_view()

        view.drag_dest_set(Gtk.DestDefaults.DROP | Gtk.DestDefaults.MOTION, [], Gdk.DragAction.COPY)
        view.drag_dest_add_uri_targets()
        view.connect('drag-data-received', self.on_drag_data_received)

    def on_drag_data_received(self, _widget, _drag_context, _x, _y, selection_data, _info, _timestamp):
        uris = selection_data.get_uris()
        if uris:
            application = self.parent_window.get_application()
            application.lookup_action('open-file').activate(
                GLib.Variant.new_string(cxutils.uri_to_path(uris[0])))

    def set_bottle(self, bottle):
        if self.bottle:
            self.bottle.remove_change_delegate(self)

        self.xml.get_object('Launchers').hide()
        self.xml.get_object('Loading').show()
        self.xml.get_object('NoResults').hide()

        self.bottle = bottle

        self.launcher_list_store.clear()
        self.update_launchers()

        self.control_panel_ready = False
        self.update_control_panels()

        self.installed_software_store.clear()

        self.bottle.add_change_delegate(self)
        self.update_header()
        self.update_sidebar()
        self.update_status()

    def get_view(self):
        return self.xml.get_object('BottleView')

    def get_icon_view(self):
        return self.xml.get_object('IconView')

    def update_launchers(self):
        if self.bottle.is_busy:
            return

        launchers = self.launchers_for_bottle(self.bottle)
        self.set_launchers(launchers)

        if launchers:
            self.xml.get_object('Launchers').show()
            self.xml.get_object('Loading').hide()
            self.xml.get_object('NoResults').hide()
        else:
            self.xml.get_object('Launchers').hide()
            self.xml.get_object('Loading').hide()
            self.xml.get_object('NoResults').show()

    def update_status(self):
        self.xml.get_object('BottleStatus').set_label(self.bottle.status)
        self.xml.get_object('BottleType').set_label(self.bottle.windows_version_display_name)

    def update_header(self):
        # FIXME: Don't duplicate this

        self.xml.get_object('PackageName').set_text(self.bottle.name)

        profile = self.bottle.profile
        if profile and profile.app_profile:
            app_profile = profile.app_profile

            rating = app_profile.medal_rating
            icon_names = []
            if not rating:
                icon_names = [cxguitools.get_icon_name(('non-starred', ), Gtk.IconSize.LARGE_TOOLBAR, symbolic=True)] * 5
            elif 1 <= rating <= 5:
                icon_names = [cxguitools.get_icon_name(('starred', ), Gtk.IconSize.LARGE_TOOLBAR, symbolic=True)] * rating \
                    + [cxguitools.get_icon_name(('non-starred', ), Gtk.IconSize.LARGE_TOOLBAR, symbolic=True)] * (5 - rating)
            else:
                icon_names = [cxguitools.get_icon_name(('dialog-warning', 'gtk-dialog-warning'), Gtk.IconSize.LARGE_TOOLBAR)]

            for i, icon in enumerate(icon_names):
                self.xml.get_object('RatingStar' + str(i + 1)).set_from_icon_name(icon, Gtk.IconSize.LARGE_TOOLBAR)
                self.xml.get_object('RatingStar' + str(i + 1)).show()

            for i in range(len(icon_names), 5):
                self.xml.get_object('RatingStar' + str(i + 1)).hide()

            self.xml.get_object('RatingDescription').set_text(app_profile.rating_description)

            if rating and app_profile.medal_version:
                date = None
                try:
                    date = iso8601.parse_date(app_profile.medal_date).strftime('%x')
                    version_tooltip = ((_("Last Ranked: %s") % date) + '\n' +
                                       (_("Number of Rankings: %s") % app_profile.medal_count))
                except iso8601.ParseError:
                    version_tooltip = None

                if date:
                    version_text = _("Last tested with CrossOver %(version)s on %(date)s") % {
                        'version': app_profile.medal_version,
                        'date': date}
                else:
                    version_text = _("Last tested with CrossOver %(version)s") % {'version': app_profile.medal_version}

                self.xml.get_object('LastTested').set_text(version_text)
                self.xml.get_object('LastTested').set_tooltip_text(version_tooltip)
                self.xml.get_object('LastTested').show()
            else:
                self.xml.get_object('LastTested').hide()

            if profile.is_component or distversion.IS_PREVIEW:
                self.xml.get_object('RatingButton').set_markup('')
            else:
                self.xml.get_object('RatingButton').set_markup('<a href="">' + _('Leave a Rating') + '</a>')

            self.xml.get_object('RatingBox').show()
        else:
            self.xml.get_object('RatingBox').hide()

        self.update_dependency_button()

    def update_dependency_button(self):
        if installtask.get_missing_dependencies(self.bottle):
            self.xml.get_object('InstallMissingButton').set_markup('<a href="">' + _('Install missing dependencies') + '</a>')
            self.xml.get_object('InstallMissingIcon').show()
            return

        self.xml.get_object('InstallMissingButton').set_markup('')
        self.xml.get_object('InstallMissingIcon').hide()

    def update_sidebar(self):
        self.xml.get_object('RunCommandButton').set_tooltip_text(
            _("Select a command to run in the '%s' bottle") % self.bottle.name)

        self.xml.get_object('RunWithOptionsButton').set_tooltip_text(
            _("Run the selected launcher with options"))

        self.xml.get_object('OpenCDriveButton').set_tooltip_text(
            _("Open the C: drive for the '%s' bottle") % self.bottle.name)

        self.xml.get_object('InstallIntoBottleButton').set_tooltip_text(
            _("Install a Windows application into the '%s' bottle") % self.bottle.name)

        self.xml.get_object('DeleteButton').set_tooltip_text(
            _("Delete the '%s' bottle") % self.bottle.name)

        self.xml.get_object('QuitButton').set_sensitive(self.bottle.is_active)

        if self.bottle.can_force_quit:
            self.xml.get_object('QuitButton').set_label(_('Force Quit'))
            self.xml.get_object('QuitButton').set_tooltip_text(
                _("Force quit all applications in the '%s' bottle") % self.bottle.name)
            self.xml.get_object('QuitButton').set_sensitive(True)
        else:
            self.xml.get_object('QuitButton').set_label(_('Quit All Applications'))
            self.xml.get_object('QuitButton').set_tooltip_text(
                _("Quit all applications in the '%s' bottle") % self.bottle.name)

        application = self.parent_window.get_application()
        backend = application.lookup_action('graphics').get_state().get_string()
        self.xml.get_object('GraphicsComboBox').set_active_id(backend)

        self.xml.get_object('HighResButtonBox').set_tooltip_text(_("High resolution mode"))
        if self.bottle.is_high_resolution_enabled_state == self.bottle.STATUS_HIGHRES_UNKNOWN:
            self.xml.get_object('HighResButtonBox').set_sensitive(False)
        elif self.bottle.is_high_resolution_enabled_state == self.bottle.STATUS_HIGHRES_UNAVAILABLE:
            self.xml.get_object('HighResButtonBox').set_sensitive(False)
            self.xml.get_object('HighResButtonBox').set_tooltip_text(
                _("High resolution mode can't be enabled due to your current screen resolution"))
        elif self.bottle.is_high_resolution_enabled_state == self.bottle.STATUS_HIGHRES_ENABLED:
            self.xml.get_object('HighResButtonBox').set_sensitive(True)
        elif self.bottle.is_high_resolution_enabled_state == self.bottle.STATUS_HIGHRES_DISABLED:
            self.xml.get_object('HighResButtonBox').set_sensitive(True)

        self.xml.get_object('PreviewButtonBox').set_tooltip_text(_("Enable this bottle for CrossOver Preview"))
        if self.bottle.is_preview_enabled_state == self.bottle.STATUS_PREVIEW_UNKNOWN:
            self.xml.get_object('PreviewButtonBox').set_sensitive(False)
        elif self.bottle.is_preview_enabled_state == self.bottle.STATUS_PREVIEW_ENABLED:
            self.xml.get_object('PreviewButtonBox').set_sensitive(True)
        elif self.bottle.is_preview_enabled_state == self.bottle.STATUS_PREVIEW_DISABLED:
            self.xml.get_object('PreviewButtonBox').set_sensitive(True)

        if not distversion.IS_PREVIEW:
            self.xml.get_object('PreviewButtonBox').hide()

        if self.bottle.is_managed or self.bottle.is_disabled:
            self.xml.get_object('GraphicsButtonBox').set_sensitive(False)
            self.xml.get_object('HighResButtonBox').set_sensitive(False)
        else:
            self.xml.get_object('GraphicsButtonBox').set_sensitive(True)

        if self.bottle.is_managed:
            self.xml.get_object('PreviewButtonBox').set_sensitive(False)

        if self.bottle.control_panel_ready != self.control_panel_ready:
            self.update_control_panels()

        self.update_installed_software()

        self.update_description()

    def on_control_panel_clicked(self, _widget, control_panel):
        self.bottle.launch_control_panel_applet(control_panel['exe'])

    def get_control_panel_button(self, control_panel):
        button = Gtk.Button(label=control_panel['name'])
        button.set_tooltip_text(control_panel['description'])
        button.connect('clicked', self.on_control_panel_clicked, control_panel)
        return button

    @staticmethod
    def control_panel_key(panel):
        known_panels = ['winecfg.exe',
                        'joy.cpl',
                        'cxassoceditui',
                        'cxmenueditui',
                        'wineboot.exe',
                        'taskmgr.exe',
                        'inetcpl.cpl']

        if panel['exe'] in known_panels:
            return str(known_panels.index(panel['exe']))

        return '9' + panel['name']

    def update_installed_software(self):
        if not self.bottle.installed_packages_ready:
            self.bottle.load_application_info()

            self.xml.get_object('InstalledSoftwareLabel').set_sensitive(False)
            self.xml.get_object('InstalledSoftware').set_sensitive(False)
            return

        applications = {}
        for package in self.bottle.installed_packages.values():
            name = cxutils.string_to_str(package.name)
            applications[name] = package

        iterator = self.installed_software_store.get_iter_first()
        while iterator:
            name = self.installed_software_store.get_value(iterator, 0)
            application = self.installed_software_store.get_value(iterator, 1)
            if name in applications and applications[name] == application:
                del applications[name]
                iterator = self.installed_software_store.iter_next(iterator)
            else:
                if not self.installed_software_store.remove(iterator):
                    break

        for name, application in applications.items():
            self.installed_software_store.append((name, application))

        self.xml.get_object('InstalledSoftwareLabel').set_sensitive(True)
        self.xml.get_object('InstalledSoftware').set_sensitive(True)

    def update_description(self):
        description_buffer = self.xml.get_object('Description').get_buffer()
        current_description = description_buffer.get_text(description_buffer.get_start_iter(),
                                                          description_buffer.get_end_iter(), True)
        if current_description != self.bottle.current_description:
            description_buffer.set_text(self.bottle.current_description)

        self.xml.get_object('Description').set_editable(self.bottle.can_edit)

    def update_control_panels(self):
        if self.control_panel_widget:
            self.xml.get_object('ControlPanels').remove(self.control_panel_widget)
            self.control_panel_widget = None

        if self.control_panel_spinner:
            self.xml.get_object('ControlPanels').remove(self.control_panel_spinner)
            self.control_panel_spinner = None

        self.control_panel_ready = self.bottle.control_panel_ready
        if not self.control_panel_ready:
            self.bottle.load_control_panel_info()

            self.control_panel_spinner = Gtk.Spinner()
            self.xml.get_object('ControlPanels').add(self.control_panel_spinner)
            self.control_panel_spinner.start()
            self.control_panel_spinner.show()
            return

        self.control_panel_widget = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=6)
        self.control_panel_widget.set_margin_top(6)
        control_panels = sorted(self.bottle.control_panel_table, key=self.control_panel_key)
        for control_panel in control_panels:
            self.control_panel_widget.pack_start(self.get_control_panel_button(control_panel), True, True, 0)

        self.xml.get_object('ControlPanels').add(self.control_panel_widget)
        self.control_panel_widget.show_all()

    def build_uninstall_menu(self):
        selection = self.xml.get_object('InstalledSoftware').get_selection()
        if not selection:
            return None

        model, iterator = selection.get_selected()
        package = model.get_value(iterator, 1)
        if not package.uninstallcommand:
            return None

        item = Gtk.MenuItem.new_with_mnemonic(_('_Uninstall'))
        item.connect('activate', self.on_uninstall_activate)
        item.set_sensitive(self.bottle.can_install)
        item.show()

        menu = Gtk.Menu()
        menu.append(item)

        return menu

    def on_uninstall_activate(self, _widget):
        selection = self.xml.get_object('InstalledSoftware').get_selection()
        if not selection:
            return

        model, iterator = selection.get_selected()
        package = model.get_value(iterator, 1)
        _retcode, _out, _err = cxutils.run(package.uninstallcommand)

    def on_InstalledSoftware_button_press_event(self, view, event):
        if event.button == 3: # right click
            path, _column, _x, _y = view.get_path_at_pos(int(event.x), int(event.y))
            if not path:
                return False

            self.xml.get_object('InstalledSoftware').set_cursor(path, None, False)

            menu = self.build_uninstall_menu()
            if menu:
                menu.attach_to_widget(view)
                cxguitools.popup_at_pointer(menu, view, event)
                return True

        return False

    def on_InstalledSoftware_popup_menu(self, view):
        menu = self.build_uninstall_menu()
        if menu:
            path, column = view.get_cursor()
            if not path:
                return False

            menu.attach_to_widget(view)

            rect = view.get_cell_area(path, column)
            cxguitools.popup_at_rect(
                menu,
                view.get_window(),
                rect,
                Gdk.Gravity.NORTH_WEST,
                Gdk.Gravity.NORTH_WEST,
                None)

            return True

        return False

    def on_GraphicsComboBox_changed(self, widget):
        backend = widget.get_active_id()
        application = self.parent_window.get_application()
        application.lookup_action('graphics').change_state(GLib.Variant.new_string(backend))

    def on_Description_changed(self, text_buffer):
        if not self.xml.get_object('Description').get_editable():
            return

        new_description = text_buffer.get_text(text_buffer.get_start_iter(), text_buffer.get_end_iter(), True)
        self.bottle.current_description = new_description
        self.bottle.write_description()

    @staticmethod
    def on_Description_insert_text(text_buffer, _iter, text, _length):
        if '\n' in text:
            text_buffer.stop_emission_by_name('insert-text')

    def on_IconView_selection_changed(self, view):
        selected_items = view.get_selected_items()
        if selected_items:
            self.xml.get_object('RunWithOptionsButton').show()
        else:
            self.xml.get_object('RunWithOptionsButton').hide()

    def on_InstallMissingButton_activate_link(self, _widget, _url):
        application = self.parent_window.get_application()
        application.lookup_action('install-missing').activate()

    def on_RatingButton_activate_link(self, _widget, _uri):
        ratingdialog.RatingController(self.bottle, self.parent_window)
        return True

    def on_QuitButton_clicked(self, _widget):
        if not self.bottle.is_usable:
            return

        bottleoperation.quit_bottle(self.bottle, self.bottle.can_force_quit)

    # BottleWrapper delegate functions
    def bottleChanged(self, _bottle):
        assert pyop.is_main_thread()

        self.update_launchers()
        self.update_sidebar()
        self.update_status()
        self.update_header()
