/*
**  Copyright (c) 2006 Sendmail, Inc. and its suppliers.
**	All rights reserved.
**
**  $Id: nesting-filter.c,v 1.64 2006/06/09 23:41:01 msk Exp $
*/

#ifndef lint
static char nesting_filter_c_id[] = "@(#)$Id: nesting-filter.c,v 1.64 2006/06/09 23:41:01 msk Exp $";
#endif /* !lint */

/* system includes */
#include <sys/types.h>
#include <sys/param.h>
#ifdef DEBUG
# include <sys/socket.h>
#endif /* DEBUG */
#ifdef SOLARIS
# include <iso/limits_iso.h>
#endif /* SOLARIS */
#include <netinet/in.h>
#include <arpa/inet.h>
#include <syslog.h>
#include <errno.h>
#include <sysexits.h>
#include <assert.h>
#include <stdlib.h>
#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>
#include <pwd.h>
#include <grp.h>
#include <ctype.h>
#ifdef DEBUG
# include <netdb.h>
# include <time.h>
#endif /* DEBUG */

#ifdef SOLARIS
# define _PATH_DEVNULL	"/dev/null"
#else /* SOLARIS */
# include <paths.h>
#endif /* SOLARIS */

/* sendmail includes */
#include <sm/cdefs.h>
#include <sm/string.h>

/* libmilter includes */
#ifndef DEBUG
# include <libmilter/mfapi.h>
#endif /* !DEBUG */

/* nesting-filter includes */
#include "nesting-filter.h"
#include "util.h"

/* DEBUGGING STUFF */
#ifdef DEBUG
# define MI_SUCCESS	1
# define MI_FAILURE	(-1)
# define SMFIS_CONTINUE	0
# define SMFIS_ACCEPT	1
# define SMFIS_REJECT	2
# define SMFIS_DISCARD	3
# define SMFIS_TEMPFAIL	4
# define sfsistat	int
# define SMFICTX	void
# define _SOCK_ADDR	struct sockaddr

int smfi_addheader __P((void *, char *, char *));
int smfi_chgheader __P((void *, char *, int, char *));
void *smfi_getpriv __P((void *));
char *smfi_getsymval __P((void *, char *));
int smfi_insheader __P((void *, int, char *, char *));
int smfi_replacebody __P((void *, char *, size_t));
int smfi_progress __P((void *));
int smfi_setconn __P((char *));
void smfi_setpriv __P((void *, void *));
int smfi_setreply __P((void *, char *, char *, char *));

char *smfis_ret[] = {
	"SMFIS_CONTINUE",
	"SMFIS_ACCEPT",
	"SMFIS_REJECT",
	"SMFIS_DISCARD",
	"SMFIS_TEMPFAIL",
};
#endif /* DEBUG */

/*
**  Header -- a handle referring to a header
*/

typedef struct Header * HEADER;
struct Header
{
	char *		hdr_hdr;
	char *		hdr_val;
	struct Header *	hdr_next;
};

/*
**  MIME -- MIME node
*/

typedef struct Mime * MIME;
struct Mime
{
	unsigned long	mime_depth;		/* current depth */
	MIME		mime_parent;		/* parent node */
	MIME		mime_child;		/* child node */
	STRING		mime_boundary;		/* boundary at this level */
};

/*
**  Context -- filter context
*/

typedef struct Context * CONTEXT;
struct Context
{
	bool		ctx_encap;		/* encapsulated msg pending */
	bool		ctx_inhdrs;		/* inside a header block */
	bool		ctx_inrfc822;		/* current type msg/rfc822 */
	bool		ctx_knownbound;		/* saw a previous boundary */
	unsigned int	ctx_lineno;		/* line number */
	unsigned long	ctx_rdepth;		/* current recursion count */
	char		ctx_lastchar;		/* last character read */
	HEADER		ctx_hqhead;		/* header queue head */
	HEADER		ctx_hqtail;		/* header queue tail */
	char *		ctx_jobid;		/* job ID */
	_SOCK_ADDR	ctx_addr;		/* client IP information */
	SMFICTX *	ctx_milter;		/* milter context */
	VECTOR		ctx_recycle;		/* recyclable MIME nodes */
	MIME		ctx_mime;		/* root of the MIME tree */
	MIME		ctx_curmime;		/* current MIME node */
	TOKENS		ctx_tokens;		/* tokens */
	STRING		ctx_bodybuf;		/* current body buffer */
	STRING		ctx_hdrbuf;		/* current header buffer */
	char		ctx_hostname[MAXHOSTNAMELEN + 1];
						/* client hostname */
	char		ctx_sender[MAXADDRESS + 1];
						/* envelope sender */
	char		ctx_from[MAXADDRESS + 1];
						/* header sender */
};

/* PROTOTYPES */
sfsistat mlfi_body __P((SMFICTX *, u_char *, size_t));
sfsistat mlfi_connect __P((SMFICTX *, char *, _SOCK_ADDR *));
sfsistat mlfi_envfrom __P((SMFICTX *, char **));
sfsistat mlfi_eoh __P((SMFICTX *));
sfsistat mlfi_eom __P((SMFICTX *));
sfsistat mlfi_header __P((SMFICTX *, char *, char *));

static void nf_msgcleanup __P((SMFICTX *));
static MIME nf_newmime __P((CONTEXT));
static int nf_parsebodyct __P((CONTEXT, unsigned char *));
#if 0
static void nf_printmime __P((MIME));
#endif /* 0 */

/* GLOBALS */
bool addxhdr;					/* add identifying header? */
bool dolog;					/* syslog */
bool no_i_whine;				/* noted ${i} is undefined */
int blockaction;				/* action to take */
unsigned long maxdepth;				/* maximum allowed depth */
char *progname;					/* program name */
char *rejectmsg;				/* rejection message */

#ifdef DEBUG
static void *fakepriv;				/* fake private space */
#endif /* DEBUG */

#define	NF_DEBUG(x)	(getenv("NFDEBUG") != NULL && \
			 strchr(getenv("NFDEBUG"), (x)) != NULL)

#define	TRYFREE(x)	do { \
				if ((x) != NULL) \
				{ \
					free(x); \
					(x) = NULL; \
				} \
			} while (0)

#ifndef PARSE_FAIL_ACTION
# define PARSE_FAIL_ACTION	SMFIS_TEMPFAIL
#endif /* ! PARSE_FAIL_ACTION */



/*
**  ==================================================================
**  BEGIN private section
*/

/*
**  NF_INITCONTEXT -- initialize filter context
**
**  Parameters:
**  	None.
**
**  Return value:
**  	A pointer to an allocated and initialized filter context, or NULL
**  	on failure.
**
**  Side effects:
**  	Crop circles near Birmingham.
*/

static CONTEXT
nf_initcontext(void)
{
	struct Context *ctx;

	ctx = malloc(sizeof(struct Context));
	if (ctx == NULL)
		return NULL;

	memset(ctx, '\0', sizeof(struct Context));

	return ctx;
}

/*
**  NF_FINDHEADER -- find a header
**
**  Parameters:
**  	nfc -- filter context
**  	hname -- name of the header of interest
**  	instance -- which instance is wanted (0 = first)
**
**  Return value:
**  	HEADER handle, or NULL if not found.
*/

HEADER
nf_findheader(CONTEXT sic, char *hname, int instance)
{
	HEADER hdr;

	assert(sic != NULL);
	assert(hname != NULL);

	hdr = sic->ctx_hqhead;

	while (hdr != NULL)
	{
		if (strcasecmp(hdr->hdr_hdr, hname) == 0)
		{
			if (instance == 0)
				return hdr;
			else
				instance--;
		}

		hdr = hdr->hdr_next;
	}

	return NULL;
}

/*
**  NF_FINDBOUNDARY -- see if a line matches a MIME boundary we know
**
**  Parameters:
**  	root -- top of the MIME list to search
**  	str -- input string
**
**  Return value:
**  	MIME node matching "boundary" or NULL if not found.
*/

MIME
nf_findboundary(MIME root, char *str)
{
	size_t blen;
	MIME mime;
	char *b;

	assert(root != NULL);
	assert(str != NULL);

	if (str[0] != '-' || str[1] != '-')
		return NULL;

	mime = root;

	while (mime != NULL)
	{
		b = string_data(mime->mime_boundary);
		blen = strlen(b);
		if (strncmp(b, &str[2], blen) == 0 &&
		    (isspace(str[blen + 2]) ||
		     str[blen + 2] == '-' ||
		     str[blen + 2] == '\0'))
			return mime;

		mime = mime->mime_child;
	}

	return NULL;
}

/*
**  NF_STDIO -- set up standard I/O descriptors and stuff
**
**  Parameters:
**  	None.
**
**  Return value:
**  	None.
**
**  Side effects:
**  	stdin, stdout and stderr are routed to /dev/null and the process
**  	starts running in its own process group.
*/

static void
nf_stdio(void)
{
	int devnull;

	/* this only fails silently, but that's OK */
	devnull = open(_PATH_DEVNULL, O_RDWR, 0);
	if (devnull != -1)
	{
		(void) dup2(devnull, 0);
		(void) dup2(devnull, 1);
		(void) dup2(devnull, 2);
		(void) close(devnull);
	}

	(void) setsid();
}

/*
**  NF_MSGCLEANUP -- release local resources related to a message
**
**  Parameters:
**  	ctx -- milter context
**
**  Return value:
**  	None.
*/

static void
nf_msgcleanup(SMFICTX *ctx)
{
	CONTEXT nfc;

#ifndef DEBUG
	assert(ctx != NULL);
#endif /* !DEBUG */

	nfc = (CONTEXT) smfi_getpriv(ctx);

	/* release memory */
	if (nfc != NULL)
	{
		if (nfc->ctx_hqhead != NULL)
		{
			HEADER hdr;
			HEADER prev;

			hdr = nfc->ctx_hqhead;
			while (hdr != NULL)
			{
				TRYFREE(hdr->hdr_hdr);
				TRYFREE(hdr->hdr_val);
				prev = hdr;
				hdr = hdr->hdr_next;
				TRYFREE(prev);
			}

			nfc->ctx_hqhead = NULL;
			nfc->ctx_hqtail = NULL;
		}

		if (nfc->ctx_mime != NULL)
		{
			MIME tmp;
			MIME next;

			tmp = nfc->ctx_mime;
			while (tmp != NULL)
			{
				next = tmp->mime_child;

				if (vector_append(nfc->ctx_recycle,
				                  tmp) == -1)
				{
					string_free(tmp->mime_boundary);
					free(tmp);
				}

				tmp = next;
			}

			nfc->ctx_mime = NULL;
		}

		string_blank(nfc->ctx_bodybuf);
		if (nfc->ctx_hdrbuf != NULL)
			string_blank(nfc->ctx_hdrbuf);
		nfc->ctx_lastchar = '\0';
		nfc->ctx_rdepth = 0;
		nfc->ctx_lineno = 0;
		nfc->ctx_encap = FALSE;
		nfc->ctx_inhdrs = FALSE;
		nfc->ctx_inrfc822 = FALSE;
		nfc->ctx_knownbound = FALSE;
	}
}

/*
**  NF_CONNCLEANUP -- release local resources related to a connection
**
**  Parameters:
**  	ctx -- milter context
**
**  Return value:
**  	None.
*/

static void
nf_conncleanup(SMFICTX *ctx)
{
	CONTEXT nfc;

#ifndef DEBUG
	assert(ctx != NULL);
#endif /* !DEBUG */

	nfc = (CONTEXT) smfi_getpriv(ctx);

	/* release memory */
	if (nfc != NULL)
	{
		MIME mime;

		nf_msgcleanup(ctx);

		if (nfc->ctx_tokens != NULL)
			tokens_free(nfc->ctx_tokens);

		while (vector_length(nfc->ctx_recycle) > 0)
		{
			mime = vector_remove(nfc->ctx_recycle, 0);
			if (mime == NULL)
				continue;

			string_free(mime->mime_boundary);
			free(mime);
		}
		vector_free(nfc->ctx_recycle);

		if (nfc->ctx_hdrbuf != NULL)
			string_free(nfc->ctx_hdrbuf);

		if (nfc->ctx_bodybuf != NULL)
			string_free(nfc->ctx_bodybuf);

		free(nfc);
		smfi_setpriv(ctx, NULL);
	}
}

/*
**  NF_NEWMIME -- create or recycle a MIME node
**
**  Parameters:
**  	nfc -- connection context
**
**  Return value:
**  	A new/recycled MIME node ready for use, or NULL on error.
*/

static MIME
nf_newmime(CONTEXT nfc)
{
	MIME new;

	assert(nfc != NULL);

	if (vector_length(nfc->ctx_recycle) != 0)
	{
		new = vector_remove(nfc->ctx_recycle, 0);
		if (new == NULL)
			return NULL;
	}
	else
	{
		new = (MIME) malloc(sizeof *new);
		if (new == NULL)
			return NULL;
		new->mime_boundary = string_new(0);
		if (new->mime_boundary == NULL)
		{
			free(new);
			return NULL;
		}
	}

	string_blank(new->mime_boundary);
	new->mime_parent = NULL;
	new->mime_child = NULL;
	new->mime_depth = 0;

	return new;
}

/*
**  NF_PARSEBODYCT -- parse and handle a Content-Type: header found in body
**
**  Parameters:
**  	nfc -- connection context
**  	hdr -- header found
**
**  Return value:
**  	1 if parse succeeded, 0 if parse failed, -1 if system error.
**
**  Side effects:
**  	Updates the MIME stack in "nfc" if necessary and successful.
*/

static int
nf_parsebodyct(CONTEXT nfc, unsigned char *hdr)
{
	int status;
	int ntokens;
	STRING str;
	unsigned char *p;
	unsigned char **tokens;

	assert(nfc != NULL);
	assert(hdr != NULL);

	str = tokens_getdata(nfc->ctx_tokens);
	string_blank(str);
	p = strchr(hdr, ':');
	if (string_cat(str, p + 1) == -1)
		return -1;
	for (p = string_data(str); *p != '\0'; p++)
	{
		if (*p == '\r' || *p == '\n')
			*p = ' ';
	}

	status = tokens_tokenize(nfc->ctx_tokens, MIME_SPECIALS,
	                         MIME_COMMENTS);
	if (status == -2)
	{
		int save_errno = errno;

		if (dolog)
		{
			syslog(LOG_ERR, "%s malloc(): %s", nfc->ctx_jobid,
			       strerror(save_errno));
		}

		errno = save_errno;
		return -1;
	}
	else if (status < 1)
	{
		if (dolog)
		{
			syslog(LOG_ERR, "%s invalid Content-Type `%s'",
			       nfc->ctx_jobid, string_data(str));
		}

		return 0;
	}

	ntokens = status;
	tokens = tokens_gettokens(nfc->ctx_tokens);

	if (strcasecmp(tokens[0], "message") == 0 &&
	    (tokens[1] != NULL && strcasecmp(tokens[1], "/") == 0) &&
	    (tokens[2] != NULL && strcasecmp(tokens[2], "rfc822") == 0))
	{

		nfc->ctx_rdepth++;
		nfc->ctx_encap = TRUE;
	}
	else if (strcasecmp(tokens[0], "multipart") == 0)
	{
		int n;
		int c;
		unsigned char *boundary;
		unsigned char *param;
		unsigned char *equal;
		unsigned char *value;
		unsigned char *params[MAXMIMEARGS];
		unsigned char *values[MAXMIMEARGS];

		memset(params, '\0', sizeof params);
		memset(values, '\0', sizeof values);

		/* start at the first token past the first ";", if any */
		for (n = 1; n < ntokens; n++)
		{
			if (tokens[n] != NULL && tokens[n][0] == ';')
			{
				n++;
				break;
			}
		}

		for (c = 0; n < ntokens && c < MAXMIMEARGS; n++)
		{
			/* assume we have "parameter=value" */
			param = tokens[n];
			if (param == NULL)
				break;
			equal = tokens[n + 1];
			if (equal == NULL)
				break;
			value = tokens[n + 2];

			/* check for empty value */
			if (value == NULL || value[0] == ';')
			{
				n += 3;
				params[c] = param;
				c++;
				continue;
			}

			/* store the parameter/value pair */
			params[c] = param;
			values[c] = value;
			c++;

			/* move forward to the next ";" */
			n += 3;
			while (n < ntokens &&
			       tokens[n] != NULL &&
			       tokens[n][0] != ';')
				n++;
		}

		/* now see if we found a single boundary */
		boundary = NULL;
		for (n = 0; n < c; n++)
		{
			if (strcasecmp(params[n], "boundary") == 0)
			{
				if (boundary == NULL)
				{
					boundary = values[n];
					continue;
				}
				else
				{
					if (dolog)
					{
						syslog(LOG_ERR,
						       "%s Content-Type: header contains multiple boundaries",
						       nfc->ctx_jobid);
					}

					break;
				}
			}
		}

		if (boundary != NULL)
		{
			MIME new;

			if (nfc->ctx_curmime->mime_depth >= maxdepth)
			{
				if (nfc->ctx_knownbound)
				{
					nfc->ctx_knownbound = FALSE;
					nfc->ctx_rdepth++;
				}
				return 1;
			}

			new = nf_newmime(nfc);
			if (new == NULL)
			{
				int save_errno = errno;

				if (dolog)
				{
					syslog(LOG_ERR, "%s malloc(): %s",
					       nfc->ctx_jobid,
					       strerror(save_errno));
				}

				errno = save_errno;
				return -1;
			}

			nfc->ctx_curmime->mime_child = new;
			new->mime_parent = nfc->ctx_curmime;
			new->mime_depth = nfc->ctx_curmime->mime_depth + 1;
			if (string_cat(new->mime_boundary, boundary) == -1)
			{
				int save_errno = errno;

				if (dolog)
				{
					syslog(LOG_ERR, "%s malloc(): %s",
					       nfc->ctx_jobid,
					       strerror(save_errno));
				}

				errno = save_errno;
				return -1;
			}
			nfc->ctx_curmime = new;
			nfc->ctx_rdepth++;
			nfc->ctx_inhdrs = FALSE;
			nfc->ctx_knownbound = FALSE;
		}
		else
		{
			if (dolog)
			{
				syslog(LOG_ERR, "%s invalid Content-Type `%s'",
				       nfc->ctx_jobid, string_data(str));
			}

			return 0;
		}
	}

	return 1;
}

/*
**  NF_HDRCHECK -- see if this line from a message body is a header we want
**
**  Parameters:
**  	str -- candidate string
**  	hdr -- name of the header we want
**
**  Return value:
** 	TRUE iff this line appears to be the header we want.
*/

static bool
nf_hdrcheck(unsigned char *str, unsigned char *hdr)
{
	unsigned char *p;

	assert(str != NULL);
	assert(hdr != NULL);

	/* is at least the name the same? */
	if (strncasecmp(str, hdr, strlen(hdr)) != 0)
		return FALSE;

	/* what's left should just be 0+ spaces and a colon */
	for (p = str + strlen(hdr); *p != '\0'; p++)
	{
		/* found the end */
		if (*p == ':')
			return TRUE;

		/* any other character means this isn't it */
		if (!isascii(*p) || !isspace(*p))
			return FALSE;
	}

	/* never found the colon! */
	return FALSE;
}

/*
**  END private section
**  ==================================================================
**  BEGIN milter section
*/

/*
**  MLFI_CONNECT -- connection handler
**
**  Parameters:
**  	ctx -- milter context
**  	host -- hostname
**  	ip -- address, in in_addr form
**
**  Return value:
**  	An SMFIS_* constant.
*/

sfsistat
mlfi_connect(SMFICTX *ctx, char *host, _SOCK_ADDR *ip)
{
	CONTEXT nfc;

	/*
	**  Initialize a context
	*/

	nfc = nf_initcontext();
	if (nfc != NULL)
		nfc->ctx_bodybuf = string_new(0);
	if (nfc != NULL && nfc->ctx_bodybuf != NULL)
		nfc->ctx_recycle = vector_new(0);

	if (nfc == NULL || nfc->ctx_bodybuf == NULL || nfc->ctx_recycle == NULL)
	{
		if (dolog)
		{
			syslog(LOG_INFO,
			       "temp-failing (internal error)");
		}

		if (nfc->ctx_bodybuf != NULL)
			string_free(nfc->ctx_bodybuf);
		if (nfc != NULL)
			free(nfc);

		return SMFIS_TEMPFAIL;
	}

	nfc->ctx_milter = ctx;

	sm_strlcpy(nfc->ctx_hostname, host, sizeof nfc->ctx_hostname);
	if (ip != NULL)
	{
		memcpy(&nfc->ctx_addr, ip, sizeof nfc->ctx_addr);
	}
	else
	{
		struct sockaddr_in sin;

		memset(&sin, '\0', sizeof sin);
		sin.sin_family = AF_INET;
		sin.sin_addr.s_addr = htonl(INADDR_LOOPBACK);

		memcpy(&nfc->ctx_addr, &sin, sizeof nfc->ctx_addr);
	}

	/*
	**  Save it in this thread's private space.
	*/

	smfi_setpriv(ctx, nfc);

	return SMFIS_CONTINUE;
}

/*
**  MLFI_ENVFROM -- handler for MAIL FROM command (start of message)
**
**  Parameters:
**  	ctx -- milter context
**  	envfrom -- envelope from arguments
**
**  Return value:
**  	An SMFIS_* constant.
*/

sfsistat
mlfi_envfrom(SMFICTX *ctx, char **envfrom)
{
	unsigned char *p;
	CONTEXT nfc;

#ifndef DEBUG
	assert(ctx != NULL);
	assert(envfrom != NULL);
#endif /* !DEBUG */

	nfc = smfi_getpriv(ctx);
	assert(nfc != NULL);

	nf_msgcleanup(ctx);

	/*
	**  Store the sender information.
	*/

	sm_strlcpy(nfc->ctx_sender, envfrom[0], sizeof nfc->ctx_sender);
	for (p = nfc->ctx_sender; *p != '\0'; p++)
	{
		if (!isascii(*p) || !isprint(*p))
			*p = '?';
	}

	/*
	**  Continue processing.
	*/

	return SMFIS_CONTINUE;
}

/*
**  MLFI_HEADER -- handler for mail headers; stores the header in a vector
**                 of headers for later perusal, removing RFC822 comment
**                 substrings
**
**  Parameters:
**  	ctx -- milter context
**  	headerf -- header
**  	headerv -- value
**
**  Return value:
**  	An SMFIS_* constant.
*/

sfsistat
mlfi_header(SMFICTX *ctx, char *headerf, char *headerv)
{
	CONTEXT nfc;
	HEADER newhdr;

#ifndef DEBUG
	assert(ctx != NULL);
#endif /* !DEBUG */
	assert(headerf != NULL);
	assert(headerv != NULL);

	nfc = (CONTEXT) smfi_getpriv(ctx);
	assert(nfc != NULL);

	newhdr = malloc(sizeof(struct Header));
	if (newhdr == NULL)
	{
		if (dolog)
			syslog(LOG_ERR, "malloc(): %s", strerror(errno));

		nf_msgcleanup(ctx);
		return SMFIS_TEMPFAIL;
	}

	newhdr->hdr_hdr = strdup(headerf);
	newhdr->hdr_val = strdup(headerv);
	newhdr->hdr_next = NULL;

	if (newhdr->hdr_hdr == NULL || newhdr->hdr_val == NULL)
	{
		if (dolog)
			syslog(LOG_ERR, "malloc(): %s", strerror(errno));

		TRYFREE(newhdr->hdr_hdr);
		TRYFREE(newhdr->hdr_val);
		nf_msgcleanup(ctx);
		return SMFIS_TEMPFAIL;
	}

	if (nfc->ctx_hqhead == NULL)
		nfc->ctx_hqhead = newhdr;

	if (nfc->ctx_hqtail != NULL)
		nfc->ctx_hqtail->hdr_next = newhdr;

	nfc->ctx_hqtail = newhdr;

	return SMFIS_CONTINUE;
}

/*
**  MLFI_EOH -- handler called when there are no more headers;
**
**  Parameters:
**  	ctx -- milter context
**
**  Return value:
**  	An SMFIS_* constant.
*/

sfsistat
mlfi_eoh(SMFICTX *ctx)
{
	bool multipart;
	bool encapsulated;
	int status;
	int ntokens;
	unsigned char *p;
	CONTEXT nfc;
	HEADER ct;
	HEADER from;
	STRING str;
	MIME mime;
	unsigned char **tokens;

#ifndef DEBUG
	assert(ctx != NULL);
#endif /* !DEBUG */

	nfc = (CONTEXT) smfi_getpriv(ctx);
	assert(nfc != NULL);

	/*
	**  Determine the message ID for logging.
	*/

	nfc->ctx_jobid = smfi_getsymval(ctx, "i");
	if (nfc->ctx_jobid == NULL)
	{
		if (no_i_whine && dolog)
		{
			syslog(LOG_WARNING,
			       "WARNING: sendmail symbol 'i' not available");
			no_i_whine = FALSE;
		}
		nfc->ctx_jobid = MSGIDUNKNOWN;
	}

	/* extract From: header's value */
	from = nf_findheader(nfc, FROMHEADER, 0);
	if (from != NULL)
	{
		sm_strlcpy(nfc->ctx_from, from->hdr_val, sizeof nfc->ctx_from);

		/* make sure it's something loggable */
		for (p = nfc->ctx_from; *p != '\0'; p++)
		{
			if (!isascii(*p) || !isprint(*p))
				*p = '?';
		}
	}
	else
	{
		sm_strlcpy(nfc->ctx_from, FROMUNKNOWN, sizeof nfc->ctx_from);
	}

	/* figure out if this message even could be interesting */
	ct = nf_findheader(nfc, CTHEADER, 0);

	/*
	**  If there's no Content-Type: header, there's nothing to worry
	**  about so pass the message.
	*/

	if (ct == NULL)
		return SMFIS_ACCEPT;

	/* initialize the header buffer */
	if (nfc->ctx_hdrbuf == NULL)
		nfc->ctx_hdrbuf = string_new(0);
	if (nfc->ctx_hdrbuf == NULL)
	{
		if (dolog)
		{
			syslog(LOG_ERR, "%s malloc(): %s", nfc->ctx_jobid,
			       strerror(errno));
		}

		return SMFIS_TEMPFAIL;
	}
	string_blank(nfc->ctx_hdrbuf);

	/* tokenize and minimally validate the Content-Type: header */
	if (nfc->ctx_tokens == NULL)
	{
		nfc->ctx_tokens = tokens_new();
		if (nfc->ctx_tokens == NULL)
		{
			if (dolog)
			{
				syslog(LOG_ERR, "%s malloc(): %s",
				       nfc->ctx_jobid, strerror(errno));
			}

			return PARSE_FAIL_ACTION;
		}
	}

	str = tokens_getdata(nfc->ctx_tokens);
	string_blank(str);
	if (string_cat(str, ct->hdr_val) == -1)
	{
		if (dolog)
		{
			syslog(LOG_ERR, "%s malloc(): %s",
			       nfc->ctx_jobid, strerror(errno));
		}

		return PARSE_FAIL_ACTION;
	}
	for (p = string_data(str); *p != '\0'; p++)
	{
		if (*p == '\r' || *p == '\n')
			*p = ' ';
	}

	status = tokens_tokenize(nfc->ctx_tokens, MIME_SPECIALS,
	                         MIME_COMMENTS);

	if (status == -2)
	{
		if (dolog)
		{
			syslog(LOG_ERR, "%s malloc(): %s", nfc->ctx_jobid,
			       strerror(errno));
		}

		return PARSE_FAIL_ACTION;
	}
	else if (status < 1)
	{
		if (dolog)
		{
			syslog(LOG_ERR, "%s invalid Content-Type `%s'",
			       nfc->ctx_jobid, ct->hdr_val);
		}

		return PARSE_FAIL_ACTION;
	}

	ntokens = status;
	tokens = tokens_gettokens(nfc->ctx_tokens);

	/*
	**  If the master Content-Type isn't something that can contain
	**  nesting, then we don't need to worry about this message and
	**  can just accept it.
	*/

	multipart = (strcasecmp(tokens[0], "multipart") == 0);
	encapsulated = (strcasecmp(tokens[0], "message") == 0 &&
	                (tokens[1] != NULL &&
	                 strcasecmp(tokens[1], "/") == 0) &&
	                (tokens[2] != NULL &&
	                 strcasecmp(tokens[2], "rfc822") == 0));

	if (!multipart && !encapsulated)
		return SMFIS_ACCEPT;

	/* prepare for MIME! */
	mime = nf_newmime(nfc);
	if (mime == NULL)
	{
		if (dolog)
		{
			syslog(LOG_ERR, "%s malloc(): %s",
			       nfc->ctx_jobid, strerror(errno));
		}

		return SMFIS_TEMPFAIL;
	}
	nfc->ctx_mime = mime;
	nfc->ctx_curmime = mime;

	nfc->ctx_encap = FALSE;

	if (encapsulated)
	{
		nfc->ctx_inhdrs = TRUE;
	}
	else
	{
		int n;
		int c;
		unsigned char *boundary;
		unsigned char *param;
		unsigned char *value;
		unsigned char *equal;
		unsigned char *params[MAXMIMEARGS];
		unsigned char *values[MAXMIMEARGS];

		nfc->ctx_inhdrs = FALSE;

		memset(params, '\0', sizeof params);
		memset(values, '\0', sizeof values);

		/* start at the first token past the first ";", if any */
		for (n = 1; n < ntokens; n++)
		{
			if (tokens[n] != NULL && tokens[n][0] == ';')
			{
				n++;
				break;
			}
		}

		for (c = 0; n < ntokens && c < MAXMIMEARGS; n++)
		{
			/* assume we have "parameter=value" */
			param = tokens[n];
			if (param == NULL)
				break;
			equal = tokens[n + 1];
			if (equal == NULL)
				break;
			value = tokens[n + 2];

			/* check for empty value */
			if (value == NULL || value[0] == ';')
			{
				n += 3;
				params[c] = param;
				c++;
				continue;
			}

			/* store the parameter/value pair */
			params[c] = param;
			values[c] = value;
			c++;

			/* move forward to the next ";" */
			n += 3;
			while (n < ntokens &&
			       tokens[n] != NULL &&
			       tokens[n][0] != ';')
				n++;
		}

		/* now see if we found a single boundary */
		boundary = NULL;
		for (n = 0; n < c; n++)
		{
			if (strcasecmp(params[n], "boundary") == 0)
			{
				if (boundary == NULL)
				{
					boundary = values[n];
					continue;
				}
				else
				{
					if (dolog)
					{
						syslog(LOG_ERR,
						       "%s Content-Type: header contains multiple boundaries",
						       nfc->ctx_jobid);
					}

					break;
				}
			}
		}

		/* multipart; extract the boundary */
		if (boundary == NULL)
		{
			if (dolog)
			{
				syslog(LOG_ERR,
				       "%s invalid Content-Type `%s'",
				       nfc->ctx_jobid, ct->hdr_val);
			}

			return PARSE_FAIL_ACTION;
		}

		if (string_cat(mime->mime_boundary, boundary) == -1)
		{
			if (dolog)
			{
				syslog(LOG_ERR, "%s malloc(): %s",
				       nfc->ctx_jobid, strerror(errno));
			}

			return PARSE_FAIL_ACTION;
		}
	}

	return SMFIS_CONTINUE;
}

/*
**  MLFI_BODY -- handler called for body chunks
**
**  Parameters:
**  	ctx -- milter context
**  	bodyp -- body data
**  	bodylen -- body size
**
**  Return value:
**  	An SMFIS_* constant.
*/

sfsistat
mlfi_body(SMFICTX *ctx, unsigned char *bodyp, size_t bodylen)
{
	size_t n;
	size_t blen;
	CONTEXT nfc;
	MIME mime;
	unsigned char *str;
	unsigned char *boundary;

#ifndef DEBUG
	assert(ctx != NULL);
#endif /* !DEBUG */
	assert(bodyp != NULL);

	nfc = (CONTEXT) smfi_getpriv(ctx);
	assert(nfc != NULL);

	/* if we already know we've blown it, just short-circuit */
	if (nfc->ctx_rdepth > maxdepth)
		return SMFIS_CONTINUE;

	/* no MIME to process?  short-circuit */
	if (nfc->ctx_mime == NULL)
		return SMFIS_CONTINUE;


	for (n = 0; n < bodylen; n++)
	{
		if (bodyp[n] == '\n')
		{
			int len;

			/* we have a full line! */
			str = string_data(nfc->ctx_bodybuf);
			len = string_length(nfc->ctx_bodybuf);
			nfc->ctx_lineno++;

			/* if the last byte is a CR, truncate it */
			if (str[len - 1] == '\r')
				string_chop(nfc->ctx_bodybuf, len - 1);

			/* is this any boundary we know? */
			mime = nf_findboundary(nfc->ctx_mime, str);
			if (mime != NULL)
			{
				if (mime != nfc->ctx_curmime)
					nfc->ctx_knownbound = TRUE;

				boundary = string_data(mime->mime_boundary);
				blen = strlen(boundary);

				if (nfc->ctx_inrfc822)
				{
					nfc->ctx_rdepth--;
					nfc->ctx_inrfc822 = FALSE;
				}

				/* is this a termination? */
				if (str[blen + 2] == '-' &&
				    str[blen + 3] == '-')
				{			/* MBT_FINAL */

					if (nfc->ctx_curmime->mime_parent != NULL)
						nfc->ctx_curmime->mime_parent->mime_child =  nfc->ctx_curmime->mime_child;

					/* recycle the "current" node */
					if (vector_append(nfc->ctx_recycle,
					                  nfc->ctx_curmime) == -1)
					{
						if (dolog)
						{
							syslog(LOG_ERR,
							       "%s temp-failing (internal error)",
							       nfc->ctx_jobid);
						}
						return SMFIS_TEMPFAIL;
					}

					/* the parent is now current */
					nfc->ctx_curmime = nfc->ctx_curmime->mime_parent;

					/* nothing left to do? */
					if (nfc->ctx_curmime == NULL)
					{
						nfc->ctx_mime = NULL;
						return SMFIS_CONTINUE;
					}


					nfc->ctx_knownbound = FALSE;

					/* MTA only pops once */
					if (nfc->ctx_curmime != NULL)
						nfc->ctx_rdepth--;
				}

				/* nope, it's a new part */
				else
				{			/* MBT_INTERMED */
					/* enter header parse mode */
					nfc->ctx_inhdrs = TRUE;
				}
			}
			else if (nfc->ctx_inhdrs)
			{
				/* continuation */
				if (strlen(str) > 0 &&
				    isascii(str[0]) && nf_ishspace(str[0]))
				{
					if (string_cat(nfc->ctx_hdrbuf,
					               str) == -1)
					{
						if (dolog)
						{
							syslog(LOG_ERR,
							       "%s temp-failing (internal error)",
							       nfc->ctx_jobid);
						}
						return SMFIS_TEMPFAIL;
					}
				}

				/* new header */
				else if (strlen(str) > 0)
				{
					/*
					**  If this is not a header and we
					**  think we're inside an
					**  message/rfc822 part, make sure
					**  we're not in header mode.
					*/

					if (nfc->ctx_inrfc822 &&
					    !nf_isheader(str))
						nfc->ctx_inhdrs = FALSE;

					if (string_length(nfc->ctx_hdrbuf) > 0)
					{
						unsigned char *str2;

						str2 = string_data(nfc->ctx_hdrbuf);

						if (nf_hdrcheck(str2, CTHEADER))
						{
							int status;

							status = nf_parsebodyct(nfc, str2);

							if (status != 1)
								return PARSE_FAIL_ACTION;
						}

						nfc->ctx_knownbound = FALSE;

						string_blank(nfc->ctx_hdrbuf);
					}

					if (string_cat(nfc->ctx_hdrbuf,
					               str) == -1)
					{
						if (dolog)
						{
							syslog(LOG_ERR,
							       "%s temp-failing (internal error)",
							       nfc->ctx_jobid);
						}
						return SMFIS_TEMPFAIL;
					}
				}

				/* blank line */
				else
				{
					if (string_length(nfc->ctx_hdrbuf) > 0)
					{
						unsigned char *str2;

						str2 = string_data(nfc->ctx_hdrbuf);

						if (nf_hdrcheck(str2, CTHEADER))
						{
							int status;

							status = nf_parsebodyct(nfc, str2);

							if (status != 1)
								return PARSE_FAIL_ACTION;
						}

						string_blank(nfc->ctx_hdrbuf);
					}

					if (nfc->ctx_encap)
					{
						nfc->ctx_inrfc822 = TRUE;
						nfc->ctx_inhdrs = TRUE;
						nfc->ctx_encap = FALSE;
					}
					else
					{
						nfc->ctx_inhdrs = FALSE;
					}
				}
			}

			string_blank(nfc->ctx_bodybuf);
		}
		else
		{
			if (nfc->ctx_lastchar == '\r')
			{
				if (string_cat1(nfc->ctx_bodybuf, '\r') == -1)
					return -2;
			}
			if (string_cat1(nfc->ctx_bodybuf, bodyp[n]) == -1)
				return -2;
		}

		nfc->ctx_lastchar = bodyp[n];

		if (nfc->ctx_rdepth > maxdepth)
			break;

		if (string_length(nfc->ctx_bodybuf) > MAXMIMELINE)
		{
			if (dolog)
			{
				syslog(LOG_NOTICE,
				       "%s MIME line length %u exceeded, rejecting",
				       nfc->ctx_jobid, MAXMIMELINE);
			}

			return PARSE_FAIL_ACTION;
		}
	}

	return SMFIS_CONTINUE;
}

/*
**  MLFI_EOM -- handler called at the end of the message
**
**  Parameters:
**  	ctx -- milter context
**
**  Return value:
**  	An SMFIS_* constant.
*/

sfsistat
mlfi_eom(SMFICTX *ctx)
{
	CONTEXT nfc;
	char *hostname;
	char myname[MAXHOSTNAMELEN + 1];

#ifndef DEBUG
	assert(ctx != NULL);
#endif /* !DEBUG */

	nfc = (CONTEXT) smfi_getpriv(ctx);
	assert(nfc != NULL);

	/* get hostname; used in the X header and in new MIME boundaries */
	hostname = smfi_getsymval(ctx, "j");
	if (hostname == NULL)
	{
		int status;

		memset(myname, '\0', sizeof myname);
		status = gethostname(myname, sizeof myname - 1);
		if (status != 0)
			sm_strlcpy(myname, HOSTUNKNOWN, sizeof myname);

		hostname = myname;
	}

	/*
	**  Identify the filter, if requested.
	*/

	if (addxhdr)
	{
		char xfhdr[MAXHEADER + 1];

		memset(xfhdr, '\0', sizeof xfhdr);

		snprintf(xfhdr, sizeof xfhdr, "%s v%s %s %s", NESTING_PRODUCT,
		         NESTING_VERSION, hostname, nfc->ctx_jobid);

		if (smfi_insheader(ctx, 1, XHEADERNAME, xfhdr) != MI_SUCCESS)
		{
			if (dolog)
			{
				syslog(LOG_ERR, "%s smfi_insheader() failed",
				       nfc->ctx_jobid);
			}

			nf_msgcleanup(ctx);
			return SMFIS_TEMPFAIL;
		}
	}

	/* if the limit was exceeded, take the requested action */
	if (nfc->ctx_rdepth > maxdepth)
	{
#ifdef SMFIR_QUARANTINE
		if (blockaction == NF_ACTION_QUARANTINE)
		{
			if (smfi_quarantine(ctx, rejectmsg) != MI_SUCCESS)
			{
				if (dolog)
				{
					syslog(LOG_ERR,
					       "%s smfi_quarantine() failed",
					       nfc->ctx_jobid);
				}

				nf_msgcleanup(ctx);
				return SMFIS_TEMPFAIL;
			}
		}
#endif /* SMFIR_QUARANTINE */

		if ((blockaction == NF_ACTION_REJECT ||
		     blockaction == NF_ACTION_TEMPFAIL) &&
		    rejectmsg != NULL)
		{
			int status;

			if (blockaction == NF_ACTION_REJECT)
			{
				status = smfi_setreply(ctx, "550", "5.6.0",
				                       rejectmsg);
			}
			else
			{
				status = smfi_setreply(ctx, "450", "4.6.0",
				                       rejectmsg);
			}

			if (status != MI_SUCCESS)
			{
				if (dolog)
				{
					syslog(LOG_ERR,
					       "%s smfi_setreply() failed",
					       nfc->ctx_jobid);
				}

				nf_msgcleanup(ctx);
				return SMFIS_TEMPFAIL;
			}
		}

		if (dolog)
		{
			syslog(LOG_NOTICE,
			       "%s blocked: recursion >= %lu from %s (envelope sender `%s', From: `%s')",
			       nfc->ctx_jobid, maxdepth, nfc->ctx_hostname,
			       nfc->ctx_sender, nfc->ctx_from);
		}

		nf_msgcleanup(ctx);

		switch (blockaction)
		{
		  case NF_ACTION_REJECT:
			return SMFIS_REJECT;

		  case NF_ACTION_TEMPFAIL:
			return SMFIS_TEMPFAIL;

		  case NF_ACTION_DISCARD:
			return SMFIS_DISCARD;

		  case NF_ACTION_QUARANTINE:
			return SMFIS_ACCEPT;
		}
	}

	/* clean up */
	nf_msgcleanup(ctx);

	return SMFIS_ACCEPT;
}

/*
**  MLFI_CLOSE -- handler called on connection shutdown
**
**  Parameters:
**  	ctx -- milter context
**
**  Return value:
**  	An SMFIS_* constant.
*/

sfsistat
mlfi_close(SMFICTX *ctx)
{
	nf_msgcleanup(ctx);
	nf_conncleanup(ctx);
	return SMFIS_CONTINUE;
}

#ifndef DEBUG
/*
**  smfilter -- the milter module description
*/

struct smfiDesc smfilter =
{
	NESTING_PRODUCT,	/* filter name */
	SMFI_VERSION,		/* version code -- do not change */
#ifdef SMFIR_QUARANTINE
	(SMFIF_ADDHDRS|SMFIF_QUARANTINE), /* flags */
#else /* SMFIR_QUARANTINE */
	SMFIF_ADDHDRS,		/* flags */
#endif /* SMFIR_QUARANTINE */
	mlfi_connect,		/* connection info filter */
	NULL,			/* SMTP HELO command filter */
	mlfi_envfrom,		/* envelope sender filter */
	NULL,			/* envelope recipient filter */
	mlfi_header,		/* header filter */
	mlfi_eoh,		/* end of header */
	mlfi_body,		/* body block filter */
	mlfi_eom,		/* end of message */
	NULL,			/* message aborted */
	mlfi_close,		/* shutdown */
};
#endif /* !DEBUG */

#ifdef DEBUG
int
smfi_chgheader(void *ctx, char *hdr, int idx, char *val)
{
	printf("smfi_chgheader(<ctx>, `%s', `%d', `%s')\n", hdr, idx,
	       val == NULL ? "(null)" : val);
	return MI_SUCCESS;
}

int
smfi_replacebody(void *ctx, char *p, size_t len)
{
	printf("smfi_replacebody(<ctx>, `%.20s%s', `%d')\n", p,
	       strlen(p) > 20 ? "..." : "", len);
	return MI_SUCCESS;
}

int
smfi_addheader(void *ctx, char *hdr, char *val)
{
	printf("smfi_addheader(<ctx>, `%s', `%s')\n", hdr, val);
	return MI_SUCCESS;
}

int
smfi_insheader(void *ctx, int idx, char *hdr, char *val)
{
	printf("smfi_insheader(<ctx>, %d, `%s', `%s')\n", idx, hdr, val);
	return MI_SUCCESS;
}

int
smfi_setconn(char *file)
{
	printf("smfi_setconn(`%s')\n", file);
	return MI_SUCCESS;
}

void
smfi_setpriv(void *ctx, void *priv)
{
	fakepriv = priv;
}

void *
smfi_getpriv(void *ctx)
{
	return fakepriv;
}

int
smfi_setreply(void *ctx, char *sc, char *esc, char *reply)
{
	printf("smfi_setreply(<ctx>, `%s', `%s', `%s')\n",
	       sc, esc, reply);
	return MI_SUCCESS;
}

char *
smfi_getsymval(void *ctx, char *sym)
{
	char *ret;
	size_t l;
	CONTEXT nfc;

	l = strlen(sym) + 6 + 1;
	nfc = fakepriv;

	printf("smfi_getsymval(<ctx>, `%s')\n", sym);
	ret = malloc(l);
	snprintf(ret, l, "DEBUG-%s", sym);
	return ret;
}

int
smfi_progress(void *ctx)
{
	printf("smfi_progress(<ctx>)\n");

	return MI_SUCCESS;
}

/*
**  NF_DEBUG -- debugging code; simulates libmilter calls
**
**  Parameters:
**  	None.
**
**  Return value:
**  	None.
*/

int
nf_debug(void)
{
	bool done;
	int status;
	size_t len;
	time_t now;
	char *p;
	char *env[2];
	char data[513];
	char block[4096];
	char tmphdr[4096];

	time(&now);
	srandom(now);

	memset(data, '\0', sizeof data);
	memset(tmphdr, '\0', sizeof tmphdr);

	for (;;)
	{
		if (fgets(data, sizeof data, stdin) == NULL)
			return 1;

		for (p = data; *p != '\0'; p++)
		{
			if (*p == '\r' || *p == '\n')
			{
				*p = '\0';
				break;
			}
		}

		if (strcmp(data, ".") == 0)
			break;

		env[0] = &data[1];
		env[1] = NULL;

		if (data[0] == 'C')
		{
			struct hostent *h;
			struct sockaddr_in sin;

			h = gethostbyname(&data[1]);
			if (h == NULL)
			{
				printf("gethostbyname(\"%s\") failed\n",
				       &data[1]);
				return 1;
			}
			sin.sin_family = AF_INET;
			sin.sin_port = htons(time(NULL) % 65536);
			memcpy(&sin.sin_addr.s_addr, h->h_addr,
			       sizeof sin.sin_addr.s_addr);

			status = mlfi_connect(NULL, &data[1],
			                      (_SOCK_ADDR *) &sin);
			printf("mlfi_connect(NULL, `%s', `%s') returns %s\n",
			       &data[1], inet_ntoa(sin.sin_addr),
			       smfis_ret[status]);
		}
		else if (data[0] == 'F')
		{
			status = mlfi_envfrom(NULL, env);
			printf("mlfi_envfrom(NULL, `%s') returns %s\n", env[0],
	       		       smfis_ret[status]);
		}
/*
		else if (data[0] == 'T')
		{
			status = mlfi_envrcpt(NULL, env);
			printf("mlfi_envrcpt(NULL, `%s') returns %s\n", env[0],
	       		       smfis_ret[status]);
		}
*/
		else
		{
			return 1;
		}

		if (status != SMFIS_CONTINUE)
			return 0;
	}

	for (;;)
	{
		memset(data, '\0', sizeof data);
		if (fgets(data, sizeof data, stdin) == NULL)
			return 1;

		for (p = data; *p != '\0'; p++)
		{
			if (*p == '\r' || *p == '\n')
			{
				*p = '\0';
				break;
			}
		}

		if (strlen(data) > 0 && isascii(data[0]) && isspace(data[0]))
		{
			sm_strlcat(tmphdr, "\r\n", sizeof tmphdr);
			sm_strlcat(tmphdr, data, sizeof tmphdr);
			continue;
		}

		if (strlen(tmphdr) != 0)
		{
			char *q;

			p = strchr(tmphdr, ':');
			if (p == NULL)
			{
				printf("Invalid header line `%s'\n",
				       tmphdr);
				return 1;
			}

			*p = '\0';
			for (q = p + 1; isascii(*q) && isspace(*q); q++)
				continue;
			status = mlfi_header(NULL, tmphdr, q);
			printf("mlfi_header(NULL, `%s', `%s') returns %s\n",
			       tmphdr, q, smfis_ret[status]);
			if (status != SMFIS_CONTINUE)
				return 0;
			memset(tmphdr, '\0', sizeof tmphdr);
		}

		if (strlen(data) == 0)
			break;

		sm_strlcat(tmphdr, data, sizeof tmphdr);
	}

	status = mlfi_eoh(NULL);
	printf("mlfi_eoh(NULL) returns %s\n", smfis_ret[status]);
	if (status != SMFIS_CONTINUE)
		return 0;

	done = FALSE;
	while (!done)
	{
		len = fread(block, 1, sizeof block, stdin);
		status = mlfi_body(NULL, block, len);
		printf("mlfi_body(NULL, <body>, %d) returns %s\n",
		       len, smfis_ret[status]);
		if (status != SMFIS_CONTINUE)
			return 0;
		if (len < 4096)
			done = TRUE;
	}

	status = mlfi_eom(NULL);
	printf("mlfi_eom(NULL) returns %s\n", smfis_ret[status]);

	(void) mlfi_close(NULL);
	printf("mlfi_close()\n");

	return 0;
}

# if 0
static void
nf_printmime(MIME mime)
{
	printf("MIME node at %p:\n", mime);
	printf("\tDepth       %lu\n", mime->mime_depth);
	if (mime->mime_boundary != NULL)
	{
		printf("\tBoundary    `%s'\n",
		       string_data(mime->mime_boundary));
	}

	if (mime->mime_child != NULL)
	{
		printf("\n");
		nf_printmime(mime->mime_child);
	}
}
# endif /* 0 */
#endif /* DEBUG */

/*
**  USAGE -- print a usage message and return the appropriate exit status
**
**  Parameters:
**  	None.
**
**  Return value:
**  	EX_USAGE.
*/

static int
usage()
{
	fprintf(stderr, "%s: usage: %s -p socketfile [options]\n"
	                "-a action  \tpreventive action to take\n"
	                "-f         \tdon't fork-and-exit\n"
	                "-h         \tprepend identifying header\n"
	                "-l         \tlog activity to system log\n"
	                "-m maxdepth\tmaximum recursion depth\n"
	                "-P pidfile \tfile to which to write pid\n"
	                "-r text    \trejection message to use\n"
	                "-u userid  \tchange to specified userid\n"
	                "-V         \tprint version number and terminate\n",
	        progname, progname);
	return EX_USAGE;
}

/*
**  MAIN -- program mainline
**
**  Process command line arguments and call the milter mainline.
*/

int
main(int argc, char **argv)
{
	bool gotp = FALSE;
	bool dofork = TRUE;
	int c;
#ifndef DEBUG
	int n;
	int status;
	char *end;
#endif /* ! DEBUG */
	const char *args = CMDLINEOPTS;
	char *become = NULL;
	char *p;
	char *pidfile = NULL;
	char *action = DEFBLOCKACTION;
#ifndef DEBUG
	char argstr[MAXARGV];
#endif /* ! DEBUG */

	/* initialize */
	addxhdr = FALSE;
	dolog = FALSE;
	no_i_whine = TRUE;
	rejectmsg = DEFREJECTMSG;
	maxdepth = DEFMAXMIMEDEPTH;

	progname = (p = strrchr(argv[0], '/')) == NULL ? argv[0] : p + 1;

	/* process command line options */
	while ((c = getopt(argc, argv, args)) != -1)
	{
		switch (c)
		{
		  case 'a':
			if (optarg == NULL || *optarg == '\0')
				return usage();
			action = optarg;
			break;

		  case 'f':
			dofork = FALSE;
			break;

		  case 'h':
			addxhdr = TRUE;
			break;

		  case 'l':
#ifndef DEBUG
			dolog = TRUE;
#endif /* !DEBUG */
			break;

		  case 'm':
			if (optarg == NULL || *optarg == '\0')
				return usage();
			maxdepth = strtoul(optarg, &p, 10);
			if (*p != '\0')
			{
				fprintf(stderr, "%s: invalid depth `%s'\n",
				        progname, optarg);
				return EX_USAGE;
			}
			break;

		  case 'p':
			if (optarg == NULL || *optarg == '\0')
				return usage();
			if (smfi_setconn(optarg) != MI_SUCCESS)
			{
				fprintf(stderr, "%s: smfi_setconn() failed\n",
				        progname);
				return EX_SOFTWARE;
			}
			gotp = TRUE;
			break;

		  case 'P':
			if (optarg == NULL || *optarg == '\0')
				return usage();
			pidfile = optarg;
			break;

		  case 'r':
			if (optarg == NULL || *optarg == '\0')
				return usage();
			rejectmsg = optarg;
			break;

		  case 'u':
			if (optarg == NULL || *optarg == '\0')
				return usage();
			become = optarg;
			break;

		  case 'V':
			printf("%s: %s v%s\n", progname, NESTING_PRODUCT,
			       NESTING_VERSION);
			return EX_OK;

		  default:
			return usage();
		}
	}

	if (!gotp)
		return usage();

	if (strcasecmp(action, "reject") == 0)
	{
		blockaction = NF_ACTION_REJECT;
	}
	else if (strcasecmp(action, "tempfail") == 0)
	{
		blockaction = NF_ACTION_TEMPFAIL;
	}
	else if (strcasecmp(action, "discard") == 0)
	{
		blockaction = NF_ACTION_DISCARD;
	}
#ifdef SMFIR_QUARANTINE
	else if (strcasecmp(action, "quarantine") == 0)
	{
		blockaction = NF_ACTION_QUARANTINE;
	}
#endif /* SMFIR_QUARANTINE */
	else
	{
		fprintf(stderr, "%s: invalid block action `%s'\n",
		        progname, action);
		return EX_USAGE;
	}

	nf_setmaxfd();

	/* change user if appropriate */
	if (become != NULL)
	{
		struct passwd *pw;
		gid_t groups[1];

		pw = getpwnam(become);
		if (pw == NULL)
		{
			uid_t uid;

			uid = atoi(become);
#ifdef UID_MAX
			if (uid != 0 && uid != UID_MAX)
#else /* UID_MAX */
			if (uid != 0)
#endif /* UID_MAX */
				pw = getpwuid(uid);
			if (pw == NULL)
			{
				fprintf(stderr, "%s: no such user `%s'\n",
				        progname, become);
				return EX_DATAERR;
			}
		}

		(void) endpwent();

		groups[0] = pw->pw_gid;

		if (setgroups(1, groups) != 0)
		{
			fprintf(stderr, "%s: setgroups(): %s\n", progname,
			        strerror(errno));
			return EX_NOPERM;
		}
		if (setgid(pw->pw_gid) != 0)
		{
			fprintf(stderr, "%s: setgid(): %s\n", progname,
			        strerror(errno));
			return EX_NOPERM;
		}
		if (setuid(pw->pw_uid) != 0)
		{
			fprintf(stderr, "%s: setuid(): %s\n", progname,
			        strerror(errno));
			return EX_NOPERM;
		}
	}

	/* activate logging */
	if (dolog)
	{
#ifdef LOG_MAIL
		openlog(progname, LOG_PID, LOG_MAIL);
#else /* LOG_MAIL */
		openlog(progname, LOG_PID);
#endif /* LOG_MAIL */
	}

#ifndef DEBUG
	/* register with the milter interface */
	if (smfi_register(smfilter) == MI_FAILURE)
	{
		if (dolog)
			syslog(LOG_ERR, "smfi_register() failed");

		fprintf(stderr, "%s: smfi_register() failed\n", progname);

		return EX_UNAVAILABLE;
	}

	/* try to establish the milter socket */
	if (smfi_opensocket(FALSE) == MI_FAILURE)
	{
		if (dolog)
			syslog(LOG_ERR, "smfi_opensocket() failed");

		fprintf(stderr, "%s: smfi_opensocket() failed\n", progname);

		return EX_UNAVAILABLE;
	}
#endif /* !DEBUG */

	if (dofork)
	{
		pid_t pid;

		pid = fork();
		switch(pid)
		{
		  case -1:
			fprintf(stderr, "%s: fork(): %s\n", progname,
			        strerror(errno));
			if (dolog)
				syslog(LOG_ERR, "fork(): %s", strerror(errno));
			return EX_OSERR;

		  case 0:
			nf_stdio();
			break;

		  default:
			return EX_OK;
		}
	}

	/* write out the pid */
	if (pidfile != NULL)
	{
		FILE *f;

		f = fopen(pidfile, "w");
		if (f != NULL)
		{
			fprintf(f, "%ld\n", (long) getpid());
			(void) fclose(f);
		}
		else
		{
			if (dolog)
			{
				syslog(LOG_ERR, "can't write pid to %s: %s",
				       pidfile, strerror(errno));
			}
		}
	}

#ifdef DEBUG
	return nf_debug();
#else /* DEBUG */
	memset(argstr, '\0', sizeof argstr);
	end = &argstr[sizeof argstr - 1];
	n = sizeof argstr;
	for (c = 1, p = argstr; c < argc && p < end; c++)
	{
		if (strchr(argv[c], ' ') != NULL)
		{
			status = snprintf(p, n, "%s \"%s\"",
			                  c == 1 ? "args:" : "",
			                  argv[c]);
		}
		else
		{
			status = snprintf(p, n, "%s %s",
			                  c == 1 ? "args:" : "",
			                  argv[c]);
		}

		if (status <= 0)
			break;

		p += status;
		n -= status;
	}

	if (dolog)
	{
		syslog(LOG_INFO, "%s v%s starting (%s)", NESTING_PRODUCT,
		       NESTING_VERSION, argstr);
	}

	/* call the milter mainline */
	errno = 0;
	status = smfi_main();

	if (dolog)
	{
		syslog(LOG_INFO,
		       "%s v%s terminating with status %d, errno = %d",
		       NESTING_PRODUCT, NESTING_VERSION, status, errno);
	}

	return status;
#endif /* DEBUG */
}
