/*
 * Copyright (c) 2000-2002 Sendmail, Inc. and its suppliers.
 *	All rights reserved.
 *
 * By using this file, you agree to the terms and conditions set
 * forth in the LICENSE file which can be found at the top level of
 * the sendmail distribution.
 *
 *	$Id: str.h,v 1.29 2002/07/10 01:48:12 ca Exp $
 */

#ifndef SM_STR_H
#define SM_STR_H	1

#include "sm/generic.h"
#include "sm/types.h"
#include "sm/error.h"
#include "sm/rpool.h"
#include "sm/limits.h"

typedef struct sm_str_S sm_str_T, *sm_str_P;

/*
**  sm_str_T -- Stores buf data and length information.
**
**	Members:
**		sm_str_base -- uchar * to data (doesn't need to end with '\0').
**		sm_str_size -- Total bytes allocated.
**		sm_str_len -- Total number of characters in buf.
**		sm_str_max -- Maximum number of characters in buf.
**		sm_str_rpool -- rpool to allocate from.
**
**	Invariants:
**		sm_str_len <= sm_str_size <= sm_str_max
**
**	Notices:
**		- the structure could be extended to a general buffer,
**		like iobuf (VSTRING) to put/get characters out of it.
**		This would make loops through the data more abstract,
**		e.g., instead of
**		for (i = 0; i < len(str); i++)
**			do something with elem(str, i)
**		it could be
**		for (ch = first(str); !EOS(str); ch = get(str))
**			do something with ch
**		however, that requires read/write pointers and hence
**		there can't be independent operations on the string.
**
**		- we could use reference counting to avoid free()ing
**		a buf twice if it is used as:
**		d = sm_str_cat(s1, s2)
**		because now d and s1 are identical (unless the
**		operation fails).
**
**		- a flag field could note whether this is statically
**		allocated and can't be changed,
**		whether the current size is fixed (could be achieve
**		wiht sm_str_max)
**
**		- reference counting could also be used for the data
**		part, which may allow sharing (use copy-on-write).
**
**		- this structure is only "public" due to macros accessing
**		components, e.g., sm_str_len(), otherwise the
**		data structure could be in buf.c or an internal
**		include file.
**
**		- sm_str_max should be always set and it should be
**		less than INT_MAX. Then we can get rid of some
**		overflow tests (sm_str_size wrap-around). Moreover,
**		this would be better for some other code which
**		assumes int is good enough as index.
**		XXX should we enforce this?
**		SM_ASSERT(sm_str_max != 0 && sm_str_max < INT_MAX)
*/

struct sm_str_S
{
	sm_magic_T	 sm_magic;
	uchar		*sm_str_base;
	size_t		 sm_str_len;
	size_t		 sm_str_size;
	size_t		 sm_str_max;
#if SM_STR_READ
	size_t		 sm_str_rd;	/* read index */
#endif
	sm_rpool_P	 sm_str_rpool;
};

extern sm_str_P		 sm_str_new(sm_rpool_P _rpool, size_t _len,
					size_t _max);
extern sm_str_P		 sm_str_crt(sm_rpool_P rpool, uchar *str, size_t len,
					size_t maxlen);

/* assign necessary elements to an existing str_buf */
#define sm_str_assign(str, rpool, s, len, maxlen) \
	do	\
	{	\
		(str).sm_str_base = (s); \
		(str).sm_str_size = (len); \
		(str).sm_str_len = (len); \
		(str).sm_str_max = (maxlen); \
		(str).sm_str_rpool = (rpool); \
	} while (0)

/* private */
extern size_t		sm_str_resize_data(sm_str_P _str, size_t _len);

#if SM_STR_CHECK
extern size_t		 sm_str_getlen(sm_str_P _str);
extern size_t		 sm_str_getsize(sm_str_P _str);
extern int		 sm_str_setmax(sm_str_P _str, size_t _max);
extern size_t		 sm_str_getmax(sm_str_P _str);
extern size_t		 sm_str_space(sm_str_P _str, size_t _new_len);
#else /* SM_STR_CHECK */
# define sm_str_getlen(str)	((str)->sm_str_len)
# define sm_str_getsize(str)	((str)->sm_str_size)
# define sm_str_setmax(str, m)	(str)->sm_str_max = (m)
# define sm_str_getmax(str)	(str)->sm_str_max
# define sm_str_space(str, new_len) 			\
	(((new_len) <= (str)->sm_str_size) ? (str)->sm_str_size	\
		: sm_str_resize_data((str), (new_len)))
#endif /* SM_STR_CHECK */

#define sm_str_elem(str, i)	((str)->sm_str_base[i])
#define sm_str_data(str)	((str)->sm_str_base)

extern void		 sm_str_free(sm_str_P _str);
#define SM_STR_FREE(ptr) do			\
	{					\
		if ((ptr) != NULL)		\
		{				\
			sm_str_free(ptr);	\
			(ptr) = NULL;		\
		}				\
	} while (0)

#if SM_STR_CHECK
extern void		 sm_str_clr(sm_str_P _str);
#else /* SM_STR_CHECK */
# define sm_str_clr(str)	(str)->sm_str_len = 0
#endif /* SM_STR_CHECK */

extern sm_ret_T		 sm_str_shorten(sm_str_P _str, int _l);

/* last character if string is non-empty otherwise -1 (XXX is -1 ok?) */
# define SM_STR_LAST(str) ((str)->sm_str_len > 0 ? ((str)->sm_str_base[(str)->sm_str_len - 1]) : (-1))

extern uchar		*sm_str_getdata(sm_str_P _str);
extern uchar		*sm_str_copydata(sm_rpool_P _rpool, sm_str_P _str);

extern sm_str_P		 sm_str_dup(sm_rpool_P _rpool, const sm_str_P _src);
extern sm_ret_T		 sm_str_cpy(sm_str_P _dst, const sm_str_P _src);
extern sm_ret_T		 sm_str_cat(sm_str_P _dst, const sm_str_P _src);
extern sm_ret_T		 sm_str_catv(sm_str_P _dst, int n, ...);

extern bool		 sm_str_eq(sm_str_P _s1, const sm_str_P _s2);
extern int		 sm_str_cmp(sm_str_P _s1, const sm_str_P _s2);

#if HAVE_SNPRINTF
/* XXX: implement your own versions? */
extern size_t		 sm_str_printf(sm_str_P _str, const char *_format, ...);
extern size_t		 sm_str_vprintf(sm_str_P _str,
					const char *_format, va_list _va);
#endif /* HAVE_SNPRINTF */

extern sm_ret_T		 sm_str_scopyn(sm_str_P _str, const char *_src,
					size_t _len);
extern sm_ret_T		 sm_str_scopy(sm_str_P _str, const char *_src);
extern sm_str_P		 sm_str_scpy(sm_rpool_P _rpool, const char *_str,
					size_t _max);
extern sm_str_P		 sm_str_scpy0(sm_rpool_P _rpool, const char *_src,
					size_t _max);
extern sm_str_P		 sm_str_scpyn(sm_rpool_P _rpool,
					const char *_src, size_t _n,
					size_t _max);
extern sm_ret_T		 sm_str_scat(sm_str_P _str, const char *_append);
extern sm_ret_T		 sm_str_scatn(sm_str_P _str,
					const char *_append, size_t _len);
extern sm_ret_T		 sm_str_put(sm_str_P _str, int _c);
extern sm_ret_T		 sm_str_term(sm_str_P _str);
extern sm_ret_T		 sm_str_putuint32(sm_str_P _str, uint32_t _n);

#if 0
extern sm_str_P		 sm_str_readline_stream(sm_str_P _str, FILE *_f);
extern size_t		 sm_str_slurp_stream(sm_str_P _str, FILE *_f);
extern size_t		 sm_str_slurpn_stream(sm_str_P _str,
					FILE *_f, size_t _n);
extern Vector		 sm_str_split_quoted(sm_rpool_P _rpool,
					const char * _str);
#endif /* 0 */

#endif /* SM_STR_H */
