/*****************************************************************************
 * playlist.cpp : Custom widgets for the playlist
 ****************************************************************************
 * Copyright © 2007-2008 the VideoLAN team
 * $Id: fcc6d2df72301f0556486923c694f9c0d427cbc6 $
 *
 * Authors: Clément Stenac <zorglub@videolan.org>
 *          Jean-Baptiste Kempf <jb@videolan.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * ( at your option ) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston MA 02110-1301, USA.
 *****************************************************************************/

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "components/playlist/panels.hpp"
#include "components/playlist/selector.hpp"
#include "components/playlist/playlist.hpp"
#include "input_manager.hpp" /* art signal */

#include <QSettings>
#include <QLabel>
#include <QSpacerItem>
#include <QCursor>
#include <QPushButton>
#include <QVBoxLayout>

/**********************************************************************
 * Playlist Widget. The embedded playlist
 **********************************************************************/

PlaylistWidget::PlaylistWidget( intf_thread_t *_p_i,
                                QSettings *settings,
                                QWidget *_parent )
               : p_intf ( _p_i ), parent( _parent )
{
    setContentsMargins( 3, 3, 3, 3 );

    /* Left Part and design */
    QSplitter *leftW = new QSplitter( Qt::Vertical, this );

    /* Source Selector */
    selector = new PLSelector( this, p_intf, THEPL );
    leftW->addWidget( selector );

    /* Create a Container for the Art Label
       in order to have a beautiful resizing for the selector above it */
    QWidget *artContainer = new QWidget;
    QHBoxLayout *artContLay = new QHBoxLayout( artContainer );
    artContLay->setMargin( 0 );
    artContLay->setSpacing( 0 );
    artContainer->setMaximumHeight( 128 );

    /* Art label */
    art = new ArtLabel;
    art->setMinimumHeight( 128 );
    art->setMinimumWidth( 128 );
    art->setMaximumHeight( 128 );
    art->setMaximumWidth( 128 );
    art->setScaledContents( true );
    art->setPixmap( QPixmap( ":/noart.png" ) );
    art->setToolTip( qtr( "Double click to get media information" ) );

    artContLay->addWidget( art, 1 );

    leftW->addWidget( artContainer );

    /* Initialisation of the playlist */
    playlist_item_t *p_root =
                  playlist_GetPreferredNode( THEPL, THEPL->p_local_category );

    rightPanel = new StandardPLPanel( this, p_intf, THEPL, p_root );

    /* Connect the activation of the selector to a redefining of the PL */
    CONNECT( selector, activated( int ), rightPanel, setRoot( int ) );

    /* Connect the activated() to the rootChanged() signal
       This will be used by StandardPLPanel to setCurrentRootId, that will
       change the label of the addButton  */
    connect( selector, SIGNAL( activated( int ) ),
             this, SIGNAL( rootChanged( int ) ) );

    /* Forward removal requests from the selector to the main panel */
    CONNECT( qobject_cast<PLSelector *>( selector )->model,
             shouldRemove( int ),
             qobject_cast<StandardPLPanel *>( rightPanel ), removeItem( int ) );

    emit rootChanged( p_root->i_id );

    /* art */
    CONNECT( THEMIM->getIM(), artChanged( QString ) , this, setArt( QString ) );

    /* Add the two sides of the QSplitter */
    addWidget( leftW );
    addWidget( rightPanel );

    QList<int> sizeList;
    sizeList << 180 << 420 ;
    setSizes( sizeList );
    //setSizePolicy( QSizePolicy::Preferred, QSizePolicy::Expanding );
    setStretchFactor( 0, 0 );
    setStretchFactor( 1, 3 );
    leftW->setMaximumWidth( 250 );
    setCollapsible( 1, false );

    /* In case we want to keep the splitter informations */
    // components shall never write there setting to a fixed location, may infer
    // with other uses of the same component...
    // settings->beginGroup( "playlist" );
    restoreState( settings->value("splitterSizes").toByteArray());
}

void PlaylistWidget::setArt( QString url )
{
    if( prevArt != url )
    {
        art->setPixmap( QPixmap( url.isEmpty() ? ":/noart.png" : url ) );
        prevArt = url;
    }
}

QSize PlaylistWidget::sizeHint() const
{
   return QSize( 600 , 300 );
}

PlaylistWidget::~PlaylistWidget()
{}

void PlaylistWidget::savingSettings( QSettings *settings )
{
    settings->setValue( "splitterSizes", saveState() );
}

