%{

#include <math.h>
#include <stdlib.h>
#include <string.h>
#include "goom_script_scanner.h"
#include "goom_script_scanner.tab.h"
void yyerror(char *);
void yyparse();

/* #define TRACE_SCRIPT */
    
#define INSTR_SETI_PARAM_INTEGER   1
#define INSTR_SETI_VAR_INTEGER     2
#define INSTR_SETI_VAR_VAR         3
#define INSTR_SETI_VAR_PARAM       4
#define INSTR_SETI_PARAM_VAR       5
/* #define INSTR_JUMP              6 */
#define INSTR_NOP                  7
#define INSTR_SETF_PARAM_FLOAT     8
#define INSTR_SETF_VAR_FLOAT       9
#define INSTR_SETF_VAR_VAR         10
#define INSTR_SETF_VAR_PARAM       11
#define INSTR_SETF_PARAM_VAR       12
#define INSTR_ISLOWERF_VAR_VAR     13
#define INSTR_ISLOWERF_VAR_FLOAT   14
#define INSTR_ISLOWERI_VAR_VAR     15
#define INSTR_ISLOWERI_VAR_INTEGER 16
#define INSTR_ADDI_VAR_INTEGER     17
#define INSTR_ADDF_VAR_FLOAT       18
#define INSTR_ADDI_VAR_VAR         19
#define INSTR_ADDF_VAR_VAR         20
#define INSTR_MULI_VAR_INTEGER     21
#define INSTR_MULF_VAR_FLOAT       22
#define INSTR_MULI_VAR_VAR         23
#define INSTR_MULF_VAR_VAR         24
#define INSTR_ISEQUALF_VAR_VAR     25
#define INSTR_ISEQUALF_VAR_FLOAT   26
#define INSTR_ISEQUALI_VAR_VAR     27
#define INSTR_ISEQUALI_VAR_INTEGER 28
/* #define INSTR_JZERO             29 */

#define VALIDATE_OK "ok"
#define VALIDATE_ERROR "error while validating"
#define VALIDATE_TODO "todo"
#define VALIDATE_SYNTHAX_ERROR "synthax error"
#define VALIDATE_NO_SUCH_INT "no such integer variable"
#define VALIDATE_NO_SUCH_PARAM "no such param"
#define VALIDATE_NO_SUCH_VAR "no such variable"
#define VALIDATE_NO_SUCH_DEST_VAR "no such destination variable"
#define VALIDATE_NO_SUCH_SRC_VAR "no such src variable"

    PluginParam *goom_script_get_param(PluginInfo *pluginInfo, const char *name);

    /* ------------- SCRIPT_EXEC_ENV ------------ */
    
    /* Instruction *instr_init(GoomScriptScanner *parent, const char *name, int id, int nb_param); */
    static void instr_free(Instruction *_this);
    /* void instr_add_param(Instruction *_this, char *param, int type); */
    static const char *instr_validate(Instruction *_this);
    /* void instr_display(Instruction *_this); */

    /* ----------- INSTRUCTION_FLOW ------------- */

    static InstructionFlow *iflow_new();
    static void iflow_add_instr(InstructionFlow *_this, Instruction *instr);
    static void iflow_clean(InstructionFlow *_this);
    static void iflow_execute(InstructionFlow *_this, ScriptExecEnv *exec_env);

    /* ----------- IMPLEMENTATIONS --------------- */

    void iflow_clean(InstructionFlow *_this) {
        /* TODO: clean chaque instruction du flot */
        _this->number = 0;
        goom_hash_free(_this->labels);
        _this->labels = goom_hash_new();
    }
    
    InstructionFlow *iflow_new() {
        
        InstructionFlow *_this = (InstructionFlow*)malloc(sizeof(InstructionFlow));
        _this->number = 0;
        _this->tabsize = 6;
        _this->instr = (Instruction**)malloc(_this->tabsize * sizeof(Instruction*));
        _this->labels = goom_hash_new();

        return _this;
    }

    void iflow_add_instr(InstructionFlow *_this, Instruction *instr) {

        if (_this->number == _this->tabsize) {
            _this->tabsize *= 2;
            _this->instr = (Instruction**)realloc(_this->instr, _this->tabsize * sizeof(Instruction*));
        }
        _this->instr[_this->number] = instr;
        instr->address = _this->number;
        _this->number++;
    }

    /**
     * instr_add_param
     */
    void instr_add_param(Instruction *instr, char *param, int type) {

        int len;
        if (instr==NULL)
            return;
        if (instr->cur_param==0)
            return;
        --instr->cur_param;
        len = strlen(param);
        instr->params[instr->cur_param] = (char*)malloc(len+1);
        strcpy(instr->params[instr->cur_param], param);
        instr->types[instr->cur_param] = type;
        if (instr->cur_param == 0) {

            const char *result = instr_validate(instr);
            if (result != VALIDATE_OK) {
                printf("ERROR LINE %d: ", instr->parent->num_lines + 1);
                instr_display(instr);
                printf("... %s\n", result);
                instr->parent->compilationOK = 0;
            }
            
            if (instr->id != INSTR_NOP)
                iflow_add_instr(instr->parent->current_flow, instr);
            else
                instr_free(instr);
        }
    }

    /**
     * instr_init
     */
    Instruction *instr_init(GoomScriptScanner *parent, const char *name, int id, int nb_param) {

        Instruction *instr = (Instruction*)malloc(sizeof(Instruction));
        instr->params = (char**)malloc(nb_param*sizeof(char*));
        instr->types = (int*)malloc(nb_param*sizeof(int));
        instr->cur_param = instr->nb_param = nb_param;
        instr->parent = parent;
        instr->id = id;
        instr->name = name;
        instr->jump_label = NULL;
        return instr;
    }

    void instr_free(Instruction *_this) {

        int i;
        free(_this->types);
        for (i=_this->cur_param; i<_this->nb_param; ++i)
            free(_this->params[i]);
        free(_this->params);
        free(_this);
    }
    
    void instr_display(Instruction *_this) {
        int i=_this->nb_param-1;
        printf("%s", _this->name);
        while(i>=_this->cur_param) {
            printf(" %s", _this->params[i]);
            --i;
        }
    }
   
    /** VALIDATE **/
    
    static const char *validate_v_v(Instruction *_this) {

        _this->data.v_v.var_dest = goom_hash_get(_this->parent->vars, _this->params[1]);
        _this->data.v_v.var_src = goom_hash_get(_this->parent->vars, _this->params[0]);

        if (_this->data.v_v.var_dest == NULL) {
            return VALIDATE_NO_SUCH_DEST_VAR;
        }
        if (_this->data.v_v.var_src == NULL) {
            return VALIDATE_NO_SUCH_SRC_VAR;
        }
        return VALIDATE_OK;
    }

    static const char *validate_v_i(Instruction *_this) {

        _this->data.v_i.var = goom_hash_get(_this->parent->vars, _this->params[1]);
        _this->data.v_i.value = atoi(_this->params[0]);

        if (_this->data.v_i.var == NULL) {
            return VALIDATE_NO_SUCH_INT;
        }
        return VALIDATE_OK;
    }

    static const char *validate_v_p(Instruction *_this) {
        _this->data.v_p.var = goom_hash_get(_this->parent->vars, _this->params[1]);
        _this->data.p_i.param = goom_script_get_param(_this->parent->pluginInfo, _this->params[0]);
        if (_this->data.p_i.param == NULL)
            return VALIDATE_NO_SUCH_PARAM;
        if (_this->data.p_i.param->type != PARAM_INTVAL)
            return VALIDATE_ERROR;
        if (_this->data.v_i.var == NULL) {
            return VALIDATE_NO_SUCH_INT;
        }
        return VALIDATE_OK;
    }

    static const char *validate_p_v(Instruction *_this) {
        _this->data.v_p.var = goom_hash_get(_this->parent->vars, _this->params[0]);
        _this->data.v_p.param = goom_script_get_param(_this->parent->pluginInfo, _this->params[1]);
        if (_this->data.v_p.param == NULL)
            return VALIDATE_NO_SUCH_PARAM;
        if (_this->data.v_p.param->type != PARAM_INTVAL)
            return VALIDATE_ERROR;
        if (_this->data.v_p.var == NULL) {
            return VALIDATE_NO_SUCH_INT;
        }
        return VALIDATE_OK;
    }

    static const char *validate_p_i(Instruction *_this) {
        _this->data.p_i.param = goom_script_get_param(_this->parent->pluginInfo, _this->params[1]);
        _this->data.p_i.value = atoi(_this->params[0]);
        if (_this->data.p_i.param == NULL)
            return VALIDATE_NO_SUCH_PARAM;
        if (_this->data.p_i.param->type == PARAM_INTVAL)
            return VALIDATE_OK;
        return VALIDATE_ERROR;
    }

    /***/
    
    static const char *validate_v_f(Instruction *_this) {

        _this->data.v_f.var = goom_hash_get(_this->parent->vars, _this->params[1]);
        _this->data.v_f.value = atof(_this->params[0]);

        if (_this->data.v_f.var == NULL) {
            return VALIDATE_NO_SUCH_VAR;
        }
        return VALIDATE_OK;
    }

    static const char *validate_v_pf(Instruction *_this) {

        _this->data.v_p.var = goom_hash_get(_this->parent->vars, _this->params[1]);
        _this->data.v_p.param = goom_script_get_param(_this->parent->pluginInfo, _this->params[0]);
        if (_this->data.v_p.param == NULL)
            return VALIDATE_NO_SUCH_VAR;
        if (_this->data.v_p.param->type != PARAM_FLOATVAL)
            return VALIDATE_ERROR;
        if (_this->data.v_p.var == NULL) {
            return VALIDATE_NO_SUCH_VAR;
        }
        return VALIDATE_OK;
    }

    static const char *validate_pf_v(Instruction *_this) {
        
        _this->data.v_p.var = goom_hash_get(_this->parent->vars, _this->params[0]);
        _this->data.v_p.param = goom_script_get_param(_this->parent->pluginInfo, _this->params[1]);
        if (_this->data.v_p.param == NULL)
            return VALIDATE_NO_SUCH_VAR;
        if (_this->data.v_p.param->type != PARAM_FLOATVAL)
            return VALIDATE_ERROR;
        if (_this->data.v_p.var == NULL) {
            return VALIDATE_NO_SUCH_VAR;
        }
        return VALIDATE_OK;
    }

    static const char *validate_p_f(Instruction *_this) {
        _this->data.p_f.param = goom_script_get_param(_this->parent->pluginInfo, _this->params[1]);
        _this->data.p_f.value = atof(_this->params[0]);
        if (_this->data.p_f.param == NULL)
            return VALIDATE_NO_SUCH_VAR;
        if (_this->data.p_f.param->type == PARAM_FLOATVAL)
            return VALIDATE_OK;
        return VALIDATE_ERROR;
    }

    static const char *validate_i(Instruction *_this, int v_i_id, int v_v_id) {

        if ((_this->types[0] == TYPE_INTEGER) && (_this->types[1] == TYPE_VAR)) {
            _this->id = v_i_id;
            return validate_v_i(_this);
        }
        else if ((_this->types[1] == TYPE_VAR) && (_this->types[0] == TYPE_VAR)) {
            _this->id = v_v_id;
            return validate_v_v(_this);
        }
        return VALIDATE_ERROR;
    }
    
    static const char *validate_f(Instruction *_this, int v_f_id, int v_v_id) {

        if ((_this->types[0] == TYPE_FLOAT) && (_this->types[1] == TYPE_VAR)) {
            _this->id = v_f_id;
            return validate_v_f(_this);
        }
        else if ((_this->types[1] == TYPE_VAR) && (_this->types[0] == TYPE_VAR)) {
            _this->id = v_v_id;
            return validate_v_v(_this);
        }
        return VALIDATE_ERROR;
    }

    /**
     * instr_validate
     */   
    const char *instr_validate(Instruction *_this) {

        switch (_this->id) {
          
            /* set.i */ 
            case INSTR_SETI:

                if ((_this->types[1] == TYPE_PARAM) && (_this->types[0] == TYPE_INTEGER)) {
                    _this->id = INSTR_SETI_PARAM_INTEGER;
                    return validate_p_i(_this);
                }
                else if ((_this->types[1] == TYPE_VAR) && (_this->types[0] == TYPE_INTEGER)) {
                    _this->id = INSTR_SETI_VAR_INTEGER;
                    return validate_v_i(_this);
                }
                else if ((_this->types[1] == TYPE_VAR) && (_this->types[0] == TYPE_VAR)) {
                    _this->id = INSTR_SETI_VAR_VAR;
                    return validate_v_v(_this);
                }
                else if ((_this->types[1] == TYPE_PARAM) && (_this->types[0] == TYPE_VAR)) {
                    _this->id = INSTR_SETI_PARAM_VAR;
                    return validate_p_v(_this);
                }
                else if ((_this->types[1] == TYPE_VAR) && (_this->types[0] == TYPE_PARAM)) {
                    _this->id = INSTR_SETI_VAR_PARAM;
                    return validate_v_p(_this);
                }
                else {
                    return VALIDATE_TODO;
                }
                return VALIDATE_SYNTHAX_ERROR;
                
            /* set.f */ 
            case INSTR_SETF:

                if ((_this->types[1] == TYPE_PARAM) && (_this->types[0] == TYPE_FLOAT)) {
                    _this->id = INSTR_SETF_PARAM_FLOAT;
                    return validate_p_f(_this);
                }
                else if ((_this->types[1] == TYPE_VAR) && (_this->types[0] == TYPE_FLOAT)) {
                    _this->id = INSTR_SETF_VAR_FLOAT;
                    return validate_v_f(_this);
                }
                else if ((_this->types[1] == TYPE_VAR) && (_this->types[0] == TYPE_VAR)) {
                    _this->id = INSTR_SETF_VAR_VAR;
                    return validate_v_v(_this);
                }
                else if ((_this->types[1] == TYPE_PARAM) && (_this->types[0] == TYPE_VAR)) {
                    _this->id = INSTR_SETF_PARAM_VAR;
                    return validate_pf_v(_this);
                }
                else if ((_this->types[1] == TYPE_VAR) && (_this->types[0] == TYPE_PARAM)) {
                    _this->id = INSTR_SETF_VAR_PARAM;
                    return validate_v_pf(_this);
                }
                else {
                    return VALIDATE_TODO;
                }
                return VALIDATE_SYNTHAX_ERROR;

            /* int */
            case INSTR_INT:

                if (_this->types[0] == TYPE_VAR) {
                    _this->id = INSTR_NOP;
                    goom_hash_put_int(_this->parent->vars, _this->params[0], 0);
                    return VALIDATE_OK;
                }
                return VALIDATE_SYNTHAX_ERROR;

            /* jump */
            case INSTR_JUMP:

                if (_this->types[0] == TYPE_LABEL) {
                    _this->jump_label = _this->params[0];
                    return VALIDATE_OK;
                }
                return VALIDATE_ERROR;

            /* jzero */
            case INSTR_JZERO:

                if ((_this->types[1] == TYPE_VAR) && (_this->types[0] == TYPE_LABEL)) {
                    _this->jump_label = _this->params[0];
                    _this->data.v.var = goom_hash_get(_this->parent->vars, _this->params[1]);
                    if (_this->data.v.var == NULL) return VALIDATE_NO_SUCH_VAR;
                    return VALIDATE_OK;
                }
                return VALIDATE_ERROR;

            /* label */
            case INSTR_LABEL:

                if (_this->types[0] == TYPE_LABEL) {
                    _this->id = INSTR_NOP;
                    goom_hash_put_int(_this->parent->current_flow->labels, _this->params[0], _this->parent->current_flow->number);
                    return VALIDATE_OK;
                }
                return VALIDATE_ERROR;

            /* isequal.i */
            case INSTR_ISEQUALI:
                return validate_i(_this, INSTR_ISEQUALI_VAR_INTEGER, INSTR_ISEQUALI_VAR_VAR);

            /* isequal.f */
            case INSTR_ISEQUALF:
                return validate_f(_this, INSTR_ISEQUALF_VAR_FLOAT, INSTR_ISEQUALF_VAR_VAR);

            /* islower.i */
            case INSTR_ISLOWERI:
                return validate_i(_this, INSTR_ISLOWERI_VAR_INTEGER, INSTR_ISLOWERI_VAR_VAR);

            /* islower.f */
            case INSTR_ISLOWERF:
                return validate_f(_this, INSTR_ISLOWERF_VAR_FLOAT, INSTR_ISLOWERF_VAR_VAR);

            /* add.i */
            case INSTR_ADDI:
                return validate_i(_this, INSTR_ADDI_VAR_INTEGER, INSTR_ADDI_VAR_VAR);

            /* add.f */
            case INSTR_ADDF:
                return validate_f(_this, INSTR_ADDF_VAR_FLOAT, INSTR_ADDF_VAR_VAR);

            /* mul.i */
            case INSTR_MULI:
                return validate_i(_this, INSTR_MULI_VAR_INTEGER, INSTR_MULI_VAR_VAR);

            /* mul.f */
            case INSTR_MULF:
                return validate_f(_this, INSTR_MULF_VAR_FLOAT, INSTR_MULF_VAR_VAR);

            default:
                return VALIDATE_TODO;
        }
        return VALIDATE_ERROR;
    }

    /** EXECUTE **/
    
    void iflow_execute(InstructionFlow *_this, ScriptExecEnv *exec_env) {
        int ip = 0;
        while (ip < _this->number) {
            Instruction *instr = _this->instr[ip];
#ifdef TRACE_SCRIPT 
            printf("execute "); instr_display(instr); printf("\n");
#endif

            switch (instr->id) {

                /* SET.I */
                case INSTR_SETI_PARAM_INTEGER:
                    IVAL(*instr->data.p_i.param) = instr->data.p_i.value;
                    instr->data.p_i.param->change_listener(instr->data.p_i.param);
                    ++ip; break;

                case INSTR_SETI_VAR_VAR:
                    *instr->data.v_v.var_dest = *instr->data.v_v.var_src;
                    ++ip; break;

                case INSTR_SETI_PARAM_VAR:
                    IVAL(*instr->data.v_p.param) = instr->data.v_p.var->i;
                    instr->data.v_p.param->change_listener(instr->data.v_p.param);
                    ++ip; break;

                case INSTR_SETI_VAR_PARAM:
                    instr->data.v_p.var->i = IVAL(*instr->data.v_p.param);
                    ++ip; break;

                case INSTR_SETI_VAR_INTEGER:
                    instr->data.v_i.var->i = instr->data.v_i.value;
                    ++ip; break;

                    /* SET.F */
                case INSTR_SETF_PARAM_FLOAT:
                    FVAL(*instr->data.p_i.param) = instr->data.p_f.value;
                    instr->data.p_f.param->change_listener(instr->data.p_f.param);
                    ++ip; break;

                case INSTR_SETF_VAR_VAR:
                    *instr->data.v_v.var_dest = *instr->data.v_v.var_src;
                    ++ip; break;

                case INSTR_SETF_PARAM_VAR:
                    FVAL(*instr->data.v_p.param) = instr->data.v_p.var->f;
                    instr->data.v_p.param->change_listener(instr->data.v_p.param);
                    ++ip; break;

                case INSTR_SETF_VAR_PARAM:
                    instr->data.v_p.var->f = FVAL(*instr->data.v_p.param);
                    ++ip; break;

                case INSTR_SETF_VAR_FLOAT:
                    instr->data.v_f.var->f = instr->data.v_f.value;
                    ++ip; break;

                    /* JUMP */
                case INSTR_JUMP:
                    ip += instr->data.jump_offset; break;

                    /* JZERO */
                case INSTR_JZERO:
                    ip += (instr->data.v.var->i ? 1 : instr->data.jump_offset); break;

                case INSTR_NOP:
                    ++ip; break;

                    /* ISEQUAL.I */
                case INSTR_ISEQUALI_VAR_VAR:
                    instr->data.v_v.var_dest->i -= instr->data.v_v.var_src->i;
                    ++ip; break;

                case INSTR_ISEQUALI_VAR_INTEGER:
                    instr->data.v_i.var->i -= instr->data.v_i.value;
                    ++ip; break;

                    /* ISEQUAL.F */
                case INSTR_ISEQUALF_VAR_VAR:
                    instr->data.v_v.var_dest->f -= instr->data.v_v.var_src->f;
                    ++ip; break;

                case INSTR_ISEQUALF_VAR_FLOAT:
                    instr->data.v_f.var->f -= instr->data.v_f.value;
                    ++ip; break;

                    /* ISLOWER.I */
                case INSTR_ISLOWERI_VAR_VAR:
                    instr->data.v_v.var_dest->i = (instr->data.v_v.var_dest->i < instr->data.v_v.var_src->i);
                    ++ip; break;

                case INSTR_ISLOWERI_VAR_INTEGER:
                    instr->data.v_i.var->i = (instr->data.v_i.var->i < instr->data.v_i.value);
                    ++ip; break;

                    /* ISLOWER.F */
                case INSTR_ISLOWERF_VAR_VAR:
                    instr->data.v_v.var_dest->f = (instr->data.v_v.var_dest->f < instr->data.v_v.var_src->f);
                    ++ip; break;

                case INSTR_ISLOWERF_VAR_FLOAT:
                    instr->data.v_f.var->f = (instr->data.v_f.var->f < instr->data.v_f.value);
                    ++ip; break;

                    /* ADD.I */
                case INSTR_ADDI_VAR_VAR:
                    instr->data.v_v.var_dest->i += instr->data.v_v.var_src->i;
                    ++ip; break;

                case INSTR_ADDI_VAR_INTEGER:
                    instr->data.v_i.var->i += instr->data.v_i.value;
                    ++ip; break;

                    /* ADD.F */
                case INSTR_ADDF_VAR_VAR:
                    instr->data.v_v.var_dest->f += instr->data.v_v.var_src->f;
                    ++ip; break;

                case INSTR_ADDF_VAR_FLOAT:
                    instr->data.v_f.var->f += instr->data.v_f.value;
                    ++ip; break;

                    /* MUL.I */
                case INSTR_MULI_VAR_VAR:
                    instr->data.v_v.var_dest->i *= instr->data.v_v.var_src->i;
                    ++ip; break;

                case INSTR_MULI_VAR_INTEGER:
                    instr->data.v_i.var->i *= instr->data.v_i.value;
                    ++ip; break;

                    /* MUL.F */
                case INSTR_MULF_VAR_VAR:
                    instr->data.v_v.var_dest->f *= instr->data.v_v.var_src->f;
                    ++ip; break;

                case INSTR_MULF_VAR_FLOAT:
                    instr->data.v_f.var->f *= instr->data.v_f.value;
                    ++ip; break;

                    /* EQUALS.I * /
                case INSTR_EQUALSI_VAR_VAR:
                    return (instr->data.v_v.var_src->i == instr->data.v_v.var_dest->i)?1:2;

                case INSTR_EQUALSI_VAR_INTEGER:
                    return (instr->data.v_i.var->i == instr->data.v_i.value)?1:2;

                    / * EQUALS.F * /
                case INSTR_EQUALSF_VAR_VAR:
                    return (instr->data.v_v.var_src->f == instr->data.v_v.var_dest->f)?1:2;
                case INSTR_EQUALSF_VAR_FLOAT:
                    return (instr->data.v_f.var->f == instr->data.v_f.value)?1:2;

                    / * LOWER.I * /
                case INSTR_LOWERI_VAR_VAR:
                    return (instr->data.v_v.var_src->i > instr->data.v_v.var_dest->i)?1:2;
                case INSTR_LOWERI_VAR_INTEGER:
                    return (instr->data.v_i.var->i < instr->data.v_i.value)?1:2;

                    / * LOWER.F * /
                case INSTR_LOWERF_VAR_VAR:
                    return (instr->data.v_v.var_src->f > instr->data.v_v.var_dest->f)?1:2;
                case INSTR_LOWERF_VAR_FLOAT:
                    return (instr->data.v_f.var->f < instr->data.v_f.value)?1:2;
                    */

                default:
                    printf("NOT IMPLEMENTED : %d\n", instr->id);
                    ++ip;
            }
        }
    }

    GoomScriptScanner *currentScanner;
%}

DIGIT    [0-9]
ID       [a-zA-Z_][a-zA-Z0-9_]*

%S COMMENT

%%

<INITIAL,COMMENT>\n          { ++currentScanner->num_lines; }

<COMMENT>"*/"                { BEGIN INITIAL; }
<COMMENT>.                   { /* eat up comment */ }

<INITIAL>"/*"                { BEGIN COMMENT; }

 /*
 <INITIAL>set"."i             { currentScanner->instr = instr_init(currentScanner, "set.i", INSTR_SETI, 2); }
 <INITIAL>set"."f             { currentScanner->instr = instr_init(currentScanner, "set.f", INSTR_SETF, 2); }
 <INITIAL>add"."i             { currentScanner->instr = instr_init(currentScanner, "add.i", INSTR_ADDI, 2); }
 <INITIAL>add"."f             { currentScanner->instr = instr_init(currentScanner, "add.f", INSTR_ADDF, 2); }
 <INITIAL>mul"."i             { currentScanner->instr = instr_init(currentScanner, "mul.i", INSTR_MULI, 2); }
 <INITIAL>mul"."f             { currentScanner->instr = instr_init(currentScanner, "mul.f", INSTR_MULF, 2); }
 <INITIAL>bool                { currentScanner->instr = instr_init(currentScanner, "bool", INSTR_INT, 1); }
 <INITIAL>jump                { currentScanner->instr = instr_init(currentScanner, "jump", INSTR_JUMP, 1); }
 <INITIAL>label               { currentScanner->instr = instr_init(currentScanner, "label", INSTR_LABEL, 1); }
 <INITIAL>equals"."i          { currentScanner->instr = instr_init(currentScanner, "equals.i", INSTR_EQUALSI, 2); }
 <INITIAL>equals"."f          { currentScanner->instr = instr_init(currentScanner, "equals.f", INSTR_EQUALSF, 2); }
 <INITIAL>lower"."i           { currentScanner->instr = instr_init(currentScanner, "lower.i", INSTR_LOWERI, 2); }
 <INITIAL>lower"."f           { currentScanner->instr = instr_init(currentScanner, "lower.f", INSTR_LOWERF, 2); }
 <INITIAL>":"{ID}":"          { instr_add_param(currentScanner->instr, yytext, TYPE_LABEL); }
 <INITIAL>{ID}"."{ID}         { instr_add_param(currentScanner->instr, yytext, TYPE_PARAM); }
 */
<INITIAL>float                   { return FLOAT_TK; }
<INITIAL>int                     { return INT_TK;   }
<INITIAL>{ID}"."{ID}             { strncpy(yylval.strValue, yytext, 2047); return TYPE_PARAM;   }
<INITIAL>{ID}                    { strncpy(yylval.strValue, yytext, 2047); return TYPE_VAR;     }
<INITIAL>"-"?{DIGIT}+            { strncpy(yylval.strValue, yytext, 2047); return TYPE_INTEGER; }
<INITIAL>"-"?{DIGIT}+"."{DIGIT}* { strncpy(yylval.strValue, yytext, 2047); return TYPE_FLOAT;   }
<INITIAL>"-"?{DIGIT}+"%"         { sprintf(yylval.strValue, "%3.2f", atof(yytext)/100.0f); return TYPE_FLOAT; }
<INITIAL>[ \t]+                  /* eat up whitespace */
<INITIAL>.                       { yylval.charValue = *yytext; return *yytext;    }

%%

static void reset_scanner(GoomScriptScanner *gss) {

    gss->num_lines = 0;
    gss->instr = NULL;
    iflow_clean(gss->iflow);
    gss->current_flow = gss->iflow;
    gss->compilationOK = 1;
}

static void calculate_labels(InstructionFlow *iflow) {
        int i = 0;
        while (i < iflow->number) {
            Instruction *instr = iflow->instr[i];
            if (instr->jump_label) {
                HashValue *label = goom_hash_get(iflow->labels,instr->jump_label);
                if (label) {
                    instr->data.jump_offset = -instr->address + label->i;
                }
                else {
                    fprintf(stderr, "ERROR: could not find label %s\n", instr->jump_label);
                    instr->id = INSTR_NOP;
                }
            }
            ++i;
        }
}

void goom_script_scanner_compile(GoomScriptScanner *_currentScanner, PluginInfo *pluginInfo, const char *script) {

    printf("\n=== Starting Compilation ===\n");
    currentScanner = _currentScanner;
    reset_scanner(currentScanner);
    currentScanner->pluginInfo = pluginInfo;
    
    yy_scan_string(script);
    yyparse();
    gsl_commit_compilation();

    calculate_labels(currentScanner->iflow);
    
    printf("=== Compilation done. # of lines: %d. # of instr: %d ===\n", currentScanner->num_lines, currentScanner->iflow->number);
}

void goom_script_scanner_execute(GoomScriptScanner *scanner) {
    
    ScriptExecEnv exec_env;
    exec_env.vars = scanner->vars;
    if (scanner->compilationOK)
        iflow_execute(scanner->iflow, &exec_env);
}

GoomScriptScanner *goom_script_scanner_new() {

    GoomScriptScanner *gss = (GoomScriptScanner*)malloc(sizeof(GoomScriptScanner));

    gss->iflow = iflow_new();
    gss->vars  = goom_hash_new();

    reset_scanner(gss);
    gss->compilationOK = 0;
    return gss;
}

int goom_script_scanner_is_compiled(GoomScriptScanner *gss) {
    return gss->compilationOK;
}

void goom_script_scanner_free(GoomScriptScanner *gss) {
    free(gss);
}

int yywrap(void) { return 1; }

