/*******************************************************************************
* parsers.cpp: Command line and configuration files parsers
*-------------------------------------------------------------------------------
* (c)1999-2001 VideoLAN
* $Id: parsers.cpp,v 1.4 2002/04/10 10:47:18 asmax Exp $
*
* Authors: Benoit Steiner <benny@via.ecp.fr>
*          Arnaud de Bossoreille de Ribou <bozo@via.ecp.fr>
*          Cyril Deguet <asmax@via.ecp.fr>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*
*-------------------------------------------------------------------------------
*
*******************************************************************************/


//------------------------------------------------------------------------------
// Preamble
//------------------------------------------------------------------------------
#include "defs.h"

#include <stdio.h>
#ifdef HAVE_OPENDIR
#include <dirent.h>
#endif
#include <sys/types.h>

#include "common.h"
#include "debug.h"
#include "reflect.h"
#include "serialization.h"
#include "string.h"
#include "vector.h"
#include "hashtable.h"
#include "stack.h"
#include "buffers.h"
#include "exception.h"
#include "file.h"
#include "log.h"
#include "stream.h"

#include "parsers.h"
#include "lexer.h"

#include "stack.cpp"
#include "stream.cpp"


//******************************************************************************
// class C_ParserException
//******************************************************************************
//
//******************************************************************************

//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
E_Parser::E_Parser(int iCode, const C_String& strMsg) :
                        E_Exception(iCode, strMsg)
{
  // Nothing to do
}


//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
E_Parser::E_Parser(int iCode, const C_String& strMsg, E_Exception e) :
                        E_Exception(iCode, strMsg, e)
{
  // Nothing to do
}





//******************************************************************************
// class C_CfgFileParser
//******************************************************************************
//
//******************************************************************************

//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
C_CfgFileParser::C_CfgFileParser(C_ParserHandler* pCallBack) : 
    m_cContextStack(10)
{
  ASSERT(pCallBack);
  m_pCallBack = pCallBack;

  // Init the file position indicators
  m_iLineNumber = 0;

}


//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
void C_CfgFileParser::Parse(const C_String& strFileName, bool bCompletePath)
{
  ASSERT(m_pCallBack);

  try
  {
    // Open the cfg file
    C_File* pCfgFile = new C_File(strFileName);
    pCfgFile->Open("r");

    yyin = pCfgFile->GetHandle();

    // Parse the file
    int iTokenType = 0;
    while ((iTokenType = yylex()) != 0)
    {
      C_String strFirstToken(yytext);
      if (iTokenType == TOK_END)
      {
        // Unstack the section
        if(m_cContextStack.Size() == 0)
        {
          throw E_Parser(GEN_ERR, strFirstToken + "tag at line " + 
                         m_iLineNumber + "in excess");
        }
        C_String* pstrSection = m_cContextStack.Pop();
        m_pCallBack->OnEndSection(*pstrSection);
        delete pstrSection;
        if (yylex() == TOK_NEWLINE)
        {
          m_iLineNumber++;
        }
        else {
          throw E_Parser(GEN_ERR,
                         C_String("Error in the configuration file at line ") +
                         m_iLineNumber);
        }
      }
      else if (iTokenType == TOK_BEGIN)
      { 
        if (yylex() != TOK_VALUE)
        {
          throw E_Parser(GEN_ERR,
                         C_String("Error in the configuration file at line ") +
                         m_iLineNumber);
        }
        else if(m_cContextStack.Size() == m_cContextStack.Capacity())
        {
          throw E_Parser(GEN_ERR, C_String("Error line ") + m_iLineNumber +
                         ": only " + m_cContextStack.Capacity() +
                         " section levels allowed");
        }
        C_String strValue(yytext);
        // Remove the quotes. FIXME: should be done by the lexer
        C_String strSection(strValue.SubString(1, strValue.Length()-1));
        m_cContextStack.Push(new C_String(strSection));
        m_pCallBack->OnStartSection(strSection);
        if (yylex() == TOK_NEWLINE)
        {
          m_iLineNumber++;
        }
        else {
          throw E_Parser(GEN_ERR,
                         C_String("Error in the configuration file at line ") +
                         m_iLineNumber);
        }
      }
      else if (iTokenType == TOK_VAR)
      {
        if (yylex() != TOK_EQUAL) 
        {
          throw E_Parser(GEN_ERR,
                         C_String("Error in the configuration file at line ") +
                         m_iLineNumber);
        }
        else if (yylex() != TOK_VALUE)
        {
          throw E_Parser(GEN_ERR,
                         C_String("Error in the configuration file at line ") +
                         m_iLineNumber);
        }
        C_String strValue(yytext);
        // Remove the quotes. FIXME: should be done by the lexer
        C_String strStrippedValue(strValue.SubString(1, strValue.Length()-1));
        m_pCallBack->OnProperty(strFirstToken, strStrippedValue);
        if (yylex() == TOK_NEWLINE)
        {
          m_iLineNumber++;
        }
        else {
          throw E_Parser(GEN_ERR,
                         C_String("Error in the configuration file at line ") +
                         m_iLineNumber);
        }
      } 
      else if (iTokenType == TOK_NEWLINE)
      {
        m_iLineNumber++;
      }
      else {
        throw E_Parser(GEN_ERR,
                       C_String("Error in the configuration file at line ") +
                       m_iLineNumber);
      }
    }
  }
  catch(E_Exception e)
  {
    throw E_Parser(GEN_ERR, "Parsing of file '" + strFileName + "' failed", e);
    
  }
}

// Needed for flex functions
extern "C" int yywrap(void)
{
  return 1;
}
